// Persistence of Vision Ray Tracer Scene Description File
// File: parsys.inc
// Vers: 3.5
// Desc: Tim Nikias' Particle System
// Date: 20.02.04 (dd.mm.yy)
// Auth: Tim Nikias Wenclawiak
// Last Update: 22.4.04 (dd.mm.yy)

//Description
// I/O-Based Particle System

/*
This set of macros are designed for a single purpose: creating particle effects like smoke, rain,
dust and the like. Different kinds of effects are "grouped", e.g. one group for rain and one group
for smoke. All groups are then processed by the Main-Core.
As the data has to be passed on from frame to frame, the system creates and uses several files for
storage and handling purposes. Those created by the Core begin with "TPS", the abbreviation for
"Tim's Particle System". Files specific for a group begin with the groupname.
Inbetween frames, all these files should remain untouched. After an animation, the files may of
course be deleted, as they will be generated anew everytime the animation is restarted.
Note that ALL Macros, especially the Core, assume that the groups have been set up properly and that
all files have been written to disk successfully. When stopping a render during parse-time, it may
well happen that you corrupt the File-I/O-Files, and thus introduce an error into the system. The
only help when the System returns an error, is to rerun the entire sequence and/or delete the
temporary files.

Another file that is used by the System is "tps_lfrm.txt", it keeps track which frame was rendered
last in order to check if the running system is uptodate and called in sequence. It may be used to
check with which frame to begin a subset-animation in case a trace had been interrupted, nontheless,
it is needed by the Core and should not be modified.
*/

//Homepage:
// www.nolights.de
//Email:
// TimNikias(@)gmx.net

#declare _parsys_inc_tnw = 1;
#declare _parsys_epsilon = .001;

#declare tnw_ParsysRun = false;
#declare Parsys_Stats_Loaded = 0;
#declare Parsys_Stats_Saved = 0;
#declare Parsys_Stats_Active = 0;
#declare Parsys_Stats_Processed = 0;

#ifndef (Animation_Time) #declare Animation_Time = 1; #end

//A random Stream for every frame
#declare TPS_Random = seed(frame_number);

//I tend to use the variable Internal_POV_Clock instead of clock throughout
// the code, so that I may easily specify several clocks for different
// include-files. And I also have means to clip the clock to a value between
// 0 and 1, as my files mainly base upon a clock running from 0 to 1, and would
// return inconsistant results if used with different values.
#ifndef (Internal_POV_Clock) #declare Internal_POV_Clock=clock; #end
#if (Internal_POV_Clock<0) #declare Internal_POV_Clock=0; #end
#if (Internal_POV_Clock>1) #declare Internal_POV_Clock=1; #end

/*
Global Variables:
-Some Variables that are declared in ParticleGroup_Begin and needed until ParticleGroup_End
  #declare tnw_ActualConstructGroup = _GroupName;
  #declare tnw_ACG_Objects = 0;
  #declare tnw_ACG_Impact = "";
  #declare tnw_ACG_Death = "";
  #declare tnw_ACG_Details = _DetailSettings;
  #declare tnw_ACG_Datasize = 4;
  #fopen _tnw_ParticleObjectOut
  #fopen _tnw_ParticleGroupOut

tnw_ResetParsys = Boolean if Parsys (all data) needs to be reset
tnw_Parsys_GroupT = Array of Active Groups
tnw_Parsys_NewGroupT [Inclusion]: Boolean to check if GroupT was newly created

tnwTPS_xxx = Group-Related Declarations

Files the System creates when included:
tps_grpt.tmp - Group-Tracker, keeps track of active Groups
tps_lfrm.txt - Last-Frame-Value, memorizes the last frame_number
tps_pm.tmp   - Particle-Memory, short-term bucket for new particles
tps_pn.tmp   - Particle-Numbers, stores two relation-arrays: Group and Particle-Count

Files the System creates per Group:
xxx1.tmp      - First Particle-Bunch
xxx2.tmp      - Second Particle-Bunch
xxx_s.tmp     - Setup for xxx-Group with Macro-Calls for modification
xxx_o.tmp     - Object-Array (using strings)
xxx_m.tmp     - Group-Memory into which objects, declarations etc are parsed and handed once
                to a running Group
*/


/****  VNormalize()  ****/
#macro VNormalize(_Vec)
 //Create proper return-value
 #if (vlength(_Vec))
  #local _Ret=vnormalize(_Vec); #else
  #local _Ret=<0,0,0>; #end
 //Return:
 _Ret
#end

/*** Parse-String ***/
//I'm supplying an own version, because when creating/using macros with this
//technique, the file needs to be valid as long as macro is used during
//parse-time. Hence, to make sure this doesn't interfere with other macros,
//that make use of ParseString, I've made my own version.
#macro _tnw_ParseString(_SomeBlaString)
  #fopen _tmp_Out "tps_parse.tmp" write
  #write (_tmp_Out, _SomeBlaString)
  #fclose _tmp_Out
  #include "tps_parse.tmp"
#end


//If this is the first frame of the render, or if the Group-Tracker-File
// doesn't exist, I reset the System. Thus, all Groups will get checked once
// when doing a continued trace.
#if (initial_frame = frame_number | !file_exists("tps_grpt.tmp"))
  #declare tnw_Parsys_NewGroupT = true;
  #fopen _tmp_FileOut "tps_grpt.tmp" write
  #write (_tmp_FileOut, "//Group-Tracker for Tim Nikias' I/O-Particle System\n")
  #write (_tmp_FileOut, "#declare tnw_Parsys_GroupT = array[1] {\"default\"}")
  #fclose _tmp_FileOut
#else
  #declare tnw_Parsys_NewGroupT = false;
#end
#include "tps_grpt.tmp"

//Particle-Memory needs to be reset in every case, but won't be used when continuing
// a simulation: if the same frame is parsed twice, new particles may get added twice.
// Since these particles are already placed in the running simulation, their clones in
// the particle memory are left out.
#fopen _tmp_FileOut "tps_pm.tmp" write
#write (_tmp_FileOut, "//Particle-Memory for Tim Nikias' I/O-Particle System\n")
#write (_tmp_FileOut, "0")
#fclose _tmp_FileOut

//Check if the System needs to be reset: if frame != current frame, it may
// only be 1 higher (next frame), otherwise, the Simulation isn't in synch
// with the frame.
#if (!file_exists("tps_lfrm.txt"))
  #declare tnw_ResetParsys = true;
  #fopen _tmp_FileOut "tps_lfrm.txt" write
  #write (_tmp_FileOut, frame_number)
  #fclose _tmp_FileOut
  #declare _TNW_ParsysCurrentFrame = frame_number;
#else
  #fopen _tmp_FileIn "tps_lfrm.txt" read
  #read (_tmp_FileIn, _TNW_ParsysCurrentFrame)
  #fclose _tmp_FileIn
  //Next frame: Move onwards with the simulation
  #if (frame_number = _TNW_ParsysCurrentFrame+1)
    #declare tnw_ResetParsys = false;
    #fopen _tmp_FileOut "tps_lfrm.txt" write
    #write (_tmp_FileOut, frame_number)
    #fclose _tmp_FileOut
    #declare _TNW_ParsysCurrentFrame = frame_number;
  #else
    //Same frame: Continued trace
    #if (frame_number = _TNW_ParsysCurrentFrame)
      #declare tnw_ResetParsys = false;
    #else
      //Future frame: Continued trace later than stopped. Simply updates
      // the present particles
      #if (frame_number > _TNW_ParsysCurrentFrame)
        #declare tnw_ResetParsys = false;
        #fopen _tmp_FileOut "tps_lfrm.txt" write
        #write (_tmp_FileOut, frame_number)
        #fclose _tmp_FileOut
        #declare _TNW_ParsysCurrentFrame = frame_number;
      #else  
      //Earlier/same frame: Update the saved Framenumber and Reset
        #declare tnw_ResetParsys = true;
        #fopen _tmp_FileOut "tps_lfrm.txt" write
        #write (_tmp_FileOut, frame_number)
        #fclose _tmp_FileOut
        #declare _TNW_ParsysCurrentFrame = frame_number;
      #end
    #end
  #end
#end

//Always Reset on beginning of animation
#if (clock=0) #declare tnw_ResetParys = true; #end

/* Particle-Processing
First of all, a Groups gets loaded from the Group-Tracker, and the particles are processed
one by one. For each one, the time-interval is generated and the particle moved. If the
distance to the new position is larger than a given threshold, the interval is reduced
and the process repeated until a sufficient amount of detail is applied. After a valid
move, object-interaction is calculated. The process repeats itself until the particle
has reached the actual moment of the animation. In this manner, the entire path of a
particle is calculated with a certain amount of detail to avoid large jumps and overcome
the iterational problem of parabolic flight vs point-to-point movement.
*/

/*Particle_Add*/
//Function:
// Saves the data of a particle to setup
//Parameters:
// 1. Group-Name
// 2. Array of Vectors
//Returns:
// 1. Changes to Data-File
//Notes:
// Appends particle to Data for next frame
#macro Particle_Add(_GroupName,_Data)
  //Check the array for consistency with main routines
  #local _Size = dimension_size(_Data,1);
  #if (_Size < 4)
    #local _tmpData = array[4];
    #local _A=0;
    #while (_A < _Size)
      #local _tmpData[_A] = _Data[_A];
      #local _A=_A+1;
    #end
    #while (_A < 4)
      #local _tmpData[_A] = <0,0,0>;
      #local _A=_A+1;
    #end
    #local _Data = _tmpData; 
  #end
  #fopen _tnw_PMemoryOut "tps_pm.tmp" append
  #write (_tnw_PMemoryOut, ",\"",_GroupName,"\"")
  _SavePM_VectorParticle(_Data)
  #fclose _tnw_PMemoryOut
#end

/*Show_Particles*/
//Function:
// Hands data of a particle to a macro for Visualization
//Parameters:
// 1. Group-Name
// 2. Macro-Name
//Returns:
// Nothing but the calls of the Macro
//Notes:
// None
#macro Show_Particles(_GroupName,_MacroName)
  #local _NextFrame = mod(_TNW_ParsysCurrentFrame+1,2);
  #fopen _tmp_Out "tps_parse.tmp" write
  #write (_tmp_Out, "#macro _MakeParticleVisibleWith(_PartData)\n")
  #write (_tmp_Out, "  #local _tmpData = _PartData;\n")
  #write (_tmp_Out, "  #ifndef (",_MacroName,")\n")
  #write (_tmp_Out, "    #macro ",_MacroName,"(_Data)\n")
  #write (_tmp_Out, "      #warning concat(\"Show_Particles: Macro named ",_MacroName," not declared!\\n\")\n")
  #write (_tmp_Out, "    #end\n")
  #write (_tmp_Out, "  #end\n")
  #write (_tmp_Out, "  ",_MacroName,"(_tmpData)\n")
  #write (_tmp_Out, "#end\n")
  #fclose _tmp_Out
  #include "tps_parse.tmp"
  #local _ActualMoment = Internal_POV_Clock*Animation_Time;
  //Read the result-data of this frame
  #local _tnw_Filename = concat(_GroupName,str(_NextFrame,1,0),".tmp");
  #if (file_exists(_tnw_Filename))
    #fopen _tnw_ParticlesIn _tnw_Filename read
    #read (_tnw_ParticlesIn, _temp_Start)
    #while (defined(_tnw_ParticlesIn))
      //Get a particle
      #local _PartData = _Load_VectorParticle();
      #local _PresumedAge = _ActualMoment-_PartData[0].z;
      #if (_PartData[0].x >= 0
           & _PartData[0].x < _PartData[0].y
           & _PresumedAge >= 0
           & _PresumedAge < _PartData[0].y)
        _MakeParticleVisibleWith(_PartData)
        #declare Parsys_Stats_Active = Parsys_Stats_Active+1;
      #end
    #end      
    #fclose _tnw_ParticlesIn
  #else
    #warning concat("The ",_GroupName,"-Group doesn't exist, no particles placed!\n")  
  #end
#end

/*Parsys_Run*/
//Function:
// Iterates the Groups and the particles
//Parameters:
// None
//Returns:
// 1. Changes to active Particles
//Notes:
// None
#macro Parsys_Run()
  #local _GroupSize = dimension_size(tnw_Parsys_GroupT,1);
  //First, process the Groups
  #local _RunGroups = tnw_Parsys_GroupT;
  #local _RunGroupsAmount = array[_GroupSize];
  #local _RunGroupsAmount[0] = 0;
  #local _GroupCounter = 1;
  #while (_GroupCounter < _GroupSize)
    #local _ActiveGroup = _RunGroups[_GroupCounter];
    _Process_ParticleGroup(_ActiveGroup)
    #local _RunGroupsAmount[_GroupCounter] = _ParticlesSaved;
    #local _RunGroupsAmount[0] = _RunGroupsAmount[0]+_ParticlesSaved;
    #local _GroupCounter = _GroupCounter+1;
  #end
  //Process the Particle-Storage
  #local _ParticleValue = -1;
  #local _ActiveFrame = mod(_TNW_ParsysCurrentFrame,2);
  #local _NextFrame = mod(_TNW_ParsysCurrentFrame+1,2);
  #fopen _tnw_ParticlesIn "tps_pm.tmp" read
  #read (_tnw_ParticlesIn, _temp_Start)
  #local _lastGroup = "";
  #while (defined(_tnw_ParticlesIn))
    //Get the Group
    #read (_tnw_ParticlesIn, _ThisGroup)
    //Open/Close appropriate data
    #if (strcmp(_lastGroup,"")!=0 & strcmp(_ThisGroup,_lastGroup)!=0)
      #fclose _tnw_ParticlesOut    
    #end
    #if (strcmp(_ThisGroup,_lastGroup)!=0)
      _Add2GroupT(_ThisGroup)
      #fopen _tnw_ParticlesOut concat(_ThisGroup,str(_NextFrame,1,0),".tmp") append
      //Environmental Data for this Group
      #include concat(_ThisGroup,"_m.tmp")
      #local _A=0;
      #local _Index = 0;
      #local _IndexFound = false;
      #while (_A<_GroupSize)
        #if (strcmp(_ThisGroup,_RunGroups[_A])=0)
          #local _Index = _A;
          #local _IndexFound = true;
        #end
        #local _A = _A + 1;
      #end
      //If the Group hasn't been found in former Group-Array, then enlargen
      // the array
      #if (!_IndexFound)
        #local _Index = _GroupSize;
        #local _GroupSize = _GroupSize + 1;
        #local _NewGroups = array[_GroupSize];
        #local _NewGroupsAmount = array[_GroupSize];
        #local _A=0;
        #while (_A<_GroupSize-1)
          #local _NewGroups[_A] = _RunGroups[_A];
          #local _NewGroupsAmount[_A] = _RunGroupsAmount[_A];
          #local _A=_A+1;
        #end
        #local _NewGroups[_GroupSize-1] = _ThisGroup;
        #local _NewGroupsAmount[_GroupSize-1] = 0;
        #local _RunGroups = _NewGroups;
        #local _RunGroupsAmount = _NewGroupsAmount;
      #end
    #end
    //Load Particle, process it and save it
    #local _PartData = _Load_VectorParticle();
    _Process_SingleVectorParticle(_ThisGroup,_PartData)
    #if (_PartData[0].x >= 0 & _PartData[0].x <= _PartData[0].y) 
      _Save_VectorParticle(_PartData)
      #local _RunGroupsAmount[_Index] = _RunGroupsAmount[_Index] + 1;
      #local _RunGroupsAmount[0] = _RunGroupsAmount[0]+1;
    #end
    //Memorize used Group to avoid multiple File I/O-Calls on the same Group
    #local _lastGroup = _ThisGroup;
    #undef _ThisGroup
    #read (_tnw_ParticlesIn, _MarkerValue)
  #end
  //Only close if we've actually parsed a particle
  #if (strcmp(_lastGroup,""))
    #fclose _tnw_ParticlesOut
  #end
  #fclose _tnw_ParticlesIn
  //Now, each setup has been passed, and the Particle-Memory got added
  //to the individual Groups. All is well. :-)
  //RunGroup-Arrays now store the amount of particles per active group, and
  //this gets saved to "tps_pn.tmp" (for Particle-Numbers)
  #fopen _tnw_Out "tps_pn.tmp" write
  #write (_tnw_Out, "//Particle-Numbers, stores Groupnames with active particles, by Tim Nikias' I/O-Particle System\n")
  #write (_tnw_Out, "#declare tnw_Parsys_ActiveGroupnames = array[",_GroupSize,"];\n")
  #write (_tnw_Out, "#declare tnw_Parsys_ActiveGroupnumbers = array[",_GroupSize,"];\n")
  #local _A=0;
  #while (_A<_GroupSize)
    #write (_tnw_Out, "#declare tnw_Parsys_ActiveGroupnames[",_A,"] = \"",_RunGroups[_A],"\";\n")
    #write (_tnw_Out, "#declare tnw_Parsys_ActiveGroupnumbers[",_A,"] = ",_RunGroupsAmount[_A],";\n")
    #local _A=_A+1;
  #end
  #fclose _tnw_Out
  #include "tps_pn.tmp"
  //Tracks if simulation was run or not
  #declare tnw_ParsysRun = true;
#end

/*_Process_ParticleGroup*/
//Function:
// Processes a Particle-Group and the accompanying particles
//Parameters:
// 1. Group-Name
//Returns:
// 1. Calculates Data for the next frame
//Notes:
// None
#macro _Process_ParticleGroup(_GroupName)
  #local _ActiveFrame = mod(_TNW_ParsysCurrentFrame,2);
  #local _NextFrame = mod(_TNW_ParsysCurrentFrame+1,2);
  //I only need to append new data, since the data-file is reset during
  //Group-Construction
  #fopen _tnw_ParticlesOut concat(_GroupName,str(_NextFrame,1,0),".tmp") append
  #local _ParticleValue = 0;
  #declare _ParticlesSaved = 0;
  //Get the Environmental Data for the running Group
  #include concat(_GroupName,"_m.tmp")
  //Read the former data
  #fopen _tnw_ParticlesIn concat(_GroupName,str(_ActiveFrame,1,0),".tmp") read
  #read (_tnw_ParticlesIn, _temp_Start)
  #while (defined(_tnw_ParticlesIn))
    //Get a particle
    #local _PartData = _Load_VectorParticle();
    //Process the particle
    _Process_SingleVectorParticle(_GroupName,_PartData)
    #local _ParticleValue = _ParticleValue + 1;
    //Save the new particle-data if Age is still there
    #if (_PartData[0].x >= 0 & _PartData[0].x < _PartData[0].y) 
      _Save_VectorParticle(_PartData)
      #declare _ParticlesSaved=_ParticlesSaved+1;
    #end
  #end
  #fclose _tnw_ParticlesOut
  //If no particles have been saved, remove the setup from the list
  #if (_ParticlesSaved = 0)
    _Sub2GroupT(_GroupName)
  #end
#end

/*_Process_SingleVectorParticle*/
//Function:
// Moves the particles in intervals that are based upon the
// Detail-Settings for the Calculations
//Parameters:
// 1. Group-Name
// 2. Particle-Data
//Returns:
// 1. Changes to Particle-Data
//Notes:
// None
#macro _Process_SingleVectorParticle(_GroupName,_PartData)
  //_ParticleValue = Index of Particle if it were in the loaded Array
  //Create current and last copy of Data: when calculations have gone
  // too far for the Detail-Settings, the System always reverts to
  // Last-Data and attempts calculation with a smaller timestep
  #local _tmp_Data = _PartData;
  #local ParticleIndex = _ParticleValue;
  #local _last_Data = _PartData;
  //Generate comparable age ("is" vs "should be")
  #local _ActualMoment = Internal_POV_Clock*Animation_Time;
  #local _RealAge = _tmp_Data[0].x;
  #local _PresumedAge = _ActualMoment-_tmp_Data[0].z;

  //Time-Step shouldn't be larger than the actual age the particle may get
  #local _tmp_TStep = tnwTPS_DetailSettings.y;
  #if (_PresumedAge-_RealAge < _tmp_TStep)
    #local _tmp_TStep = _PresumedAge-_RealAge; 
  #end
  #local _particleDead=false;
  #local _ProxTest = -1;
  //Iterate the Particle in a loop:
  #local _Object_Steps = 0;
  #while (_PresumedAge > _RealAge & !_particleDead)
    /* Move by a Timestep */
    //Memorizing former and new position (with "tmp" vs "last") I can check for
    // distances and other significant changes to judge, wether an adaption is
    // required for the timestep, if I can save environmental-sampling, etc
    #include concat(_GroupName,".tmp")
    #local _tmp_Data[2] = _tmp_Data[2]+_tmp_TStep*_tmp_Data[3];
    #local _last_TStep = _tmp_TStep;
    #local _Travelled = vlength(_tmp_Data[2]-_last_Data[2]);
    
    /* Adaptive-Iteration */
    //If Distance-Check is actually set to on and TStep isn't underbid, I
    // have room for adaption.
    #if (tnwTPS_DetailSettings.z > 0
         & _tmp_TStep > tnwTPS_DetailSettings.x
         & _tmp_TStep < tnwTPS_DetailSettings.y)
      //...check if distance meets Detail-Settings...
      #if (_Travelled > tnwTPS_DetailSettings.z)
        //...and if its too far, dump the new data and revert to former one,
        // but then use lower TStep for next iteration
        #local _tmp_Data = _last_Data;
        #local _tmp_TStep = _tmp_TStep/(_Travelled/tnwTPS_DetailSettings.z+.02);
        #local _last_TStep = 0;
        #local _Travelled = 0;
      #else
        //If we have some space, we increase the Time-Step for next iteration
        #local _tmp_TStep = _tmp_TStep*(tnwTPS_DetailSettings.z/(max(_Travelled,.15)-.1));
      #end
      //Check the proper boundaries for the new Time-Step
      #if (_tmp_TStep < tnwTPS_DetailSettings.x)
        #local _tmp_TStep = tnwTPS_DetailSettings.x;
      #end
      #if (_tmp_TStep > tnwTPS_DetailSettings.y)
        #local _tmp_TStep = tnwTPS_DetailSettings.y;
      #end
    #end
    
    /* Object-Interaction */
    #local _particleDead = false;
    #local _Heading = VNormalize(_tmp_Data[2]-_last_Data[2]);
    #local _rem_TStep = _last_TStep;
    
    #local _ModifiedVelocity = false;
    #while (_rem_TStep > 0 & tnwTPS_ObjectInteraction & !_particleDead & _Travelled)
      #local _Object_Steps = _Object_Steps + 1;
      //Matrix for Forward-Direction, used to align samples, only needed, when
      // particle-Radius > 0
      #if (_tmp_Data[1].y > 0)
        #local _ZMatrix = -_Heading;
        #if (vlength(_ZMatrix)<=.9)
          #local _ZMatrix = VNormalize(tnwTPS_gravDir);
          #if (vlength(_ZMatrix)<=.9)
            #local _ZMatrix = <0,-1,0>;
          #end
        #end
        #if (abs(_ZMatrix.y) != 1)
          #local _XMatrix = vnormalize(vcross(<0,1,0>,_ZMatrix));
        #else
          #local _XMatrix = vnormalize(vcross(<1,0,0>,_ZMatrix));
        #end
        #local _YMatrix = vnormalize(vcross(_ZMatrix,_XMatrix));
        #local _thisParticleMatrix =
          function{
            transform{
              matrix<_XMatrix.x,_XMatrix.y,_XMatrix.z,
                     _YMatrix.x,_YMatrix.y,_YMatrix.z,
                     _ZMatrix.x,_ZMatrix.y,_ZMatrix.z,
                      0,0,0>
            }
          };
      #end
      
      //For Object-Interaction, I sample every object and then sort through the
      // the results and use closest sample
      #local _tnw_ObjCounter = 0;
      #while (_tnw_ObjCounter < tnwTPS_ObjectInteraction)
        #local _ObjectToHit = tnwTPS_Objects[_tnw_ObjCounter]
        #local _ObjectIDamp = tnwTPS_ObjectValues[_tnw_ObjCounter][0];
        #local _ObjectBDamp = tnwTPS_ObjectValues[_tnw_ObjCounter][1];
        #local _ObjectFX    = tnwTPS_ObjectValues[_tnw_ObjCounter][2];
        #local _ObjectIMacro = tnwTPS_ObjectMacro[_tnw_ObjCounter][0];
        #local _ObjectDMacro = tnwTPS_ObjectMacro[_tnw_ObjCounter][1];

        //Default values in case no intersection is found
        #local _Intersection = <0,0,0>;
        #local _IsectDir = <0,0,0>;
        #local _Normal = <0,0,0>;
        #local _Relative2Center = <0,0,0>;

        #local _tnw_Sample = 0;
        #if (tnwTPS_ParticleSamplingSwitch = 0)
          #local _tnw_MaxSamples = _tmp_Data[1].x+5;
        #else
          #local _tnw_MaxSamples = _tmp_Data[1].x;
        #end
        #local _tnw_KillVelocity = <0,0,0>;
        #while (_tnw_Sample <= _tnw_MaxSamples)
        
        #if (tnwTPS_ParticleSamplingSwitch = 0) //=> Hemispherical Hull Samples
          //First three samples are strictly from center and along velocity-
          // and gravity-vector, then come User-Defined extra Samples if
          // a particle's radius is larger than 0. Samples are first generated
          // at the origin and then moved to the particle
          #if (_tnw_Sample <= 5)
            #if (_tnw_Sample = 0)
              #local _TraceFrom = <0,0,0>;
              #local _TraceTo = <0,0,0>;
            #end
            #if (_tnw_Sample = 1)
              #local _TraceFrom = VNormalize(tnwTPS_gravDir)*_tmp_Data[1].y;
              #local _TraceTo = <0,0,0>;
            #end
            #if (_tnw_Sample = 2)
              #local _TraceFrom = _Heading*_tmp_Data[1].y;
              #local _TraceTo = <0,0,0>;
            #end
            #if (_tnw_Sample = 3)
              #local _TraceFrom = VNormalize(vcross(vcross(_Heading,tnwTPS_gravDir),_Heading))*_tmp_Data[1].y;
              #local _TraceTo = <0,0,0>;
            #end
            #if (_tnw_Sample = 4)
              #local _TraceFrom = VNormalize(vcross(_Heading,tnwTPS_gravDir))*_tmp_Data[1].y;
              #local _TraceTo = <0,0,0>;
            #end
            #if (_tnw_Sample = 5)
              #local _TraceFrom = VNormalize(vcross(tnwTPS_gravDir,_Heading))*_tmp_Data[1].y;
              #local _TraceTo = <0,0,0>;
            #end
            //If radius of particle is 0, only one sample is needed
            #if (_tnw_Sample = 0 & _tmp_Data[1].y = 0)
              #local _tnw_Sample = _tnw_MaxSamples;
            #end
          #else
            //Spread samples across front-hemisphere of particle
            #local _TraceY = 1-sqrt((_tnw_Sample-3)/_tmp_Data[1].x);
            #local _TraceZ = -sqrt(1-pow(_TraceY,2));
            #local _TracePre = vrotate(<0,_TraceY,_TraceZ>,z*360/.618*(_tnw_Sample-3));
            //Orient the Samples to the heading of the particle
            #local _TraceFrom = _thisParticleMatrix(_TracePre.x,_TracePre.y,_TracePre.z)*_tmp_Data[1].y;
            #local _TraceTo = <0,0,0>;
          #end
          #local _thisRelative2Center = _TraceFrom;
          #local _TraceFrom = _TraceFrom + _last_Data[2];
        #else //=> Macro-Generated Samples
          #local _giveData = _last_Data;
          #local _giveDataB = _tmp_Data;
          #local _giveSampleValue = _tnw_Sample;
          #local _giveMaxSampleValue = _tnw_MaxSamples;
          #local _TraceFrom = _tmp_Data[2];
          #local _TraceTo = <0,0,0>;
          _tnw_ParseString(
            concat(
              tnwTPS_ParticleSamplingMacro,
              "(_giveData,_giveDataB,_giveSampleValue,",
              "_giveMaxSampleValue,_TraceFrom,_TraceTo)\n"))
          #local _thisRelative2Center = _TraceFrom - _last_Data[2];
        #end

        #if (vlength(_TraceTo))
          #local _TraceDir = vnormalize(_TraceTo);
        #else
          #local _TraceDir = _Heading;
        #end
        //Gather Intersections with Object
        #local _thisNormal = <0,0,0>;
        #local _thisIntersection = trace(_ObjectToHit,         //Object
                                         _TraceFrom,_TraceDir, //Start and End
                                         _thisNormal);         //Normal
        #local _thisIsectDir = _thisIntersection-_TraceFrom;
        
        //Check, if the sample isn't inside the object already
        #if (tnwTPS_ParticleSamplingSwitch = 0 & vlength(_thisRelative2Center))
          #local _checkNormal = <0,0,0>;
          #local _sampleCheck = trace(_ObjectToHit,_TraceFrom,-_thisRelative2Center,_checkNormal);
          #if (vlength(_checkNormal))
            #if (vlength(_TraceFrom-_sampleCheck) < vlength(_thisRelative2Center))
              #local _thisNormal = <0,0,0>; //Switch Sample to invalid
              #local _tnw_KillVelocity = _tnw_KillVelocity - _thisRelative2Center;
            #end
          #end
        #end
        
        #if (vlength(_thisNormal))
          //If _Normal points in same direction as movement, trace was applied
          // to what POV-Ray considers the inside of an object. This may occur
          // with patches or open objects, e.g. cylinders. We reverse the
          // normal as it has to point towards the particle to be handled
          // correctly.
          #if (vdot(_thisNormal,_Heading) > 0)
            #local _thisNormal = -_thisNormal;
          #end

          //Get closest hit from traces
          #if (!_tnw_Sample)
            #local _Normal = _thisNormal;
            #local _Intersection = _thisIntersection;
            #local _IsectDir = _thisIsectDir;
            #local _Relative2Center = _thisRelative2Center;
          #else
            #if ( (!vlength(_Normal) & vlength(_thisNormal))
                 | (vlength(_IsectDir) > vlength(_thisIsectDir)
                 & vlength(_thisNormal)) )
              #local _Intersection = _thisIntersection;
              #local _IsectDir = _thisIsectDir;
              #local _Normal = _thisNormal;
              #local _Relative2Center = _thisRelative2Center;
            #end
          #end
        #end
        //If the hits from the first four traces are too far away, we skip
        // the extra traces
        #if (tnwTPS_ParticleSamplingSwitch = 0
             & _tnw_Sample = 3
             & !(vlength(_Normal)
             & vlength(_IsectDir) <= max(_tmp_Data[1].y,_parsys_epsilon)*2))
          #local _tnw_Sample = _tnw_MaxSamples;
        #end
        
        #local _tnw_Sample = _tnw_Sample + 1;
        #end
               
        //Memorize closest hit on object and move on through the other objects
        #if (_tnw_ObjCounter = 0)
          #local _mNormal = _Normal;
          #local _mIntersection = _Intersection;
          #local _mIsectDir = _IsectDir;
          #local _mRelative2Center = _Relative2Center;
          #local _mObjectIDamp = _ObjectIDamp;
          #local _mObjectBDamp = _ObjectBDamp;
          #local _mObjectFX = _ObjectFX;
          #local _mObjectIMacro = _ObjectIMacro;
          #local _mObjectDMacro = _ObjectDMacro;
        #else
          //Compare z-distance and use closest
          #if ( (!vlength(_mNormal) & vlength(_Normal))
               | (vlength(_mIsectDir) > vlength(_IsectDir)
               & vlength(_Normal)) )
            #local _mNormal = _Normal;
            #local _mIntersection = _Intersection;
            #local _mIsectDir = _IsectDir;
            #local _mRelative2Center = _Relative2Center;
            #local _mObjectIDamp = _ObjectIDamp;
            #local _mObjectBDamp = _ObjectBDamp;
            #local _mObjectFX = _ObjectFX;
            #local _mObjectIMacro = _ObjectIMacro;
            #local _mObjectDMacro = _ObjectDMacro;
          #end
        #end

      #local _tnw_ObjCounter = _tnw_ObjCounter + 1;
      #end
      
      //Now we have the shortest sample. If that one is closer than the
      // the distance the particle wants to travel, then we rebounce it
      #if (vlength(_mNormal) & vlength(_mIsectDir) <= _Travelled)
        #local _ModifiedVelocity = true;
        //Now, the path is further processed with linear calculations
        #local _ImpactTime = vlength(_mIsectDir)/_Travelled*_rem_TStep;
        #local _rem_TStep = _rem_TStep - _ImpactTime;        
        //Repel velocity (and take a bit, based on Object's dampening)
        #local _reverse_Vel = (1+_mObjectIDamp)*vdot(_mNormal,_tmp_Data[3])*_mNormal;
        #local _tmp_Data[3] = (_tmp_Data[3]-_reverse_Vel)*_mObjectBDamp
                              -VNormalize(_mRelative2Center)*(_parsys_epsilon+_tmp_Data[1].y/2)
                              +VNormalize(_tnw_KillVelocity)*_parsys_epsilon;
        
        //Sliding-Mode actually just switches if Impact-Effects shall be used or not
        #local _Sliding = off;
        
        //Switch to Sliding if repulsion force is too weak
        #if (vlength(vdot(_mNormal,_last_Data[3])*_mNormal) <= _last_Data[1].y*_tmp_TStep*2 )
          #local _Sliding = on;
        #end

        //Gravity killer + Sliding-Switch
        #if ( vdot(_mNormal,-tnwTPS_gravDir)>=.99
             & vlength(vdot(_mNormal,_tmp_Data[3])*_mNormal) <= tnwTPS_gravAmount*_tmp_TStep*1.01
             & _mObjectIDamp != -1)
          #local _tmp_Data[3] = _tmp_Data[3]-_mNormal*vdot(_mNormal,_tmp_Data[3]);
          #local _Sliding = on;
        #end
               
        //If speed drops to 0, particle can't move anymore, so iteration is stopped at once
        #if (!vlength(_tmp_Data[3]))
          #local _ImpactTime = _rem_TStep;
          #local _rem_TStep = 0;
        #end
        
        //Update _last_Data for next iteration
        #local _last_Data[0] = _last_Data[0]+x*_ImpactTime;
        #local _last_Data[2] = _mIntersection+_mNormal*_parsys_epsilon-_mRelative2Center;
        //Update _tmp_Datawith rebounced position
        #local _tmp_Data[0] = _tmp_Data[0]+x*_ImpactTime;
        #local _tmp_Data[2] = _mIntersection+_mNormal*_parsys_epsilon-_mRelative2Center+_tmp_Data[3]*_rem_TStep;
        
        //Different Macro-Calls
        //Impact-Effect based on Particle-Group
        #if (_mObjectFX != 2 & !_Sliding & _ImpactTime & tnwTPS_ImpactMacroSwitch)
          #local _giveNormal = _mNormal;
          #local _giveImpact = _mIntersection;
          #local _giveTime = _last_Data[0].x+_last_Data[0].z;
          _tnw_ParseString(concat(tnw_ImpactMacroName,"(_last_Data,_giveTime,_giveImpact,_giveNormal)\n"))
        #end
        //Impact-Effect based on object
        #if (_mObjectFX = 1 & !_Sliding & _ImpactTime & strcmp(_mObjectIMacro,"off"))
          #local _giveTime = _last_Data[0].x+_last_Data[0].z;
          #local _giveNormal = _mNormal;
          #local _giveImpact = _mIntersection;
          _tnw_ParseString(concat(_mObjectIMacro,"(_last_Data,_giveTime,_giveImpact,_giveNormal)\n"))
        #end
        //Death-Effect based on object
        #if (_mObjectFX = 2)
          //Start up Death-Macro, if needed
          #if (strcmp(_mObjectDMacro,"off"))
            #local _giveTime = _last_Data[0].x+_last_Data[0].z;
            #local _giveNormal = _mNormal;
            #local _giveImpact = _mIntersection;
            _tnw_ParseString(concat(_mObjectDMacro,"(_last_Data,_giveTime,_giveImpact,_giveNormal)\n"))
          #end
          //Set Particle's age to dead
          #local _tmp_Data = _last_Data;
          #local _tmp_Data[0] = <_tmp_Data[0].y,_tmp_Data[0].y,_tmp_Data[0].z>;
          #local _particleDead = true;
        #end        
        #local _ProxTest = 0;
      #else //No Intersection between old and new position
        #if (_ProxTest=-1) #local _ProxTest = 0; #end
        #if (_tmp_Data[1].z = 0)
          //Very crude method to keep particles from penetrating surfaces: if they
          // are too close, their velocity just gets reversed away from the surface
          #local _Proximity=vlength(_mIntersection-_tmp_Data[2]);
          #if (vlength(_mNormal) & _Proximity*1.5 < _tmp_Data[1].y)
            #local _tmp_Data[3] = _tmp_Data[3]+_mNormal*_tmp_Data[1].y*(1-pow((_Proximity*1.5/_tmp_Data[1].y),2));
            #if (vdot(_mNormal,_tmp_Data[3]) < 0)
              #local _tmp_Data[3] = _tmp_Data[3]-2.1*vdot(_mNormal,_tmp_Data[3])*_mNormal;
            #end
            #local _ModifiedVelocity=true;
            #local _ImpactTime=0;
            //Counts how often in sequence this test applies. If more than once, particle cannot
            //move any further away from object without tampering with age or other possible
            //problems (e.g. object penetration)
            #if (_ProxTest=0) #local _ProxTest=1; #end
            #if (_ProxTest=1) #local _ProxTest=2; #end
          #end
        #end
        #if (_ModifiedVelocity)
          //Reset new final position
          #local _tmp_Data[2] = _last_Data[2]+_tmp_Data[3]*_rem_TStep;
          #local _ModifiedVelocity = false;
          #if (_ProxTest=2)
            #local _ImpactTime = _rem_TStep;
            #local _rem_TStep = 0;
            #local _tmp_Data[0] = _tmp_Data[0]+x*_ImpactTime;
          #end
        #else
          //Age the particle when no modification of velocity has taken place
          #local _ImpactTime = _rem_TStep;
          #local _rem_TStep = 0;
          #local _tmp_Data[0] = _tmp_Data[0]+x*_ImpactTime;
        #end
      #end
      
      //New for the iteration through the linear bounce-loop
      #local _Heading = _tmp_Data[2]-_last_Data[2];
      #local _Travelled = vlength(_Heading);
      #local _Heading = VNormalize(_Heading);
      
      //Remove Function
      #if (_tmp_Data[1].y>0)  #undef _thisParticleMatrix  #end
    
    #end //of Object-Interaction-Loop
    
    #local _tmp_Data[0] = _tmp_Data[0]+x*_rem_TStep;
    
    //Update Internal Measurement-Variables
    #local _RealAge = _tmp_Data[0].x;
    #local _last_Data = _tmp_Data;
    //If remaining time is smaller than actual interval, just do the remaining time...
    #if (_PresumedAge-_RealAge < _tmp_TStep)
      #local _tmp_TStep = _PresumedAge-_RealAge; 
    #end
  #end //of Iteration-Loop
  //If Particle is dead now, we call the DeathMacro
  #if (tnwTPS_DeathMacroSwitch & _tmp_Data[0].x >= _tmp_Data[0].y)
    #local _giveData = _tmp_Data;
    #local _giveTime = _tmp_Data[0].z+_tmp_Data[0].y;
    #local _giveImpact = _tmp_Data[2];
    #local _giveNormal = <0,0,0>;
    _tnw_ParseString(concat(tnw_DeathMacroName,"(_last_Data,_giveTime,_giveImpact,_giveNormal)\n"))
  #end
  //Replace parameter-array with calculated one
  #declare _PartData = _tmp_Data;
  //Update the Stats
  #declare Parsys_Stats_Processed = Parsys_Stats_Processed+1;
#end

/****************** Particle-Array Related ******************/

/*Particles_Get*/
//Function:
// Gets all particles from a certain group and places them into an array
//Parameters:
// 1. Group-Name
//Returns:
// 1. Array of 1D-arrays (Particle-Data)
//Notes:
// None
#macro Particles_Get(_GroupName)
  #local _ActiveFrame = mod(_TNW_ParsysCurrentFrame,2);
  #local _NextFrame = mod(_TNW_ParsysCurrentFrame+1,2);
  #if (tnw_ParsysRun)
    #local _tnw_Filename = concat(_GroupName,str(_NextFrame,1,0),".tmp");
  #else
    #local _tnw_Filename = concat(_GroupName,str(_ActiveFrame,1,0),".tmp");
  #end
  #if (file_exists("tps_pn.tmp"))
    #include "tps_pn.tmp"
  #else
    #declare tnw_Parsys_ActiveGroupnames=array[1] {"default"};
    #declare tnw_Parsys_ActiveGroupnumbers=array[1] {0};
  #end
  //Get index of array if the simulation wasn't reset
  #if (tnw_ResetParsys)
    #local _Index = -1;
  #else
    #local _Size = dimension_size(tnw_Parsys_ActiveGroupnames,1);
    #local _Index = -1;
    #local _A=0;
    #while (_A<_Size)
      #if (!strcmp(_GroupName,tnw_Parsys_ActiveGroupnames[_A]))
        #local _Index = _A;    
      #end
      #local _A=_A+1;
    #end
  #end
  //Fill particle-data into the array
  #if (_Index = -1 | _Index = 0)
    #local Return_Array = array[1];
    #local Return_Array[0] = array[1] {<0,0,0>};
  #else
    #local _A=0;
    #if (file_exists(_tnw_Filename))
      #local _Size = tnw_Parsys_ActiveGroupnumbers[_Index];
      #if (_Size > 0)
        #include concat(_GroupName,"_m.tmp")
        #local Return_Array = array[_Size];
        #fopen _tnw_ParticlesIn _tnw_Filename read
        #read (_tnw_ParticlesIn, _temp_Start)
        #while (defined(_tnw_ParticlesIn))
          #local _PartData = _Load_VectorParticle();
          //Datasize
          #local _Size = dimension_size(_PartData,1);
          #if (_Size < tnwTPS_Datasize)
            #local _tmpData = array[tnwTPS_Datasize];
            #local _B=0;
            #while (_B < _Size)
              #local _tmpData[_B] = _PartData[_B];
              #local _B=_B+1;
            #end
            #while (_B < tnwTPS_Datasize)
              #local _tmpData[_B] = <0,0,0>;
              #local _B=_B+1;
            #end
            #local _PartData = _tmpData; 
          #end
          #local Return_Array[_A] = _PartData;
          #local _A=_A+1;
        #end      
        #fclose _tnw_ParticlesIn
      #else
        #local Return_Array = array[1];
        #local Return_Array[0] = array[1] {<0,0,0>};
      #end
    #else
      #warning concat("The ",_GroupName,"-File doesn't exist!\n")  
      #local Return_Array = array[1];
      #local Return_Array[0] = array[1] {<0,0,0>};
    #end //of File-There-Check
  #end //of Index-Check
  //Return
  Return_Array
#end

/*Particles_Add*/
//Function:
// Saves the data of a particle to setup
//Parameters:
// 1. Group-Name
// 2. Array of Particle-Data
//Returns:
// 1. Changes to Data-File
//Notes:
// Appends particles to Data for next frame
#macro Particles_Add(_GroupName,_DataBundle)
  #fopen _tnw_PMemoryOut "tps_pm.tmp" append
  #local _Size = dimension_size(_DataBundle,1);
  #if (dimension_size(_DataBundle[0],1)!=1)
    #local _A=0;
    #while (_A<_Size)
      #write (_tnw_PMemoryOut, ",\"",_GroupName,"\"")
      _SavePM_VectorParticle(_DataBundle[_A])
      #local _A=_A+1;
    #end
  #end
  #fclose _tnw_PMemoryOut
#end

/*Check_Particles*/
//Function:
// Boolean return if the particle-array returned by Get_Particles()
// contains particles.
//Parameters:
// 1. Particle-Array
//Returns:
// 1. Boolean if particles are present
//Notes:
// None
#macro Particles_Check(_Array)
 #local _Ret=(dimension_size(_Array[0],1)=1 ? false : true);
 _Ret
#end

/*Particles_Save*/
//Function:
// Saves array of particles to disk
//Parameters:
// 1. Particle-Array
// 2. Filename
//Returns:
// Writes Filename
//Notes:
// None
#macro Particles_Save(_SaveArray,_OutputFile)
  #if (Particles_Check(_SaveArray))
    #local _Amount = dimension_size(_SaveArray,1);
    #fopen _FileOut _OutputFile write
    #write (_FileOut, "//Particles for Tim Nikias' Particle-System\n")
    #write (_FileOut, _Amount,", //Amount of Particles stored\n")
    #local _A=0;
    #while (_A<_Amount)
      #local _Size=dimension_size(_SaveArray[_A],1);
      #write (_FileOut, _Size,",\n")
      #local _B=0;
      #while (_B<_Size)
          #write (_FileOut, _SaveArray[_A][_B],",")
        #local _B=_B+1;
      #end
      #local _A=_A+1;
    #end
  #else
    #local _Amount = dimension_size(_SaveArray,1);
    #fopen _FileOut _OutputFile write
    #write (_FileOut, "//Particles for Tim Nikias' Particle-System\n")
    #write (_FileOut, "0 //Amount of Particles stored\n")
  #end  
  #fclose _FileOut
#end

/*Particles_Load*/
//Function:
// Loads particles from disk and returns the array
//Parameters:
// 1. Filename
//Returns:
// Array of Particles
//Notes:
// None
#macro Particles_Load(_InputFile)
  #fopen _FileIn _InputFile read
  #read (_FileIn, _Amount)
  #if (_Amount != 0)
    #local _Array=array[_Amount];
    #local _A=0;
    #while (_A<_Amount)
      #read (_FileIn, _Size)
      #local _Array[_A] = array[_Size];
      #local _B=0;
      #while (_B<_Size)
        #read (_FileIn, _Vector)
        #local _Array[_A][_B] = _Vector;     
        #local _B=_B+1;
      #end
      #local _A=_A+1;
    #end
    #fclose _FileIn
  #else
    #local _Array=array[1];
    #local _Array[0]=array[1]{<0,0,0>};
  #end
  _Array
#end


/*Particles_ResetAge*/
//Function:
// Modifies Array of particles
//Parameters:
// 1. Particle-Array
// 2. Base-Age
// 3. Random-Delta added to Base-Age
//Returns:
// Nothing, but modifies Array of Particles
//Notes:
// None
#macro Particles_ResetAge(_Array,_BaseAge,_DeltaAge)
  #local _TempArray = _Array;
  #if (Particles_Check(_Array))
    #local _Size = dimension_size(_Array,1);
    #local _A=0;
    #while (_A<_Size)
      #local _TempArray[_A][0] = <0,_TempArray[_A][0].y,_BaseAge+rand(TPS_Random)*_DeltaAge>;    
      #local _A=_A+1;
    #end
  #end
  #declare _Array = _TempArray;
#end

/*Particles_Show*/
//Function:
// Hands data of particles to a macro for Visualization
//Parameters:
// 1. Array of particles
// 2. Macro-Name
//Returns:
// Nothing but the calls of the Macro
//Notes:
// None
#macro Particles_Show(_Array,_MacroName)
  #fopen _tmp_Out "tps_parse.tmp" write
  #write (_tmp_Out, "#macro _MakeParticleVisibleWith(_PartData)\n")
  #write (_tmp_Out, "  #local _tmpData = _PartData;\n")
  #write (_tmp_Out, "  #ifndef (",_MacroName,")\n")
  #write (_tmp_Out, "    #macro ",_MacroName,"(_Data)\n")
  #write (_tmp_Out, "      #warning concat(\"Show_Particles: Macro named ",_MacroName," not declared!\\n\")\n")
  #write (_tmp_Out, "    #end\n")
  #write (_tmp_Out, "  #end\n")
  #write (_tmp_Out, "  ",_MacroName,"(_tmpData)\n")
  #write (_tmp_Out, "#end\n")
  #fclose _tmp_Out
  #include "tps_parse.tmp"
  #local _ActualMoment = Internal_POV_Clock*Animation_Time;
  #if (Particles_Check(_Array))
    #local _Size = dimension_size(_Array,1);
    #local _A=0;
    #while (_A<_Size)
      #local _PartData = _Array[_A];
      #local _PresumedAge = _ActualMoment-_PartData[0].z;
      #if (_PartData[0].x >= 0
           & _PartData[0].x <= _PartData[0].y
           & _PresumedAge >= 0)
        _MakeParticleVisibleWith(_PartData)
        #declare Parsys_Stats_Active = Parsys_Stats_Active+1;
      #end
      #local _A=_A+1;
    #end      
  #end
#end

/********** Single-Particle I/O **************/
/*_Load_VectorParticle*/
//Function:
// Loads the data of a particle from disk
//Parameters:
// None
//Returns:
// 1. Array of Vectors
//Notes:
// Requires a read-opened file with handle _tnw_ParticlesIn
#macro _Load_VectorParticle()
  #declare Parsys_Stats_Loaded = Parsys_Stats_Loaded+1;
  #read (_tnw_ParticlesIn, _Size)
  #local _Data = array[_Size];
  #local _A=0;
  #while (_A<_Size)
    #read (_tnw_ParticlesIn, _tmp_Vector)
    #local _Data[_A] = _tmp_Vector;
    #local _A=_A+1;
  #end
  //Return
  _Data
#end

/*_Save_VectorParticle*/
//Function:
// Saves the data of a particle to disk
//Parameters:
// 1. Array of Vectors
//Returns:
// 1. Changes to Data-File
//Notes:
// Requires a write-opened file with handle _tnw_ParticlesOut
#macro _Save_VectorParticle(_Data)
  #declare Parsys_Stats_Saved = Parsys_Stats_Saved+1;
  #local _Size = dimension_size(_Data,1);
  #write (_tnw_ParticlesOut, ",",_Size)
  #local _A=0;
  #while (_A<_Size)
    #write (_tnw_ParticlesOut, ",",_Data[_A])
    #local _A=_A+1;
  #end
  #write (_tnw_ParticlesOut, "\n")
#end
//Same as above, just for the Particle-Memory
#macro _SavePM_VectorParticle(_Data)
  #declare Parsys_Stats_Saved = Parsys_Stats_Saved+1;
  #local _Size = dimension_size(_Data,1);
  #write (_tnw_PMemoryOut, ",",_Size)
  #local _A=0;
  #while (_A<_Size)
    #write (_tnw_PMemoryOut, ",",_Data[_A])
    #local _A=_A+1;
  #end
  #write (_tnw_PMemoryOut, ",0")
  #write (_tnw_PMemoryOut, "\n")
#end

/**************** Groups ******************/

/* Particle-Groups
The general idea here is to create different setups for different kinds of
effects, e.g. fire (doesn't really need gravity) vs water (relies heavily on
gravity). The particles of a setup are saved in two files, one for even, one
for odd frames. Data is always loaded from one frame and then generated for
the next. Basically, in a Group-File, the macro-calls are saved which are
applied to the particle's data to modify it's position, orientation etc.
*/

/*ParticleGroup_Begin*/
//Function:
// Creates Group-File for a particle-type
//Parameters:
// 1. Group-Name (used as front-part for filenames)
//Returns:
// 1. Empty and open Group-File with _tnw_ParticleGroupOut
// 2. Empty and open Object-File with _tnw_ParticleObjectOut
// 3. Declaration of present Group being processed
//Notes:
// None
#macro ParticleGroup_Begin(_GroupName,_DetailSettings)
  #declare tnw_ActualConstructGroup = _GroupName;
  #declare tnw_ACG_Objects = 0;
  #declare tnw_ACG_Impact = "";
  #declare tnw_ACG_Death = "";
  #declare tnw_ACG_Details = _DetailSettings;
  #declare tnw_ACG_Datasize = 4;
  #fopen _tnw_ParticleGroupOut concat(_GroupName,".tmp") write
  #write (_tnw_ParticleGroupOut, "//Particle-Group File, created by Tim Nikias' I/O-Particle-System\n")
  #write (_tnw_ParticleGroupOut, "// Following lines are calls to macros that modify\n")
  #write (_tnw_ParticleGroupOut, "// the velocities and/or positions of particles\n")
  #write (_tnw_ParticleGroupOut, "// belonging to this Group.\n\n")
  #write (_tnw_ParticleGroupOut, "//Default values needed for within Parsys:\n")
  #write (_tnw_ParticleGroupOut, "#declare tnwTPS_gravAmount = 0;\n")
  #write (_tnw_ParticleGroupOut, "#declare tnwTPS_gravDir = <0,0,0>;\n\n")
  #write (_tnw_ParticleGroupOut, "#declare tnwTPS_ImpactMacroSwitch = 0;\n")
  #write (_tnw_ParticleGroupOut, "#declare tnwTPS_DeathMacroSwitch = 0;\n\n")
  #write (_tnw_ParticleGroupOut, "#declare tnwTPS_ParticleSamplingSwitch = 0;\n\n")
  #write (_tnw_ParticleGroupOut, "//Now User-Specified Macros\n")
  #if (tnw_Parsys_NewGroupT) _Add2GroupT(_GroupName) #end
  //Object-Bucket
  #fopen _tnw_ParticleObjectOut concat(_GroupName,"_o.tmp") write
  #write (_tnw_ParticleObjectOut, "//Object-Data for ",tnw_ActualConstructGroup,"-Group\n")
  #write (_tnw_ParticleObjectOut, "0")
#end

/*ParticleGroup_End*/
//Function:
// Finalizes the Group of a Particle-Group and cleans the next
// frame's data to easily append new particles to the file.
//Parameters:
// None, but requires ParticleGroup_Begin as a previous call somewhere
//Returns:
// Several Filechanges
//Notes:
// None
#macro ParticleGroup_End()
  //Closes the open Group-File
  #fclose _tnw_ParticleGroupOut
  #fclose _tnw_ParticleObjectOut
  //Check actual frame and clear the next frame's data, adding comments and such
  #local _ActiveFrame = mod(_TNW_ParsysCurrentFrame,2);
  #local _NextFrame = mod(_TNW_ParsysCurrentFrame+1,2);
  #fopen _tnw_Out concat(tnw_ActualConstructGroup,str(_NextFrame,1,0),".tmp") write
  #write (_tnw_Out, "//Particles of ",tnw_ActualConstructGroup,"-Group, for frame ",(_TNW_ParsysCurrentFrame+1),"\n")
  #write (_tnw_Out, "0")
  #fclose _tnw_Out
  //Remove data of Active-Frame if Parsys got Reset
  #local _NextFile = concat(tnw_ActualConstructGroup,str(_ActiveFrame,1,0),".tmp");
  #if (tnw_ResetParsys | !file_exists(_NextFile))
    #fopen _tnw_Out _NextFile write
    #write (_tnw_Out, "//Particles of ",tnw_ActualConstructGroup,"-Group, for frame ",_TNW_ParsysCurrentFrame,"\n")
    #write (_tnw_Out, "0")
    #fclose _tnw_Out
  #end
  //Build Object-File
  #fopen _tnw_Out concat(tnw_ActualConstructGroup,"_m.tmp") write
  #write (_tnw_Out, "//Detail-Settings for the ",tnw_ActualConstructGroup,"-Group\n")
  #write (_tnw_Out, "#declare tnwTPS_DetailSettings = ",tnw_ACG_Details,";\n")
  #write (_tnw_Out, "//Datasize\n")
  #write (_tnw_Out, "#declare tnwTPS_Datasize = ",tnw_ACG_Datasize,";\n")
  #write (_tnw_Out, "//Object-Arrays for this Group\n")
  #if (!tnw_ACG_Objects)
    //No objects, just add simple comment into the file
    #write (_tnw_Out, "#declare tnwTPS_ObjectInteraction = 0; // No objects used!\n")
  #else
    //First, we build the internal arrays
    #local _ObjCounter = 0;
    #local _Objects = array[tnw_ACG_Objects]
    #local _DampFX  = array[tnw_ACG_Objects][3]
    #local _FXMacro = array[tnw_ACG_Objects][2]
    #fopen _tnw_In concat(tnw_ActualConstructGroup,"_o.tmp") read
    #read (_tnw_In, _tempStart)
    #undef _tempStart
    #while (defined(_tnw_In))
      #read (_tnw_In, _tnw_tps_ObjName, _tnw_tps_IDamp, _tnw_tps_BDamp, _tnw_tps_FX, _tnw_tps_IM, _tnw_tps_DM)
      #local _Objects[_ObjCounter] = _tnw_tps_ObjName;
      #local _DampFX[_ObjCounter][0] = _tnw_tps_IDamp;
      #local _DampFX[_ObjCounter][1] = _tnw_tps_BDamp;
      #local _DampFX[_ObjCounter][2] = _tnw_tps_FX;
      #local _FXMacro[_ObjCounter][0] = _tnw_tps_IM;
      #local _FXMacro[_ObjCounter][1] = _tnw_tps_DM;
      #local _ObjCounter = _ObjCounter + 1;
      #undef _tnw_tps_ObjName
      #undef _tnw_tps_IDamp
      #undef _tnw_tps_BDamp
      #undef _tnw_tps_FX
      #undef _tnw_tps_IM
      #undef _tnw_tps_DM
    #end    
    #fclose _tnw_In
    //Then, we write these to the Memory:
    // Object-Declarations into an array, Values in a seperate array, and Macros
    // as Strings that get parsed at runtime
    #write (_tnw_Out, "#declare tnwTPS_ObjectInteraction = ",tnw_ACG_Objects,"; //Objects, yay! :-)\n")
    //Object-Array
    #write (_tnw_Out, "#declare tnwTPS_Objects = array[",tnw_ACG_Objects,"]\n{\n")
    #local _tmpCounter = 0;
    #while (_tmpCounter < tnw_ACG_Objects)
      #write (_tnw_Out, "  ",_Objects[_tmpCounter],",\n")
      #local _tmpCounter = _tmpCounter + 1;
    #end
    #write (_tnw_Out, "}\n")
    //Friction and Effect-Array
    #write (_tnw_Out, "#declare tnwTPS_ObjectValues = array[",tnw_ACG_Objects,"][3]\n{\n")
    #local _tmpCounter = 0;
    #while (_tmpCounter < tnw_ACG_Objects)
      #write (_tnw_Out, "  {",max(_DampFX[_tmpCounter][0],0),",")
      #write (_tnw_Out, "",max(_DampFX[_tmpCounter][1],0),",")
      #write (_tnw_Out, "",_DampFX[_tmpCounter][2],"")
      #write (_tnw_Out, "},\n")
      #local _tmpCounter = _tmpCounter + 1;
    #end
    #write (_tnw_Out, "}\n")
    //Macro-Array
    #write (_tnw_Out, "#declare tnwTPS_ObjectMacro = array[",tnw_ACG_Objects,"][2]\n{\n")
    #local _tmpCounter = 0;
    #while (_tmpCounter < tnw_ACG_Objects)
      #write (_tnw_Out, "  {\"",_FXMacro[_tmpCounter][0],"\",")
      #write (_tnw_Out, "\"",_FXMacro[_tmpCounter][1],"\",")
      #write (_tnw_Out, "},\n")
      #local _tmpCounter = _tmpCounter + 1;
    #end
    #write (_tnw_Out, "}\n")
    //Environment-Union
    #write (_tnw_Out, "#declare tnwTPS_Environment = ")
    #if (tnw_ACG_Objects > 1)
      #write (_tnw_Out, "union{\n")
    #else
      #write (_tnw_Out, "\n")
    #end
    #local _tmpCounter = 0;
    #while (_tmpCounter < tnw_ACG_Objects)
      #write (_tnw_Out, "  object{",_Objects[_tmpCounter],"}\n")
      #local _tmpCounter = _tmpCounter + 1;
    #end
    #if (tnw_ACG_Objects > 1)
      #write (_tnw_Out, "}\n")
    #else
      #write (_tnw_Out, "\n")
    #end
  #end
  //Impact-Switch-Tracker
  #if (strcmp(tnw_ACG_Impact,"")!=0)
    #write (_tnw_Out, "#ifndef (",tnw_ACG_Impact,") #macro ",tnw_ACG_Impact,"(_Data,_T,_I,_N) #end #end\n")
    #write (_tnw_Out, "#declare tnwTPS_ImpactMacroSwitch = 1;\n")
  #else
    #write (_tnw_Out, "#declare tnwTPS_ImpactMacroSwitch = 0;\n")
  #end
  //Death-Switch-Tracker
  #if (strcmp(tnw_ACG_Death,"")!=0)
    #write (_tnw_Out, "#ifndef (",tnw_ACG_Death,") #macro ",tnw_ACG_Death,"(_Data,_T,_I,_N) #end #end\n")
    #write (_tnw_Out, "#declare tnwTPS_DeathMacroSwitch = 1;\n")
  #else
    #write (_tnw_Out, "#declare tnwTPS_DeathMacroSwitch = 0;\n")
  #end
  #fclose _tnw_Out
#end

/*ParticleGroup_AddObject*/
//Function:
// Adds an Object to the Environment for a Group
//Parameters:
// 1. Object-Name as String
// 2. Dampening on repel when particle hits the object (clipped below 0)
// 3. Dampening on entire energy after an impact (clipped below 0)
// 4. Special FX on impact: 0=off, 1=Impact, 2=Kill particle
// 5. Macro-Name for Impact-FX ("off" for no extra effect)
// 6. Macro-Name for Kill-FX ("off" for no extra effect)
//Returns:
// None, but adds data to "parsing" Step of Parsys for Environmental interaction
// for the particles
//Notes:
// The Impact/Kill macro that got attached to the particles will occur in any
// case, they are not overriden with these settings. E.g. if you want all but one
// wall to have a certain effect upon impact, you need to define the macro for
// every wall, not for the partiles.
#macro ParticleGroup_Object(_ObjectName,_ColIDamp,_ColBDamp,_ColFX,_MImpactName,_MKillName)
  #declare tnw_ACG_Objects = tnw_ACG_Objects + 1;
  #local _tmp_ColIDamp =_ColIDamp;
  #local _tmp_ColBDamp =_ColBDamp;
  #if (_tmp_ColIDamp != -1)
    #local _tmp_ColIDamp = min(max(_tmp_ColIDamp,0),1);
  #end
  #if (_tmp_ColIDamp != -1)
    #local _tmp_ColBDamp = max(_tmp_ColBDamp,0);
  #end  
  #write (_tnw_ParticleObjectOut, ",\n\"",_ObjectName,"\",",_tmp_ColIDamp,",",_tmp_ColBDamp,",",_ColFX,",\"",_MImpactName,"\",\"",_MKillName,"\"")
#end


/********** Group-Tracking ***************/
/* GroupT - Group-Tracker
Keeps track of active Groups. When particles are processed, the GroupTracker-Array is loaded
and the particles of active Groups are calculated. Once a Group has no remaining
particles, the Group gets removed. When new particles are generated, the Group gets
added again. Especially saves parsing time when there are several Groups running
at once.
*/

/*_Add2GroupT*/
//Function:
// Adds a Group-Name, if not present, to the Group-Tracker. Done initially when
// Groups are built in the first frame, Groups get removed when no particles
// remaining.
//Parameters:
// 1. The Group-Name to be added
//Returns:
// 1. Changes to Group-Tracker-File
//Notes:
// None
#macro _Add2GroupT(_GroupName)
  #local _GroupT_Size = dimension_size(tnw_Parsys_GroupT,1);
  #local _Found = false;
  #local _A=0;
  #while (_A < _GroupT_Size)
    #if (!strcmp(tnw_Parsys_GroupT[_A],_GroupName))
      #local _Found = true;
    #end
    #local _A = _A+1;
  #end
  #if (!_Found)
    #fopen _tmp_FileOut "tps_grpt.tmp" write
    #write (_tmp_FileOut, "//Group-Tracker for Tim Nikias' I/O-Particle System\n")
    #write (_tmp_FileOut, "#declare tnw_Parsys_GroupT = array[",_GroupT_Size+1,"]\n{\n")
    #write (_tmp_FileOut, "  \"default\"")
    #local _A=1;
    #while (_A < _GroupT_Size)
      #write (_tmp_FileOut, ",\n  \"",tnw_Parsys_GroupT[_A],"\"")
     #local _A=_A+1;
    #end
    #write (_tmp_FileOut, ",\n  \"",_GroupName,"\"\n}\n")
    #fclose _tmp_FileOut
    #include "tps_grpt.tmp"
  #end
#end

/*_Sub2GroupT*/
//Function:
// Removes a Group-Name, if present, from the Group-Tracker. Meant to be used
// only when the System recognizes that no particles of a given
// Group are remaining. Group should then get added when new
// particles appear.
//Parameters:
// 1. The Group-Name to be removed
//Returns:
// 1. Changes to Group-Tracker-File
//Notes:
// None
#macro _Sub2GroupT(_GroupName)
  #local _GroupT_Size = dimension_size(tnw_Parsys_GroupT,1);
  #local _Found = false;
  #local _Pos = 0;
  #local _A=0;
  #while (_A < _GroupT_Size)
    #if (!strcmp(tnw_Parsys_GroupT[_A],_GroupName))
      #local _Found = true;
      #local _Pos = _A;
    #end
    #local _A = _A+1;
  #end
  #if (_Found)
    #fopen _tmp_FileOut "tps_grpt.tmp" write
    #write (_tmp_FileOut, "//Group-Tracker for Tim Nikias' I/O-Particle System\n")
    #write (_tmp_FileOut, "#declare tnw_Parsys_GroupT = array[",_GroupT_Size-1,"]\n{\n")
    #write (_tmp_FileOut, "  \"default\"")
    #local _A=1;
    #while (_A < _GroupT_Size)
      #if (_A != _Pos)
        #write (_tmp_FileOut, ",\n  \"",tnw_Parsys_GroupT[_A],"\"")
      #end
      #local _A=_A+1;
    #end
    #write (_tmp_FileOut, "\n}\n")
    #fclose _tmp_FileOut
    #include "tps_grpt.tmp"
  #end
#end


/************** Parsys EDK **********************/

//Basic Building-Blocks to add a macro to the Group
#macro ParsysEDK_AddMacro(_MacroName)
  #write (_tnw_ParticleGroupOut, _MacroName,"(_tmp_Data,_tmp_TStep")
#end
#macro ParsysEDK_AddParameterValue(_Parameter)
  #write (_tnw_ParticleGroupOut, ",",_Parameter)
#end
#macro ParsysEDK_AddParameterString(_Parameter)
  #write (_tnw_ParticleGroupOut, ",\"",_Parameter,"\"")
#end
#macro ParsysEDK_AddParameterVariable(_Parameter)
  #write (_tnw_ParticleGroupOut, ",",_Parameter)
#end
#macro ParsysEDK_CloseMacro()
  #write (_tnw_ParticleGroupOut, ")\n")
#end
#macro ParsysEDK_AddDeclarationValue(_Name,_Parameter)
  #write (_tnw_ParticleGroupOut, "#declare ",_Name,"=",_Parameter,";\n")
#end
#macro ParsysEDK_AddDeclarationString(_Name,_Parameter)
  #write (_tnw_ParticleGroupOut, "#declare ",_Name,"=\"",_Parameter,"\";\n")
#end
#macro ParsysEDK_AddDeclarationVariable(_Name,_Parameter)
  #write (_tnw_ParticleGroupOut, "#declare ",_Name,"=",_Parameter,";\n")
#end
#macro ParsysEDK_AddLineFeed()
  #write (_tnw_ParticleGroupOut, "\n")
#end
#macro ParsysEDK_AddSDL(_SDLString)
  #write (_tnw_ParticleGroupOut, _SDLString)
#end

/******** Parsys Effects ************/
/** Datasize **/
//-Constructor
#macro ParticleGroup_Datasize(_MinSize)
  #if (_MinSize > 4)
    #declare tnw_ACG_Datasize = _MinSize;
    ParsysEDK_AddMacro("_VectorParticle_Datasize")
    ParsysEDK_AddParameterValue(_MinSize)
    ParsysEDK_CloseMacro()
  #end
#end
//-Actual Macro
#macro _VectorParticle_Datasize(_Data,_TStep,_Size)
  #local _thisSize = dimension_size(_Data,1);
  #if (_thisSize < _Size)
    #local _tmpData = array[_Size];
    #local _A=0;
    #while (_A<_thisSize)
      #local _tmpData[_A] = _Data[_A];
      #local _A=_A+1;
    #end
    #while (_A<_Size)
      #local _tmpData[_A] = <0,0,0>;
      #local _A=_A+1;
    #end
    #declare _Data = _tmpData;
  #end
#end

//ImpacrMacro, DeathMacro and SamplingMacro actually only provide new declarations for internal
// processes to switch their effect on/off and attach them to a macro.
/*ImpactMacro*/
#macro ParticleGroup_FXImpact(_ImpactName)
  #if (strcmp(_ImpactName,"off"))
    ParsysEDK_AddDeclarationValue("tnwTPS_ImpactMacroSwitch",1)
    ParsysEDK_AddDeclarationString("tnwTPS_ImpactMacroName",_ImpactName)
    #declare tnw_ACG_Impact = _ImpactName;
    ParsysEDK_AddLineFeed()
  #end
#end

/*DeathMacro*/
#macro ParticleGroup_FXDeath(_DeathName)
  #if (strcmp(_DeathName,"off"))
    ParsysEDK_AddDeclarationValue("tnwTPS_DeathMacroSwitch",1)
    ParsysEDK_AddDeclarationString("tnwTPS_DeathMacroName",_DeathName)
    #declare tnw_ACG_Death = _DeathName;
    ParsysEDK_AddLineFeed()
  #end
#end

/*SamplingMacro*/
//Consult Help for usage and preparation of Sampling-Macro
#macro ParticleGroup_Sampler(_SamplerName)
  ParsysEDK_AddDeclarationValue("tnwTPS_ParticleSamplingSwitch",1)
  ParsysEDK_AddDeclarationString("tnwTPS_ParticleSamplingMacro",_SamplerName)
  ParsysEDK_AddSDL(concat("#ifndef (",_SamplerName,") #macro ",_SamplerName,"(_d1,_d2,_s1,_s2,_rp,_rd) #declare _rp = _d1[2]; #end #end\n"))
#end

/** Gravity **/
//-Constructor
#macro ParticleGroup_Gravity(_GravityAmount,_GravityDirection)
  #local _GravDir = VNormalize(_GravityDirection);
  #local _GravAm = _GravityAmount;
  #if (_GravityAmount < 0)
    #local _GravDir = -_GravDir;
    #local _GravAm = -_GravAm;
  #end
  ParsysEDK_AddMacro("_VectorParticle_Gravity")
  ParsysEDK_AddParameterValue(_GravAm)
  ParsysEDK_AddParameterValue(_GravDir)
  ParsysEDK_CloseMacro()
  //Values to be accessed via "Parsys.inc"
  ParsysEDK_AddDeclarationValue("tnwTPS_gravAmount",_GravAm)
  ParsysEDK_AddDeclarationValue("tnwTPS_gravDir",_GravDir)
  ParsysEDK_AddLineFeed()
#end
//-Actual Macro
#macro _VectorParticle_Gravity(_Data,_TStep,_Gravity,_Direction)
  #local _tmp_Data = _Data;
  #local _tmp_Data[3] = _tmp_Data[3]+_TStep*_Gravity*VNormalize(_Direction);
  #declare _Data = _tmp_Data;
#end

/** Damping **/
//-Constructor
#macro ParticleGroup_Damping(_DampingAmount)
  ParsysEDK_AddMacro("_VectorParticle_Damping")
  ParsysEDK_AddParameterValue(_DampingAmount)
  ParsysEDK_CloseMacro()
  ParsysEDK_AddLineFeed()
#end
//-Actual Macro
#macro _VectorParticle_Damping(_Data,_TStep,_DampingAmount)
  #local _tmp_Data = _Data;
  #local _tmp_Data[3] = _tmp_Data[3]-VNormalize(_tmp_Data[3])*(1-_DampingAmount)*_TStep;
  #declare _Data = _tmp_Data;
#end

/** Linear Wind **/
//-Constructor
#macro ParticleGroup_LinearWind(_Direction,_Force,_Friction)
  ParsysEDK_AddMacro("_VectorParticle_LinearWind")
  #local _Dir = VNormalize(_Direction);
  #local _For = _Force;
  #if (_For < 0)
    #local _Dir = -_Dir;
    #local _For = -_For;
  #end
  ParsysEDK_AddParameterValue(_Dir)
  ParsysEDK_AddParameterValue(_For)
  ParsysEDK_AddParameterValue(min(max(_Friction,0),1))
  ParsysEDK_CloseMacro()
  ParsysEDK_AddLineFeed()
#end
//-Actual Macro
#macro _VectorParticle_LinearWind(_Data,_TStep,_WindDirection,_WindForce,_WindFriction)
  #local _tmp_Data = _Data;
  #local _alongWind = vdot(_tmp_Data[3],_WindDirection)*_WindDirection;
  #local _Wind = _WindDirection*_WindForce;  
  #local _tmp_Data[3] = _tmp_Data[3]+(_Wind-_tmp_Data[3])*(1-_WindFriction)*_TStep;
  #declare _Data = _tmp_Data;
#end

/** Turbulenced-field Wind **/
//-Constructor
#macro ParticleGroup_TurbWind(_Lambda,_Omega,_Octaves,_Scale,_PosMod,_Force,_Friction)
  #local _Turb = <_Lambda,_Omega,_Octaves>;
  ParsysEDK_AddMacro("_VectorParticle_TurbWind")
  ParsysEDK_AddParameterValue(_Turb)
  ParsysEDK_AddParameterValue(_Scale)
  ParsysEDK_AddParameterValue(_PosMod)
  ParsysEDK_AddParameterValue(_Force)
  ParsysEDK_AddParameterValue(min(max(_Friction,0),1))
  ParsysEDK_CloseMacro()
  ParsysEDK_AddLineFeed()
#end
//-Actual Macro
#macro _VectorParticle_TurbWind(_Data,_TStep,_TSet,_TScl,_TPMod,_TPow,_WindFriction)
  #local _tmp_Data = _Data;
  #local _Vturb =vturbulence(_TSet.x,_TSet.y,_TSet.z,_tmp_Data[2]*_TScl+_TPMod); 
  #local _WindDirection = VNormalize(_Vturb);
  #local _WindForce = vlength(_Vturb)*_TPow;
  #local _alongWind = vdot(_tmp_Data[3],_WindDirection)*_WindDirection;
  #local _Wind = _WindDirection*_WindForce;  
  #local _tmp_Data[3] = _tmp_Data[3]+(_Wind-_tmp_Data[3])*(1-_WindFriction)*_TStep;
  #declare _Data = _tmp_Data;
#end
