
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>


#define BUFLEN	256
#define MAXARG	128
#define MAXCPU	16

typedef struct cpu_struct
{
	char *machine;
	char *dir;
	char *prog;
	char *tmp;
	char *task;
	int cputime;
	int jobs;
	pid_t pid;
	time_t stime,etime;
} CPU;

CPU cpu[MAXCPU];
int ncpu=0;
int acpu=0;

int sframe=(-1),eframe=(-1);

char **av;
int ac;

char *rsh=NULL;
char *sh=NULL;
char *logfile=NULL;
FILE *logfp;

int
isup(char *host)
{
	char buf[64];
	int r;
	sprintf(buf,"/bin/ping -c 1 %s\n",host);
	r=system(buf);
	if(WEXITSTATUS(r)==0)
		return(1);
	return(0);
}

void
read_ini(char *fname)
{
	FILE *fp;
	char buf[BUFLEN],*p;
	int line=0;
	fp=fopen(fname,"r");
	if(fp==NULL)
	{
		perror(fname);
		exit(1);
	}
	while(fgets(buf,BUFLEN-1,fp)!=NULL)
	{
		line++;
		if(*buf=='#')
			continue;
		if(buf[strlen(buf)-1]=='\n');
			buf[strlen(buf)-1]='\0';
		if(*buf=='\0')
			continue;
		p=strchr(buf,'=');
		if(p!=NULL)
		{
			*p++='\0';
			if(strcmp(buf,"Initial_Frame")==0)
			{
				sframe=atoi(p);
				printf("%s:%d start frame=%d\n",fname,line,sframe);
			}
			if(strcmp(buf,"Final_Frame")==0)
			{
				eframe=atoi(p);
				printf("%s:%d end frame=%d\n",fname,line,eframe);
			}
		}
	}
	fflush(stdout);
}

void
read_config(char *fname)
{
	FILE *fp;
	char buf[BUFLEN],*p,*dir=NULL,*prog=NULL;
	int line=0;
	fp=fopen(fname,"r");
	if(fp==NULL)
	{
		perror(fname);
		exit(1);
	}
	dir=strdup(getcwd(buf,BUFLEN));
	while(fgets(buf,BUFLEN-1,fp)!=NULL)
	{
		line++;
		if(*buf=='#')
			continue;
		if(buf[strlen(buf)-1]=='\n');
			buf[strlen(buf)-1]='\0';
		if(*buf=='\0')
			continue;
		p=strchr(buf,'=');
		if(p==NULL)
		{
			fprintf(stderr,"%s: syntax error on line %d %s\n",fname,line,buf);
			exit(1);
		}
		*p++='\0';
		if(strcmp(buf,"dir")==0)
			dir=strdup(p);
		if(strcmp(buf,"rsh")==0)
			rsh=strdup(p);
		if(strcmp(buf,"sh")==0)
			sh=strdup(p);
		if(strcmp(buf,"prog")==0)
			prog=strdup(p);
		if(strcmp(buf,"logfile")==0)
			logfile=strdup(p);
		if(strcmp(buf,"machine")==0)
		{
			if(dir==NULL)
			{
				fprintf(stderr,"%s:%d dir not set\n",fname,line);
				exit(1);
			}
			if(prog==NULL)
			{
				fprintf(stderr,"%s:%d prog not set\n",fname,line);
				exit(1);
			}
			if(rsh==NULL)
			{
				fprintf(stderr,"%s:%d rsh not set\n",fname,line);
				exit(1);
			}
			if(sh==NULL)
			{
				fprintf(stderr,"%s:%d sh not set\n",fname,line);
				exit(1);
			}
			if(ncpu==MAXCPU)
			{
				fprintf(stderr,"%s:%d max number of CPU's reached (%d) recompile with larger MAXCPU\n",fname,line,MAXCPU);
				exit(1);
			}
			cpu[ncpu].machine=strdup(p);
			cpu[ncpu].dir=dir;
			cpu[ncpu].prog=prog;
			cpu[ncpu].tmp=NULL;
			cpu[ncpu].task=NULL;
			cpu[ncpu].pid=0;
			cpu[ncpu].stime=0;
			cpu[ncpu].etime=0;
			cpu[ncpu].cputime=0;
			cpu[ncpu].jobs=0;
			printf("%s:%d cpu %d host=%s dir=%s\n",fname,line,ncpu,p,dir);
			if(isup(cpu[ncpu].machine))
				ncpu++;
			else
				printf("No ping from host %s - removed\n",cpu[ncpu].machine);
		}
	}
	fflush(stdout);
}

void
run_task(int c,int frame)
{
	char buf[BUFLEN];
	int fd,i;
	pid_t cpid;
	char *pv[MAXARG];
	int pc=0;

	sprintf(buf,"/tmp/pdist.%04x.%04d",getpid(),frame);
	cpu[c].tmp=strdup(buf);

	fd=open(cpu[c].tmp,O_WRONLY|O_CREAT,0644);
	if(fd<0)
	{
		perror(cpu[c].tmp);
		exit(1);
	}

	if(strcmp(cpu[c].machine,"localhost")!=0)
	{
		pv[pc++]=rsh;
		pv[pc++]=cpu[c].machine;
	}
	else
	{
		pv[pc++]=sh;
		pv[pc++]="-c";
	}

	strcpy(buf,"cd ");
	strcat(buf,cpu[c].dir);
	strcat(buf,";");
	strcat(buf,cpu[c].prog);
	sprintf(buf+strlen(buf)," +SF%d +EF%d",frame,frame);	

	for(i=2;i<ac;i++)
	{
		strcat(buf," ");
		strcat(buf,av[i]);
	}
	cpu[c].task=strdup(buf);
	pv[pc++]=cpu[c].task;
	pv[pc]=NULL;

	cpu[c].stime=time(0);

	cpid=fork();

	sprintf(buf,"** Starting pid %d (cpu %d) on %s at %s",(cpid==0)?getpid():cpid,c,cpu[c].machine,ctime(&cpu[c].stime));
	sprintf(buf+strlen(buf),"Task=");
	for(i=0;i<pc;i++)
		sprintf(buf+strlen(buf)," %s",pv[i]);
	sprintf(buf+strlen(buf),"\n");

	switch(cpid)
	{
	case 0:	/* child */
		write(fd,buf,strlen(buf));
		close(0);
		close(1);
		close(2);
		dup2(fd,1);
		dup2(fd,2);
		execv(pv[0],pv);
		perror("execv");
		exit(0);
		break;
	case -1:
		perror("fork()");
		exit(1);
		break;
	default:	/* parent */
		cpu[c].pid=cpid;
		fwrite(buf,strlen(buf),1,stdout);
		fflush(stdout);
	}
	close(fd);
}

void
start_task(int frame)
{
	int i;
	if(acpu==ncpu)
	{
		fprintf(stderr,"task_start called with no free cpu's n=%d a=%d\n",ncpu,acpu);
		sleep(60);
		return;
	}
	for(i=0;i<ncpu;i++)
	{
		if(cpu[i].pid==0)
		{
			run_task(i,frame);
			acpu++;
			break;
		}
	}
}

void
end_task(void)
{
	int n,status,len;
	pid_t pid;
	FILE *fp;
	char buf[BUFLEN];

	pid=wait(&status);

	for(n=0;n<ncpu;n++)
	{
		if(pid==cpu[n].pid)
		{
			cpu[n].etime=time(0);
			cpu[n].cputime+=cpu[n].etime-cpu[n].stime;
			printf("** Finish pid %d (cpu %d) (%d seconds) at %s",pid,n,(int)(cpu[n].etime-cpu[n].stime),ctime(&cpu[n].etime));
			if(status!=0)
			{
				printf("==== Finish pid %d (cpu %d) status is 0x%x\n",pid,n,status);
			}
			fflush(stdout);
			fp=fopen(cpu[n].tmp,"r");
			if(fp==NULL)
			{
				perror(cpu[n].tmp);
			}
			else
			{
				do
				{
					len=fread(buf,1,BUFLEN,fp);
					fwrite(buf,1,len,logfp);
					fflush(logfp);
				}
				while(len!=0);
				if(logfp!=stdout)
					fprintf(logfp,"** Finish pid %d (cpu %d) at %s",pid,n,ctime(&cpu[n].etime));
				fclose(fp);
			}
			unlink(cpu[n].tmp);
			free(cpu[n].tmp);
			cpu[n].tmp=NULL;
			free(cpu[n].task);
			cpu[n].pid=0;
			cpu[n].jobs++;
			acpu--;
			break;
		}
	}
	if(n==ncpu)
	{
		fprintf(stderr,"wait() just got unexpected pid %d\n",pid);
	}
}

void
do_tasks(void)
{
	int tasks=(1+eframe-sframe);
	int ntask=sframe;
	do
	{
		printf("tasks=%d ncpu=%d acpu=%d ntask=%d\n",tasks,ncpu,acpu,ntask);
		if((tasks>0)&&(ncpu!=acpu))
		{
			start_task(ntask);
			ntask++;
			tasks--;
		}
		if((ncpu==acpu)||((tasks==0)&&(acpu>0)))
		{
			end_task();
		}
	}
	while((tasks!=0)||(acpu!=0));
}

int
main(int argc, char *argv[])
{
	int i;
	av=argv;
	ac=argc;
	if(argc<4)
	{
		fprintf(stderr,"Usage: %s machine.conf pov.ini pov-parameters\n",argv[0]);
		exit(1);
	}
	read_config(argv[1]);
	if(ncpu==0)
	{
		fprintf(stderr,"%s: No cpu's defined in %s\n",argv[0],argv[1]);
		exit(1);
	}
	read_ini(argv[2]);
	if((sframe==(-1))||(eframe==(-1))||(sframe>eframe))
	{
		fprintf(stderr,"%s: start and/or end not defined in %s\n",argv[0],argv[2]);
		exit(1);
	}
	if(logfile!=NULL)
	{
		logfp=fopen(logfile,"w");
		if(logfp==NULL)
		{
			perror(logfile);
			exit(1);
		}
	}
	else
		logfp=stdout;
	do_tasks();
	for(i=0;i<ncpu;i++)
	{
		if(logfp!=stdout)
			fprintf(logfp,"%s cpu=%ds jobs=%d (%fs/job)\n",cpu[i].machine,cpu[i].cputime,cpu[i].jobs,(float)cpu[i].cputime/(float)cpu[i].jobs);
		printf("%s cpu=%ds jobs=%d (%fs/job)\n",cpu[i].machine,cpu[i].cputime,cpu[i].jobs,(float)cpu[i].cputime/(float)cpu[i].jobs);
	}
	exit(0);
}
