import Numeric, LinearAlgebra, os, sys, random
from OpenGL.quaternion import quaternion

def q_to_m(q):
	return Numeric.array([
		[1-2*q.j*q.j-2*q.k*q.k, 2*q.i*q.j-2*q.real*q.k, 2*q.k*q.i+2*q.real*q.j],
		[2*q.i*q.j+2*q.real*q.k, 1-2*q.i*q.i-2*q.k*q.k, 2*q.j*q.k-2*q.real*q.i],
		[2*q.k*q.i-2*q.real*q.j, 2*q.j*q.k+2*q.real*q.i, 1-2*q.i*q.i-2*q.j*q.j]
	])

class State:
	def __init__(self):
		pass
	def __getattr__(self, key):
		try:
			return self.__dict__[key]
		except KeyError:
			return quaternion(1)
	def __repr__(self):
		return repr(self.__dict__)
	def __str__(self):
		return repr(self.__dict__)
	def copy(self):
		tmpstate=State()
		for k,v in self.__dict__.items():
			tmpstate.__dict__[k]=v
		return tmpstate
	def normalize(self):
		for k,v in self.__dict__.items():
			s=v.i*v.i+v.j*v.j+v.k*v.k
			if s>=1:
				self.__dict__[k]=v/abs(v)
			else:
				v.real=(1-s)**0.5

def interpolate(s1, s2, alpha):
	s=State()
	for k in s1.__dict__.keys()+s2.__dict__.keys():
		v=getattr(s1, k)*alpha+getattr(s2, k)*(1-alpha)
		v=v/abs(v)
		setattr(s, k, v)
	return s
	
def vtransform(q, v):
	return Numeric.matrixmultiply(q_to_m(q), v)

class Model:
	_CENTER=Numeric.array([0,3,0])
	_CJOINT=Numeric.array([1.4,0,0])
	_UPPER=Numeric.array([1,0,0])
	_LL11=Numeric.array([0,-0.5,1])
	_LL12=Numeric.array([2,1.5,2])
	_LL13=Numeric.array([1,-3,1])
	_LL14=Numeric.array([2,-1,1])
	_RL11=Numeric.array([0,-0.5,-1])
	_RL12=Numeric.array([2,1.5,-2])
	_RL13=Numeric.array([1,-3,-1])
	_RL14=Numeric.array([2,-1,-1])
	
	_LL21=Numeric.array([0.8,0.2,1])
	_LL22=Numeric.array([0.2,0.8,2.5])
	_LL23=Numeric.array([0.5,-3.6,1])
	_LL24=Numeric.array([0.5,-0.4,1.5])
	_RL21=Numeric.array([0.8,0.2,-1])
	_RL22=Numeric.array([0.2,0.8,-2.5])
	_RL23=Numeric.array([0.5,-3.6,-1])
	_RL24=Numeric.array([0.5,-0.4,-1.5])
	
	_LL31=Numeric.array([-0.8,-0.4,1])
	_LL32=Numeric.array([-1.2,1.4,2])
	_LL33=Numeric.array([-1,-3,1])
	_LL34=Numeric.array([-2,-1,2])
	_RL31=Numeric.array([-0.8,-0.4,-1])
	_RL32=Numeric.array([-1.2,1.4,-2])
	_RL33=Numeric.array([-1,-3,-1])
	_RL34=Numeric.array([-2,-1,-2])
	def __init__(self):
		self.transform(State())
	def transform(self, state):
		self.CENTER=vtransform(state.T_ROT, self._CENTER)
		self.LL21=vtransform(state.T_ROT*state.T_LL21, self._LL21)+self.CENTER
		self.LL22=vtransform(state.T_ROT*state.T_LL21*state.T_LL22,
			self._LL22)+self.LL21
		self.LL23=vtransform(state.T_ROT*state.T_LL21*state.T_LL22*state.T_LL23,
			self._LL23)+self.LL22
		self.LL24=vtransform(state.T_ROT*state.T_LL21*state.T_LL22*state.T_LL23*state.T_LL24,
			self._LL24)+self.LL23
			
		self.RL21=vtransform(state.T_ROT*state.T_RL21, self._RL21)+self.CENTER
		self.RL22=vtransform(state.T_ROT*state.T_RL21*state.T_RL22,
			self._RL22)+self.RL21
		self.RL23=vtransform(state.T_ROT*state.T_RL21*state.T_RL22*state.T_RL23,
			self._RL23)+self.RL22
		self.RL24=vtransform(state.T_ROT*state.T_RL21*state.T_RL22*state.T_RL23*state.T_RL24,
			self._RL24)+self.RL23
		
		self.LL31=vtransform(state.T_ROT*state.T_LL31, self._LL31)+self.CENTER
		self.LL32=vtransform(state.T_ROT*state.T_LL31*state.T_LL32,
			self._LL32)+self.LL31
		self.LL33=vtransform(state.T_ROT*state.T_LL31*state.T_LL32*state.T_LL33,
			self._LL33)+self.LL32
		self.LL34=vtransform(state.T_ROT*state.T_LL31*state.T_LL32*state.T_LL33*state.T_LL34,
			self._LL34)+self.LL33
		self.RL31=vtransform(state.T_ROT*state.T_RL31, self._RL31)+self.CENTER
		self.RL32=vtransform(state.T_ROT*state.T_RL31*state.T_RL32,
			self._RL32)+self.RL31
		self.RL33=vtransform(state.T_ROT*state.T_RL31*state.T_RL32*state.T_RL33,
			self._RL33)+self.RL32
		self.RL34=vtransform(state.T_ROT*state.T_RL31*state.T_RL32*state.T_RL33*state.T_RL34,
			self._RL34)+self.RL33
			
		self.CJOINT=vtransform(state.T_ROT, self._CJOINT)+self._CENTER
		self.UPPER=vtransform(state.T_ROT*state.T_UPPER, self._UPPER)+self.CJOINT
		self.LL11=vtransform(state.T_ROT*state.T_UPPER*state.T_LL11, self._LL11)+self.UPPER
		self.LL12=vtransform(state.T_ROT*state.T_UPPER*state.T_LL11*state.T_LL12,
			self._LL12)+self.LL11
		self.LL13=vtransform(state.T_ROT*state.T_UPPER*state.T_LL11*state.T_LL12*state.T_LL13,
			self._LL13)+self.LL12
		self.LL14=vtransform(state.T_ROT*state.T_UPPER*state.T_LL11*state.T_LL12*state.T_LL13*state.T_LL14,
			self._LL14)+self.LL13
		self.RL11=vtransform(state.T_ROT*state.T_UPPER*state.T_RL11, self._RL11)+self.UPPER
		self.RL12=vtransform(state.T_ROT*state.T_UPPER*state.T_RL11*state.T_RL12,
			self._RL12)+self.RL11
		self.RL13=vtransform(state.T_ROT*state.T_UPPER*state.T_RL11*state.T_RL12*state.T_RL13,
			self._RL13)+self.RL12
		self.RL14=vtransform(state.T_ROT*state.T_UPPER*state.T_RL11*state.T_RL12*state.T_RL13*state.T_RL14,
			self._RL14)+self.RL13
			

def get_target_vector(model, targets):
	tlist=[]
	for t in targets:
		p=getattr(model, t)
		tlist=tlist+list(p)
	return Numeric.array(tlist)

def find_improvement(model, state, variables, targets, d, tval, start_residual):
	opt_residual=start_residual
	opt_scale=0
	
	scale=1.0
	for n in range(0, 20):
		tmp=state.copy()
		index=0
		for var in variables:
			setattr(tmp, var, getattr(tmp, var)+quaternion(0,
				d[index+0], d[index+1], d[index+2])*scale)
			index=index+3
		tmp.normalize()
		model.transform(tmp)
		v=get_target_vector(model, targets)
		residual=Numeric.matrixmultiply(tval-v, tval-v)
		if residual<opt_residual:
			opt_residual=residual
			opt_scale=scale
		scale=scale/2
	if opt_residual>=start_residual: return False
	index=0
	for var in variables:
		setattr(state, var, getattr(state, var)+quaternion(0,
			d[index+0], d[index+1], d[index+2])*opt_scale)
		index=index+3
	state.normalize()
	model.transform(state)
	return True
	
def inv_step(model, state, variables, targets, tval):
	model.transform(state)
	orig=get_target_vector(model, targets)
	
	delta=1e-3
	
	cols=[]
	for var in variables:
		tmp=state.copy()
		setattr(tmp, var, getattr(tmp, var)+quaternion(0,delta,0,0))
		model.transform(tmp)
		cols.append(get_target_vector(model, targets))
		
		tmp=state.copy()
		setattr(tmp, var, getattr(tmp, var)+quaternion(0,0,delta,0))
		model.transform(tmp)
		cols.append(get_target_vector(model, targets))
		
		tmp=state.copy()
		setattr(tmp, var, getattr(tmp, var)+quaternion(0,0,0,delta))
		model.transform(tmp)
		cols.append(get_target_vector(model, targets))
	
	
	M=Numeric.transpose(Numeric.array(cols))/delta
	b=tval-orig
	lsq, residual, rank, sv=LinearAlgebra.linear_least_squares(M, b)
	m1=max(lsq)
	m2=min(lsq)
	if m1<0: m=-m2
	elif m2>0: m=m1
	else: m=max(abs(m1), abs(m2))
	if m<1: m=1
	lsq=lsq/m*2
	
	residual=Numeric.matrixmultiply(tval-orig, tval-orig)
	if find_improvement(model, state, variables, targets, lsq, tval, residual):
		return True
	for n in xrange(0, 30):
		tmp=[2*random.random()-1 for x in xrange(0, len(variables)*3)]
		if find_improvement(model, state, variables, targets, tmp, tval, residual):
			return True
	return False
		
def inv_kinematics(model, state, variables, targets, tval, tolerance=0.1):
	while True:
		model.transform(state)
		p=get_target_vector(model, targets)
		residual=Numeric.matrixmultiply(p-tval, p-tval)
		if residual<tolerance: break
		sys.stderr.write("%f\r" % residual)
		inv_step(model, state, variables, targets, tval)

def serialize_pose(state):
	p=""
	for k,v in state.__dict__.items():
		p=p+"#declare %s=transform{QUATERNION(%f, %f, %f, %f)}\n" \
			% (k, v.real, v.i, v.j, v.k)
	return p

def write_pose(state, filename):
	f=file(filename, "w")
	f.write(serialize_pose(state))
	f.close()

m=Model()

tolerance=0.1
# front leg
LL1=[None]*13
st=State()
for n in range(0, 4):
	tval=Numeric.array([7.4-n*0.8,0,5])
	st=st.copy()
	st.T_UPPER=quaternion((1-0.005*n)**0.5,0,-(0.005*n)**0.5,0)
	st.T_HEAD=quaternion((1-0.0025*n)**0.5,0,(0.0025*n)**0.5,0)
	st.T_LOWER=quaternion((1-0.0025*n)**0.5,0,-(0.0025*n)**0.5,0)
	m.transform(st)
	inv_kinematics(m, st, [
		"T_LL12", "T_LL13", "T_LL14",
		],
		["LL14"], tval, tolerance)
	LL1[n]=st
	print n, "***"

st=State()
for n in range(0, 4):
	tval=Numeric.array([7.4+n*0.8,0,5])
	st=st.copy()
	st.T_UPPER=quaternion((1-0.005*n)**0.5,0,(0.005*n)**0.5,0)
	st.T_HEAD=quaternion((1-0.0025*n)**0.5,0,-(0.0025*n)**0.5,0)
	st.T_LOWER=quaternion((1-0.0025*n)**0.5,0,(0.0025*n)**0.5,0)
	m.transform(st)
	inv_kinematics(m, st, [
		"T_LL12", "T_LL13", "T_LL14",
		],
		["LL14"], tval, tolerance)
	LL1[12-n]=st
	print 12-n, "***"


s=State()
s.T_LL12=quaternion(0.9**0.5,-0.1**0.5,0,0)
LL1[6]=s

for n in range(1, 3):
	LL1[6+n]=interpolate(LL1[9], LL1[6], n/3.0)
	LL1[6+n].T_UPPER=quaternion((1-0.005*n)**0.5,0,(0.005*n)**0.5,0)
	LL1[6+n].T_HEAD=quaternion((1-0.0025*n)**0.5,0,-(0.0025*n)**0.5,0)
	LL1[6+n].T_LOWER=quaternion((1-0.0025*n)**0.5,0,(0.0025*n)**0.5,0)
	LL1[6-n]=interpolate(LL1[3], LL1[6], n/3.0)
	LL1[6-n].T_UPPER=quaternion((1-0.005*n)**0.5,0,-(0.005*n)**0.5,0)
	LL1[6-n].T_HEAD=quaternion((1-0.0025*n)**0.5,0,(0.0025*n)**0.5,0)
	LL1[6-n].T_LOWER=quaternion((1-0.0025*n)**0.5,0,-(0.0025*n)**0.5,0)

# middle leg
LL2=[None]*13
st=State()
for n in range(0, 4):
	tval=Numeric.array([2-n*0.8,0,6])
	st=st.copy()
	m.transform(st)
	inv_kinematics(m, st, [
		"T_LL22", "T_LL23", "T_LL24",
		],
		["LL24"], tval, tolerance)
	LL2[6+n]=st
	print n, "***"

st=State()
for n in range(0, 4):
	tval=Numeric.array([2+n*0.8,0,6])
	st=st.copy()
	m.transform(st)
	inv_kinematics(m, st, [
		"T_LL22", "T_LL23", "T_LL24",
		],
		["LL24"], tval, tolerance)
	LL2[6-n]=st
	print 12-n, "***"


s=State()
s.T_LL22=quaternion(0.9**0.5,-0.1**0.5,0,0)
LL2[0]=s

for n in range(1, 3):
	LL2[n]=interpolate(LL2[3], LL2[0], n/3.0)
	LL2[12-n]=interpolate(LL2[9], LL2[0], n/3.0)

# rear leg
LL3=[None]*13
st=State()
for n in range(0, 4):
	tval=Numeric.array([-5-n*0.8,0,6])
	st=st.copy()
	m.transform(st)
	inv_kinematics(m, st, [
		"T_LL32", "T_LL33", "T_LL34",
		],
		["LL34"], tval, tolerance)
	LL3[n]=st
	print n, "***"

st=State()
for n in range(0, 4):
	tval=Numeric.array([-5+n*0.8,0,6])
	st=st.copy()
	m.transform(st)
	inv_kinematics(m, st, [
		"T_LL32", "T_LL33", "T_LL34",
		],
		["LL34"], tval, tolerance)
	LL3[12-n]=st
	print 12-n, "***"


s=State()
s.T_LL32=quaternion(0.9**0.5,-0.1**0.5,0,0)
LL3[6]=s

for n in range(1, 3):
	LL3[6+n]=interpolate(LL3[9], LL3[6], n/3.0)
	LL3[6-n]=interpolate(LL3[3], LL3[6], n/3.0)

# now put it all together
for n in range(0, 12):
	d=(n+6)%12
	s=State()
	s.T_UPPER=LL1[n].T_UPPER
	s.T_HEAD=LL1[n].T_HEAD
	s.T_LOWER=LL1[n].T_LOWER
	s.T_LL12=LL1[n].T_LL12
	s.T_LL13=LL1[n].T_LL13
	s.T_LL14=LL1[n].T_LL14
	s.T_RL12=LL1[d].T_LL12
	s.T_RL13=LL1[d].T_LL13
	s.T_RL14=LL1[d].T_LL14
	s.T_LL22=LL2[n].T_LL22
	s.T_LL23=LL2[n].T_LL23
	s.T_LL24=LL2[n].T_LL24
	s.T_RL22=LL2[d].T_LL22
	s.T_RL23=LL2[d].T_LL23
	s.T_RL24=LL2[d].T_LL24
	s.T_LL32=LL3[n].T_LL32
	s.T_LL33=LL3[n].T_LL33
	s.T_LL34=LL3[n].T_LL34
	s.T_RL32=LL3[d].T_LL32
	s.T_RL33=LL3[d].T_LL33
	s.T_RL34=LL3[d].T_LL34
	write_pose(s, "antpose%d.inc" % n)
