// POV-Ray 3.0 scene file by Ian Armstrong   -  May 1996
//
// Subject: home-built electric clock
// Entry for IRTC competition May/June 1996
//
// (c)1996 Ian Armstrong (iarmstrong@cix.compulink.co.uk)
//
#include "colors.inc"
#include "textures.inc"
#include "shapes.inc"
#include "metals.inc"
#include "glass.inc"
#include "woods.inc"
#include "stones.inc"

#declare Area_Lights = 1
#declare Full_Scenery = 1


/* ==== Cameras ==== */

#declare Camera1 =    // Main view
camera {
//   location <0, 9, -28>
   location <0, 10, -34>
   direction <0, 0,  2.25>
   right x*1.33
   look_at <0,4.4,0>
}
#declare Camera2 =    // Close-up views
camera {
   location <0, 2, -35>
   direction <0, 0,  5.25>
   right x*1.33
   look_at <1,0,0>
}
#declare Camera3 =    // Top view
camera {
   location <0, 25, -8>
   direction <0, 0,  2.25>
   right x*1.33
   look_at <0,3,0>
}
camera { Camera1 }

/* ==== Lights ==== */

#declare Dist=80.0
light_source {< -30, 30, -50> color White		// Left Side Light
   fade_distance Dist fade_power 2
#ifdef (Area_Lights)
   area_light <0, 0, 20>, <0, 20, 0>, 3, 3
   adaptive 1
   jitter
#end
}
light_source {< 50, 10,  -40> color Gray60		// Right Side Light
   fade_distance Dist fade_power 2
#ifdef (Area_Lights)
   area_light <0, 0, 10>, <0, 10, 0>, 2, 2
   adaptive 1
   jitter
#end
}
light_source {< -12, 100,  0> color Gray40		// Top Light
   fade_distance Dist fade_power 2
}
/* ==== Textures & Colours ==== */

#declare Bakelite = color red 0.08 green 0.04 blue 0.035

#declare Board_Texture =
texture { Cork }

#declare Board_Edge_Texture =
  texture { T_Wood30 rotate 90*y scale <10,3,3> }

#declare Brick_Texture =
texture {
    pigment {
        brick Gray65, Wheat*0.7
    }
    normal {
        brick normal{granite 0.5}, normal{granite 0.5} }
    finish {
        crand 0.003
        diffuse 0.6 ambient 0.20
    }
    scale <0.6, 0.6, 0.3>
    translate 1*x
}

#declare Floor_Texture =
    texture {T_Wood7 finish { specular 0.15 roughness 0.025 ambient 0.25 reflection 0.15}}

#declare Box_Texture =
    texture {T_Wood35 finish { specular 0.15 roughness 0.025 ambient 0.25 reflection 0.05}
        rotate y*90 rotate <10, 0, 15> scale <2,1,1> translate -3*x
    }

/* ==== Scenery ==== */
sky_sphere {
    pigment {
        gradient y
        color_map {
            [0, 1  color Gray70 color Gray90]
        }
    }
}

#declare Paper1 =
union {
  box { <0,0,-0.01>,<4,8,0>
    texture { pigment { White } finish { Dull ambient 0.2 } }
    texture {
      pigment {image_map { <1, -1, 0> gif "t_gears1.gif" once interpolate 2 /* filter 1,1 */ } scale <4,8, 1> }
      finish { ambient 0.7 }
    }
  }
  sphere { 0,0.15 scale <1,1,0.2> translate <0.25,0.25,-0.11> pigment { Red } finish { Shiny } }
  sphere { 0,0.15 scale <1,1,0.2> translate <3.75,0.25,-0.11> pigment { Red } finish { Shiny } }
  sphere { 0,0.15 scale <1,1,0.2> translate <0.25,7.75,-0.11> pigment { Red } finish { Shiny } }
  sphere { 0,0.15 scale <1,1,0.2> translate <3.75,7.75,-0.11> pigment { Red } finish { Shiny } }
}
#declare Paper2 =
union {
  box { <0,0,-0.01>,<6,4,0>
    texture { pigment { White } finish { Dull ambient 0.15 } }
    texture {
      pigment {image_map { <1, -1, 0> gif "t_cct1.gif" once interpolate 2 filter 1,1 } scale <6,4, 1> }
      finish { ambient 0.99 }
    }
  }
  sphere { 0,0.15 scale <1,1,0.2> translate <0.25,0.25,-0.11> pigment { Blue } finish { Shiny } }
  sphere { 0,0.15 scale <1,1,0.2> translate <5.75,0.25,-0.11> pigment { Blue } finish { Shiny } }
  sphere { 0,0.15 scale <1,1,0.2> translate <0.25,3.75,-0.11> pigment { Blue } finish { Shiny } }
  sphere { 0,0.15 scale <1,1,0.2> translate <5.75,3.75,-0.11> pigment { Blue } finish { Shiny } }
}
#declare Board_Edge_Top =
intersection {
  superellipsoid { <0.3, 0.3> scale <100,0.5,0.5> }
  plane { <1,1,0>,0 translate 24*x }
  plane { <-1,1,0>,0 translate -24*x }
  bounded_by { box { <-25,-1,-1>,<25,1,1> } }
}
#declare Board_Edge_Side =
intersection {
  superellipsoid { <0.3, 0.3> scale <100,0.5,0.5> }
  plane { <1,1,0>,0 translate 18*x }
  plane { <-1,1,0>,0 translate -18*x }
  bounded_by { box { <-19,-1,-1>,<19,1,1> } }
}
#declare Board =
union {
  box { <0,0,-0.5>,<48,36,0> texture { Board_Texture } }
  object { Board_Edge_Top scale <1,1,1> texture { Board_Edge_Texture } translate <24,0,-0.5> }
  object { Board_Edge_Top scale <1,-1,1> texture { Board_Edge_Texture } translate <24,36,-0.5> }
  object { Board_Edge_Side scale <1,1,1> texture { Board_Edge_Texture } rotate 90*z translate <48,18,-0.5> }
  object { Board_Edge_Side scale <1,-1,1> texture { Board_Edge_Texture } rotate -90*z translate <0,18,-0.5> }
  object { Paper1 translate <27,1,-0.52> }
  object { Paper2 translate <36,1,-0.52> }
  sphere { 0,0.15 pigment { Red } finish { Shiny } scale <1,1,0.2> translate <46,1,-0.6> }
  sphere { 0,0.15 pigment { Green } finish { Shiny } scale <1,1,0.2> translate <47,2,-0.6> }
  sphere { 0,0.15 pigment { Blue } finish { Shiny } scale <1,1,0.2> translate <46.5,3,-0.6> }
  sphere { 0,0.15 pigment { Yellow } finish { Shiny } scale <1,1,0.2> translate <47,1.2,-0.6> }
}

#declare Walls =
union {
#ifdef (Full_Scenery)
  object { plane {-z,-7.5} texture { Brick_Texture }}   // Back (main) wall
  object { Board translate <-38,5,7.5> }
#else
  object { plane {-z,-7.5} pigment { Wheat } finish { Dull }}
#end
}
#declare Floor =
intersection {
  plane { y,0
    texture {
#ifdef (Full_Scenery)
      Floor_Texture
      rotate y*90
      rotate <10, 0, 15>
      translate -2*x
#else
      pigment { White } finish { Dull }
#end
    }
  }
  plane { z,7 }
  plane { -z,10 }
}

/* ==== Tools ==== */

#declare Screwdriver =
union {
  union {	// Handle
    intersection {
      box { <0, -0.25, -0.25>,<1, 0.25, 0.25> }
      box { <0, -0.25, -0.25>,<1, 0.25, 0.25> rotate 45*x}
    }
    sphere { <0.1, 0, 0>, 0.24 }
    cone { <1.0, 0, 0>, 0.25 <2.25, 0, 0>,0.18 }
    cylinder { <2.25, 0, 0>, <2.3, 0, 0>, 0.22 }
    pigment { DarkOliveGreen } finish { Dull }
  }
  difference {	// Shaft
    cylinder { <2.3, 0, 0>, <4.6, 0, 0>, 0.05 }
    plane { z,0 rotate -5.7*y translate 4.6*x }
    plane { -z,0 rotate 5.7*y translate 4.6*x }
    texture { Chrome_Metal }
  }
  translate <-1,0.25,0> rotate -1.375*z    // lie flat on the table
}

#declare Smoke =      // smoke from Soldering Iron tip
sphere { <0,0,0>,1
  pigment { color rgbt <1, 1, 1, 1> }
  halo {
    attenuating
    spherical_mapping
    linear
    turbulence 0.75
    color_map {
      [ 0 color rgbt <0.8, 0.8, 0.8, 1> ]
      [ 1 color rgbt <1, 1, 1, 0.2> ]
    }
    samples 10
    scale 0.75
  }
  hollow
  scale <1.5,15,1.5>
  translate 8*y
}
#declare S_I_Tex =    // Burnt tip for Soldering Iron
texture {
  gradient x
  texture_map {
    [ 0 Chrome_Metal ]
    [ 0.3 Chrome_Metal ]
    [ 0.7 Rusty_Iron ]
  }
}

#declare Soldering_Iron =
union {
  cone { <-4,0,0>,0.15 <-2.75,0,0>,0.25 }         // Handle rear
  cylinder { <-2.75,0,0>,<0,0,0>,0.25 }
  cylinder { <-2.75,0,0>,<0,0,0>,0.05 translate 0.25*y rotate 0*x }   // Handle
  cylinder { <-2.75,0,0>,<0,0,0>,0.05 translate 0.25*y rotate 60*x }
  cylinder { <-2.75,0,0>,<0,0,0>,0.05 translate 0.25*y rotate 120*x }
  cylinder { <-2.75,0,0>,<0,0,0>,0.05 translate 0.25*y rotate 180*x }
  cylinder { <-2.75,0,0>,<0,0,0>,0.05 translate 0.25*y rotate 240*x }
  cylinder { <-2.75,0,0>,<0,0,0>,0.05 translate 0.25*y rotate 300*x }
  cylinder { <-2.75,0,0>,<-2.5,0,0>,0.3 }
  torus { 0.25,0.075 rotate 90*z }                // Handle front
  cylinder { <0,0,0>,<0.1,0,0>,0.15 }
  union {                                         // bit
    difference {
      cylinder { <0,0,0>,<3,0,0>,0.05 }
      plane { -y,0 rotate -45*z translate 2.75*x }
    }
    cylinder { <0.9,0,0>,<2.4,0,0>,0.1 }
    cone { <2.4,0,0>,0.1 <2.5,0,0>,0.05 }
    texture { S_I_Tex scale <3,1,1> } 
  }
  #declare S0 = 0
  #declare S1 = 0.3
  #declare Z0 = 0
  union {                                         // lead
    #while (S1 < 10)
      #declare Z1 = -sin(S1*2*pi/10)*0.5
      cylinder { <-S0,0,Z0>,<-S1,0,Z1>,0.1 }
      #declare S0 = S1
      #declare Z0 = Z1
      #declare S1 = S1 + 0.3
    #end
    pigment { White } finish { Dull ambient 0.2 }
    translate -4*x
  }
  object { Smoke scale 0.3 translate <2.75,0,0> }
  pigment { Bakelite } finish { Dull }
  translate 0.26*y
}

#declare Solder =
union {
  difference {
    union {
      cylinder { <0,0,0>,<0,0.1,0>,0.75 }
      cylinder { <0,1.9,0>,<0,2,0>,0.75 }
      cylinder { <0,0,0>,<0,2,0>,0.55 }
    }
    cylinder { <0,-0.11,0>,<0,2.1,0>,0.4 }
    pigment { DkGreenCopper } finish { Dull }
  }
  #declare S = 0.15
  #while (S < 1.9)
    torus { 0.6,0.05 translate S*y }
    #declare S = S + 0.1
  #end
  cylinder { <0,1.75,-0.6>,<0.3,1.75,-0.6>,0.05 }
  sphere { <0.3,1.75,-0.6>,0.05 }
  cylinder { <0.3,1.75,-0.6>,<0.6,1.75,-0.5>,0.05 }
  sphere { <0.6,1.75,-0.5>,0.05 }
  cylinder { <0.6,1.75,-0.5>,<1,1.75,-0.3>,0.05 }
  sphere { <1,1.75,-0.3>,0.05 }
  texture { T_Silver_4A }
}

/* ==== Mechanics ==== */

#declare RoundScrewHead =
difference {
  sphere { 0,1 scale <1,0.7,1> }
  plane { y,0 }
  box { <-1.1,0.3,-0.15>,<1.1,1.1,0.15> }
  texture { T_Brass_5A }
  bounded_by { sphere { 0,1.1 } }
}
#declare BlockScrewHead =
difference {
  superellipsoid { <1, 0.2> rotate 90*x scale <1,0.7,1> }
  plane { y,0 }
  box { <-1.1,0.3,-0.15>,<1.1,1.1,0.15> }
  texture { T_Brass_5A }
  bounded_by { sphere { 0,1.1 } }
}
#declare BlockScrewHead1 = object { BlockScrewHead rotate 37*y }
#declare BlockScrewHead2 = object { BlockScrewHead rotate 65*y }

#declare Knob =
union {		// This is a knurled knob to fix the cables on to
  intersection {
    cylinder { <0,0,-0.15>,<0,0,-0.3>,0.3 }
    #declare S = 0
    #while (S < 2*pi)
      cylinder { <0,0,1>,<0,0,-1>,0.1 translate 0.33*x rotate degrees(S)*z inverse }
      #declare S = S + pi/4
    #end
    texture { T_Brass_5A }
  }
  cylinder { <0,0,0>,<0,0,-0.35>,0.1 texture { T_Silver_3B }}
}

#declare Bolt =     // along X-axis, origin at shoulder
union {             // length = 1(head)+3(shank), head diam=2, shank diam=1.1
  intersection {
    box { <-1,-1,-1>,<0,1,1> }
    box { <-1,-1,-1>,<0,1,1> rotate 45*x}
    sphere { <1.5,0,0>,2.6 }
    }
  #declare S = 0
  #while (S < 2)
    sphere { <0,0,0>,0.55 scale <0.2, 1, 1> translate S*x }
    #declare S = S + 0.15
  #end
  texture { T_Brass_5A }
}

#declare Control =
intersection {
  union {
    torus { 1.35, 0.15 translate 0.15*y }
    cylinder { <0,0,0> <0,0.3,0>,1.35 }
    cone { <0,0.3,0>,1.0 <0,1.5,0>,0.8 }
    torus { 0.65, 0.15 translate 1.5*y }
  }
  #declare S = 0
  #while (S < 360)
    sphere { <1.0,0.5,0>,0.2 rotate S*y inverse }
    cylinder { <1.0,0.5,0>,<0.8,1.7,0>,0.2 rotate S*y inverse }
    #declare S = S + 36
  #end
  pigment { Bakelite } finish { Dull }
  scale 0.45
}

#declare Gear_Hub =
difference {
  cylinder { <0,0,0>,<0.3,0,0>,0.25 }
  cylinder { <0.15,-0.5,0>,<0.15,0.5,0>,0.05 }
  texture { T_Brass_4C }
  bounded_by { cylinder { <-.01,0,0>,<0.31,0,0>,0.3 }}
}

#declare Gear36 =
#declare N = 36         // no. of teeth
union {
  intersection {
#include "gear1.inc"
    #declare N = 7      // no. of lightening holes
    #declare H = 0.15   // lightening hole radius
#include "gear2.inc"
    bounded_by { cylinder { <-0.15,0,0>,<0.15,0,0>,1.1*R }}
  }
  object { Gear_Hub translate 0.05*x }
  texture { T_Brass_4C }
}

#declare Gear39 =
#declare N = 39         // no. of teeth
union {
  intersection {
#include "gear1.inc"
    #declare N = 7      // no. of lightening holes
    #declare H = 0.2   // lightening hole radius
#include "gear2.inc"
    bounded_by { cylinder { <-0.15,0,0>,<0.15,0,0>,1.1*R }}
  }
  object { Gear_Hub translate 0.05*x }
  texture { T_Brass_4C }
}

#declare Gear12 =
#declare N = 12         // no. of teeth
union {
  intersection {
#include "gear1.inc"
    scale <3,1,1>
    bounded_by { cylinder { <-0.35,0,0>,<0.35,0,0>,1.1*R }}
  }
  object { Gear_Hub translate 0.3*x }
  texture { T_Brass_4C }
}

#declare Gear9 =
#declare N = 9          // no. of teeth
union {
  intersection {
#include "gear1.inc"
    scale <2,1,1>
    bounded_by { cylinder { <-0.25,0,0>,<0.25,0,0>,1.1*R }}
  }
  object { Gear_Hub translate 0.2*x }
  texture { T_Brass_4C }
}
#declare Minute_Hand =
union {
  cylinder { <0,0,-0.1>,<0,0,0.1>,0.2 }
  cone { <0,0,0>,0.2 <0.4,0,0>,0.075 scale <1,1,0.2> }
  cylinder { <0,0,0>,<2.7,0,0>,0.075 scale <1,1,0.2> }
  cone { <2.5,0,0>,0.2 <3,0,0>,0 scale <1,1,0.2> }
  
  texture { T_Silver_3B }
}
#declare Hour_Hand =
union {
  cylinder { <0,0,-0.1>,<0,0,0.1>,0.25 }
  cone { <0,0,0>,0.25 <0.4,0,0>,0.1 scale <1,1,0.2> }
  cylinder { <0,0,0>,<1.7,0,0>,0.1 scale <1,1,0.2> }
  cone { <1.7,0,0>,0.2 <2,0,0>,0 scale <1,1,0.2> }
  texture { T_Silver_3B }
}
#declare Spindle =
cylinder { <0,0,0>,<-2.5,0,0>,0.1 texture { Chrome_Metal } }

#declare Gear_Train =
union {
  object { Spindle }
  object { Spindle translate 1.5*x }
  object { Gear12 translate -2*x }
  object { Spindle translate 1.33333*y }
  object { Gear36 translate <-2,1.33333,0> }
  object { Gear12 translate <-1,1.33333,0> }
  object { Spindle translate 2.66666*y }
  object { Gear36 translate <-1,2.66666,0> }
  object { Gear12 translate <-1.5,2.66666,0> }
  object { Spindle translate 4*y }
  object { Gear36 translate <-1.5,4,0> }
  object { Gear12 translate <-2,4,0> }
  object { Spindle translate 5.33333*y }
  object { Spindle translate <2,5.33333,0> }        // Hands go here
  object { Minute_Hand rotate -90*y rotate 60*x translate <1.4,5.33333,0> }
  object { Hour_Hand rotate -90*y rotate 12.5*x translate <1.0,5.33333,0> }
  object { Gear36 translate <-2,5.33333,0> }
  object { Gear39 scale <-1,1,1> translate <0.5,5.33333,0> }
  object { Gear12 translate <-0.7,5.33333,0> }
  object { Spindle translate 6.66666*y }
  object { Spindle translate <1,6.66666,0> }        // Hour Drive gear goes here
  object { Gear36 translate <-0.7,6.66666,0> }
  object { Gear9 rotate 20*x scale <-1,1,1> translate <0.5,6.66666,0> }
}


/* ==== Electronics ==== */

#declare VeroBoardBit =
difference {
  box { <0,0,0>,<0.5,0.05,0.5> }
  #declare X=0
  #while (X <= 0.5)
    #declare Z=0
    #while (Z <= 0.5)
      cylinder {<0,-1,0>,<0,1,0>,0.025 translate <X,0,Z> }
      #declare Z = Z + 0.1
    #end
    #declare X = X + 0.1
  #end
  pigment { Tan } finish { Dull }
  bounded_by { box { <-0.01,-0.01,-0.01>,<0.51,0.06,0.51> }}
}
#declare VeroBoardPiece =
union {
  object { VeroBoardBit }
  object { VeroBoardBit translate 0.5*z }
  object { VeroBoardBit translate 1.0*z }
  object { VeroBoardBit translate 1.5*z }
}
#declare VeroBoard =
union {
  object { VeroBoardPiece }
  object { VeroBoardPiece translate 0.5*x }
  object { VeroBoardPiece translate 1.0*x }
  object { VeroBoardPiece translate 1.5*x }
  object { VeroBoardPiece translate 2.0*x }
  object { VeroBoardPiece translate 2.5*x }
}

#declare WireBend =
union {
  difference {
    torus { 0.1, 0.01 }
    plane { x,0 }
    plane { z,0 }
  }
  cylinder { <-0.3,0,0.1>,<0,0,0.1>,0.01 }
  cylinder { <0.1,0,-0.3>,<0.1,0,0>,0.01 }
  texture { Chrome_Metal }
}
#declare RMap =     // Colour map for resistors
color_map {
  [ 0.0  color Tan ]
  [ 0.09 color Tan ]
  [ 0.10 color Red ]
  [ 0.20 color Red ]
  [ 0.21 color Tan ]
  [ 0.29 color Tan ]
  [ 0.30 color Blue ]
  [ 0.40 color Blue ]
  [ 0.41 color Tan ]
  [ 0.49 color Tan ]
  [ 0.50 color Green ]
  [ 0.60 color Green ]
  [ 0.61 color Tan ]
  [ 0.69 color Tan ]
  [ 0.70 color Gold ]
  [ 0.80 color Gold ]
  [ 0.81 color Tan ]
  [ 1.00 color Tan ]
}
  
#declare Resistor =
union {
  superellipsoid { <1, 0.2> rotate 90*y
                   pigment { gradient x color_map { RMap } scale 2 translate -1*x } finish { Dull }
                   scale <0.25,0.1,0.1>
                   }
  object { WireBend rotate <-90,0,0> translate <0.3,-0.1,0> }
  object { WireBend rotate <-90,180,0> translate <-0.3,-0.1,0> }
}

#declare Transistor1 =
union {
  superellipsoid { <1, 0.2> rotate 90*x scale <0.15,0.1,0.15> }
  cylinder { <0,-0.12,0>,<0,-0.1,0>,0.17 }
  box { <-0.2,-0.12,-0.02>,<0,-0.1,0.02> }
  cylinder { <0.1,-0.1,0>,<0.1,-0.5,0>,0.01 }
  cylinder { <-0.1,-0.1,0>,<-0.1,-0.5,0>,0.01 }
  cylinder { <0,-0.1,-0.1>,<0,-0.5,-0.1>,0.01 }
  texture { Chrome_Metal }
}

#declare Link =
union {
  cylinder { <-0.5,0,0>,<0.5,0,0>,0.025 pigment { Yellow } finish { Dull } }
  object { WireBend rotate <-90,0,0> translate <0.5,-0.1,0> }
  object { WireBend rotate <-90,180,0> translate <-0.5,-0.1,0> }
}

#declare CapTex =     // Capacitor texture
texture { pigment { CadetBlue } finish { Dull } }

#declare Capacitor_Base =
difference {
  union {
    torus { 0.95, 0.05 translate 0.05*y }
    torus { 0.95, 0.05 translate 4*y }
    cylinder { <0,0.05,0>,<0,4,0>,1 }
    texture { CapTex }
  }
  torus { 1.05, 0.1 translate 3.75*y texture { CapTex } }
  cylinder { <0, 3.99, 0>,<0, 4.1, 0>,0.9 pigment { Black } finish { Dull } }
  bounded_by { cylinder { <0,-0.1,0>,<0,4.1,0>,1.1 } }
}
#declare Capacitor1 =     // small electrolytic capacitor with leads
union {
  object { Capacitor_Base translate -4*y scale 0.15 }
  cylinder { <-0.1,0,0>,<-0.1,0.5,0>,0.01 texture { Chrome_Metal }}
  cylinder { <0.1,0,0>,<0.1,0.5,0>,0.01 texture { Chrome_Metal }}
}
#declare Capacitor2 =     // large electrolytic capacitor with terminals
union {
  object { Capacitor_Base }
  object { Knob rotate 90*x translate <-0.4,4,0> }
  object { Knob rotate 90*x translate <0.4,4,0> }
}

#declare Capacitor3 =     // small plastic capacitor
union {
  superellipsoid { <0.3, 0.3> 
                  texture {pigment { Tan } finish { Dull }}
                  texture {pigment { image_map { gif "t_cap1.gif" once interpolate 2 filter 1,1 }} scale 2 translate <-1,-1,0>}
                  scale <0.15,0.15,0.05>
                  }
  cylinder { <-0.1,0,0>,<-0.1,-0.5,0>,0.01 texture { Chrome_Metal }}
  cylinder { <0.1,0,0>,<0.1,-0.5,0>,0.01 texture { Chrome_Metal }}
}

#declare Potentiometer =
union {
  difference {
    superellipsoid { <1, 0.1> rotate 90*y scale <0.45,0.45,0.45> }
    plane { -x,-0.1 }
    cylinder { <-0.05,0.65,0>,<1,0.65,0>,0.45 }
    cylinder { <-0.05,-0.65,0>,<1,-0.65,0>,0.45 }
    texture { Chrome_Metal }
  }
  cylinder { <-0.1,0,0>,<-0.05,0,0>,0.46 pigment { Tan } finish { Dull } }
  union {
    sphere { 0,1 scale <0.05,0.2,0.2> translate 0.15*x }
    sphere { 0,1 scale <0.05,0.2,0.2> translate 0.20*x }
    sphere { 0,1 scale <0.05,0.2,0.2> translate 0.25*x }
    sphere { 0,1 scale <0.05,0.2,0.2> translate 0.30*x }
    sphere { 0,1 scale <0.05,0.2,0.2> translate 0.35*x }
    texture { Chrome_Metal }
  }
  cylinder { <0,0,0>,<1,0,0>,0.125 pigment { White } finish { Dull } }
  object { Control rotate -90*z translate 1*x }
}

#declare Spacer1 =      // plastic spacer to mount the circuit board
cylinder { <0,-0.2,0>,<0,0,0>,0.1 pigment { White } finish { Dull } }

#declare Circuit =
union {
  object { VeroBoard }
  object { Resistor translate <1.2,0.25,1.6> }
  object { Resistor translate <1.4,0.25,1.0> }
  object { Resistor translate <1.2,0.25,0.2> }
  object { Transistor1 translate <0.5,0.55,0.5> }
  object { Transistor1 translate <1.5,0.55,0.5> }
  object { Link rotate 90*y translate <0.2,0.07,1.2> }
  object { Link rotate 90*y translate <2.0,0.07,0.8> }
  object { Capacitor1 rotate 180*x translate <2.2,0.25,1.0> }
  object { Capacitor3 translate <0.7,0.55,0.9> }
  object { Capacitor3 translate <2.2,0.55,0.3> }
  object { Potentiometer scale 0.75 translate <2.9,0.35,1> }
  object { RoundScrewHead scale 0.07 rotate 35*y translate <0.1,0.05,0.1> }
  object { RoundScrewHead scale 0.07 rotate 75*y translate <2.9,0.05,0.1> }
  object { RoundScrewHead scale 0.07 rotate 55*y translate <0.1,0.05,1.9> }
  object { RoundScrewHead scale 0.07 rotate 125*y translate <2.9,0.05,1.9> }
  object { Spacer1 translate <0.1,0.0,0.1> }
  object { Spacer1 translate <2.9,0.0,0.1> }
  object { Spacer1 translate <0.1,0.0,1.9> }
  object { Spacer1 translate <2.9,0.0,1.9> }
}

#declare Wire1 =    // kinked straight wire from X=0 to 1, thickness = 0.02
union {
  #declare R1 = seed(197)
  #declare R2 = seed(5632)
  #declare P = <0,0,0>
  #declare S = 0.05
  #declare DY = 0
  #declare DZ = 0
  #while (S <= 1)
    #declare DY = DY + (rand(R1)-0.5)*0.3
    #declare DZ = DZ + (rand(R2)-0.5)*0.3
    #declare Q = <S,DY*S*(1-S),DZ*S*(1-S)>
    cylinder { P,Q,0.01 }
    sphere { Q,0.01 }
    #declare S = S + 0.05
    #declare P = Q
  #end
  pigment { Red } finish { Dull }
}

#declare Wire2 =    // kinked wire from X=0 to 1, thickness = 0.02, curved in -Y dirn
union {
  #declare R1 = seed(197)
  #declare R2 = seed(5632)
  #declare P = <0,0,0>
  #declare S = 0.05
  #declare DY = 0
  #declare DZ = 0
  #while (S <= 1)
    #declare DY = DY + (rand(R1)-0.5)*0.3
    #declare DZ = DZ + (rand(R2)-0.5)*0.3
    #declare Q = <S,DY*S*(1-S)-0.2*sqrt(sin(S*pi)),DZ*S*(1-S)>
    cylinder { P,Q,0.0075 }
    sphere { Q,0.0075 }
    #declare S = S + 0.05
    #declare P = Q
  #end
  pigment { Red } finish { Dull }
}

#declare EMagnet1 =     // round electromagnet
union {
  cylinder { <0,0,0>,<1,0,0>,0.2 texture { T_Silver_5A } }
  cylinder { <0.01,0,0>,<0.03,0,0>,0.4 pigment { Black } finish { Dull } }
  cylinder { <0.97,0,0>,<0.99,0,0>,0.4 pigment { Black } finish { Dull } }
  #declare R1 = seed(674)
  #declare R2 = seed(239)
  #declare S = 0.05
  #while (S < 1.0)
    #declare RY = (rand(R1)-0.5)*5
    #declare RZ = (rand(R2)-0.5)*5
    superellipsoid { <1, .4> rotate 90*y scale <0.04,0.39,0.39> rotate <0,RY,RZ> translate S*x pigment { Maroon } finish { Dull } }
    #declare S = S + 0.05
  #end
  translate -0.5*x
}

#declare EMagnet2 =     // square electromagnet/coil
union {
  superellipsoid { <.2, .2> scale <0.03,0.4,0.4> translate 0.025*x pigment { Black } finish { Dull } }
  superellipsoid { <.2, .2> scale <0.03,0.4,0.4> translate (0.975)*x pigment { Black } finish { Dull } }
  #declare R1 = seed(674)
  #declare R2 = seed(239)
  #declare S = 0.03
  #while (S < 0.97)
    #declare RY = (rand(R1)-0.5)*20*S*(1-S)   // random tilt to give hand-wound appearance..
    #declare RZ = (rand(R2)-0.5)*20*S*(1-S)   // ...an amateur armature?
    superellipsoid { <.4, .4> scale <0.03,0.39,0.39> rotate <0,RY,RZ> translate S*x pigment { Maroon } finish { Dull } }
    #declare S = S + 0.04
  #end
  translate -0.5*x
}

#declare Rotor =
union {
  cylinder { <-0.55,0,0>,<0.55,0,0>,0.15 }
  box { <-0.5,-0.35,-0.05>,<0.5,0.35,0.05> }
  box { <-0.5,-0.35,-0.05>,<0.5,0.35,0.05> rotate 45*x }
  box { <-0.5,-0.35,-0.05>,<0.5,0.35,0.05> rotate -45*x }
  box { <-0.5,-0.35,-0.05>,<0.5,0.35,0.05> rotate 90*x }
  texture { Chrome_Metal }
}

#declare Armature =
union {
  difference {
    superellipsoid { <.15, .15> scale <1.2,1.2,0.25>   }
    box { <-0.7,-0.7,-1>,<0.7,0.7,1> }
    cylinder { <0,0.95,-1>,<0,0.95,1>,0.375 }
    texture { T_Silver_3A }
    bounded_by { box { <-1.5,-1.5,-0.3>,<1.5,1.5,0.3> }}
  }
  object { EMagnet2 rotate 90*z translate <-0.95,0,0> }
  object { EMagnet2 rotate 90*z rotate 90*y translate <0.95,0,0> }
  object { Rotor scale <0.5,1,1> rotate 90*y rotate 37*z translate 0.95*y }
  object { Wire2 scale <3.3,2,2> rotate <0,0,-6> translate <1.1,-0.5,-0.25> pigment { Maroon } }
  object { Wire2 scale <3.3,2,2> rotate <0,4,-6> translate <0.8,-0.5,-0.25> pigment { Maroon } }
  object { Wire2 scale <5.0,2,2> rotate <25,4,-6> translate <-0.8,-0.5,-0.25> pigment { Maroon } }
  object { Wire2 scale <5.0,2,2> rotate <25,8,-6> translate <-1.1,-0.5,-0.25> pigment { Maroon } }
  object { Bolt scale 0.15 rotate -90*y translate <-0.85,-0.9,-0.25> }
  object { Bolt scale 0.15 rotate -90*y translate <0.85,-0.9,-0.25> }
}

#declare Coil1 =
#declare R0 = 0.15          // Coil radius
#declare W = 0.05           // Wire radius
#declare C = 12             // Distance/Coil ratio
#declare TFirst = -pi/4
#declare TLast = pi*4/3
#declare T = TFirst
#declare S = 0
#declare Height = 1.65       // Actually semi-height
#declare Width = 6
#declare X0 =Width*TFirst/(TLast-TFirst)
#declare V0 = <0,Height*cos(TFirst),0>
#declare P1 = V0
union {
  #while (T < TLast)
    #declare R = R0*min(1,(T-TFirst)*5/(TLast-TFirst))
    #declare R = R*min(1,(TLast-T)*5/(TLast-TFirst))
    #declare Y = Height*cos(T)
    #if (T > pi)
      #declare Y = -Height
    #end
    #declare X = Width*T/(TLast-TFirst) - X0
    #declare V1 = <X,Y,0>
    #declare DS = vlength(V1-V0)
    #declare S = S + DS
    #if (DS > 0)
      #declare P1 = vaxis_rotate(<0,0,R>,V1-V0,degrees(S*C)) + V1
      sphere { P1,W }
      cylinder { P0,P1,W }
    #end
    #declare V0 = V1
    #declare P0 = P1
    #declare T = T + (TLast-TFirst)/350
  #end
  pigment { Red } finish { Dull }
  translate -Height*cos(TFirst)*y
}

#declare Battery =
union {
  difference {
    union {
      superellipsoid { <.3, .3> scale <1.25,0.3,1.25> translate 0.3*y }
      superellipsoid { <.3, .3> scale <1.25,4/3,1.25> translate 4/3*y }
      superellipsoid { <.3, .3> scale <1.25,4/3,1.25> translate 8/3*y }
      superellipsoid { <.3, .3> scale <1.25,0.3,1.25> translate 3.7*y }
    }
    superellipsoid { <.2, .2> scale <1.1,0.1,1.1> translate 4*y pigment { Red }}
    texture { pigment { color red 0.063 green 0.389 blue 0.709 } finish { Dull } }
    texture { pigment { image_map { gif "t_bat1.gif" once interpolate 2 filter 3,1 }} finish { ambient 0.7 } scale 2 translate <-1,1,0>}
  }
  object { Knob rotate 90*x translate <-0.6,3.9,0> }
  object { Knob rotate 90*x translate <0.6,3.9,0> }
  object { Coil1 pigment { Black } rotate 91*y translate <0.6,3.9,-0.1> }
  object { Coil1 rotate 89*y translate <-0.6,3.9,-0.1> }
}

/* ==== Clock ==== */

#declare Girder =
difference {
  superellipsoid { <.3, .3> scale <0.2,6,0.2> translate 5*y }
  superellipsoid { <.3, .3> scale <0.2,7,0.2> translate <0.02,5,0.02> }
  #declare S = 0.1
  #while ( S < 10 )
    cylinder { -x,x,0.05 translate S*y }
    cylinder { -z,z,0.05 translate S*y }
    #declare S = S + 0.25
  #end
  plane { y,0 }
  plane { -y,-10 }
  bounded_by { box { <-0.21,-0.01,-0.21>,<0.21,10.1,0.21> } }
  texture { T_Silver_5A }
}

#declare Base_Girder1 = object { Girder clipped_by { plane { -y,-5.8 }} rotate 90*z translate <10,0.1,0> }
#declare Base_Girder2 = object { Girder clipped_by { plane { -y,-7.3 }} rotate 90*z translate <10,0.1,0> }
#declare Base_Girder3 = object { Girder clipped_by { plane { -y,-8.1 }} rotate 90*z translate <10,0.1,0> }

#declare Base_Frame =
union {
  object { Base_Girder1 translate <-2.1,0,-0.02> }
  object { Base_Girder1 scale <1,1,-1> translate <-2.1,0,2.52> }
  object { Base_Girder2 scale <1,1,1> rotate 90*y translate <-2.02,0,2.6> }
  object { Base_Girder2 scale <1,1,-1> rotate 90*y translate <2.02,0,2.6> }
  object { BlockScrewHead scale 0.1 rotate -90*x translate <2,0.1,-0.22> }    // Front
  object { BlockScrewHead1 scale 0.1 rotate -90*x translate <-2,0.1,-0.22> }
  object { BlockScrewHead2 scale 0.1 rotate -90*z translate <2.22,0.1,0> }    // R.Side
  object { BlockScrewHead scale 0.1 rotate -90*z translate <2.22,0.1,2.5> }
  object { BlockScrewHead2 scale 0.1 rotate -90*x translate <0,0.1,-0.22> }   // Centre
  object { BlockScrewHead2 scale 0.1 translate <-1.5,0.02,0> }      // Mounting
  object { BlockScrewHead scale 0.1 translate <1.5,0.02,0> }        // Mounting
  object { BlockScrewHead2 scale 0.1 translate <-1.5,0.02,2.5> }    // Mounting
  object { BlockScrewHead1 scale 0.1 translate <1.5,0.02,2.5> }     // Mounting
}
#declare Frame =
union {
  box { <-0.2,0,0>,<0.2,10,0.05> translate 0.05*z }           // Gear train supports
  box { <-0.2,0,0>,<0.2,10,0.05> translate 2.45*z }           //        "
  object { Girder translate <-2,0,0> }                  // Corners
  object { Girder rotate -90*y translate <2,0,0> }      //
  object { Girder rotate 90*y translate <-2,0,2.5> }    //
  object { Girder rotate 180*y translate <2,0,2.5> }    //
  object { Base_Frame translate 0.0*y }                       // bottom frame
  object { Base_Frame scale <1,-1,1> translate 10*y }         // top frams
  texture { T_Brass_5A }
}

#declare Face =
union {
  difference {
    cylinder { <0,0,0>,<0,0,-0.05>,3 }
    cylinder { <0,0,1>,<0,0,-1>,2.2 }
  }
  #declare S = 1
  #while (S <=12)
    text {
      ttf "timrom.ttf",
      str(S,0,0)
      1,0
      translate <-0.25*strlen(str(S,0,0)),-0.37,0>
      scale <0.5,0.5,1>
      translate <2.6*sin(radians(S*30)),2.6*cos(radians(S*30)),-0.2>
      pigment { White } finish { Dull }
    }
    #declare S = S + 1
  #end
  torus { 3,0.05 rotate 90*x scale <1,1,2> translate -0.025*z }
  torus { 2.2,0.05 rotate 90*x scale <1,1,2> translate -0.025*z }
  object { RoundScrewHead scale 0.1 rotate <-90,0,0> translate <2,1.75,-0.05> }
  object { RoundScrewHead scale 0.1 rotate <-90,0,35> translate <2,-1.75,-0.05> }
  object { RoundScrewHead scale 0.1 rotate <-90,0,55> translate <-2,1.75,-0.05> }
  object { RoundScrewHead scale 0.1 rotate <-90,0,73> translate <-2,-1.75,-0.05> }
  texture { T_Brass_4B }
  clipped_by { cylinder { <0,0,-.3>,<0,0,.1>,3.2 }}
}
#declare Bell =
union {
  union {
    difference {
      sphere { 0,1 }
      sphere { 0,0.9 }
      plane { y,0 }
    }
    sphere { <0,1,0>,0.2 }
    scale <1,0.75,1>
  }
  cylinder { <0,-2,0>,<0,0.75,0> 0.1 }
  texture { T_Brass_4C }
}
#declare Hammer =
union {
  superellipsoid { <1, .3> rotate 90*y scale <0.5,0.3,0.3> translate 0.1*y }
  cylinder { <0,-3,0>,<0,0,0>,0.1 }		          	// stem
  cylinder { <0,-2,-0.2>,<0,-2,0.2>,0.2 }	       	// pivot
  cylinder { <0,-2,-1.5>,<0,-2,1.5>,0.075 }	      // pivot
  cylinder { <-0.15,-1,0>,<0.15,-1,0>,0.15 }     	// boss for plate
  cylinder { <0,-2,0>,<0.5,-2,0>,0.1 }           	// stop
  cylinder { <-0.25,-1,0>,<-0.15,-1,0>,0.4 texture { T_Silver_5C } }
  texture { T_Brass_4C }
}
#declare Alarm_Frame =
union {
  object { Base_Girder2 scale <1,1,1> rotate 90*y translate <-0.77,-0.12,2.6> }
  object { Base_Girder2 scale <1,1,-1> rotate 90*y translate <0.77,-0.12,2.6> }
  object { Base_Girder3 scale <1,1,1> translate <-0.97,-0.1,-0.02> }
  object { Base_Girder3 scale <1,1,-1> translate <-0.97,-0.1,2.52> }

  object { BlockScrewHead1 scale 0.1 translate <-0.75,-0.1,0> }
  object { BlockScrewHead scale 0.1 translate <0.75,-0.1,0> }
  object { BlockScrewHead2 scale 0.1 translate <-0.75,-0.1,2.5> }
  object { BlockScrewHead scale 0.1 translate <0.75,-0.1,2.5> }

  object { BlockScrewHead1 scale 0.1 rotate -90*z translate <-0.95,0.02,0.25> }
  object { BlockScrewHead2 scale 0.1 rotate -90*z translate <-0.95,0.02,2.25> }

  superellipsoid { <.3, .3> scale <1.2,1.2,0.05> rotate 45*z rotate 90*y translate <-1.03,0,1.25>
                   texture { T_Silver_3A } }
}
#declare Spring =
#declare R0 = 0.15          // Coil radius
#declare W = 0.01           // Wire radius
#declare N = 12             // No. turns
#declare DT = 1/100         // Step increment
#declare T = DT
#declare P0 = <0,0,0>
union {
  #while (T < 1)
    #declare R = R0*min(1,T*7)
    #declare R = R*min(1,(1-T)*7)
    #declare P1 = vaxis_rotate(<0,0,R>,<1,0,0>,degrees(T*2*pi*N)) + <T,0,0>
    sphere { P1,W }
    cylinder { P0,P1,W }
    #declare V0 = V1
    #declare P0 = P1
    #declare T = T + DT
  #end
  texture { Chrome_Metal }
}
#declare Alarm =
union {
  object { Bell }
  object { Hammer translate <1.625,0,0> }
  object { EMagnet1 translate <0.7,-1,0> }
  object { Wire1 scale <2,1,1> rotate 88*y translate <0.25,-0.8,0> pigment { Maroon }}
  object { Wire1 scale <2.2,1,1> rotate <90,88,0> rotate -5*x translate <0.25,-1.2,0> pigment { Maroon }}
  object { Alarm_Frame translate <1.2,-2,-1.25> }
  object { Spring scale <1.625,1,1> translate <0,-2.9,0> }
}
#declare Clock =
union {
    object { Face translate <0,5.33333+2.15,-0.7> }
    object { Frame }
    object { Armature translate <0,1.2,-0.6> }
    object { Gear_Train rotate 90*y translate 2.15*y }
//    object { Alarm translate <2.12,5,1.25> }
}

#declare Base =
superellipsoid { <.15, .15>
		scale <5,0.4,4> translate 0.4*y
		texture { Box_Texture }
}
#declare Model =
union {
  object { Base }
  object { Clock translate <-1.7,0.875,-1> }
  object { Circuit translate <1.5,1,-3> }
  object { Battery translate <3,0,5.3> }
}

/* ==== odds & ends ==== */
#declare Bits =
union {
  object { Bolt rotate -12.68*z rotate -35*y translate <0, 1, 0> scale 0.15 translate <6, 0, -4> }
  object { Bolt rotate -12.68*z rotate  15*y translate <0, 1, 0> scale 0.15 translate <6.3, 0, -5> }
  object { Bolt rotate -12.68*z rotate -30*y translate <0, 1, 0> scale 0.15 translate <6.6, 0, -5.8> }
  object { Bolt rotate -12.68*z rotate  78*y translate <0, 1, 0> scale 0.15 translate <5.5, 0, -6.5> }
  object { Bolt rotate -12.68*z rotate 175*y translate <0, 1, 0> scale 0.15 translate <4.8, 0, -6.6> }
}

/* **** Build the model **** */
union {
    object { Floor }
    object { Walls }
    object { Model rotate 25*y translate <-0.3,0,-2> }
    object { Screwdriver rotate 15*y translate <-5,0,-6> }
    object { Soldering_Iron rotate -75*y translate <-7.5,0,0> }
    object { Solder translate <-9.5,0,2> }
    object { Bits translate <0,0,-2> }
    object { Alarm translate 2*y rotate -45*x rotate 45*y translate <6.8,1.2,-2.5> }

/* testing */
//    object { Clock scale 0.7 rotate -25*y translate <0,0,-2> }
//    object { Capacitor2 translate <-3,0,0> }
//    object { Armature scale 1 rotate -25*y translate <1,2,-2.5> }
//    object { Gear_Train scale 1 rotate 90*y translate <0,0,-3> }
//    object { Base rotate -25*y translate <0,0,-4> }
//    object { Soldering_Iron rotate 15*y translate <0,0,-6> }
//    object { Circuit scale 2 translate <-4.5,0.2,-2.5> }
//    object { EMagnet2 scale 3 rotate -25*y translate <-1,2,-2.5> }
//    object { Battery scale 1.1 rotate -25*y translate <-1,0,-2.5> }

//    object { Alarm scale 3 translate 2*y rotate -45*x translate <0,1.2,-2.5> rotate 30*y }
}

