// Persistence of Vision Ray Tracer Scene Description File
// File: time.pov
// Vers: 3
// Desc: Time Machine - for Raytracing competition
//      The machine was built by a hobbiest and taken back in time
//      to the year 1226 AD. He arrived near a castle (As can be seen here)
//      He has just materialized, and his time field generators are
//      dissipating
// Date: May 11, 1996 (to May 24, 1996)
// Auth: Morgan Hamish Fraser

#version 3.0
global_settings { assumed_gamma 2.2}

#include "colors.inc"
#include "textures.inc"
#include "skies.inc"
#include "stones.inc"
#include "metals.inc"

#declare GoodColors=1
#declare DrawHalos=1
#declare GoodGrass=1
#declare GoodTree=1
#declare GoodStones=1

// note that all dimensions are in metres
#declare NiceCamera=camera {
   location  <5.0, 1.1,-3>
   direction <0, 0,  2.3>
   up        <0, 1,  0>
   right   <4/3, 0,  0>
   look_at   <0.5, 0.6, 1.5>
}

// ================================
// ===========REAL CAMERA =========
// ================================
camera {
   location  <5.0, 0.7+0.4,-3>
   direction <0, 0,  2.3>
   up        <0, 1,  0>
   right   <4/3, 0,  0>
   look_at   <0.5, 0.6, 1.5>
}

light_source {<50, 7, -30> color White}
//light_source {<5, 1.5, 70> color White}

sky_sphere { S_Cloud3 }


// wheels
// origin is absolute centre
#declare Wheel=object{
    cylinder{<0,0,0>,<0,0,0.04>,0.075}
    texture{pigment{color Gray10}}
    }

// **** finish definitions
#declare N_Dented=normal{dents turbulence 0.5 bump_size 0.5 scale 0.02}

// ****** color definitions
#if (GoodColors=1)
    global_settings {max_trace_level 25}
    #declare T_CageBars=texture{T_Copper_3B normal{N_Dented}}
    #declare T_NoseCone=texture{T_Chrome_4C normal{N_Dented}}
    #declare T_ArmRestMetal=texture{T_Chrome_4C}
    #declare T_ArmRest=texture{T_Chrome_1A}
    #declare T_Chair=texture{T_Copper_5A}
    #declare T_Base=texture{
                    pigment{ granite
                        color_map{
                            [0.0 rgb<0.8,0.8,0.8>]
                            [0.4 rgb<0.5,0.6,0.65>]
                            [0.5 rgb<0.2,0.3,0.31>]
                            [0.6 rgb<0,0.1,0>]
                            [0.65 rgb<0.5,0.4,0.3>]
                            [0.8 rgb<0.65,0.7,0.75>]
                            [1.0 rgb<0.7,0.75,0.85>]
                        }
                        turbulence  <0.5,0.5,0.5>
                    }
                    finish{
                        ambient 0.2
                        diffuse 0.4
                    }
                }
    #declare T_CordInner=texture{pigment{color rgb<0.9,0.9,1.0>} finish{phong 0.2 phong_size 4 diffuse 0.6 ambient 0.6}}
    #declare T_CordOuter=texture{pigment{color rgb<1.0,0.4,0.4> filter 0.5 transmit 0.5}}
    #declare T_SubBase=texture{
                        pigment{
                            gradient <0.5,0.5,0>
                            color_map{
                                [0.0 rgb<0.8,0.1,0.1>]
                                [0.1 rgb<0.1,0.1,0.1>]
                                [0.15 rgb<0.1,0.1,0.1>]
                                [0.25 rgb<0.7,0.2,0.2>]
                                [0.3 rgb<0.7,0.7,0.75>]
                                [0.45 rgb<0.6,0.6,0.6>]
                                [0.6 rgb<0.1,0.1,0.1>]
                                [0.75 rgb<0.6,0.6,0.6>]
                                [1 rgb<0.8,0.8,0.9>]
                            }
                            turbulence <0.2,0.05,0.05>
                            scale 0.3
                        }
                    }
                    texture{
                        pigment{
                            bozo
                            color_map{
                                [0.0 rgbft<1,1,1,0.5,1>]
                                [0.6 rgbft<0.2,0.2,1,0.2,0.3>]
                                [0.6 rgb<0.2,0.2,0.2>]
                                [1.0 rgb<0.1,0.1,0.1>]
                            }
                            scale 0.01
                        }
                    }
    #declare T_CastleWall=texture{
            pigment{brick pigment{color rgb<0.4,0.3,0.3>},
                        pigment{granite
                            color_map{
                                    [0 rgb<0.7,0.7,0.7>]
                                    [0.4 rgb<0.4,0.4,0.4>]
                                    [0.4 rgb<0.1,0.1,0.1>]
                                    [0.5 rgb<0.1,0.1,0.1>]
                                    [0.5 rgb<0.6,0.6,0.6>]
                                    [1 rgb<0.6,0.6,0.8>]
                                }
                                scale 3
                            }

                    brick_size <1,0.5,1.2>
                    mortar 0.1
                    turbulence 0.04
                }
            normal{ brick 0.2
                    brick_size <1,0.5,1.2>
                    mortar 0.1
                    turbulence 0.04
            }
            finish{ diffuse 0.5}
            translate<0.2,0,0>
        }
    #declare T_Engine=texture{T_Chrome_4B normal{N_Dented}}
    #declare T_EngineRings=texture{T_Chrome_1A}
    #declare T_EngineUpper=texture{pigment{rgbft<0.7,0.7,1,0.7,0.7>}
                                    finish{refraction 1 ior 1.2}}
    #declare T_EngineConnectors=texture{pigment{color White}
                        finish{diffuse 0.5 ambient 0.5}
                    }
    // Water texture is derived from Dan Farmer's Wavy Water
    #declare T_Water = texture {
        pigment{ granite
                 color_map{
                    [0 color rgbf <0.0, 0.3, 0.8, 0.9> ]
                    [0.4 color rgbf <0.0, 0.6, 0.4, 0.9> ]
                    [0.6 color rgbf <0.0, 0.4, 0.7, 0.9> ]
                    [0.65 color rgbf <0.5, 0.4, 0.2, 0.9> ]
                    [0.7 color rgbf <0.1, 0.4, 0.6, 0.9> ]
                    [1 color rgbf <0.0, 0.3, 0.8, 0.9> ]
                 }
                turbulence 0.5
            }
        normal {
            ripples 0.75
            frequency 10
        }
        finish {
            reflection 0.3
            refraction 0.5
            ior 1.33
        }
    }
#else
    global_settings {max_trace_level 5}
    #declare T_CageBars=texture{pigment{color Brown}}
    #declare T_NoseCone=texture{pigment{color Orange}}
    #declare T_ArmRestMetal=texture{pigment{color Gray80}}
    #declare T_ArmRest=texture{pigment{color Gray20}}
    #declare T_Chair=texture{pigment{color Yellow}}
    #declare T_Base=texture{pigment{color Gray70}}
    #declare T_SubBase=texture{pigment{color Red}}
    #declare T_CordInner=texture{pigment{color rgb<0.7,0.7,1.0>}}
    #declare T_CordOuter=texture{pigment{color rgb<1.0,0.4,0.4>}}
    #declare T_CastleWall=texture{pigment{color rgb<0.6,0.6,0.8>}}
    #declare T_Grass=texture{pigment{color Green}}
    #declare T_Engine=texture{pigment{color Gray80}}
    #declare T_EngineUpper=texture{pigment{rgbft<0.7,0.7,1,1,0.7>}
                                    finish{refraction 1 ior 1.7}}
    #declare T_EngineConnectors=texture{pigment{color White}}
    #declare T_EngineRings=texture{T_Chrome_1A}
    #declare T_Water=texture{pigment{color Blue}}
#end

#if (GoodGrass=1)
    #declare T_Grass=
        texture{
            pigment{granite
                color_map{
                    [0.0 rgb<0.2,1,0.2>]
                    [0.4 rgb<0.5,0.9,0.4>]
                    [0.6 rgb<0.4,0.8,0.5>]
                    [0.8 rgb<0.7,0.85,0.4>]
                    [1 rgb<0.5,0.9,0.4>]
                }
                turbulence 0.3
                scale <0.6,1,0.3>
                scale 0.5
            }
            normal{crackle bump_size 0.1 scale 0.4}
            finish{diffuse 0.2 ambient 0.3}
        }
#else
    #declare T_Grass=texture{pigment{color Green}}
#end


// mandatory textures
#declare T_Mountain1=texture{T_Stone10 finish{phong 0.1 phong_size 2 diffuse 0.4 ambient 0.4}}

// Ground
object{
    difference{
        plane { y, 0}
        box{<0,0.01,-10>,<-15,-3,438> rotate<0,-45,0>}
    }
    texture{T_Grass}
}

// Castle moat
#declare Moat=object{union{
    height_field {gif "Moat.gif"
        smooth
        scale<11,-1,450>
        translate<-11,0,-10>
        rotate<0,-45,0>
        texture{
            gradient <0,-1,0>
            texture_map{
                [0.0 T_Grass]
                [0.2 T_Mountain1]
                [1.0 T_Mountain1]
            }
            scale 1.1
        }
    }
    object{plane{y,-0.9} texture{T_Water}}
    }
}

// The timefield is an aura that goes around the posts
#declare TimeField=object{
        sphere{<0,0,0>,0.3}
    #if (DrawHalos=1)
        pigment{ rgbft<1,1,1,0,1>}
        hollow
        halo {
          emitting
          spherical_mapping
          turbulence <1.5,0.5,1>
          frequency 3
          linear
          color_map {
            [ 0 color rgbt <0.2, 0.5, 0.5, 0.8> ]
            [ 0.4 color rgbt <0.1, 0.6, 0.8, 0.2> ]
            [ 0.5 color rgbt <0, 0, 1, -0.5> ]
            [ 0.6 color rgbt <0, 0, 0.9, 0> ]
            [ 0.8 color rgbt <1, 1, 1, -1> ]
            [ 1 color rgbt <1, 1, 1, -1> ]
          }
          samples 15
          scale 0.2
        }
    #else
        pigment{rgbft<0.4,0.4,1,0.75,0.8>}
    #end
    }



// Far off mountains (Height Field)
#declare Mountains=object{
    height_field {tga "Mountain.tga"
        smooth
        translate<-0.25,0,0>
        scale<400,30,400>
        translate<0,-0.01,400>
        rotate<0,-45,0>
        texture{
            gradient y
            texture_map{
                [0.0 T_Grass]
                [0.2 T_Mountain1]
                [1.0 T_Mountain1]
            }
            scale 31
        }
    }
}

// Flag flying on top of castle tower
#declare Flag=object{union{
        cylinder{<0,0,0>,<0,4+1.3,0>,0.15}
        sphere{<0,4+1.3,0>,0.3}

        object{union{
            // draw flag using equation z=(6x-0.6)*(1.1x-0.7)*(1.1x-0.8)
            #declare XPos=0
            #declare XStep=0.05
            #declare EndWidth=0.2
            #while (XPos<1)
                #declare ZPos=(XPos*6-0.6)*(XPos*1.1-0.7)*(XPos*1.1-0.8)+0.336
                #declare NextXPos=XPos+XStep
                #declare NextZPos=(NextXPos*6-0.6)*(NextXPos*1.1-0.7)*(NextXPos*1.1-0.8)+0.336
                #declare BottomY=0.5+(0.25-EndWidth/2)*XPos
                #declare TopY=1-(0.25-EndWidth/2)*XPos
                #if (XPos<0.7)
                    triangle{
                         <XPos,BottomY,ZPos>
                         <NextXPos,BottomY,NextZPos>
                         <XPos,TopY,ZPos>
                            }
                    triangle{
                         <XPos,TopY,ZPos>
                         <NextXPos,TopY,NextZPos>
                         <NextXPos,BottomY,NextZPos>
                            }
                #else
                    #declare InnerBottomY=0.75-(EndWidth/2)*(XPos-0.7)/0.3
                    #declare InnerTopY=0.75+(EndWidth/2)*(XPos-0.7)/0.3
                    triangle{
                         <XPos,BottomY,ZPos>
                         <NextXPos,BottomY,NextZPos>
                         <XPos,InnerBottomY,ZPos>
                            }
                    triangle{
                         <XPos,InnerBottomY,ZPos>
                         <NextXPos,InnerBottomY,NextZPos>
                         <NextXPos,BottomY,NextZPos>
                            }
                    triangle{
                         <XPos,InnerTopY,ZPos>
                         <NextXPos,InnerTopY,NextZPos>
                         <XPos,TopY,ZPos>
                            }
                    triangle{
                         <XPos,TopY,ZPos>
                         <NextXPos,TopY,NextZPos>
                         <NextXPos,InnerTopY,NextZPos>
                            }
                #end
                #declare XPos=XPos+XStep
            #end
            }
            scale<5,2.25,0.25>
            translate<0,2.75,0>
    texture{pigment{color Red}}
        }
    }
    texture{T_Copper_5C}
}

#declare Battlement=object{
    box{<0,0,0>,<-0.4,0.4,1>}
    }

// Cut for archer in castle wall
#declare MerlinCut=object{
    union{
        box{<0.01,0,-0.25>,<-0.6,1,0.25>}
        box{<-0.5,-0.5,-0.5>,<-1,1.5,0.5>}
    }
    texture{T_CastleWall finish{diffuse 0.2 ambient 0.2}}
 }

#declare CastleButtress=object{
        difference{
            union{
                box{<0,0,-4>,<2,5+3.1,4>}
            }
            object{plane{y,0} rotate<0,0,-78.7+180> translate<0,5+3.1,0>}
            object{plane{x,0} rotate<0,22.5+180,0> translate<0,0,-4>}
            object{plane{x,0} rotate<0,-22.5+180,0> translate<0,0,4>}
        }
        translate<0,-3.1,0>
        texture{T_CastleWall}
    }

#declare Castle=object{union{
        difference{
            object{box{<-10,-3.1,-15>,<-20,10,150>}}
            object{MerlinCut translate<-10,7,120>}
            object{MerlinCut translate<-10,5,100>}
            object{MerlinCut translate<-10,6,80>}
            object{MerlinCut translate<-10,7.5,60>}
            object{MerlinCut translate<-10,5.5,35>}
        }

        object{CastleButtress translate<-10,0,0>}
        object{CastleButtress translate<-10,0,20>}
        object{CastleButtress translate<-10,0,40>}
        object{CastleButtress translate<-10,0,60>}
        object{CastleButtress translate<-10,0,80>}
        object{CastleButtress translate<-10,0,100>}
        object{CastleButtress translate<-10,0,120>}
        object{CastleButtress translate<-10,0,140>}

        object{Flag translate<-15,15,155>}
        difference{
            object{cylinder{<-15,-3.1,155>,<-15,15,155>,10}}
            object{MerlinCut translate<10,10,0> rotate<0,40,0> translate<-15,0,155>}
            #declare Count=0
            #while(Count<180)
                object{box{<-16,14.5,-0.25>,<16,15.01,0.25>
                        rotate<0,Count,0>
                        translate<-15,0,155>}}
                #declare Count=Count+9
            #end
        }

        // draw battlements on wall
        #declare Count=0
        #while (Count<150/2)
            object{Battlement translate<-10,10,2*Count>}
            #declare Count=Count+1
        #end

    }
    texture{T_CastleWall}
    translate<-0.1,0,0>
    rotate<0,-45,0>
}

// SubBase - Support cushion
// origin is Global Time Machine origin
#declare SubBase=object{
    union{
        box{<0.1,0,0.05>,<2.4,-0.1,0.85>}
        object{difference{
            union{
                cylinder{<0.1,0,0.05>,<0.1,0,0.85>,0.1}
                cylinder{<2.4,0,0.05>,<2.4,0,0.85>,0.1}
            }
            object{plane{y,0} inverse}
        }}

        // wheels underneath
        object{Wheel translate<0.3,-0.1,0.2>}
        object{Wheel translate<0.3,-0.1,0.7>}
        object{Wheel translate<0.3+0.625,-0.1,0.2>}
        object{Wheel translate<0.3+0.625,-0.1,0.7>}
        object{Wheel translate<0.3+0.625*2,-0.1,0.2>}
        object{Wheel translate<0.3+0.625*2,-0.1,0.7>}
        object{Wheel translate<0.3+0.625*3,-0.1,0.2>}
        object{Wheel translate<0.3+0.625*3,-0.1,0.7>}
    }
    texture{T_SubBase}
}
#declare Base=object{
    union{
        object{box{<0,0,0>,<2.6,0.04,0.9>}}

        // front end of machine
        object{difference{
            object{box{<0,0,0>,<-0.5,0.3,0.9>}}
            object{union{
                object{cylinder{<0,0,0>,<0,0.3,0>,0.5}
                        scale<1,1,0.45/0.5> translate<0,0,0.45>}
                object{box{<0,0,0.45>,<-0.5,0.3,0.9>}}
            } inverse}
            object{union{
                object{cylinder{<0,0,0>,<0,0.3,0>,0.5}
                        scale<1,1,0.45/0.5> translate<0,0,0.9-0.45>}
                object{box{<0,0,0.9-0.45>,<-0.5,0.3,0>}}
            } inverse}
            object{cylinder{<0,0.3,0>,<0,0.3,0.9>,0.3}
                    scale<0.5/0.3,1,1> inverse}
            object{cylinder{<0,0.3,-0.01>,<0,0.3,0.901>,0.3-0.04}
                    scale<(0.5-0.04)/(0.3-0.04),1,1>}
        }}
    }
    texture{T_Base}
}

// this is the enclosing cage for the engine device
// cross beams are all 3 cm in diameter
// origin is the front lower left - middle of column
#declare Cage=object{
    union{
        object{cylinder{<0,0,0>,<0,0.7+0.3,0>,0.015}}
        object{sphere{<0,1,0>,0.04}}
        object{TimeField scale<3,1.4,2> translate<0,1,0>}
        object{cylinder{<0,0,0.87>,<0,0.7+0.3,0.87>,0.015}}
        object{sphere{<0,1,0.87>,0.04}}
        object{TimeField rotate<40,0,0> scale<3,1.4,2> translate<0,1,0.87>}
        object{cylinder{<0.95,0,0>,<0.95,0.7,0>,0.015}}
        object{cylinder{<0.95,0,0.87>,<0.95,0.7,0.87>,0.015}}
        object{cylinder{<0,0,0>,<0.95,0.7,0>,0.015}}
        object{cylinder{<0,0,0.87>,<0.95,0.7,0.87>,0.015}}
        object{cylinder{<0,0.7,0>,<0.95,0,0>,0.015}}
        object{cylinder{<0,0.7,0.87>,<0.95,0,0.87>,0.015}}
        object{cylinder{<0,0.7,0>,<0,0.7,0.87>,0.015}}
        object{cylinder{<0.95,0.7,0>,<0.95,0.7,0.87>,0.015}}
        object{cylinder{<1.22,0,0>,<1.22,0.3,0>,0.015}}
        object{cylinder{<1.22,0,0.87>,<1.22,0.3,0.87>,0.015}}
        object{cylinder{<1.22,0.3,0>,<1.22,0.3,0.87>,0.015}}
        object{cylinder{<0.95,0,0>,<1.22,0.3,0>,0.015}}
        object{cylinder{<0.95,0,0.87>,<1.22,0.3,0.87>,0.015}}
        object{cylinder{<0.95,0.7,0>,<1.22,0.3,0>,0.015}}
        object{cylinder{<0.95,0.7,0.87>,<1.22,0.3,0.87>,0.015}}
        object{cylinder{<0,0.7,0>,<0.95,0.7,0>,0.015}}
        object{cylinder{<0,0.7,0.87>,<0.95,0.7,0.87>,0.015}}
        object{cylinder{<0.95,0.7,0>,<1.22,0.3,0.87>,0.015}}
        object{cylinder{<0.95,0.7,0.87>,<1.22,0.3,0>,0.015}}
        object{cylinder{<1.22,0.3,0>,<1.22,0,0.87>,0.015}}
        object{cylinder{<1.22,0.3,0.87>,<1.22,0,0>,0.015}}
        object{sphere{<0,0.7,0>,0.015}}
        object{sphere{<0.95,0.7,0>,0.015}}
        object{sphere{<1.22,0.3,0>,0.015}}
        object{sphere{<0,0.7,0.87>,0.015}}
        object{sphere{<0.95,0.7,0.87>,0.015}}
        object{sphere{<1.22,0.3,0.87>,0.015}}
    }
    texture{T_CageBars}
}

// chair - origin is front, base centre of back of chair
#declare Chair=object{
    union{
        object{box{<0,0,-0.35>,<0.05,0.75,0.35>}} // chair back

        // seat of chair
        object{box{<0,0,-0.25>,<-0.25,0.15,0.25>}}
        object{prism{linear_sweep -0.25,0.25
                     4,<0,0>,<0.15,0>,<0,0.3>,<0,0>}
                rotate<-90,0,90>
                translate<-0.25,0,0>
            }

        // Left side arm rest
        object{cylinder{<0,0.35,-0.3>,<-0.2,0.35,-0.3>,0.075/2} texture{T_ArmRest}}
        object{cylinder{<-0.5,0,-0.3>,<-0.2,0.35,-0.3>,0.04/2} texture{T_ArmRestMetal}}
        object{sphere{<-0.2,0.35,-0.3>,0.04/2} texture{T_ArmRestMetal}}

        // Right side arm rest
        object{cylinder{<0,0.35,0.3>,<-0.2,0.35,0.3>,0.075/2} texture{T_ArmRest}}
        object{cylinder{<-0.5,0,0.3>,<-0.2,0.35,0.3>,0.04/2} texture{T_ArmRestMetal}}
        object{sphere{<-0.2,0.35,0.3>,0.04/2} texture{T_ArmRestMetal}}
    }
    texture{T_Chair}
}

// Nose cone of machine
// origin is 0.04 above absolute origin for machine
#declare NoseCone=object{
    union{
        // portion nearest the driver
        difference{
            union{
                object{box{<0,0,0>,<1.2,0.26,0.01>}}
                object{box{<0,0,0.9>,<1.2,0.26,0.89>}}
            }
            object{cylinder{<0,0.3,-0.001>,<0,0.3,0.901>,0.25}
                    scale<0.9/0.25,1,1> translate<1.2,0,0>}
        }
        difference{
                object{cylinder{<0,0,0.45>,<0.3,0,0.45>,0.45}
                        scale<1,0.14/0.45,1> translate<0,0.26,0>}
                object{box{<0,0.26,0>,<0.301,0,0.9>}}
        }

        // nosecone posts
        object{union{
            object{cylinder{<0.25,0,0.015>,<0.25,0.55,0.015>,0.015}}
            object{sphere{<0.25,0.55,0.015>,0.04}}
            object{TimeField scale<1,1,1> translate<0.25,0.55,0.015>}
            object{cylinder{<0.25,0,0.885>,<0.25,0.55,0.885>,0.015}}
            object{sphere{<0.25,0.55,0.885>,0.04}}
            object{TimeField scale<1,1,1> translate<0.25,0.55,0.885>}
        } texture{T_CageBars}}

        // front portion of nosecone
        difference{
            object{box{<0,0,0>,<-0.5,0.4,0.9>}}
            object{
                union{
                    object{sphere{<0,0,0>,0.5}
                        scale<1,0.14/0.5,0.45/0.5>
                        translate<0,0.26,0.45>}
                    object{box{<0,0,0>,<-0.5,0.26,0.9>}}
                    }
                inverse}
            object{union{
                object{cylinder{<0,0.26,0>,<0,0.26,0.9>,0.26}
                        scale<0.46/0.26,1,1>}
                object{box{<0,0.26,0>,<-0.5,0.4,0.9>}}
            }inverse}
            object{cylinder{<0,-0.01,0.45>,<0,0.4,0.45>,0.45}
                    scale<0.5/0.45,1,1> inverse}
        }

        // Side bar GNDN thingys
        object{union{
                cylinder{<-0.1,0.1,0>,<0.35,0.1,0>,0.02}
                sphere{<-0.1,0.1,0>,0.02}
                sphere{<0.35,0.1,0>,0.02}
                cylinder{<0,0.13,0>,<0.30,0.13,0>,0.01}
                sphere{<0,0.13,0>,0.01}
                sphere{<0.30,0.13,0>,0.01}
                cylinder{<0.1,0.05,0>,<0.5,0.05,0>,0.03}
                sphere{<0.1,0.05,0>,0.03}
                sphere{<0.5,0.05,0>,0.03}
            }
            texture{T_Brass_5C}
        }
    }
    texture{T_NoseCone}
}

// Keyboard
//origin is top of nosecone left inner corner
#declare Keyboard=object{
    union{
        object{prism{linear_sweep -0.88,0,5
                <0,0>,<0.12,0>,<0.12,0.01>,<0,0.04*2>,<0,0>}
                rotate<-90,0,0> translate<0,-0.04*2,0>}

        //Start Button
        object{cylinder{<0.06,-0.01,0.65>,<0.06,-0.08,0.65>,0.05}
                texture{pigment{color Red}}
                }
    }
    texture{pigment{color OldGold}}
    texture{pigment{image_map{
        gif "keyboard.gif"
        filter 0, 0
        once
        }
        rotate<90,-90,0>
        scale<0.12/1*0.75,1,0.88/1*0.5>
        translate<0.12*0.75+0.12*0.25/2,1,0.05>
    }}
}

// Readout screen
//origin is bottom left front corner
#declare DisplayScreen=object{
    union{
        difference{
            union{
                object{sphere{<0,0,0.44>,0.44}
                        scale<0.1/0.44,0.2/0.44,1>
                        translate<-0.05,0,0>}
                object{cylinder{<0,0,0.44>,<-0.05,0,0.44>,0.44}
                        scale<1,0.2/0.44,1>}
            }
            object{box{<0,0,0>,<0.101,0.21,0.88>}}
            object{box{<-0.16,-0.21,0>,<0.101,0,0.88>}}
        }
        object{box{<0.0,0,0.19>,<0.001,0.15,0.88-0.19>}
            texture{pigment{
                image_map{gif "readout.gif"
                filter 0, 1
                once
                }
                rotate<0,-90,0>
                scale<1,99/334,1>
                scale 0.5
                translate<0,0,0.19>
            }}
        }
    }
    texture{T_NoseCone}
}

// Time machine engine
// origin is back of sled
// cables enter at -
#declare Engine=object{union{
        // main engine part
        cylinder{<0.45,0,0.45>,<0.45,0.5,0.45>,0.3}
        object{
            union{
                object{torus{0.3,0.025} translate<0.45,0.5,0.45>}
                object{torus{0.3,0.025} translate<0.45,0.25,0.45>}
                object{torus{0.3,0.025} translate<0.45,0,0.45>}
            }
            texture{T_EngineRings}
        }
            // Main Hatch
        object{intersection{
                    box{<0,0,0>,<0.256,0.16,0.3>}
                    cylinder{<0,0,0.31>,<0,0.16,0.31>,0.31}
                }
            texture{
                material_map{
                        gif "hatch.gif"
                        texture{T_EngineRings}
                        texture{T_Engine}
                        texture{T_Brass_1A}
                    }
                    scale<0.256,0.16,1>
                }
            translate<0,0.375-0.08,-0.31>
            rotate<0,-20,0>
            translate<0.45,0,0.45>
        }
            // secondary hatch
        object{
                union{
                    // main hatch part
                    intersection{
                        box{<0,0,0>,<0.256,0.12,0.3>}
                        cylinder{<0.128,0,0.31>,<0.128,0.12,0.31>,0.31}
                    }
                    // bolts
                    object{union{
                        #declare Counter=0
                        #while (Counter<5)
                             object{sphere{<0,0,-0.31>,0.009}
                                rotate<0,-24.4+4.88+9.76*Counter,0>
                                translate<0.256/2,0.03,0.31>
                             }
                             object{sphere{<0,0,-0.31>,0.009}
                                rotate<0,-24.4+4.88+9.76*Counter,0>
                                translate<0.256/2,0.09,0.31>
                             }
                            #declare Counter=Counter+1
                        #end
                        }
                        texture{T_Brass_1B}
                    }
                }
            texture{T_Engine scale 0.4}
            translate<-0.128,0.125-0.06,-0.31>
            rotate<0,-10,0>
            translate<0.45,0,0.45>
        }


        // 2 tank-like structures beside main engine part (near chair)
        cylinder{<0.15302,0,0.15302>,<0.15302,0.75,0.15302>,0.12}
        cylinder{<0.15302,0,0.9-0.15302>,<0.15302,0.75,0.9-0.15302>,0.12}
        sphere{<0.15302,0.75,0.15302>,0.12}
        sphere{<0.15302,0.75,0.9-0.15302>,0.12}
        object{
            union{
            #declare Angle=0
                #while (Angle<360)
                    object{union{
                            cone{<0,0.3,-0.12>,0.05,<0,0.6,-0.12>,0.02}
                            sphere{<0,0.3,-0.12>,0.05}
                            sphere{<0,0.6,-0.12>,0.02}
                            cone{<0,-0.03,-0.12>,0.03,<0,0.2,-0.12>,0.005}
                        }
                        rotate<1.4,0,0>
                        rotate<0,Angle+20,0>
                    }
                    object{union{
                            cone{<0,0.3,-0.12>,0.05,<0,0.6,-0.12>,0.02}
                            sphere{<0,0.3,-0.12>,0.05}
                            sphere{<0,0.6,-0.12>,0.02}
                        }
                        rotate<1.4,0,0>
                        rotate<0,Angle+20,0>
                        translate<0,0,0.59396>
                    }
                    #declare Angle=Angle+60
                #end
            }
            texture{T_Copper_5C}
            translate<0.15302,0,0.15302>
            }

        // funky tardis thingy
        object{merge{
                cylinder{<0.45,0.5,0.45>,<0.45,0.7,0.45>,0.15}
                sphere{<0.45,0.7,0.45>,0.15}
             }
             texture{T_EngineUpper}
        }

        // glowing rods inside funky tardis thingy
        object{union{
                cylinder{<0.45-0.075,0.5,0.45-0.075>,<0.45-0.075,0.75,0.45-0.075>,0.025}
                cylinder{<0.45+0.075,0.5,0.45-0.075>,<0.45+0.075,0.75,0.45-0.075>,0.025}
                cylinder{<0.45-0.075,0.5,0.45+0.075>,<0.45-0.075,0.75,0.45+0.075>,0.025}
                cylinder{<0.45+0.075,0.5,0.45+0.075>,<0.45+0.075,0.75,0.45+0.075>,0.025}
            }
            texture{T_EngineConnectors}
        }

        // Rear End Cylinders
            // main rear one
            difference{
                cylinder{<1,0,0.45>,<1,0.3,0.45>,0.15}
                sphere{<1,0.3,0.45>,0.15}
            }
            // connectors & other funky things
            object{union{
                object{torus{0.1,0.01} rotate<0,0,90> scale<1,2,1> translate<1,0.3,0.3>}
                object{torus{0.1,0.01} rotate<0,0,90> scale<1,2,1> translate<1,0.3,0.6>}
                }
                texture{T_EngineRings}
            }
            object{union{
                    cylinder{<1.08,0.2,0.4>,<1.08,0.45,0.4>,0.02}
                    sphere{<1.08,0.45,0.4>,0.02}
                    cylinder{<1.06,0.2,0.5>,<1.06,0.38,0.5>,0.015}
                    sphere{<1.06,0.38,0.5>,0.015}
                    cylinder{<1.01,0.2,0.46>,<1.01,0.51,0.46>,0.02}
                    sphere{<1.01,0.51,0.46>,0.02}
                    cylinder{<0.95,0.2,0.42>,<0.95,0.51,0.42>,0.017}
                    sphere{<0.95,0.51,0.42>,0.017}
                }
                texture{T_Brass_4C}
            }
            // light panel on main rear cylinder
            object{
                union{
                    object{
                        intersection{
                            box{<0,0,0>,<0.1,0.2,0.15>}
                            cylinder{<0,0,0.16>,<0,0.2,0.16>,0.16}
                        }
                        texture{T_Engine scale 0.4}
                    }
                    object{sphere{<0.05,0.15,0>,0.015}
                        texture{pigment{color rgbft<0,1,0,0.7,0.3>}
                                finish{phong 1}}}
                    object{sphere{<0.05,0.10,0>,0.015}
                        texture{pigment{color rgbft<0,0,1,0.7,0.3>}
                                finish{phong 1}}}
                    object{sphere{<0.05,0.05,0>,0.015}
                        texture{pigment{color rgbft<1,0,0,0.7,0.3>}
                                finish{phong 1}}}
                }
                translate<0,0.02,-0.16>
                rotate<0,-30,0>
                translate<1,0,0.45>
            }

        cylinder{<1,0,0.2>,<1,0.2,0.2>,0.1}
        cylinder{<1,0,0.7>,<1,0.2,0.7>,0.1}
        sphere{<1,0.2,0.2>,0.1}
        sphere{<1,0.2,0.7>,0.1}
        object{cylinder{<1,0.15,0.45>,<0.45,0.15,0.45>,0.05}
            texture{T_Brass_1A}
            }
        // connectors from rear cylinders to front cylinders
        object{union{
                object{cone{<1,0.1,0.2>,0.03,<0.15302,0.3,0.15302-0.07>,0.01}}
                //object{cone{<1,0.1,0.7>,0.03,<0.15302,0.3,0.9-0.15302>,0.01}}
            }
            texture{T_Brass_2B}
        }
    }
    texture{T_Engine}
}

#declare TimeMachine=object{
    union{
        object{SubBase}
        object{Base}
        object{Cage translate<1.35,0.04,0.015>}
        object{Engine translate<1.35,0.04,0>}
        object{Chair translate<1.25,0.04,0.45>}
        object{NoseCone translate<0,0.04,0>}
        object{Keyboard translate<0.3,0.3,0.01>}
        object{DisplayScreen translate<0.301,0.3,0.01>}

    }
    translate<0,0.175,0>
}

#if (GoodTree=1)
    #include "mhftree.pov"
#else
    #declare mhftree=object{sphere{<0,0,0>,1} scale<2.5,10,2.5>
             texture{pigment{color rgbft<0.2,0.8,0.2,0.6,0.6>}}}
#end

#declare Forest=object{union{
    object{mhftree translate<12,0,150> rotate<0,-45,0>}
    object{mhftree scale<1.2,1.2,1.2> translate<25,0,100> rotate<0,-45,0>}
    object{mhftree scale<1,1.2,1.1> translate<42,0,150> rotate<0,-45,0>}
}}

#if (GoodStones=1)
/*    #declare T_Stone1=texture{pigment{color rgb<0.7,0.7,0.8>}
                    finish{ ambient 0.3 }
                }*/
    #include "Stone1.inc"
#else
    #declare Stone1=object{
                sphere{<0,0,0>,1}
                texture{pigment{color Gray80}}
        }
#end

// This is a series of rocks lying on the ground
//  they are for atmosphere
#declare Rocks=object{union{
        object{Stone1 scale<0.2,0.05,0.1> translate<2,0,-0.4>}
        object{Stone1 scale<1.5,0.3,0.7> translate<5,0,10> rotate<0,-45,0>}
        object{Stone1 scale<0.8,0.5,0.7> translate<5.5,0,9> rotate<0,-45,0>}
    }
    texture{pigment{color Red}}
}

object{Forest}
object{Rocks}
object{Mountains}
object{TimeMachine rotate<0,0,0>}
object{Castle}
object{Moat}
