// Names of the various shading flags
define shadow_check       1
define reflect_check      2
define transmit_check     4
define two_sided_surface  8
define cast_shadows      16
define primary_rays      32

// Default shading flags during raytracing - everything gets checked.
define all_shading_flags
   shadow_check + reflect_check + transmit_check + cast_shadows +
   two_sided_surface + primary_rays

//
// Define a set of shading flags for surfaces that shouldn't have their
// backsides illuminated
define one_sided_surface
   all_shading_flags - two_sided_surface

// Useful definitions to use with noise surfaces.  These
// will come in handy below when we declare marble and
// wood textures.
define position_plain       0
define position_objectx     1
define position_worldx      2
define position_cylindrical 3
define position_spherical   4
define position_radial      5

define lookup_plain    0
define lookup_sawtooth 1
define lookup_cos      2
define lookup_sin      3

define plain_normal  0
define bump_normal   1
define ripple_normal 2
define dented_normal 3

define blue_ripple
texture {
   noise surface {
      color <0.4, 0.4, 1.0>
      normal 2
      frequency 100
      bump_scale 2
      ambient 0.3
      diffuse 0.4
      specular white, 0.7
      reflection 0.5
      microfacet Reitz 10
      }
   scale <10, 1, 10>
   }

// The standard sort of marble texture
define white_marble
texture {
   noise surface {
      color white
      position_fn position_objectx
      lookup_fn lookup_sawtooth
      octaves 3
      turbulence 3
      ambient 0.3
      diffuse 0.8
      specular 0.3
      microfacet Reitz 5
      color_map(
         [0.0, 0.8, <1, 1, 1>, <0.6, 0.6, 0.6>]
         [0.8, 1.0, <0.6, 0.6, 0.6>, <0.1, 0.1, 0.1>])
      }
   }

// Nice blue agate texture
define sapphire_agate
texture {
   noise surface {
      ambient 0.4
      diffuse 0.6
      specular white, 0.2
      microfacet Cook 4
      position_fn position_objectx
      position_scale 1.1
      lookup_fn lookup_sawtooth
      octaves 3
      turbulence 2
      color_map(
         [0.0, 0.3, <0, 0, 0.9>, <0, 0, 0.8>]
         [0.3, 1,   <0, 0, 0.8>, <0, 0, 0.4>])
      }
   scale <0.5, 0.5, 0.5>
   }

// Simple color map texture
define whorl_texture
texture {
   noise surface {
      color green
      ambient 0.3
      diffuse 0.8
      lookup_fn lookup_sawtooth
      octaves 2
      turbulence 2
      color_map(
         [0.0, 0.3, green,   blue]
         [0.3, 0.6, blue,    skyblue]
         [0.6, 0.8, skyblue, orange]
         [0.8, 1.0, orange,  red])
      }
   scale <0.5, 0.5, 0.5>
   }

// Create a wood texture.  Concentric rings of color
// are wrapped around the z-axis.  There is some turbulence
// in order to keep the rings from looking too perfect.
define light_wood <0.6, 0.24, 0.1>
define median_wood <0.3, 0.12, 0.03>
define dark_wood <0.05, 0.01, 0.005>
define wooden
texture {
   noise surface {
      position_fn position_cylindrical
      position_scale 1
      lookup_fn lookup_sawtooth
      octaves 1
      turbulence 1
      ambient 0.2
      diffuse 0.7
      specular white, 0.5
      microfacet Reitz 10
      color_map(
         [0.0, 0.2, light_wood, light_wood]
         [0.2, 0.3, light_wood, median_wood]
         [0.3, 0.4, median_wood, light_wood]
         [0.4, 0.7, light_wood, light_wood]
         [0.7, 0.8, light_wood, median_wood]
         [0.8, 0.9, median_wood, light_wood]
         [0.9, 1.0, light_wood, dark_wood])
      }
   }

// Define a texture using a color wheel
define xz_wheel_texture
texture {
   special surface {
      color color_wheel(x, y, z)
      ambient 0.2
      diffuse 0.8
      specular white, 0.2
      microfacet Reitz 10
      }
   }

// This is an example of a gradient texture.
define mountain_colors
texture {
   noise surface {
      ambient 0.2
      diffuse 0.8
      specular 0.2
      position_fn position_objectx
      color_map(
         [-128,   0, blue,  blue]
         [   0,  20, green, green]
         [  20,  40, green, tan]
         [  40,  90, tan,   tan]
         [  90, 128, white, white])
      }
   rotate <0, 0, 90>
   }

// This makes a nice texture to use on leaves
define bumpy_green
texture {
   special surface {
      normal N + dnoise(3*W)
      ambient green, 0.2
      diffuse green, 0.3
      specular white, 0.7
      microfacet Reitz 10
      }
   scale <0.02, 0.02, 0.02>
   }

// Red dented/wrinkled appearance
define dented_red
texture {
   noise surface {
      color <1, 0.2, 0.2>
      normal 1
      frequency 2
      bump_scale 3
      ambient 0.2
      diffuse 0.5
      specular 0.7
      microfacet Reitz 10
      }
   scale <0.2, 0.2, 0.2>
   }

define dented_brown
texture {
   noise surface {
      color Tan
      normal 1
      frequency 2
      bump_scale 3
      ambient 0.2
      diffuse 0.5
      specular 0.7
      microfacet Reitz 10
      }
   scale <0.2, 0.2, 0.2>
   }

// When used on a big sphere, this makes a nice cloudy sky.  You will
// probably need to scale to make it look good.
define cloudy_sky
texture {
   noise surface {
      ambient 0.9
      diffuse 0
      specular 0
      turbulence 6.0
      position_fn 2
      lookup_fn 1
      octaves 4
      color_map(
	 [0.0, 0.6, <0.4, 0.4, 0.4>, <1, 1, 1>]
	 [0.6, 0.8, <1, 1, 1>, <0.196078, 0.6, 0.8>]
	 [0.8, 1.0, <0.196078, 0.6, 0.8>, <0.196078, 0.6, 0.8>])
      }
   }

define erode_exper (noise(10*P, 2) < 0.2 ? 1 :
		     (noise(10*P, 2) < 0.4 ?
			1 - 5 * (noise(10*P, 2) - 0.2) : 0))

define Red_Brick
texture {
   special surface {
      normal N + 4 * (dnoise(10*P, 4) - <0.5, 0.5, 0.5>)
      ambient (1 - erode_exper) * red, 0.2
      diffuse (1 - erode_exper) * red, 0.5
      specular (1 - erode_exper) * white, 0.7
      microfacet Reitz 15
      transmission erode_exper, 1
      }
   }

define Grey_Brick
texture {
   special surface {
      normal N + 4 * (dnoise(10*P, 4) - <0.5, 0.5, 0.5>)
      ambient (1 - erode_exper) * DarkSlateGrey, 0.2
      diffuse (1 - erode_exper) * DarkSlateGrey, 0.5
      specular (1 - erode_exper) * white, 0.7
      microfacet Reitz 15
      transmission erode_exper, 1
      }
   }

