
//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// PTD_Tree.inc - Almost totally random nested loop tree!
//
// Created by: Paul T. Dawson
//             ptdawson@voicenet.com
//             http://www.voicenet.com/~ptdawson
//
// <=>-<=>-<=>-<=>-<=>-<=>-<=>-<=>-<=>-<=>-<=>-<=>
// <=>-<=>-<=>-<=>                 <=>-<=>-<=>-<=>
// <=>-<=>-<=>-<=>  PUBLIC DOMAIN  <=>-<=>-<=>-<=>
// <=>-<=>-<=>-<=>                 <=>-<=>-<=>-<=>
// <=>-<=>-<=>-<=>-<=>-<=>-<=>-<=>-<=>-<=>-<=>-<=>
//
// Requires POV-Ray 3.0, with the new "rand" function, and loops!
//
// Thank you, POV-Team, for your magnificent program!!!
//
//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// Note: This file should be broken into two separate files, with the
//       first file containing the "variable setting", and the second
//       file with the actual "tree building". I kept both parts together
//       this time, because it's a little easier that way.
//
// Note: This file does NOT display the tree. It just makes an object
//       called "Complete_Tree". Then your scene file can show that!
//
// Note: Your main scene file must have all of the usual include files,
//       including "colors.inc" and "textures.inc". If you use any
//       special textures, like "golds" or "stones", you will need to
//       include those files, too (but you-all knew all that!).
//
// Note: Realistic trees use a *LOT* of memory!!!
//
//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// So, here's PART ONE, with the variable setting...
//
// You can change anything in this section (within reason!).
//
//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// The seed(#) value controls the PSEUDO "random" numbers.

        #declare TREE_RAND = seed(0)

//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// These settings control the number of branches on the tree. Each number
// is NOT a total number. It's just in relation to the "parent branch".
// For example, you might have 1 trunk and 5 large branches, and then put
// 3 medium branches on EACH large branch. Then you could put, say, 10 small
// branches on each medium one. The grand total (if you figure it all out)
// would be 1 + 5 + (5*3) + (5*3*10) = 171 branches. Be careful when you
// change these, because you can easily end up with thousands of branches!

        #declare Number_Of_Large_Branches = 5
        #declare Number_Of_Medium_Branches = 5
        #declare Number_Of_Small_Branches = 10

// The branches start out standing straight up. If you rotate the branch
// by 10 degrees, it will tilt just a little. If you rotate it 90 degrees,
// it will be at a right angle to the parent branch. These variables set
// the allowable range of "tilt" for each branch. Remember, in this tree,
// each branch gets a different RANDOM tilt. These are just the limits!

        #declare Branch_Minimum_Angle = 20
        #declare Branch_Maximum_Angle = 80

// Set the size of the tree trunk, and the different branches. This is
// for the length only, not the diameter. Usually, the branches get
// shorter as they go outwards. You can put in any size you want, for some
// very strange trees!

        #declare Tree_Trunk_Size = 10

        #declare Large_Branch_Size_Min = 2
        #declare Large_Branch_Size_Max = 4

        #declare Medium_Branch_Size_Min = 2
        #declare Medium_Branch_Size_Max = 4

        #declare Small_Branch_Size_Min = 2
        #declare Small_Branch_Size_Max = 4

//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// There are five different leaf styles to choose from.
// These five numbers are constants - don't change them!

        #declare Leaf_Type_Realistic = 0
        #declare Leaf_Type_Fast      = 1
        #declare Leaf_Type_Strange   = 2
        #declare Leaf_Type_Torus     = 3
        #declare Leaf_Type_Mesh      = 4

// Change the Leaf_Type variable right here.

        #declare Leaf_Type = Leaf_Type_Mesh

// Set the complete texture for the leaves.

        #declare Leaf_Texture = texture { pigment { LimeGreen } }

//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// Now, here's PART TWO, where the tree is built...
//
// Don't change anything below here (unless you really want to!).
//
//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// Don't change these - they are calculated from the above variables.

        #declare Bmin = Branch_Minimum_Angle
        #declare Bmax = Branch_Maximum_Angle - Branch_Minimum_Angle

        #declare Large_Branch_Size_Range =
        Large_Branch_Size_Max - Large_Branch_Size_Min

        #declare Medium_Branch_Size_Range =
        Medium_Branch_Size_Max - Medium_Branch_Size_Min

        #declare Small_Branch_Size_Range =
        Small_Branch_Size_Max - Small_Branch_Size_Min

//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// Create one of the five leaf styles.
//
// Note: To save memory, the leaves do not get a texture until the
//       very end of the Complete_Tree union. That way, all of the leaves
//       can share one texture. This saves a LOT of memory. You can
//       change this around if you want to, and put the textures up
//       here (memory is cheap, I know, I know!).
//
//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// Now create the actual REALISTIC (sort-of) leaf.

        #if ( Leaf_Type = Leaf_Type_Realistic )

        #declare One_Leaf =

        union{

                // Leaf part #1
                sphere{ 0, 3 scale < 0.4, 0.1, 1.0 >
                        translate z * 3
                        rotate x * -30
                        rotate y * 0 }

                // Leaf part #2
                sphere{ 0, 3 scale < 0.4, 0.1, 1.0 >
                        translate z * 3
                        rotate x * -30
                        rotate y * 120 }

                // Leaf part #3
                sphere{ 0, 3 scale < 0.4, 0.1, 1.0 >
                        translate z * 3
                        rotate x * -30
                        rotate y * 240 }

                // Scale the entire union. Make the leaf small for
                // testing branch placement, then make it big!

                        scale 0.2

        } // End of union.

        #end

// Now create the actual FAST leaf.

        #if ( Leaf_Type = Leaf_Type_Fast )

                #declare One_Leaf = sphere{ 0, 0.5 }

        #end

// Now create the actual STRANGE leaf. Change this to anything!

        #if ( Leaf_Type = Leaf_Type_Strange )

        #declare One_Leaf = union {

                sphere { 0, 0.3 }

                difference {

                        box { < -1, -1, -1 > < 1, 1, 1 > }

                        sphere { 0, 1.3 }

                } // End of difference.

        } // End of union.

        #end

// Now create the actual TORUS leaf. This one has a texture!!!

        #if ( Leaf_Type = Leaf_Type_Torus )

        #declare One_Leaf = union {

                torus { 1, 0.25 pigment { checker Red, Yellow scale 10 } }

                cylinder { < -1, 0,  0 > < 1, 0, 0 >, 0.1
                pigment { checker Blue, Cyan scale 0.1 } }

                cylinder { <  0, 0, -1 > < 0, 0, 1 >, 0.1
                pigment { checker Blue, Cyan scale 0.1 } }

        } // End of union.

        #end

// Create the MESH leaf object - with 200 little triangles!!!
        
        #if ( Leaf_Type = Leaf_Type_Mesh )

        #declare One_Leaf = mesh {

        #declare A = 1 #while ( A <= 200 )
        
                // Calculate random location for first point.
                #declare X1 = ( rand(R) * 2 ) - 1
                #declare Y1 = ( rand(R) * 2 ) - 1
                #declare Z1 = ( rand(R) * 2 ) - 1

                // Move a little way from *first* point.
                #declare X2 = X1 + ( rand(R) * 0.5 ) - 0.25
                #declare Y2 = Y1 + ( rand(R) * 0.5 ) - 0.25
                #declare Z2 = Z1 + ( rand(R) * 0.5 ) - 0.25

                // Move a little way from *first* point.
                #declare X3 = X1 + ( rand(R) * 0.5 ) - 0.25
                #declare Y3 = Y1 + ( rand(R) * 0.5 ) - 0.25
                #declare Z3 = Z1 + ( rand(R) * 0.5 ) - 0.25
                
                // Make the triangle.
                triangle { <X1, Y1, Z1>, <X2, Y2, Z2>, <X3, Y3, Z3> }

        #declare A = A + 1 #end

        } // End of mesh.

        #end

//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// Now create the tree trunk.

        #declare Tree_Trunk = cone {
                < 0, 0, 0 >, 1.00 < 0, Tree_Trunk_Size, 0 >, 0.80
                pigment { Brown_Agate scale 0.1 } }

//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// Now we begin the CONFUSING nested loops. They create a big union
// called "Complete_Tree". We start with one trunk, and add some large
// branches to it. Then, for each large branch, we add a few medium
// branches - each one is at a random angle. Then, in the inner loop,
// we add small branches. Each one of the small branches gets one leaf
// object attached to it. Whew!
//
// The loops MUST be nested like this - it's the only way to give each
// and every branch a random tilt (relative to the parent branch).
//
// The loops are NOT indented properly, because that put the inner loop
// way off the screen, making things even more confusing!

        #declare Complete_Tree =

        union {

        object { Tree_Trunk }

        #declare A = 0 #while ( A < 360 )

        union {

// Make one large branch.

        #declare This_Large_Branch_Size =
                ( rand(TREE_RAND) * Large_Branch_Size_Range ) +
                Large_Branch_Size_Min

        cone {<0,0,0>,0.60 <0,This_Large_Branch_Size,0>,0.40
                pigment{Brown_Agate scale 0.1 } }

// Loop to put medium branches on that large branch.

        #declare B = 0 #while ( B < 360 )

        union {

        #declare This_Medium_Branch_Size =
                ( rand(TREE_RAND) * Medium_Branch_Size_Range ) +
                Medium_Branch_Size_Min

        cone { <0,0,0>,0.30 <0,This_Medium_Branch_Size,0>,0.20
                pigment{Brown_Agate scale 0.1 } }

// Loop to put small branches on that medium branch.

        #declare C = 0 #while ( C < 360 )

        #declare This_Small_Branch_Size =
                ( rand(TREE_RAND) * Small_Branch_Size_Range ) +
                Small_Branch_Size_Min

        union {

        cone { <0,0,0>,0.10 <0,This_Small_Branch_Size,0>,0.05
                pigment { Brown_Agate scale 0.1 }
                } // End of cone.

// The leaf at the end of the small branch.

        object { One_Leaf translate y * This_Small_Branch_Size }

        // First, spin the vertical branch to a random angle.
        // The branch doesn't really change - this actually
        // just spins the leaf around!

                #declare SpinAngle = (rand(TREE_RAND)*360)
                rotate y * SpinAngle

        // Now, tilt it over a little.
                #declare BranchAngle = (rand(TREE_RAND)*Bmax)+Bmin
                rotate z * BranchAngle

        // Rotate it into place, with a little random wiggle.
                #declare Wiggle=(rand(TREE_RAND)*20)-10
                rotate y * (C + Wiggle)

        // Move it up to the top of the Medium_Branch.
                translate y * This_Medium_Branch_Size

        } // End of object.

        #declare C = C + ( 360 / Number_Of_Small_Branches ) #end

        #declare BranchAngle = (rand(TREE_RAND)*Bmax)+Bmin
        rotate z * BranchAngle

        #declare Wiggle=(rand(TREE_RAND)*20)-10
        rotate y * ( B + Wiggle )

        // Move the Medium_Branch up to the top of the Large_Branch.
                translate y * This_Large_Branch_Size

        } // End of union.

        #declare B = B + ( 360 / Number_Of_Medium_Branches ) #end

        #declare BranchAngle = (rand(TREE_RAND)*Bmax)+Bmin
        rotate z * BranchAngle

        #declare Wiggle=(rand(TREE_RAND)*20)-10
        rotate y * ( A + Wiggle )

        // Move the Large_Branch up to the top of the Tree_Trunk.
                translate y * Tree_Trunk_Size

        } // End of union.

        #declare A = A + ( 360 / Number_Of_Large_Branches ) #end

        // Now, turn all of the leaves green! This doesn't change the
        // branches, because they already have a pigment.

                texture { Leaf_Texture }

    } // end of Complete_Tree union

//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// Now return to your main file and show "Complete_Tree".
//
//------------------------------------------------------------------->
//------------------------------------------------------------------->
//
// End of this file - bye!

