/*
	Generate the rib file for the Ringworld, the Shadow Squares and
	the Sun.

	Copyright, James W. Williams, April 1995
*/

#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "ri.h"
/* Basic Ringworld geometry is in rw.h */
#include "rw.h"	

/*
	The ring lies in the XY plane, centered on Z = 0.
	The Sun is at location (0, 0, 0) with the axis along the Z axis.
*/

/* Define the Shadow Squares */
static RtPoint  SS_points[4] = {
				{ 0, -SS_length/2, -SS_width/2 },
				{ 0, -SS_length/2,  SS_width/2 },
				{ 0,  SS_length/2,  SS_width/2 },
				{ 0,  SS_length/2, -SS_width/2 },
				};

/* some base colors */

RtColor SUNcolor	= { 1.0, 1.0, 0.8 };
RtColor SScolor		= { 0.05, 0.05, 0.05 };
RtColor MOONcolor	= { 1.0, 1.0, 0.2 };
RtColor White	= { 1.0, 1.0, 1.0 };
RtColor Red	= { 1.0, 0.0, 0.0 };

int
main(int argc, char **argv)
{
	int i, c, gflag = 0, errflg = 0;
	/* phase is used to adjust where noon falls on the ring. */
	float	SS_Kd = 0.2,	/* very low, for the Shadow Squares */
		phase = 9,	/* 0 == Midnight; 3 == Terminator; 9 == Noon */
		sunIntensity = 4e4,
		coronaRadius,
		earthRadius = 0.006371,
		moonRadius = 0.002;
	RtFloat	sweep;		/* used to make gap in ring. */
	RtPoint transparent = { 0, 0, 0 };
	RtFloat landscale = 8.0,
		octaves = 7,
                sealevel = -.1, /* land vs. sea.  more + means more sea */
		multifractal = 0;
	char 	*shadows = "on", *opaque = "opaque", *texturename;
	RtLightHandle	sunHandle1, sunHandle2, sunHandle3, sunHandle4,
	    sunHandle5, sunHandle6;
	extern char *optarg;
	extern int optind;

	/* Parse arguments. */
        while ((c = getopt(argc, argv, "gp:")) != EOF) {
		switch (c) {
		case 'g':	/* put a gap in the ring */
			gflag++;
			break;
                case 'p':	/* set the phase for time of day */
			phase = atof(optarg);
			break;
                case '?':
			errflg++;
		}
	}
	if (errflg) {
               (void)fprintf(stderr, "usage: ring [-p <phase>] ribfile\n");
               exit (2);
	}

        RiBegin(RI_NULL);

        RiDeclare("octaves", "uniform float");
        RiDeclare("sealevel", "uniform float");
        RiDeclare("threshold_clouds", "uniform float");
	RiDeclare("landscale", "uniform float");
	RiDeclare("multifractal", "uniform float");
        RiDeclare("density", "uniform float");
        RiDeclare("falloff", "uniform float");
        RiDeclare("minstepsize", "uniform float");
        RiDeclare("maxstepsize", "uniform float");
        RiDeclare("use_lighting", "uniform float");

	/* Create the Sun */
#define AREASUN
#ifdef AREASUN
	RiAttributeBegin();
	/* RiRotate(45, 0, 0, 1); position so that y=0 on ring sees three
				   sections of the sun */
	RiColor(SUNcolor);
	RiSurface("constant", RI_NULL);
	RiAttributeBegin();
	  opaque = "none";
          RiAttribute("render", "casts_shadows", &opaque, RI_NULL);
	  sunHandle1 =
	      RiAreaLightSource("areasun", "intensity", &sunIntensity, RI_NULL);
	  RiSphere(SUN_radius, -SUN_radius/2, SUN_radius/2, (RtFloat)90, RI_NULL);
	RiAttributeEnd();

	RiRotate(90, 0, 0, 1);
	RiAttributeBegin();
	  opaque = "none";
          RiAttribute("render", "casts_shadows", &opaque, RI_NULL);
	  sunHandle2 =
	      RiAreaLightSource("areasun", "intensity", &sunIntensity, RI_NULL);
	  RiSphere(SUN_radius, -SUN_radius/2, SUN_radius/2, (RtFloat)90, RI_NULL);
	RiAttributeEnd();
	
	RiRotate(90, 0, 0, 1);
	RiAttributeBegin();
	  opaque = "none";
          RiAttribute("render", "casts_shadows", &opaque, RI_NULL);
	  sunHandle3 =
	      RiAreaLightSource("areasun", "intensity", &sunIntensity, RI_NULL);
	  RiSphere(SUN_radius, -SUN_radius/2, SUN_radius/2, (RtFloat)90, RI_NULL);
	RiAttributeEnd();
	
	RiRotate(90, 0, 0, 1);
	RiAttributeBegin();
	  opaque = "none";
          RiAttribute("render", "casts_shadows", &opaque, RI_NULL);
	  sunHandle4 =
	      RiAreaLightSource("areasun", "intensity", &sunIntensity, RI_NULL);
	  RiSphere(SUN_radius, -SUN_radius/2, SUN_radius/2, (RtFloat)90, RI_NULL);
	RiAttributeEnd();

	/* South polar cap */
	RiRotate(90, 0, 0, 1);
	RiAttributeBegin();
	  opaque = "none";
          RiAttribute("render", "casts_shadows", &opaque, RI_NULL);
	  sunHandle5 =
	      RiAreaLightSource("areasun", "intensity", &sunIntensity, RI_NULL);
	  RiSphere(SUN_radius, -SUN_radius, -SUN_radius/2, (RtFloat)360, RI_NULL);
	RiAttributeEnd();

	/* North polar cap */
	RiRotate(90, 0, 0, 1);
	RiAttributeBegin();
	  opaque = "none";
          RiAttribute("render", "casts_shadows", &opaque, RI_NULL);
	  sunHandle6 =
	      RiAreaLightSource("areasun", "intensity", &sunIntensity, RI_NULL);
	  RiSphere(SUN_radius, SUN_radius/2, SUN_radius, (RtFloat)360, RI_NULL);
	RiAttributeEnd();
	RiAttributeEnd();

#define CORONA 1
#ifdef CORONA
	/* Create the Corona */
	RiAttributeBegin();
	opaque = "none";
        RiAttribute("render", "casts_shadows", &opaque, RI_NULL);
	RiColor(SUNcolor);
	RiSurface("glow", RI_NULL);
	coronaRadius = SUN_radius * 1.2;
	RiSphere(coronaRadius, -coronaRadius, coronaRadius, (RtFloat)360, RI_NULL);
	RiAttributeEnd();
#endif
	/* Turn on all the Sun parts */
	RiIlluminate(sunHandle1, RI_TRUE);
	RiIlluminate(sunHandle2, RI_TRUE);
	RiIlluminate(sunHandle3, RI_TRUE);
	RiIlluminate(sunHandle4, RI_TRUE);
/*	RiIlluminate(sunHandle5, RI_TRUE);
	RiIlluminate(sunHandle6, RI_TRUE);
*/
#else	/* point light sun - actual point light is output by mkrib */
        /* The Sun is transparent and surrounds a point light source */
        RiAttributeBegin();
        RiColor(SUNcolor);
        RiOpacity(transparent);
        RiSurface("sun", RI_NULL);
        RiSphere(SUN_radius, -SUN_radius, SUN_radius, (RtFloat)360, RI_NULL);
        RiAttributeEnd();
#endif

	/* Create the Shadow Squares */
	RiAttributeBegin();
	opaque = "opaque";
        RiAttribute("render", "casts_shadows", &opaque, RI_NULL);
	RiColor(SScolor);
	RiSurface("matte", "Kd", &SS_Kd, RI_NULL);
	for (i = 0; i < SS_number; i++) {
		RiTransformBegin();
		RiRotate(i * 360.0/SS_number + phase, 0, 0, 1);
		RiTranslate(SS_radius, 0, 0);
		RiPolygon(4, RI_P, (RtPointer) SS_points, RI_NULL);
		RiTransformEnd();
	}
	RiAttributeEnd();

	/* Create the Ring */
	RiAttributeBegin();
	opaque = "none";
        RiAttribute("render", "casts_shadows", &opaque, RI_NULL);
	multifractal = 1;
	RiSurface("ringsurf",
		"octaves", &octaves,
		"sealevel", &sealevel,
		"landscale", &landscale,
		"multifractal", &multifractal,
		RI_NULL);
	sweep = 360.0;
	if (gflag) {
		RiRotate(1.0, 0, 0, 1);
		sweep = 358.0;
	}
	RiCylinder(RW_radius, -RW_width/2, RW_width/2, sweep, RI_NULL);
	RiAttributeEnd();

	RiEnd();
	exit(0);
}

