/*
 * A surface shader for the Ringworld inner surface.
 * This is basically the same as the fBm planetary surface shader from
 * the earlier project.  I have included the multifractal code from Ken's
 * "terran" (sic) planetary shader.
 *
 * To correctly shade the underside of the Ringworld, I check the direction
 * of the normal and shade with "scrith" color if we are looking at the
 * outside of the ring.
 *
 * Copyright, James W. Williams, April 1995
 */

#define snoise(x) (2*(noise(x)-0.5))

surface
ringsurf(float	Ka = 0.3,
		Kd = 0.5,
		spectral_exp = 0.5,
		lacunarity = 2,
		octaves = 7,
		dist_scale = 0.2,
		multifractal = 0,
		sealevel = 0, /* a NEGATIVE value means LESS sea. */
		landscale = 8,
		threshold_clouds = 0.0)
{
	float	i, o, l, landOrSea, land, weight, value, cloudscale;
	float	octaves_clouds = octaves, omega_clouds = 0.5, value_clouds;
	point	Nf, PP, PP_clouds, Psh, tp;
	color	Cg, Ct,
		water = color (0.05, 0.05, 0.2),
		scrith = color (0.1, 0.1, 0.1),
		cloudcolor = color (0.9, 0.9, 0.9);

	Nf = faceforward(normalize(N), I);
	Psh = transform("shader", P);

	if ((Ng . I) < 0) { /* looking at the back-side */
		Ct = scrith;  /* Really need some sort of cool scrith shader */
	}
	else {
		/* value is an offset added to sealevel that keeps the biggest
		 * lakes from forming near the ring edges.  The regular,
		 * unmodified sealevel is used for the fine scale lakes.
		 */
		value = abs(v - 0.5);
		value = sealevel - (4 * value * value * value);
		PP = landscale * Psh;
		landOrSea = 0;
		o = 1;
		l = 1;
		for (i = 0;  i < octaves;  i += 1) {
			landOrSea += o * snoise (l * PP);
			l *= lacunarity;
			o *= spectral_exp;
		}
		land = filterstep(value, landOrSea); 

		if (land > 0) { /* if on land, compute a finer grain shader */
			PP = 6 * landscale * Psh;
			landOrSea = 0;
			o = 1;
			l = 1;
			for (i = 0; i < octaves; i += 1) {
				landOrSea += o * snoise(l * PP);
				l *= lacunarity;
				o *= spectral_exp;
			}
			land = filterstep(sealevel, landOrSea); 
			if (land > 0) { /* shade land with a very fine
					   grain texture */
				PP = 100 * landscale * Psh;
				value = 0;
				o = 1;
				l = 1;
				for (i = 0;  i < octaves;  i += 1) {
					value += o * snoise (l * PP);
					l *= lacunarity;
					o *= spectral_exp;
				}
				value = abs(value) / 1.2;
				Cg = spline(value,
					color(0.05, 0.05, 0.2), /* water */
					color(0.05, 0.05, 0.2),
					color(0.12, 0.20, 0.10), /* green */
					color(0.27, 0.19, 0.17), /* brown */
					color(0.94, 0.83, 0.25), /* desert */
					color(0.12, 0.20, 0.10), /* green */
					color(0.12, 0.20, 0.10)
					);
			}
			else {
				Cg = water;
			}
		}
		else {
			Cg = water;
        	}
		/* scale clouds bigger than land, but less so along Z,
		 * which stretches the clouds
		 */
		/* clouds are bigger than fine grain land features */
		cloudscale = 8 * landscale;
		PP_clouds = point(cloudscale, cloudscale, cloudscale) * Psh;
		value_clouds = 0;
		l = 1;  o = 1;  value_clouds = 0;
		for (i = 0;  i < octaves_clouds; i += 1) {
			value_clouds += o * snoise (PP_clouds*l);
			l *= lacunarity;
			o *= omega_clouds;
		}

        	Ct = mix(Cg, cloudcolor,
			smoothstep(threshold_clouds, 1, value_clouds)); 
	}

	/* Matte model */
	Oi = Os;
	Ci = Os * Ct * (Ka * ambient() + Kd * diffuse(Nf));
}

