//////////////////////////////////////////////////////////////////////
// Matthew Buckley                                                  //
// sauron@cmu.edu                                                   //
// Catmull-Rom Spline POV-Ray "Plugin"                              //
// Begun: 11-11-96                                                  //
// Completed: 12-6-96                                               //
// Version: 2                                                       //
// For: POV-Ray 3.0                                                 //
//                                                                  //
// Usage: declare the following:                                    //
//                                                                  //
//  +------------------------+-----------+--------------------------++--+
//  |  name                  | type      | contents                     |
//  +------------------------+-----------+------------------------------+
//  |  Points                | float     | Number of control points in  |
//  |                        |           | spline: IF THIS IS NOT 10,   |
//  |                        |           | CHANGE THE CASE STATEMENT    |
//  |                        |           | AND THE DECLARATION OF       |
//  |                        |           | ControlPoints ACCORDINGLY    |
//  |- - - - - - - - - - - - + - - - - - + - - - - - - - - - - - - - - -|
//  |  Control0 .. Controln  | vector    | The control points           |
//  |                        |           |      n = ( Points - 1 )      |
//  |- - - - - - - - - - - - + - - - - - + - - - - - - - - - - - - - - -|
//  |  splinethickness       | float     | thickness of tube created    |
//  |                        |           | by spline                    |
//  |- - - - - - - - - - - - + - - - - - + - - - - - - - - - - - - - - -|
//  |  smoothness            | float     | number of subdivisions       |
//  |                        |           | between control points       |
//  |- - - - - - - - - - - - + - - - - - + - - - - - - - - - - - - - - -|
//  |  looseness             | float     | how loose should the spline  |
//  |                        |           | be between control points    |
//  +------------------------+-----------+--------------------------++--+
//                                                                  //
// Then include this file.                                          //
//. . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . //
// Show the spline like this:                                       //
//                                                                  //
//    object {                                                      //
//      CatmullRomSpline                                            //
//      texture { ... }                                             //
//    }                                                             //
//                                                                  //
// Show the control points like this:                               //
//                                                                  //
//    object {                                                      //
//      ControlPoints                                               //
//    }                                                             //
//                                                                  //
//  vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv  //
//  > The spline will not touch Control0 or the last two Control <  //
//  > points.  This is due to the way the slopes are calculated. <  //
//  >  eg                                                        <  //
//  >  (0)  (1)==(2)==(3)==...==(n-4)==(n-3)==(n-2)  (n-1)  (n)  <  //
//  >                                                            <  //
//  ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^  //
//                                                                  //
//. . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . . //
//                                                                  //
// Variable names used ( Other than those named above):             //
//    yz lastxyz tt lastTT prmtr tval                               //
//    currslope nextslope                                           //
//    pc cc nc n2c                                                  //
//    Aval Bval Cval Dval                                           //
//    ControlPoints CatmullRomSpline                                //
//    renderingthisfile                                             //
//                                                                  //
//////////////////////////////////////////////////////////////////////
      //                                                         //
     //  If you use this for anything, please give me credit.   //
    //   If you make any improvements, please mail me.         //
   //         Have Fun                                        //
  //                                        Matthew Buckley  //
 //                                                         //
/////////////////////////////////////////////////////////////

  /************/
 /* Defaults */
/************/
#ifndef (Points)
  #declare Points = 10
#end //ifndef(Points)

#ifndef (smoothness)
  #declare smoothness = 10
#end //ifndef(smoothness)

#ifndef (looseness)
  #declare looseness = 2
#end //ifndef(looseness)

#ifndef (splinethickness)
  #declare splinethickness = 0.4
#end //ifndef(splinethickness)

#ifndef (Control0)
  #declare Control0 = <-0.1,0,0>
  #declare Control1 = <0,0,0>
  #declare Control2 = <1,-1,-1>
  #declare Control3 = <2,0,0>
  #declare Control4 = <3,-1,1>
  #declare Control5 = <4,0,0>
  #declare Control6 = <5,3,-1>
  #declare Control7 = <6,0,0>
  #declare Control8 = <7,-1,1>
  #declare Control9 = <8,-2,2>

  background { color rgb<1,1,1> }
  camera {
    location <4,5,-5>
    look_at<4,0,0>
  }
  light_source {
    <-30,30,-30>
    color rgb<1,1,1>
  }

  light_source {
    <30,30,-30>
    color rgb<1,1,1>
  }

  #declare renderingthisfile = 1

#end  //ifndef (Control0)

  /****************/
 /* END DEFAULTS */
/****************/


  /***********************/
 /* The Spline Function */
/***********************/
#declare CatmullRomSpline =
union {

  #declare lastTT = 0
  #declare lastxyz = Control1 + 0.001

  #declare nextslope = (Control2 - Control0 + 0.0001) / 2

  #declare prmtr = 1.0
  #while (prmtr <= (Points - 3))
    #declare tt = int(prmtr)
    #if (tt != lastTT)  // Only recalculate cubics if at a control point
      #switch (tt)
        #case (1)
          #declare pc = Control0
          #declare cc = Control1
          #declare nc = Control2
          #declare n2c = Control3
        #break
        #case (2)
          #declare pc = Control1
          #declare cc = Control2
          #declare nc = Control3
          #declare n2c = Control4
       #break
        #case (3)
          #declare pc = Control2
          #declare cc = Control3
          #declare nc = Control4
          #declare n2c = Control5
        #break
        #case (4)
          #declare pc = Control3
          #declare cc = Control4
          #declare nc = Control5
          #declare n2c = Control6
        #break
        #case (5)
          #declare pc = Control4
          #declare cc = Control5
          #declare nc = Control6
          #declare n2c = Control7
        #break
        #case (6)
          #declare pc = Control5
          #declare cc = Control6
          #declare nc = Control7
          #declare n2c = Control8
        #break
        #case (7)
          #declare pc = Control6
          #declare cc = Control7
          #declare nc = Control8
          #declare n2c = Control9
        #break
        // Add more here or remove some off the end.  You should be able to figure out the pattern. ;)
      #end

      #declare lastTT = tt

      // Slopes of current and next points

      #declare currslope = nextslope
      #declare nextslope = (n2c - cc + 0.0001) / 2

      #declare l  = (1 / looseness)
      #declare l2 = (1 / (looseness * looseness))
      #declare l3 = (1 / (looseness * looseness * looseness))

      // Coefficients for cubics
      #declare Aval = ( (l2 * (currslope + nextslope)) + (l3 * 2 * (cc - nc)) )
      #declare Bval = ( (l2 * 3 * (nc - cc)) - (l * (nextslope + (2 * currslope))) )
      #declare Cval = currslope
      #declare Dval = cc

    #end //if (tt != lastTT)

    #declare tval = (prmtr - tt) * looseness
    // The cubics
    #declare xyz = ((Aval * tval * tval * tval) + (Bval * tval * tval) + (Cval * tval) + Dval)

    #declare prmtr = prmtr + (1/smoothness)

    // The splined objects
    sphere { xyz, splinethickness }
    cylinder{ lastxyz, xyz, splinethickness }



    #declare lastxyz = xyz
#end

} // CatmullRomSpline



// So you want to see where your control points are?
#declare ControlPoints =
union {
  sphere {
    Control0, (splinethickness + 0.05)
  }
  sphere {
    Control1, (splinethickness + 0.05)
  }
  sphere {
    Control2, (splinethickness + 0.05)
  }
  sphere {
    Control3, (splinethickness + 0.05)
  }
  sphere {
    Control4, (splinethickness + 0.05)
  }
  sphere {
    Control5, (splinethickness + 0.05)
  }
  sphere {
    Control6, (splinethickness + 0.05)
  }
  sphere {
    Control7, (splinethickness + 0.05)
  }
  sphere {
    Control8, (splinethickness + 0.05)
  }
  sphere {
    Control9, (splinethickness + 0.05)
  }

  // Add more here or comment out some at the end.  You should be able to figure out the pattern. ;)

  texture {
    pigment { color rgbt<0,1,0,0.50> }
    finish { ior 1.5 }
  }
}

  /////////////////////////////////////////////////////
 // If rendering this file, put the demo spline in. //
/////////////////////////////////////////////////////
#ifdef (renderingthisfile)
  object {
    CatmullRomSpline
    texture { pigment { checker color rgb<1,0.5,0> color rgb<1,1,0> } }
  }
  object { ControlPoints }
#end
