// Persistence of Vision Ray Tracer Scene Description File

// File: pen1.pov
// Vers: 1.00
// Desc: Retractable ball point pen, blue. Based on the Pilot PBS. Created for the
//       May-June round of the internet raytracing comp.
// Date: May 1997
// Auth: Chris Hernandez


/* -------------------------------- Declare Global Attributes ----------------------------- */

#declare Pen_OuterRadius = 0.2
#declare Pen_InnerRadius = 0.16

#declare Pen_Chrome = texture {
    pigment { color rgb 0.85 }
    finish {
        ambient 0.2
        diffuse 0.9
        reflection 0.5
        brilliance 8
        specular 1.0
        roughness 0.005
        metallic
    }
}

#declare Pen_TransPlastic = texture {
    pigment { color rgbft<0.8, 0.8, 0.9, 0.9, 0.1> }
    finish {
        ambient 0.2
        diffuse 0.2
        reflection 0.2
        refraction on
        ior 1.5
        specular 0.3
        roughness 0.05
        caustics 1
    }
}

#declare Pen_MattPlastic = texture {
    pigment { color rgb < 0.5, 0.5, 0.7 > }
    finish {
       ambient 0.3
       diffuse 0.25
       specular 0.35
       roughness 0.05
       reflection 0.05
    }
}


/* ------------------------------------- Pen Body ----------------------------------- */

#declare PenBody_Length = 3.5
#declare PenBody_PlaneDist = 0.19

#declare PenBody = object {
   difference {
      cylinder { -(PenBody_Length/2)*z,  (PenBody_Length/2)*z,  Pen_OuterRadius hollow }
      plane { y, -PenBody_PlaneDist rotate z*000 }
      plane { y, -PenBody_PlaneDist rotate z*045 }
      plane { y, -PenBody_PlaneDist rotate z*090 }     //  Give the pen body 8 sides,
      plane { y, -PenBody_PlaneDist rotate z*135 }     //  while making the joins between
      plane { y, -PenBody_PlaneDist rotate z*180 }     //  between the sides rounded.
      plane { y, -PenBody_PlaneDist rotate z*225 }
      plane { y, -PenBody_PlaneDist rotate z*270 }
      plane { y, -PenBody_PlaneDist rotate z*315 }
      //plane { y, -PenBody_PlaneDist rotate z*240 }
      //plane { y, -PenBody_PlaneDist rotate z*270 }
      //plane { y, -PenBody_PlaneDist rotate z*300 }
      //plane { y, -PenBody_PlaneDist rotate z*330 }
      cylinder { -((PenBody_Length/2)+0.1)*z,  ((PenBody_Length/2)+0.1)*z,  Pen_InnerRadius }
      rotate x*90
      translate y*(PenBody_Length/2)
   }
}



/* ------------------------------------- Pen Grip ----------------------------------- */

#declare PenGrip_Units = 4              //  Number of units used to make the grip.
#declare PenGrip_UnitRadius = 0.02      //  Minor radius of each torus.
#declare PenGrip_UnitCount  = 10        //  Number of torii in each Pen Grip "unit".
#declare PenGrip_UnitOffset = 0.038     //  Translation offset for each torus.
#declare PenGrip_UnitLength = PenGrip_UnitOffset * PenGrip_UnitCount


// Declare the basic unit for construction of Pen grip. Approx 0.38u long.

#declare PenGrip_Unit = object {
   merge {
      torus { Pen_OuterRadius, PenGrip_UnitRadius   translate -y*(PenGrip_UnitOffset*00) }
      torus { Pen_OuterRadius, PenGrip_UnitRadius   translate -y*(PenGrip_UnitOffset*01) }
      torus { Pen_OuterRadius, PenGrip_UnitRadius   translate -y*(PenGrip_UnitOffset*02) }
      torus { Pen_OuterRadius, PenGrip_UnitRadius   translate -y*(PenGrip_UnitOffset*03) }
      torus { Pen_OuterRadius, PenGrip_UnitRadius   translate -y*(PenGrip_UnitOffset*04) }
      torus { Pen_OuterRadius, PenGrip_UnitRadius   translate -y*(PenGrip_UnitOffset*05) }
      torus { Pen_OuterRadius, PenGrip_UnitRadius   translate -y*(PenGrip_UnitOffset*06) }
      torus { Pen_OuterRadius, PenGrip_UnitRadius   translate -y*(PenGrip_UnitOffset*07) }
      torus { Pen_OuterRadius, PenGrip_UnitRadius   translate -y*(PenGrip_UnitOffset*08) }
      torus { Pen_OuterRadius, PenGrip_UnitRadius   translate -y*(PenGrip_UnitOffset*09) }
   }
}

#declare PenGrip = object {
   merge {
      object { PenGrip_Unit  translate -y*(PenGrip_UnitLength*00) }
      object { PenGrip_Unit  translate -y*(PenGrip_UnitLength*01) }
      object { PenGrip_Unit  translate -y*(PenGrip_UnitLength*02) }
      object { PenGrip_Unit  translate -y*(PenGrip_UnitLength*03) }
   }
}



/* ------------------------------------- End Cover ---------------------------------- */

#declare PenEC_Length     = 0.6
#declare PenEC_LengthVect = -PenEC_Length*y

#declare PenEC = object {
   union {
      cone { 0.00*PenEC_LengthVect,  0.195, 0.25*PenEC_LengthVect, 0.17   open }
      cone { 0.25*PenEC_LengthVect,  0.17,  0.28*PenEC_LengthVect, 0.15   open }
      cone { 0.28*PenEC_LengthVect,  0.15,  0.96*PenEC_LengthVect, 0.06   open }
      cone { 0.96*PenEC_LengthVect,  0.06,  1.00*PenEC_LengthVect, 0.04   open }
   }
}



/* -------------------------------------- Pen Nib ---------------------------------- */

#declare PenNib = object {
   union {
      cylinder { 0*y,  -0.06*y,  0.04 }
      cone { -0.06*y,  0.04,  -0.14*y, 0.01 }
      sphere { <0, -0.14, 0> 0.01 }
   }
}



/* ------------------------------------ Push Button ---------------------------------- */

#declare PenPB = object {
   union {
      torus    { (Pen_OuterRadius*0.7)-0.05, 0.05 translate y*0.4 texture{ Pen_Chrome } }
      cylinder { 0.4*y,  0.45*y, (Pen_OuterRadius*0.7)-0.05  texture { Pen_Chrome }     }
      cylinder { 0.4*y,  0.0*y,  Pen_OuterRadius*0.7  texture { Pen_Chrome }            }
      cylinder { 0.0*y, -0.45*y, Pen_OuterRadius*1.0+0.02  texture { Pen_MattPlastic }  }
      box      { < -0.0, -0.05, -0.1> < -0.3, -0.1,  0.1>  texture { Pen_MattPlastic }  }
   }
}



/* ------------------------------------ Pen Handle ---------------------------------- */

#declare PH_Vertex1 = <  0.00, 0.05,  0.00 >
#declare PH_Vertex2 = < -1.65, 0.00,  0.10 >
#declare PH_Vertex3 = <  0.27, 0.00,  0.10 >
#declare PH_Vertex4 = <  0.27, 0.00, -0.10 >
#declare PH_Vertex5 = < -1.65, 0.00, -0.10 >
#declare PH_Vertex6 = < -1.35, 0.025, 0.00 >

#declare PH_Normal1 = <  0.00, 1.00,  0.00 >
#declare PH_Normal2 = < -1.65, 0.50,  0.10 >
#declare PH_Normal3 = <  0.27, 0.25,  0.10 >
#declare PH_Normal4 = <  0.27, 0.25, -0.10 >
#declare PH_Normal5 = < -1.65, 0.50, -0.10 >
#declare PH_Normal6 = < -1.35, 1.00,  0.00 >


// ----- Make the pen handle top have a smooth, rounded shape.

#declare PenHandle_Top = object {
    mesh {
       smooth_triangle { PH_Vertex1 PH_Normal1, PH_Vertex2 PH_Normal2, PH_Vertex3 PH_Normal3 }
       triangle        { PH_Vertex1, PH_Vertex3, PH_Vertex4 }
       smooth_triangle { PH_Vertex1 PH_Normal1, PH_Vertex4 PH_Normal4, PH_Vertex5 PH_Normal5 }
       smooth_triangle { PH_Vertex1 PH_Normal1, PH_Vertex5 PH_Normal5, PH_Vertex2 PH_Normal2 }
    }
}

#declare PenHandle_Endpiece = object {
   difference {
      box { < 0.0, 0.0, -0.1> < -0.15, 0.2,  0.1> }
      cylinder { -0.11*z,  0.11*z,  0.12 }
      scale <1, 2, 1>
   }
}

#declare PenHandle = object {
   union {
      difference {
         box { < -0.2, -0.1, -0.1> < -0.35, -0.25,  0.1> }
         plane { x, -0.25 rotate -z*18.43 }
      }

      object { PenHandle_Endpiece translate <-0.2, -0.65, 0> }
      box { < -0.35, -0.65, -0.1> < -0.32, -1.5,  0.1> }
      object {
         PenHandle_Endpiece
         rotate x*180
         translate <-0.2, -1.5, 0>
         clipped_by { plane {y, -1.75 inverse} }
      }

      object {
         union {
            difference {
               box { < 0.0, 0.0, -0.1> < -0.15, -0.35, 0.1> }
               plane { -x, 0 rotate -z*15.95 }
            }
            cylinder { -0.1*z, 0.1*z, 0.02 translate < -0.125, -0.35, 0 > }
            translate < -0.2, -1.75, 0 >
         }
      }

      object { PenHandle_Top  rotate z*90  translate < -0.35, -0.5, 0 > }
   }
}


/* ------------------------------------ Inner Piece --------------------------------- */

#declare PenIP_Length = 2.0
#declare PenIP_InnerRadius = 0.15

#declare PenIP = object {
   difference {
      cylinder { 0*y, -PenIP_Length*y, Pen_InnerRadius }
      cylinder { 0*y, -PenIP_Length*y, PenIP_InnerRadius }

      box { <-0.05, 0, -1> < 0.05, -0.33*PenIP_Length, 1> }                // Upper cutaway
      cylinder { -z, z, 0.05 translate -0.33*y*PenIP_Length }

      box { <-0.03, -0.85*PenIP_Length, -1> < 0.03, -PenIP_Length, 1> }    // Lower cutaway
      cylinder { -z, z, 0.05 translate -0.85*y*PenIP_Length }
   }
}


/* -------------------------------------- Refill ----------------------------------- */

#declare PenRefill_Length = PenBody_Length + ( PenGrip_UnitLength * PenGrip_Units )
#declare PenRefill_Radius = 0.054


#declare PenRefill_Empty = texture {
    pigment { color rgbft < 0.875, 0.875, 0.9, 0.7, 0.0 > }
    finish {
       ambient 0.6
       diffuse 0.6
       specular 0.5
       roughness 0.1
    }
}

#declare PenRefill_Filled = texture {
    pigment { color rgbft < 0.05, 0.05, 0.15, 0.1, 0.1 > }
    finish {
       ambient 0.2
       diffuse 0.2
    }
}

#declare PenRefill = object {
   union {
      cylinder {
         0.00*y, -0.60*y*PenRefill_Length, PenRefill_Radius
         hollow
         texture { PenRefill_Empty }
      }
      cylinder {
         -0.60*y*PenRefill_Length, -1.00*y*PenRefill_Length, PenRefill_Radius
         hollow
         texture { PenRefill_Filled }
      }
   }
}


/*
   *----------------------------------------------------------------------------------------*
   |                                    Main Pen Object                                     |
   *----------------------------------------------------------------------------------------*
*/

#declare Pen = object {
   union {
      object { PenBody texture { Pen_TransPlastic } }
      object { PenGrip texture { Pen_TransPlastic } }
      object { PenEC texture { Pen_Chrome } translate -y*1.475 }
      object { PenNib texture { Pen_Chrome } translate -y*2.07 }
      object { PenPB translate y*PenBody_Length }
      object { PenHandle texture { Pen_Chrome } translate y*PenBody_Length }
      object { PenIP texture { Pen_MattPlastic } translate y*(PenBody_Length-0.4) }
      object { PenRefill translate y*PenBody_Length }
   }
}



