// Persistence of Vision Ray Tracer Scene Description File
// File: Kepler.pov
// Vers: 3
// Desc: Kepler's Laws
// Date: 5/12/1997
// Auth: Steve Sloan II

#version 3

global_settings {
  assumed_gamma 1.0
}

#include "colors.inc"
#include "textures.inc"

// Use halos: Takes more time to render
#declare use_halo = 1

// Don't use halos
//#declare use_halo = 0

// Use starfield: Takes more time to render
//#declare use_stars = 1

// Don't use starfield
#declare use_stars = 0

/*PHYSICAL UNITS*/
/* Distance */
#declare km = 1                                // kilometers
#declare AU = 1.5e11*km                        // Astronomical Units
#declare m = 1*km/1000                         // meters

/*CONSTANTS*/

/* Orbit Constants */
#declare start1 = 45
#declare start2 = 180.0
#declare increment = 0.1                       // Angle of the "slivers" used to approximate the area of a sector of an ellipse
#declare total_angle = 80.0                    // Total angle of Area1
#declare a = 10
#declare e = 0.5
#declare b = a*sqrt(1 - e*e)
#declare c = sqrt(a*a - b*b)
#declare k = a * ((1.0 / e) - 1.0)


/* Radii */
#declare sun_radius = 0.15*a                   // Radius of Sun
#declare planet_radius = 0.1*a                 // Radius of the planet

/* Orbital path constants */
#declare orbit_path_radius = 0.01*a            // Radius of the path traced out by an orbit
#declare orbit_color = color Red

#declare sun_position = 0

// LnsFlare declarations
#declare cam_loc = <1.8*a, 3*a/4, -0.3*a>      // use the x, y, and z of your camera
#declare light_loc = sun_position              // use the x, y, and z of the light
                                               // you wish to flare
#declare lookat = <0, 0, c - a>                // the look_at vector of your camera
#declare sky_vect = <0,1,0>                    // the sky vector of your camera

// LensFlar declarations
#declare camera_location = cam_loc
#declare camera_look_at = lookat
#declare flare_position = light_loc

camera {
    location cam_loc
    look_at lookat
    sky sky_vect
}

// Call Chris Colefax's LensFlare
#declare flare_type = 2
#declare flare_brightness = 0.3
#declare flare_size = 0.5
#declare flare_color = <1, 1, 0>
#declare flare_rays = false
#include "LensFlar.inc"                        // create the lens flare


#declare area_texture =
texture {
    pigment { color Green filter 0.5 }
}

#declare star_texture =
    pigment {
        granite
        color_map {
            [ 0.000  0.270 color rgb < 0, 0, 0> color rgb < 0, 0, 0> ]
            [ 0.270  0.280 color rgb <.5,.5,.4> color rgb <.8,.8,.4> ]
            [ 0.280  0.470 color rgb < 0, 0, 0> color rgb < 0, 0, 0> ]
            [ 0.470  0.480 color rgb <.4,.4,.5> color rgb <.4,.4,.8> ]
            [ 0.480  0.680 color rgb < 0, 0, 0> color rgb < 0, 0, 0> ]
            [ 0.680  0.690 color rgb <.5,.4,.4> color rgb <.8,.4,.4> ]
            [ 0.690  0.880 color rgb < 0, 0, 0> color rgb < 0, 0, 0> ]
            [ 0.880  0.890 color rgb <.5,.5,.5> color rgb < 1, 1, 1> ]
            [ 0.890  1.000 color rgb < 0, 0, 0> color rgb < 0, 0, 0> ]
        }
        turbulence 1
        sine_wave
        scale .5
    }


/* COMPUTE THE "AREAS" */
// Compute the total ellipse area for the
// arc ranging from start1 to start1+total_angle
#declare area1 = 0.0
#declare theta = start1
#while (theta < mod(start1+total_angle, 360))
    #declare temp = 1.0 + e*cos(radians(theta))
    #declare r = e * k / temp
    #declare wedge_area = degrees(r*r*increment)
    #declare area1 = area1 + wedge_area
    #declare theta = mod(theta + increment, 360) // increment our counter
#end

// Output results of the computation
#render concat("Angle is ", str(theta - start1,5,1), ", Area1 is: ",str(area1,5,1),"\n")

// Determine the total angle it takes to equal the previous area
#declare area2 = 0.0
#declare theta = start2
#while (area2 < area1)
    #declare temp = 1.0 + e*cos(radians(theta))
    #declare r = e * k / temp
    #declare wedge_area = degrees(r*r*increment)
    #declare area2 = area2 + wedge_area
    #declare theta = mod(theta + increment, 360) // increment our counter
#end

// Output results of the computation
#render concat("Angle is ",str(theta - start2,5,1), ", Area2 is: ",str(area2,5,1),"\n")

#declare end_angle2 = mod(theta, 360)


/* DECLARE USEFUL OBJECTS */
/* Orbit plane */
#declare full_orbit =
intersection {
    plane {
        y, 0
        clipped_by{
            cylinder {
                -y, y, 1
                scale <b, a, a>
            }
        }
    }
    translate -c*z
}

// Kepler's Third Law: P^2 = a^3
#declare text_thickness = 0.5
#declare Kepler_formula =
union {
    // "P = a"
    text {
        ttf "timrom.ttf" "P = a" text_thickness, 0
    }

    // Exponent for "P"
    text {
        ttf "timrom.ttf" "2" text_thickness * 10/3, 0
        scale 0.3
        translate <0.65, 0.5, 0>
    }

    // Exponent for "a"
    text {
        ttf "timrom.ttf" "3" text_thickness * 10/3, 0
        scale 0.3
        translate <2.2, 0.5, 0>
    }
}

/* DRAW THE SCENE OBJECTS */

// Create an infinite sphere around scene and allow any texture on it
sky_sphere {
    #if (use_stars = 1)
        pigment { star_texture }
    #else
        pigment { color Black }
    #end
}

/* Draw the orbit */
torus {
    a, orbit_path_radius
    scale <b/a, 1, 1>
    translate -c*z
    pigment { orbit_color }
}

/* Draw the first "area" */
#declare sector_height = 500*km
#declare sector_radius = 2*a
#declare sector_start = start1
#declare sector_end = start1 + total_angle
#include "sector.inc"

object {
    full_orbit
    clipped_by {
        object {
            sector_cutter
        }
    }
    texture { area_texture }
}

/* Position the first planet */
#declare temp = 1.0 + e*cos(radians(start1))
#declare curr_distance = e * k / temp

#include "planet.inc"
object {
    planet
    scale planet_radius
    translate <0, 0, 1.5*curr_distance>
    rotate start1*y
}

/* Draw the second "area" */
#declare sector_height = 500*km
#declare sector_radius = 2*a
#declare sector_start = start2
#declare sector_end = end_angle2
#include "sector.inc"

object {
    full_orbit
    clipped_by {
        object {
            sector_cutter
        }
    }
    texture { area_texture }
}

/* Position the second planet */
#declare temp = 1.0 + e*cos(radians(start2))
#declare curr_distance = e * k / temp

object {
    planet
    scale planet_radius
    translate <0, 0, 1.5*curr_distance>
    rotate start2*y
}

/* Draw the sun */
#include "Sun.inc"
object {
    sun
    translate sun_position
}

/* Draw the formula */
object {
    Kepler_formula
    texture {
        pigment { Blue }
        finish { ambient 0.5 }
    }
    scale a/8
    rotate -90*y
    translate <a, 0.6*a, -0.5*a>
}


