// A small exploration robot is crawling the surface of a far away planet
// and discovers many strange things before it suddenly sees it's own image
// in a mirror...
// This picture was designed for the December 1998 round of the Internet
// Raytracing Competition (see http://www.irtc.org for details)
// ---------------------------------------------------
// Hannes Rohde <rohde@xmailer.informatik.uni-bonn.de>
// visit my Homepage at http://titan.informatik.uni-bonn.de/~rohde/

#include "colors.inc"	// Standard Color definitions
#include "textures.inc"	// Standard Texture definitions
#include "metals.inc"   // nice metal textures
                        
// neccessary for the bubbles in the pipe:
#global_settings { max_trace_level 7 }

#version 3.1;                        
                        
// set viewer's position in the scene
camera {
   location  <-28, 30, -70> // position of camera <X Y Z>
// location  <0,45,-40>
// location  <-28, 23, -60> // position of camera <X Y Z>
   up        y              // which way is +up <X Y Z>
   right     4/3*x          // which way is +right <X Y Z> and aspect ratio
   look_at   <-5, -31, 0>   // point center of view at this point <X Y Z>
// look_at   <0,0,-30>
// look_at   <-5, -34, 0>   // point center of view at this point <X Y Z>
}

#declare SunColor = rgb <.9,.8,.85>;
#declare SunPos   = <-1200,1200,200>;
#declare FlowerVal = 0;

light_source { SunPos SunColor }
light_source { <0,1000,0> rgb <.6,.6,.6> shadowless }

#declare WheelDiameter = 8;
#declare ProfileHeight = 0.7;
#declare WheelCenter = <0,WheelDiameter+ProfileHeight,0>;
#declare AxleDiameter = 1;

#declare MetalTexture = texture { T_Chrome_4C }
#declare RubberTexture = texture {
                            pigment { color rgb <0.2,0.2,0.25> }
                            finish  { diffuse 0.6 }
                         }

// texture for the solar panels:
#declare SolarTexture = texture {
   pigment { rgb <0.07,0.1,0.24> }
   finish {
      ambient 0.5
      diffuse 0.2
      specular 0.9 roughness 0.0025
      metallic 0.14
      reflection .05
      irid {
         0.35 thickness 0.1 turbulence 0.085
       }
   }
}

#declare BoltTexture = texture {
   T_Brass_3B
}

#declare SandColor = rgb <.8,.65,.50>;
#declare WetSandColor = rgb <.4,.32,.20>;
#declare SoakedSandColor = rgb <.2,.16,.08>;

#declare SandFinish = finish {
   crand .27   // Sorry, but this was the easiest way to get it look right :-)
   ambient .6*SunColor
}

#declare AlienSandTexture = texture {
   gradient y
   texture_map {
      [0.15  pigment { SoakedSandColor } finish { SandFinish } ]
      [0.3   pigment { WetSandColor } finish { SandFinish } ]
      [0.32  pigment { SandColor } finish { SandFinish } ]
      [1.0   pigment { SandColor } finish { SandFinish } ]
   }
}

// A little bit of cheating was neccessary here: because of the odd camera angle, a plane of water
// would be visible in the mirrors where the viewer would not expect it.
// Now they show the sky-sphere which looks more credible...

#declare AlienWater = box { <-75,-10,-75>, <75,0,75>
   pigment { rgbf <.8,.8,.8,1> }
   finish {
      ambient 0
      diffuse 0
      specular .8
      roughness 0.0001
      reflection .22
   }
   normal {
      bumps .04
      scale 2.5
   }
   interior {
      ior 2
   }
}

#declare Profile = box { <-0.2,WheelDiameter,-4>,<0.2,WheelDiameter+ProfileHeight,+4> texture { RubberTexture } }
#declare Support = difference {
   box { <-0.75,0,-0.1>, <0.75, 0.8*WheelDiameter,+0.1> }
   cylinder { <0,AxleDiameter+0.25*(0.8*WheelDiameter-AxleDiameter),-0.2>,
              <0,AxleDiameter+0.25*(0.8*WheelDiameter-AxleDiameter),0.2>,0.3 }
   cylinder { <0,AxleDiameter+0.50*(0.8*WheelDiameter-AxleDiameter),-0.2>,
              <0,AxleDiameter+0.50*(0.8*WheelDiameter-AxleDiameter),0.2>,0.3 }
   cylinder { <0,AxleDiameter+0.75*(0.8*WheelDiameter-AxleDiameter),-0.2>,
              <0,AxleDiameter+0.75*(0.8*WheelDiameter-AxleDiameter),0.2>,0.3 }
   texture { MetalTexture }
}

#declare SolarPanel = difference {
   union {
      box { <-4,  0,-4>, <4,0.1, 4> pigment { rgb <0.9,0.6,0.5> } finish { ambient 0.6  } }
      box { <-4,0.1,-4>, <4,0.15,4> texture { SolarTexture } }
   }
   union {
      box { <-6,-1,-2>, <-5,1,2> rotate <0,45,0> }
      box { <-6,-1,-2>, <-5,1,2> rotate <0,135,0> }
      box { <-6,-1,-2>, <-5,1,2> rotate <0,225,0> }
      box { <-6,-1,-2>, <-5,1,2> rotate <0,315,0> }
      pigment { rgb <0.9,0.6,0.5> } finish { ambient 0.6 }
   }
}

#declare Nut = difference {
   union {                   
      box { <-0.57,0,0>, <0.57,0.5,1> }
      box { <-0.57,0,0>, <0.57,0.5,1> rotate y*60 }
      box { <-0.57,0,0>, <0.57,0.5,1> rotate y*120 }
      box { <-0.57,0,0>, <0.57,0.5,1> rotate y*180 }
      box { <-0.57,0,0>, <0.57,0.5,1> rotate y*240 }
      box { <-0.57,0,0>, <0.57,0.5,1> rotate y*300 }
   }
   cylinder { <0,-1,0>, <0,1,0>, 0.6 }
   texture { BoltTexture }
}

#declare Wheel = union {
   difference {
      cylinder { <0,0,-4>, <0,0,4>, WheelDiameter }
      cylinder { <0,0,-4.1>, <0,0,4.1>, 0.95*WheelDiameter }
      texture { RubberTexture }
   }
   difference {
      cylinder { <0,0,-3.8>, <0,0,3.8>, 0.95*WheelDiameter }
      cylinder { <0,0,-4>, <0,0,-3.7>, 0.92*WheelDiameter }
      cylinder { <0,0, 4>, <0,0, 3.7>, 0.90*WheelDiameter }
      cylinder { <0,0,-4>, <0,0,4>, 0.8*WheelDiameter }
      texture { MetalTexture }
   }
   #declare Count=0;
   #while (Count<13)
      object { Profile rotate z*30*Count }
      #declare Count=Count+1;
   #end

   object { Support translate <0,0,-3.5> }
   object { Support rotate <0,0,90> translate  <0,0,-3.5> }
   object { Support rotate <0,0,180> translate <0,0,-3.5> }
   object { Support rotate <0,0,270> translate <0,0,-3.5> }

   object { Support translate <0,0, 3.5> }
   object { Support rotate <0,0,90> translate  <0,0,3.5> }
   object { Support rotate <0,0,180> translate <0,0,3.5> }
   object { Support rotate <0,0,270> translate <0,0,3.5> }
   
   object { Nut scale 1/0.6 rotate <-90,0,18> translate <0,0,-3.6> }
}

// the damper for the wheels / axles:
#declare damper = union {
   sphere { <0,-0.4,0>, 0.7 }                        // connection to the axle
   cylinder { <0,0,0>, <0,5,0>, 1 }                  // inner shaft
   cylinder { <0,4.5,0>, <0,10,0>, 1.2 pigment { rgb <.7,.06,.1> } finish { F_MetalA } } // outer shaft 
   cylinder { <-1.2,10.9,0>, <1.2,10.9,0>, 0.3 }     // and a screw
   difference {
      union {
         sphere { <0,10.9,0>, 0.9 }                   // the piece for attaching the suspension
         cylinder { <0,10,0>, <0,10.9,0>, 0.9 }
      }
      box { <-0.25,10,-1>, <0.25,12,1> }
      texture { T_Silver_3A }
   }
   texture { MetalTexture }
}   

// the suspension connects the wheel/axle/damper assembly:
#declare Suspension = union {
   difference {
      cylinder { <0,0,-9.4>, <0,0,-8>, 1.6*AxleDiameter }    // some kind of a ball bearing
      cylinder { <0,0,-9.5>, <0,0,-7.9>, 1.05*AxleDiameter }
      texture { BoltTexture }
   }
   object { damper rotate <17,0,0> translate <0,1.6*AxleDiameter+0.7,-8.5> }
   difference {
      box { <-0.25,11.8,-6.3>, <0.25,13.3,0> }    // this piece connects body and suspension
      cylinder { <-0.3,11.8+(13.3-11.8)/2,0>, <0.3,11.8+(13.3-11.8)/2,0>, 0.35 }
      texture { MetalTexture }
   }
   
   union {
      cylinder { <-0.8,11.8+(13.3-11.8)/2,-6.3/3.5>, <0.8,11.8+(13.3-11.8)/2,-6.3/3.5>, 0.35 }
      cylinder { <-0.8,11.8+(13.3-11.8)/2,-6.3/3.5*2>, <0.8,11.8+(13.3-11.8)/2,-6.3/3.5*2>, 0.35 }
      texture { BoltTexture }
   }
   union {
      box { < 0.6,13.1,-6.3/3.5+0.5>, < 0.2501,12,-6.3/3.5*2-0.5> }
      box { < 0.6,13.1, 6.3/3.5-0.5>, < 0.2501,12, 6.3/3.5*2+0.5> }
      texture { MetalTexture }
   }
   
   union {
      box { <-2.3,-4,-6.3/3.5*1.5+0.6>, <-2,7.5,-6.3/3.5*1.5-0.6> }
      box { <-2.7,-4,-6.3/3.5*1.5+0.15>, <-2.3,8,-6.3/3.5*1.5-0.15> }
      rotate z*-36
      translate <-3,5,0>
      texture { MetalTexture }
   }
   
   union {
      box { < 2.3,-4,-6.3/3.5*1.5+0.6>, < 2,7.5,-6.3/3.5*1.5-0.6> }
      box { < 2.7,-4,-6.3/3.5*1.5+0.15>, < 2.3,8,-6.3/3.5*1.5-0.15> }
      rotate z*36
      translate <3,5,0>
      texture { MetalTexture }
   }
}                 
                   
#declare HoldPlate = union {
   union {
      box { < 4,8.6,-6.3/3.5+0.5>, < 3.7,7,-6.3/3.5*2-0.5> }
      box { < 4,8.6, 6.3/3.5-0.5>, < 3.7,7, 6.3/3.5*2+0.5> }
      texture { MetalTexture }
   }
   union {
      cylinder { < 4,7.8,-6.3/3.5>  , < 3.5,7.8,-6.3/3.5>, 0.35 }
      cylinder { < 4,7.8,-6.3/3.5*2>, < 3.5,7.8,-6.3/3.5*2>, 0.35 }
      cylinder { < 4,7.8, 6.3/3.5>,   < 3.5,7.8, 6.3/3.5>, 0.35 }
      cylinder { < 4,7.8, 6.3/3.5*2>, < 3.5,7.8, 6.3/3.5*2>, 0.35 }
      texture { BoltTexture }
   }
}                      
                   
#declare RangeSensor = union {
   difference {
      box { <-1,0,0>, <1,1.2,-1> pigment { Black } }
      cylinder { <0,.6,-1.1>, <0,.6,-.6>, .45 scale <1.7,1,1> }
   }
   union {
      #declare i=0;
      #declare maxi=10;
      #while (i<maxi)
         cylinder { <i*(1/maxi),.1,-1>, <i*(1/maxi),1.1,-1>, .02 }
         cylinder { <i*(-1/maxi),.1,-1>, <i*(-1/maxi),1.1,-1>, .02 }
         cylinder { <-.9,i*(1.2/maxi),-1>, <.9,i*(1.2/maxi),-1>, .02 }
         #declare i=i+1;
      #end
      box { <-1.01,1.21,-1>, <1.01,1.11,-1.1> }
      box { <-1.01,1.21,-1>, <-0.91,-0.01,-1.1> }
      box { <-1.01,-.01,-1>, <1.01,.09,-1.1> }
      box { < 1.01,1.21,-1>, <0.91,-0.01,-1.1> }
      texture { MetalTexture }
   }
}

#declare YellowSign = union {
   difference {
      cylinder { <0,0,0>, <0,0,-0.01>, 1 }
      box { <0,0,-.1>, <2,2,.1> }
      box { <0,0,-.1>, <-2,-2,.1> }
      pigment { Yellow }
      finish { ambient 1 }
   }      
   difference {
      cylinder { <0,0,0>, <0,0,-0.01>, 1 }
      box { <0,0,-.1>, <-2,2,.1> }
      box { <0,0,-.1>, <2,-2,.1> }
      pigment { Black }
      finish { ambient 1 }
   }      
}

#declare CamBody = union {
   difference {
      union {
         cylinder { <.8,0,0>, <2.5,0,0>, 1.1 }
         cylinder { <2,0,0>, <4,0,0>, 1 }
         cylinder { <2.6,0,0>, <6.5,0,0>, 1.35 }
         cone { <-.4,0,0>, 1.4, <.8,0,0>, 1.1 }
      }
      cone { <-.401,0,0>, 1.3, <.801,0,0>, 1 }
  
      #declare HelpCylinder = cylinder { <.9,1.1,0>, <2.4,1.1,0>, .075 }
      #declare Count=0;
      #while (Count<36)
         object { HelpCylinder rotate x*10*Count }
         #declare Count=Count+1;
      #end

      #declare HelpCylinder = cylinder { <3,1.32,0>, <6.6,1.32,0>, .075 }
      #declare Count=0;
      #while (Count<36)
         object { HelpCylinder rotate x*10*Count }
         #declare Count=Count+1;
      #end

   }
   cylinder { <0.3,0,0>, <0.4,0,0>, 1 pigment { Black } }
   sphere { <0,0,0>, 1 scale <.3,1,1> texture { NBglass } }
   pigment { Gray50 }
   finish {
      specular .8
      roughness .002
      metallic
      reflection .1
   }
}

#declare CCDcam = union {                     
   object { CamBody translate <-3.5,5,0> rotate z*10 }      // Tilt of the camera
   cylinder { <0,5,0>, <0,5,3>, 0.4 }
   box { <-1.5,2,2>, <1.5,6.5,2.3> }
   box { <-.7,4.3,3.5>, <.7,5.7,2> }
   box { <-1.5,2,2.3>, <1.5,1.7,0> }
   cylinder { <0,-2,0>, <0,2,0>, 1.5 }
   scale 1.5
   texture { MetalTexture }
}

// the rear part of the robot:
#declare RearBody = union {
   difference {
      box { <-18, 2, 6>, <12, 9, -6> }         // the large body has a piece cut out for the 
      box { <-4, 13, 7>, <4, 5, -7> }          // suspension assembly.
   }
   cylinder { <0,0,-18.9>, <0,0,18.9>, AxleDiameter }  // the axle
   object { Wheel translate <0,0,-14> }                // two wheels
   object { Suspension }                               // and the suspensions 
   object { HoldPlate }

   object { Wheel rotate x*180 translate <0,0,14> }
   object { Suspension rotate y*180 }
   object { HoldPlate rotate y*180 }

   // mounting plate for the solar panels
   union {
      box { <-20.5,9.8,8.4>, <-3.6,10,-8.4> }
      box { <-20.5,9.8,-8.4>, <-20.3,10.2, 8.4> }
      box { <-3.6,9.8,-8.4>, <-3.4,10.2, 8.4> }
      box { <-20.5,9.8,-8.4>, <-3.6,10.2,-8.2> }
      box { <-20.5,9.8, 8.4>, <-3.6,10.2, 8.2> }
   }
   object { SolarPanel translate <-7.9,10,-4.1> }
   object { SolarPanel translate <-7.9,10, 4.1> }
   object { SolarPanel translate <-7.9-8.2,10,-4.1> }
   object { SolarPanel translate <-7.9-8.2,10, 4.1> }
   
   union {
      box { <12.3,9.8, 8.4>, < 3.6,10,  -8.4> }
      box { <12.3,9.8,-8.4>, <12.1,10.2, 8.4> }
      box { < 3.6,9.8,-8.4>, < 3.4,10.2, 8.4> }
      box { <12.3,9.8,-8.4>, < 3.6,10.2,-8.2> }
      box { <12.3,9.8, 8.4>, < 3.6,10.2, 8.2> }
   }
   object { SolarPanel translate <7.9,10,-4.1> }
   object { SolarPanel translate <7.9,10, 4.1> }
   
   object { YellowSign scale 2 translate <8,5.5,-6> }
   object { YellowSign scale 2 rotate y*180 translate <8,5.5, 6> }

   object { YellowSign scale 2 rotate z*90 translate <-8,5.5,-6> }
   object { YellowSign scale 2 rotate z*90 rotate y*180 translate <-8,5.5, 6> }

   union {
      box { <0, -.25, -.35>, <-5.2, .25, .35> rotate y*20 translate <-17.5,4.8,-3.5> }
      box { <0, -.25, -.35>, <-5.2, .25, .35> rotate y*-20 translate <-17.5,4.8,3.5> }
      box { <0, -.25, -.35>, <-5.2, .25, .35> rotate y*20 translate <-17.5,6.8,-3.5> }
      box { <0, -.25, -.35>, <-5.2, .25, .35> rotate y*-20 translate <-17.5,6.8,3.5> }
      difference {
         cylinder { <-23,4.55,0>, <-23,5.05,0>, 1.7 }
         cylinder { <-23,4.54,0>, <-23,5.06,0>, 1.3 }
      }
      difference {
         cylinder { <-23,6.35,0>, <-23,7.05,0>, 1.7 }
         cylinder { <-23,6.49,0>, <-23,7.06,0>, 1.3 }
      }
      translate x*.75
   }
   
   texture { MetalTexture }
}

// the front part of the robot:
#declare FrontBody = union {
   difference {
      box { <-10, 2, 6>, <12, 9, -6> }           // the large body has a piece cut out for the 
      box { <-4, 13, 7>, <4, 4.5, -7> }          // suspension assembly.
      box { <-10.1,4.5,6.1>, <-3.9,9.1,-6.1> }
   }
   object { CCDcam rotate y*15 translate <-5.5,7.5,0>}
   difference {
      cylinder { <-5.5,3,0>, <-5.5,8.6,0>, 2.7 }
      cylinder { <-5.5,3,0>, <-5.5,8.7,0>, 2.3 }
   }

   object { YellowSign scale 2 translate <8,5.5,-6> }
   object { YellowSign scale 2 rotate y*180 translate <8,5.5,6> }
   object { RangeSensor rotate y* 90     scale 2 translate <-8,4.6,0> }
   object { RangeSensor rotate y*(90-40) scale 2 translate <-7,4.6,-4> }
   object { RangeSensor rotate y*(90+40) scale 2 translate <-7,4.6, 4> }
   cylinder { <0,0,-18.9>, <0,0,18.9>, AxleDiameter }  // the axle

   // mounting plate for the solar panels
   union {
      box { <12.2,9.8,8.4>, <3.6,10,-8.4> }
      box { <12.2,9.8,-8.4>, <12,10.2, 8.4> }
      box { <3.6,9.8,-8.4>, <3.4,10.2, 8.4> }
      box { <12.2,9.8,-8.4>, <3.6,10.2,-8.2> }
      box { <12.2,9.8, 8.4>, <3.6,10.2, 8.2> }
   }
   
   object { SolarPanel translate <7.9,10,-4.1> }
   object { SolarPanel translate <7.9,10, 4.1> }

   object { Wheel translate <0,0,-14> }                // two wheels
   object { Wheel rotate x*180 translate <0,0,14> }

   object { Suspension }                               // and the suspensions 
   object { Suspension rotate y*180}
   object { HoldPlate }
   
   box { <0, -.25, -.35>, <5.2, .25, .35> rotate y*-20 translate <11.5,5.3,-3.5> }
   box { <0, -.25, -.35>, <5.2, .25, .35> rotate y*20 translate <11.5,5.3,3.5> }
   sphere { <16.5,5.3,0>, 1.7 }
   cylinder { <16.5,5.3,-2.6>, <16.5, 5.3, 2.6>, .5 texture { BoltTexture } }

   texture { MetalTexture }
}

// needed something to fill the lower right corner of the picture:
#declare BubbleTank = cylinder { <0,0,0>, <0,100,0>, 1.9
   pigment { Gray90 transmit .94 }
   finish {
      specular .9
      roughness .0004
      reflection .15
   }
   interior { ior 1.6 }
   scale .7
}

#declare BubbleContent = difference {
   cylinder { <0,0,0>, <0,100,0>, 1.7
      pigment {
         Orange filter .8
      }
      interior {
         ior 1.9
      }
   }
   #declare S=seed(30);
   #declare i=0;
   #while (i<120)
      #declare dia=.2+.5*rand(S);
      sphere { <(1.7-dia)*rand(S),100*rand(S),0>, dia
         pigment { transmit 1 }
         rotate y*360*rand(S)
         interior {
            ior 1
         }
      }
      #declare i=i+1;
   #end
   scale .7
}

#declare AlienWorld = difference {
   height_field {
      png "world5.png"
      texture { AlienSandTexture }
      translate <-.5,0,-.5>
      rotate y*90
      translate <0,-.3,0> scale <150,17,150>
   }
   object { BubbleTank translate <-9.5,0,-57.5> } 
}

#declare AlienMirror = box {
   <-3,-10,-1>, <6,20,0>
   pigment { White }
   finish { reflection 1.0 ambient .1 diffuse 0 }
}
           
#declare AlienSky = sky_sphere {
   pigment {
      MediumOrchid*2
      // Black
   }
   pigment {
      bozo
      color_map {
         [ 0 transmit 1]
         [.5 transmit 1]
         [.8 DarkOrchid ]
         [.9 White ]
      }
      turbulence .9
   }
   scale <1,.1,1>
   rotate x*-12
}           

#declare StrangeObj = union {
   #declare Count=0;
   #while (Count<150)
      cylinder {
         <-2,Count/10, 0>,
         < 0,Count/10, 0>, .2  
         pigment {
            rgb <abs(sin(Count/12+FlowerVal))*.8,abs(cos(Count/12-FlowerVal))*.8,.6>
         }
         finish { ambient .2 diffuse .7 }
         rotate y*Count*7.2
      }
      sphere {
         <-2,Count/10,0>, 0.3
         pigment { rgb <abs(sin(Count/5)),.7-FlowerVal/20,FlowerVal/10> }
         finish { ambient .5 specular .6 roughness .008 }
         rotate y*Count*7.2
      }
      #declare Count=Count+1;
   #end
   rotate y*-10*FlowerVal
}

#declare StrangeBloom = union {
   #declare Petals=20;
   #declare PetalDist=5;
   #declare i=0;
   #while (i<Petals)
      box { <PetalDist,-.25,-.25>, <0,.25,.25>
         rotate x*45
         rotate z*15
         rotate y*i*360/Petals
         pigment { Red }
         finish {
            irid { .6 thickness .5 turbulence .2 }
            specular .6 roughness .03
         }  
         normal {
            bumps 0.3
            scale .1
         }
      }
      sphere {<PetalDist,0,0>, PetalDist*pi*.8/Petals
         rotate z*15
         rotate y*i*360/Petals
         pigment { Black }
         finish {
            reflection 1
            ambient 0
            diffuse 0
            specular .6 roughness .03
         }
      }
      #declare i=i+1;
   #end
   sphere { <0,.8,0> 1.8
      pigment { Red }
      finish {
         irid { .6 thickness .5 turbulence .2 }
         specular .6 roughness .03
      }
      normal {
         bumps 0.4
         scale .1
      }
   }
   scale .7
   translate y*15
}           

// the world:

object { AlienWorld }
object { AlienWater }
sky_sphere { AlienSky }
box { <-75,-10,-75>, <75,-11,75> pigment { Black } }

// some flowers:

object { StrangeObj   rotate y*120 translate <-32.5, -2,-41> }
object { StrangeBloom translate <-32.5, -2,-41> }
#declare FlowerVal = FlowerVal+1;

object { StrangeObj   scale .8 rotate y*30 translate <-41, -3.3,-43> }
object { StrangeBloom scale .8 translate <-41, -3.3,-43> }
#declare FlowerVal = FlowerVal+1;

object { StrangeObj   scale .9 rotate y*-80 translate <4, -3, -8> }
object { StrangeBloom scale .9 translate <4, -3, -8> }
#declare FlowerVal = FlowerVal+1;

object { StrangeObj   rotate y*170 scale 1.3 translate <20, -2, 3> }
object { StrangeBloom scale 1.3 translate <20, -2, 3> }
#declare FlowerVal = FlowerVal+1;

object { StrangeObj   rotate y*-40 translate <-42, -5, 36> }
object { StrangeBloom translate <-42, -5, 36> }
#declare FlowerVal = FlowerVal+1;

object { StrangeObj   rotate y*-130 scale .77 translate <-16, -2, -48> }
object { StrangeBloom scale .77 translate <-16, -2, -48> }
#declare FlowerVal = FlowerVal+1;

object { StrangeObj   rotate y*-80 scale 1.1 translate <13, -2, 38> }
object { StrangeBloom scale 1.1 translate <13, -2, 38> }
#declare FlowerVal = FlowerVal+1;

// some of those mirrors:
object { AlienMirror rotate z*-12 rotate x*-11 rotate y*87 translate < 37,-3,6>}           
object { AlienMirror rotate z*17 rotate y*-47 translate <-22,-3,-24>}           

object { BubbleTank translate <-9.5,0,-57.5> } 
object { BubbleContent translate <-9.5,0,-57.5> }
  
// our little robot: 
object { FrontBody scale .45 rotate x*-1 rotate y*23 translate <-13,4.3,-30.8> }
object { RearBody  scale .45 rotate z*10 rotate x*5 rotate y*-1.5 translate <4,6.2,-33.7> }
