
// POV-Ray Scene File - Pearl Harbor (Version 6)
// created 20-Nov-98 for the Nov/Dec 1998
// Internet Ray Tracing Competition (www.irtc.org)
// last update 27-Dec-98
// copyright 1998 Glenn McCarter (gmccarter@hotmail.com)
//                and N.B. (beliaev@utu.fi)
//
// Notes:
// Scene origin is almost directly under camera, at sea level
// (ground level is 8 feet above this)
// +Z axis is along the centerline of the drydock


//------------- Scene Complexity Switches ------------------------

// (Turn all switches to 1 for full render all objects)
#declare Docks = 1; // turns off/on docks, fences, drydock, mooring lines
#declare BldgCraneEquipVeh = 1; // turns off/on buildings, cranes, equipment, vehicles
#declare Ships = 1; // turns off/on ships & smoke
#declare GoodSmoke = 1; // turns on real smoke & fire, otherwise use simple spheres
#declare Aircraft = 1; // turns off/on airplanes

//----------------------------------------------------------------

#include "GUnits.inc"

#switch ( Ships )
	#case (1)
	#include "Ships.inc"
	#include "dd01.inc"
	#include "bb01.inc"
	#include "SmokeFire.inc"
	#break
#end
#switch ( Docks )
	#case (1)
	#include "Docks.inc"
	#break
#end
#switch ( BldgCraneEquipVeh )
	#case (1)
	#include "Buildings.inc"
	#include "Equipment.inc"
	#include "Cranes.inc"
	#include "Vehicles.inc"
	#break
#end
#switch ( Aircraft )
	#case (1)
	#include "Zero01.inc"
	#include "torpedo_bomber01.inc"
	#break
#end


#version 3.1;
 global_settings {
	adc_bailout 0.003922
//	ambient_light <1.0,1.0,1.0>
	ambient_light <.5,.55,.6> // test
	assumed_gamma 1.0
	hf_gray_16 off
	irid_wavelength <0.25,0.18,0.14>
	max_intersections 64
	max_trace_level 15
	number_of_waves 10
	 // Radiosity section: uncomment this to enable radiosity
	// calculations; must also turn on radiosity in your ini file.
	radiosity {
		brightness       3.3
		count            100
		distance_maximum 5
		error_bound      0.4
		gray_threshold   0.5
		low_error_factor 0.8
		minimum_reuse    0.015
		nearest_count    6
		recursion_limit  1
	}
	 // end radiosity section
}

//===============================================
//             Variables
//===============================================

// Use this to position the camera and the airplanes
#declare CameraPosition = <80*Meters,80*Meters,-120*Meters>;
#declare CamPos = CameraPosition;

// This sets the height of all dock objects above sea level
#declare GroundLevel = 8*Feet;


//===============================================
//             Cameras
//===============================================

#declare Render_Cam =
	camera {
		perspective
		direction <0.0,     0.0,  1.0>
		sky       <-0.18,     1.0,  0.0>  // Use left handed-system
		up        <0.0,     1.0,  0.0>  // Where Y is up
		right     <1.33333, 0.0,  0.0>
	/* // Focal Blur section: uncomment to enable focal blur
		blur_samples 6
		aperture .5    //lower value = wider range in focus
		focal_point <0,0,0>
	*/ // end focal blur
	angle 26
	location  CamPos
//	look_at   <-50*Meters, 0, 800*Meters> // straight down the line
	look_at   <-90*Meters, 0, 800*Meters> // more to the left
}

#declare Special_Cam = // move this camera anywhere for testing
	camera {
		perspective
		direction <0.0,     0.0,  1.0>
		sky       <0.0,     1.0,  0.0>  // Use left handed-system
		up        <0.0,     1.0,  0.0>  // Where Y is up
		right     <1.33333, 0.0,  0.0>
		angle 30
//	location  CamPos+<0,200*Meters,0> // above render view
//	look_at   <-50*Meters, 100*Meters, 800*Meters>
//	location  <0,80*Feet,0> // photo view
//	look_at   <0, 0, 600*Feet>
//	location <400*Feet,1900*Feet,600*Feet>
//	look_at <0,0,900*Feet> // Pennsylvania
//	location <400*Feet,600*Feet,200*Feet>
//	look_at <0,0,400*Feet> // Cassin & Downes
//	location  <1200*Feet,440*Meters,00*Feet>
//	look_at   <-12*Feet, 5*Meters,720*Feet> // Docks view
//	location  <600*Feet,600*Feet,1100*Feet>
//	look_at   <100*Feet, 0,1600*Feet> // Cachalot
//	location  <600*Feet,600*Feet,1800*Feet>
//	look_at   <100*Feet, 0,2600*Feet> // Oglala, Helena
//	location  <800*Feet,600*Feet,2100*Feet>
//	look_at   <350*Feet, 0,3000*Feet> // Sacramento, Argonne
//	location  <200*Meters,260*Meters,2800*Feet>
//	look_at   <-1200*Feet, 5*Meters,3350*Feet> // California
//	location  <200*Feet,200*Meters,3200*Feet>
//	look_at   <-1500*Feet, 5*Meters,5100*Feet> // Ford Island buildings
//	location  <200*Meters,260*Meters,3400*Feet>
	location  <60*Feet,2600*Feet,2100*Feet>
	look_at   <-550*Feet, 0,2700*Feet> // Nevada overhead wake
//	location  <60*Feet,120*Feet,1700*Feet>
//	look_at   <-600*Feet, 0,2300*Feet> // Nevada front
//	location  <200*Meters,260*Meters,4000*Feet>
//	look_at   <-1000*Feet, 5*Meters,4500*Feet> // Neosho
//	location  <200*Meters,260*Meters,4000*Feet>
//	look_at   <-150*Feet, 5*Meters,5000*Feet> // Battleship Row Oklahoma
//	location  <200*Meters,260*Meters,4000*Feet>
//	look_at   < -50*Feet, 5*Meters,5800*Feet> // Battleship Row West Virginia
//	location  <200*Meters,260*Meters,4000*Feet>
//	look_at   < 150*Feet, 5*Meters,6600*Feet> // Battleship Row Vestal
//	location  <000*Feet,1460*Meters,6000*Feet>
//	look_at   <-2500*Feet, 5*Meters,9000*Feet> // East Loch
//	location  <000*Feet,600*Meters,6000*Feet>
//	look_at   <-2500*Feet, 5*Meters,14000*Feet> // Far shore
//	location  <200*Feet,260*Meters,4600*Feet>
//	look_at   <-1300*Feet, 5*Meters,5000*Feet> // Ford Island buildings detail
//	location  <600*Feet,260*Feet,6000*Feet>
//	look_at   <-1300*Feet, 1*Meters,5000*Feet> // Ford Island buildings further
//	location  <600*Feet,260*Feet,6000*Feet>
//	look_at   <-1300*Feet, 1*Meters,6500*Feet> // Ford Island buildings further
	}

#declare Map_Cam = // this camera is a top-down orthographic view, for aligning maps
	camera {
		orthographic
		direction <0.0,     -1.0,  0.0> // points straight down
		sky       <0.0,     1.0,  0.0>  // Use left handed-system
//		up        <0.0,     0.0,  16*Miles>  // Set size of viewing window - Oahu view
//		right     <16*1.33333*Miles, 0.0,  0.0>
//		up        <0.0,     0.0,  5*Miles>  // Set size of viewing window - Pearl Harbor view
//		right     <5*1.33333*Miles, 0.0,  0.0>
		up        <0.0,     0.0,  1*Miles>  // Set size of viewing window - Local view
		right     <1*1.33333*Miles, 0.0,  0.0>
//	location  <3*Mile,1*Mile,5*Mile> // center on Oahu map
//	look_at   <3*Mile,0,5*Mile>
	location  <.1*Mile,1*Mile,.3*Mile> // center on Pearl Harbor map
	look_at   <.1*Mile,0,.3*Mile>
//	location  <0,1*Mile,0> // center on local origin
//	look_at   <0,0,0>
	}

camera {
//	Special_Cam
//	Map_Cam
	Render_Cam
	}


//===============================================
//             Textures
//===============================================

#declare ChromeTex = // for testing only
	texture {
		pigment { color rgb<1,1,1>*.03 }
		normal { bumps .39 scale 2 } // test
		finish { ambient .1 diffuse .3 specular .1 roughness .005 reflection .5 }
	}

#declare SeaWaterNearbyTex = // brownish, clear, not too reflective
	texture {
//		pigment { color rgbt<0.227,0.416,0.451,0.64> } // too clear
//		pigment { color rgb<0.227,0.416,0.451> } // no trans, too opaque
//		pigment { color rgb<1,1,1> } // test white
		pigment { color rgbf<0.227,0.416,0.282,0.47> }
		normal { bumps .11 scale <2*Meters,1*Meters,2*Meters> }
		finish { ambient .001 diffuse .2 specular .1 roughness .005 reflection 0.25}
	}
#declare SeaWaterInt =
	interior { ior 1.33 }
#declare SeaWaterMat = // material
	material {
		texture { SeaWaterNearbyTex }
		interior { SeaWaterInt }
	}
#declare SeaWaterTex = texture { SeaWaterNearbyTex }

#declare SeaFloorTex =
	texture {
		pigment {
			bozo
				color_map {
					[ 0.0 color rgb <0.268,0.174,0.084> ]
					[ 0.5 color rgb <0.403,0.304,0.124> ]
					[ 1.0 color rgb <0.537,0.489,0.315> ]
				}
				scale 500*Feet
				turbulence .3
		}
		normal { bumps .5 scale 2.7*Meter }
		finish { ambient .1 diffuse .7 specular .2 roughness .15 }
	}


#declare DistantMountainsTex = // for testing
	texture {
		pigment { color rgb<.77,.67,.45> }
	}


//===============================================
//                Lighting
//===============================================

#declare MorningSun =
	light_source {
		<0,0,0>
//		color rgb <0.973,0.988,0.984>*2.0 // too harsh bright
//		color rgb <0.973,0.988,0.984>*1.4
		color rgb <0.988,0.988,0.98>*1.35
	}

object { MorningSun
	translate <0,50*KM,0> // noon at Equator: directly overhead
	rotate z*-4/24*360 // 8:00 am
	rotate x*-21 // Hawaii latitude
	rotate x*-23 // Earth tilt on 7th of December
  rotate y*-22 // North in scene is rotated
}


#declare FillLight =
	light_source {
		<1600*Feet,3000*Feet,-1800*Feet>
		color rgb <0.890,0.925,0.961>*.5
		spotlight
		point_at <0,0,700*Feet>
		radius 20    // degrees fully lit (*2)
		falloff 42   // degrees no lighting (*2)
		tightness 1
		media_interaction off
		fade_distance 40*Meters
		fade_power 1
	}
object {FillLight }

#declare NegativeLight = // just an experiment
	light_source {
//		<1600*Feet,3000*Feet,-1800*Feet>
		CamPos+<100*Feet,50*Feet,1400*Feet>
		color rgb <-0.890,-0.925,-0.961>*.3
		spotlight
		point_at <0,0,700*Feet>
		radius 22    // degrees fully lit (*2)
		falloff 32   // degrees no lighting (*2)
		tightness 1
		media_interaction off
//		fade_distance 40*Meters
//		fade_power 1
	}
//object {NegativeLight }

#declare ZeroLight = // nice highlights on fuselage
	light_source {
		CamPos+<7*Meters,14*Meters,56*Meters>
		color rgb <0.890,0.925,0.961>*0.8
		spotlight
		point_at CamPos+<-7*Meters,-4*Meters,28*Meters>
		radius 20    // degrees fully lit (*2)
		falloff 32   // degrees no lighting (*2)
		tightness 1
		fade_distance 40*Feet
		fade_power 1
		media_interaction off
	}
//object {ZeroLight }

//===============================================
//              Environment
//===============================================

background { color rgb <0.2, 0.4, 0.8> } // light blue

sky_sphere {
	pigment {
		gradient y
		color_map {
//			[ 0.0 color rgb<0.32,0.48,0.68> ] // light blue, use this with no fog
//			[ 0.5 color rgb<0.02,0.15,0.32> ]
			[ 0.00 color rgb<0.62,0.68,0.68> ] // white to deep blue, use this with fog
			[ 0.04 color rgb<0.62,0.68,0.68> ] // this is the lowest visible sky
			[ 0.15 color rgb<0.21,0.32,0.52> ] // this is the top of the scene view
		}
	}
}

#declare FakeFog = // use a simple filtered box to simulate fog.  Very quick to render.
	box {
		<-4*KM,0,0>,<4*KM,1*KM,1>
		texture {
			pigment {
		    gradient y
					color_map{
						[ 0.000 color rgbf <1,1,1,1.00> ] // bottom transparent
						[ 0.001 color rgbf <1,1,1,1.00> ] // sea surface transparent
						[ 0.008 color rgbf <.85,.93,.89,0.86> ] // hide shoreline
						[ 0.018 color rgbf <1,1,1,0.95> ] // hills clear
						[ 0.046 color rgbf <1,1,1,0.75> ] // mountaintops opaque
						[ 0.100 color rgbf <1,1,1,1.00> ] // sky transparent
						[ 1.000 color rgbf <1,1,1,1.00> ] // top transparent
					}
					scale <1,4*KM,1>
		  }
			finish { ambient .9 }
		}
	}
object { FakeFog rotate y*10 translate <0,0,8000*Feet> }


//===============================================
//                Objects
//===============================================

//---------------- Landscape --------------------

#declare SeaWater =
	union {
		difference {
				box { <-30*KM,-100*Feet,-30*KM>,<30*KM,0,30*KM> // this is the transparent surface
				}
				box { <-110*Feet,-100*Feet,-300*Feet>,<100*Feet,1,1275*Feet> // cutaway for the drydocks
				}
				box { <-100*Feet*.707,-100*Feet,-100*Feet>,<100*Feet*.707,1,100*Feet> // cutaway for drydock doors
					scale <1,1,.7> rotate y*45 translate <0,0,1275*Feet>
				}
				material { SeaWaterMat }
		} // end difference
		plane { y, -65*Feet // this is the ground underneath
			texture { SeaFloorTex }
		}
	} // end union
#declare SeaWaterT =  // for testing only
	box { <-30*KM,-100*Feet,-30*KM>,<30*KM,0,30*KM>
		texture { pigment { color rgb<.45,.5,.7> } }
	}

#declare NevadaWake =
	union {
		height_field {
			png "Wake01.png"
			material { SeaWaterMat }
			scale <2400*Feet,18*Feet,2400*Feet>
			water_level .0001
		}
//		box { <0,0,0>,<607*Feet,30*Feet,80*Feet> rotate y*-53 translate <180*Feet,0,280*Feet> texture {WhitePaint} } // test
		box { // this is the bright area behind the ship
			<0,-1,-40*Feet>,<1000*Feet,1*Feet,40*Feet>
			texture {
				pigment {
					bozo
						color_map {
							[ 0.0 color rgbt<1,1,1,1> ]
							[ 0.2 color rgbt<1,1,1,1> ]
							[ 1.0 color rgb<1,1,1> ]
						}
					scale 8*Feet
				}
				finish { ambient 3.0 }
			}
			rotate z*-.2
			rotate y*-53
			translate <500*Feet,0,500*Feet>
		}
	}

#declare PearlMap =
	height_field {
		png "PHhfmap04.png" // render
		smooth
		texture {
			pigment {
				image_map { // image is always projected from -z, with front facing +z, top to +Y
				png "PHcolormap04.png" // render
					once
				}
				rotate x*90 // align map to height_field
			}
		}
		water_level .001 // only keep the land
		translate <-0.40,0,-0.08> // scene origin near dry docks
		translate <0,-0.3,0> // ensure local area is below ground - render
		scale <2.88*Miles,100*Feet,2.44*Miles> // render
	}
#declare PearlMapT = // for testing positions of objects only
	height_field {
		gif "PHhfmap01.gif" // highlight object positions
		texture {
			pigment {
				image_map { // image is always projected from -z, with front facing +z, top to +Y
					gif "PHcolormap01.gif" // test ship positions
					once
				}
				rotate x*90 // align map to height_field
			}
		}
		water_level .001 // only keep the land
		translate <-0.40,0,-0.08> // scene origin near dry docks
	translate <0,-0.5,0> // test
	scale <2.88*Miles,40*Feet,2.44*Miles> // test
	}

#declare OahuMap =
	height_field {
		png "OahuHFmap3.png"
//		texture { DistantMountainsTex }
		smooth
		texture {
			pigment {
				image_map { // image is always projected from -z, with front facing +z, top to +Y
					png "OahuColormap2test.png"
					once
				}
				rotate x*90 // align map to height_field
			}
		}
		water_level .015
		translate <-0.085,0,-0.008> // scene origin must align with PearlMap
		scale <11.5*Miles,.96*KM,11.5*Miles*.8265> // XZ scale must match PearlMap, Y-scale sets max height
		translate <0,-85*Feet,0> // ensure local area is below ground
	}

//---------------- Other Stuff --------------------

#declare StrafingTex =
	texture {
		pigment { color rgb<.99,.99,.99> }
		normal { bumps .3 scale 10*cm }
		finish { ambient .98 diffuse .3 specular .6 roughness .05 }
	}

#local SSpacing = 36*Feet;
#declare Strafing =
	union {
		#local BulletCount = 0;
		#while ( BulletCount < 8 )
			torus {
				(0.8+BulletCount*.3)*Feet,6*inches
				scale <1,.1,1>
				translate <-7*Feet,0,-SSpacing*BulletCount>
				texture { StrafingTex }
			}
			torus {
				(0.8+BulletCount*.3)*Feet,6*inches
				scale <1,.1,1>
				translate < 7*Feet,0,-SSpacing*BulletCount>
				texture { StrafingTex }
			}
			#local BulletCount = BulletCount + 1;
		#end
	}
#local SSpacing = 36*Feet;


//===============================================
//              Object Placement
//===============================================

// ENVIRONMENT
object { SeaWater }
object { OahuMap  rotate y*-30 translate <60*Feet,0,-30*Feet> } // align to drydock
object { PearlMap rotate y*-30 translate <60*Feet,0,-30*Feet> } // align to drydock

// LEFT DOCK OBJECTS
#switch ( Docks )
#case (1)
object { SimpleDocks translate <0,GroundLevel,0> } // centerline of drydock is at x=0
object { DryDock }
object { MooringGroup translate <0,0,0*Feet> }
object { Fences translate <0,GroundLevel,0*Feet> }
#break
#end // Docks
#switch ( BldgCraneEquipVeh )
#case (1)
object { Crane(14,40,60*Feet) translate <128*Feet,GroundLevel,800*Feet> } // right
object { CraneRails(1200*Feet,800*Feet) translate <128*Feet,GroundLevel,800*Feet> }
object { WhiteDockBuilding rotate y*-90 translate <-150*Feet,GroundLevel,600*Feet> }
object { WhiteDockBuildingJunk rotate y*-90 translate <-150*Feet,GroundLevel,600*Feet> }
object { WaterTankVertGroup translate <-220*Feet,GroundLevel,830*Feet> }
object { CargoGroup translate <-200*Feet,GroundLevel,450*Feet> }
object { CargoGroup rotate y*4 translate <-220*Feet,GroundLevel,1050*Feet> }
object { PipeGroup rotate y*-82 translate <-128*Feet,GroundLevel,674*Feet> }
object { Vehicles translate <0,GroundLevel,0> } // all vehicles in the scene, see Vehicles.inc
object { People translate <0,GroundLevel,0> } // all people in the scene, see Vehicles.inc

// RIGHT DOCK OBJECTS
object { AdminBuilding translate <220*Feet,GroundLevel,700*Feet> }
object { SigBuilding translate <220*Feet,GroundLevel,900*Feet> }
object { SignalTower translate <640*Feet,GroundLevel,2400*Feet> }
object { CachalotGroup translate <240*Feet,GroundLevel,470*Meters> }
object { SignalTower scale <.2,.2,.2> translate <218*Feet,0,1200*Feet> }
object { AdminBuilding scale <.4,1,.64> rotate y*90 translate <170*Feet,GroundLevel-27*Feet,1210*Feet> }
object { FarGroup translate <240*Feet,GroundLevel,2900*Feet> }
object { Crane(-148,65,0) translate <350*Feet,GroundLevel,2750*Feet> } // right far
object { WhiteDockBuilding translate <310*Feet,GroundLevel,1400*Feet> } // far right
object { WaterTankVert translate <360*Feet,GroundLevel,1440*Feet> }
object { WaterTankVert translate <390*Feet,GroundLevel,1440*Feet> }
object { Warehouse translate <300*Feet,GroundLevel,1600*Feet> }
object { Warehouse translate <300*Feet,GroundLevel,1800*Feet> }
object { Warehouse translate <300*Feet,GroundLevel,2000*Feet> }
object { WhiteDockBuilding scale <1,1.3,1> translate <360*Feet,GroundLevel,2400*Feet> } // very far

// FORD ISLAND OBJECTS
object { Crane(80,30,0) rotate y*24 translate <-1400*Feet,8*Feet,4800*Feet> }
object { IslandBuilding rotate y*-22 translate <-1700*Feet,14*Feet,4800*Feet> } // behind crane left
object { IslandBuilding rotate y*-22 translate <-1350*Feet,14*Feet,5300*Feet> } // behind crane right
object { LowBuilding scale <1,1.4,1> rotate y*-22 translate <-1600*Feet,14*Feet,5000*Feet> } // behind crane
object { LowBuilding scale <.8,1.65,.8> rotate y*-22 translate <-1360*Feet,14*Feet,4850*Feet> } // beside crane
object { WhiteDockBuilding scale <1.3,1.2,1.2> rotate y*-34 translate <-1800*Feet,10*Feet,4330*Feet> }
object { WhiteDockBuilding scale <1.3,1.2,1.2> rotate y*-34 translate <-1600*Feet,14*Feet,4500*Feet> }
object { OilTankGroup rotate y*-35 translate <-1300*Feet,8*Feet,5500*Feet> }
object { IslandBuilding rotate y*-40 translate <-1200*Feet,8*Feet,4500*Feet> }
object { IslandBuilding rotate y*-76 translate <-1000*Feet,10*Feet,4900*Feet> }
object { IslandBuilding rotate y*-76 translate <-1000*Feet,12*Feet,5100*Feet> }
object { IslandBuilding rotate y*-76 translate <-1000*Feet,14*Feet,5300*Feet> }
object { LowBuilding rotate y*-76 translate <-900*Feet,8*Feet,5100*Feet> }
object { AdminBuilding scale <1,1.3,1> rotate y*-75 translate <-600*Feet,8*Feet,6200*Feet> }
object { LowBuilding rotate y*-65 translate <-700*Feet,8*Feet,5950*Feet> }
object { IslandBuilding rotate y*-22 translate <-2600*Feet,14*Feet,6500*Feet> } // far side

object { FarawayRandomBuildings translate <-4000*Feet,8*Feet,12500*Feet> } // far shore
object { FarawayRandomTrees translate <-4000*Feet,8*Feet,12500*Feet> } // far shore
object { AllTrees } // see "Buildings.inc"
#break
#end // BldgCraneEquipVeh

#switch ( Ships )
#case (1)
// NEAR SHIPS
object { Pennsylvania scale 1*Feet rotate y*-90 translate <0*Feet,0,650*Feet>} // battleship Pennsylvania
object { Destroyer01 ( 10, -30, 12, -29, 10, 43, 15, 32 ) scale 1*Feet rotate y*-90 translate <-35*Feet,0,250*Feet>} // destroyer Downes
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-88 translate <35*Feet,0,250*Feet>} // destroyer Cassin
object { BB_36_37 scale 1*Feet rotate y*-67 translate <-510*Feet,0,2550*Feet> } // battleship Nevada
object { NevadaWake rotate y*(-67+53) translate <-680*Feet,0,2400*Feet> }
object { Cachalot_Sub scale 1 rotate y*-87 translate <55*Meters,-1*Meters,470*Meters> } // submarine Cachalot
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1.30*Feet rotate y*-90 translate <55*Meters,0,650*Meters> } // light cruiser Helena
object { Oglala rotate <8,-90,0> translate <33*Meters,0,680*Meters> } // minelayer Oglala
object { Argonne rotate y*-92 translate <97*Meters,0,890*Meters> } // auxiliary Argonne
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-90 translate <165*Meters,0,960*Meters> } // gunboat Sacramento
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-90 translate <155*Meters,0,890*Meters> } // far dock destroyer
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-90 translate <169*Meters,0,890*Meters> } // far dock destroyer

// FORD ISLAND SHIPS
object { BB_43_44 scale 1*Feet rotate z*-1 rotate y*-83 translate <-1220*Feet,0,2970*Feet> } // California
object { Neosho rotate y*-30 translate <-1160*Feet,-12*Feet,4150*Feet> } // oiler Neosho
object { BB_36_37 scale 1*Feet rotate <-18,-80,0> translate <-230*Feet,0,4730*Feet> } // Oklahoma
object { BB_46_48 scale 1*Feet rotate y*-80 translate <-380*Feet,0,4770*Feet> } // Maryland
object { BB_46_48 scale 1*Feet rotate y*-84 translate < 40*Feet,0,5500*Feet> } // West Virginia
object { BB_43_44 scale 1*Feet rotate y*-84 translate <-140*Feet,0,5520*Feet> } // Tennessee
object { Oglala scale <2,1.1,1.4> rotate y*-84 translate < 190*Feet,0,6430*Feet> } // Vestal
object { Arizona scale 1*Feet rotate y*-82 translate < 40*Feet,0,6430*Feet> } // Arizona

// FAR SHIPS
object { Tender rotate y*-80 translate < -2600*Feet,0,10000*Feet> } // destroyer tender Whitney
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-80 translate <-2700*Feet,0,10100*Feet> } // destroyer
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-80 translate <-2800*Feet,0,10100*Feet> } // destroyer
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-80 translate <-2900*Feet,0,10100*Feet> } // destroyer
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-80 translate <-3000*Feet,0,10100*Feet> } // destroyer
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-80 translate <-3100*Feet,0,10100*Feet> } // destroyer

object { Tender rotate y*-80 translate < -3000*Feet,0,8700*Feet> } // destroyer tender Dobbin
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-80 translate <-3100*Feet,0,8700*Feet> } // destroyer
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-80 translate <-3200*Feet,0,8700*Feet> } // destroyer
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-80 translate <-3300*Feet,0,8700*Feet> } // destroyer
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-80 translate <-3400*Feet,0,8700*Feet> } // destroyer
object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) scale 1*Feet rotate y*-80 translate <-3500*Feet,0,8700*Feet> } // destroyer

object { Destroyer01 ( 0, 0, 0, 0, 0, 0, 0, 0 ) rotate y*-80 translate <-1200*Feet,0,10700*Feet> } // destroyer Blue
object { Solace rotate y*-80 translate < -1030*Feet,0,8660*Feet> } // hospital ship Solace

// SMOKE
object { SmokeGroup rotate y*196 translate <-1200*Feet,15*Meters,3350*Feet> } // California
object { FireGroup rotate y*196 translate <-1100*Feet,15*Meters,3200*Feet> } // California
object { SmokeGroup rotate y*150 translate <-180*Feet,15*Meters,5130*Feet> } // Oklahoma
object { FireGroup rotate y*170 translate <-180*Feet,15*Meters,4960*Feet> } // Oklahoma
object { SmokeGroup scale <1.89,1.52,-2.08> rotate y*-179 translate <  70*Feet,15*Meters,6700*Feet> } // Arizona
object { FireGroup scale <1.89,1.52,-2.08> rotate y*-189 translate <  45*Feet,15*Meters,6660*Feet> } // Arizona
object { LtSmoke scale <1.5,1,1> rotate z*-65 translate <22*Meters,18*Meters,750*Meters> } // Oglala
#break
#end // Ships

// AIRPLANES
#switch ( Aircraft )
#case (1)
object { Zero_Fighter rotate < 24,0,8> rotate <0,88,0> translate CamPos+<-8*Meters,-4.8*Meters,28*Meters> } // nearest
object { Zero_Fighter rotate <8,0,6> rotate <0,95,0> translate CamPos+< -9*Meters,-12*Meters,115*Meters> } // near
object { Strafing rotate y*5 translate <174*Feet,2,1520*Feet> } // splashes near Cachalot
object { Zero_Fighter rotate <40,0,-16> rotate <0,120,0> translate <85*Meters,130*Meters,600*Meters> } // mediumfar
object { Zero_Fighter rotate <-35,0,0> rotate <0,20,0> translate <-220*Meters,170*Meters,1200*Meters> } // veryfarUp
object { Zero_Fighter rotate <-65,0,0> rotate <0,20,0> translate <-270*Meters,150*Meters,1200*Meters> } // veryfarDn
object { Torpedo_bomber rotate <8,0,6> rotate <0,123,0> translate CamPos+< -65*Meters,-17*Meters,201*Meters> }
#break
#end


