
// POV-Ray Scene File - Dawn Patrol II (Version 4)
// created 28-Jan-98 for the Jan/Feb 1999
// Internet Ray Tracing Competition (www.irtc.org)
// last update 5-Feb-99
// copyright 1998 Glenn McCarter (gmccarter@hotmail.com)
//
// Notes:
// Scene origin is almost directly under camera, at sea level
// +Z axis is nearly straight away from the camera, towards the center of the scene


//#include "GIncludeRH.inc"
//#include "GRuler.inc"


#version 3.1;
 global_settings {
	adc_bailout 0.003922
//	ambient_light <1.0,1.0,1.0>
	ambient_light <0.2,0.2,0.2>
//	ambient_light <.5,.55,.6> // test
	assumed_gamma 1.0
	hf_gray_16 off
	irid_wavelength <0.25,0.18,0.14>
	max_intersections 64
	max_trace_level 15
	number_of_waves 10
	 // Radiosity section: uncomment this to enable radiosity
	// calculations; must also turn on radiosity in your ini file.
/*
	radiosity {
		brightness       3.3
		count            100
		distance_maximum 5
		error_bound      0.4
		gray_threshold   0.5
		low_error_factor 0.8
		minimum_reuse    0.015
		nearest_count    6
		recursion_limit  1
	}
*/
	 // end radiosity section
}

//===============================================
//             Variables
//===============================================

// Basic unit in scene is Yards
// Maximum safe dimension is 5700 Miles
#declare Miles = 1760;
#declare Mile = 1760;
#declare Feet = 1/3;
#declare in = 1/36;
#declare Meters = Feet*3.28084;
#declare Meter = Feet*3.28084;
#declare KM = Meters*1000;

// Use this to position the camera
//#declare CameraPosition = 	<1*Feet,10*Feet,-2*Feet>;
//#declare CameraPosition = 	<1*Feet,100*Feet,-2*Feet>;
#declare CameraPosition = 	<1*Feet,100*Feet,-2*Feet>;
#declare CamPos = CameraPosition;

// Use this function to apply a random rotation to any object
#declare RSeed = seed(10195);
#macro RandRot()
	<rand(RSeed)*360,rand(RSeed)*360,rand(RSeed)*360>
#end

#declare GoodSmoke = 1; // turns on real vapor, otherwise use simple spheres
#include "VaporTrail.inc"
#include "Dreadnought.inc"

//===============================================
//             Cameras
//===============================================

#declare Render_Cam =
	camera {
		perspective
		direction <0.0,     0.0,  1.0>
		sky       <-0.03,     1.0,  0.0>  // Use left handed-system
		up        <0.0,     1.0,  0.0>  // Where Y is up
		right     <1.33333, 0.0,  0.0>
	/* // Focal Blur section: uncomment to enable focal blur
		blur_samples 6
		aperture .5    //lower value = wider range in focus
		focal_point <0,0,0>
	*/ // end focal blur
	angle 50
	location  CamPos
//	look_at   <-50*Meters, 0, 800*Meters> // straight down the line
	look_at   <-20*Feet, 180*Feet, 800*Feet> // more to the left
}

#declare Special_Cam = // move this camera anywhere for testing
	camera {
		perspective
		direction <0.0,     0.0,  1.0>
		sky       <0.0,     1.0,  0.0>  // Use left handed-system
		up        <0.0,     1.0,  0.0>  // Where Y is up
		right     <1.33333, 0.0,  0.0>
	angle 20
	location  CamPos
	look_at   <-240*Feet,200*Feet,1500*Feet> // patrol ship
//	angle 16
//	location  CamPos
//	look_at   <-240*Feet,40*Feet,1500*Feet> // patrol ship wake
	}

#declare Normal_Cam = // this camera aligns the normal map to the image
	camera {
		orthographic
		direction <0.0,     -1.0,  0.0> // points straight down
		sky       <0.0,     1.0,  0.0>  // Use left handed-system
		up        <0.0,     0.0,  1*Miles>  // Set size of viewing window - Local view
		right     <1*1.33333*Miles, 0.0,  0.0>
	location  <.1*Mile,1*Mile,.3*Mile> // center on image map
	look_at   <.1*Mile,0,.3*Mile>
	}

camera {
//	Special_Cam
//	Normal_Cam
	Render_Cam
	}


//===============================================
//             Textures
//===============================================

#declare ChromeTex = // for testing only
	texture {
		pigment { color rgb<1,1,1>*.03 }
		normal { bumps .39 scale 2*in } // test
		finish { ambient .1 diffuse .3 specular .1 roughness .005 reflection .5 }
	}

#declare WhitePaint = // for testing only
	texture {
		pigment { color rgb<1,1,1> }
		finish { ambient .9 }
	}

#declare StarTex =
	texture {
		pigment { color rgb<1,1,1> }
		finish { ambient .9 }
	}

#declare MoonTex = // for testing only
	texture {
		pigment { color rgb<1,1,1> }
		normal { bumps .09 scale 20*Feet }
		finish { ambient .2 diffuse .7 specular .6 roughness .15 }
	}

/*
#declare RingTex = // for testing only
	texture {
		pigment { color rgb<1,1,1> }
		normal { bumps .09 scale 20*Feet }
		finish { ambient .1 diffuse .3 specular .1 roughness .005 }
	}
*/
#declare RingColor = rgb<0.598,0.674,0.984>
#declare RingTrans = .95; // maximum transparancy
#declare RingTex =
	texture {
		pigment {
			onion
				color_map {
//					[ 0.00 color rgbt <0.098,0.074,0.984,0.0> ] // test blue
//					[ 0.03 color rgbt <0.098,0.074,0.984,0.0> ]
					[ 0.00 RingColor transmit 1.0 ]
					[ 0.01 RingColor transmit 0.92*RingTrans ]

					[ 0.07 RingColor transmit 0.92*RingTrans ] // gap start
					[ 0.07 RingColor transmit 1.0 ]
					[ 0.10 RingColor transmit 1.0 ]
					[ 0.10 RingColor transmit 0.78*RingTrans ]

					[ 0.20 RingColor transmit 0.78*RingTrans ] // gap start
					[ 0.20 RingColor transmit 1.0 ]
					[ 0.22 RingColor transmit 1.0 ]
					[ 0.22 RingColor transmit 0.87*RingTrans ]

  				[ 0.32 RingColor transmit 0.87*RingTrans ] // mini gap start
					[ 0.32 RingColor transmit 1.0 ]
					[ 0.325 RingColor transmit 1.0 ]
					[ 0.325 RingColor transmit 0.87*RingTrans ]

					[ 0.46 RingColor transmit 0.87*RingTrans ] // mini gap start
					[ 0.46 RingColor transmit 1.0 ]
					[ 0.465 RingColor transmit 1.0 ]
					[ 0.465 RingColor transmit 0.87*RingTrans ]

					[ 0.60 RingColor transmit 0.87*RingTrans ] // gap start
					[ 0.60 RingColor transmit 1.0 ]
					[ 0.68 RingColor transmit 1.0 ]
					[ 0.68 RingColor transmit 0.93*RingTrans ]

					[ 0.80 RingColor transmit 0.90*RingTrans ] // gap start
					[ 0.80 RingColor transmit 1.0 ]
					[ 0.82 RingColor transmit 1.0 ]
					[ 0.82 RingColor transmit 0.90*RingTrans ]

 					[ 1.00 RingColor transmit 0.90*RingTrans ]
				}
				scale 900*Miles
				turbulence .02
		}
		normal { bumps .005 scale 20*Feet }
//		finish { ambient .1 diffuse .3 specular .1 roughness .005 }
		finish { ambient .1 diffuse .7 specular .1 roughness .005 }
	}

#declare TowerTex =
	texture {
		pigment { color rgb<0.99,0.75,0.45> }
		normal { bumps .09 scale 3*Feet }
		finish { ambient .1 diffuse .3 specular .1 roughness .050 }
	}
#declare TowerWindowTex =
	texture {
		pigment { color rgb<0.49,0.35,0.35> }
		finish { ambient .1 diffuse .3 specular .96 roughness .005 reflection .6 }
	}


#declare EmitterWhite = // beam emitter
	texture {
		pigment { color rgb<1,1,1> }
		finish { ambient 2.5 }
	}

#declare SeaWaterSurfTex =
	texture {
//		pigment { color rgbf<0.227,0.416,0.282,0.47> }
		pigment { color rgbf<0.227,0.216,0.032,0.47> }
		normal { bumps .11 scale <2*Feet,.3*Feet,2*Feet> }
		finish { ambient .001 diffuse .2 specular .1 roughness .005 reflection 0.35}
	}
#declare SeaWaterInt =
	interior { ior 1.33 }
#declare SeaWaterMat = // material
	material {
		texture { SeaWaterSurfTex }
		interior { SeaWaterInt }
	}

#declare SeaFloorTex =
	texture {
		pigment {
			bozo
				color_map {
					[ 0.0 color rgb <0.268,0.174,0.084> ]
					[ 0.5 color rgb <0.403,0.304,0.124> ]
					[ 1.0 color rgb <0.537,0.489,0.315> ]
				}
				scale 500*Feet
				turbulence .3
		}
		normal { bumps .5 scale 2.7*Meter }
		finish { ambient .1 diffuse .7 specular .2 roughness .15 }
	}

#declare FogSwirlTex =
	texture {
		pigment {
			bozo
				color_map {
					[ 0.00 color rgbf <0.268,0.174,0.084,0.60> ]
					[ 0.32 color rgbf <0.403,0.504,0.424,0.83> ]
					[ 0.75 color rgbf <0.453,0.304,0.624,0.94> ]
					[ 1.00 color rgbf <0.637,0.889,0.715,0.98> ]
				}
				scale 200*Feet
				turbulence .2
				octaves 6
				lambda 2.5
				omega 0.75
		}
//		normal { bumps .5 scale 2.7*Meter }
		finish { ambient .1 diffuse .7 specular .2 roughness .15 }
	}

#declare MountainTex = // for testing
	texture {
		pigment { color rgb<.77,.67,.45> }
		finish { ambient .1 diffuse .45 specular .08 roughness .15 }
	}

#declare BridgeTex =
	texture {
		pigment { color rgb<1,1,1> }
		finish { ambient .1 diffuse .3 specular .7 roughness .05 }
	}

#declare BridgeTowerTex =
	texture {
		pigment { color rgb<1.0,.95,.86> }
		finish { ambient .1 diffuse .3 specular .5 roughness .08 }
	}
#declare ClampInnerTex =
	texture {
		pigment {
			gradient y
				color_map {
					[ 0.00 color rgb <1,1,1> ]
					[ 0.60 color rgb <1,1,1> ]
					[ 0.60 color rgb <0.653,0.704,0.724> ]
					[ 1.00 color rgb <0.637,0.689,0.815> ]
				}
				scale 3.3*Feet
		}
		finish { ambient .1 diffuse .3 specular .7 roughness .05 }
	}



//===============================================
//                Lighting
//===============================================

#declare MorningSun =
	light_source {
		<0,0,0>
//		color rgb <0.973,0.988,0.984>*2.0 // too harsh bright
//		color rgb <0.973,0.988,0.984>*1.4
//		color rgb <0.988,0.938,0.78>*1.35
		color rgb <0.988,0.838,0.54>*3
	}

object { MorningSun
//	translate <-300*Miles,50*Miles,-300*Miles>
//	translate <500*Miles,50*Miles,100*Miles>
	translate <500000*Miles,040000*Miles,200000*Miles>
} 


#declare FillLight =
	light_source {
		<0,0,0>
		color rgb <0.890,0.525,0.361>*0.2
//		spotlight
//		point_at <0,0,700*Feet>
//		radius 20    // degrees fully lit (*2)
//		falloff 42   // degrees no lighting (*2)
//		tightness 1
		media_interaction off
		fade_distance 2400*Feet
		fade_power 1
		shadowless
	}
object {FillLight translate <-3000*Feet,100*Feet,-3000*Feet> }

#declare MoonSun =
	light_source {
		<0,0,0>
		color rgb <0.890,0.525,0.361>*0.2
//		spotlight
//		point_at <0,0,700*Feet>
//		radius 20    // degrees fully lit (*2)
//		falloff 42   // degrees no lighting (*2)
//		tightness 1
		media_interaction off
		fade_distance 2400*Feet
		fade_power 1
		shadowless
	}
//object {MoonSun translate <-3000*Feet,100*Feet,-3000*Feet> }

//===============================================
//              Environment
//===============================================

background { color rgb <0.0, 0.0, 0.0> } // black
/*
sky_sphere {
	pigment {
		gradient y
		color_map {
			[ 0.03 color rgb<0.62,0.68,.38> ] //
			[ 0.10 color rgb<0.062,0.068,0.38> ] // this is
			[ 0.5 color rgb<0.180.02,0.06> ] // this is
		}
	}
}
*/

#declare AtmoSphere =
	sphere {
		<0,0,0>,30*Miles
		texture {
			pigment {
				gradient y
				color_map {
//					[ 0.53 color rgbt<0.880,0.820,0.860,.0> ] // test white
//					[ 0.40 color rgbt<0.980,0.870,0.630,.08> ] // below ground level
					[ 0.463 color rgbt<0.199,0.088,0.062,.08> ] // below ground level
					[ 0.47 color rgbt<0.680,0.610,0.130,.13> ] // ground level
					[ 0.48 color rgbt<0.462,0.268,0.080,.20> ]
					[ 0.50 color rgbt<0.362,0.128,0.080,.36> ]
					[ 0.60 color rgbt<0.102,0.020,0.052,.80> ] // to zenith
				}
				scale 1
				translate <0,-0.45,0>
				scale 30*Miles

				turbulence .04
				octaves 6
				lambda 2.5
				omega 0.55

				rotate y*-60
				rotate <2,0,8>
			}
			finish { ambient 2.0 }
		}
		translate CamPos
//	scale 100*Feet // test
//	translate <0,90*Feet,900*Feet> // test
		hollow
//		inverse
	}


object { AtmoSphere
	clipped_by { //clipping allows the sun to illuminate the inside of the sphere
		box {
			<-50*Miles,-50*Miles,-5*Miles>,<50*Miles,50*Miles,50*Miles>
			rotate y*-45
		}
	}
}

#declare ShadowSphere = // shadows some of the bridge
	sphere {
		<0,0,0>,200*Feet
		translate <0,-120*Feet,0>
		texture { MountainTex }
	}

//===============================================
//                Objects
//===============================================

//---------------- Celestial --------------------

#declare StarShape =
	sphere {
		<0,0,0>,2.3*Miles
		texture { StarTex }
		no_shadow
	}
#declare StarCount = 1;
#declare MaxStars = 500;
#declare RS = seed(57143);
#declare DimStarField =
	union {
		#while ( StarCount < MaxStars )
			object { StarShape
				scale rand(RS)*.7+.3
				translate <0,0,5000*Miles>
				rotate <4+rand(RS)*-30,-30+rand(RS)*60,0>
				texture { pigment{color rgb<1,1,1>} finish {ambient rand(RS)*1.5+.2} }
			}
			#declare StarCount = StarCount + 1;
		#end
	}

#declare Moon =
	sphere {
		<0,0,0>,30*Miles
		texture { MoonTex }
	}

#declare RingRad = 1800*Miles;
#declare RingSystem =
	difference {
		cylinder {
			<0,0,0>,<0,1*Feet,0>,RingRad*1.5 // horizontal
		}
		cylinder {
			<0,-1,0>,<0,1.1*Feet,0>,RingRad*1.0 // horizontal
		}
		texture { RingTex }
	} // end difference

//---------------- Landscape --------------------

#declare SeaWater =
	union {
		box { <-30*KM,-100*Feet,-30*KM>,<30*KM,0,30*KM> // this is the transparent surface
			material { SeaWaterMat }
		}
		plane { y, -65*Feet // this is the ground underneath
			texture { SeaFloorTex }
		}
	} // end union
#declare SeaWaterT =  // for testing only
	box { <-30*KM,-100*Feet,-30*KM>,<30*KM,0,30*KM>
		texture { pigment { color rgb<.95,.95,1.37> } finish { ambient .3 diffuse .8 } }
	}
#declare SeaFog = // experiment
	box { <-30*KM,1*in,-30*KM>,<30*KM,1.01*in,30*KM>
		texture { FogSwirlTex }
	}


#declare MountainMap =
	height_field {
		png "HFmap01.png"
//		texture { DistantMountainsTex }
		smooth
		texture {
			pigment {
				image_map { // image is always projected from -z, with front facing +z, top to +Y
					png "Colormap.png"
					once
				}
				rotate x*90 // align map to height_field
			}
		}
		water_level .015
		translate <-0.085*in,0,-0.008*in> // align to scene origin
		scale <11.5*Miles,3000*Feet,11.5*Miles*.8265> // scale it up, Y-scale sets max height
		translate <0,-85*Feet,0> // ensure local area is below ground
	}

#declare LocalMap =
	height_field {
		png "NearHFmap02.png"
//		texture { DistantMountainsTex }
		smooth
		texture {
			pigment {
				image_map { // image is always projected from -z, with front facing +z, top to +Y
					png "Colormap.png"
					once
				}
				rotate x*90 // align map to height_field
			}
		}
		water_level .015
		translate <-0.085*in,0,-0.008*in> // align to scene origin
		scale <500*Feet,70*Feet,800*Feet*.8265> // scale it up, Y-scale sets max height
		translate <0,-5*Feet,0> // ensure local area is below ground
	}


#declare Formation =
// create a 3-D slice of a 4-D julia fractal object
	julia_fractal { 
		<-0.083,0.0,-0.83,-0.025> 
		quaternion // hypercomplex?
		cube // TYPE: sqr cube 
		max_iteration 8 
		precision 20 // 10...500? FIXME
		//  slice xx?
		rotate z*90
		scale <30*Feet,100*Feet,30*Feet>
		texture { MountainTex }
	}
#declare WierdLand =
// create a 3-D slice of a 4-D julia fractal object
	julia_fractal { 
		<-0.044,0.02,-0.33,-0.045> 
		quaternion // hypercomplex?
		cube // TYPE: sqr cube 
		max_iteration 8 
		precision 20 // 10...500? FIXME
		//  slice xx?
		rotate z*90
		scale <430*Feet,40*Feet,360*Feet>
		texture { MountainTex }
	}



//---------------- Towers --------------------

#declare THT = 200*Feet; // Tower Height

#declare SentryTower =
	union {
		cone {
			<0,-50*Feet,0>,7*Feet,<0,THT,0>,2.4*Feet
		}
		#declare FDeg = 1;
		#while ( FDeg < 180 )
			cone {
				<0,-50*Feet,0>,20*Feet,<0,THT*.7,0>,2*Feet
				scale <1,1,.1>
				rotate y*FDeg
			}
			#declare FDeg = FDeg + 30;
		#end
		cylinder { // main
			<0,0,0>,<0,4*Feet,0>,12*Feet
			translate <0,THT-34*Feet,0>
		}
		cylinder {
			<0,-3*Feet,0>,<0,1*Feet,0>,10*Feet
			translate <0,THT-34*Feet,0>
		}
		cone { // window
			<0,4*Feet,0>,11.6*Feet,<0,12*Feet,0>,9.0*Feet
			translate <0,THT-34*Feet,0>
			texture { TowerWindowTex }
		}
		cylinder { // roof
			<0,0,0>,<0,.8*Feet,0>,10*Feet
			translate <0,THT-22*Feet,0>
		}
		#declare SDeg = 1;
		#while ( SDeg < 360 )
			cylinder {
				<0,-10*Feet,0>,<0,0,0>,2.5*Feet
				translate <0,THT-34*Feet,6*Feet>
				rotate y*SDeg
			}
			cone {
				<0,-19*Feet,0>,1*in,<0,-10*Feet,0>,2.5*Feet
				translate <0,THT-34*Feet,6*Feet>
				rotate y*SDeg
			}
			#declare SDeg = SDeg + 360/4;
		#end
		texture { TowerTex }
	}

#declare BeamEmitter =
	union {
		cylinder {
			<0,0,0>,<12*Feet,0,0>,2.3*Feet
			texture { TowerTex }
//			texture { WhitePaint }
		}

		cylinder {
			<12*Feet,0,0>,<12.6*Feet,0,0>,2.7*Feet
			texture { TowerTex }
		}
		cylinder {
			<12.5*Feet,0,0>,<12.8*Feet,0,0>,1.7*Feet
			texture { EmitterWhite }
		}

	}
#macro EnergyBeam(ELength)
	union {
		object { BeamEmitter }
		object { BeamEmitter rotate y*180 translate <ELength,0,0> }

		cylinder {
			<0,0,0>,<ELength,0,0>,0.2*Feet
			texture {
				pigment { color rgbt<3,0,2,.10> } // very hot!
				finish { ambient 1.0 diffuse 0 }
			}
		}

		cylinder {
			<0,0,0>,<ELength,0,0>,1.1*Feet
			texture {
				pigment { color rgbt<1.5,0,0,.75> }
				finish { ambient 1.0 diffuse 0 }
			}
		}

		cylinder {
			<0,0,0>,<ELength,0,0>,2.2*Feet
			texture {
				pigment { color rgbt<1,0.2,0,.95> }
				finish { ambient 1.0 diffuse 0 }
			}
		}

//		no_shadow
	}
#end
#macro EnergyFence(EFLength)
	union {
		object { EnergyBeam(EFLength) translate <0,THT*.95,0> }
		object { EnergyBeam(EFLength) translate <0,THT*.64,0> }
		object { EnergyBeam(EFLength) translate <0,THT*.35,0> }
		object { EnergyBeam(EFLength) translate <0,THT*.05,0> }
//		object { EnergyBeam(EFLength) translate <0,20*Feet,0> }
	}
#end

#declare TowerGroup1 =
	union {
		object { EnergyFence(2000*Feet) rotate y*-120 }
		object { SentryTower }
	}
#declare TowerGroup2 =
	union {
		object { SentryTower }
		object { EnergyFence(1200*Feet) rotate y*-100 }
		object { TowerGroup1 rotate y*100 translate x*1200*Feet rotate y*-100 }
	}
#declare TowerGroup3 =
	union {
		object { SentryTower }
		object { EnergyFence(3400*Feet) rotate y*-100 }
		object { TowerGroup2 rotate y*100 translate x*3400*Feet rotate y*-100 }
	}
#declare TowerGroup4 =
	union {
		object { SentryTower }
		object { EnergyFence(1400*Feet) rotate y*-140 }
		object { TowerGroup3 rotate y*140 translate x*1400*Feet rotate y*-140 }
	}
#declare TowerGroup5 =
	union {
		object { SentryTower }
		object { EnergyFence(1800*Feet) rotate y*-100 }
		object { TowerGroup4 rotate y*100 translate x*1800*Feet rotate y*-100 }
		object { EnergyFence(500*Feet) rotate y*60 } // off-screen right
	}

//---------------- Bridges --------------------

#declare SBridgeTower = // simple version
	cylinder { // tower
		<0,0,0>,<0,24*Feet,0>,9*Feet
		texture { BridgeTowerTex }
	}
#declare BridgeTower =
	difference {
		cylinder { // main
			<0,0,0>,<0,24*Feet,0>,9*Feet
		}
		cylinder { // tower roof cutout
			<0,20*Feet,0>,<0,25*Feet,0>,8.2*Feet
		}
		box { // window
			<-1.5*Feet,16*Feet,-10*Feet>,<1.5*Feet,19*Feet,10*Feet>
		}
		box { // door
			<7*Feet,10*Feet,-10*Feet>,<10*Feet,16*Feet,10*Feet>
		}

		texture { BridgeTowerTex }
	} // end difference


#declare BL = 300*Feet; // bridge length
#declare Bridge1 =
	union {
		box { // main
			<-6*Feet,8*Feet,0>,<6*Feet,10*Feet,BL>
		}
		#declare SPos = 0;
		#while ( SPos < BL )
			cylinder { <0,-10*Feet,0>,<0,13*Feet,0>,0.8*Feet translate < 6.5*Feet,0,SPos> }
			cylinder { <0,-10*Feet,0>,<0,13*Feet,0>,0.8*Feet translate <-6.5*Feet,0,SPos> }
			#declare SPos = SPos + 35*Feet;
		#end
		cylinder { // guardrail
			<0,0,0>,<0,0,BL>,.23*Feet
			translate <6.5*Feet,13*Feet,0>
		}
		cylinder { // guardrail
			<0,0,0>,<0,0,BL>,.23*Feet
			translate <-6.5*Feet,13*Feet,0>
		}
		object { BridgeTower translate <-13*Feet,0,BL/2> }
		texture { BridgeTex }
	}

#declare DockingStation = // origin at sea level
	union {
		difference { // clamp LH
			cylinder { // outer
				< 0,0,0>,<30*Feet,0,0>,7*Feet
				texture { BridgeTex }
				texture { Stripe1Tex rotate y*0 scale <20*Feet,50*Feet,40*Feet> translate <14*Feet,-15*Feet,0> }
			}
			cylinder { // inner
				<-1,0,0>,<31*Feet,0,0>,6*Feet
				texture { ClampInnerTex translate <0,2*Feet,0> }
			}
			cylinder { <15*Feet,17*Feet,-8*Feet>,<15*Feet,17*Feet,8*Feet>13*Feet }
			box { <-1,-8*Feet,-8*Feet>,<31*Feet,8*Feet,1*Feet> }
			translate <0,15*Feet,11*Feet>
		}
		difference { // clamp RH
			cylinder {
				< 0,0,0>,<30*Feet,0,0>,7*Feet
				texture { BridgeTex }
				texture { Stripe1Tex rotate y*0 scale <20*Feet,50*Feet,40*Feet> translate <14*Feet,-15*Feet,0> }
			}
			cylinder { <-1,0,0>,<31*Feet,0,0>,6*Feet }
			cylinder { <15*Feet,17*Feet,-8*Feet>,<15*Feet,17*Feet,8*Feet>13*Feet }
			box { <-1,-8*Feet,-1*Feet>,<31*Feet,8*Feet,8*Feet> }
			translate <0,15*Feet,-11*Feet>
		}
		cylinder { <0,0, 10*Feet>,<0,0, 18*Feet>,3*Feet translate < 5*Feet,5*Feet,0> } // clamp cyl
		cylinder { <0,0,-11*Feet>,<0,0, 11*Feet>,1.7*Feet translate < 5*Feet,5*Feet,0> } // clamp cyl
		cylinder { <0,0,-10*Feet>,<0,0,-18*Feet>,3*Feet translate < 5*Feet,5*Feet,0> } // clamp cyl

		cylinder { <0,0, 10*Feet>,<0,0, 18*Feet>,3*Feet translate <25*Feet,5*Feet,0> } // clamp cyl
		cylinder { <0,0,-11*Feet>,<0,0, 11*Feet>,1.7*Feet translate <25*Feet,5*Feet,0> } // clamp cyl
		cylinder { <0,0,-10*Feet>,<0,0,-18*Feet>,3*Feet translate <25*Feet,5*Feet,0> } // clamp cyl

		box { <-22*in,0,0>,<22*in,6*Feet,5*Feet> translate <15*Feet,10*Feet,10*Feet> } // door
		cylinder { <0,-30*Feet,0>,<0,5*Feet,0>,14*Feet scale <1,1,.3> translate <15*Feet,0,0> } // support
		box { <0,-30*Feet,0>,<30*Feet,10*Feet,6*Feet> translate <0,0,15*Feet> } // support
		box { <-2.3*Feet,-8*in,14.5*Feet>,<2.3*Feet,0,65*Feet> translate <15*Feet,10*Feet,0> } // walkway
		cylinder { <-2.3*Feet,18*in,14.5*Feet>,<-2.3*Feet,18*in,65*Feet>,3*in translate <15*Feet,10*Feet,0> } // rail
		cylinder { <-2.3*Feet,36*in,14.5*Feet>,<-2.3*Feet,36*in,65*Feet>,3*in translate <15*Feet,10*Feet,0> } // rail
		cylinder { < 2.3*Feet,18*in,14.5*Feet>,< 2.3*Feet,18*in,65*Feet>,3*in translate <15*Feet,10*Feet,0> } // rail
		cylinder { < 2.3*Feet,36*in,14.5*Feet>,< 2.3*Feet,36*in,65*Feet>,3*in translate <15*Feet,10*Feet,0> } // rail
		texture { BridgeTex }
	}

//---------------- Other Stuff --------------------

#declare ShipWake =
	height_field {
		png "Wake01.png"
		smooth
//		texture { WhitePaint } // test
		material { SeaWaterMat }
		translate <-0.5,0,0>
		water_level .01
		scale <200*Feet,6*Feet,600*Feet>

	}
/*
#declare NormBox =
	box {
		<0,0,0>,<1,1,1>
		texture {
			pigment { rgb<1,1,1> }
			normal {
				bump_map {
					png "NormalMapTest.png"
				}
				bump_size 4
			}
		}
		scale <30*Feet,30*Feet,3*Feet>
	}
object { NormBox rotate <0,-40,0> rotate <20,0,0> translate <0, 100*Feet,70*Feet> }                               
*/

#declare CheckerPlane =
plane { z, 6550*Feet hollow texture { pigment { checker pigment { rgb<1,1,1>} pigment { rgb<0,0,0>} scale 230*Feet } finish { ambient .4}} rotate y*-40}
//object { CheckerPlane }


//===============================================
//              Object Placement
//===============================================

// ENVIRONMENT

object { DimStarField }
object { Moon translate <0,0,5500*Miles> rotate <-22,15,0> }
object { RingSystem  rotate <45,-235,0> translate <0,-500*Miles,0>}
object { SeaWater }

object { ShadowSphere translate <550*Feet,-10*Feet,930*Feet> }
object { ShadowSphere translate <550*Feet,  0*Feet,560*Feet> }

object { MountainMap rotate y*-38 translate <0*Feet,0,0*Feet> }
object { LocalMap rotate y*-27 translate <-220*Feet,0,180*Feet> }

object { Formation scale <1,.6,1> rotate y*-90 translate <120*Feet,0,660*Feet> } // near right tower
object { Formation scale 1 rotate <12,180,0> translate <-130*Feet,0,2700*Feet> } // second tower
object { Formation scale 1 rotate <12,180,0> translate <-1350*Feet,0,4000*Feet> } // third tower

object { WierdLand rotate <0,0,0> translate <-1000*Feet,0,2100*Feet> } // centerleft
object { WierdLand scale <1,1.3,1> rotate <0,90,0> translate <-1400*Feet,0,3100*Feet> } // centerleft
object { Formation scale <1,.34,1> rotate <12,180,0> translate <-670*Feet,0,1700*Feet> } // centerleft
object { Formation scale <1,.30,.8> rotate <-7,98,0> translate <-700*Feet,0,1650*Feet> } // centerleft
object { Formation scale <.6,.23,.7> rotate <-17,248,0> translate <-500*Feet,0,1400*Feet> } // centerleft
object { Formation scale <.46,.13,.41> rotate < 10,323,4> translate <-400*Feet,0,1380*Feet> } // centerleft
object { Formation scale <.26,.10,.27> rotate <  4,129,2> translate <-320*Feet,0,1410*Feet> } // centerleft

object { TowerGroup5 scale <1,1,1> translate <260*Feet,0,880*Feet> }

object { Bridge1 rotate y*30 translate <120*Feet,0,660*Feet> } // near right
object { Bridge1 rotate y*170 translate <120*Feet,0,660*Feet> } // nearest right
object { DockingStation translate <0,0,-60*Feet> rotate y*(60) translate <110*Feet,0,600*Feet> } // nearest right

object { PatrolShip rotate z*0 rotate <0,-24,0> translate <-240*Feet,235*Feet,1500*Feet> }
object { VaporTrail rotate z*0 rotate <0,-24,0> translate <-240*Feet,2*Feet,1500*Feet> }
object { ShipWake rotate z*0 rotate <0,-24,0> translate <-240*Feet,0*Feet,1500*Feet> }

// TEST OBJECTS - DO NOT RENDER
//object { PatrolShip rotate z*0 rotate <0,-28,0> translate <-240*Feet,260*Feet,1500*Feet> }
//object { SeaFog }
//object { SeaWaterT }