// ===========================================================================
//
// IRTC 'Gardens' - grass.inc
//
// (C) Ian Shumsky / OuterArm 1999
//
// ===========================================================================
//
// This include file defines individual grass blades, creates 'patches' of
// blades and provides a positioning macro. The individual blades are mesh
// objects which can be generated for each image, written to files or imported
// from files. The write_mesh flag indicates if the mesh objects are written
// out to a file. If Nathan Kopp's POV MegaMatch is being used, the mesh2
// object definition is available. This thype of mesh is far more memory
// efficient, which is good for the number of blades in the scene. By setting
// both the load_mesh and use_mesh2 flags to 1, mesh2 versions of the blade
// will be loaded and used. The grass blade meshes are based on macros by
// Gilles Tran. The trace_grass flag plays an important part of this file.
// Valid values are:
//
// Value   Effect
// ------ -------
//     0 - No grass
//     1 - Simple grass blade, 1/2 density grass patches
//     2 - Full defined grass blade, full patch density
//     3 - Extra grass patches which only add a handful of blades
//
// ===========================================================================

//
// The following defines if a mesh is written out. 0 don't, 1 do.
//

#declare write_mesh = 0;

//
// the following says if we just load predefined blade meshs. 0 don't, 1 do
//

#declare load_mesh = 1;

#declare grass_random = seed (100);

#if (trace_grass > 0)
	//
	// Build a proper patch of grass from individual blades.
	//

	#declare Blanc1 = rgb <0.90, 0.81, 0.81>;
	#declare Green1 = rgb <230, 170, 50> / 255;
	#declare Green2 = rgb <110, 160, 8> / 255;

	#declare SpottedPig = pigment
	{
		spotted
		color_map
		{
			[0.0, 0.2 color Tan*1.3 color Tan*0.4]
			[0.2, 0.3 color Tan*0.4 color Green2]
			[0.3, 1.01 color Green2 color Green2*1.2]
		}
		scale 0.01
	}

	#declare EarthPig = pigment
	{
		spotted
		color_map
		{
			[0 color Tan*0.3]
			[0.6 color Tan*0.3]
			[0.6 color Clear]
			[1 color Clear]
		}
		scale <0.001, 2, 0.001>
	}

	#declare GreenPig = pigment
	{
		Green1 * 1.3
	}

	#declare GrassTxt = texture
	{
		pigment
		{
			gradient y
			turbulence 0.2
			pigment_map
			{
				[0.00 EarthPig]
				[0.15 GreenPig]
				[0.9 SpottedPig]
				[1.00 EarthPig]
			}
		}
		finish
		{
			ambient 0.01
			diffuse 0.7
			specular 0.4
			roughness 0.015
		}
	}

	#if (load_mesh = 1)
		#if (use_mesh2 = 1)
			#include "grass_blade_1_mesh2.inc"
			#include "grass_blade_2_mesh2.inc"
		#else
			#include "grass_blade_1_mesh.inc"
			#include "grass_blade_2_mesh.inc"
		#end
	#else
		//
		// Build an individual blade
		//

		#if (write_mesh = 1)
			#fopen fh1 "grass_blade_1_mesh.inc" write
			#write (fh1, "#declare Blade1 = mesh\n{\n")
		#end

		#declare zBrin0 = 7;
		#declare zBrin = zBrin0;
		#declare TetaZDebut = 0;
		#declare TetaZFin = 2*pi;
		#declare TetaYDebut = 0;
		#declare TetaYFin = pi;
		#declare rTerre = 150;

		#declare iTetaZ = pi/50;
		#declare iTetaY = pi/50;
		#declare TetaY = 0;
		#declare yMax = 0;
		#declare zBrin0 = 4;
		#declare zBrin = zBrin0;

		#declare BScale = 10;

		#declare Blade1 = mesh
		{
			#declare CountRand = 1;
			#declare xS = 0;
			#declare zS0 = 0;
			#declare zBrin = zBrin0;
			#declare zBrinFin = zBrin0*0.1;
			#declare xS0 = 0;
			#declare yS0 = 0;
			#declare zS0 = zBrin;
			#declare Count = 1;
			#declare X3 = -1;
			#declare X2 = 46;
			#declare X1 = 3;
			#declare DV1 = 1;
			#declare DV2 = 2500;

			#if (trace_grass = 1)
				#declare nS = 2;
			#else
				#declare nS = 50;
			#end

			#declare dDebut = 5;
			#declare dFin = 0.5;
			#declare nC = 55;

			#while (Count<nS+1)
				#declare yS = Count*nC/nS;
				#declare xS = ((X3*pow(yS/DV1, 3))+(X2*pow(yS/DV1, 2))+(X1*yS/DV1))/DV2;
				#declare zS = (zBrin-(Count*(zBrin-zBrinFin)/nS));
				#declare ralpha = atan2(zS, xS);
				#declare ralpha0 = atan2(zS0, xS0);
				#declare xL = sqrt(pow(xS, 2)+pow(zS, 2));
				#declare xL0 = sqrt(pow(xS0, 2)+pow(zS0, 2));

				#declare xS0a = (xL0*cos(-ralpha0));
				#declare xS0b = (xL0*cos(ralpha0));
				#declare yS0a = yS0;
				#declare zS0a = (xL0*sin(-ralpha0));
				#declare zS0b = (xL0*sin(ralpha0));

				#declare xSa = (xL*cos(-ralpha));
				#declare xSb = (xL*cos(+ralpha));
				#declare ySa = yS;
				#declare zSa = (xL*sin(-ralpha));
				#declare zSb = (xL*sin(ralpha));

				#declare xM0 = xS0-(zS0/3);
				#declare xM = xS-(zS/3);

				#declare xM0a = xM0;
				#declare xMa = xM;

				#declare yM0a = yS0a;
				#declare yMa = ySa;

				#declare zM0a = 0;
				#declare zMa = 0;

				triangle
				{
					<xS0a/BScale, yS0a/BScale, zS0a/BScale>,
					<xMa/BScale, yMa/BScale, zMa/BScale>,
					<xSa/BScale, ySa/BScale, zSa/BScale>
				}

				triangle
				{
					<xS0a/BScale, yS0a/BScale, zS0a/BScale>,
					<xMa/BScale, yMa/BScale, zMa/BScale>,
					<xM0a/BScale, yM0a/BScale, zM0a/BScale>
				}

				triangle
				{
					<xS0b/BScale, yS0a/BScale, zS0b/BScale>,
					<xMa/BScale, yMa/BScale, zMa/BScale>,
					<xM0a/BScale, yM0a/BScale, zM0a/BScale>
				}

				triangle
				{
					<xS0b/BScale, yS0a/BScale, zS0b/BScale>,
					<xMa/BScale, yMa/BScale, zMa/BScale>,
					<xSb/BScale, ySa/BScale, zSb/BScale>
				}

				#if (write_mesh = 1)
					#write (fh1, "triangle {<",
						xS0a/BScale, ",",
						yS0a/BScale, ",",
						zS0a/BScale, ">,<",
						xMa/BScale,  ",",
						yMa/BScale,  ",",
						zMa/BScale, ">,<",
						xSa/BScale,  ",",
						ySa/BScale,  ",",
						zSa/BScale, ">}\n")

					#write (fh1, "triangle {<",
						xS0a/BScale, ",",
						yS0a/BScale, ",",
						zS0a/BScale, ">,<",
						xMa/BScale, ",",
						yMa/BScale, ",",
						zMa/BScale, ">,<",
						xM0a/BScale, ",",
						yM0a/BScale, ",",
						zM0a/BScale, ">}\n")

					#write (fh1, "triangle {<",
						xS0b/BScale, ",",
						yS0a/BScale, ",",
						zS0b/BScale, ">,<",
						xMa/BScale, ",",
						yMa/BScale, ",",
						zMa/BScale, ">,<",
						xM0a/BScale, ",",
						yM0a/BScale, ",",
						zM0a/BScale, ">}\n")

					#write (fh1, "triangle {<",
						xS0b/BScale, ",",
						yS0a/BScale, ",",
						zS0b/BScale, ">,<",
						xMa/BScale, ",",
						yMa/BScale, ",",
						zMa/BScale, ">,<",
						xSb/BScale, ",",
						ySa/BScale, ",",
						zSb/BScale, ">}\n")
				#end

				#if (ySa>yMax)
					#declare yMax = ySa;
				#end

				#declare yS0 = yS;
				#declare xS0 = xS;
				#declare zS0 = zS;
				#declare Count = Count+1;
			#end
		}

		#if (write_mesh = 1)
			#write (fh1, "}\n")
			#write (fh1, "#declare yMax = ", yMax, ";\n")
			#fclose fh1
		#end

		//
		// Build an individual blade
		//

		#if (write_mesh = 1)
			#fopen fh2 "grass_blade_2_mesh.inc" write
			#write (fh2, "#declare Blade2 = mesh\n{\n")
		#end

		#declare zBrin0 = 7;
		#declare zBrin = zBrin0;
		#declare TetaZDebut = 0;
		#declare TetaZFin = 2*pi;
		#declare TetaYDebut = 0;
		#declare TetaYFin = pi;
		#declare rTerre = 150;

		#declare iTetaZ = pi/50;
		#declare iTetaY = pi/50;
		#declare TetaY = 0;
		#declare yMax = 0;
		#declare zBrin0 = 4;
		#declare zBrin = zBrin0;

		#declare BScale = 10;

		#declare Blade2 = mesh
		{
			#declare CountRand = 1;
			#declare xS = 0;
			#declare zS0 = 0;
			#declare zBrin = zBrin0;
			#declare zBrinFin = zBrin0*0.1;
			#declare xS0 = 0;
			#declare yS0 = 0;
			#declare zS0 = zBrin;
			#declare Count = 1;
			#declare X3 = 0;
			#declare X2 = 0;
			#declare X1 = 0;
			#declare DV1 = 1;
			#declare DV2 = 2500;

			#if (trace_grass = 1)
				#declare nS = 2;
			#else
				#declare nS = 50;
			#end

			#declare dDebut = 5;
			#declare dFin = 0.5;
			#declare nC = 55;

			#while (Count<nS+1)
				#declare yS = Count*nC/nS;
				#declare xS = ((X3*pow(yS/DV1, 3))+(X2*pow(yS/DV1, 2))+(X1*yS/DV1))/DV2;
				#declare zS = (zBrin-(Count*(zBrin-zBrinFin)/nS));
				#declare ralpha = atan2(zS, xS);
				#declare ralpha0 = atan2(zS0, xS0);
				#declare xL = sqrt(pow(xS, 2)+pow(zS, 2));
				#declare xL0 = sqrt(pow(xS0, 2)+pow(zS0, 2));

				#declare xS0a = (xL0*cos(-ralpha0));
				#declare xS0b = (xL0*cos(ralpha0));
				#declare yS0a = yS0;
				#declare zS0a = (xL0*sin(-ralpha0));
				#declare zS0b = (xL0*sin(ralpha0));

				#declare xSa = (xL*cos(-ralpha));
				#declare xSb = (xL*cos(+ralpha));
				#declare ySa = yS;
				#declare zSa = (xL*sin(-ralpha));
				#declare zSb = (xL*sin(ralpha));

				#declare xM0 = xS0-(zS0/3);
				#declare xM = xS-(zS/3);

				#declare xM0a = xM0;
				#declare xMa = xM;

				#declare yM0a = yS0a;
				#declare yMa = ySa;

				#declare zM0a = 0;
				#declare zMa = 0;

				triangle
				{
					<xS0a/BScale, yS0a/BScale, zS0a/BScale>,
					<xMa/BScale, yMa/BScale, zMa/BScale>,
					<xSa/BScale, ySa/BScale, zSa/BScale>
				}

				triangle
				{
					<xS0a/BScale, yS0a/BScale, zS0a/BScale>,
					<xMa/BScale, yMa/BScale, zMa/BScale>,
					<xM0a/BScale, yM0a/BScale, zM0a/BScale>
				}

				triangle
				{
					<xS0b/BScale, yS0a/BScale, zS0b/BScale>,
					<xMa/BScale, yMa/BScale, zMa/BScale>,
					<xM0a/BScale, yM0a/BScale, zM0a/BScale>
				}

				triangle
				{
					<xS0b/BScale, yS0a/BScale, zS0b/BScale>,
					<xMa/BScale, yMa/BScale, zMa/BScale>,
					<xSb/BScale, ySa/BScale, zSb/BScale>
				}

				#if (write_mesh = 1)
					#write (fh2, "triangle {<",
						xS0a/BScale, ",",
						yS0a/BScale, ",",
						zS0a/BScale, ">,<",
						xMa/BScale,  ",",
						yMa/BScale,  ",",
						zMa/BScale, ">,<",
						xSa/BScale,  ",",
						ySa/BScale,  ",",
						zSa/BScale, ">}\n")

					#write (fh2, "triangle {<",
						xS0a/BScale, ",",
						yS0a/BScale, ",",
						zS0a/BScale, ">,<",
						xMa/BScale, ",",
						yMa/BScale, ",",
						zMa/BScale, ">,<",
						xM0a/BScale, ",",
						yM0a/BScale, ",",
						zM0a/BScale, ">}\n")

					#write (fh2, "triangle {<",
						xS0b/BScale, ",",
						yS0a/BScale, ",",
						zS0b/BScale, ">,<",
						xMa/BScale, ",",
						yMa/BScale, ",",
						zMa/BScale, ">,<",
						xM0a/BScale, ",",
						yM0a/BScale, ",",
						zM0a/BScale, ">}\n")

					#write (fh2, "triangle {<",
						xS0b/BScale, ",",
						yS0a/BScale, ",",
						zS0b/BScale, ">,<",
						xMa/BScale, ",",
						yMa/BScale, ",",
						zMa/BScale, ">,<",
						xSb/BScale, ",",
						ySa/BScale, ",",
						zSb/BScale, ">}\n")
				#end

				#if (ySa>yMax)
					#declare yMax = ySa;
				#end

				#declare yS0 = yS;
				#declare xS0 = xS;
				#declare zS0 = zS;
				#declare Count = Count+1;
			#end
		}

		#if (write_mesh = 1)
			#write (fh2, "}\n")
			#write (fh2, "#declare yMax = ", yMax, ";\n")
			#fclose fh2
		#end

	#end // load or build.

	//
	// Build the first grass patch
	//

	#if (trace_grass = 2)
		//
		// Maximum density.
		//

		#declare nBrinX = 25;
		#declare nBrinZ = 25;
		#declare iBrin = 10;
		#declare BScale = 20 ;
	#else
		//
		// 1/2 density
		//

		#declare nBrinX = 12.5;
		#declare nBrinZ = 12.5;
		#declare iBrin = 10;
		#declare BScale = 10 ;
	#end

	#declare grass_patch_1 = union
	{
		#declare cX = -nBrinX;
		#while (cX<nBrinX)
			#declare cZ = -nBrinZ;
			#while (cZ<nBrinZ)
				#declare scBrin = (rand(grass_random)+rand(grass_random))*0.5 +0.5;
				#declare tetaBrin = 360*(rand(grass_random)+rand(grass_random));
				#declare mvX = iBrin*(1-rand(grass_random)-rand(grass_random));
				#declare mvZ = iBrin*(1-rand(grass_random)-rand(grass_random));

				#declare blade_type = rand(grass_random) * 2;
				#declare place_type = rand(grass_random) * 5;

				#if (blade_type < 1)
					#if (place_type < 1)
						object
						{
							Blade1
							scale 0.5
							rotate <tetaBrin/50, tetaBrin, tetaBrin/50>
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#else
						object
						{
							Blade1
							scale 0.5
							rotate x * (-30 + (60 * rand (grass_random)))
							rotate y * (-30 + (60 * rand (grass_random)))
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}

					#end
				#else
					#if (place_type < 1)
						object
						{
							Blade2
							scale 0.5
							rotate <tetaBrin/50, tetaBrin, tetaBrin/50>
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#else
						object
						{
							Blade2
							rotate x * (-30 + (60 * rand (grass_random)))
							rotate y * (-30 + (60 * rand (grass_random)))
							scale 0.8 + (rand(grass_random) / 2.5)
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#end
				#end

				#declare cZ = cZ+1;
			#end
			#declare cX = cX+1;
		#end

		texture
		{
			GrassTxt
			scale (yMax*1.2) / 10
			translate (y-0.1) / 10
		}
	}

	//
	// Build the second grass patch
	//

	#if (trace_grass = 2)
		//
		// Maximum density.
		//

		#declare nBrinX = 25;
		#declare nBrinZ = 25;
		#declare iBrin = 10;
		#declare BScale = 20 ;
	#else
		//
		// 1/2 density
		//

		#declare nBrinX = 12.5;
		#declare nBrinZ = 12.5;
		#declare iBrin = 10;
		#declare BScale = 10 ;
	#end

	#declare grass_patch_2 = union
	{
		#declare cX = -nBrinX;
		#while (cX<nBrinX)
			#declare cZ = -nBrinZ;
			#while (cZ<nBrinZ)
				#declare scBrin = (rand(grass_random)+rand(grass_random))*0.5 +0.5;
				#declare tetaBrin = 360*(rand(grass_random)+rand(grass_random));
				#declare mvX = iBrin*(1-rand(grass_random)-rand(grass_random));
				#declare mvZ = iBrin*(1-rand(grass_random)-rand(grass_random));

				#declare blade_type = rand(grass_random) * 2;
				#declare place_type = rand(grass_random) * 5;

				#if (blade_type < 1)
					#if (place_type < 1)
						object
						{
							Blade1
							scale 0.5
							rotate <tetaBrin/50, tetaBrin, tetaBrin/50>
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#else
						object
						{
							Blade1
							scale 0.5
							rotate x * (-30 + (60 * rand (grass_random)))
							rotate y * (-30 + (60 * rand (grass_random)))
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}

					#end
				#else
					#if (place_type < 1)
						object
						{
							Blade2
							scale 0.5
							rotate <tetaBrin/50, tetaBrin, tetaBrin/50>
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#else
						object
						{
							Blade2
							rotate x * (-30 + (60 * rand (grass_random)))
							rotate y * (-30 + (60 * rand (grass_random)))
							scale 0.8 + (rand(grass_random) / 2.5)
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#end
				#end

				#declare cZ = cZ+1;
			#end
			#declare cX = cX+1;
		#end

		texture
		{
			GrassTxt
			scale (yMax*1.2) / 10
			translate (y-0.1) / 10
		}
	}

	//
	// Build the third grass patch
	//

	#if (trace_grass = 2)
		//
		// Maximum density.
		//

		#declare nBrinX = 25;
		#declare nBrinZ = 25;
		#declare iBrin = 10;
		#declare BScale = 20 ;
	#else
		//
		// 1/2 density
		//

		#declare nBrinX = 12.5;
		#declare nBrinZ = 12.5;
		#declare iBrin = 10;
		#declare BScale = 10 ;
	#end

	#declare grass_patch_3 = union
	{
		#declare cX = -nBrinX;
		#while (cX<nBrinX)
			#declare cZ = -nBrinZ;
			#while (cZ<nBrinZ)
				#declare scBrin = (rand(grass_random)+rand(grass_random))*0.5 +0.5;
				#declare tetaBrin = 360*(rand(grass_random)+rand(grass_random));
				#declare mvX = iBrin*(1-rand(grass_random)-rand(grass_random));
				#declare mvZ = iBrin*(1-rand(grass_random)-rand(grass_random));

				#declare blade_type = rand(grass_random) * 2;
				#declare place_type = rand(grass_random) * 5;

				#if (blade_type < 1)
					#if (place_type < 1)
						object
						{
							Blade1
							scale 0.5
							rotate <tetaBrin/50, tetaBrin, tetaBrin/50>
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#else
						object
						{
							Blade1
							scale 0.5
							rotate x * (-30 + (60 * rand (grass_random)))
							rotate y * (-30 + (60 * rand (grass_random)))
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}

					#end
				#else
					#if (place_type < 1)
						object
						{
							Blade2
							scale 0.5
							rotate <tetaBrin/50, tetaBrin, tetaBrin/50>
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#else
						object
						{
							Blade2
							rotate x * (-30 + (60 * rand (grass_random)))
							rotate y * (-30 + (60 * rand (grass_random)))
							scale 0.8 + (rand(grass_random) / 2.5)
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#end
				#end

				#declare cZ = cZ+1;
			#end
			#declare cX = cX+1;
		#end

		texture
		{
			GrassTxt
			scale (yMax*1.2) / 10
			translate (y-0.1) / 10
		}
	}

	//
	// Build the fourth grass patch
	//

	#if (trace_grass = 2)
		//
		// Maximum density.
		//

		#declare nBrinX = 25;
		#declare nBrinZ = 25;
		#declare iBrin = 10;
		#declare BScale = 20 ;
	#else
		//
		// 1/2 density
		//

		#declare nBrinX = 12.5;
		#declare nBrinZ = 12.5;
		#declare iBrin = 10;
		#declare BScale = 10 ;
	#end

	#declare grass_patch_4 = union
	{
		#declare cX = -nBrinX;
		#while (cX<nBrinX)
			#declare cZ = -nBrinZ;
			#while (cZ<nBrinZ)
				#declare scBrin = (rand(grass_random)+rand(grass_random))*0.5 +0.5;
				#declare tetaBrin = 360*(rand(grass_random)+rand(grass_random));
				#declare mvX = iBrin*(1-rand(grass_random)-rand(grass_random));
				#declare mvZ = iBrin*(1-rand(grass_random)-rand(grass_random));

				#declare blade_type = rand(grass_random) * 2;
				#declare place_type = rand(grass_random) * 5;

				#if (blade_type < 1)
					#if (place_type < 1)
						object
						{
							Blade1
							scale 0.5
							rotate <tetaBrin/50, tetaBrin, tetaBrin/50>
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#else
						object
						{
							Blade1
							scale 0.5
							rotate x * (-30 + (60 * rand (grass_random)))
							rotate y * (-30 + (60 * rand (grass_random)))
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#end
				#else
					#if (place_type < 1)
						object
						{
							Blade2
							scale 0.5
							rotate <tetaBrin/50, tetaBrin, tetaBrin/50>
							scale scBrin*2
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#else
						object
						{
							Blade2
							rotate x * (-30 + (60 * rand (grass_random)))
							rotate y * (-30 + (60 * rand (grass_random)))
							scale 0.8 + (rand(grass_random) / 2.5)
							translate <((cX*iBrin) +mvX)/BScale, 0, ((cZ*iBrin)+mvZ)/BScale>
						}
					#end
				#end

				#declare cZ = cZ+1;
			#end
			#declare cX = cX+1;
		#end

		texture
		{
			GrassTxt
			scale (yMax*1.2) / 10
			translate (y-0.1) / 10
		}
	}

	//
	// Create a simple macro that takes an x and y and rotation
	// and places a grass patch at the specified point.
	//

	#macro put_grass_patch (grass_x, grass_z, grass_rot)
		#local use_grass_patch = rand (grass_random) * 4;

		object
		{
			#if (use_grass_patch < 1)
				grass_patch_1
			#else
				#if (use_grass_patch < 2)
					grass_patch_2
				#else
					#if (use_grass_patch < 3)
						grass_patch_3
					#else
						grass_patch_4
					#end
				#end
			#end

			rotate y*grass_rot
			translate <grass_patch_size * grass_x, 3.5, grass_patch_size * grass_z>
		}
	#end
#end

