// ============================================================================
//
// Atlantic Still Life
//
// Entry for July/August IRTC - Topic 'Sea'
//
// All code and images copyright (c) Ian Shumsky / Outerarm 2000
//
// ============================================================================

#version unofficial Megapov 0.5;

// ============================================================================
//
// This file defines a nice frame-type globe. As almost everything is
// controlled by a few parameters (defined in asl.pov) this object could be
// easily converted to a macro.
//
// ============================================================================

// ============================================================================
//
// First we need to build the frame.
//
// ============================================================================

#declare globe_vframe_element = torus
{
	inner_max - (vframe_thickness / 2), vframe_thickness / 2
	rotate <90, 0, 0>
}

#declare globe_vertical_frame = union
{
	#declare vangle = 18;
	#declare vcount = 0;

	#while (vcount < (180 / vangle))
		object
		{
			globe_vframe_element
			rotate <0, vcount * vangle, 0>

			#if (vcount = 0)
				texture
				{
					globe_silver_t
				}
			#else
				texture
				{
					globe_bronze_t
				}
			#end

			#if (do_photons = true)
				interior
				{
					ior 4
				}
				photons
				{
					target reflection on
				}
			#end
		}

		#declare vcount = vcount + 1;
	#end
}

#declare globe_pole_minus = torus
{
	inner_max - (vframe_thickness / 2), (vframe_thickness / 2) + 0.0002
	rotate <90, 0, 0>
}

#declare globe_ice_caps = difference
{
	cylinder
	{
		<0, (inner_max - (vframe_thickness / 2)) + 0.005, 0>, <0, -((inner_max - (vframe_thickness / 2)) + 0.005), 0>, 0.03
	}
	box
	{
		<-1, (inner_max - (vframe_thickness / 2)) -0.005, -1>, <1, -((inner_max - (vframe_thickness / 2)) - 0.005), 1>
	}

	#declare vangle = 18;
	#declare vcount = 0;

	#while (vcount < (180 / vangle))
		object
		{
			globe_pole_minus
			rotate <0, vcount * vangle, 0>
		}

		#declare vcount = vcount + 1;
	#end

	texture
	{
		globe_silver_t
	}

	#if (do_photons = true)
		interior
		{
			ior 4
		}
		photons
		{
			target reflection on
		}
	#end
}

#declare globe_horizontal_frame = union
{
	//
	// We are assuming 7 horizontal rings... This gives us angles
	// of 67.5, 45, 22.5, 0, -22.5 -45, -67.5. If we were having 5 rings
	// the angles would be 60, 30, 0, -30, -60.
	//

	#declare base_radius = inner_max - vframe_thickness - (hframe_thickness / 2);

	//
	// equator
	//

	torus
	{
		base_radius, hframe_thickness / 2
		texture
		{
			globe_silver_t
		}
		#if (do_photons = true)
			interior
			{
				ior 4
			}
			photons
			{
				target reflection on
			}
		#end
	}

	//
	// The rest...
	//

	union
	{
		#declare new_rad = cos (radians (22.5)) * base_radius;
		#declare new_height = sin (radians (22.5)) * base_radius;

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, new_height, 0>
		}

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, -(new_height), 0>
		}

		#declare new_rad = cos (radians (45)) * base_radius;
		#declare new_height = sin (radians (45)) * base_radius;

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, new_height, 0>
		}

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, -(new_height), 0>
		}

		#declare new_rad = cos (radians (67.5)) * base_radius;
		#declare new_height = sin (radians (67.5)) * base_radius;

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, new_height, 0>
		}

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, -(new_height), 0>
		}

		texture
		{
			globe_bronze_t
		}

		#if (do_photons = true)
			interior
			{
				ior 4
			}
			photons
			{
				target reflection on
			}
		#end
	}
}

// ============================================================================
//
// Now define the landmass on the globe.
// First we declare a function with the map_type 1 (spherical) mapping
//
// ============================================================================

#declare land_heightfield = function
{
	pigment
	{
		image_map
		{
			png "land_hf.png"
			map_type 1
			interpolate 2
		}
	}
}

//
// Now the isosurface
// try several rs values <1; the higher it is the higher is the text
// Also adjust max_gradient to your picture otherwise strange things happen
//

#declare rs=0.3;

#declare base_land_mass = isosurface
{
function
	{
		x*x+y*y+z*z + rs - land_heightfield(x,y,z)*rs
	}
	contained_by
	{
		sphere
		{
			<0, 0, 0>, 1
		}
	}
	eval
	max_gradient 150 
	threshold 1
}
	
#declare scaled_land_mass =  object
{
	base_land_mass
	scale (inner_max + land_mass_thickness)
}

#declare globe_land_mass = difference
{
	object
	{
		scaled_land_mass
	}
	sphere
	{
		<0, 0, 0>, inner_max
	}
	texture
	{
		globe_gold_t
	}
	#if (do_photons = true)
		interior
		{
			ior 4
		}
		photons
		{
			target reflection on
		}
	#end
}

// ============================================================================
//
// Add the frame and the land together.
//
// ============================================================================

#declare globe_earth = union
{
	#if (do_globe_frame = true)
		object
		{
			globe_vertical_frame 
		}
		object
		{
			globe_horizontal_frame 
		}
		object
		{
			globe_ice_caps 
		}
	#end
	#if (do_globe_land = true)
		object
		{
			globe_land_mass
		}
	#end

	//
	// And rotate it where requested.
	//

	rotate <0, globe_rotation, 0>
}

// ============================================================================
//
// Now build the globe holder.
//
// ============================================================================

//
// The screw cap at the top and bottom of the globe
//

#declare screw_cap = difference
{
	#declare cap_radius = 0.006; // 6mm
	#declare cap_height = 0.01;  // 1cm

	union
	{
		cylinder
		{
			<0, 0, 0>, <0, cap_height, 0>, cap_radius
		}
		sphere
		{
			<0, 0, 0>, cap_radius
			scale <1, 0.25, 1>
			translate <0, cap_height, 0>
		}
	}

	cylinder
	{
		<0, -1, 0>, <0, cap_height / 2, 0>, 0.005
	}

	#declare diff_count = 0;
	#declare diff_angle = 20;
	#declare max_diff_count = 360 / diff_angle;

	#declare diff_radius = 0.0006; // 0.6mm

	#while (diff_count < max_diff_count)
		cylinder
		{
			<0, -1, 0>, <0, 1, 0>, diff_radius
			scale <.5, 1, 1>
			translate <cap_radius, 0, 0>
			rotate y*diff_count*diff_angle
		}
		#declare diff_count = diff_count + 1;
	#end
}

#declare holder_diff_long = union
{
	cylinder
	{
		<0, 0, 0>, <-(inner_max + land_mass_thickness + 0.0280),0, 0>, 0.001
	}
	sphere
	{
		<-(inner_max + land_mass_thickness + 0.0280),0, 0>, 0.001
	}
}

#declare holder_diff_mid= union
{
	cylinder
	{
		<0, 0, 0>, <-(inner_max + land_mass_thickness + 0.027),0, 0>, 0.0005
	}
	sphere
	{
		<-(inner_max + land_mass_thickness + 0.027),0, 0>, 0.0005
	}
}

#declare holder_diff_short = union
{
	cylinder
	{
		<0, 0, 0>, <-(inner_max + land_mass_thickness + 0.0245), 0, 0>, 0.0005
	}
	sphere
	{
		<-(inner_max + land_mass_thickness + 0.0245),0, 0>, 0.0005
	}
}

#declare globe_holder_01 = difference
{
	//
	// Base start
	//

	cylinder
	{
		<0, 0, -0.0025>, <0, 0, 0.0025>, inner_max + land_mass_thickness + 0.04
	}

	//
	// remove middle
	//

	cylinder
	{
		<0, 0, -1>, <0, 0, 1>, inner_max + land_mass_thickness + 0.02
	}

	//
	// create ridge on eaither side
	//

	cylinder
	{
		<0, 0, -0.0015>, <0, 0, -1>, inner_max + land_mass_thickness + 0.03//5
	}
	cylinder
	{
		<0, 0, 0.0015>, <0, 0, 1>, inner_max + land_mass_thickness + 0.03//5
	}

	//
	// remove half of it
	//

	box
	{
		<0, -1, -1>, <1, 1, 1>
	}

	#if (do_globe_holder_markings = true)
		#declare my_scale = 0.0085;
		#declare text_extrusion_depth = 0.001;
		#declare text_extrusion_depth = 0.0015 / my_scale;

		#declare thing_angle = 0;

		#while (thing_angle < 90)
			#if (mod (thing_angle,10) = 0)
				object
				{
					holder_diff_long
					rotate <0,0,thing_angle>
					translate <0, 0, -0.0015>
				}
				object
				{
					holder_diff_long
					rotate <0,0,thing_angle>
					translate <0, 0,  0.0015>
				}
				object
				{
					holder_diff_long
					rotate <0,0,-thing_angle>
					translate <0, 0, -0.0015>
				}
				object
				{
					holder_diff_long
					rotate <0,0,-thing_angle>
					translate <0, 0,  0.0015>
				}

				//
				// Now add the number
				//

				#declare _0_base = text
				{
					ttf "Times.ttf",
					str (thing_angle, 0, 0),
					text_extrusion_depth,
					0
				}

				#declare _0_max = max_extent (_0_base);
				#declare _0_min = min_extent (_0_base);

				#declare near_side_0 = object
				{
					_0_base
					translate (-1 * (_0_min + _0_max)) / 2
					scale (my_scale / (abs (_0_min.y) + abs (_0_max.y) ) )
					rotate z*90
				}

				#declare far_side_0 = object
				{
					_0_base
					translate (-1 * (_0_min + _0_max)) / 2
					scale (my_scale / (abs (_0_min.y) + abs (_0_max.y)))
					rotate y*180
					rotate z*90
				}

				object
				{
					near_side_0
					translate <-(inner_max + land_mass_thickness + 0.035), 0, -0.0025>
					rotate <0,0,thing_angle>
				}
				object
				{
					near_side_0
					translate <-(inner_max + land_mass_thickness + 0.035), 0, -0.0025>
					rotate <0,0,-thing_angle>
				}
				object
				{
					far_side_0
					translate <-(inner_max + land_mass_thickness + 0.035), 0, 0.0025>
					rotate <0,0,thing_angle>
				}
				object
				{
					far_side_0
					translate <-(inner_max + land_mass_thickness + 0.035), 0, 0.0025>
					rotate <0,0,-thing_angle>
				}
			#else
				#if (mod (thing_angle,5) = 0)
					object
					{
						holder_diff_mid
						rotate <0,0,thing_angle>
						translate <0, 0, -0.0015>
					}
					object
					{
						holder_diff_mid
						rotate <0,0,thing_angle>
						translate <0, 0,  0.0015>
					}
					object
					{
						holder_diff_mid
						rotate <0,0,-thing_angle>
						translate <0, 0, -0.0015>
					}
					object
					{
						holder_diff_mid
						rotate <0,0,-thing_angle>
						translate <0, 0,  0.0015>
					}
				#else
					object
					{
						holder_diff_short
						rotate <0,0,thing_angle>
						translate <0, 0, -0.0015>
					}
					object
					{
						holder_diff_short
						rotate <0,0,thing_angle>
						translate <0, 0,  0.0015>
					}
					object
					{
						holder_diff_short
						rotate <0,0,-thing_angle>
						translate <0, 0, -0.0015>
					}
					object
					{
						holder_diff_short
						rotate <0,0,-thing_angle>
						translate <0, 0,  0.0015>
					}
				#end
			#end
			#declare thing_angle = thing_angle + 1;
		#end
	#end

	//
	// Add a bounded_by statement. This improves render time by a factor of 10!
	// The bounding shape is the base shaoe of the holder.
	//

	bounded_by
	{
		difference
		{
			cylinder
			{
				<0, 0, -0.0025>, <0, 0, 0.0025>, inner_max + land_mass_thickness + 0.04
			}
			cylinder
			{
				<0, 0, -1>, <0, 0, 1>, inner_max + land_mass_thickness + 0.02
			}
			box
			{
				<0, -1, -1>, <1, 1, 1>
			}
		}
	}
}

//
// the ends of the angle count 
//

#declare globe_holder_02 = difference
{
	cylinder
	{
		<0, 0, -0.0025>, <0, 0, 0.0025>, 0.01
	}
	box
	{
		<-1, -1, -1>, <-0.001, 1, 1>
	}
	scale <3.5, 1, 1>
}

//
// globe_holder_03 and globe_holder_04 are the bits at the top and bottom
// of the globe inbetween the angle counter, the end bit
//

#declare globe_holder_03 = union
{
	cylinder
	{
		<0, 0, 0>, <0, 0.05, 0>, 0.004
	}
	object
	{
		screw_cap
		translate <0, 0.045, 0>
	}
}

#declare globe_holder_04 = object
{
	globe_holder_03
	scale <1, -1, 1>
}

#declare globe_holder = union
{
	object
	{
		globe_holder_01
	}
	object
	{
		globe_holder_02
		translate <0, inner_max + land_mass_thickness + 0.03, 0>
	}
	object
	{
		globe_holder_02
		translate <0, -(inner_max + land_mass_thickness + 0.03), 0>
	}
	object
	{
		globe_holder_03
		translate <0, inner_max + land_mass_thickness, 0>
	}
	object
	{
		globe_holder_04
		translate <0, -(inner_max + land_mass_thickness ), 0>
	}

	texture
	{
		globe_silver_t
	}
	#if (do_photons = true)
		interior
		{
			ior 4
		}
		photons
		{
			target reflection on
		}
	#end
}

// ============================================================================
//
// And now put it all together...
//
// ============================================================================

#declare globe_top = union
{
	object
	{
		globe_earth
	}
	object
	{
		globe_holder
	}

	//
	// Rotate the object so that it is correctly tilted for the earth's
	// axis.
	//

	rotate <0, 0, earth_axis_rotation>
}
		
// ============================================================================
//
// Now create the stand for the globe.
//
// ============================================================================


#declare stand_screw_1 = difference
{
	#declare cap_radius = 0.015;
	#declare cap_height = 0.01;

	union
	{
		cylinder
		{
			<0, 0, 0>, <0, cap_height, 0>, cap_radius
		}
		sphere
		{
			<0, 0, 0>, cap_radius
			scale <1, 0.1, 1>
			translate <0, cap_height, 0>
		}
	}

	cylinder
	{
		<0, -1, 0>, <0, cap_height / 2, 0>, 0.0101
	}

	#declare diff_count = 0;
	#declare diff_angle = 10;
	#declare max_diff_count = 360 / diff_angle;

	#declare diff_radius = 0.001; // 0.5mm

	#while (diff_count < max_diff_count)
		cylinder
		{
			<0, -1, 0>, <0, 1, 0>, diff_radius
			scale <.5, 1, 1>
			translate <cap_radius, 0, 0>
			rotate y*diff_count*diff_angle
		}
		#declare diff_count = diff_count + 1;
	#end
}

#declare stand_screw_2 = difference
{
	#declare cap_radius = 0.015;
	#declare cap_height = 0.01;

	union
	{
		cylinder
		{
			<0, 0, 0>, <0, cap_height, 0>, cap_radius
		}
		sphere
		{
			<0, 0, 0>, cap_radius
			scale <1, 0.1, 1>
		}
		sphere
		{
			<0, 0, 0>, cap_radius
			scale <1, 0.1, 1>
			translate <0, cap_height, 0>
		}
	}

	cylinder
	{
		<0, -1, 0>, <0, 1, 0>, 0.0101
	}

	#declare diff_count = 0;
	#declare diff_angle = 10;
	#declare max_diff_count = 360 / diff_angle;

	#declare diff_radius = 0.001; // 0.5mm

	#while (diff_count < max_diff_count)
		cylinder
		{
			<0, -1, 0>, <0, 1, 0>, diff_radius
			scale <.5, 1, 1>
			translate <cap_radius, 0, 0>
			rotate y*diff_count*diff_angle
		}
		#declare diff_count = diff_count + 1;
	#end
}

//
// globe_stand_01 is the main support
//

#declare globe_stand_01 = union
{
	cylinder
	{
		<0, 0.02, 0>, <0, 0.11, 0>, 0.01
	}
	object
	{
		stand_screw_1
		translate <0, -(cap_height / 2), 0> // to centre on y=0
		translate <0, 0.11 + 0.002, 0> // at top
	}
	object
	{
		stand_screw_2
		translate <0, -(cap_height / 2), 0> // to centre on y=0
		translate <0, 0.11, 0> // at top
		translate <0, -((4*0.0025)+0.02+ 0.002), 0>
	}
}

#declare globe_stand_02 = union
{
	#declare nominal_base_height = 0.05; // 5cm
	#declare centre_radius = 0.05;
	#declare outer_radius = 0.175;
	#declare bottom_lip_height = 0.005;

	difference
	{
		cylinder
		{
			<0, 0, 0>, <0, nominal_base_height, 0>, outer_radius
		}
		torus
		{
			outer_radius, outer_radius - centre_radius
			sturm
			scale <1, (nominal_base_height - bottom_lip_height) / (outer_radius - centre_radius), 1>
			translate <0, nominal_base_height, 0>
		}
	}
	sphere
	{
		<0, 0, 0>, centre_radius
		scale <1, .25, 1>
		translate <0, nominal_base_height, 0>
	}
	torus
	{
		outer_radius, bottom_lip_height / 2
		sturm
		translate <0, bottom_lip_height / 2, 0>
	}
}

#declare globe_stand = union
{
	object
	{
		globe_stand_01
	}
	object
	{
		globe_stand_02
	}

	texture
	{
		globe_silver_t
	}
	#if (do_photons = true)
		interior
		{
			ior 4
		}
		photons
		{
			target reflection on
		}
	#end
}

// ============================================================================
//
// Finally put everything together...
//
// ============================================================================
#declare globe = union
{
	object
	{
		globe_stand
		translate <0, -(inner_max + land_mass_thickness + 0.11 +0.015), 0>
	}
	object
	{
		globe_top
	}
	// move to rest on y=0
	translate <0, (inner_max + land_mass_thickness + 0.11 +0.015), 0>
}

