// ============================================================================
//
// arms.inc - Copyright 2001 Ian Shumsky / OuterArm
//
// ============================================================================
//
// This file contains both the definitions for the various arm meshes as well
// as a placement and choice macro (put_arm). There are a total of 24
// different arms created from the following variables:
//
//   Sex     : male/female
//   Handed  : left/right
//   Pose    : fist/point/pointer/pointer2/reach/reaching
//   Texture : t1
//
// Combinations of these values are used to specify a mesh. For example:
//
//    male_right_fist_t1
//
// specifies a male arm, right handed with a fist pose. The arm is finished in
// texture map 1.
//
// The arms are defined in alhpabetical order.
//
// ============================================================================

#version unofficial MegaPov 0.6;

// ============================================================================
//
// Define each of the possible arms and orient them so their base is at
// <0, 0, 0> and they point in +y.
//
// ============================================================================

#declare female_left_fist_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_left_fist_mesh.inc"

		translate <-0.061096, 0, 0>
		translate <0, -0.540596, 0>
		translate <0, -0.0223105, 0>
		translate <0, 0, 0.007868>
		translate <0, 0, -0.030532>
		rotate z*90
		rotate y*-90
	}
}

#declare female_left_point_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_left_point_mesh.inc"

		translate <-0.061096, 0, 0>
		translate <0, -0.540596, 0>
		translate <0, -0.0223105, 0>
		translate <0, 0, 0.007868>
		translate <0, 0, -0.030532>
		rotate z*90
		rotate y*-90
	}
}

#declare female_left_pointer_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_left_pointer_mesh.inc"

		translate <-0.061096, 0, 0>
		translate <0, -0.540596, 0>
		translate <0, -0.0223105, 0>
		translate <0, 0, 0.007868>
		translate <0, 0, -0.030532>
		rotate z*90
		rotate y*-90
	}
}

#declare female_left_pointer2_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_left_pointer2_mesh.inc"

		translate <-0.061096, 0, 0>
		translate <0, -0.540596, 0>
		translate <0, -0.0223105, 0>
		translate <0, 0, 0.007868>
		translate <0, 0, -0.030532>
		rotate z*90
		rotate y*-90
	}
}

#declare female_left_reach_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_left_reach_mesh.inc"

		translate <-0.061096, 0, 0>
		translate <0, -0.540596, 0>
		translate <0, -0.0223105, 0>
		translate <0, 0, 0.007868>
		translate <0, 0, -0.030532>
		rotate z*90
		rotate y*-90
	}
}

#declare female_left_reaching_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_left_reaching_mesh.inc"

		translate <-0.061096, 0, 0>
		translate <0, -0.540596, 0>
		translate <0, -0.0223105, 0>
		translate <0, 0, 0.007868>
		translate <0, 0, -0.030532>
		rotate z*90
		rotate y*-90
	}
}

#declare female_right_fist_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_right_fist_mesh.inc"

		translate <0.061341, 0, 0>
		translate <0, -0.542317, 0>
		translate <0, -0.0216215, 0>
		translate <0, 0, 0.007567>
		translate <0, 0, -0.031176>
		rotate z*-90
		rotate y*90
	}
}

#declare female_right_point_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_right_point_mesh.inc"

		translate <0.061341, 0, 0>
		translate <0, -0.542317, 0>
		translate <0, -0.0216215, 0>
		translate <0, 0, 0.007567>
		translate <0, 0, -0.031176>
		rotate z*-90
		rotate y*90
	}
}

#declare female_right_pointer_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_right_pointer_mesh.inc"

		translate <0.061341, 0, 0>
		translate <0, -0.542317, 0>
		translate <0, -0.0216215, 0>
		translate <0, 0, 0.007567>
		translate <0, 0, -0.031176>
		rotate z*-90
		rotate y*90
	}
}

#declare female_right_pointer2_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_right_pointer2_mesh.inc"

		translate <0.061341, 0, 0>
		translate <0, -0.542317, 0>
		translate <0, -0.0216215, 0>
		translate <0, 0, 0.007567>
		translate <0, 0, -0.031176>
		rotate z*-90
		rotate y*90
	}
}

#declare female_right_reach_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_right_reach_mesh.inc"

		translate <0.061341, 0, 0>
		translate <0, -0.542317, 0>
		translate <0, -0.0216215, 0>
		translate <0, 0, 0.007567>
		translate <0, 0, -0.031176>
		rotate z*-90
		rotate y*90
	}
}

#declare female_right_reaching_t1 = object
{
	#include "female_texture1.inc"

	union
	{
		#include "female_right_reaching_mesh.inc"

		translate <0.061341, 0, 0>
		translate <0, -0.542317, 0>
		translate <0, -0.0216215, 0>
		translate <0, 0, 0.007567>
		translate <0, 0, -0.031176>
		rotate z*-90
		rotate y*90
	}
}

#declare male_left_fist_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_left_fist_mesh.inc"

		translate <-0.062183, 0, 0>
		translate <0, -0.576523, 0>
		translate <0, -0.0257135, 0>
		translate <0, 0, 0.003051>
		translate <0, 0, -0.0383455>
		rotate z*90
		rotate y*-90
	}
}

#declare male_left_point_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_left_point_mesh.inc"

		translate <-0.062183, 0, 0>
		translate <0, -0.576523, 0>
		translate <0, -0.0257135, 0>
		translate <0, 0, 0.003051>
		translate <0, 0, -0.0383455>
		rotate z*90
		rotate y*-90
	}
}

#declare male_left_pointer_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_left_pointer_mesh.inc"

		translate <-0.062183, 0, 0>
		translate <0, -0.576523, 0>
		translate <0, -0.0257135, 0>
		translate <0, 0, 0.003051>
		translate <0, 0, -0.0383455>
		rotate z*90
		rotate y*-90
	}
}

#declare male_left_pointer2_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_left_pointer2_mesh.inc"

		translate <-0.062183, 0, 0>
		translate <0, -0.576523, 0>
		translate <0, -0.0257135, 0>
		translate <0, 0, 0.003051>
		translate <0, 0, -0.0383455>
		rotate z*90
		rotate y*-90
	}
}

#declare male_left_reach_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_left_reach_mesh.inc"

		translate <-0.062183, 0, 0>
		translate <0, -0.576523, 0>
		translate <0, -0.0257135, 0>
		translate <0, 0, 0.003051>
		translate <0, 0, -0.0383455>
		rotate z*90
		rotate y*-90
	}
}

#declare male_left_reaching_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_left_reaching_mesh.inc"

		translate <-0.062183, 0, 0>
		translate <0, -0.576523, 0>
		translate <0, -0.0257135, 0>
		translate <0, 0, 0.003051>
		translate <0, 0, -0.0383455>
		rotate z*90
		rotate y*-90
	}
}

#declare male_right_fist_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_right_fist_mesh.inc"

		translate <0.058985, 0, 0>
		translate <0, -0.577406, 0>
		translate <0, -0.0254495, 0>
		translate <0, 0, 0.002669>
		translate <0, 0, -0.03892485>
		rotate z*-90
		rotate y*90
	}
}

#declare male_right_point_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_right_point_mesh.inc"

		translate <0.058985, 0, 0>
		translate <0, -0.577406, 0>
		translate <0, -0.0254495, 0>
		translate <0, 0, 0.002669>
		translate <0, 0, -0.03892485>
		rotate z*-90
		rotate y*90
	}
}

#declare male_right_pointer_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_right_pointer_mesh.inc"

		translate <0.058985, 0, 0>
		translate <0, -0.577406, 0>
		translate <0, -0.0254495, 0>
		translate <0, 0, 0.002669>
		translate <0, 0, -0.03892485>
		rotate z*-90
		rotate y*90
	}
}

#declare male_right_pointer2_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_right_pointer2_mesh.inc"

		translate <0.058985, 0, 0>
		translate <0, -0.577406, 0>
		translate <0, -0.0254495, 0>
		translate <0, 0, 0.002669>
		translate <0, 0, -0.03892485>
		rotate z*-90
		rotate y*90
	}
}

#declare male_right_reach_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_right_reach_mesh.inc"

		translate <0.058985, 0, 0>
		translate <0, -0.577406, 0>
		translate <0, -0.0254495, 0>
		translate <0, 0, 0.002669>
		translate <0, 0, -0.03892485>
		rotate z*-90
		rotate y*90
	}
}

#declare male_right_reaching_t1 = object
{
	#include "male_texture1.inc"

	union
	{
		#include "male_right_reaching_mesh.inc"

		translate <0.058985, 0, 0>
		translate <0, -0.577406, 0>
		translate <0, -0.0254495, 0>
		translate <0, 0, 0.002669>
		translate <0, 0, -0.03892485>
		rotate z*-90
		rotate y*90
	}
}



// ============================================================================
//
// Now need a series of macros to 'pick' an arm, orient it to point at a
// specified point and add the arm to the scene. To achieve this we will use
// three macros:
//
//   choose_arm
//   get_arm_mesh
//   orient_arm
//
// We also need to add a series of arrays which define the possible arm
// combinations and some macros to allow us to seed the random values.
//
// ============================================================================

//
// Create an array of the available sexes.
//

#declare sex_options = array[2]
{
	"female",
	"male"
}

//
// Create an array of the available hands.
//

#declare handed_options = array[2]
{
	"left",
	"right"
}

//
// Create an array of the availiable poses.
//

#declare pose_options = array[6]
{
	"fist",
	"point",
	"pointer",
	"pointer2",
	"reach",
	"reaching"
}

//
// Finally, declare an array of the availiable textures.
//

#declare texture_options = array[1]
{
	"t1"
}

//
// Declare the various random number generators we need and macros to re-seed
// them.
//

#declare arm_random = seed (0);

#macro set_arm_random_seed (new_seed)
	#declare arm_random = seed (new_seed);
#end



// ============================================================================
//
// The coose_arm macro 'picks' one of the availiable arm meshes at random. We
// hava a level of control over the choice by a series of threshold values we
// pass in. The three parameters hold (in the following order) the male/female
// threshold, the left hand/right hand split and the choices of the various
// poses.
//
// ============================================================================

#macro choose_arm (fm_thresh, lr_thresh, p_thresh)
	//
	// First choose a sex
	//

	#declare arm_rand = rand (arm_random);

	#if (arm_rand < fm_thresh)
		// female
		#declare sex_type = 0;
	#else
		// male
		#declare sex_type = 1;
	#end

	//
	// Choose a left or right arm.
	//

	#declare arm_rand = rand (arm_random);

	#if (arm_rand < lr_thresh)
		// left
		#declare handed_type = 0;
	#else
		// right
		#declare handed_type = 1;
	#end

	//
	// Choose a pose.
	//

	#declare arm_rand = rand (arm_random);

	#if (arm_rand < p_thresh[0])
		// fist
		#declare pose_type = 0;
	#else
		#if (arm_rand < p_thresh[1])
			// point
			#declare pose_type = 1;
		#else
			#if (arm_rand < p_thresh[2])
				// pointer
				#declare pose_type = 2;
			#else
				#if (arm_rand < p_thresh[3])
					// pointer2
					#declare pose_type = 3;
				#else
					#if (arm_rand < p_thresh[4])
						// reach
						#declare pose_type = 4;
					#else
						// reaching
						#declare pose_type = 5;
					#end
				#end
			#end
		#end
	#end

	//
	// Choose a texture. Currently we only have a single texture to choose
	// from.
	//

	#declare texture_type = 0;

	//
	// Tie all the choices together into a single text description.
	//

	#declare arm_choice = concat (sex_options[sex_type], "_", handed_options[handed_type], "_", pose_options[pose_type], "_", texture_options[texture_type])

//	#debug concat (arm_choice, "\n")
#end



// ============================================================================
//
// The get_arm_orientation macro calculates the x and y rotations required to
// orient a vertical line from an arbitary location to pass through any other
// arbitary location.
//
// In this case, these rotations will be used to orient the arm mesh.
//
// This macro does not completely do the job. If the y value of the origin is
// larger (i.e. above) the look at point, then something unexpected will
// happen. Maybe one day I'll get round to fixing this!
//
// ============================================================================

#macro get_arm_orientation (arm_location, arm_point_at)
	//
	// Get a 'sanitised' look at point. This allows us to simplify the work
	// as our origin is treated as <0, 0, 0>.
	//

	#declare normal_point_at = arm_point_at - arm_location;

	#declare norm_x = abs (normal_point_at.x);
	#declare norm_y = abs (normal_point_at.y);
	#declare norm_z = abs (normal_point_at.z);

	//
	// Calculate the x rotation (or elevation) of the arm. Note, we are
	// starting from vertical and we will never be going further than 90.
	//

	#declare arm_x_rot = 90 - degrees (asin (norm_y / vlength (arm_point_at - arm_location)));

	//
	// Calculate the y rotation.
	//

	#if (normal_point_at.x < 0)
		#if (normal_point_at.z < 0)
			#if (norm_z = 0)
				#declare norm_z = 0.0000001;
			#end

			#declare arm_y_rot = 180 + degrees (atan2 (norm_x / norm_z, 1));
		#else
			#if (norm_x = 0)
				#declare norm_x = 0.0000001;
			#end

			#declare arm_y_rot = 270 + degrees (atan2 (norm_z / norm_x, 1));
		#end
	#else
		#if (normal_point_at.z < 0)
			#if (norm_x = 0)
				#declare norm_x = 0.0000001;
			#end

			#declare arm_y_rot = 90 + degrees (atan2 (norm_z / norm_x, 1));
		#else
			#if (norm_z = 0)
				#declare norm_z = 0.0000001;
			#end

			#declare arm_y_rot = degrees (atan2 (norm_x / norm_z, 1));
		#end
	#end

//	#debug concat ("Arm x rotation: ", str (arm_x_rot, 3, 3), "\n")
//	#debug concat ("Arm y rotation: ", str (arm_y_rot, 3, 3), "\n")
#end



// ============================================================================
//
// This macro gets an appropriate object based on a text request. I did it this
// way as I don't think that you can go  object{text_string}  to get an object.
//
// ============================================================================

#macro get_arm_mesh (arm_choice)
	#if (strcmp (arm_choice, "female_left_fist_t1") = 0)
		object
		{
			female_left_fist_t1
		}
	#end

	#if (strcmp (arm_choice, "female_left_point_t1") = 0)
		object
		{
			female_left_point_t1
		}
	#end

	#if (strcmp (arm_choice, "female_left_pointer_t1") = 0)
		object
		{
			female_left_pointer_t1
		}
	#end

	#if (strcmp (arm_choice, "female_left_pointer2_t1") = 0)
		object
		{
			female_left_pointer2_t1
		}
	#end

	#if (strcmp (arm_choice, "female_left_reach_t1") = 0)
		object
		{
			female_left_reach_t1
		}
	#end

	#if (strcmp (arm_choice, "female_left_reaching_t1") = 0)
		object
		{
			female_left_reaching_t1
		}
	#end

	#if (strcmp (arm_choice, "female_right_fist_t1") = 0)
		object
		{
			female_right_fist_t1
		}
	#end

	#if (strcmp (arm_choice, "female_right_point_t1") = 0)
		object
		{
			female_right_point_t1
		}
	#end

	#if (strcmp (arm_choice, "female_right_pointer_t1") = 0)
		object
		{
			female_right_pointer_t1
		}
	#end

	#if (strcmp (arm_choice, "female_right_pointer2_t1") = 0)
		object
		{
			female_right_pointer2_t1
		}
	#end

	#if (strcmp (arm_choice, "female_right_reach_t1") = 0)
		object
		{
			female_right_reach_t1
		}
	#end

	#if (strcmp (arm_choice, "female_right_reaching_t1") = 0)
		object
		{
			female_right_reaching_t1
		}
	#end

	#if (strcmp (arm_choice, "male_left_fist_t1") = 0)
		object
		{
			male_left_fist_t1
		}
	#end

	#if (strcmp (arm_choice, "male_left_point_t1") = 0)
		object
		{
			male_left_point_t1
		}
	#end

	#if (strcmp (arm_choice, "male_left_pointer_t1") = 0)
		object
		{
			male_left_pointer_t1
		}
	#end

	#if (strcmp (arm_choice, "male_left_pointer2_t1") = 0)
		object
		{
			male_left_pointer2_t1
		}
	#end

	#if (strcmp (arm_choice, "male_left_reach_t1") = 0)
		object
		{
			male_left_reach_t1
		}
	#end

	#if (strcmp (arm_choice, "male_left_reaching_t1") = 0)
		object
		{
			male_left_reaching_t1
		}
	#end

	#if (strcmp (arm_choice, "male_right_fist_t1") = 0)
		object
		{
			male_right_fist_t1
		}
	#end

	#if (strcmp (arm_choice, "male_right_point_t1") = 0)
		object
		{
			male_right_point_t1
		}
	#end

	#if (strcmp (arm_choice, "male_right_pointer_t1") = 0)
		object
		{
			male_right_pointer_t1
		}
	#end

	#if (strcmp (arm_choice, "male_right_pointer2_t1") = 0)
		object
		{
			male_right_pointer2_t1
		}
	#end

	#if (strcmp (arm_choice, "male_right_reach_t1") = 0)
		object
		{
			male_right_reach_t1
		}
	#end

	#if (strcmp (arm_choice, "male_right_reaching_t1") = 0)
		object
		{
			male_right_reaching_t1
		}
	#end
#end



// ============================================================================
//
// This is the main 'grunt' macro. It first chooses an arm and then base on the
// look at point and the location, calculates the rotation for the arm. It then
// places the arm, rotates it and moves it to its final location.
//
// ============================================================================

#macro put_arm (fm_thresh, lr_thresh, p_thresh, arm_location, arm_point_at)
	choose_arm (fm_thresh, lr_thresh, p_thresh)
	get_arm_orientation (arm_location, arm_point_at)

	object
	{
		get_arm_mesh (arm_choice)
		rotate x*arm_x_rot
		rotate y*arm_y_rot
		translate arm_location
	}
#end

