//****************************************************************************
//
// Author: Ali Lakhia
// File:   uwalls.inc
// Date:   January 18th, 2001
//
//***************************************************************************/

// Variable to determine offset of roof along z-axis
#declare Upper_roof_offset = 0.25;
#declare Upper_lights_color = rgbft <236/255, 141/255, 21/255, 0, 200/255>;
#declare Upper_wall_height = 2.36;

// Ceiling for upper section
#declare Upper_ceiling = box
{
  <0, 0, 0>,
  <Wall_length, 0.04, Wall_depth + Upper_roof_offset>
}

// Half-arch for upper walls
#declare Upper_arch_half = prism
{
  quadratic_spline
  0, 0.16, 11
  // Control point followed by 2 base points
  <-1, 0>, <0, 0>, <0.1, 0>,
  // Create the arch
  <0.1, 0.31>, <0.188, 0.488>, <0.39, 0.65>, <0.652, 0.78>, <0.7, 0.8>,
  // Go back down to close prism
  <0.7, 1>, <0, 1>, <0, 0>
}

// Create full arch for upper walls
#declare Upper_arch = union
{
  object {
    Upper_arch_half
    rotate -90*x
    translate <0, 0, 0.08>
  }
  object {
    Upper_arch_half
    rotate -90*x + 180*y
    translate <1.4, 0, -0.08>
  }
}

// Pillar connector to arch
#declare Upper_pillar_connector = prism
{
  linear_spline
  0, 0.16, 4,
  <0, 0>, <0.1, 0.16>, <0, 0.16>, <0, 0>
}

// Pillar with connectors and arch
#declare Upper_pillar = union
{
  // Actual pillar
  box {
    <-0.08, 0, -0.08>,
    <0.08, Upper_wall_height, 0.08>
    texture { light_brown_concrete }
  }
  // Right connector
  object {
    Upper_pillar_connector
    rotate -90*x
    translate <0.08, 1.2, 0.08>
  }
  // Left connector
  object {
    Upper_pillar_connector
    rotate -90*x + 180*y
    translate <Wall_length - 0.08, 1.2, -0.08>
  }
  // Top connector to roof
  object {
    Upper_pillar_connector
    rotate -90*x + 90*y
    scale <1, 6.75, 1>                 // (0.16 + 0.9)/0.16 = 6.625
    translate <0.08, Upper_wall_height - 0.16*6.6, -0.08>
  }
  // Top flat connector to roof
  box {
    <-0.15, Upper_wall_height - 0.09, -0.25>,
    <0.15, Upper_wall_height, 0>
  }
}

// Upper section, includes arch, pillar and ceiling
#macro Upper_section (Lights_color)
  union {
    object {
      Upper_pillar
    }
    object {
      Upper_arch
      translate <0.08, 1.36, 0>
    }
    object {
      Upper_ceiling
      translate <-0.08, Upper_wall_height, -0.08 - Upper_roof_offset>
    }
    #if (Lights_on > 0)
      #if (Extra_lights_on)
        light_source {
          <Wall_length/2, Upper_wall_height, 0>
          color Lights_color
          spotlight
          point_at <0, -2, 0>
          fade_distance 0.6
          fade_power 1
          radius ros
          falloff fos
          tightness 10
          media_attenuation on
        }
      #end
      difference
      {
        Create_halo(Lights_color, 1, 1)
        box {
          <-.75, 0.02, -.75>,
          <.75, -.75, .75>
          pigment { Clear }
        }
        no_shadow
        scale <.6, 2.2, .6>
        translate <Wall_length/2, 0, 0>
      }
    #end
    texture { very_light_brown_concrete }
    normal {
      brick -0.05
      brick_size 0.35
      mortar 0.02
    }
  }
#end

// Macro for generating sections along positive x-axis.
#macro Upper_wall (NumberOfSections, Draw_back_wall, Lights_color)
  #local sections = NumberOfSections;
  #while (sections > 0)
    #local sections=sections-1;
    object {
      Upper_section(Lights_color)
      translate sections*Wall_length*x
    }
    #if (Draw_back_wall)
      box {
        <0.0, 0, Wall_depth>,
        <Wall_length, Upper_wall_height, Wall_depth + 0.01>
        translate sections*Wall_length*x
        texture { very_light_brown_concrete }
      }
    #end
  #end
#end

// Macro for creating a wall that is generated along X & Z axis.
#macro Create_upper_wall(Number_x, Number_z, Lights_color)
  #local backwall = 1;
  #local num_x = Number_x;
  #local num_z = Number_z;
  union
  {
    #while (num_z > 0)
      #local num_z = num_z-1;
      union {
        Upper_wall(num_x, backwall, Lights_color)
        translate num_z*Wall_depth*z
      }
      #if (backwall = 1)
        #local backwall = 0;
      #end
    #end
  }
#end

// Redefine macro to creating bounding box for testing
#if (!Draw_walls_upper)
  #macro Create_upper_wall(Number_x, Number_z, Lights_color)
    box {
      0,
      <Number_x*Wall_length, Upper_wall_height, Number_z*Wall_depth>
    }
  #end
#end

// Create both lower and upper sections of the wall
// "lwalls.inc" should be included before calling this
#macro Create_wall(Number_x, Number_z)
  union {
    // Draw front lower wall
    difference {
      Create_lower_wall(Number_x, Number_z)
      box {
        <(Number_x*Wall_length) - 1.6201, -0.0001, -0.0401>,
        <(Number_x*Wall_length) + 0.0001, 1.2401, 0.2001>
      }
      box {
        <(Number_x*Wall_length) - 1.0001, Lower_wall_height - 0.1601, 0.16>,
        <(Number_x*Wall_length) - 1.6701, Lower_wall_height + 0.7601, Wall_depth>
        texture { offwhite_concrete }
      }
      box {
        <1, Lower_wall_height - 0.1601, 0.16>,
        <1.4601, Lower_wall_height + 0.7601, Wall_depth>
        texture { offwhite_concrete }
      }
    }

    // Draw back lower walls unless disabled
    #if (Draw_walls_lower_back)
      object {
        Create_upper_wall(Number_x, Number_z, Lower_lights_color)
        scale <1, Wall_depth, 1>
        translate <0, 0, Wall_depth*Number_z>
      }
    #end

    // Draw upper walls
    object {
      Create_upper_wall(Number_x, Number_z, Upper_lights_color)
      translate <0, Upper_wall_height*1.4, Wall_depth*Number_z>
    }
  }
#end

// Macro to clip on left. Number_x is placeholder and not used.
#macro Clip_left(Number_x, Number_z, Cut_angle)
plane {
  -x, 0
  rotate -Cut_angle*y
  translate <0, 0, Number_z*Wall_depth*2>
}
#end

// Macro to clip on right.
#macro Clip_right(Number_x, Number_z, Cut_angle)
plane {
  x, 0
  rotate Cut_angle*y
  translate <Number_x*Wall_length, 0, Number_z*Wall_depth*2>
}
#end

// Join two symmetric walls at a specified angle
#macro Join_sym_walls(Number_x, Number_z, Cut_angle)
  object {
    Create_wall(Number_x, Number_z)

    // Cut with plane at specified angle
    clipped_by {
      Clip_right(Number_x, Number_z, Cut_angle)
    }
  }
  
  object {
    Create_wall(Number_x, Number_z)

    // Cut with plane at specified angle
    clipped_by {
      Clip_left(Number_x, Number_z, Cut_angle)
    }

    // Move right side to its position
    Position_wall(Number_x, Number_z, Cut_angle)
  }
#end

// Join two walls at a specified angle
#macro Join_two_walls(Number_x, Number_z, Cut_angle)
  object {
    Create_wall(Number_x, Number_z)
    
    // Cut with plane at specified angle
    clipped_by {
      Clip_right(Number_x,  Number_z, Cut_angle)
    }
  }
#end

// Macro to position wall section in alignment with another
// wall section that has dimensions specified.
#macro Position_wall(Number_x, Number_z, Cut_angle)
   #local theta = radians(Cut_angle);
   #local _X = Wall_length*Number_x;
   #local _Z = 2*Wall_depth*Number_z;
   #local _diff = _Z*tan(theta);
   #local new_x = _X - _diff;
   translate -_diff*x                  // Move front cut corner to origin
   rotate 2*Cut_angle*y                // Now rotate so corner stays at origin
   translate new_x*x                   // Bring front corners together
#end