/*
 *	House.pov
 *
 *	POV-Ray modelling of house designed in 10th grade CAD
 *	Copyright 2001 Mark Wagner
 */

#ifndef(included)
	#declare included = true;
	#include "colors.inc"
	#include "textures.inc"
	#include "woods.inc"
	
	#declare cm = 1/2.54;	/* 1 POV unit = 1 inch */
	#include "units.inc"	/* www.geocities.com/Rengaw03/Pov-Ray.html */
	#undef included
	global_settings{assumed_gamma 1}
#end

/************************************************************
	Textures
************************************************************/

#declare TrimTexture = texture{
	pigment{
		SummerSky
	}
	finish{
		ambient .1
		diffuse .8
	}
}

#declare WallTexture = texture{
	pigment{rgb <1,1,1>}
	finish{
		ambient .1
		diffuse .95
	}
}

#declare WindowGlass1=material {
	texture {
		pigment {LightBlue filter .99} 
		finish {reflection .2}
	}
	interior {ior 1.1}
}

#declare Concrete = texture{
	pigment{rgb .5}
}

#declare Shingles = texture{
	brick
	texture{pigment{rgb 0}},
	texture{
		pigment{VeryDarkBrown}
		normal{crackle scale 1*inch}
		finish{
			ambient .01
			diffuse .7
		}
	}
	brick_size <3*feet,15*inch,6*inch>
	mortar .5*inch
	rotate x*90
}


/************************************************************
	Wall parts: Doors, windows, corners, wall sections
	All wall sections have the outside in the -z direction
************************************************************/
#include "windows.inc" // Fabien Mosen's window macro, available on news://news.povray.org/

#macro HouseWindow(_width)
	union{
		box{<0,6*feet+10*inch,0>,<_width,8*feet,6*inch>}
		box{<0,0,0>,<_width,3*feet,6*inch>}
		
		/* DUMMY WINDOWPANE */
		object{Dbl_Open_Window_1_1 (_width,3*feet+10*inch,  2*inch,2*inch,1*inch,6*inch,1*inch,.25*inch,.5*inch,  2,3,
			TrimTexture,WindowGlass1, 0,0,sphere {<0,0,0>,0})
			rotate y*90
			translate y*3*feet
		}
		box{<0,3*feet,2>,<_width,6*feet+10*inch,4*inch> pigment{rgbt <1,1,1,.9>}}
	}
#end

#macro HouseWall(_width)
	union{
		box{<0,0,0>,<_width,8*feet,6*inch>}
	}
#end

#macro HouseCorner()
	union{
		box{<-.5,0,-.5>,<4*inch,8*feet,.5*inch>}
		box{<-.5,0,.5*inch>,<.5*inch,8*feet,3.5*inch>}
	}
#end

#macro HouseDoor(_width,_pct_open)
	union{
		box{<0,6*feet+10*inch,0>,<_width,8*feet,6*inch>}
		
		/* DUMMY DOOR */
		box{<0,0,2*inch>,<_width,6*feet+10*inch, 2*inch> pigment{Brown}}
	}
#end

#macro GarageWall(_width)
	union{
		box{<0,-.25,0>*feet,<_width,8*feet,6*inch>}
	}
#end

#macro GarageDoor(_width,_pct_open)
	union{
		box{<0,6*feet+10*inch,0>,<_width,8*feet,6*inch>}
		
		/* DUMMY DOOR */
		box{<0,0,2*inch>,<_width,6*feet+13*inch,2*inch> scale <1,-((100-_pct_open)/100),1> translate y*(6*feet+10*inch) pigment{rgb .7}}
	}
#end

/* Interior walls are centered on the x-axis and extend 2 inches beyond the specified ends */
#macro InsideWall(_width)
	union{
		box{<-2*inch,0,-2*inch>,<_width+2*inch,8*feet,2*inch>}
	}
#end

/************************************************************
	House definition
************************************************************/

#local Foundation = union{
	box{<0,0,0>*feet,  <23,-2,36>*feet}
	box{<23,0,-4>*feet,<38,-2,36>*feet}
	box{<38,0,18>*feet,<60,-2,36>*feet}
	box{<38,0,-4>*feet,<41,-2,0 >*feet}
	/* Garage floor */
	box{<38,-.25, 0>*feet,<41,-2,8>*feet}
	box{<41,-.25,-4>*feet,<63,-2,18>*feet}
	texture{Concrete}
}

#local ExteriorWalls = union{
	object{HouseCorner() texture{TrimTexture}}
	/* Front walls */
	object{HouseWall(5.5*feet)}
	object{HouseWindow(3*feet) translate x*5.5*feet}
	object{HouseWall(6*feet) translate x*8.5*feet}
	object{HouseWindow(3*feet) translate x*14.5*feet}
	object{HouseWall(5.5*feet) translate x*17.5*feet}
	object{HouseWall(4.5*feet) rotate y*90 translate x*23*feet+z*6*inch}
	union{
		object{HouseCorner() texture{TrimTexture}}
		object{HouseWall(1*feet+8*inch)}
		object{HouseWindow(6*feet)         translate x*(1*feet+8*inch)}
		object{HouseWall(2*feet+11*inch)   translate x*(7*feet+8*inch)}
		object{HouseDoor(2*feet+10*inch,0) translate x*(10*feet+7*inch)}
		object{HouseWall(1*feet+7*inch)    translate x*(13*feet+5*inch)}
		object{HouseWall(3*feet+0*inch)    translate x*(15*feet+0*inch)}

		object{GarageWall(1*feet+6*inch)   translate x*(18*feet+0*inch)}		
		object{GarageDoor(8*feet,0)        translate x*(19*feet+6*inch)}		
		object{GarageWall(3*feet+0*inch)   translate x*(27*feet+6*inch)}		
		object{GarageDoor(8*feet,0)        translate x*(30*feet+6*inch)}		
		object{GarageWall(1*feet+6*inch)   translate x*(38*feet+6*inch)}		
		translate x*23*feet+z*-4*feet
	}
	
	/* Left wall */
	union{
		object{HouseWall(9*feet+8*inch)}
		object{HouseWindow(3*feet) translate x*(9*feet+8*inch)}
		object{HouseWall(10*feet+6*inch) translate x*(12*feet+8*inch)}
		object{HouseWindow(3*feet) translate x*(23*feet+2*inch)}
		object{HouseWall(9*feet+10*inch) translate x*(26*feet+2*inch)}
		
		scale <1,1,-1>
		rotate y*-90
	}		
	
	/* Back walls */
	union{
		object{HouseCorner() texture{TrimTexture}}
		object{HouseWall(16*feet+2*inch)}
		object{HouseWindow(6*feet)         translate x*(16*feet+2*inch)}
		object{HouseWall(1*foot+5*inch)    translate x*(22*feet+2*inch)}
		object{HouseDoor(2*feet+10*inch,0) translate x*(23*feet+7*inch)}
		object{HouseWall(1*foot+7*inch)    translate x*(26*feet+5*inch)}
		object{HouseWindow(3*feet)         translate x*(28*feet)}
		object{HouseWall(2*feet)           translate x*(31*feet)}
		object{HouseWall(1*feet+2*inch)    translate x*(33*feet)}
		object{HouseWindow(6*feet)         translate x*(34*feet+2*inch)}
		object{HouseWall(3*feet+6*inch)    translate x*(40*feet+2*inch)}
		object{HouseDoor(6*feet,0)         translate x*(43*feet+8*inch)}
		object{HouseWall(2*feet+8*inch)    translate x*(49*feet+8*inch)}
		object{HouseWindow(6*feet)         translate x*(52*feet+4*inch)}
		object{HouseWall(1*feet+8*inch)    translate x*(58*feet+4*inch)}
		object{HouseCorner() scale <-1,1,1> translate x*60*feet texture{TrimTexture}}

		scale <1,1,-1>
		translate z*36*feet
	}
	
	/* Right wall */
	union{
		object{GarageWall(22*feet) rotate y*-90 translate x*63*feet+z*-4*feet}
		object{GarageWall(3.5*feet) scale <1,1,-1> translate x*59.5*feet+z*18*feet}
	}
	union{
		object{HouseWall(2*feet+6*inch)}
		object{HouseWindow(3*feet) translate x*(2*feet+6*inch)}
		object{HouseWall(5*feet+10*inch) translate x*(5*feet+6*inch)}
		object{HouseWindow(3*feet) translate x*(11*feet+4*inch)}
		object{HouseWall(3*feet+8*inch) translate x*(14*feet+4*inch)}

		rotate y*-90
		translate x*60*feet+z*18*feet
	}
	
	texture{WallTexture}
}

#local InteriorWalls = union{
	object{InsideWall(6.5*feet) rotate y*-90 translate x*13*feet+z*6*inch}
	object{InsideWall(3*feet) translate x*10*feet+z*7*feet}
	object{InsideWall(7*feet) rotate y*-90 translate x*10*feet+z*7*feet}
	object{InsideWall(3*feet) translate x*10*feet+z*14*feet}
	object{InsideWall(6.5*feet) translate x*.5*feet+z*14*feet}
	object{InsideWall(4*feet) rotate y*-90 translate x*4*feet+z*14*feet}
	object{InsideWall(3*feet) translate x*4*feet+z*18*feet}
	object{InsideWall(9.5*feet) translate x*.5*feet+z*27*feet}
	object{InsideWall(12*feet) rotate y*-90 translate x*10*feet+z*18*feet}
	object{InsideWall(3*feet) rotate y*-90 translate x*(5*feet+8*inch)+z*27*feet}

	object{InsideWall(3.5*feet) rotate y*-90 translate x*41*feet+z*-3.5*feet}
	object{InsideWall(3*feet) translate x*38*feet}
	object{InsideWall(18*feet) rotate y*-90 translate x*38*feet}

	pigment{rgb <.8,.8,1>}
}

#local Floors = union{
}

#local Ceiling = union{
}	

#local Roof = union{
	/* Left half */
	union{
		difference{
			intersection{
				box{<-2.02,.5,-5>*feet,<60,-20,60>*feet
					texture{Shingles}
					rotate -x*degrees(atan2(4,12))
				}
				box{<-2.01,.5,-5>*feet,<60,-20,60>*feet
					texture{Shingles}
					rotate -x*degrees(atan2(4,12))
					scale <1,1,-1>
					translate z*40*feet
				}
				box{<-2.00,-1,-2>*feet,<42,15,55>*feet}
			}
			box{<0,0,-10*feet>,<30*feet,-10*feet, 40*feet>
				rotate z*degrees(atan2(3,12))
				translate x*20*feet
			}
			intersection{
				box{<-2.3,0,-5>*feet,<60,-20,60>*feet
					rotate -x*degrees(atan2(4,12))
				}
				box{<-2.2,0,-5>*feet,<60,-20,60>*feet
					rotate -x*degrees(atan2(4,12))
					scale <1,1,-1>
					translate z*40*feet
				}
				box{<-2.1,-1.1,-0>*feet,<40,15,60>*feet}
				texture{WallTexture}
			}
		}
		/* Upper wall */
		intersection{
			box{<-.1,0,-5>*feet,<60,-20,60>*feet
				rotate -x*degrees(atan2(4,12))
			}
			box{<-.1,0,-5>*feet,<60,-20,60>*feet
				rotate -x*degrees(atan2(4,12))
				scale <1,1,-1>
				translate z*40*feet
			}
			box{<0,0,-2>*feet,<.5,15,60>*feet}
			texture{WallTexture}
		}
		translate y*8*feet
		texture{TrimTexture}
	}
	/* Right half */
	union{
		difference{
			intersection{
				box{<-150,6*inch,-10*feet>,<70*feet,-20*feet, 44*feet>
					texture{Shingles rotate y*90}
					rotate z*degrees(atan2(4,12))
					translate x*20*feet
				}
				box{<0,6*inch,-12*feet>,<80*feet,-20*feet, 44*feet>
					texture{Shingles rotate y*90}
					rotate z*degrees(atan2(4,12))
					scale <-1,1,1>
					translate x*67*feet
				}
				box{<17.8,-1,-6>*feet,<69,25,44>*feet}
			}
			intersection{
				box{<0,-6*inch,-10.3*feet>,<70*feet,-20*feet, 40.3*feet>
					texture{Shingles rotate y*90}
					rotate z*degrees(atan2(4,12))
					translate x*20*feet
				}
				box{<0,-6*inch,-10.4*feet>,<70*feet,-20*feet, 40.4*feet>
					texture{Shingles rotate y*90}
					rotate z*degrees(atan2(4,12))
					scale <-1,1,1>
					translate x*67*feet
				}
				box{<10,-1.1,-6.5>*feet,<69,15,42.5>*feet}
				texture{WallTexture}
			}
		}
		/* Upper wall */
		intersection{
			box{<0,-6*inch,-10.3*feet>,<70*feet,-20*feet, 40.3*feet>
				texture{Shingles rotate y*90}
				rotate z*degrees(atan2(4,12))
				translate x*20*feet
			}
			box{<0,-6*inch,-10.4*feet>,<70*feet,-20*feet, 40.4*feet>
				texture{Shingles rotate y*90}
				rotate z*degrees(atan2(4,12))
				scale <-1,1,1>
				translate x*67*feet
			}
			union{
				box{<10,-1.1,-4>*feet,<69,15,-3.5>*feet}
				box{<10,-1.1,35.5>*feet,<69,15,36>*feet}
			}
			texture{WallTexture}
		}
		
		translate y*8*feet
		texture{TrimTexture}
	}
}

#declare House = union{
	object{Foundation}
	object{ExteriorWalls}
	object{InteriorWalls}
	object{Floors}
	object{Ceiling}
	object{Roof}
}


#ifndef(included)
	plane{y, -1*feet pigment{rgb <0,1,0>}}
	object{House}
	camera{
		location <-10,20,-20>*feet
		look_at  < 20, 0, 10>*feet
	}
//	light_source{<1,1,-1>*1000*feet, 1}
	light_source{y*10*mile, 1}
#end



