// Persistence of Vision Ray Tracer Scene Description File
// File: Macros.inc
// Vers: 3.1
// Desc: A collection of commonly used macros and useful declarations
// Date: 4 July 2000 
// Auth: Gail Shaw
              
#version unofficial MegaPov 0.5;                

#declare rnd=seed(9329);
              
#macro WriteNumber(Name,Value)
 #debug concat ("Value of ", Name , " is: ",str(Value,0,3),"\n")
#end

#macro WriteVector(Name,Value)
 #debug concat ("Value of ", Name , " is: <",str(Value.x,0,3),",",str(Value.y,0,3),",",str(Value.z,0,3),">\n")
#end

#macro HalfTorus(Major,Minor)
 difference {
  torus {
   Major, Minor
  }
  box {
   <0,-Minor*2,-Major*2>,<Major*2,Minor*2,Major*2>
  }
 }
#end  

#macro HalfTorus(Major,Minor)
 intersection {
  torus { Major, Minor }
  box { <0,-Major,-Major-Minor*2>,<Major+Minor*2,Major,Major+Minor*2> }
 }
#end

#macro QuarterTorus(Major,Minor)
 intersection {
  torus { Major, Minor }
  box { <0,-Major,0>,<Major+Minor*2,Major,Major+Minor*2> }
 }
#end
 
#macro HalfSphere(Radius)
 difference {
  sphere {
   <0,0,0>,Radius
  }
  box {
   <-Radius*2,-Radius*2,-Radius*2>,<Radius*2,0,Radius*2>
  }
 }
#end

#macro SlicedBox(Length,Height,Depth,Bevelled)
 difference {
  box {
   <0,0,0>,<Length,Height,Depth>
  }    
  box {
   <0,0,-Depth*2>,<Height*2,Height*2,Depth*2>
   rotate z*45
  }
  #if (Bevelled!=0)
   box {
    <-0.1,-Depth/8,-Depth/8>,<Length*2,Depth/8,Depth/8>
    rotate x*45
    translate <0,Height,0>
  }
   box {
    <-0.1,-Depth/8,-Depth/8>,<Length*2,Depth/8,Depth/8>
    rotate x*45
   }
   box {
    <-Depth/8,-0.1,-Depth/8>,<Depth/8,Height*2,Depth/8>
    rotate y*45
    translate <1.5,0,0>
   } 
   box {
    <-Depth/8,-0.1,-Depth/8>,<Depth/8,Height*2,Depth/8>
    rotate y*45
    rotate z*-45
   }
  #end 
 }
#end               

#macro BevelledBoxold(Length,Height,Depth)
 difference {
  box {
   <0,0,0>,<Length,Height,Depth>
  }
  box {
    <-0.1,-Depth/8,-Depth/8>,<Length*2,Depth/8,Depth/8>
    rotate x*45
    translate <0,Height,0>  
    pigment {rgb <1,0,0>}
  }
  box {
   <-0.1,-Depth/8,-Depth/8>,<Length*2,Depth/8,Depth/8>
   rotate x*45     
   pigment {rgb <1,1,0>}
  }
  box {
   <-Depth/8,-0.1,-Depth/8>,<Depth/8,Height*2,Depth/8>
   rotate y*45
   translate <1.5,0,0>  
   pigment {rgb <0,1,0>}
  } 
  box {
   <-Depth/8,-0.1,-Depth/8>,<Depth/8,Height*2,Depth/8>
   rotate y*45                 
   pigment {rgb <0,0,1>}
  }
 }
#end

#macro BevelledBox(Start,End,BevelWidth)
 // note bevelling is in the x-y plane
 // Ensuring that box orientation is known. There's probably a much easier way
 #if (Start.x<End.x)
  #local BoxMin = <Start.x,0,0>;
  #local BoxMax = <End.x,0,0>;
 #else
  #local BoxMin = <End.x,0,0>;
  #local BoxMax = <Start.x,0,0>;
 #end
 #if (Start.y<End.y)
  #local BoxMin = <BoxMin.x,Start.y,0>;
  #local BoxMax = <BoxMax.x,End.y,0>;
 #else
  #local BoxMin = <BoxMin.x,End.y,0>;
  #local BoxMax = <BoxMax.x,Start.y,0>;
 #end#if (Start.z<End.z)
  #local BoxMin = <BoxMin.x,BoxMin.y,Start.z>;
  #local BoxMax = <BoxMax.x,BoxMax.y,End.z>;
 #else
  #local BoxMin = <BoxMin.x,BoxMin.y,End.z>;
  #local BoxMax = <BoxMax.x,BoxMax.y,Start.z>;
 #end   
 #local Width=BoxMax.x-BoxMin.x;
 #local Height=BoxMax.y-BoxMin.y;
 #local Depth=BoxMax.y-BoxMin.y; 
 #if ((BevelWidth>Width/2) | (BevelWidth>Height/2))
  #error "Bevel Width may not be more than half the width or height of the box"
 #end
 difference {
  box {
   BoxMin,BoxMax
  }
  box {
    <BoxMin.x-0.1,-BevelWidth,-BevelWidth>,<BoxMax.x+0.1,BevelWidth,BevelWidth>
    rotate x*45
    translate <0,BoxMax.y,BoxMin.z>  
  }
  box {
   <BoxMin.x-0.1,-BevelWidth,-BevelWidth>,<BoxMax.x+0.1,BevelWidth,BevelWidth>
   rotate x*45 
   translate <0,BoxMin.y,BoxMin.z>    
  }
  box {
   <-BevelWidth,BoxMin.y-0.1,-BevelWidth>,<BevelWidth,BoxMax.y+0.1,BevelWidth>
   rotate y*45
   translate <BoxMax.x,0,BoxMin.z>  
  } 
  box {
   <-BevelWidth,BoxMin.y-0.1,-BevelWidth>,<BevelWidth,BoxMax.y+0.1,BevelWidth>
   rotate y*45         
   translate <BoxMin.x,0,BoxMin.z>        
  }
 }
#end

#macro RoundedBox(Start,End,Radius)
 union {
  box {
   <Start.x+Radius,Start.y,Start.z+Radius>,<End.x-Radius,End.y,End.z -Radius> 
  }
  box {
   <Start.x+Radius,Start.y+Radius,Start.z>,<End.x-Radius,End.y-Radius,End.z>
  }
  box {
   <Start.x,Start.y+Radius,Start.z+Radius>,<End.x,End.y-Radius,End.z-Radius>
  }
  cylinder {
   <Start.x+Radius,Start.y+Radius,Start.z+Radius>,<End.x-Radius,Start.y+Radius,Start.z+Radius>,Radius
  }
  cylinder {
   <Start.x+Radius,End.y-Radius,Start.z+Radius>,<End.x-Radius,End.y-Radius,Start.z+Radius>,Radius
  }                                                   
  cylinder {
   <Start.x+Radius,Start.y+Radius,End.z-Radius>,<End.x-Radius,Start.y+Radius,End.z-Radius>,Radius
  }                    
  cylinder {
   <Start.x+Radius,End.y-Radius,End.z-Radius>,<End.x-Radius,End.y-Radius,End.z-Radius>,Radius
  }
  cylinder {
   <Start.x+Radius,Start.y+Radius,Start.z+Radius>,<Start.x+Radius,Start.y+Radius,End.z-Radius>,Radius
  }
  cylinder {
   <End.x-Radius,Start.y+Radius,Start.z+Radius>,<End.x-Radius,Start.y+Radius,End.z-Radius>,Radius
  }
  cylinder {
   <Start.x+Radius,End.y-Radius,Start.z+Radius>,<Start.x+Radius,End.y-Radius,End.z-Radius>,Radius
  }
  cylinder {
   <End.x-Radius,End.y-Radius,Start.z+Radius>,<End.x-Radius,End.y-Radius,End.z-Radius>,Radius
  }
  cylinder {
   <Start.x+Radius,Start.y+Radius,Start.z+Radius>,<Start.x+Radius,End.y-Radius,Start.z+Radius>,Radius
  }
  cylinder {
   <End.x-Radius,Start.y+Radius,Start.z+Radius>,<End.x-Radius,End.y-Radius,Start.z+Radius>,Radius
  } 
  cylinder {
   <Start.x+Radius,Start.y+Radius,End.z-Radius>,<Start.x+Radius,End.y-Radius,End.z-Radius>,Radius
  }
  cylinder {
   <End.x-Radius,Start.y+Radius,End.z-Radius>,<End.x-Radius,End.y-Radius,End.z-Radius>,Radius
  }
  sphere {
   <Start.x+Radius,Start.y+Radius,Start.z+Radius>,Radius
  }
  sphere {
   <End.x-Radius,Start.y+Radius,Start.z+Radius>,Radius
  }                            
  sphere {
   <Start.x+Radius,End.y-Radius,Start.z+Radius>,Radius
  } 
  sphere {
   <End.x-Radius,End.y-Radius,Start.z+Radius>,Radius
  }
  sphere {
   <Start.x+Radius,Start.y+Radius,End.z-Radius>,Radius
  }
  sphere {
   <End.x-Radius,Start.y+Radius,End.z-Radius>,Radius
  }                            
  sphere {
   <Start.x+Radius,End.y-Radius,End.z-Radius>,Radius
  } 
  sphere {
   <End.x-Radius,End.y-Radius,End.z-Radius>,Radius
  }
 }
#end  

// by Ron Parker
#macro RoundedCylinder( StartPt, EndPt, Radius, EndRadius )
  #local startpt=<0,0,0>+StartPt;
  #local endpt=<0,0,0>+EndPt;  

  #if ( EndRadius > Radius/2 ) 
    #error "End radius must be less than half the radius\n";
  #end
  
  #local endcenter=(vlength(endpt-startpt)-EndRadius);
  #if ( endcenter < EndRadius ) 
    #error "End radius must be less than half the length\n";
  #end
               
  #local vX1=y;
  #local vX2=vnormalize(endpt-startpt);
  #if ( vX2.x != 0 | vX2.z != 0 )
    #local vY=vnormalize(vcross(vX1,vX2));
    #local vZ1=vnormalize(vcross(vX1,vY));
    #local vZ2=vnormalize(vcross(vX2,vY));
    
    #local trans=transform {
      matrix < vX1.x, vY.x,vZ1.x, vX1.y,vY.y,vZ1.y, vX1.z,vY.z, vZ1.z, 0,0,0 >
      matrix < vX2.x,vX2.y,vX2.z,  vY.x,vY.y, vY.z, vZ2.x,vZ2.y,vZ2.z, 0,0,0 >
      translate startpt
    }
  #else
    #local trans={ translate startpt }
  #end
  cylinder {startpt, endpt, Radius-EndRadius}
  cylinder {y*EndRadius, y*endcenter, Radius transform trans}
  torus    {Radius-EndRadius, EndRadius translate y*EndRadius transform trans}
  torus    {Radius-EndRadius, EndRadius translate y*endcenter transform trans}
#end

  
#macro SphereHelix(Major,Minor,Turns,Spacing) 
 sphere_sweep {
  b_spline_sphere_sweep
  4*Turns+2,
  <0,-Spacing/4,-Major>,Minor,
  #local Height=0;
  #local TurnNo=0;
  #while (TurnNo<Turns)
   <Major,Height,0>,Minor,
   <0,Height+Spacing/4,Major>,Minor,
   <-Major,Height+Spacing/2,0>,Minor,
   <0,Height+Spacing*3/4,-Major>,Minor,
   #local TurnNo=TurnNo+1;
   #local Height=Height+Spacing;
  #end
  <Major,Height,0>,Minor
 }
#end

#macro Stars (Size,Count,Distance)
 #local StarSeed=seed(3929); // was 4632
 #local ColorSeed=seed(9292);
 #local SizeSeed=seed(4843);
 #local Ind=0;
 #while(Ind<Count)
  #local Location = <-1+rand(StarSeed)*2, -1+rand(StarSeed)*2, -1+rand(StarSeed)*2>;
  #while(vlength(Location) > 1)
    #local Location = <-1+rand(StarSeed)*2, -1+rand(StarSeed)*2, -1+rand(StarSeed)*2>;
  #end
  #local Location = Distance*Location/vlength(Location);
  sphere { 
   Location, Size+rand(SizeSeed)*Size/2 
   pigment {rgb <1-rand(ColorSeed)*0.15,1-rand(ColorSeed)*0.15,1-rand(ColorSeed)*0.15> }
   finish {ambient 1}
  }
  #local Ind=Ind+1;
 #end
#end

#macro RndRange(rndStream,Max,Min)
 (rand(rndStream)*(Max-Min)+Min)
#end

#macro gauss(Seed,StdDeviation)
  #local l=rand(Seed);
  #if (rand(Seed)>0.5)
   (sqrt(-2*log(l)/log(1.543873443971))*cos(2*pi*l))*StdDeviation
  #else
   -1*(sqrt(-2*log(l)/log(1.543873443971))*cos(2*pi*l))*StdDeviation
  #end   
#end

#declare F_Granite=
 function {pigment {granite}}
 
#declare F_Bumps=
 function {pigment {bumps}}
 
#declare F_Wrinkles=
 function {pigment {wrinkles}}
 
#declare F_Crackle=
 function {pigment {crackle color_map {[0.01 rgb 0][1 rgb 1]}}} 