/*-------------------------------------------------------

            Scatter Macros for POV-Ray 3.5

Version: 0.1            
Author: Jonathan Rafael Ghiglia 
Date: 25th November 2001

                -Terms of use-

- You can freely use these macros in your works.
- You can freely use and change this code (credit is always welcome :).

The comments should make it easy to adapt it to your specific needs.
Of course, if you make interesting changes or find bugs I'd like you to contact me!

//---------------------------------------------------------//

How the macro works (brief explanation):

This macro uses a fairly simple geometrical algorithm.

At first the macro creates an array of points called object_wire. 
This array contains the coordinates of every point of the object's bottom in a grid of accuracy_x*accuracy_z.
The macro uses the trace() function to store these coordinates. These coordinates will be used later when placing
the object. If no intersection occurs, the macro gives that point a huge y coordinate, in order to avoid that 
that point is taken into account while looking for the closest point to the ground (see below).

Then a position is chosen by the position_macro (), together with a random rotation amount around the y axis.
The coordinates contained in object_wire are transformed and stored into another array called present_wire.
Notice that the object_wire array is never transformed.

The coordinates contained in present_wire are used to find the corresponding coordinates of the scatter_land object. 
Again the trace() function is used. The coordinates found are stored into an array called land_wire. If no intersection
occurs, the point is given a huge (negative) y coordinate (see above).

The macro looks for the minimum distance between two corresponding points of present_wire and land_wire.
This is the first point of contact between the ground and the object. It's the only point of contact that has got
the 100% probability to be truly a point of contact :)

Now the macro uses that point to find other two points of contact. The research is restricted to that part of the present_wire
grid limited by the first point of contact and containing the ideal mass centre of the grid. Truly, the macro searches for two
angles. The first angle will represent the amount of rotation performed around a fixed axis passing through the first point of
contact (this axis is parallel to the z axis before being rotated by the random amount). Once this first rotation has been done,
the second angle will represent the amount of rotation around the axis passing through the first and the second points of contact.
The macro looks for the smallest rotation angle in a specific direction by calculating the angle between the projection of the vectors
(coordinates) of present_wire onto the plane passing through the first point of contact and normal to that direction, and the 
corresponding vectors (coordinates) of land_wire. The first point of contact is treated as the origin (so the vectors I'm referring to 
are the vectors linking every point of present_wire and land_wire to the first point of contact).

Then the macro writes on a file the scatter_object with all the tranformations performed. The same object is added to the scatter_land.

This was just a brief explanation. The code speaks volumes...

------------------------------------------------------------------*/
                 
/*----------------------------------------------------------------

The macro ang_vec is used to calculate the angle between two vectors.
The formula is elementary:
vdot (v1,v2) := vlength(v1)*vlength(v2)*costheta
==> theta = acos (vdot(v1,v2)/(vlength(v1)*vlength(v2)))

---------------------------------------------------------------*/
 
#macro ang_vec (v1,v2)
#if (vdot(v1,v2)/(vlength (v1)*vlength(v2))>1) // Inside approximations can lead to this error, which is theoretically impossible. 
#local ang=0; #else
#local ang= acos(vdot(v1,v2)/(vlength (v1)*vlength(v2)));
#end
ang/pi*180 
#end 

//-----scatter macro

#macro scatter (name_file)  

#ifdef (max_height)  // The objects cannot be placed higher than max_height from the ground
#local old_land= object {scatter_land} // The scatter_land grows, but I need to know the y distance from the real ground
#end

#declare epsilon = 0.0001; // Using trace for the edges of the bounding box might lead to unexpected results. So I use trace just inside it.
#declare huge = 10000;  // a huge value
#declare _min = min_extent (scatter_object);
#declare _max = max_extent (scatter_object);
#declare object_wire = array[x_accuracy+1][z_accuracy+1]  // I think of accuracy as the parts which the object is divided in. For this reason, the number of points is accuracy +1.
#declare present_wire = array[x_accuracy+1][z_accuracy+1]
#declare land_wire = array[x_accuracy+1][z_accuracy+1]
#local norm = <0,0,0>;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare object_wire[i][j] = trace (scatter_object, _min +epsilon*(x+z) +i*(_max.x-_min.x-2*epsilon)/x_accuracy*x +j*(_max.z-_min.z-2*epsilon)/z_accuracy*z -huge*y,y, norm);
#if (vlength (norm)=0) #declare object_wire[i][j]=_min +epsilon*(x+z) +i*(_max.x-_min.x-2*epsilon)/x_accuracy*x +j*(_max.z-_min.z-2*epsilon)/z_accuracy*z +huge*y; #end // If intersection does not occur I use a huge y coordinate.
#local j=j+1;
#end  
#local i=i+1;
#end

#fopen scatter_file name_file write
#write (scatter_file, "/*------Scatter array file------*/ \n") 
#write (scatter_file,"union { \n" )

#declare present_object = 1;
#while (present_object<= total_objects)

#declare rand_rot = 360*rand(RS);//  random rotation amount around y axis
#declare pos= position_macro(); // the chosen position 

#declare ok_pos=0;
#while (ok_pos=0) // If the position isn't okay, this variable will be set to 0.  
#declare ok_pos=1;  

//-------- the object_wire's coordinates are transformed and stored into present_wire
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare present_wire[i][j] = vtransform (object_wire[i][j], transform {rotate rand_rot*y translate pos}); 
#local j=j+1;
#end  
#local i=i+1;
#end

//-------- the present_wire's coordinates are used to calculate the corresponding coordinates of land_wire
#local norm = <0,0,0>;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare land_wire[i][j] = trace (scatter_land,present_wire[i][j]+huge*y,-y,norm) ;
#if (vlength (norm)=0) #declare land_wire[i][j]=land_wire[i][j]-huge*y; #end // If intersection does not occur I use a huge (negative) y coordinate.
#local j=j+1;
#end  
#local i=i+1;
#end

//------- find the minimum distance between the present_wire's points and the corresponding land_wire's ones
#declare i_min=0;
#declare j_min=0;
#declare min_distance = huge;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#if (present_wire[i][j].y-land_wire[i][j].y<min_distance) #declare min_distance = present_wire[i][j].y-land_wire[i][j].y; #declare i_min=i; #declare j_min=j; #end
#local j=j+1;
#end  
#local i=i+1;
#end 

//------- check whether the first point of contact (land_wire[i_min][j_min]) is higher than max_height or not. 
// If it's higher the macro looks for the lowest point in the four directions and changes the position accordingly.
#ifdef (max_height)
#if (land_wire[i_min][j_min].y>trace(old_land,land_wire[i_min][j_min]+huge*y,-y).y+max_height)
#local vec_n=trace(scatter_land,land_wire[i_min][j_min]+huge*y+vlength(_max-_min)*z,-y);
#local vec_s=trace(scatter_land,land_wire[i_min][j_min]+huge*y-vlength(_max-_min)*z,-y);
#local vec_e=trace(scatter_land,land_wire[i_min][j_min]+huge*y+vlength(_max-_min)*x,-y);
#local vec_w=trace(scatter_land,land_wire[i_min][j_min]+huge*y-vlength(_max-_min)*x,-y);
#if (min(vec_n.y,vec_s.y)<min(vec_e.y,vec_w.y)) #declare pos=pos+(vec_n.y<vec_s.y?vec_n:vec_s)*<1,0,1>;
#else #declare pos=pos+(vec_e.y<vec_w.y?vec_e:vec_w)*<1,0,1>;
#end
#declare ok_pos=0; 
#debug "Object's position too high! Position changed.\n"
#end
#end

#if (ok_pos!=0) 

//----- Notice that the present_wire's points are not transformed here for speed reasons. When the other transformations are performed, 
// I will have to remember that the real coordinates have to be translated by -min_distance*y.

/*-----find the first angle of rotation (here called second_angle because it refers to the second point of contact)
The object will be rotated by this angle around the axis passing through land_wire[i_min][j_min] (the first point of contact) and parallel
to vtransform (z, transform {rotate rand_rot*y})
The projection of a generic vector present_wire[i][j]-present_wire[i_min][j_min] onto that plane (passing through the first point of contact
and normal to that direction) is here <present_wire[i][j_min].x,present_wire[i][j].y,present_wire[i][j_min].z>-present_wire[i_min][j_min].
*/
#if (i_min< ceil (x_accuracy/2)) #declare i_step=1; #else #declare i_step=-1; #end // if the first point of contact is *on the left* of the medium point you go to the right and vice versa 
#if (j_min< ceil (z_accuracy/2)) #declare j_step=1; #else #declare j_step=-1; #end // same here
#declare second_angle=huge;
#declare i=i_min+i_step;
#while (i<= x_accuracy & i>=0)
#declare j=j_min+j_step;
#while (j<=z_accuracy & j>=0)
#if (ang_vec(present_wire[i][j_min]*<1,0,1>+present_wire[i][j].y*y-present_wire[i_min][j_min],land_wire[i][j_min]*<1,0,1>+land_wire[i][j].y*y-land_wire[i_min][j_min])<second_angle)//ang_vec(present_wire[second_i][second_j]-present_wire[i_min][j_min],land_wire[second_i][second_j]-land_wire[i_min][j_min]))
#declare second_angle = ang_vec(present_wire[i][j_min]*<1,0,1>+present_wire[i][j].y*y-present_wire[i_min][j_min],land_wire[i][j_min]*<1,0,1>+land_wire[i][j].y*y-land_wire[i_min][j_min]); 
#declare second_i=i;
#declare second_j=j;
#end
#declare j=j+j_step;
#end
#declare i=i+i_step;
#end  

//-----transform present_wire
#declare dir1 = -i_step*vtransform (z, transform {rotate rand_rot*y}); // The sign of the angle depends on i_step

#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare present_wire[i][j] = vtransform (present_wire[i][j], transform {translate -(land_wire[i_min][j_min]+min_distance*y) Axis_Rotate_Trans (dir1,second_angle) translate land_wire[i_min][j_min]+min_distance*y}); 
#local j=j+1;
#end  
#local i=i+1;
#end   

//----recalculate land_wire: this can lead to more accurate results, but slows things down... :(
#local norm = <0,0,0>;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare land_wire[i][j] = trace (scatter_land,present_wire[i][j]+huge*y,-y,norm) ;
#if (vlength (norm)=0) #declare land_wire[i][j]=land_wire[i][j]-huge*y; #end // see above
#local j=j+1;
#end  
#local i=i+1;
#end




/*-----find the second angle of rotation (here called third_angle because it refers to the third point of contact).
The object will be rotated by this angle around the axis passing through land_wire[i_min][j_min] (the first point of contact) and
land_wire[second_i][second_j] (the second point of contact).
The projection of a generic vector present_wire[i][j]-present_wire[i_min][j_min] onto that plane (passing through the first point of contact
and normal to that direction) is here present_wire[i][j]-present_wire[i_min][j_min]-vdot(present_wire[i][j]-present_wire[i_min][j_min],dir1)*dir1
where dir1 is the normal of that plane.
*/

#declare i=i_min+i_step; // start from i_min
#if (j_step=1)  // this and the following two lines do have to be improved. The macro tries to find the more *logical* rotation direction.
#if (second_j< ceil (j_min+(z_accuracy-j_min)/2)) #declare j_step=1; #else #declare j_step=-1; #end
#else #if (second_j< ceil(j_min/2)) #declare j_step=1; #else #declare j_step=-1; #end #end

#declare third_angle=huge;
#while (i<= x_accuracy & i>=0 & i!=i_min)
#declare j=second_j+j_step;
#while (j<=z_accuracy & j>=0 & j!=j_min)
#if (ang_vec(present_wire[i][j]-present_wire[i_min][j_min]-vdot(present_wire[i][j]-present_wire[i_min][j_min],dir1)*dir1,land_wire[i][j]-land_wire[i_min][j_min]-vdot(land_wire[i][j]-land_wire[i_min][j_min],dir1)*dir1)<third_angle)//ang_vec(present_wire[second_i][second_j]-present_wire[i_min][j_min],land_wire[second_i][second_j]-land_wire[i_min][j_min]))
#declare third_angle = ang_vec(present_wire[i][j]-present_wire[i_min][j_min]-vdot(present_wire[i][j]-present_wire[i_min][j_min],dir1)*dir1,land_wire[i][j]-land_wire[i_min][j_min]-vdot(land_wire[i][j]-land_wire[i_min][j_min],dir1)*dir1);

#declare third_i=i;
#declare third_j=j;

#end
#declare j=j+j_step;
#end
#declare i=i+i_step;
#end  

#declare dir2=-j_step*vnormalize(present_wire[second_i][second_j]-present_wire[i_min][j_min]);

//---- Here I don't have to tranform all the present_wire, just the third point of contact (the other two points lay on the rotation axis)
#declare present_wire[third_i][third_j] = vtransform (present_wire[third_i][third_j], transform {translate -(land_wire[i_min][j_min]+min_distance*y) Axis_Rotate_Trans (dir2,third_angle) translate land_wire[i_min][j_min]+min_distance*y}); 

//---- Check if the three points of contact are really touching the corresponding points of land_wire. Since these points are rotated, and 
// the corresponding points of land_wire are just their projection, they truly never touch. That's why a height_accuracy float is needed. 
// the first point of contact doesn't need checking.
#if (present_wire[second_i][second_j].y>land_wire[second_i][second_j].y+min_distance+height_accuracy | present_wire[third_i][third_j].y>land_wire[third_i][third_j].y+min_distance+height_accuracy)
#declare pos=pos+vnormalize(present_wire[second_i][second_j]/2-present_wire[i_min][j_min]+present_wire[third_i][third_j]/2)*<1,0,1>*vlength(_max-_min);
#declare ok_pos=0;
#debug concat("Object ", str(present_object,0,0), " badly placed on point ",str(i,5,3),", "str(j,5,3),". Changing...\n")
#else
#declare ok_pos=1;
#end 
#end
#end  

//----- write file 

#debug concat ("Placed ",str (present_object,0,0)," objects.\n")

#ifndef (scatter_texture_macro)
#write (scatter_file,"object {scatter_object_macro() rotate ",rand_rot,"*y translate ",pos-min_distance*y," translate ",-land_wire[i_min][j_min]," Axis_Rotate_Trans(",dir1,",",second_angle,") Axis_Rotate_Trans(",dir2,",",third_angle,") translate ",land_wire[i_min][j_min]," }\n")

#else
#write (scatter_file,"object {scatter_object rotate ",rand_rot,"*y translate ",pos-min_distance*y," translate ",-land_wire[i_min][j_min]," Axis_Rotate_Trans(",dir1,",",second_angle,") Axis_Rotate_Trans(",dir2,",",third_angle,") translate ",land_wire[i_min][j_min]," scatter_texture_macro() }\n")
#end

//----- add object to scatter_land
#declare scatter_land=
union {
object {scatter_land}
object {scatter_object rotate rand_rot*y translate pos-min_distance*y translate -land_wire[i_min][j_min] Axis_Rotate_Trans(dir1,second_angle) Axis_Rotate_Trans(dir2,third_angle) translate land_wire[i_min][j_min]}
}

#declare present_object= present_object+1;
#end  

#write (scatter_file, "}")
#fclose scatter_file  

#end 

              

#macro scatter_multiple (N,name_file)  

#ifdef (max_height)
#local old_land= object {scatter_land}
#end

#declare epsilon = 0.0001;
#declare huge = 10000; 
#declare object_wire = array[x_accuracy+1][z_accuracy+1][N]
#declare _min=array[N]
#declare _max=array[N]   
#declare scatter_object=array[N]
#declare present_wire = array[x_accuracy+1][z_accuracy+1]
#declare land_wire = array[x_accuracy+1][z_accuracy+1]
#local _n=0;
#while (_n<N) 
#declare scatter_object[_n]=object {scatter_object_macro(_n+1)}
#declare _min[_n] = min_extent (scatter_object[_n]);
#declare _max[_n] = max_extent (scatter_object[_n]);
#local norm = <0,0,0>;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare object_wire[i][j][_n] = trace (scatter_object[_n], _min[_n] +epsilon*(x+z) +i*(_max[_n].x-_min[_n].x-2*epsilon)/x_accuracy*x +j*(_max[_n].z-_min[_n].z-2*epsilon)/z_accuracy*z -huge*y,y, norm);
#if (vlength (norm)=0) #declare object_wire[i][j][_n]=_min[_n] +epsilon*(x+z) +i*(_max[_n].x-_min[_n].x-2*epsilon)/x_accuracy*x +j*(_max[_n].z-_min[_n].z-2*epsilon)/z_accuracy*z +huge*y; #end // so that it's too high
#local j=j+1;
#end  
#local i=i+1;
#end 
#local _n=_n+1;
#end

#fopen scatter_file name_file write
#write (scatter_file, "/*------Scatter array file------*/ \n") 
#write (scatter_file,"union { \n" )

#declare present_object = 1;
#while (present_object<= total_objects)

#declare _N=floor(rand(RS)*N);   //-----choose an object
#declare rand_rot = 360*rand(RS);
#declare pos= position_macro();  

#declare ok_pos=0;
#while (ok_pos=0)   
#declare ok_pos=1;  

#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare present_wire[i][j] = vtransform (object_wire[i][j][_N], transform {rotate rand_rot*y translate pos}); 
#local j=j+1;
#end  
#local i=i+1;
#end

#local norm = <0,0,0>;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare land_wire[i][j] = trace (scatter_land,present_wire[i][j]+huge*y,-y,norm) ;
#if (vlength (norm)=0) #declare land_wire[i][j]=land_wire[i][j]-huge*y; #end // so that it's too low
#local j=j+1;
#end  
#local i=i+1;
#end

#declare i_min=0;
#declare j_min=0;
#declare min_distance = huge;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#if (present_wire[i][j].y-land_wire[i][j].y<min_distance) #declare min_distance = present_wire[i][j].y-land_wire[i][j].y; #declare i_min=i; #declare j_min=j; #end
#local j=j+1;
#end  
#local i=i+1;
#end 

#ifdef (max_height)
#if (land_wire[i_min][j_min].y>trace(old_land,land_wire[i_min][j_min]+huge*y,-y).y+max_height)
#local vec_n=trace(scatter_land,land_wire[i_min][j_min]+huge*y+vlength(_max[_N]-_min[_N])*z,-y);
#local vec_s=trace(scatter_land,land_wire[i_min][j_min]+huge*y-vlength(_max[_N]-_min[_N])*z,-y);
#local vec_e=trace(scatter_land,land_wire[i_min][j_min]+huge*y+vlength(_max[_N]-_min[_N])*x,-y);
#local vec_w=trace(scatter_land,land_wire[i_min][j_min]+huge*y-vlength(_max[_N]-_min[_N])*x,-y);
#if (min(vec_n.y,vec_s.y)<min(vec_e.y,vec_w.y)) #declare pos=pos+(vec_n.y<vec_s.y?vec_n:vec_s)*<1,0,1>;
#else #declare pos=pos+(vec_e.y<vec_w.y?vec_e:vec_w)*<1,0,1>;
#end
#declare ok_pos=0; 
#debug "Object's position too high! Position changed.\n"
#end
#end

#if (ok_pos!=0) 


#if (i_min< ceil (x_accuracy/2)) #declare i_step=1; #else #declare i_step=-1; #end
#if (j_min< ceil (z_accuracy/2)) #declare j_step=1; #else #declare j_step=-1; #end
#declare second_angle=huge;
#declare i=i_min+i_step;
#while (i<= x_accuracy & i>=0)
#declare j=j_min+j_step;
#while (j<=z_accuracy & j>=0)
#if (ang_vec(present_wire[i][j_min]*<1,0,1>+present_wire[i][j].y*y-present_wire[i_min][j_min],land_wire[i][j_min]*<1,0,1>+land_wire[i][j].y*y-land_wire[i_min][j_min])<second_angle)//ang_vec(present_wire[second_i][second_j]-present_wire[i_min][j_min],land_wire[second_i][second_j]-land_wire[i_min][j_min]))
#declare second_angle = ang_vec(present_wire[i][j_min]*<1,0,1>+present_wire[i][j].y*y-present_wire[i_min][j_min],land_wire[i][j_min]*<1,0,1>+land_wire[i][j].y*y-land_wire[i_min][j_min]); 
#declare second_i=i;
#declare second_j=j;
#end
#declare j=j+j_step;
#end
#declare i=i+i_step;
#end  

#declare dir1 = -i_step*vtransform (z, transform {rotate rand_rot*y}); 

#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare present_wire[i][j] = vtransform (present_wire[i][j], transform {translate -(land_wire[i_min][j_min]+min_distance*y) Axis_Rotate_Trans (dir1,second_angle) translate land_wire[i_min][j_min]+min_distance*y}); 
#local j=j+1;
#end  
#local i=i+1;
#end   

#local norm = <0,0,0>;
#local i=0;
#while (i<=x_accuracy)
#local j=0; 
#while (j<=z_accuracy)
#declare land_wire[i][j] = trace (scatter_land,present_wire[i][j]+huge*y,-y,norm) ;
#if (vlength (norm)=0) #declare land_wire[i][j]=land_wire[i][j]-huge*y; #end // so that it's too low
#local j=j+1;
#end  
#local i=i+1;
#end


#declare i=i_min+i_step;
#if (j_step=1)
#if (second_j< ceil (j_min+(z_accuracy-j_min)/2)) #declare j_step=1; #else #declare j_step=-1; #end
#else #if (second_j< ceil(j_min/2)) #declare j_step=1; #else #declare j_step=-1; #end #end

#declare third_angle=huge;
#while (i<= x_accuracy & i>=0 & i!=i_min)
#declare j=second_j+j_step;
#while (j<=z_accuracy & j>=0 & j!=j_min)
#if (ang_vec(present_wire[i][j]-present_wire[i_min][j_min]-vdot(present_wire[i][j]-present_wire[i_min][j_min],dir1)*dir1,land_wire[i][j]-land_wire[i_min][j_min]-vdot(land_wire[i][j]-land_wire[i_min][j_min],dir1)*dir1)<third_angle)//ang_vec(present_wire[second_i][second_j]-present_wire[i_min][j_min],land_wire[second_i][second_j]-land_wire[i_min][j_min]))
#declare third_angle = ang_vec(present_wire[i][j]-present_wire[i_min][j_min]-vdot(present_wire[i][j]-present_wire[i_min][j_min],dir1)*dir1,land_wire[i][j]-land_wire[i_min][j_min]-vdot(land_wire[i][j]-land_wire[i_min][j_min],dir1)*dir1);

#declare third_i=i;
#declare third_j=j;

#end
#declare j=j+j_step;
#end
#declare i=i+i_step;
#end  

#declare dir2=-j_step*vnormalize(present_wire[second_i][second_j]-present_wire[i_min][j_min]);


#declare present_wire[third_i][third_j] = vtransform (present_wire[third_i][third_j], transform {translate -(land_wire[i_min][j_min]+min_distance*y) Axis_Rotate_Trans (dir2,third_angle) translate land_wire[i_min][j_min]+min_distance*y}); 

#if (present_wire[second_i][second_j].y>land_wire[second_i][second_j].y+min_distance+height_accuracy | present_wire[third_i][third_j].y>land_wire[third_i][third_j].y+min_distance+height_accuracy)
#declare pos=pos+vnormalize(present_wire[second_i][second_j]/2-present_wire[i_min][j_min]+present_wire[third_i][third_j]/2)*<1,0,1>*vlength(_max[_N]-_min[_N]);
#declare ok_pos=0;
#debug concat("Object ", str(present_object,0,0), " badly placed on point ",str(i,5,3),", "str(j,5,3),". Changing...\n")
#else
#declare ok_pos=1;
#end 
#end
#end  

#debug concat ("Placed ",str (present_object,0,0)," objects.\n")

#ifndef (scatter_texture_macro)
#write (scatter_file,"object {scatter_object_macro(",_N+1,") rotate ",rand_rot,"*y translate ",pos-min_distance*y," translate ",-land_wire[i_min][j_min]," Axis_Rotate_Trans(",dir1,",",second_angle,") Axis_Rotate_Trans(",dir2,",",third_angle,") translate ",land_wire[i_min][j_min]," }\n")

#else
#write (scatter_file,"object {scatter_object_macro(",_N+1,") rotate ",rand_rot,"*y translate ",pos-min_distance*y," translate ",-land_wire[i_min][j_min]," Axis_Rotate_Trans(",dir1,",",second_angle,") Axis_Rotate_Trans(",dir2,",",third_angle,") translate ",land_wire[i_min][j_min]," scatter_texture_macro (",_N+1,") }\n")
#end

#declare scatter_land=
union {
object {scatter_land}
object {scatter_object[_N] rotate rand_rot*y translate pos-min_distance*y translate -land_wire[i_min][j_min] Axis_Rotate_Trans(dir1,second_angle) Axis_Rotate_Trans(dir2,third_angle) translate land_wire[i_min][j_min]}
}

#declare present_object= present_object+1;
#end  

#write (scatter_file, "}")
#fclose scatter_file  

#end 
              











   









            