// Persistence of Vision Ray Tracer Scene Description File
// File: ?.pov
// Vers: 3.5
// Desc: Basic Scene Example
// Date: mm/dd/yy
// Auth: Jeff M. Thomas
//

#version 3.5;

#include "functions.inc"
#include "colors.inc"
#include "textures.inc"
#include "shapes.inc"
#include "skies.inc"
#include "metals.inc"
#include "glass.inc"
#include "stones.inc"
#include "woods.inc"     

#include "transforms.inc"


////////////////////////////////////////////////////////////
//
// --------------------------------------- Option Flags
//
////////////////////////////////////////////////////////////


#local FULL_RENDER = false;

#if (FULL_RENDER)

  #local GROUND = true;
  #local CHURCH_WALL = true;
  #local BRICK_WALL = true;
  #local ISO_CHURCH_WALL = false;
  #local WINDOW_CUTOUT = true;
  #local BUTTRESSES =true;
  #local BRICK_BUTTRESSES = true;
  #local PILE_ON_SNOW = true;
  #local ICICLES = true;    

  #local WINDOW = true;
  #local SIMPLE_WINDOW = false;
  #local NO_GLASS = false;
  #local WINDOW_FRAME = true;

  #local BACKDROP = false;
  #local BACK_COLUMNS = false;    

  #local BENCH = true;
  #local CRYPT = true;    
  #local CROSS = true;
  #local TREE = true;    
  #local MAIN_TREE = true;
  #local STATUE = true;
  #local CAT = true;    
  #local CAT_BENCH_PRINTS = true;    

  #local SKY_SPHERE = true;
  
  #local CAMERA = 0;
  
  #local WORKLIGHT = false;
  #local PHOTONS = true;
  #local PHOTONS_LOAD = true;
  #local WINDOW_LIGHT = true;
  #local AREA_LIGHT = true;

  
  #local LAMPS = true;
  #local LAMPS_LIT = true;

#else // working render

  #local GROUND = true;
  #local CHURCH_WALL = true;
  #local BRICK_WALL = true;
  #local ISO_CHURCH_WALL = false;
  #local WINDOW_CUTOUT = true;    
  #local BUTTRESSES =true;
  #local BRICK_BUTTRESSES = true;    
  #local PILE_ON_SNOW = true;    
  #local ICICLES = true;
  
  #local WINDOW = true;
  #local SIMPLE_WINDOW = false;
  #local NO_GLASS = false;    
  #local WINDOW_FRAME = true;

  #local BACKDROP = true;
  #local BACK_COLUMNS = false;    

  #local BENCH = true;
  #local CRYPT = true;
  #local CROSS = true;
  #local TREE = true;    
  #local MAIN_TREE = true;
  #local STATUE = false;
  #local GRAVE_FENCE = true;
  #local CAT = true;
  #local CAT_BENCH_PRINTS = true;    
  
  #local SKY_SPHERE = true;  

  #local CAMERA = 0;
  
  #local WORKLIGHT = false;
  #local PHOTONS = true;
  #local PHOTONS_LOAD = true;
  #local WINDOW_LIGHT = true;
  #local AREA_LIGHT = true;
  
  #local LAMPS = true;
  #local LAMPS_LIT = true;

#end // FULL_RENDER

////////////////////////////////////////////////////////////
//
// --------------------------------------- Declares
//
////////////////////////////////////////////////////////////

#local Camera_Height = 3;

#if (ICICLES)
  #include "icicle.inc"
#end

////////////////////////////////////////////////////////////
//
// --------------------------------------- Textures
//
////////////////////////////////////////////////////////////

#declare Texture_Snow =
  texture
  {
    pigment{rgb <0.5,0.5,1>} //<0.9,0.9,1.9>}
    finish{ambient 0.10 diffuse 0.85}
    normal
    {
      average 2
      normal_map{
        [dents 2 scale 1/35]
        [bozo 1 scale 4 turbulence 0.75]
      }
    }
  }

#declare Texture_NaveImageMap =
  texture
  {
    pigment
    {
      image_map 
      {
        png "nave.png"
      }
      translate <.5,.5,0>
    }
    finish{ ambient 8.2 }
  }


#declare Texture_WhiteMarble =
  texture
  {
    pigment{ White_Marble }                    
    scale .8
    finish{phong 1.0 phong_size 90 ambient 0.2}
  }

#declare Texture_Stone=
  texture{ pigment{ DimGray } }//finish{ ambient .3} }
  texture{ T_Grnt11a }//finish{ ambient .3} }  

#declare Texture_StoneWall=
  texture{ pigment{ DimGray } }//finish{ ambient .3} }
  texture
  { 
    T_Grnt8a 
    normal
    { 
      agate 
      scale<1,1,.25>
    } 
  }//finish{ ambient .3} }

#declare Texture_WindowFrame=
  texture
  { 
    T_Grnt1 
    normal
    { 
      granite
      scale<1,1,.15>
    } 
  }

#declare Interior_Stone=
  interior {}

#declare Texture_StoneCrypt=
  texture{ pigment{ Gray } }//finish{ ambient .3} }
  texture{ T_Grnt22a }//finish{ ambient .3} }  
   
#declare LightIronTexture=
    texture // a rusty iron
    { 
      pigment
      {   
        granite
        color_map 
        {
          [0.0 rgb <.0,.0,.0>]
          [0.45 rgb <.05,.05,.05>]
          [0.6 rgb <.1,.051,.051>]
          [0.6 rgb <.0,.0,.0>]
        }
      } 
      scale .3
      finish{ ambient 0.2 diffuse 0.6 } 
      normal { wrinkles 1 scale 0.5 }
    }

////////////////////////////////////////////////////////////
//
// --------------------------------------- Global Settings
//
////////////////////////////////////////////////////////////

global_settings 
{
  assumed_gamma 1.0
  ambient_light .1
  #if (PHOTONS)
    photons
    {
      #if(PHOTONS_LOAD)
        load_file "Basking.ph"
        #undef PHOTONS
        #declare PHOTONS = false; 
      #else
        // use non-adaptive search radius
    	  spacing 0.01
        gather 20, 100
        autostop 0
        //media 100
        jitter 0.9
        save_file "Basking.ph"
      #end
    } 
  #end // PHOTONS
}

////////////////////////////////////////////////////////////
//
// --------------------------------------- Cameras
//
////////////////////////////////////////////////////////////

#if (CAMERA = 0)// main camera
  camera 
  {
    location  <0.0, Camera_Height, 0.0>
    direction 1.5*z
    right     x*image_width/image_height
    look_at   <0.0,3.4,20.0>
  }
#end // CAMERA = 0


#if (CAMERA = 1)// top down
  camera 
  {
    location  <0.0, 45, 19.0>
    //location  <0.0, 45, 19.0>
    direction 1.5*z
    right     x*image_width/image_height
    look_at   <0.0, 0.0, 19.0>
  }
#end // CAMERA = 1

#if (CAMERA = 2)// lamp
  camera 
  {
    location  <-7.0, 2, 28.0>
    //location  <0.0, 45, 19.0>
    direction 1.5*z
    right     x*image_width/image_height
    look_at   <-8.0, 1.8, 32.0>
  }
#end // CAMERA = 2

#if (CAMERA = 3)// window sill
  camera 
  {
    location  <0.0, 4.5, 25.0>
    direction 1.5*z
    right     x*image_width/image_height
    look_at   <0.0,3.0,39.0>
  }
#end // CAMERA = 3

#if (CAMERA = 4)// Celtic Cross
  camera 
  {
    location  <4.0, 1, 14.0>
    direction 1.5*z
    right     x*image_width/image_height
    look_at   <4.0,.3,18.0>
  }
#end // CAMERA = 4

#if (CAMERA = 5)// Back Stage
  camera 
  {
    location  <4.0, 60, 75.0>
    direction 1.5*z
    right     x*image_width/image_height
    look_at   <4.0,.3,18.0>
  }
#end // CAMERA = 5

#if (SKY_SPHERE)
  sky_sphere 
  {
    pigment 
    {
      gradient y
      color_map 
      {
        [0.0 rgb <0.3,0.35,0.5>]
        [0.7 rgb <0.0,0.5,0.4>]
      }
    }
  }
#end // SKY_SPHERE

////////////////////////////////////////////////////////////
//
// --------------------------------------- Global Lighting
//
////////////////////////////////////////////////////////////

light_source 
{
  <0, 0, 0>
  color rgb <1.0, 1.0, 0.8>*.15 //1
  translate <150000,200000,-100000>

  area_light <0, 0, 0>, <10000,10000,-10000>, 50, 50
  adaptive 1
  jitter
  #if (PHOTONS)
    photons 
    {
      reflection off
      refraction off
    }
  #end
}

// the light rays
#if (WINDOW_LIGHT)
  light_source 
  {
    <0, 0, 0>
    color rgb <0.5, 0.35, 0.1>*2  // light's color
    translate <0.0, 26.5, 64.0>
    parallel
    spotlight
    #if (AREA_LIGHT)
      area_light <1, 0, 0>, <0,1,0>, 10, 10
      adaptive 1                    
      orient
    #else
      cylinder
    #end
    point_at <0, 0, 18>
    falloff 40
    radius 9
    #if (PHOTONS)
      photons 
      {
        reflection on
        refraction on
        #if (AREA_LIGHT)
          //area_light This breaks it for some reason
        #end
      }
    #end
  }
#end // WINDOW_LIGHT

#if (WORKLIGHT)
  light_source {
    <0, 0, 0>            // light's position (translated below)
    color rgb <1, 1, 1>  // light's color
    translate <-30, 30, -10>
  #if (PHOTONS)
    photons {
      reflection off
      refraction off
    }
  #end
  } 
#end // WORKLIGHT

////////////////////////////////////////////////////////////
//
// --------------------------------------- Macros
//
////////////////////////////////////////////////////////////

#macro HeapSnowOn( Target, RandomSeed, Amount, Size, Height, Width, Length )
  #if (PILE_ON_SNOW)
    #local SnowRandom = seed(RandomSeed);
    #local iFlakes = 0;
    
    #while( iFlakes < Amount )
      #local HitNormal = <0,0,0>;
      #local xStart = Width*rand(SnowRandom) - (Width/2); 
      #local zStart = Length*rand(SnowRandom) - (Length/2);
      #local Intercept = trace( Target, <xStart, Height, zStart>, <0,-Height,0>, HitNormal );
      #local Transition = <xStart, Intercept.y, zStart>;
      #if ( vlength(HitNormal) != 0 )
        sphere
        {
          0,Size,1
          scale<1,0.5,1>
          transform{Reorient_Trans(y, HitNormal)}
          translate Transition
        }
        #local iFlakes = iFlakes + 1;
      #end
    #end // while
  #else // PILE_ON_SNOW
    sphere
    {
      0,.0001,1
    }
  #end // PILE_ON_SNOW
#end // HeapSnowOn

#macro BlanketSnowOn( Target, Size, Height, Width, Length )
  #if (PILE_ON_SNOW)
    #local iFlakes = 0;
    #local iDebugCount = 0;
    
    #local zLoc = 0;
    #local increment = Size / 2;
    
    #debug concat( "Snowing . . . \n" )
    
    #while( zLoc < Length )
      #local xLoc = 0;
      #while( xLoc < Width )
        #local HitNormal = <0,0,0>;
        #local xStart = xLoc - (Width/2);
        #local zStart = zLoc - (Length/2);
        #local Intercept = trace( Target, <xStart, Height, zStart>, <0,-Height,0>, HitNormal );
        #local Transition = <xStart, Intercept.y, zStart>;
        #if ( vlength(HitNormal) != 0 )
          sphere
          {
            0,Size,1
            scale<1,0.5,1>
            transform{Reorient_Trans(y, HitNormal)}
            translate Transition
          }
          #local iFlakes = iFlakes + 1;
          //#debug concat( "." )
          //#local iDebugCount = iDebugCount + 1;
          #if (iDebugCount = 100)
            #local iDebugCount = 0;
            #debug concat("\n")
          #end
        #end
        #local xLoc = xLoc + increment;
      #end // xLoc while
      #local zLoc = zLoc + increment;
    #end // zLoc while
  #else // PILE_ON_SNOW
    #local iFlakes = 0;
  #end // PILE_ON_SNOW
  #if( iFlakes = 0 )
    sphere
    {
      0,.0001,1
    }
  #end
  #debug concat( " Snowflake Count = ",str(iFlakes,7,2), "\n" )
#end // BlanketSnowOn

////////////////////////////////////////////////////////////
//
// --------------------------------------- The Ground
//
////////////////////////////////////////////////////////////

#if (GROUND)

  #declare SnowField =
  height_field 
  {
    png
    "snow.png"
    smooth
    translate <-0.5,-0.5,-0.5>
    scale <27,1.7,27>
    translate <0,0,19>
    #if (false & PHOTONS)
      photons
      {
        target
        reflection on
        refraction on
      }
    #end
      
  }

#else // quick ground

  #local SnowField =
  plane 
  {
    y, -1
  }

#end // GROUND             


////////////////////////////////////////////////////////////
//
// --------------------------------------- The Church Wall
//
////////////////////////////////////////////////////////////

#if (CHURCH_WALL)
  #if (BRICK_WALL)    
    #ifndef (MakeStoneWall)
      #include "stonewall.inc"
    #end
    
    MakeStoneWall(<.5,0.25,.5>, <.015,.015,.015>, <.015,.015,.015>, <15,16,1>, Texture_StoneWall, Interior_Stone, .05, .5, 2, ISO_CHURCH_WALL, false, .25)
  
    #local ChurchBricks =
      object
      {
        StoneWall
        scale <4,4,1>
      }
  #else  // put in a false wall
  
    #local ChurchBricks =
      box
      {
        <-15,-8,-.25>,<15,8,.25>
        pigment{ DimGray }
      }  
    
  #end // BRICK_WALL
#end // CHURCH_WALL  

#if (CHURCH_WALL & WINDOW_CUTOUT)

  #local Arch =
    union
    {
      difference
      {
        cylinder
        {
          <0,0,-.26>,<0,0,0.26>, 4.12
        }
        box{ <-4.12,4.12,-.26>,<2.06,-4.12,.26> }
        box{ <-4.12,-4.12,-.26>,<4.12,0,.26> }
        texture{ Texture_StoneWall }
        translate <-2.06,0,0>
      }
      difference
      {
        cylinder
        {
          <0,0,-.26>,<0,0,0.26>, 4.12
        }
        box{ <-4.12,4.12,-.26>,<2.06,-4.12,.26> }
        box{ <-4.12,-4.12,-.26>,<4.12,0,.26> }
        texture{ Texture_StoneWall }
        rotate< 0, 180, 0 >
        translate <2.06,0,0>
      }
      scale< 1, .75, 1 >
    }


  #local WindowCutout =
    union
    {
      box{ < -2.06,-4.08,-.26 >,< 2.06,4.08,.26 >}
      object{ Arch translate <0, 4.07, 0> }
      texture{ Texture_StoneWall }
    } 
  

  #local ChurchWall =
    union
    { 
      difference
      {                    
        object{ ChurchBricks }
        object{ WindowCutout translate <0,-0.1,0> }
      } // difference
      #if (WINDOW_FRAME)
      
        #local ArchFrameTop =
          difference
          {
            box{ <-2.06,-1.544,-.25>,<2.06,1.545,.25> translate<0,1.545,0> }
            object{ Arch }
            scale 1.15
          }
        //object{ ArchFrameTop } 
        #local ArchFrame =
          difference
          {
            box{ <-2.31,-1.544,-.26>,<2.31,1.545,.26> translate<0,1.545,0> }
            object{ ArchFrameTop }
            object{ Arch scale<1,1,1.1>}            
          }
        
        #local TotalHeight = 8.16;
        #local Count = 20;
        #local WidthType = 1;
        #local Height = TotalHeight/Count;
        #local xStart = 2.06;
        #local yStart = -TotalHeight/2;
        #local Width1 = .25;
        #local Width2 = Width1/2;
        #local Thickness = .6;
        #local MorterWidth = .025;
        
        #local Frame1=
        union
        {
          #while (Count > 0)
            #local NowWidth = Width1;
            #if (WidthType = 2)
              #local NowWidth = Width2;
            #end
            box
            {
              <xStart,yStart,-(Thickness/2)>,
              <xStart + NowWidth,yStart + Height - MorterWidth,Thickness/2>
            }
            #local WidthType = 3 - WidthType;
            #local yStart = yStart + Height;
            #local Count = Count - 1;
          #end // while
        } // union
        
        #local Frame2=
        object 
        {
          Frame1
          rotate<0,180,0>
        }
        
        union
        {
          object{ ArchFrame translate <0, 4.07, 0> }
          object{ Frame1 }
          object{ Frame2 }
          texture{ Texture_WindowFrame }
        }
        
      #end // WINDOW_FRAM
      translate <0,0.80,0>
    } // union
//    #local ChurchWallSnow =
//    blob
//    {
//      HeapSnowOn( ChurchWall, 3255, 200, 8, 4.12, .26 )
//      texture{ Texture_Snow }
//    }

#else
  #if (CHURCH_WALL)  
    #local ChurchWall = 
      object{ ChurchBricks }
  #end
#end // CURCH_WALL && WINDOW_CUTOUT

#if (CHURCH_WALL)
    
  #local WindowSill1 =
    box
    {
      <-2.09,-3.4,-.27>,<2.09,-3.3,.27>
      texture{ Texture_StoneWall }
    }
  #local WindowSillSnow11 =
    blob
    {
      //HeapSnowOn( WindowSill1, 4567, 400, 0.14, 1, 4.19, .54 )
      BlanketSnowOn( WindowSill1, 0.2, 1, 4.19, .54 )
      texture{ Texture_Snow } 
    }
  
  #local WindowSillSnow12 =
    blob
    {
      //HeapSnowOn( WindowSillSnow11, 543, 400, 0.14, 1, 4.19, .54 )
      BlanketSnowOn( WindowSillSnow11, 0.15, 1, 4.19, .54 )
      texture{ Texture_Snow } 
    }
  
  #local WindowSillSnow13 =
    blob
    {
      //HeapSnowOn( WindowSillSnow12, 6772, 400, 0.14, 1, 4.19, .54 )
      BlanketSnowOn( WindowSillSnow12, 0.1, 1, 4.19, .54 )
      texture{ Texture_Snow } 
    }
    
  #if (ICICLES)
    // on window sill
    MakeIcicle( 2, 2, .3 )
    #local Icicle1=
      object
      {
        Icicle
        translate <-1,-4,-.3>
      }
    MakeIcicle( 1.1, 1.5, .25 )
    #local Icicle2=
      object
      {
        Icicle
        translate <.3,-3.57,-.3>
      }
    MakeIcicle( 1.75, 1.75, .25 )
    #local Icicle3=
      object
      {
        Icicle
        translate <.90,-3.78,-.3>
      }
  #end // ICICLES
  
  #local FullWall =
    union
    {
      object{ ChurchWall }
      #ifdef (ChurchWallSnow)
        object { ChurchWallSnow }
      #end
      union
      {
        object{ WindowSill1 }
        object{ WindowSillSnow11 }
        object{ WindowSillSnow12 }
        object{ WindowSillSnow13 }
        #if (ICICLES)  
          object{ Icicle1 }
          object{ Icicle2 }
          object{ Icicle3 }
        #end // ICICLES
      }
    }
#end // CHURCH_WALL

////////////////////////////////////////////////////////////
//
// --------------------------------------- The Backdrop
//
////////////////////////////////////////////////////////////

#if (BACKDROP)  

#declare Backdrop=
  object
  {
    box
    {
      <-.5,-.5,-.001>, <.5,.5,.001>
      //texture{ Texture_NaveImageMap }
      pigment { White }
      finish{ ambient 10 }
      no_shadow
      #if (PHOTONS)
        photons 
        {
          pass_through
        }
      #end
    }
  }

#end // BACKDROP

////////////////////////////////////////////////////////////
//
// --------------------------------------- The Window
//
////////////////////////////////////////////////////////////

#if (WINDOW)

  #declare iNumberOfPanes = 25;

  #declare asMasks=array[25] 
  {
   "WindowLead.png", "WindowLightRed2.png", "WindowGray.png", "WindowCatPurple.png", "WindowGrassYellow.png", 
   "WindowYellow2.png", "WindowLightGreen.png", "WindowLightGreen2.png", "WindowGreen.png", "WindowDarkGreen.png", 
   "WindowRed.png", "WindowRed2.png", "WindowLightRed.png", "WindowDarkGreen2.png", "WindowBrown.png", 
   "WindowBrown2.png", "WindowDarkBrown.png", "WindowAqua.png", "WindowAqua2.png", "WindowLightAqua.png", 
   "WindowGrassBrown.png", "WindowGrassGreen.png", "WindowLightAqua2.png", "WindowDarkYellow.png", "WindowYellow1.png"
  }
  #declare argbfNoFadeColors=array[25] 
  {
   rgbf<0.8, 0.8, 0.2, 1>, rgbf<0.4980, 0.0824, 0.2275, 1>, rgbf<0.8, 0.9, 0.85, 1>, rgbf<0.0627, 0.2980, 0.2118, 1>, rgbf<0.5020, 0.4118, 0.3529, 1>, 
   rgbf<0.75, 0.78, 0.10, 1>, rgbf<0.35, 0.85, 0.10, 1>, rgbf<0.20, 0.75, 0.25, 1>, rgbf<0.8, 1, 0.95, 1>, rgbf<0.1, 0.7, 1>, 
   rgbf<0.85, 0.10, 0.20, 1>, rgbf<0.85, 0.35, 0.10, 1>, rgbf<0.50, 0.10, 0.10, 1>, rgbf<0.10, 0.50, 0.10, 1>, rgbf<0.3725, 0.4235, 0.2941, 1>, 
   rgbf<0.2627, 0.2549, 0.1451, 1>, rgbf<0.3875, 0.3875, 0.15859375, 1>, rgbf<0.5333, 0.6157, 0.6196, 1>, rgbf<0.7412, 0.8235, 0.8431, 1>, rgbf<0.7255, 0.8510, 0.9176, 1>, 
   rgbf<0.0627, 0.2980, 0.2118, 1>, rgbf<0.10, 0.85, 0.20, 1>, rgbf<0.7255, 0.8510, 0.9176, 1>, rgbf<0.8875, 0.78125, 0.0615, 1>, rgbf<0.78, 0.75, 0.10, 1>
  }
  #declare argbFadeColors=array[25] 
  {
   Col_Glass_Yellow, Col_Ruby_01, Col_Glass_Old, Col_Tourmaline_05, Col_Citrine_01, 
   Col_Yellow_02, Col_Green_02, Col_Green_03, Col_Glass_Green, Col_Glass_Dark_Green,
   Col_Red_01, Col_Red_02, Col_Red_04, Col_Green_04, Col_Tourmaline_01,
   Col_Tourmaline_03, rgb<0.3875, 0.3875, 0.15859375>, Col_Aquamarine_01, Col_Aquamarine_03, Col_Aquamarine_06,
   Col_Tourmaline_05, Col_Green_01, Col_Aquamarine_06, rgb<0.4875, 0.39125, 0.0315>, Col_Yellow_03  
  }
  #declare i = 0;

  #declare Stained_Glass_Window =
  union 
  {
    #while (i < iNumberOfPanes)
      #ifdef (fn_SDents)
        #undef fn_SDents
      #end
      #declare fn_SDents=function {
          pigment {
            dents  
            warp{ turbulence <0.1,0.1,0> }
            translate <2*i,4*i,0>
            scale <0.03,0.03,.01>
          }
        }
      #ifdef (fn_Picmap)
        #undef fn_Picmap
      #end
      #declare fn_Picmap=function {
          pigment {
            #if (i=0)
              image_map {png asMasks[i] interpolate 4}     
            #else
              image_map {png asMasks[i] }     
            #end
            translate<0.5,0.5,0>
          }
        }
      #if ( (NO_GLASS = false) | (i = 0) )  
        object
        {
          isosurface 
          {
            #if (i=0) // first
              function { y +(fn_Picmap(x/2,z/2,0).gray*.0065)}
            #else
              #if (SIMPLE_WINDOW)
                function { y +(fn_Picmap(x/2,z/2,0).gray*.008) }
              #else
                function { y +(fn_Picmap(x/2,z/2,0).gray*.0080)-(fn_SDents(x,z,0).gray*.003) }
              #end
            #end
            contained_by { box { <-1,-.005,-1>, <1,.005,1> } }  
            rotate<-90,0,0>
            #if (PHOTONS)
            photons 
            {
              target
              refraction on
              reflection off
              //collect off
            }                
            #end
          }  
          //pigment{ Gray }
          //finish{ IsoFinish }
          #if (i=0) // first
            //texture{ pigment{ Black } }
            texture { 
               pigment { DimGray }
              scale .6   
              translate .5
              //finish { phong 0 }
            }      
          #else
            texture { pigment{ color argbfNoFadeColors[i] } }
            interior {
              ior 1.5
              //fade_distance 5//1.0
              //fade_power .1//2
              fade_color argbFadeColors[i]
            }
            finish
            {
              specular 0.7
              roughness 0.001
              diffuse 0
              reflection
              {
                0.2, 1.0
                fresnel on
              }
              conserve_energy
              ambient 1
            }
            #if (PHOTONS)
              photons 
              {
              	target 1.0
                reflection off
                refraction on
                collect off
              }
            #end // PHOTONS
          #end // not first
        }
      #end // NO_GLASS
      #local i = i + 1;
    #end // while
  } // union

#end // WINDOW

////////////////////////////////////////////////////////////
//
// --------------------------------------- The Flying Buttresses
//
////////////////////////////////////////////////////////////

#if (BUTTRESSES)

  #macro MakeButtress( ISOBRICKS )
    #if (BRICK_BUTTRESSES)  
      #ifndef (MakeStoneWall)
        #include "stonewall.inc"
      #end
      
      MakeStoneWall(<1,1,1>, <.040,.040,.040>, <.040,.040,.040>, <1,10,1>, Texture_StoneWall, Interior_Stone, .2, .5, 2, ISOBRICKS, false, 0)  
      #local ButtressTowerBricks =
        difference
        {
          object
          {
            StoneWall
            //scale <4,4,4>
          }
          box
          {
            <-.75,-.75,-.75>,<.75,.75,.75>
            texture{ Texture_StoneWall } 
            rotate <0,0,45> 
            translate <.48,5.5,0>
          }
        }
        
      MakeStoneWall(<.6,2,.6>, <.040,.040,.040>, <.040,.040,.040>, <1,5,1>, Texture_StoneWall, Interior_Stone, .2, .5, 2, ISOBRICKS, false, 0)  
      #local ButtressTowerBricksA =
        difference
        {
          object
          {
            StoneWall
            //scale <4,4,4>
          }
          box
          { 
            <-.5,-.5,-.5>,<.5,.5,.5>
            texture{ Texture_StoneWall } 
            rotate <0,0,45> 
            translate <.3,5.0,0>
          }
        }
      
      MakeStoneWall(<1,1,1>, <.040,.040,.040>, <.040,.040,.040>, <1,13,1>, Texture_StoneWall, Interior_Stone, .2, .5, 2, ISOBRICKS, false, 0)  
      #local ButtressTowerBricksB =
        union
        {
          difference
          {
            object
            {
              StoneWall
              //scale <4,4,4>
            }
            box
            {
              <-.75,-.75,-.75>,<.75,.75,.75>
              texture{ Texture_StoneWall } 
              rotate <0,0,45> 
              translate <.48,7.0,0>
            }
          }
          #if (LAMPS)
            #include "Lamp.inc"
            BrassLamp( true, LAMPS_LIT )
            object
            {
              BLamp
              scale .20
              translate <.75,-4.0,0>
            }
          #end
        }
        
      #local ButtressWing=
        difference
        {
          box
          {
            <-1.75,-.55,-.15>,<1.75,3,.15>
            texture{ Texture_StoneWall }
          }
          box
          {
            <-2.25,-1,-.26>,<2.25,1,.26>
            texture{ Texture_StoneWall }
            rotate <0,0,-22.5>
            translate <.55,2.75,0>
          }
          cylinder
          {
            <0,0,-.26>,<0,0,.26>,
            2
            texture{ Texture_StoneWall }
            scale <1,.45,1>
            rotate <0,0,-22.75>
            translate <-.1,.2,0>
          }
          box
          {
            <-2.25,-1,-.26>,<2.25,1,.26>
            texture{ Texture_StoneWall }
            rotate <0,0,-22.5>
            translate <0,-1.2,0>
          }
        }
        
    #end // BRICK_BUTTRESSES
    
    #declare Buttress=  
      union
      {
        object{ ButtressTowerBricks }
        object
        {
          ButtressTowerBricksB 
          translate <-4.5,1,0>
        }

        object
        {
          ButtressTowerBricksA
          translate <(.075+.125)*4,-1,0>
        }
        object
        {
          ButtressWing
          translate <-2.25,3,0>
        }
        #local Wing1Snow1 =
          blob
          {
            HeapSnowOn( ButtressWing, 25632, 800, 0.1, 4, 6, 1 )
            texture{ Texture_Snow } 
          }
          
    /*      #local Wing1Snow2 =
          blob
          {
            HeapSnowOn( ButtressWing, 4328, 800, 0.1, 2, 6, 1 )
            texture{ Texture_Snow } 
          }
    */
          #local LowerWing=
          union
          {
            object{ ButtressWing }
            object{ Wing1Snow1 }
            translate <-2.25,-1,0>
          }
        object{ LowerWing }                 
        //box{ <-3.5,-.1,-.5>, <3.5,.1,.5> pigment { DimGray } translate <-2,7.0,0> }
        //box{ <-1.5,-.1,-.5>, <1.5,.1,.5> pigment { DimGray } translate <-2,2.0,0> }
        //translate <0,-3,0>
      }
  #end // MakeButtress

  MakeButtress( false )
  
  #local Buttress1Snow1 =
    blob
    {
      //HeapSnowOn( Buttress, 728, 800, 0.1, 10, 10, 1 )
      BlanketSnowOn( Buttress, 0.25, 10, 10, 1 )
      texture{ Texture_Snow } 
    }
  #local Buttress1Snow2 =
    blob
    {
      //HeapSnowOn( Buttress1Snow1, 654, 800, 0.15, 10, 10, 1 )
      BlanketSnowOn( Buttress1Snow1, 0.20, 10, 10, 1 )
      texture{ Texture_Snow } 
    }
  #local Buttress1Snow3 =
    blob
    {
      //HeapSnowOn( Buttress1Snow2, 9087, 800, 0.15, 10, 10, 1 )
      BlanketSnowOn( Buttress1Snow2, 0.1, 10, 10, 2 )
      texture{ Texture_Snow } 
    }
  #local Buttress2Snow1 =
    blob
    {
      //HeapSnowOn( Buttress, 8765, 800, 0.15, 10, 10, 1 )
      BlanketSnowOn( Buttress, 0.25, 10, 10, 1 )
      texture{ Texture_Snow } 
    }
  #local Buttress2Snow2 =
    blob
    {
      //HeapSnowOn( Buttress2Snow1, 987, 800, 0.15, 10, 10, 1 )
      BlanketSnowOn( Buttress2Snow1, 0.20, 10, 10, 1 )
      texture{ Texture_Snow } 
    }
  #local Buttress2Snow3 =
    blob
    {
      //HeapSnowOn( Buttress2Snow2, 11223, 800, 0.15, 10, 10, 1 )
      BlanketSnowOn( Buttress2Snow2, 0.15, 10, 10, 2 )
      texture{ Texture_Snow } 
    }
    
  #if (ISO_CHURCH_WALL)
    #undef Buttress
    MakeButtress( true )
  #end 
        
  #local Buttress1=
    union
    {
      object{ Buttress }
      //object{ Wing1Snow1 }
      object{ Buttress1Snow1}
      object{ Buttress1Snow2}// translate<0,-.02,0>}
      object{ Buttress1Snow3}// translate<0,-.04,0>}
      #if (ICICLES)
        MakeIcicle( 1.4, 1.25, .20 )
        object
        {
          Icicle
          scale<.75,1,1>
          translate <((.075+.125)*4)+.37,3.3,-.2>
        }
        MakeIcicle( .45, .800, .15 )
        object
        {
          Icicle
          scale<.75,1,1>
          translate <((.075+.125)*4)+.37,3.6,.13>
        }
      #end // ICICLES
    }
  #local Buttress2=
    union
    {
      object{ Buttress }
      //object{ Wing1Snow2 }
      object{ Buttress2Snow1}
      object{ Buttress2Snow2}// translate<0,-.02,0>}
      object{ Buttress2Snow3}// translate<0,-.04,0>}
      #if (ICICLES)
        MakeIcicle( .35, 1.3, .20 )
        object
        {
          Icicle
          scale<.75,1,1>
          translate <((.075+.125)*4)+.37,3.2,.13>
        }
        MakeIcicle( 1.3, .9900, .20 )
        object
        {
          Icicle
          scale<.75,1,1>
          translate <((.075+.125)*4)+.37,3.5,-.2>
        }
      #end // ICICLES
    }
#end // BUTTRESSES

////////////////////////////////////////////////////////////
//
// --------------------------------------- The Bench(es)
//
//////////////////////////////////////////////////////////// 

#if (BENCH)

  #if (CAT_BENCH_PRINTS)
    #declare CatPrints =
    height_field 
    {
      png
      "catbench.png"
      smooth
      translate <-0.5,-0.5,-0.5>
      scale <1,1.7,.70>
      rotate<0,0,180>
      texture{ Texture_Snow }
    }
  #end // CAT_BENCH_PRINTS   
  
  
  #ifndef (MakeStoneWall)
    #include "stonewall.inc"
  #end

  #include "Bench.inc"
  
  #local Bench1 =
    object
    {
      Bench( BenchTexture )
    }
  #local Bench11Snow =
    blob
    {
      HeapSnowOn( Bench1, 341, 400, 0.1, 2, 1.45, .5 )
      texture{ Texture_Snow } 
    }
  #local Bench12Snow =
    blob
    {
      HeapSnowOn( Bench11Snow, 2352, 400, 0.1, 2, 1.24, .5 )
      texture{ Texture_Snow } 
    }
  #local Bench13Snow =
    blob
    {
      HeapSnowOn( Bench12Snow, 685, 400, 0.1, 2, 1.45, .5 )
      texture{ Texture_Snow } 
    }
  
  #declare BenchOne=  
    union
    {
      object{ Bench1 }
      object{ Bench11Snow }
      object{ Bench12Snow translate <0,-.01,0> }
      object{ Bench13Snow translate <0,-.02,0> }
    }
  
  #local Bench2 =
    object
    {
      Bench( BenchTexture )
    }

  #local Bench21Snow =
    blob
    {
      //HeapSnowOn( Bench2, 2345, 400, 0.1, 2, 1.45, .5 )
      BlanketSnowOn( Bench2, 0.2, 2, 1.45, .5 )
      texture{ Texture_Snow } 
    }
  #local Bench22Snow =
    blob
    {
      //HeapSnowOn( Bench21Snow, 423, 400, 0.1, 2, 1.24, .5 )
      BlanketSnowOn( Bench21Snow, 0.15, 2, 1.45, .5 )
      texture{ Texture_Snow } 
    }
  #local Bench23Snow =
    blob
    {
      //HeapSnowOn( Bench22Snow, 5233, 400, 0.1, 2, 1.45, .5 )
      BlanketSnowOn( Bench22Snow, 0.16, 2, 1.45, .5 )
      texture{ Texture_Snow } 
    }
    
  #declare BenchTwo=  
    union
    {
      object{ Bench2 }
      object{ Bench21Snow }
      #if (CAT_BENCH_PRINTS)
      difference
      {
        union
        {
      #end // CAT_BENCH_PRINTS
          object{ Bench22Snow translate <0,-.01,0> }
          object{ Bench23Snow translate <0,-.02,0> }
      #if (CAT_BENCH_PRINTS)
        }
        object
        {
          CatPrints
          translate<.75,.48,0>
        }
      }
      #end // CAT_BENCH_PRINTS
    #if (false & PHOTONS)
      photons
      {
        target
        reflection on
        refraction on
      }
    #end
    }

#end // BENCH


////////////////////////////////////////////////////////////
//
// --------------------------------------- The Crypt
//
////////////////////////////////////////////////////////////
  

#if (CRYPT)

  #declare Lids = array[3]{ "CryptTopper1.png", "CryptTopper2.png", "CryptTopper2.png" }
  #declare Sides = array[3]{ "CryptSide1a.png", "CryptSide2.png", "CryptSide3.png" }
  #declare Caps = array[2]{ "CryptEndCap1.png", "CryptEndCap2.png" }
  
  #macro CryptLid( LidNumber, Texture )
    #undef fn_CryptPicmap
    #declare fn_CryptPicmap=function {
        pigment {
          image_map {png Lids[LidNumber] interpolate 4}     
          translate<0.5,0.5,0>
          scale <.75,.25,1>
        }
      }  
    object
    {
      isosurface {
        function { y +(fn_CryptPicmap(x/2,z/2,0).gray*.01)}
        contained_by { box { <-0.775,-.005,-.25>, <(1.55/2),.005,.25> } }
        all_intersections    
        //rotate<180,0,0>
        //scale <1,10,1>
        //translate<0,.058,0>
        translate<0,.063,0>
      }  
      texture { Texture }      
    }
  #end
  
  #macro CryptSide( SideNumber, Texture )
    #ifdef (fn_CryptPicmap)
      #undef fn_CryptPicmap
    #end
    #declare fn_CryptPicmap=function {
        pigment {
          image_map {png Sides[SideNumber] interpolate 4}     
          translate <0.5,0.5,0>
          scale <.775,.26,1>
        }
      }  
    object
    {
      isosurface {
        // a rather more complicated series of objects and transitions
        // boiled down to this. It can probably be reduced further
        // but I'm not going to touch it anymore.
        function { y +(fn_CryptPicmap(x/2,z/2,0).gray*.01)}
        contained_by { box { <-0.775,-.005,-.25>, <(1.55/2),.005,.25> } }  
        all_intersections  
        scale<1,8,1>
        rotate<180,0,0>
        rotate <90,0,0>
        translate <0,.02,-.278>
      }  
      texture { Texture }      
    }
  #end
  
  #macro CryptEndCap( CapNumber, Texture )
    #undef fn_CryptPicmap
    #declare fn_CryptPicmap=function {
        pigment {
          image_map {png Caps[CapNumber] interpolate 4}     
          translate <0.5,0.5,0>
          scale <.23,.23,1>
        }
      }  
    object
    {
      isosurface {
        function { y +(fn_CryptPicmap(x/2,z/2,0).gray*.01)}
        contained_by { box { <-.25,-.005,-.25>, <.25,.005,.25> } }
        all_intersections  
        scale<1,8,1>
        rotate<0,0,90>
        translate <0.76,.02,0>
      }  
      texture { Texture }      
    }
  #end
  
  #declare Finial=
    isosurface

    {
      function { f_quartic_cylinder(x,y,z, -.1, 0.8, 0.4) }
      contained_by { box { <-0.85, -2.5, -0.85>, <0.85, 2.5, 0.85> } }
      max_gradient 8.8
      scale .05
      texture { T_Silver_1E }
    }
  
  #macro Lid( Number, LidTexture, DecorationTexture, Finials )
    union
    {
      MakeStoneWall( <1.55,.07,.5>, <.05,.05,.05>, <0,0,0>, <1,1,1>, LidTexture, Interior_Stone, 0, .025, 0, true, true, 0 )
      object{ StoneWall }
      object
      {     
        CryptLid( Number, DecorationTexture )
      }
      #if (Finials)
        object{ Finial translate <.750,.11,.255> }
        object{ Finial translate <.750,.11,-.245> }
        object{ Finial translate <-.750,.11,.255> }
        object{ Finial translate <-.750,.11,-.245> }
      #end
  
      //object{ Finial translate <0,.1,0> }
         
      translate<0,.37,0> 
    }
  #end
/*  
    // center
    MakeStoneWall( <1.55,.55,.5>, <.05,.05,.05>, <.2,.2,.2>, <1,1,1>, T_Stone7, StoneWallInterior, 0, .02, true, true )
    union
    {
      difference
      {
        object{ StoneWall }
        object
        {    
          CryptSide( 2, T_Gold_1A )//T_Gold_1A )
        }
        object
        {    
          CryptEndCap( 0, T_Gold_1A )
        }
      }
      object
      {
        Lid( 2, T_Stone7, T_Silver_1E, true )
      }
      translate < 0, -2.3, -2.36 >
      translate <0,-.44,0>
    }
      
    // left side
    //MakeStoneWall( <1.55,.55,.5>, <.05,.05,.05>, <.2,.2,.2>, <1,1,1>, Texture_Stone, StoneWallInterior, 0, .02, true, true )
    union
    {
      object
      {
        Lid( 1, TestMarble, DecrotaveMetal, false )
      }
      MakeBoxFence( 1.8, .9, .15, <.02,.75,.02>, 2, texture{ LightIronTexture scale<0,2,0> }, 0 )
      object{ BoxFence translate<0,0.35,0> }
      translate < -4, -2.4, -2.35 >
      //translate <0,-.44,0>
      translate <0,-.97,0>
    }
*/  
    // right side
    MakeStoneWall( <1.55,.55,.5>, <.05,.05,.05>, <.2,.2,.2>, <1,1,1>, Texture_Stone, Interior_Stone, 0, .02, 0, true, true, 0 )
    #local Crypt1=
    //union
    //{
      difference
      {
        object{ StoneWall }
        object
        {    
          CryptSide( 1, Texture_StoneCrypt )//T_Gold_1A )
        }
        object
        {
          CryptEndCap( 0, Texture_StoneCrypt )
          // this is on the right side, we need it on the left    
          translate <-1.585,0,0>
        }
        object
        {     
          object{ CryptLid( 0, Texture_StoneCrypt ) }
          scale<1,6,1>
          translate <0,-.065,0>
        }
        scale 1.2
      }
    //}
    
  #local Crypt11Snow =
    blob
    {
      //HeapSnowOn( Crypt1, 5343, 400, 0.1, 1, 1.56, .6 )
      BlanketSnowOn( Crypt1, 0.15, 1, 2, .6 )
      texture{ Texture_Snow } 
    }
  #local Crypt12Snow =
    blob
    {
      //HeapSnowOn( Crypt11Snow, 6275, 400, 0.1, 1, 1.56, .6 )
      BlanketSnowOn( Crypt11Snow, 0.2, 1, 2, .6 )
      texture{ Texture_Snow } 
    }
  #local Crypt13Snow =
    blob
    {
      //HeapSnowOn( Crypt12Snow, 444, 400, 0.1, 1, 1.56, .6 )
      BlanketSnowOn( Crypt12Snow, 0.15, 1, 2, .6 )
      texture{ Texture_Snow } 
    }
  
  #declare CryptOne=  
    union
    {
      object{ Crypt1 }
      object{ Crypt11Snow }
      object{ Crypt12Snow }//translate <0,-.005,0> }
      object{ Crypt13Snow }//translate <0,-.01,0> }
    }

#end // CRYPT

////////////////////////////////////////////////////////////
//
// --------------------------------------- The Celtic Cross
//
////////////////////////////////////////////////////////////


#if (CROSS)  
  #include "CelticCross.inc"
  #local CCRossSized=
    object
    {
      CCross
      scale .8
    }
/*
  #local Cross11Snow =
    blob
    {
      HeapSnowOn( CCRossSized, 624, 400, 0.025, 1, 1.2, .6 )
      texture{ Texture_Snow } 
    }
  #local Cross12Snow =
    blob
    {
      HeapSnowOn( Cross11Snow, 6848, 400, 0.025, 1, 1.2, .6 )
      texture{ Texture_Snow } 
    }
  #local Cross13Snow =
    blob
    {
      HeapSnowOn( Cross12Snow, 5431, 400, 0.025, 1, 1.2, .6 )
      texture{ Texture_Snow } 
    }
*/  
  #local Cross11Snow =
    blob
    {
      BlanketSnowOn( CCRossSized, 0.1, 1, 1.2, .6 )
      texture{ Texture_Snow } 
    }
  #local Cross12Snow =
    blob
    {
      BlanketSnowOn( Cross11Snow, 0.09, 1, 1.2, .6 )
      texture{ Texture_Snow } 
    }
  #local Cross13Snow =
    blob
    {
      BlanketSnowOn( Cross12Snow, 0.08, 1, 1.2, .6 )
      texture{ Texture_Snow } 
    }
  #declare CrossOne=  
    union
    {
      object{ CCRossSized }
      object{ Cross11Snow }
      object{ Cross12Snow }//translate <0,-.005,0> }
      object{ Cross13Snow }//translate <0,-.01,0> }
    }  
#end // cROSS


////////////////////////////////////////////////////////////
//
// --------------------------------------- The Icicles
//
////////////////////////////////////////////////////////////

#if (ICICLES)
#end // ICICLES

////////////////////////////////////////////////////////////
//                                                 
// --------------------------------------- The Tree(s)
//
////////////////////////////////////////////////////////////

#if (TREE | MAIN_TREE)
  
  // Include the leaf definition and splinetree files
  #include "leafdefs.inc"
  #include "splinetree.inc"
#end
#if (TREE)  
  //#declare recursionmax = 6; // Defines the complexity of the tree
  //#declare divisions = array[6] {7,3,3,2,2,4}
  #declare recursionmax = 4; // Defines the complexity of the tree
  #declare divisions = array[4] {7,3,3,2}//,2,4}
  init_rand (757,4622,8976)     // Supply seeds to initialize the random variable
  
  // Variables modeled exponentially
  // Use: init_variable (start_size, end_size, base)
  init_tdev (0.5, 0.8, 0.9)
  init_rdev (70, 40, 1)
  init_twigprob (0, 0.7, 0.5)
  init_branchl (3.4, 0.2, 0.6)
  init_branchw (0.35, 0.004, 0.4)
  
  // Odd variables
  init_roots (2)			// Use: init_roots (rootrecursionmax)
  init_splinedev (0.15,0.3)	// Use: init_splinedev (min_dev, max_dev)
  init_simplify (2,2)		// Use: init_simplify (num_simplify_levels, num_segments)
  
  // Add some gravity
  #declare tendancy = <-0.05,0.2,0.05>;
  
  // Assign the branch texture
  #declare branchtexture = texture { standard_branch } // standard_branch is defined in leafdefs.inc
  
  // Define transformations assigned to the whole tree here.
  #declare global_transform = transform {
  	scale .75
  }
  
  // Provide feedback in the first 2 recursion levels
  #declare feedback = 2;
  
  // Call the macro
	#declare Tree=
	  union
	  {
      splinetree(0,0)
    }
  #debug concat( "\nTree done.\n")
  #local TreeSnow1=
    blob
    {
      BlanketSnowOn( Tree, .1, 7, 5, 5 )
      //HeapSnowOn( Tree, 5235, 800, .01, 10, 10, 10 )
      texture{ Texture_Snow }
    }

  #declare Tree1=
    union
    {
      object { Tree }
      object { TreeSnow1 }
    }
  
  init_rand (56464,433,24532)     // Supply seeds to initialize the random variable
  #declare global_transform = transform {
  	scale .73
  }
	#declare Tree=
	  union
	  {
      splinetree(0,0)
    }
  #debug concat( "\nTree done.\n")  
  #local TreeSnow2=
    blob
    {
      BlanketSnowOn( Tree, .1, 7, 5, 5 )
      //HeapSnowOn( Tree, 2341, 800, .01, 10, 10, 10 )
      texture{ Texture_Snow }
    }

  #declare Tree2=
    union
    {
      object { Tree }
      object { TreeSnow2 }
    }
    
#end // TREE

#if (MAIN_TREE)

  #declare recursionmax = 4; // Defines the complexity of the tree
  #declare divisions = array[4] {7,3,3,2}//,2,4}
  init_rand (534,76445,2345)     // Supp]ly seeds to initialize the random variable
  
  // Variables modeled exponentially
  // Use: init_variable (start_size, end_size, base)
  init_tdev (0.5, 0.8, 0.9)
  init_rdev (70, 40, 1)
  init_twigprob (0, 0.7, 0.5)
  init_branchl (3.4, 0.2, 0.6)
  init_branchw (0.35, 0.004, 0.4)
  
  // Odd variables
  init_roots (2)			// Use: init_roots (rootrecursionmax)
  init_splinedev (0.15,0.3)	// Use: init_splinedev (min_dev, max_dev)
  init_simplify (2,2)		// Use: init_simplify (num_simplify_levels, num_segments)
  
  // Add some gravity
  #declare tendancy = <-0.05,0.2,0.05>;
  
  // Assign the branch texture
  #declare branchtexture = texture { standard_branch } // standard_branch is defined in leafdefs.inc
  
  // Define transformations assigned to the whole tree here.
  #declare global_transform = transform {
  	scale .85
  }
  
  // Provide feedback in the first 2 recursion levels
  #declare feedback = 2;
  
  // Call the macro
	#declare MainTree=
	  union
	  {
      splinetree(0,0)
    }
  #debug concat( "\nTree done.\n")
  #local TreeSnow3=
    blob
    {
      BlanketSnowOn( MainTree, .1, 7, 5, 5 )
      //HeapSnowOn( Tree, 2341, 800, .01, 10, 10, 10 )
      texture{ Texture_Snow }
    }

  #declare Tree3=
    union
    {
      object { MainTree }
      object { TreeSnow3 }
    }
  
#end // MAIN_TREE

////////////////////////////////////////////////////////////
//
// --------------------------------------- CAT
//
////////////////////////////////////////////////////////////

#if (CAT)  

#include "cat.inc"

#declare BrownCat=
  object
  {
    Cat
    scale .0025
    rotate <-90,90,0>
  }

#end // CAT

////////////////////////////////////////////////////////////
//
// --------------------------------------- STATUE
//
////////////////////////////////////////////////////////////

#if (STATUE)  

  #include "statue.inc"
  
  #declare AStatue=
    object
    {
      Statue
      //inverse
      rotate< -90,180,0 >
      scale 2
      //translate <0,-1,0>
    }

  #local StatueSnow1=
    blob
    {
      BlanketSnowOn( AStatue, .1, 4, 1.5, 1.5 )
      texture{ Texture_Snow }
    }
  #local StatueSnow2=
    blob
    {
      BlanketSnowOn( StatueSnow1, .05, 4, 1.5, 1.5 )
      texture{ Texture_Snow }
    }
  #local StatueSnow3=
    blob
    {
      BlanketSnowOn( StatueSnow2, .06, 4, 1.5, 1.5 )
      texture{ Texture_Snow }
    }
    
  #local StoneStatue=
    union
    {
      object{ AStatue }
      object{ StatueSnow1 }
      object{ StatueSnow2 }
      object{ StatueSnow3 }
    }
#end // STATUE

////////////////////////////////////////////////////////////
//
// --------------------------------------- STATUE
//
////////////////////////////////////////////////////////////

#if (GRAVE_FENCE)  

  #include "fence.inc"
  
  // MakeBoxFence( FenceWidth, FenceDepth, FencePostSpace, FencePostSize, FenceBars, FenceTexture, Top )
  MakeBoxFence( 1.66, .9, .15, <.02,.85,.02>, 3, texture{ LightIronTexture scale<1,2,1> }, 2 )

  #declare TheFence=
    object
    {
      BoxFence
    }

  #local GraveFenceSnow1=
    blob
    {
      BlanketSnowOn( BoxFence, .1, 2, 1.9, 1 )
      texture{ Texture_Snow }
    }
/*  #local StatueSnow2=
    blob
    {
      BlanketSnowOn( StatueSnow1, .05, 4, 1.5, 1.5 )
      texture{ Texture_Snow }
    }
  #local StatueSnow3=
    blob
    {
      BlanketSnowOn( StatueSnow2, .06, 4, 1.5, 1.5 )
      texture{ Texture_Snow }
    }
*/    
  #local GraveFence=
    union
    {
      object{ BoxFence }
      object{ GraveFenceSnow1 }
    }
#end // GRAVE_FENCE

////////////////////////////////////////////////////////////
//
// --------------------------------------- Background Columns
//
////////////////////////////////////////////////////////////

#if (BACK_COLUMNS)
  #include "13_col.inc"
  
  
#end // BACK_COLUMNS

////////////////////////////////////////////////////////////
//
// --------------------------------------- The Scene Layout
//
////////////////////////////////////////////////////////////

object{ SnowField texture{ Texture_Snow } }


#if (CHURCH_WALL)
  #debug concat( "Placing Church Wall\n" )
  object { FullWall translate <0,6.5,32> }
#end // CHURCH_WALL
#if (WINDOW)  
  #debug concat( "Placing Stained Glass Window\n" )
  object { Stained_Glass_Window  scale 5.60 translate<0,8.8,32> }
#end // WINDOW
#if (BACKDROP)  
  #debug concat( "Placing Backdrop\n" )
  object { Backdrop scale 11.75 translate<0,9.1,35> no_shadow }
#end // BACKDROP
#if (BUTTRESSES)
  #debug concat( "Placing Buttresses\n" )
  object { Buttress1 scale 1.20 rotate <0,90,0> translate <-8,5.5,26> }
  object { Buttress2 scale 1.20 rotate <0,90,0> translate <8,5.5,26> }
  //object { Buttress1 scale 1.10 rotate <0,00,0> translate <0,5.5,26> }
#end // BUTTRESSES
#if (BENCH)  
  #debug concat( "Placing Benches\n" )
  object { BenchOne translate <0,-.20,31> }
  object { BenchTwo translate <0,-.20,11> }
#end // BENCH
#if (CRYPT)
  #debug concat( "Placing Tomb\n" )
  object{ CryptOne translate <-4.7,0.2,21.8> }
#end // CRYPT
#if (CROSS) 
  #debug concat( "Placing Stone Cross\n" )
  object{ CrossOne translate <4,0.3,18> rotate z*.20 }
#end //CROSS
#if (TREE)
  #debug concat( "Placing Side Trees\n" )
  object{ Tree1 rotate <0,-65,0> translate <-8,-1,20>}
  object{ Tree2 rotate <0,180,0> translate <8,-1,20>}
#end // TREE
#if (MAIN_TREE)
  #debug concat( "Placing Main Tree\n" )
  object{ Tree3 rotate <0,90,0> translate <0,-1,20>}
#end // MAIN_TREE
#if (CAT)  
  #debug concat( "Placing Cat\n" )
  object{ BrownCat translate <0,.75,11>}
#end // CAT
#if (STATUE)  
  #debug concat( "Placing Statue\n" )
  object{ StoneStatue rotate <0,90,-10> translate <4.5,-.5,14> }
#end // STATUE
#if (GRAVE_FENCE)
  #debug concat( "Placing Grave Fence\n" )
  object{ GraveFence translate <-4.5,-.5,16> }
#end // GRAVE_FENCE

#debug concat( "Rendering...\n" )
