/*beacon scene
 
 This pov file was written by Sebastian Stratbcker -  www.chano.de - 21.9.2002
 
 Creates a deserted moon landscape, with a rising planet on the horizon 
 and a pov-logo shaped red-glowing beacon, sticking in the dusty surface
 between randomly placed rocks. 
 
 The 800x600 pixel image with all features at full quality , took 20 hours and 46 minutes to render on a 600 MHz Pentium III.
 
 Remarks:
 
 - POV-Logo was copied from logo.inc file.
 - Clouds were copied from skies.inc.
 

*/   

#version 3.5  ;

#include "functions.inc" 
#include "stars.inc"  


  //constant values needed
 
 #declare cam_location = 
                    <1,2,-13>; // <1,2,-13>
 #declare cam_look_at = 
                   <-9,0,-27>; // <-9,0,-27>
 #declare moon_rad = 5000;     // 5000, set radius of the moon  
 #declare planet_location = 
               <-290,35,-230>; // <-290,35,-230>
 #declare nr_of_rocks = 3000;  // 3000, set this value for original rock placement
 #declare  delta_x = -330;     // -330 translates the moon relief in any direction
 #declare  delta_y = -230;     // -230
 #declare  delta_z = -370;     // -370
 #declare agaf = -10;          // lower value means higher hills
 #declare pf = 0.001;          // dimensions of moon landscape increases with decreasing this value
 #declare R = seed(4);         // 4 is original random seed used
 
 #declare radiosityOn = false; // not recommended - takes probably months to render
 #declare photonsOn = false ;  // for photon shots on pov-logo, turn this true  
 #declare loadPhotons = false; // if you want to load the photon map of the original scene from file 
 #declare noiseOn = false;     // switches crand and iso noise 

 
 // global settings
 
 global_settings {  
  noise_generator 3
  number_of_waves 10
  assumed_gamma 1
  #if(photonsOn)
   max_trace_level 5
   photons {
      spacing 0.04  // higher value than 0.01 means 'lower' quality, faster parsing.
      autostop 0
      jitter 0.5
      max_trace_level 15  
      #if(loadPhotons)
      load_file "beacon.ph"   
      #end
  } 
  
 #end
  
  #if(radiosityOn)
    radiosity {
    pretrace_start 0.08
    pretrace_end   0.04
    count 70

    nearest_count 5
    error_bound 0.1
    recursion_limit 1

    low_error_factor .5
    gray_threshold 0.0
    minimum_reuse 0.015
    brightness 1

    adc_bailout 0.01/2  
    

  } #end


} 
 

 
 
 //materials and textures section
 

//standard glass material 
#declare M_Glass=    
material {
  texture {
    pigment {rgbt 1}
    finish {
      ambient 0.0
      diffuse 0.05
      specular 0.6
      roughness 0.005
      reflection {
        0.1, 1.0
        fresnel on
      }
      conserve_energy
    }
  }
  interior {
    ior 1.5
    fade_power 1001
    fade_distance 1.5
    fade_color <0.8,0.8,0.86>
  }
}     

// used to simulate moon dust with very grainy appearance
#declare T_Moon = texture{

  pigment{ color rgb <0.8,0.8,0.85> } 
  finish{ phong 0.01 diffuse 0.5 ambient 0.1   
  #if(noiseOn)
  crand 0.08
  #end
  }  


}
// used for the rocks, covering the moon surface
#declare T_Rocky = texture{

  pigment{ color rgb <0.85,0.85,0.9> } 
  finish{ phong 0.01 diffuse 0.4 ambient 0.1  
  #if(noiseOn)
  crand 0.04 
  #end
  }  


} 

// used for the land areas of the tiny planet beyond the horizon
#declare T_Planet = texture{
   
   pigment{ 
   
      agate
      agate_turb 1.2 
      scale 1 
      translate y*10
      color_map{
          [0 color rgb <0.8,0.8,0.2>]
          [.2 color rgb <0.1,0.94,0.2>]
          [.4 color rgb <0.41,0.71,0.1>]
          [.7 color rgb <0.951,0.961,0.3>] 
          [1 color rgb <0.8,0.8,0.2>]
          
         }
      
      }
      
   finish{ diffuse 0.5 ambient 0.0 
   //crand 0.02 
   }   

}

// used for the cloud coverage of the planet
#declare T_Clouds =
pigment {
    bozo
    turbulence 0.65
    octaves 6
    omega 0.7
    lambda 2
    color_map {
        [0.0, 0.1   color rgb 1
                    color red 0.95 green 0.95 blue 0.98]
        [0.1, 0.4   color red 0.95 green 0.95 blue 0.98
                    color rgbt <0.8,0.8,1,1>]
        [0.4, 1.001 color rgbt <0.8,0.8,1,1>
                    color rgbt <0.8,0.8,1,1>]
    }
    scale <0.3, 0.1, 0.2>
}

//the atmosphere gets various cloud layers, taken from "skies.inc"
#declare T_Atmosphere =
texture {                              
    pigment {
        T_Clouds
    }
    finish {
        ambient 0.0
        diffuse 0.5
    }
}
texture {                              
    pigment {
        T_Clouds
        translate -0.15*y 
        scale 1.5
    }
    finish {
        ambient 0.0
        diffuse 0.5
    }                 
    
    
}
// simulates the sea areas of the planets
#declare T_Ocean = texture {
  
  pigment{ color rgb <0.0,0.1,0.6> }
  finish{ reflection 0.0 brilliance 0.7 specular 0.6 phong 0.3 ambient 0.0 diffuse 0.3}
  normal{ dents 1 scale 0.01 }

} 



// simple and complex objects section



// the povray logo taken from "logo.inc", used as red-glowing glass beacon
#declare Povray_Beacon = union{
  merge { 
   difference{
   sphere {2*y, 1}
   sphere {2*y, 0.9}
   }
   difference {
      cone {2*y, 1, -4*y, 0}
      sphere {2*y, 1.4 scale <1,1,2>}
   }
   difference {
      sphere {0, 1 scale <2.6, 2.2, 1>}
      sphere {0, 1 scale <2.3, 1.8, 2> translate <-0.35, 0, 0>}
      rotate z*30 translate 2*y
   } 
   
   
  material { M_Glass }
   
  } 
  
  light_source {
 
   <0.0, 2.0 , 0.0>     
   color rgb <0.45,0.2,0.1>  
    
   fade_distance 60
   fade_power 10
  
 
 }
  
  rotate <0, 0, -25>
  translate <-0.5,-0.35,0>
  
 }

//the moon is an iso-sphere with slight random variation of surface 
#declare moon = isosurface{
  
     
      function{ f_sphere(x,y,z,moon_rad) 
      
      - f_agate(x*pf+ delta_x,y*pf+ delta_y,z*pf+ delta_z) * agaf * f_noise3d(x*0.01,y*0.01,z*0.01) 
      
      
      
      }
      
      accuracy 0.005
      max_gradient 2
      contained_by{ sphere{ <0,0,0>,moon_rad*1.1} }
      
  }   
  
  
// a rock is realized with a macro, to garuantee uniqueness                                      
#macro rock(xr,yr,zr,sr,rf)
 
 #declare boscale = 1.3;
 isosurface{
  
     #declare xrand = 0.6+rand(R)*0.4;
     #declare zrand = 0.6+rand(R)*0.4;
     
     function{
      
    f_sphere(x*xrand,y,z*zrand,sr)
    
    - f_hetero_mf(x*rf - xr,y*rf - yr ,z*rf - zr, 1.2, 2.4, 2, 1, 1, 2)*0.5 
    
    #if(noiseOn) 
    - f_agate(x+xr,y+yr,z+zr)*f_noise3d(x*boscale,y*boscale,z*boscale)*0.2 
    #end
    
   }
   threshold 0

   accuracy 0.001
   max_gradient 5
      contained_by{ sphere {<0,0,0>,10*sr} }
      translate <xr,yr,zr>
  }

#end


// this is the soil of the planet, randomized in height to reach over sea-level
#declare Iso_Planet = isosurface{
  
     #declare prf = .3 ; 
     #declare paf = 600;
     #declare pxr = 100;
     #declare pyr = 500;
     #declare pzr = 0;
     
     
     
     function{
      
     f_sphere(x,y,z,9.7)
    
   - f_hetero_mf(x*prf - pxr,y*prf - pyr ,z*prf - pzr, 1.2, 2.4, 2, 1, 1, 2)*0.2 
   
    #if(noiseOn) 
    // - f_agate(x*paf,y*paf,z*paf)*0.001 for very grainy land masses 
    #end
    
   }
   threshold 0

   accuracy 0.001
    max_gradient 5
      contained_by{ sphere {<0,0,0>,11} }
     
  }

// the isoclouds are built as difference of this iso ball an a sphere
#declare Iso_Clouds = isosurface{
  
     #declare crf = 1.8 ; 
     #declare caf = 0.5;
     #declare cxr = 0;
     #declare cyr = 0;
     #declare czr = 0;
     
     #declare prf = .28 ; 
     #declare paf = 601;
     #declare pxr = 103;
     #declare pyr = 500.5;
     #declare pzr = 0;
     
     function{
      
     f_sphere(x,y,z,9.6)
    
    - f_hetero_mf(x*crf - cxr,y*crf - cyr ,z*crf - czr, 1.2, 2.8, 2, 1, 1, 2)*0.24   
    
    * f_hetero_mf(x*prf - pxr,y*prf - pyr ,z*prf - pzr, 1.2, 2.4, 2, 1, 1, 2)
   
    
   }
   threshold 0

   accuracy 0.001
    max_gradient 5
      contained_by{ sphere {<0,0,0>,11} }
     
  }

//this is the CSG difference that represents the clouds of the planet
#declare planet_clouds = difference {
    object{ Iso_Clouds }
    sphere{ <0,0,0>,10.1 }
    
    pigment{ color rgb 1 }
    finish{ diffuse 1 ambient 0 crand 0.05 phong 0.4}

}

//the blue athmosphere
#declare atmosphere = sphere { <0,0,0>,11
 
 pigment{ color rgbt 1 }

  
 hollow on
 interior{
  media{
    emission 0.00
    scattering {1, 0.9}
    density { 
     spherical scale 12
     
      color_map {
        [0.0 rgb 0]
        [0.1 rgb 0]
        [1.0 rgb <0.0,0.0,0.9>]
      }
    }
  }
  }
 } 


//the planet is a union of atmosphere, sea-sphere , soil and clouds 
#declare planet = union {
 
 object{ Iso_Planet texture{ T_Planet } }
 
 
 sphere { <0,0,0>, 10.25  texture {T_Atmosphere scale 6} }
 
 object{ planet_clouds }
 
  
 sphere{ <0,0,0>, 10 texture{ T_Ocean  } }
 
 object{ atmosphere }
 

}


//the full_moon is just called that, because a while loop places the preferred number of rocks on its surface
#declare full_moon = union {

difference{
        object{ moon  } 
        cone {2*y, 1, -4*y, 0 rotate <0, 0, -25> translate <-0.5,-0.35,0> rotate y*180 scale 2.5 translate <-6.0,-2.4+moon_rad,-28> }
        texture{ T_Moon } 
      }
     
     
   #declare rocks = 0;  //the loop for randomly placing rocks onto the moon surface  
   #while( rocks < nr_of_rocks)   
     #declare x_val = rand(R)*400 - 200; //placing inside a 400*400 square
     #declare z_val = rand(R)*400 - 200; 
     #declare y_val = sqrt(pow(moon_rad,2) - pow(x_val,2) - pow(z_val,2)) - 10; 
     #declare sphere_rad = exp(-5 + rand(R)*6); 
     #if(sphere_rad < 0.3) 
        #declare sphere_rad = 0.3;
     #end
     #while (y_val < moon_rad*2)
     #if ( abs(f_sphere(x_val,y_val,z_val,moon_rad)- f_agate(x_val*pf+ delta_x,y_val*pf+ delta_y,z_val*pf+ delta_z) *agaf * f_noise3d(x_val*0.01,y_val*0.01,z_val*0.01)  - sphere_rad) < 0.4)
     object{ rock (x_val,y_val,z_val,sphere_rad,0.5+rand(R)*0.1) texture{ T_Rocky } }
      #declare y_val = moon_rad*2;
     #end
     #declare y_val = y_val + 0.1; 
     #end 
     
   #declare rocks = rocks + 1;  
   #end //rock loop  
   
   
   
rotate <0,0,0> //<60,0,0>
translate <0,-moon_rad,0>


}




//object placement section

//the visible part of the moon surface
object{ full_moon bounded_by{ box{<-200,-10,-200>,<200,25,200>}}}  

//the little planet in the background
object{ planet scale 4 translate planet_location }    

//the pov beacon in the foreground
object { Povray_Beacon rotate y*180 scale 2.3 translate <-6.0,-2.4,-28> 

  #if(photonsOn)
  photons{ target 1 refraction on reflection off}
  #end 
}


//the eternal universe 
sphere{<0,0,0>,10000 hollow   texture{ Starfield1 }   }
  
//the sun 
light_source {    
   
   <-5000.0, 3200.0, 8000>     
   color rgb <0.9,0.85,0.7>  
    
 
}   


  
//the camera
 camera { 
   
    location cam_location
   
    look_at  cam_look_at
      
  }
  
