/*
 * mtlily.pov - Maurizio Tomasi 2002 (zio_tom78@hotmail.com)
 *              Visit my website at http://www.geocities.com/zio_tom78
 *
 * "The Lilypond" - image submitted to the September/October round of the
 *                  Internet Ray-Tracing Competition (IRTC).
 *
 * Created with POV-Ray 3.5 under Linux version 2.4.18-3 (Red Hat 7.3)
 */

#include "colors.inc"
#include "rand.inc"
#include "debug.inc"
#include "arrays.inc"

#include "lily.inc"
#include "grass.inc"

// Uncomment this to get useful informations when rendering the image.
Set_Debug (false)

#declare USE_WATERLILIES = 1;
#declare USE_WATER = 1;
#declare USE_LAND = 1;
#declare USE_WILLOWS = 1;      // Can be used only with USE_LAND
#declare USE_GRASS = 1;        // Ditto
#declare USE_REEDS = 1;
#declare CAMERA_NUM = 0;

//#declare USE_FIREFLIES = 1;    // Quite ugly: better leave it commented!
//#declare USE_QUICK_WATER = 1;  // Faster rendering of water

//////////////////////////////////////////////////////////////////////
// Cameras (more than one...)
//////////////////////////////////////////////////////////////////////

#switch (CAMERA_NUM)
  // Default camera: it is used for the final rendering
  #case (0)
    camera {
      location  <10.0, 10.0, -70.0>
      look_at   <0.0, 10.0,  0.0>
      right     x * image_width / image_height
    }
  #break
  
  // Top camera: useful to check if the objects are rightly placed
  #case (1)
    camera {
      location  <0.0, 1000.0, 0.0>
      look_at   <0.0,  0.0, 0.0>
      right     x * image_width / image_height
    }
  #break
#end

// Two light sources: the first come from the moon while the second
// one is shadowless, and it is used to brighten a bit the colors in
// the shadowed side of the objects in the scene.

light_source {
  <120, 40, -120> color rgb 0.7
}

/*
light_source {
  <-100, 100, 100> color 0.4
  shadowless
}
*/

global_settings {
  ambient_light rgb <0.93, 0.66, 0.32>
  max_trace_level 10
}


// This is Starfield1 (from stars.inc) with some modifications...
#declare CustomStarfield =
  pigment {
    granite
    color_map {
      [ 0.000 0.270 color rgbt 1.0 color rgbt 1.0 ]
      [ 0.270 0.280 color rgb <0.3,0.3,0.15> * 2
	color rgb <0.6,0.6,0.4> * 2]
      [ 0.280 0.680 color rgbt 1.0 color rgbt 1.0 ]
      [ 0.680 0.690 color rgb <0.15,0.15,0.15> * 2
	color rgb <0.4,0.4,0.4> * 2]
      [ 0.690 1.000 color rgbt 1.0 color rgbt 1.0 ]
    }
    
    turbulence 1
    sine_wave
    scale .5
  }


// The sky sphere.  It is composed by two pigments: the first one
// gives a faint blue shade near the horizon (this means the sun went
// down not too many hours ago...), while the second give the starred
// sky

sky_sphere
{
  pigment {
    gradient y
    color_map {
      [0.50 0.80 color rgb <0.93, 0.66, 0.32> * 0.1 color rgb 0.0]
      [0.80 1.00 color rgb 0.0 color rgb 0.0]
    }
    scale 2
    translate -1
  }

  pigment {
    CustomStarfield
  }
  
  // Some clouds...
  pigment {
    bozo
    turbulence 1.65
    octaves 6
    omega 0.7
    lambda 2

    color_map {
      [0.0 0.2 color rgb <0.060, 0.051, 0.051> transmit 0.3
	color rgb <0.060, 0.051, 0.051> * 2 transmit 0.3]
      [0.2 0.4 color rgb <0.060, 0.051, 0.051> * 2 transmit 0.3
	color rgb <0.060, 0.051, 0.051> * 2 transmit 1]
      [0.5 1.0 color rgb <0.060, 0.051, 0.051> * 2 transmit 1
	color rgb <0.060, 0.051, 0.051> * 2 transmit 1]
    }

    scale <3.5, 1.0, 1.0>
  }
}


//////////////////////////////////////////////////////////////////////
// Waterlilies
//////////////////////////////////////////////////////////////////////

#ifdef (USE_WATERLILIES)
  #local BunchStream = seed (27);
  #local Count = 0;
  #local bunchRadius = 60;
  
  // A "bunch" is a set of waterlilies and leaves. We create 15
  // of them.
  #while (Count < 15)
    #local X = RRand (- bunchRadius/2, bunchRadius/2, BunchStream);
    #local Z = RRand (- bunchRadius/2, bunchRadius/2, BunchStream);
    #local Radius = sqrt (X * X + Z * Z);

    #if (Radius <= bunchRadius)
      object {
	WaterlilyBunch (RRand (6, 14, lilyStream), 18,
	  RRand (60, 80, leafStream), 30)
	
	translate <X, 0.0, Z + 10.0>
      }
      #local Count = Count + 1;
    #end
  #end
#end


//////////////////////////////////////////////////////////////////////
// Land
//////////////////////////////////////////////////////////////////////

#ifdef (USE_LAND)
  #declare Land =
  height_field {
    png "land.png"

    // This is a slowly-varying wavy texture
    texture {
      pigment {
	waves
	turbulence 15.0
	scale 1.0
	color_map {
	  [0.00 color rgb <0.74, 0.68, 0.64> * 0.25]
	  [1.00 color rgb <0.95, 0.93, 0.75> * 0.25]
	}
      }
    }
    
    // This give some small "dark" spots (but not too much dark)
    texture {
      pigment {
	wrinkles
	turbulence 15.0
	scale 0.1
	color_map {
	  [0.00 color rgbt <1, 1, 1, 1>]
	  [0.50 color rgbt <1, 1, 1, 1>]
	  [0.51 color rgb  <0.95, 0.93, 0.75> * 0.90 * 0.25]
	  [1.00 color rgb  <0.95, 0.93, 0.75> * 0.65 * 0.25]
	}
      }
    }
    
    translate <-0.5, -0.33, -0.5>
//    scale <750, 50.0, 750>
    scale <800, 50.0, 800>
  }

  #declare PartialLand =
  intersection {
    object { Land }
    
    box {
      <-200, 0, 0>, <150, 50, 250>
    }
    
    // This is to turn off a warning message
    pigment { color rgb 1 }
  }

  object { Land }
  
  ////////////////////////////////////////////////////////////////////
  // Grass
  ////////////////////////////////////////////////////////////////////

  // Place some grass on the land
  #ifdef (USE_GRASS)
    TraceGrass (PartialLand, 0.0, 20000, 4.0,
      1.5, 2.0, <0.34, 0.45, 0.25>, <0.39, 0.30, 0.01>, false)
  #end
  
  ////////////////////////////////////////////////////////////////////
  // Fireflies
  ////////////////////////////////////////////////////////////////////
  
  #ifdef (USE_FIREFLIES)
    #declare Firefly =
    sphere {
      0, 0.25
      
      no_shadow
      
      finish { ambient 0 }
      
      pigment { color rgb <0.49, 1.0, 0.09> * 3.5 }
    }
    
    #local FlyStream = seed (14);
    #local Count = 0;
    #while (Count < 20)
      #local Normal = <0, 0, 0>;
      #local Point = trace (PartialLand,
	<RRand (-200, 150, FlyStream), 100, RRand (0, 250, FlyStream)>,
	<0, -1, 0>, Normal);
      
      #if (vlength (Point) != 0)
	object {
	  Firefly
	  
	  translate Point + 5*y
	}

	#declare Count = Count + 1;
      #end
    #end
  #end

  ////////////////////////////////////////////////////////////////////
  // Weeping Willows
  ////////////////////////////////////////////////////////////////////

  #ifdef (USE_WILLOWS)
    
    #include "willow.inc"

    #local Pos = trace (Land, <-30, 100, 225>, <0, -1, 0>);
    object {
      Willow1
      
      scale 2.5
      translate Pos - 15*y
    }

    #local Pos = trace (Land, <-200, 100, 225>, <0, -1, 0>);
    object {
      Willow2
      
      scale 1.7
      
      translate Pos - 10*y
    }

    #local Pos = trace (Land, <-120, 100, 300>, <0, -1, 0>);
    object {
      Willow2
      
      scale 1.8
      rotate 40*y
      
      translate Pos - 20*y
    }

    #local Pos = trace (Land, <50, 100, 300>, <0, -1, 0>);
    object {
      Willow2
      
      scale 2.1
      rotate 160*y
      
      translate Pos - 15*y
    }

    #local Pos = trace (Land, <130, 100, 260>, <0, -1, 0>);
    object {
      Willow2
      
      scale 1.6
      rotate 70*y
      
      translate Pos - 15*y
    }

    #local Pos = trace (Land, <80, 100, 300>, <0, -1, 0>);
    object {
      Willow3
      
      scale 2.1
      
      translate Pos - 13*y
    }
  #end
#end


//////////////////////////////////////////////////////////////////////
// Reeds
//////////////////////////////////////////////////////////////////////

#ifdef (USE_REEDS)
  #declare ReedBase =
  height_field {
    png "reed-base.png"
    water_level 0.1

    translate <-0.5, 0.0, -0.5>
    scale <150, 0.01, 100.0>
    translate <-75, -2, 100>
    
    // This is only to avoid a warning message: the object will never
    // be viewable...
    pigment { color White }
  }

  TraceGrass (ReedBase, -10, 150, 16.0, 4.0, 5.0,
    <0.00, 0.45, 0.10>, <0.70, 0.45, 0.20>, false)
#end


//////////////////////////////////////////////////////////////////////
// Water
//////////////////////////////////////////////////////////////////////

#ifdef (USE_WATER)
  
  // This ground fog adds a lot of realism to the water
  fog {
    distance 10.0
    color rgbt <0.07, 0.07, 0.10, 0>
    fog_type 2
    fog_offset -20.0
    fog_alt 0.01
  }


  // It is not clear to me if the "quick water" (box with a bozo normal)
  // is really worse than the normal one (isosurface)...

  #ifdef (USE_QUICK_WATER)
    box {
      <-300, -100, -300>, <300, -0.05, 300>

      hollow
      
      normal { bozo scale 2.5 turbulence 3 }

      pigment {
	color rgbt <0.2, 0.2, 0.7, 0.8>
      }

      interior {
	ior 1.43
	
	// Fading is required even if a ground fog is present !
	fade_distance 10
	fade_power 1001
	fade_color <0.07, 0.07, 0.10>
      }

      finish {
	reflection { 0.03, 1 fresnel }
	ambient 0
	diffuse 0
	conserve_energy
      }
    }

  #else

    #local WaterFunction =
    function {
      pigment {
	bozo
	scale 3.5
	turbulence 3
	color_map {
	  [0 color rgb 0]
	  [1 color rgb 1]
	}
      }
    }

    isosurface {
      function { y + 0.1 + 0.100 * WaterFunction(x, y, z).gray }
      contained_by { box { <-300, -5, -300>, <300, -0.05, 1000> } }
      accuracy 0.1
      max_gradient 1.2
      
      pigment {
	color rgbt <0.2, 0.2, 0.7, 0.8>
      }


      interior {
	ior 1.43
	fade_distance 10
	fade_power 1001
	fade_color <0.07, 0.07, 0.10>
      }

      finish {
	reflection { 0.03, 1 fresnel }
	ambient 0
	diffuse 0
	conserve_energy
      }
    }
  #end

#end // USE_WATER

