// Persistence of Vision Ray Tracer Scene Description File
// File: Transv.inc
// Vers: 3.5
// Desc: Transversal Wall
// Date: 2003/03/30
// Auth: Maurizio Tomasi
//

#ifdef (USE_TRANSVERSAL_WALL)
    
    #local A = 15 * (1 - SmallAisleRatio);
    #local B = 12 - 8.4;
    #local TransvLen = sqrt (pow(A, 2) + pow (B, 2));
    #local TransvMinHeight = LowWallHeight + 0.2;
    #local TransvMaxHeight = TransvMinHeight + 1.0;
    #local TransvThickness = WallThickness;
    #local Pendence = 1.5;
    #local TransvSideWallLen = 1.5;
    
    // Shear and translate by TransvMinHeight along y
    #local ShearTransform =
    transform
    {
        matrix
        <
            1, Pendence,   0,       // the first 3 lines form a rotation matrix
            0, 1,   0,       // since it is not orthogonal, shearing occurs
            0, 0,   1,
            
            // Translation
            0,
            TransvMinHeight,
            0
        >
    }

    #local CopertureElement =
    union
    {
        box
        {
            <0,  0.00, -0.01>,
            <1,  0.05, TransvThickness * 1.5>
        }

        cylinder
        {
            <0, 0.025, TransvThickness * 1.5>
            <1, 0.025, TransvThickness * 1.5>, 0.025
        }
    }
    
    #local TopCopertureElement =
    union
    {
        object { CopertureElement }
        
        cylinder
        {
            <0, 0.05, TransvThickness * 1.1>,
            <1, 0.05, TransvThickness * 1.1>, 0.03
        }
    }
    
    #local Coperture =
    union
    {
        object
        {
            CopertureElement
            scale <1, 1, -1>
            rotate -10*x
        }
        
        object
        {
            CopertureElement
            rotate 10*x
        }
    }
    
    #local TopCoperture =
    union
    {
        object
        {
            TopCopertureElement
            scale <1, 1, -1>
            rotate -10*x
        }
        
        object
        {
            TopCopertureElement
            rotate 10*x
        }
    }

    #local Deco1Size = 0.2;
    
    #local HalfDeco1 =
    height_field
    {
        png "deco1.png"
        water_level 0.01
        smooth
        
        translate <-0.5, -0.5, 0>
        rotate 45*y
        translate <0, 0.5, 0>
        scale <1, 0.1, 1>
        translate <0, 0, 0.25>
    }

    #local StickHeight = 0.9;
    #local StickRadius = 0.06;
    #local StickFlipped = false;
    
    #macro CreateDeco1 (RndSeed)
    union
    {
        union
        {
            object
            {
                HalfDeco1
                scale <1, 0.5, 1>
            }
        
            object
            {
                HalfDeco1
                rotate 180*z
                scale <-1, 0.5, 1>
            }
            
            rotate -90*x
        }

        box
        {
            <-0.50, -0.05, -0.1>,
            <-0.45,  1.05,  0.1>
        }
       
        union
        {
            #local SphereNum = 10;
            #local Sweep =
            sphere_sweep
            {
                b_spline
                SphereNum + 2,
                
                #local Count = -1;
                #while (Count <= SphereNum)
                    StickRadius * <cos (4 * pi * Count / SphereNum), 
                                   1 + 20 * StickHeight * Count / (SphereNum - 1), 
                                   sin (4 * pi * Count / SphereNum)>, StickRadius
                    #declare Count = Count + 1;
                #end
            }
            
            object { Sweep }
            object { Sweep scale <-1, 1, -1> }

            #if (StickFlipped)
                scale <-1, 1, 1>
                #declare StickFlipped = ! StickFlipped;
            #else
                #declare StickFlipped = ! StickFlipped;
            #end
            
            scale 0.67
            translate <-0.475, 1.0, 0>
        }

        object
        {
            CreateBlobbedObj (50, RndSeed)
            scale 0.25*0.67
            translate <-0.475, 1.0 + StickHeight, 0>
        }
    }
    #end

    union
    {    
        // The wall

        difference
        {
            object
            {
                CreateWall (false, TransvLen, TransvMinHeight + Pendence,
                    false, 0.0, 0.0, 2 * TransvThickness, seed(8), false, 0, 
                    BrickTexture)
                
                translate -TransvThickness*z
            }

            union
            {
                box
                {
                    <0, 0, -TransvThickness * 2>,
                    <1.1, TransvMaxHeight - TransvMinHeight + Pendence, TransvThickness * 2>
                    
                    transform { ShearTransform }
                    scale <TransvLen - TransvSideWallLen, 1, 1>
                    translate TransvSideWallLen*x
                }
                
                box
                {
                    <-0.1, TransvMaxHeight, -TransvThickness * 2>,
                    <TransvSideWallLen + 0.1, TransvMaxHeight + Pendence, TransvThickness * 2>
                }

                pigment { color rgbt <1, 0, 0, 0.7> }
            }
                        
            cutaway_textures
        }
        
    
        // The coperture
        
        object
        {
            Coperture
            
            CreateArchTexture (seed (6))
    
            transform { ShearTransform }
            scale <TransvLen - TransvSideWallLen, 1, 1>
            translate TransvSideWallLen*x
        }

        object
        {
            Coperture
            
            CreateArchTexture (seed (2))
    
            transform { ShearTransform }
            scale <TransvLen - TransvSideWallLen, 1, 1>
            translate TransvSideWallLen*x
            translate (TransvMaxHeight - TransvMinHeight) * y
        }
        
        object
        {
            TopCoperture
            
            CreateArchTexture (seed (14))
    
            transform { ShearTransform }
            scale <TransvLen - TransvSideWallLen, 1, 1>
            translate TransvSideWallLen*x
            translate (TransvMaxHeight - TransvMinHeight + 1.15 * Deco1Size) * y
        }

        object
        {
            Coperture
            
            CreateArchTexture (seed (6))
            
            translate TransvMinHeight*y
            scale <TransvSideWallLen, 1, 1>
        }

        object
        {
            Coperture
            
            CreateArchTexture (seed (2))
            
            translate TransvMaxHeight*y
            scale <TransvSideWallLen, 1, 1>
        }

        object
        {
            TopCoperture
            
            CreateArchTexture (seed (14))
            
            translate (TransvMaxHeight + 1.15 * Deco1Size)*y
            scale <TransvSideWallLen, 1, 1>
        }
        
        // Decorations
      
        #local DecoSeed = seed(26);
        
        #local Decorations =
        union
        {
            // Horizontal half of the wall
            
            #local CurPos = TransvSideWallLen;
            #while (CurPos > Deco1Size)
            
                object
                {
                    CreateDeco1 (DecoSeed)
                    
                    scale Deco1Size
                    translate <CurPos - Deco1Size/2 + 0.03, 0, 0>
                    CreateArchTexture (DecoSeed)
                }
                #declare CurPos = CurPos - Deco1Size;
            
            #end
            
            // Oblique part of the wall
            
            #local CurPos = TransvSideWallLen;
            #while (CurPos < TransvLen - 3 * Deco1Size)
            
                object 
                {
                    CreateDeco1 (DecoSeed)
                    
                    scale Deco1Size
                    #local CurX = CurPos + Deco1Size/2;
                    translate <CurX,
                        (CurX - TransvSideWallLen)*Pendence/(TransvLen - TransvSideWallLen) + 0.03, 0>
                    CreateArchTexture (DecoSeed)
                }
                #declare CurPos = CurPos + Deco1Size;
            
            #end
        }
        
        object
        {
            Decorations
            translate <0, TransvMaxHeight - 0.01, TransvThickness * 1.3>
        }
        
        object
        {
            Decorations
            translate <0, TransvMaxHeight - 0.01, -TransvThickness * 1.3>
        }


        // Column and arches

        #local TransvArchRadius = (TransvMaxHeight - TransvMinHeight) 
            / pow (1 + f_sqr(Pendence/TransvLen), 0.5);
            
        #macro CreateTransvArch (Pos)
        union
        {
            #local CenterPos = <Pos * (TransvLen - TransvSideWallLen), Pos * Pendence, 0>;

            #local TorusDecos =
            union
            {
                torus
                {
                    TransvArchRadius * 0.975,
                    TransvArchRadius * 0.025

                    rotate 90*x
                    translate CenterPos - TransvThickness/2 * z
                }

                torus
                {
                    TransvArchRadius * 0.975,
                    TransvArchRadius * 0.030
                    
                    rotate 90*x
                    translate CenterPos - 0.5 * TransvThickness/2 * z
                }
                
                #local Count = 20;
                #while (Count < 220)
                    object
                    {
                        CreateBlobbedObj (30, DecoSeed)
                        
                        scale TransvArchRadius * 0.030
                        translate 
                            1.05 * TransvArchRadius * 0.975 * <cosd (Count), sind (Count), 0.0>
                            + CenterPos - TransvThickness/2 * z
                    }
                    #declare Count = Count + 10;
                #end
            }
            
            object { TorusDecos }
            object { TorusDecos scale <1, 1, -1> }

            difference
            {
                cylinder
                {
                    CenterPos - TransvThickness/2 * z,
                    CenterPos + TransvThickness/2 * z,
                    TransvArchRadius
                }
                
                cylinder
                {
                    CenterPos - TransvThickness * z,
                    CenterPos + TransvThickness * z,
                    TransvArchRadius * 0.95
                }
            }
            
            translate <TransvSideWallLen, TransvMinHeight, 0>
        }
        #end

        intersection
        {        
            union
            {
                #local Count = 1;
                #while (Count <= 6)
                    object
                    {
                        CreateTransvArch (Count / 6)
        
                        CreateArchTexture (DecoSeed)
                    }
                    #declare Count = Count + 1;
                #end
            }
            
            box
            {
                <0, TransvMinHeight, -TransvThickness>,
                <TransvLen, TransvMaxHeight + Pendence, TransvThickness>
                
                pigment { color rgb 0 }
            }
        }
                
        rotate 180 * (1 + 1/pi*atan2(A, B))*y
        translate <12, 0, -15 + 0.1>
    }

#end // USE_TRANSVERSAL_WALL
