// Persistence of Vision Ray Tracer Scene Description File
// File: walls.inc
// Vers: 3.5
// Desc: CreateBrickedWall macro
// Date: 2003/03/30
// Auth: Maurizio Tomasi
//

#version 3.5;

#include "rand.inc"

/* Create a wall with the following size:
   
   ^ Height (y)
   |
   |
   |
   |  . Thickness (z)
   | /
   |/                  Length (x)
   +------------------->
   
   Each brick will have a mean length equal to BrickWidth and a mean height equal to RowHeight.
   BrickTextures is an array of textures; a single texture will be randomly picked, rotated
   and scaled.
 */
#macro CreateBrickedWall (Length, Height, Thickness, BrickWidth, RowHeight,
                          BrickTextures, MatteTexture, RndSeed)

    #local NumOfTextures = dimension_size (BrickTextures, 1);
    #local LastTextureNum = -1;
    
    union
    {
        #local CurRowPos = 0;    
        #while (CurRowPos < Height)
            // Create a row of bricks
            #local CurRowHeight = (1.0 + RRand (-0.3, 0.3, RndSeed)) * RowHeight;
            #local CurBrickPos = 0;
            
            // If this row is too tall, shorten it
            #if (CurRowPos + CurRowHeight > Height)
                #declare CurRowHeight = Height - CurRowPos;
            #end
            
            #if (CurRowHeight > RowHeight / 10)
                #while (CurBrickPos < Length)
                    #local CurBrickLength = (1.0 + RRand (-0.3, 0.3, RndSeed)) * BrickWidth;
                    
                    // If this brick is too long, truncate it
                    #if (CurBrickPos + CurBrickLength > Length)
                        #declare CurBrickLength = Length - CurBrickPos;
                    #end
                    
                    // Place the brick only if it is not too small
                    #if (CurBrickLength > BrickWidth / 10)
                        box
                        {
                            < CurBrickPos, Height - CurRowPos, 0 >,
                            
                            < CurBrickPos + 0.96 * CurBrickLength,
                              Height - CurRowPos - 0.96 * CurRowHeight,
                              Thickness >
                            
                            texture
                            {
                                // Pick a texture different from the last one
                                #local CurTextureNum = LastTextureNum;
                                #while (CurTextureNum = LastTextureNum)
                                    #declare CurTextureNum = floor(RRand(0, NumOfTextures, RndSeed));
                                #end
                                #declare LastTextureNum = CurTextureNum;
                                
                                BrickTextures[CurTextureNum]
                                
                                rotate 45 * VRand_In_Sphere (RndSeed)
                                scale RRand(0.8, 1.2, RndSeed)
                            }
                        }
                    #end
                                    
                    #declare CurBrickPos = CurBrickPos + CurBrickLength;
                #end
            #end
                        
            #declare CurRowPos = CurRowPos + CurRowHeight;
        #end
        
        // Matte        
        box
        {
            <0, 0, Thickness * 0.05>, <Length, Height, (1 - 0.05) * Thickness>
            texture { MatteTexture }
        }
    }
#end
