// The blob tree
// Copyright (C) 2003 Nicolas P. Rougier (rougier@loria.fr)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// ---------------------------------------------------------------------------
// Persistence of Vision Ray Tracer Scene Description File
// File: blobtree.pov
// Vers: 3.5
// Desc: The blob tree
// Date: 06/2003
// Auth: Nicolas Rougier
// Cmd : povray +Iblobtree.pov +w1200 +h900 +L./macro +a0.1 +am2 +fn 
// Cmd : povray +Iblobtree.pov +w800 +h600 +L./macro +fn  
// ---------------------------------------------------------------------------
#version 3.5;
#include "colors.inc"
#include "finish.inc"
#include "transforms.inc"
#include "blob.inc"


// ========================================
//  Notes
// ========================================
//
//  All positions (include camera) depend on the "dunes" object
//
//

// ========================================
//  Switches
// ========================================
#declare use_radiosity       = 1;
#declare radiosity_level     = 3; // 1:low, 2:medium, 3:high
#declare use_area_light      = 1;
#declare show_tree           = 1;
#declare show_blobs          = 1;
#declare use_fast_blob       = 0;
#declare show_weeds          = 1;

// ========================================
//  Settings
// ========================================
global_settings {
  assumed_gamma 2.2
  max_trace_level 25
  #if (use_radiosity)
    radiosity {
      pretrace_start 0.08
      #switch (radiosity_level)
      #case (1)
        pretrace_end 0.04
        count 35      
        error_bound 1.8
      #break
      #case (2)
        pretrace_end 0.02
        count 200
        error_bound .5
      #break
      #case (3)
        pretrace_end 0.01
        count 400
        error_bound .1
      #break      
      #end
      nearest_count 5
      recursion_limit 1
      low_error_factor .5
      gray_threshold 0.0
      minimum_reuse 0.015
      brightness 1
      adc_bailout 0.01/2
    }
  #end
}

// ========================================
//  Lights
// ========================================
light_source {
  <-500, 1400, -1000>
  color rgb <1.0, 0.99, 0.98>*2.1
  #if (use_area_light)
    area_light 200*x,200*z, 20,20
    circular
  #end
  jitter
  adaptive 1
  orient
  parallel
  point_at 0
}

// ========================================
//  Sky from demo file for the DF3 clouds
//   Copyright Gilles Tran May 2003
//   http://www.oyonale.com
// ========================================
sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 color rgb <0.7,0.7,1.0>]
      [1.0 color blue 0.5]
    }
  }
}
plane {
  y,0 
  pigment {
    wrinkles
    turbulence 1
    omega 0.1
    lambda 4
    color_map {
      [0.6 color Clear]
      [1.0 rgb <0.7,0.7,1.0>]
    }
  }
  finish {ambient 1 diffuse 0}
  hollow
  no_shadow
  scale <3,1,1>
  translate y
  scale 300000
  translate -z*600000
}    
plane {
  y,0 
  pigment {
    wrinkles
    turbulence 1
    omega 0.2
    lambda 4
    color_map {
      [0.5 color Clear]
      [1.0 rgb <0.7,0.7,1.0>*1.2]
    }
  }
  finish {ambient 1 diffuse 0}
  hollow
  no_shadow
  scale <1,1,2>
  rotate y*30
  translate y*2
  scale 300000
  translate -z*100000
}    

fog {
  rgb <0.7,0.7,1.0>
  fog_type 2
  distance 500
  fog_alt 50
}


// ========================================
//  Dunes
// ========================================
#declare Wave1 =
  pigment {
    dents
    turbulence .333
    octaves 3
    lambda .75
    omega 1.5
    color_map {
      [0,1 color rgb .5 color rgb 0]
    }
    ramp_wave
  }
#declare Wave2 =
  pigment {
    bumps
    turbulence .125
    octaves 2
    lambda 1.25
    omega .5
    color_map {
      [0,1 color rgb 1 color rgb 0]
    }
    ramp_wave
  }

#declare dunes =
  height_field {
    function 250,250 {
      pigment {
	average
	pigment_map {
	  [0.5 Wave1 scale .5]
	  [1.0 Wave2 translate .25]
	}
	ramp_wave
	scale .3
	rotate 5
      }
    }
    smooth
    translate <-.5,0,-.25>
    scale <400,50,400>
    texture {
      #local r = .55;
      pigment {Gold*1.15*r+OrangeRed*.15*r+White*.25*r}
      normal {granite .1 scale <.01,.01,.01>*600}
      finish {ambient 0.0 diffuse .95}
    }
  }
object {dunes}


// ========================================
//  Camera
// ========================================
#declare EyePos = <0,1000,-99>;
#declare EyeLook = <0,1000,0>;

#local Normal=<0, 0, 0>;
#local Where = trace (dunes, EyePos, <0, -1, 0>, Normal);
#if (vlength (Normal) != 0)
  #declare EyePos = Where+4*y;
#end
#local Normal=<0, 0, 0>;
#local Where = trace (dunes, EyeLook, <0, -1, 0>, Normal);
#if (vlength (Normal) != 0)
  #declare EyeLook = Where+15*y;
#end

camera {
  location EyePos
  look_at   EyeLook
  direction 1.5*z
  right     x*image_width/image_height
}



// ========================================
//  Ball
// ========================================
#declare BallPos = <-10,1000,-50>;
#local Normal=<0, 0, 0>;
#local Where = trace (dunes, BallPos, <0, -1, 0>, Normal);
#if (vlength (Normal) != 0)
  #declare BallPos = Where;
#end
sphere{
  0,3
  texture{
    pigment{
      radial
      color_map{
	[0.0 White]
	[0.5 White]
	[0.5 Red]
	[1.0 Red] 
      }
      frequency 6
    }
    finish {Shiny reflection{.05,.2}}
  }
  rotate <30,24,147>
  translate BallPos+3*y
}


// ========================================
//  Tree
//   from the MakeTree macro
//   (c) 1999 Gilles Tran http://www.oyonale.com
//=========================================
             
// ========================================
#declare TreePos = <0,1000,0>;
#local Normal=<0, 0, 0>;
#local Where = trace (dunes, TreePos, <0, -1, 0>, Normal);
#if (vlength (Normal) != 0)
  #declare TreePos = Where-1*y;
#end
#if (show_tree)
  #include "tree.inc"
  object {tree translate TreePos}
#end


// ========================================
//  Blobs in the tree
// ========================================
#declare n = 37;
#include "blob_pos.inc"
#declare i=0;
#declare _seed = seed(1);
#if (show_blobs)
  #while (i < (n-1))
    blobby (rand(_seed), EyePos, .75, 2, 4, _pos[i]+TreePos.y*y, <-_pos[i].y*1.5, (_pos[i].x+10)*2, 0>, <-10,0,-50>, <1,.5,0>, use_fast_blob)
    #declare i = i+1;
  #end
  blobby (rand(_seed), EyePos, .75, 2, 4, _pos[36]+TreePos.y*y, <0,0,0>, <30,70,-50>, <1,.5,0>, use_fast_blob)
#end

  

// ========================================
//  Weeds
//   by Jeremy Praay
//   Modified from "Example of use of the Makegrass macro by Gilles Tran - March 1999"
//   Modified by Jeremy Praay
//   requires Gilles Tran's mgrass.inc (available from www.oyonale.com)
// ========================================
#if (show_weeds)
  #include "Weeds.inc"

  #declare WeedPos = <0,1000,-5>;
  #local Normal=<0, 0, 0>;
  #local Where = trace (dunes, WeedPos, <0, -1, 0>, Normal);
  #if (vlength (Normal) != 0)
    #declare WeedPos = Where;
  #end
  object { 
    makeWeedPatch(10,10,15,0.3,3,10,35,0.6,0.3,seed(192))
    scale 2
    translate WeedPos
  }

  #declare WeedPos = <3,1000,-3>;
  #local Normal=<0, 0, 0>;
  #local Where = trace (dunes, WeedPos, <0, -1, 0>, Normal);
  #if (vlength (Normal) != 0)
    #declare WeedPos = Where;
  #end
  object { 
    makeWeedPatch(10,10,10,0.2,6,8,35,0.05,0.6,seed(1))
    scale 4
    translate WeedPos
  }

  #declare WeedPos = <1,1000,-5>;
  #local Normal=<0, 0, 0>;
  #local Where = trace (dunes, WeedPos, <0, -1, 0>, Normal);
  #if (vlength (Normal) != 0)
    #declare WeedPos = Where;
  #end
  object {
    makeWeedPatch(10,10,15,0.3,3,10,35,0.6,0.3,seed(10362))
    scale 3
    translate WeedPos
  }

  #declare WeedPos = <20,1000,-41>;
  #local Normal=<0, 0, 0>;
  #local Where = trace (dunes, WeedPos, <0, -1, 0>, Normal);
  #if (vlength (Normal) != 0)
    #declare WeedPos = Where;
  #end
  object {
    makeWeedPatch(5,5,20,0.3,3,10,35,0.6,0.3,seed(84132))
    scale 3
    Point_At_Trans(Normal)
    translate WeedPos
  }

  #declare WeedPos = <10,1000,-55>;
  #local Normal=<0, 0, 0>;
  #local Where = trace (dunes, WeedPos, <0, -1, 0>, Normal);
  #if (vlength (Normal) != 0)
    #declare WeedPos = Where;
  #end
  object {
    makeWeedPatch(10,10,40,0.8,3,10,35,0.6,0.3,seed(8941))
    scale 4
    Point_At_Trans(Normal)
    translate WeedPos
  }
#end
