//////////////////////////////////////////////////////////////////////////
// Entry.inc  Copyright(c) 2003, K.E. Ayers Designs                     //
//////////////////////////////////////////////////////////////////////////
// This file constructs the entire entry -- the door and the door       //
// frame with its side window panels.                                   //
//////////////////////////////////////////////////////////////////////////

// This file should only be included once!
#ifndef (_Entry_Inc_)
#declare _Entry_Inc = true;

#debug "\nParsing: Entry.inc\n"

#include "Textures.inc"
#include "SceneConstants.inc"
#include "Door.inc"
#include "PorchLight.inc"


// The door is the central element in the entry way.
#declare DoorFrameElemThick     = 2;
#declare DoorFrameElemDepth     = 4;
#declare DoorFrameClearance     = 1/4;

#declare ThresholdHgt           = 3/4;
#declare ThresholdDepth         = DoorFrameElemDepth;

#declare EntryStepHgt           = 3;
#declare SideWindowWid          = 18;

#declare InteriorFloorHgt       = -ThresholdHgt;


#declare FrameLeft              = -(3 * DoorFrameElemThick
                                        + DoorFrameClearance
                                        + SideWindowWid);
#declare FrameRight             = DoorWid 
                                        + 3 * DoorFrameElemThick
                                        + DoorFrameClearance
                                        + SideWindowWid;
#declare FrameBottom            = InteriorFloorHgt;
#declare FrameTop               = DoorHgt
                                    + DoorFrameClearance
                                    + DoorFrameElemThick;
#declare FrameBack              = (DoorFrameElemDepth - DoorThick) / 2;
#declare FrameFront             = FrameBack - DoorFrameElemDepth;

#declare EntryWid               = FrameRight - FrameLeft;

#declare FloorDepth             = 24;
#declare FloorThick             = 3/4;
#declare FloorBoardWidth        = 2;

#declare HallwayBoxDepth        = FloorDepth * 1.25;


// These macros create frame elements with the specified lengths.

#local Bev = 1/8;

#macro VerticalFrameElement(Left, Bot, Top, Back)
    intersection {
        box {
            < 0, Bot, -DoorFrameElemDepth >,
            < DoorFrameElemThick, Top, 0 >
        }
        plane {
            < -1, 0, 0 >, 0
            rotate < 0, -45, 0 >
            translate < 1/16, 0, -(DoorFrameElemDepth - Bev) >
        }
        plane {
            < -1, 0, 0 >, 0
            rotate < 0, 45, 0 >
            translate < 1/16, 0, -1/16 >
        }
        plane {
            < 1, 0, 0 >, 0
            rotate < 0, -45, 0 >
            translate < DoorFrameElemThick - Bev, 0, -Bev >
        }
        plane {
            < 1, 0, 0 >, 0
            rotate < 0, 45, 0 >
            translate < DoorFrameElemThick - Bev, 0, -(DoorFrameElemDepth - 1/16) >
        }
        translate < Left, 0, Back >
    }
#end

#macro HorizontalFrameElement(Left, Right, Bot, Back)
    intersection {
        box {
            < Left, 0, -DoorFrameElemDepth >,
            < Right, DoorFrameElemThick, 0 >
        }
        plane {
            < 0, 1, 0 >, 0
            rotate < 45, 0, 0 >
            translate < 0, DoorFrameElemThick - Bev, -Bev >
        }
        plane {
            < 0, 1, 0 >, 0
            rotate < -45, 0, 0 >
            translate < 0, DoorFrameElemThick - Bev, -(DoorFrameElemDepth - 1/16) >
        }
        plane {
            < 0, -1, 0 >, 0
            rotate < -45, 0, 0 >
            translate < 0, Bev, -Bev >
        }
        plane {
            < 0, -1, 0 >, 0
            rotate < 45, 0, 0 >
            translate < 0, Bev, -(DoorFrameElemDepth - Bev) >
        }
        translate < 0, Bot, Back >
    }
#end


// This is the frame around the door and side window panels.

#declare DoorFrame =
union {
    object {
        VerticalFrameElement(-(DoorFrameElemThick + DoorFrameClearance),
                             FrameBottom, FrameTop, FrameBack)
        texture { DoorFrameTexture rotate < 87, 0, 0 > }
    }
    object {
        VerticalFrameElement(-(2 * DoorFrameElemThick + DoorFrameClearance),
                             FrameBottom, FrameTop, FrameBack)
        texture { DoorFrameTexture rotate < 91, 0, 0 > }
    }
    object {
        VerticalFrameElement(DoorWid + DoorFrameClearance,
                             FrameBottom, FrameTop, FrameBack)
        texture { DoorFrameTexture rotate < 93, 0, 0 > }
    }
    object {
        VerticalFrameElement(DoorWid
                                + DoorFrameElemThick
                                + DoorFrameClearance,
                             FrameBottom, FrameTop, FrameBack)
        texture { DoorFrameTexture rotate < 93, 0, 0 > }
    }
    object {
        VerticalFrameElement(FrameLeft,
                             FrameBottom + DoorFrameElemThick,
                             FrameTop, FrameBack)
        texture { DoorFrameTexture rotate < 87, 0, 0 > }
    }
    object {
        VerticalFrameElement(FrameRight - DoorFrameElemThick,
                             FrameBottom + DoorFrameElemThick,
                             FrameTop, FrameBack)
        texture { DoorFrameTexture rotate < 91, 0, 0 > }
    }
    object {
        HorizontalFrameElement(FrameLeft, -DoorFrameClearance,
                               FrameBottom, FrameBack)
        texture { DoorFrameTexture rotate < 0, 87, 0 > }
    }
    object {
        HorizontalFrameElement(DoorWid + DoorFrameClearance, FrameRight,
                               FrameBottom, FrameBack)
        texture { DoorFrameTexture rotate < 0, 87, 0 > }
    }
    object {
        HorizontalFrameElement(FrameLeft, FrameRight,
                               FrameTop - DoorFrameElemThick, FrameBack)
        texture { DoorFrameTexture rotate < 0, 87, 0 > }
    }
}


// The side window panels will have a "shadow box" behind
// krinkled glass, to give the impression of a hallway
// beyond the door.

#declare HallwayBoxColor        = color rgb<  1.0,  0.9, 0.8 >;
#declare GlassColor             = color rgbf< 0.95, 1.0, 0.95, 1.0 >;
#declare HallBoxLeft            = -SideWindowWid / 2;
#declare HallBoxRight           = SideWindowWid / 2 + DoorFrameElemDepth + 1/2;
#declare HallBoxBottom          = FrameBottom;
#declare HallBoxTop             = FrameTop;
#declare HallBoxFront           = 0;
#declare HallBoxBack            = HallwayBoxDepth;
#declare HallwayBoxWid          = HallBoxRight - HallBoxLeft;

#declare HallTableWid           = HallwayBoxWid * 0.6;
#declare HallTableHgt           = HallTableWid * 2.5;
#declare HallTableDepth         = HallTableWid * 0.75;
#declare HallTableTopHgt        = HallTableHgt * 0.25;
#declare HallTableRight         = HallBoxRight - 1;
#declare HallTableLeft          = HallTableRight - HallTableWid;
#declare HallTableBot           = HallBoxBottom;
#declare HallTableTop           = HallBoxBottom + HallTableHgt;
#declare HallTableBack          = HallBoxBack - 1;
#declare HallTableFront         = HallTableBack - HallTableDepth;
#declare HallTableTopBot        = HallTableTop - HallTableTopHgt;
#declare HallTableLegLen        = HallTableHgt - HallTableTopHgt;
#declare HallTableLegTopRad     = HallTableWid * 0.125;
#declare HallTableLegBotRad     = HallTableLegTopRad * 0.5;

#declare HallLampHgt            = HallTableHgt * 0.75;
#declare HallLampShadeHgt       = HallLampHgt / 2;
#declare HallLampShadeBotRad    = HallTableDepth * 0.6;
#declare HallLampShadeTopRad    = HallLampShadeBotRad * 0.4;
#declare HallLampBaseHgt        = HallLampHgt - HallLampShadeHgt;
#declare HallLampBaseBotRad     = HallLampShadeTopRad;
#declare HallLampBaseTopRad     = HallLampBaseBotRad * 0.5;


#declare HallTableLeg =
cone {
    < 0, HallTableBot, 0 >,    HallTableLegBotRad,
    < 0, HallTableTopBot, 0 >, HallTableLegTopRad
    texture { FinishedDarkOak rotate 90 * x }
}

#declare HallTableTopShape =
box {
    < HallTableLeft,  HallTableTopBot, HallTableFront >,
    < HallTableRight, HallTableTop,    HallTableBack >
    texture { FinishedDarkOak rotate 90 * y }
}

#declare HallLampBase =
cone {
    < 0, 0, 0 >, HallLampBaseBotRad,
    < 0, HallLampBaseHgt, 0 >, HallLampBaseTopRad
    texture { SatinBrass }
}

#declare HallLampShade =
cone {
    < 0, HallLampBaseHgt, 0 >, HallLampShadeBotRad,
    < 0, HallLampHgt,     0 >, HallLampShadeTopRad
    texture {
        pigment { color rgb < 0.8, 0.4, 0.2 > }
        finish { ambient 0.8 diffuse 0 }
    }
}

#declare HallTableLamp =
union {
    object { HallLampBase }
    object { HallLampShade }
    translate < (HallTableLeft + HallTableRight) / 2,
                HallTableTop,
                (HallTableBack + HallTableFront) / 2 >
}


#declare HallTable =
union {
    object { HallTableTopShape }
    #object {
        HallTableLeg
        translate < HallTableLeft + 1, 0, HallTableBack - 1 >
    }
    object {
        HallTableLeg
        translate < HallTableRight - 1, 0, HallTableBack - 1 >
    }
    object {
        HallTableLeg
        translate < HallTableLeft + 1, 0, HallTableFront + 1 >
    }
    object {
        HallTableLeg
        translate < HallTableRight - 1, 0, HallTableFront + 1 >
    }
    object { HallTableLamp }
}

#declare HallwayBox =
union {
    box {
        < HallBoxLeft, HallBoxBottom, HallBoxFront >,
        < HallBoxLeft + (1/8), HallBoxTop, HallBoxBack >
    }
    box {
        < HallBoxRight - (1/8), HallBoxBottom, HallBoxFront >,
        < HallBoxRight, HallBoxTop, HallBoxBack >
    }
    box {
        < HallBoxLeft,  HallBoxBottom, HallBoxBack - (1/8) >,
        < HallBoxRight, HallBoxTop,    HallBoxBack >
    }
    box {
        < HallBoxLeft,  HallBoxTop - (1/8), HallBoxFront >,
        < HallBoxRight, HallBoxTop,         HallBoxBack >
    }
    pigment { HallwayBoxColor }
}

#local SideWinThick     = 1/4;
#local SideWinLeft      = -SideWindowWid / 2 - 1/2;
#local SideWinRight     =  SideWindowWid / 2 + 1/2;
#local SideWinTop       =  FrameTop + 1/2;
#local SideWinBottom    =  FrameBottom - 1/2;
#local SideWindowHgt    =  SideWinTop - SideWinBottom;
#local DentScale        =  SideWindowWid * 0.02;

#if (UseDimpledGlass)
    #declare SideWindowGlass =
    union {
        box {
            < SideWinLeft,  SideWinBottom, -SideWinThick / 2 >,
            < SideWinRight, SideWinTop   ,  SideWinThick / 2 >
            texture {
                pigment { GlassColor }
                normal  { dents 1.0 scale < DentScale, DentScale, 2.0 > }
                finish  {
                    ambient     0
                    diffuse     0
                    specular    1.0
                    reflection  0.5
                }
            }
            interior { ior 1.75 } 
        }

        #local RND  = seed(0);
        #local D    = 3/4;
        #local R    = D / 2;
        #local NX   = SideWindowWid / D;
        #local NY   = SideWindowHgt / D;
        #local NS   = NX * NY;
        #local X0   = (SideWinLeft + SideWinRight) / 2;
        #local Y0   = (SideWinBottom + SideWinTop) / 2;
        #local XS   = SideWindowWid / 2;
        #local YS   = SideWindowHgt / 2;
        #local S    = sphere { < 0, 0, 0 >, R pigment { Clear } interior { ior 1.1 } }
        #local Z    = R + (SideWinThick / 2);

        #while (NS > 0)
            #local X = rand(RND) * XS * 2 - XS + X0;
            #local Y = rand(RND) * YS * 2 - YS + Y0;

            object { S translate < X, Y, Z > }

            #local NS = NS - 1;
        #end
    }
#end

#declare SideWindowPanel =
union {
    object { HallwayBox }

    #if (UseDimpledGlass)
        object { SideWindowGlass translate < 0, 0, -DoorFrameElemDepth / 2 > }
        light_source {
            < (HallBoxLeft + HallBoxRight) / 2,
              HallBoxTop - 1/2,
              HallBoxFront + 1/2 >
            color rgb < 0.7, 0.4, 0.2 >
            spotlight
            point_at < (HallBoxLeft + HallBoxRight) / 2,
                       (HallBoxBottom + HallBoxTop) * 0.25,
                       HallwayBoxDepth / 2 >
            radius  80
            falloff 90
        }
    #end

    object { HallTable }
}


// There is a hardwood floor extending a short distance
// into the entry hall.

#declare Threshold =
intersection {
    cylinder {
        < -DoorFrameClearance, 0, 0 >,
        <  DoorWid + DoorFrameClearance, 0, 0 >,
        ThresholdHgt
    }
    plane { < 0, -1, 0 >, 0 }
    texture { BrushedAluminum }
    scale < 1, 1, (ThresholdDepth / 2) / ThresholdHgt >
    translate < 0, -ThresholdHgt, - ThresholdDepth / 2 >
}


#local FloorBoardLen = FloorDepth * 3.0;

#declare FloorBoardTexture =
texture {
    gradient z
    turbulence 0.05
    texture_map {
        [ 0.0   FinishedOak rotate < 3, 0, 0 > ]
        [ 0.4   FinishedOak rotate < 3, 0, 0 > ]
        [ 0.9   pigment { color rgbf < 1.0, 1.0, 1.0, 1.0 > } ]
        [ 1.01  pigment { color rgbf < 1.0, 1.0, 1.0, 1.0 > } ]
    }  
}

#declare FloorBoard =
box {
    < 0, -(ThresholdHgt + FloorThick), 0 >,
    < FloorBoardWidth - 1/8, -ThresholdHgt, 0.9 >
}

#declare Floor =
union {
    #local XOfs = FrameLeft - FloorBoardWidth;

    #while (XOfs <= (FrameRight + FloorBoardWidth))
        object { FloorBoard translate < XOfs, 0, 0 > }
        #local XOfs = XOfs + FloorBoardWidth;
    #end

    texture { FloorBoardTexture }
    scale < 1, 1, FloorBoardLen / 0.9 >
}


#declare AddressText    = "2 Some Place"
#declare AddressFont    = "cyrvetic.ttf"
#declare AddressThick   = 1/2;

#declare AddressPlaque =
text {
    ttf AddressFont,
    AddressText,
    AddressThick,
    0.0
}

#declare AddressPlaqueTexture =
texture {
    pigment { Black }
    finish {
        ambient     Ambient
        diffuse     Diffuse
        specular    1
        roughness   0.001
    }
    normal { bumps 1.0 scale < 0.01, 0.01, 1.0 > }
}


//////////////////////////////////////////////////////////////
// These values need to be adjusted if the text string or   //
// font is changed.                                         //

#declare AddressXScale  = DoorWid * strlen(AddressText) * 0.01;
#declare AddressYScale  = AddressXScale * 1.0;
#declare AddressXOfs    = DoorXOfs
                            - (DoorWid / 2)
                            + ((strlen(AddressText) / AddressXScale) * 2.5);
#declare AddressYOfs    = DoorYOfs + DoorHgt + (WallHgtAboveDoor * 0.325);
#declare AddressZOfs    = DoorZOfs
                            - (DoorThick * 1.1)
                            - SidingThick
                            - AddressThick;
//////////////////////////////////////////////////////////////


// This is the composite entry way.

#declare EntryWay =
union {
    object { Door }
    object { Floor }
    object { Threshold }
    object { DoorFrame }
    object {
        SideWindowPanel
        translate < FrameRight
                        - SideWindowWid / 2
                        - DoorFrameElemThick,
                    0, 0 >
    }
    object {
        SideWindowPanel
        scale < -1, 1, 1 >
        translate < FrameLeft
                        + SideWindowWid / 2
                        + DoorFrameElemThick,
                    0, 0 >
    }
    object {
        AddressPlaque
        rotate 7.5 * x
        scale < AddressXScale, AddressYScale, 1 >
        texture { AddressPlaqueTexture }
        translate < AddressXOfs, AddressYOfs, AddressZOfs >
    }
    object {
        PorchLight
        translate < FrameRight + (PorchLampBaseRad * 2),
                    DoorYOfs + (DoorHgt * 0.75),
                    DoorZOfs - DoorThick - SidingThick >
    }
    object {
        PorchLight
        translate < FrameLeft - (PorchLampBaseRad * 2),
                    DoorYOfs  + (DoorHgt * 0.75),
                    DoorZOfs  - DoorThick - SidingThick >
    }
}

#end    // _Entry_Inc_

