//////////////////////////////////////////////////////////////////////////
// PicketFence.inc  Copyright(c) 2003, K.E. Ayers Designs               //
//////////////////////////////////////////////////////////////////////////
// This file constructs the picket fence and flower beds.               //
//////////////////////////////////////////////////////////////////////////

// This file should only be included once!
#ifndef (_PicketFence_Inc_)
#declare _PicketFence_Inc_ = true;

#debug "\nParsing: PicketFence.inc\n"

#include "SceneConstants.inc"
#include "Textures.inc"
#include "Plants.inc"

#local FenceTexture =
texture {
    pigment { White }
    finish { ambient Ambient diffuse Diffuse }
}


// Create the picket fence.
#local FenceHgt         = 48;
#local FenceWid         = 600;
#local GateWid          = 39;
#local FenceBoardWid    = 3;
#local FenceBoardThick  = 3/4;
#local FenceBoardSep    = 6;
#local FenceLowRailHgt  = 12;
#local FenceTopRailHgt  = 36;
#local FenceRailLen     = (FenceWid - GateWid) / 2;
#local FencePostHgt     = FenceTopRailHgt + FenceBoardWid / 2;
#local FencePostWid     = 4;
#local FencePostSep     = 120;


#local FenceBoard =
intersection {
    box {
        < 0, 0, 0 >,
        < FenceBoardWid, FenceHgt, FenceBoardThick >
    }
    plane {
        < -1, 0, 0 >, 0
        rotate -45 * z
        translate < FenceBoardWid / 2, FenceHgt, 0 >
    }
    plane {
        < 1, 0, 0 >, 0
        rotate 45 * z
        translate < FenceBoardWid / 2, FenceHgt, 0 >
    }
    texture { FenceTexture }
}

#local FenceRail =
box {
    < GateWid / 2,  -FenceBoardWid / 2, -FenceBoardThick >,
    < FenceRailLen,  FenceBoardWid / 2,  0 >
    texture { FenceTexture }
}

#local FencePost =
box {
    < 0, 0, -FencePostWid >,
    < FencePostWid, FencePostHgt, 0 >
    texture { FenceTexture }
}

#local PicketFenceSection =
union {
    // Set the posts first.
    #local X = GateWid / 2;

    #while (X <= FenceRailLen)
        object { FencePost translate X * x }
        #local X = X + FencePostSep;
    #end

    // Attach the horizontal rails.
    object { FenceRail translate FenceLowRailHgt * y }
    object { FenceRail translate FenceTopRailHgt * y }

    // Attach the picket boards.
    #local X = GateWid / 2;

    #while (X <= FenceRailLen)
        object { FenceBoard translate X * x }
        #local X = X + FenceBoardSep;
    #end
}

#local GateHorizRail =
box {
    < -GateWid / 2,  -FenceBoardWid / 2, -FenceBoardThick >,
    <  GateWid / 2,   FenceBoardWid / 2,  0 >
    texture { FenceTexture }
}

#local GateVertRail =
box {
    < -FenceBoardWid / 2,
       FenceLowRailHgt - FenceBoardWid / 2,
      -FenceBoardThick >,
    <  FenceBoardWid / 2,
       FenceTopRailHgt + FenceBoardWid / 2,
       0 >
    texture { FenceTexture }
}

#macro GateBoard(Wid, Hgt, Thick)
    intersection {
        box { < -Wid / 2, 0, 0 >, <Wid / 2, Hgt, Thick > }
        plane {
            < -1, 0, 0 >, 0
            rotate -45 * z
            translate Hgt * y
        }
        plane {
            < 1, 0, 0 >, 0
            rotate 45 * z
            translate Hgt * y
        }
        texture { FenceTexture }
    }

#end    // #macro GateBoard

#local GateAngle = 45;

#local GateHingePlate =
box {
    < 0, -3, -3 >, < 1/8, 3, 0 >
    pigment { Black }
}
        
#local Gate =
union {
    object {
        GateVertRail
        translate ((GateWid - FenceBoardWid) / 2) * x
    }
    object {
        GateVertRail
        translate -((GateWid - FenceBoardWid) / 2) * x
    }
    object {
        GateHorizRail
        translate FenceLowRailHgt * y
    }
    object {
        GateHorizRail
        translate FenceTopRailHgt * y
    }
    object {
        GateHingePlate
        translate < GateWid / 2, FenceLowRailHgt, 0 >
    }
    object {
        GateHingePlate
        translate < GateWid / 2, FenceTopRailHgt, 0 >
    }

    #local DHgt = 9;
    #local X    = 0;

    #while (X < GateWid / 2)
        object {
            GateBoard(FenceBoardWid, FenceHgt + DHgt, FenceBoardThick)
            translate X * x
        }

        #if (X > 0)
            object {
                GateBoard(FenceBoardWid, FenceHgt + DHgt, FenceBoardThick)
                translate -X * x
            }
        #end

        #local DHgt = DHgt / 2;
        #local X    = X + FenceBoardSep;
    #end
}


#local FlowerBedDepth   = 24;
#local FlowerBedHgt     = 2;
#local FlowerBedTexture =
texture {
    pigment {
        granite turbulence 1.25
        color_map {
            [ 0.0   color rgb < 0.2, 0.1, 0.0 > ]
            [ 0.3   color rgb < 0.2, 0.1, 0.0 > ]
            [ 0.4   color rgb < 0.4, 0.2, 0.1 > ]
            [ 0.5   color rgb < 0.2, 0.1, 0.0 > ]
            [ 0.7   color rgb < 0.2, 0.1, 0.0 > ]
            [ 0.8   color rgb < 0.4, 0.2, 0.1 > ]
            [ 0.9   color rgb < 0.2, 0.1, 0.0 > ]
        }
    }
    finish { ambient Ambient diffuse Diffuse }
}

#local FloweringBushHgt       = 24;
#local FloweringBushStemLen   = 12;

#local LeafShapes =
array[2] { object { GreenLeaf scale 4 }, object { LightGreenLeaf scale 4 } }

#local FlowerShapes =
array[1] { object { RedImpatienFlower scale 2.0 } }

#local BackyardBush =
object {
    LeafyBush
    (
        // StalkHeight
        FloweringBushHgt,
        // StalkBaseRad
        1/4,
        // StalkEndRad
        1/16,
        // StemHeight
        4,
        // StemLength
        FloweringBushStemLen,
        // StemLengthScale
        0.85,
        // StemRad
        1/8,
        // StemSpacing
        20,
        // StemAngle
        60,
        // StemAngleScale
        0.9,
        // LeafAngle
        70,
        // LeafAngleScale
        1.1,
        // Rings
        7,
        // RingStagger
        10,
        // TheLeaf
        LeafShapes,
        // TheFlower
        FlowerShapes,
        // FlowerFrequency
        0.40,
        // Slop
        0.30,
        // StalkTexture
        texture { pigment { color rgb < 0.2, 0.6, 0.3 > } }
    )
}

#local Rnd = seed(123);

#local FlowerBed =
union {
    box {
        < GateWid / 2 + 6, 0, -FlowerBedDepth >,
        < FenceRailLen, FlowerBedHgt, 0 >
        texture { FlowerBedTexture }
    }

    #local X = (GateWid / 2) + FloweringBushStemLen * 2;

    #while (X < FenceRailLen)
        #local Rot = rand(Rnd) * 360;
        #local Scl = 1.1 - rand(Rnd) * 0.2;

        object {
            BackyardBush
            rotate Rot * y
            scale < 1, Scl, 1 >
            translate < X, FlowerBedHgt, -FlowerBedDepth / 2 >
        }

        #local X = X + FloweringBushStemLen * 3;
    #end
}


#declare PicketFence =
union {
    object { PicketFenceSection }
    object { PicketFenceSection scale < -1, 1, 1 > }
    object {
        GateHingePlate
        translate < (GateWid / 2) - 1/8, FenceLowRailHgt, 0 >
    }
    object {
        GateHingePlate
        translate < (GateWid / 2) - 1/8, FenceTopRailHgt, 0 >
    }
    object {
        Gate
        translate -(GateWid / 2) * x
        rotate 45 * y
        translate (GateWid / 2) * x
    }
    object {
        FlowerBed
        translate -FencePostWid * z
    }
    object {
        FlowerBed
        scale < -1, 1, 1 >
        translate -FencePostWid * z
    }
}


#end    // #ifndef (_PicketFence_Inc_)

