// Persistence of Vision Ray Tracer Scene Description File
// File: cake.pov
// Vers: 3.5
// Desc: Scene file for "Birthday in the Clouds" - IRTC Surrealism round, July-August 2003
// Date: August 24, 2003
// Auth: Philip Chan

// +W768 +H768 +a0.3 +am2

#include "colors.inc"
#include "skies.inc"
#include "finish.inc"
#include "woods.inc"
#include "metals.inc"

#include "balloon.inc"

// quality settings
#declare use_area_light = true;  // use area light
#declare C_lights = true;  // use light sources on candles
#declare place_sprinkles = true;  // place sprinkles on the cake

// global settings
global_settings {	assumed_gamma 1 }

// background gradient
sky_sphere {
	pigment { 
		gradient y
		colour_map {
			[0.00 rgb <0.35, 0.75, 1.0>]
			[0.20 rgb <0.25, 0.65, 1.0>]
			[0.90 rgb <0.25, 0.65, 1.0>]
			[1.00 rgb <0.35, 0.75, 1.0>]
		}
	}
}

// Clouds (O_Cloud2 from skies.inc, modified slightly to work while looking down)
union {
    plane { y, -500
        texture {
            T_Cloud3
            scale 600
        }
    }
    plane { y, -3000
        texture {
            T_Cloud2
            scale <900,1,6000>
            translate x*3000
            rotate -30*y
        }
    }
}

light_source {
	<0, 100, -100>
	colour White
	#if ( use_area_light )
		area_light <4, 0, 0>, <0, 4, 0>, 3, 3
		adaptive 1
		jitter
	#end
}

camera {
	location <0, 16, -18>
	up y
	right x
	look_at <0, 1, 5>
}

//----- Rainbow -----\\
// declare rainbow's colors
#declare r_violet1 = color rgbft<1.0, 0.5, 1.0, 1.0, 0.75>;
#declare r_violet2 = color rgbft<1.0, 0.5, 1.0, 0.8, 0.75>;
#declare r_indigo  = color rgbft<0.5, 0.5, 1.0, 0.8, 0.75>;
#declare r_blue    = color rgbft<0.2, 0.2, 1.0, 0.8, 0.75>;
#declare r_cyan    = color rgbft<0.2, 1.0, 1.0, 0.8, 0.75>;
#declare r_green   = color rgbft<0.2, 1.0, 0.2, 0.8, 0.75>;
#declare r_yellow  = color rgbft<1.0, 1.0, 0.2, 0.8, 0.75>;
#declare r_orange  = color rgbft<1.0, 0.5, 0.2, 0.8, 0.75>;
#declare r_red1    = color rgbft<1.0, 0.2, 0.2, 0.8, 0.75>;
#declare r_red2    = color rgbft<1.0, 0.2, 0.2, 1.0, 0.75>;
// create the rainbow
rainbow {
  angle 25
  width 6.5
  distance 200
  direction <0.563, -0.73, 1>
  jitter 0.01
  arc_angle 200
  falloff_angle 160
  color_map {
    [0.000  color r_violet1]
    [0.100  color r_violet2]
    [0.214  color r_indigo]
    [0.328  color r_blue]
    [0.442  color r_cyan]
    [0.556  color r_green]
    [0.670  color r_yellow]
    [0.784  color r_orange]
    [0.900  color r_red1]
  }
}
//----- End Rainbow -----\\

//----- Glaze Flower -----\\
#declare M_Flower_Centre = material {
	texture {
		pigment { colour rgbft <1.2, 1.2, 0.3, 0.6, 0.4> }
		finish {
			specular 0.7
			roughness 0.1
		}
		normal { bumps 0.3 scale 0.1 }
	}
	interior {
		ior 1.5
	}
}

#declare M_Flower_Petal = material {
	texture {
		pigment { colour rgbft <1.0, 0.50, 0.80, 0.6, 0.4> }
		finish {
			specular 0.7
			roughness 0.1
		}
		normal { bumps 0.3 scale 0.1 }
	}
	interior {
		ior 1.5
	}
}

#declare Glaze_Flower = union {
	sphere { <0, 0, 0>, 1
		scale <0.25, 0.075, 0.25>
		material { M_Flower_Centre }
	}	
	difference {
		merge {
			#declare i = 0;
			#while ( i < 360 )
				sphere { <0, 0, 0>, 1
					scale <0.3, 0.075, 0.5>
					translate <0, 0, 0.6>
					rotate y*i
				}
				#declare i = i + 60;
			#end
		}
		sphere { <0, 0, 0>, 1
			scale <0.3, 0.15, 0.3>
			translate <0, 0.075, 0>
		}
		material { M_Flower_Petal }
	}
}
//----- End Glaze Flower -----\\

//----- Icing Flowers -----\\
#declare CF_Rotations = 6;
#declare CR_Rotations = 7;

#declare Cake_Flower = isosurface {
	function { (0.75 - 0.25*y*(y < 0) )*sqrt( x*x + z*z ) + (0.125 + 0.125*y*(y < 0) - (0.125 + 0.125*y*(y < 0) )*sin( CF_Rotations*( atan2( z, x ) + y ) ) ) + (y > 0)*pow( (y/2), 3 ) }
	contained_by { box { <-2, -1, -2>, <2, 2, 2> } }
	threshold 1
	max_gradient 3
	translate <0, 1, 0>
	pigment { colour Cyan }
}

#declare Cake_Rim = isosurface {
	function { 0.75*sqrt( pow(y - 0.3*x*(x < 0), 2) + z*z + x*x*(x > 0) ) + (0.125 - 0.125*sin( CR_Rotations*( atan2( z, y ) ) ) ) }
	contained_by { box { <-4, 0, -2>, <2, 2, 2> } }
	threshold 1
	max_gradient 3
	translate <0, 1, 0>
	pigment { colour Cyan }
}
//----- End Icing Flowers -----\\

//----- Cake Body -----\\
#declare Cake = union {
	cylinder { <0, 0, 0>, <0, 2, 0>, 4.9 }
	cylinder { <0, 0, 0>, <0, 1.9, 0>, 5 }
	// smooth out edge
	torus { 4.9, 0.1
		translate <0, 1.9, 0>
	}
	texture {
		pigment { colour White }
		finish {
			diffuse 0.8
			brilliance 0.5
			specular 0.7
			roughness 0.02
		}
	}
}
//----- End Cake Body -----\\

//----- Sprinkles -----\\
// sprinkle object
#declare Sprinkle = union {
	sphere { <0, 0, 0>, 1
		scale 0.025
		translate <0, 0, -0.05>
	}
	cylinder { <0, 0, -0.05>, <0, 0, 0.05>, 0.025 }
	sphere { <0, 0, 0>, 1
		scale 0.025
		translate <0, 0, 0.05>
	}
}

// sprinkle placing algorithm
#if ( place_sprinkles )
	// points to check for intersection with other sprinkles
	#declare S_points = array[15] {
		<5.05, 0, -0.075>,
		<5.05, 0, -0.0375>,
		<5.05, 0, 0>,
		<5.05, 0, 0.0375>,
		<5.05, 0, 0.075>,
		<5.05, 0.025, -0.075>,
		<5.05, 0.025, -0.0375>,
		<5.05, 0.025, 0>,
		<5.05, 0.025, 0.0375>,
		<5.05, 0.025, 0.075>,
		<5.05, -0.025, -0.075>,
		<5.05, -0.025, -0.0375>,
		<5.05, -0.025, 0>,
		<5.05, -0.025, 0.0375>,
		<5.05, -0.025, 0.075>,
	}
	
	#debug "Placing sprinkles:\n"
	#declare S_rand = seed(1989);
	#declare thresh = 5.01;  // distance from centre of cake to stop placing sprinkles
	#declare i = 1;
	#while ( ( i < 1500 ) )
		// choose colour
		#switch( rand(S_rand) )
			#range (0, 1/5)
				#declare S_colour = colour Red;
			#break
			#range (1/5, 2/5)
				#declare S_colour = colour rgb <1, 1, 0>;
			#break
			#range (2/5, 3/5)
				#declare S_colour = colour Blue;
			#break
			#range (3/5, 4/5)
				#declare S_colour = colour Green;
			#break
			#range (4/5, 1)
				#declare S_colour = colour <1.4, 0.5, 0.8>;
			#break
		#end
		// get random rotation/translation information
		#declare xrotate = 360*rand(S_rand);
		#declare yrotate = 180*rand(S_rand);
		#declare ytrans = 0.3 + 1.6*rand(S_rand);
		// check for intersection with other sprinkles
		#declare place = true;  // start condition for checks is no intersection
		#declare j = 0;
		#while ( (j < 15) & place )  // loop while not all points checked, and one hasn't already failed
			// apply transformations to a point
			#declare Pos = vrotate( S_points[j], x*xrotate );
			#declare Pos = Pos + y*ytrans;
			#declare Pos = vrotate( Pos, y*yrotate );
			// find out the distance to the centre of the cake
			#declare hit = trace( Cake, Pos, <-Pos.x, Pos.y, -Pos.z> );
			#declare dist = vlength( hit - y*ytrans );
			// check threshold
			#if ( dist > thresh )
				#declare place = false;
			#end
			#declare j = j + 1;
		#end
		// place sprinkle (adding it to the Cake object used in the call to trace)
		#if ( place )
			#declare Cake = union {
				object { Cake }
				object { Sprinkle
					rotate y*5*( rand(S_rand) - 0.5 )
					rotate x*xrotate
					translate <5.02, ytrans, 0>
					rotate y*yrotate
					pigment { colour S_colour }
				}
			}
			#declare i = i + 1;  // increment number of sprinkles placed
			// debug output to view progress
			#if ( mod(i, 50) = 0 )
				#debug concat( "i is: ", str(i, 0, 0), "\n" )
			#end
		#end
	#end
#end
//----- End Sprinkes -----\\

// cake object
object { Cake }

//----- Place Icing Flowers -----\\
#declare CF_rand = seed(33984);
#declare theta = 0;
#while ( theta < 2*pi )
	object { Cake_Flower
		scale <1/6, 1/7, 1/6>
		rotate y*360*rand( CF_rand )
		translate <4.75*sin( theta ), 1.95, 4.75*cos( theta )>
	}
	#declare theta = theta + pi/30;
#end

#declare theta = 0;
#while ( theta < 360 )
	object { Cake_Rim
		scale 1/6
		translate <0, 0, -5>
		rotate y*theta
	}
	#declare theta = theta + 7.5;
#end
//----- End of Icing Flower Placement -----\\

//----- Text HF -----\\
// material for text
#declare M_Text = material {
	texture {
		pigment { colour rgbf <0.3, 0.8, 1.0, 0.7> }
		finish {
			specular 0.7
			roughness 0.1
		}
		normal { bumps 0.5 scale 0.2 }
	}
	interior {
		ior 1.5
	}
}

// "Happy Birthday"
height_field {
	png
	"bday_hf.png"
	scale <5, 0.075, 3.92>
	translate <-2.5, 1.99, -2.5>
	material { M_Text }
}
//----- End Text HF ----\\

//----- Glaze Flower Placement -----\\
union {
	#declare GF_rand = seed(81627);
	#declare GF_scale = array[15] {
		1, // below centre
		0.35, // below 'da'
		0.4, // below 'ay'
		0.5, // below-right 'y'
		0.65, // right 'ay'
		0.6, // right 'py'
		0.4, // above-right 'py'
		0.7, // above 'py'
		0.8, // above 'ap'
		0.5, // above 'H'
		0.55, // left 'H'
		0.8, // left 'B'
		0.6, // below-left 'B'
		0.4, // far below 'Bi'
		0.35, // below 'Bi'
	}
	#declare GF_trans = array[15] {
		<-0.25, 2, -3.0>, // below centre
		<1.1, 2, -2.5>, // below 'da'
		<1.5, 2, -3.4>, // below 'ay'
		<2.5, 2, -2.75>, // below-right 'y'
		<3.4, 2, -1.2>, // right 'ay'
		<2.4, 2, 0>, // right 'py'
		<2.2, 2, 1.1>, // above-right 'py'
		<1.3, 2, 2>, // above 'py'
		<-0.4, 2, 2.45>, // above 'ap
		<-1.9, 2, 1.85>, // above 'H'
		<-2.4, 2, 0.4>, // left 'H'
		<-3.3, 2, -1.2>, // left 'B'
		<-2.5, 2, -2.7>, // below-left 'B'
		<-1.6, 2, -3.5>, // far below 'Bi'
		<-1.5, 2, -2.3>, // below 'Bi'
	}
	
	#declare i = 0;
	#while ( i < 15 )
		object { Glaze_Flower
			scale GF_scale[i]
			rotate y*360*rand( GF_rand )
			translate GF_trans[i]
		}
		#declare i = i + 1;
	#end
}
//----- End Flower Placement -----\\

//----- Glaze Rainbow -----\\
// Rainbow Colours
#declare C_Rainbow = array[7] {
	colour rgbft <1.000, 0.000, 0.000, 0.6, 0.4>,  // R
	colour rgbft <1.200, 0.300, 0.000, 0.6, 0.4>,  // O
	colour rgbft <1.000, 1.000, 0.000, 0.6, 0.4>,  // Y
	colour rgbft <0.000, 1.000, 0.000, 0.6, 0.4>,  // G
	colour rgbft <0.000, 0.000, 1.000, 0.6, 0.4>,  // B
	colour rgbft <0.500, 0.000, 0.866, 0.6, 0.4>,  // I
	colour rgbft <0.808, 0.442, 0.808, 0.6, 0.4>   // V
}
	
// rainbow (light group is for highlights)
light_group {
	light_source {
		<0, 5, 0>
		colour <0.5, 0.5, 0.5>
	}
	light_source {
		<4.5, 5, 1>
		colour White
	}
	light_source {
		<-4.5, 5, 1>
		colour White
	}
	
	#declare i = 0;
	#while ( i < 7 )
		merge {
			difference {
				torus { 4.25 - 0.1*i, 0.05
					scale <1, 1.5, 1>
					translate <0, 2, 0>
				}
				plane { z, 0 }
			}
			sphere { <0, 0, 0>, 0.05
				scale <1, 1.5, 1>
				translate <-4.25 + 0.1*i, 2, 0>
			}
			sphere { <0, 0, 0>, 0.05
				scale <1, 1.5, 1>
				translate <4.25 - 0.1*i, 2, 0>
			}
			material {
				texture {
					pigment { colour C_Rainbow[i] }
					finish {
						specular 0.7
						roughness 0.025
					}
					normal { bumps 0.15 scale 0.2 }
				}
				interior {
					ior 1.5
				}
			}
		}
		#declare i = i + 1;
	#end
	
	global_lights off
}
//----- End Glaze Rainbow -----\\

//----- Candles -----\\
// candle flame macro
#declare CFRand = seed(39857);
#macro C_Flame ( scaling, translation )
	merge {
		sphere { <0, 0, 0>, 2.0 }
		difference {
			sphere { <0, 0, 0>, 2.0 
				scale <1, 2, 1>
				translate <0, 0, 0>
			}
			plane { y, 0 }
		}
		hollow
		pigment { rgbt <1, 1, 1, 1> }
		interior {
			media {
				emission 1
				intervals 50
				density {
					spherical
					turbulence 0.2
					colour_map {
						[0.00 Clear ]
						[0.10 Clear ]
						[0.30 Orange*0.5/scaling ]
						[0.50 Orange*0.8/scaling ]
						[0.80 Orange/scaling ]
						[0.90 Orange*1.5/scaling ]
						[1.00 Yellow/scaling ]
					}
					scale <2.0, 4, 2.0>
					translate <0, -0.5, 0> + <4, 0, 0> * int(rand(CFRand))*100
				}
			}
		}
		scale scaling
		translate translation
	}
#end

// White swirl on candles
#declare Candle_Swirl = sphere_sweep {
	b_spline
	14,
	#declare i = -1;
	#while ( i < 13 )
		<0.075*cos(i*pi/4), 0 + 1/88*i, 0.075*sin(i*pi/4)>, 0.01
		#declare i = i + 1;
	#end
	scale <1, 8, 1>
}

// Candle
#declare Candle = union {
	// body
	cylinder { <0, 0, 0>, <0, 1, 0>, 0.075 }
	// wick
	cylinder { <0, 1, 0>, <0, 1.1, 0>, 0.01
		pigment { colour Black }
	}
	// swirls
	intersection {
		union {
			#declare i = 0;
			#while ( i < 360 )
				object { Candle_Swirl
					rotate y*i
				}
				#declare i = i + 90;
			#end
		}
		cylinder { <0, 0, 0>, <0, 1, 0>, 0.085}
		pigment { colour White }
	}
	finish { Dull }
}

// place candles
#declare C_rand = seed(62764);
#declare C_trans = array[12] {
	<-4.2, 2, -1>,
	<-3.6, 2, -0.4>,
	<-3.4, 2, -2.4>,
	<-3.2, 2, 0.3>,
	<-2.8, 2, 1.1>,
	<-1.6, 2, 2.5>,
	<2.4, 2, 1.8>,
	<2.8, 2, -2>,
	<2.9, 2, 0.8>,
	<3.3, 2, -0.2>,
	<3.5, 2, -2.5>,
	<4, 2, -0.7>
}

#declare i = 0;
#while ( i < 12 )
	// choose colour
	#switch( rand(C_rand) )
		#range (0, 1/5)
			#declare C_colour = colour Red;
		#break
		#range (1/5, 2/5)
			#declare C_colour = colour rgb <1.5, 1.5, 0>;
		#break
		#range (2/5, 3/5)
			#declare C_colour = colour Blue;
		#break
		#range (3/5, 4/5)
			#declare C_colour = colour Green;
		#break
		#range (4/5, 1)
			#declare C_colour = colour rgb <1.4, 0.5, 0.8>;
		#break
	#end
	// Candle
	object { Candle
		pigment { colour C_colour }
		rotate y*360*rand(C_rand)
		translate C_trans[i]
	}
	// Flame
	C_Flame( 0.04, C_trans[i] + y*1.125 )
	// Flame light
	#if ( C_lights )
		light_source {
			C_trans[i] + y*1.15
			colour Orange
			fade_distance 0.25
			fade_power 2
		}
	#end
	#declare i = i + 1;
#end
//----- End Candles -----\\

//----- Board -----\\
// brackets for balloons
#declare Bracket = sphere_sweep {
	b_spline
	13,
	<0, -0.1, 0>, 0.05
	<0, 0, 0>, 0.05
	<0, 0.1, 0>, 0.05
	<0, 0.2, 0>, 0.05
	<0, 0.3, 0>, 0.05
	#declare i = 1;
	#while ( i <= 8 )
		<-0.15*sin(i*pi/4), 0.45 - 0.15*cos(i*pi/4), 0>, 0.05
		#declare i = i + 1;
	#end
	texture { T_Silver_4B }
}

#declare S_Thickness = 0.0175;  // string Thickness

#declare String1 = sphere_sweep {
	b_spline
	32,
	// balloon end first
	// circle balloon
	#declare i = 0;
	#while ( i <= 8 )
		<0.075*sin(pi/4*i), 7.50, -0.075 + 0.075*cos(pi/4*i)>, S_Thickness
		#declare i = i + 1;
	#end
	<-0.15, 7.3, 0.06>, S_Thickness
	<-0.1, 7.45, 0.03>, S_Thickness
	<-0.05, 7.55, S_Thickness>, S_Thickness
	<0, 7.50, 0>, S_Thickness
	<0, 7.40, 0>, S_Thickness
	<0, 2.00, 0>, S_Thickness
	<0, 1.20, 0>, S_Thickness
	<0, 1.05, 0>, S_Thickness
	<0, 0.90, 0.05*1/3>, S_Thickness
	<0, 0.75,	0.05*2/3>, S_Thickness
	// circle around bracket
	#declare i = 0;
	#while ( i < 5 )
		<0, 0.50 - 0.05*sin(i*pi/4), 0.05*cos(i*pi/4)>, S_Thickness
		#declare i = i + 1;
	#end
	<0, 0.75,	-0.05*2/3>, S_Thickness
	<0, 0.90,	-0.05*1/3>, S_Thickness
	<0, 1.00, -0.05*1/9>, S_Thickness
	<0, 1.05, 0>, 2.5*S_Thickness  // knot
	// dangling bit
	<0, 1.10, 0.05*1/9>, 2.5*S_Thickness  // knot
	<0.05, 1.075, 0.05*1/3>, S_Thickness
	<0.1, 1.0375, 0.05*2/3>, S_Thickness
	<0.35, 0.925, 0.05*5/3>, S_Thickness
	pigment { colour White }
	finish { ambient 0.7 }
}

#declare String2 = sphere_sweep {
	b_spline
	32,
	// balloon end first
	// circle balloon
	#declare i = 0;
	#while ( i <= 8 )
		<0.075*sin(pi/4*i), 7.50, -0.075 + 0.075*cos(pi/4*i)>, S_Thickness
		#declare i = i + 1;
	#end
	<-0.15, 7.3, 0.06>, S_Thickness
	<-0.1, 7.45, 0.03>, S_Thickness
	<-0.05, 7.55, S_Thickness>, S_Thickness
	<0, 7.50, 0>, S_Thickness
	<0, 7.40, 0>, S_Thickness
	<0, 2.00, 0>, S_Thickness
	<0, 1.20, 0>, S_Thickness
	<0, 1.05, 0>, S_Thickness
	<0, 0.90, 0.05*1/3>, S_Thickness
	<0, 0.75,	0.05*2/3>, S_Thickness
	// circle around bracket
	#declare i = 0;
	#while ( i < 5 )
		<0, 0.50 - 0.05*sin(i*pi/4), 0.05*cos(i*pi/4)>, S_Thickness
		#declare i = i + 1;
	#end
	<0, 0.75,	-0.05*2/3>, S_Thickness
	<0, 0.90,	-0.05*1/3>, S_Thickness
	<0, 1.00, -0.05*1/9>, S_Thickness
	<0, 1.05, 0>, 2.5*S_Thickness	// know
	// dangling bit
	<0, 1.10, 0.05*1/9*2/3>, 2.5*S_Thickness	// knot
	<-0.05*2/3, 1.1 - 0.025*4/3, 0.05*1/3*2/3>, S_Thickness
	<-0.1*2/3, 1.1 - 0.0625*4/3, 0.05*2/3*2/3>, S_Thickness
	<-0.35*2/3, 1.1 - 0.175*4/3, 0.05*5/3*2/3>, S_Thickness
	pigment { colour White }
	finish { ambient 0.7 }
}

#declare B_rand = seed(52107);
union {
	box { <-6, -0.25, -6>, <6, 0, 6> }
	// smooth edges
	sphere { <-6, -0.125, -6>, 0.125 }
	cylinder { <-6, -0.125, -6>, <-6, -0.125, 6>, 0.125 }
	sphere { <-6, -0.125, 6>, 0.125 }
	cylinder { <-6, -0.125, 6>, <6, -0.125, 6>, 0.125 }
	sphere { <6, -0.125, 6>, 0.125 }
	cylinder { <6, -0.125, 6>, <6, -0.125, -6>, 0.125 }
	sphere { <6, -0.125, -6>, 0.125 }
	cylinder { <6, -0.125, -6>, <-6, -0.125, -6>, 0.125 }
	// brackets, strings and balloons
	union {
		object { Bracket }
		object { String1 }
		object { Balloon
			scale 2.5
			rotate y*210*rand(B_rand)
			translate <0, 7.5, -0.075>
		}
		rotate 20*( rand(B_rand) - 0.5 )
		translate <-5.75, 0, -5.75>
	}
	union {	
		object { Bracket }
		object { String2 }
		object { Balloon
			scale 2.5
			rotate y*60*rand(B_rand)
			translate <0, 7.55, -0.075>
		}
		rotate 20*( rand(B_rand) - 0.5 )
		translate <5.75, 0, -5.75>
	}
	union {
		object { Bracket }
		object { String2 }
		object { Balloon
			scale 2.5
			rotate y*60*rand(B_rand)
			translate <0, 7.5, -0.075>
		}
		rotate 20*( rand(B_rand) - 0.5 )
		translate <-5.75, 0, 5.75>
	}
	union {
		object { Bracket }
		object { String1 }
		object { Balloon
			scale 2.5
			rotate y*60*rand(B_rand)
			translate <0, 7.5, -0.075>
		}
		rotate 20*( rand(B_rand) - 0.5 )
		translate <5.75, 0, 5.75>
	}	
	texture { T_Wood29
		scale 2
		rotate <180, 90, 0>
	}
}
//----- End Board -----\\