// Persistence of Vision Ray Tracer Include File
// File: StandMag.inc
// Vers: 3.5
// Desc: A combination stand-magnifier-and-alligator-clip thingy
// Date: 03/11/2004
// Auth: Copyright  2004 by Sherry K. Shaw

//******************************************************************************
//                      ***  HOW TO USE THIS FILE  ***
//
// The stand magnifier object may be "posed" by #declaring any or all of the variables
// listed below.  All of the numeric variables have default values and need not be
// #declared if they're not going to be changed.  Make all necessary changes to the
// posing variables, then #include this file.  About 1500 lines from now, this file
// #declares the object "SSMC_STAND_MAG_OBJ," which is the finished stand magnifier.
//
// The two variables without defaults, "SSMC_Left_AC_Obj" and "SSMC_Right_AC_Obj,"
// refer to optional objects that may be held by the left and right alligator clips.
// These variables are not used if "SSMC_Left_AC_UseObj" and "SSMC_Right_AC_UseObj"
// are set to "false," and need not be declared if an object won't be used.  (Please
// note that some degree of "nudging" will almost certainly be necessary to line up
// the object in the jaws of the clip.)
//
// These variables and their use will be described in a little bit of detail at the
// points in this file where they are first defined.  See "StandMag.pov" for a couple
// of examples of "StandMag.inc" in action.
//
// The following variables may be used to pose the object:
//   SSMC_Mag_on_FPC_Rot
//   SSMC_Mag_FPC_SW_Rot
//   SSMC_Mag_FPC_on_BC_Rot
//   SSMC_Mag_BC_S_XRot
//   SSMC_Mag_X_Trans
//   SSMC_Mag_BC_on_Rod_Rot
//   SSMC_Mag_UsePhotons
//   SSMC_Left_AC_UseObj
//   SSMC_Left_AC_Obj
//   SSMC_Left_AC_J_XRot
//   SSMC_Left_AC_Rot
//   SSMC_Left_AC_CPC_SW_Rot
//   SSMC_Right_AC_UseObj
//   SSMC_Right_AC_Obj
//   SSMC_Right_AC_J_XRot
//   SSMC_Right_AC_Rot
//   SSMC_Right_AC_CPC_SW_Rot
//   SSMC_Rod_in_BC_Rot
//   SSMC_Rod_BC_S_XRot
//   SSMC_Rod_on_FPC_Rot
//   SSMC_Rod_FPC_SW_Rot
//   SSMC_Ass_Rot
//
// Please note that there is NO checking for valid positions of the poseable parts!
// It's up to you to see to it that the alligator clip doesn't poke a hole in the
// lens, or whatever.
//
// Scale:  1 POV unit = 1 inch
// Unit stands on y=0, centered on x & z.
//
// Note: the #local variables that are defined throughout this file were not
// intended for use in modifying the finished object, but rather to save me some
// typing (and confusion), as well as for documentation, and generally should not
// be modified unless otherwise noted. (For example, see the description of
// "Lens_Edge" and "Lens_Ctr," starting at about line 1074 below.)
//******************************************************************************

//------------------------------------------------------------------------------
// The base
//------------------------------------------------------------------------------

#local Base_LL = 2.1875 ;           // Leg length (at bottom)
#local Base_LW = 0.6250 ;           // Leg width (at bottom)
#local Base_ML = 1.1250 ;           // Length of middle part (at bottom)
#local Base_MW = 0.8750 ;           // Width of middle part (distance between legs) (at bottom)
#local Base_PH = 0.0625 ;           // Height of rubber pad
#local Base_MH = 0.5625 ;           // Height of metal part of base
#local Base_Ht = Base_PH + Base_MH; // Height of entire base

#local CornerCutter =                           // (a)
difference {                                    //  |      TOP VIEW:
  box {                                         //  |
    < -0.5, -0.5, -0.5 >, < 0.5, 0.5, 0.5 >     //  | <--  These 2 sides flat; curve runs between
  }                                             //  |      point (a) and point (b)
  cylinder {                                    //  |
    < 0.5, -0.51, 0.5 >, < 0.5, 0.51, 0.5 >, 1  //  |      |
  }                                             //  |      v
}                                               //  |_____________(b)

#local BaseShape =
difference {
  union {
    box {       // Left leg
      < -( Base_LW + ( Base_MW * 0.5 ) ), 0, -( Base_LL * 0.5) >,
      < -( Base_MW * 0.5 ), Base_PH, ( Base_LL * 0.5 ) >
    }
    box {       // Right leg
      < ( Base_MW * 0.5 ), 0, -( Base_LL * 0.5) >,
      < ( Base_LW + ( Base_MW * 0.5 ) ), Base_PH, ( Base_LL * 0.5 ) >
    }
    box {       // Middle part
      < -( Base_MW * 0.51 ), 0, ( ( Base_LL * 0.5 ) - Base_ML ) >,
      < ( Base_MW * 0.51 ), Base_PH, ( Base_LL * 0.5) >
    }
  }
  union {
    object {    // Front-left outside curve of left leg
      CornerCutter
      //rotate y*0
      scale < 0.125, ( Base_PH + 0.01 ), 0.125 >
      translate < -( Base_LW - 0.0624 + ( Base_MW * 0.5 ) ), ( Base_PH * 0.5 ), -( ( Base_LL * 0.5 ) - 0.0624 ) >
    }
    object {    // Front-right outside curve of left leg
      CornerCutter
      rotate y*270
      scale < 0.125, ( Base_PH + 0.01 ), 0.125 >
      translate < -( ( Base_MW * 0.5 ) + 0.0624 ), ( Base_PH * 0.5 ), -( ( Base_LL * 0.5 ) - 0.0624 ) >
    }
    object {    // Front-left outside curve of right leg
      CornerCutter
      rotate y*270
      scale < 0.125, ( Base_PH + 0.01 ), 0.125 >
      translate < ( Base_LW - 0.0624 + ( Base_MW * 0.5 ) ), ( Base_PH * 0.5 ), -( ( Base_LL * 0.5 ) - 0.0624 ) >
    }
    object {    // Front-right outside curve of right leg
      CornerCutter
      //rotate y*0
      scale < 0.125, ( Base_PH + 0.01 ), 0.125 >
      translate < ( ( Base_MW * 0.5 ) + 0.0624 ), ( Base_PH * 0.5 ), -( ( Base_LL * 0.5 ) - 0.0624 ) >
    }
    object {    // Back-left outside curve of base
      CornerCutter
      rotate y*90
      scale < 0.125, ( Base_PH + 0.01 ), 0.125 >
      translate < -( Base_LW - 0.0624 + ( Base_MW * 0.5 ) ), ( Base_PH * 0.5 ), ( ( Base_LL * 0.5 ) - 0.0624 ) >
    }
    object {    // Back-right outside curve of base
      CornerCutter
      rotate y*180
      scale < 0.125, ( Base_PH + 0.01 ), 0.125 >
      translate < ( Base_LW - 0.0624 + ( Base_MW * 0.5 ) ), ( Base_PH * 0.5 ), ( ( Base_LL * 0.5 ) - 0.0624 ) >
    }
  }
}

#local RubberPad =
object {
  BaseShape
  texture {
    pigment {
      color rgb < 0.10, 0.10, 0.10 >
    }
    finish {
      ambient 0.3
      diffuse 0.6
    }
    normal {
      granite 0.5
      scale 0.08
    }
  }
}

#local BottomOfBase =
difference {
  object {
    BaseShape
  }
  union {
    object {    // Left edge
      CornerCutter
      rotate x*90
      scale < ( Base_LW * 0.25 ), ( Base_PH * 1.01 ), ( Base_LL * 1.1 ) >
      translate < -( ( Base_LW * 0.876 ) + ( Base_MW * 0.5 ) ), ( Base_PH * 0.5 ), 0 >
    }
    object {    // Inside edge of left leg
      CornerCutter
      rotate x*90
      rotate y*180
      scale < ( Base_LW * 0.25 ), ( Base_PH * 1.01 ), ( ( Base_LL - Base_ML ) * 1.01 ) >
      translate < -( Base_MW * 0.5875 ), ( Base_PH * 0.5 ), ( Base_LL * -0.26 ) >
    }
    object {    // Inside edge of right leg
      CornerCutter
      rotate x*90
      scale < ( Base_LW * 0.25 ), ( Base_PH * 1.01 ), ( ( Base_LL - Base_ML ) * 1.01 ) >
      translate < ( Base_MW * 0.5875 ), ( Base_PH * 0.5 ), ( Base_LL * -0.26 ) >
    }
    object {    // Right edge
      CornerCutter
      rotate x*90
      rotate y*180
      scale < ( Base_LW * 0.25 ), ( Base_PH * 1.01 ), ( Base_LL * 1.1 ) >
      translate < ( ( Base_LW * 0.876 ) + ( Base_MW * 0.5 ) ), ( Base_PH * 0.5 ), 0 >
    }
    object {    // Back edge
      CornerCutter
      rotate x*90
      rotate y*90
      scale < ( ( Base_LW * 2.01 ) + Base_MW ), ( Base_PH * 1.01 ), ( Base_LW * 0.25 ) >
      translate < 0, ( Base_PH * 0.5 ), ( Base_LL * 0.4645 ) >
    }
  }
  scale < 1, ( 0.5 / Base_PH ), 1 >
}

#local Topcutter =
difference {
  box {
    < -0.5, -0.5, -0.5 >, < 0.5, 0.5, 0.5 >
  }
  cylinder {
    < 0, -0.5, -0.51 >, < 0, -0.5, 0.51 >, 0.5
  }
}

#local TopOfBase =
difference {
  union {
    object {
      CornerCutter
      rotate x*270
      rotate y*270
      translate < 0, 0.5, 0 >
      scale < Base_MW, 1.625, ( Base_ML * 0.5) >
      translate < 0, 0.499, ( ( Base_LL - Base_ML ) * 0.5 ) >
    }
    box {
      < -( Base_MW * 0.5), 0.0, -0.03126 >,
      < ( Base_MW * 0.5), 2.125, ( ( Base_LL - Base_ML ) * 0.251 ) >
    }
  }
  union {
    object {
      Topcutter
      translate < 0, 0.5, 0 >
      scale < ( Base_MW * 1.001 ), Base_MW, Base_ML >
      translate < 0, 1.6249, 0 >
    }
    cylinder {
      < 0, 1.6875, -1 >, < 0, 1.6875, 1 >, 0.125
    }
  }
}

#local MetalPartOfBase =
union {
  object {
    BottomOfBase
  }
  object {
    TopOfBase
  }
  texture {
    pigment {
      color rgb < 0.05, 0.05, 0.05 >
    }
    finish {
      ambient 0.3
      diffuse 0.6
      metallic
    }
    normal {
      granite 0.5
      scale 0.08
    }
  }
}

#local Base =
union {
  object {
    RubberPad
  }
  object {
    MetalPartOfBase
    translate < 0, Base_PH, 0 >
  }
}

//------------------------------------------------------------------------------
// The shiny chrome parts
//------------------------------------------------------------------------------

// This is a really bright, shiny chrome.  Feel free to mess with it.
#local ChromeTex =
texture {
  pigment {
    color rgb < 0.97, 0.99, 0.87 >
  }
  finish {
    ambient 0.1
    diffuse 0.65
    specular 0.8
    roughness 0.0008
    metallic
    brilliance 6
    reflection {
      0.75
      metallic
    }
  }
}

#local CR_Rad = 0.0625 ;    // Radius of short rod part of connecting rod
#local CR_Len = 0.25 ;      // Length of short rod part of connecting rod (to center of sphere)
#local CR_Sph = 0.125 ;     // Radius of sphere at end of connecting rod
#local Rod_Rad = 0.15625 ;  // Radius of main rod
#local Rod_Len = 3.875 ;    // Length of main rod (not counting 1/16" curved bits or connecting rods)
#local Rod_X = Rod_Len * 0.5 ;  // X-offset of rod ends

#local ConnectingRod =
union {
  cylinder {
    < -0.01, 0, 0 >, < CR_Len, 0, 0 >, CR_Rad
  }
  sphere {
    < CR_Len, 0, 0 >, CR_Sph
  }
}

// ABOUT THE SCREW:
// Yes, the threads are fake.  They're just a series of skinny little cylinders, tipped
// over a bit, and threaded along the central shaft.  99.999% of the time, this won't
// matter.  Feel free to convert this to an actual spiral-thingy, if the voices insist
// that you must.

#local Screw_Len = 0.5 ;          // Length of threaded bit of screw
#local Screw_Rad = 0.0625 ;       // Radius of threaded bit of screw (not counting threads
#local Thread_Ht = 0.01 ;         // Height of threads above screw shaft
#local Head_Rad = 0.1875 ;        // Radius of screw head
#local SH_Rad = Screw_Rad + Thread_Ht ; // Radius of screw holes

#local Thread =
cylinder {
  < -( Thread_Ht * 0.5 ), 0, 0 >, < ( Thread_Ht * 0.5 ), 0, 0 >, SH_Rad
  rotate z*-5
}

#local Screw =      // Runs lengthwise on x-axis, end of screw at 0, head at right
union {
  union {
    cylinder {
      < -( Screw_Len * 0.5 ), 0, 0 > , < ( Screw_Len * 0.501 ), 0, 0 >, Screw_Rad
    }
    #local Ix = -( ( Screw_Len * 0.5 ) - ( Thread_Ht * 1.125 ) ) ;    // This is so fake!
    #while ( Ix < ( Screw_Len * 0.5 ) - ( Thread_Ht * 0.875 ) )
      object {
        Thread
        translate < Ix, 0, 0 >
      }
      #local Ix = Ix + ( Thread_Ht * 1.75 ) ;
    #end
    cone {
      < ( Screw_Len * 0.5 ), 0, 0 >, Screw_Rad, < ( Screw_Len * 0.75 ), 0, 0 >, 0
    }
    texture {
      ChromeTex
    }
  }
  cylinder {
    < 0, 0, -( Screw_Rad * 0.25 ) >, < 0, 0, ( Screw_Rad * 0.25 ) >, Head_Rad
    scale < 0.75, 1, 1 >
    translate < ( ( Screw_Len * 0.5 ) + ( Head_Rad * 0.75 ) ), 0, 0 >
    texture {
      ChromeTex
      normal {
        cells
        scale Head_Rad * 0.25
        rotate z*45
      }
    }
  }
  cylinder {
    < 0, 0, -( Screw_Rad * 0.249 ) >, < 0, 0, ( Screw_Rad * 0.249 ) >, ( Head_Rad * 1.01 )
    scale < 0.75, 1, 1 >
    translate < ( ( Screw_Len * 0.5 ) + ( Head_Rad * 0.75 ) ), 0, 0 >
    texture {
      ChromeTex
    }
  }
  translate < ( Screw_Len * 0.5 ), 0, 0 > // Put end of screw at x=0
}

#local Nut_Rad = 0.4375 ;     // Radius of wing-nut wings
#local Nut_FTB = 0.109375 ;   // 1/2 front-to-back thickness of wings
#local Nut_Butt = ( Nut_Rad - Nut_FTB ) ; // Bottom of wing-nut, if it were sitting on its flat bottom
#local Nut_C_Rad = 0.21875 ;  // Radius of circular base of wing-nut

#local WingNutShape1 =
union {
  difference {
    union {
      cylinder {
        < 0, -Nut_FTB, 0 >, < 0, Nut_FTB, 0 >, ( Nut_Rad - Nut_FTB )
      }
      torus {
        ( Nut_Rad - Nut_FTB ), Nut_FTB
      }
    }
    union {
      box {
        < -( Nut_Rad * 1.01), -( Nut_FTB * 1.01), -( Nut_Rad * 1.01) >,
        < 0, ( Nut_FTB * 1.01), ( Nut_Rad * 1.01) >
      }
      box {
        < ( Nut_Butt * 1.01 ), -( Nut_FTB * 1.01), -( Nut_Rad * 1.01) >,
        < Nut_Rad, ( Nut_FTB * 1.01), ( Nut_Rad * 1.01) >
      }
      cylinder {
        < -0.01, 0, 0 >, < ( Nut_FTB * 2 ), 0, 0 >, ( Nut_C_Rad * 0.85 )
      }
      object {
        CornerCutter
        translate < 0, 0, 0.5 >
        scale < 0.25, 1, 0.275 >
        translate < Nut_FTB, 0, Nut_FTB >
      }
      object {
        CornerCutter
        rotate y*90
        translate < 0, 0, -1.25 >
        scale < 0.25, 1, 0.275 >
        translate < Nut_FTB, 0, Nut_FTB >
      }
    }
  }
  cylinder {
    < ( Nut_FTB * 2 ), 0, 0 >, < Nut_Butt, 0, 0 >, Nut_C_Rad
  }
}

#local WingNutShape2 =
difference {
  union {
    cylinder {
      < 0, -Nut_FTB, 0 >, < 0, Nut_FTB, 0 >, ( Nut_Rad - Nut_FTB )
    }
    torus {
      ( Nut_Rad - Nut_FTB ), Nut_FTB
    }
  }
  union {
    box {
      < -( Nut_Rad * 1.01), -( Nut_FTB * 1.01), -( Nut_Rad * 1.01) >,
      < 0, ( Nut_FTB * 1.01), ( Nut_Rad * 1.01) >
    }
    box {
      < ( Nut_Butt * 1.01 ), -( Nut_FTB * 1.01), -( Nut_Rad * 1.01) >,
      < Nut_Rad, ( Nut_FTB * 1.01), ( Nut_Rad * 1.01) >
    }
    cylinder {
      < -0.01, 0, 0 >, < ( Nut_FTB * 2 ), 0, 0 >, ( Nut_C_Rad * 0.85 )
    }
    object {
      CornerCutter
      translate < 0, 0, 0.5 >
      scale < 0.25, 1, 0.275 >
      translate < Nut_FTB, 0, Nut_FTB >
    }
    object {
      CornerCutter
      rotate y*90
      translate < 0, 0, -1.25 >
      scale < 0.25, 1, 0.275 >
      translate < Nut_FTB, 0, Nut_FTB >
    }
  }
  scale < 1.01, 0.95, 0.95 >
  translate < -0.01, 0, 0 >
}

#local WingNut =
difference {
  object {
    WingNutShape1
  }
  union {
    object {
      WingNutShape2
    }
    cylinder {
      < ( Nut_FTB * 1.9 ), 0, 0 >, < ( Nut_Butt * 0.85 ), 0, 0 >, ( Nut_C_Rad * 0.8 )
    }
    cylinder {
      < ( Nut_FTB * 1.9 ), 0, 0 >, < ( Nut_Butt * 1.1 ), 0, 0, >, SH_Rad
    }
  }
  translate < -Nut_Butt, 0, 0 >
  texture {
    ChromeTex
  }
}

#local FPC_T = 0.0625 ;           // Thickness of flat plate
#local FPC_L = 1.126 ;            // Length of flat plate connector
#local FPC_W = 0.375 ;            // Width of flat plate connector
#local FPC_X = FPC_T * 0.5 ;      // Thickness of flat plate * 0.5
#local FPC_Y = FPC_L * 0.5 ;      // Length of flat plate connector * 0.5
#local FPC_Z = FPC_W * 0.5 ;      // Width of flat plate connector * 0.5
#local CPC_Rad = 0.09375 ;        // Radius of (split) cylinder at end of curved plate connector

#local FlatPlate =
difference {
  union {
    box {
      < -FPC_X, -( FPC_Y - FPC_Z ), -FPC_Z >, < FPC_X, ( FPC_Y - FPC_Z ), FPC_Z >
    }
    cylinder {
      < -FPC_X, 0, 0 >, < FPC_X, 0, 0 >, FPC_Z
      translate < 0, ( FPC_Y - FPC_Z ), 0 >
    }
    cylinder {
      < -FPC_X, 0, 0 >, < FPC_X, 0, 0 >, FPC_Z
      translate < 0, -( FPC_Y - FPC_Z ), 0 >
    }
  }
  union {
    cylinder {
      < -( FPC_X * 1.01 ), ( FPC_Y - FPC_Z ), 0 >, < ( FPC_X * 1.01 ), ( FPC_Y - FPC_Z ), 0 >, SH_Rad
    }
    cylinder {
      < -( FPC_X * 1.01 ), 0, 0 >, < ( FPC_X * 1.01 ), 0, 0 >, SH_Rad
    }
    cylinder {
      < -( FPC_X * 1.01 ), -( FPC_Y - FPC_Z ), 0 >, < ( FPC_X * 1.01 ), -( FPC_Y - FPC_Z ), 0 >, SH_Rad
    }
  }
  texture {
    ChromeTex
  }
}

#local FlatPlateConnector =
union {
  object {
    FlatPlate
    translate < -CR_Sph, 0, 0 >
  }
  object {
    FlatPlate
    translate < CR_Sph, 0, 0 >
  }
}

#local CurvedPlate =
difference {
  union {
    box {
      < -FPC_X, -( FPC_Y - FPC_Z ), -FPC_Z >,
      < FPC_X, ( SH_Rad * 1.25 ), FPC_Z >
    }
    cylinder {
      < -FPC_X, 0, 0 >, < FPC_X, 0, 0 >, FPC_Z
      translate < 0, -( FPC_Y - FPC_Z ), 0 >
    }
    union {
      box {
        < -FPC_X, 0, -FPC_Z >,
        < FPC_X, ( ( FPC_Y - FPC_Z ) - ( SH_Rad * 2.75 ) ), FPC_Z >
      }
      cylinder {
        < 0, 0, -FPC_Z >, < 0, 0, FPC_Z >, FPC_X
      }
      cylinder {
        < 0, 0, -FPC_Z >, < 0, 0, FPC_Z >, FPC_X
        translate < 0, ( ( FPC_Y - FPC_Z ) - ( SH_Rad * 2.75 ) ), 0 >
      }
      rotate z*-25
      translate < 0, ( SH_Rad * 1.25 ), 0 >
    }
    difference {
      cylinder {
        < 0, 0, -FPC_Z >, < 0, 0, FPC_Z >, ( CPC_Rad + FPC_T )
        translate < 0, ( FPC_Y - FPC_Z ), 0 >
      }
      union {
        cylinder {
          < 0, 0, -( FPC_Z + 0.01 ) >, < 0, 0, ( FPC_Z + 0.01 ) >, CPC_Rad
          translate < 0, ( FPC_Y - FPC_Z ), 0 >
        }
        box {
          < -( FPC_T * 0.5 ), -( FPC_Z + 0.01 ), -( FPC_Z + 0.01 ) >,
          < ( CPC_Rad + FPC_T + 0.01 ), ( CPC_Rad + FPC_T + 0.01 ), ( FPC_Z + 0.01 ) >
          translate < 0, ( FPC_Y - FPC_Z ), 0 >
        }
      }
      translate < CR_Sph, 0, 0 >
    }
  }
  union {
    cylinder {
      < -( FPC_X * 1.01 ), 0, 0 >, < ( FPC_X * 1.01 ), 0, 0 >, SH_Rad
    }
    cylinder {
      < -( FPC_X * 1.01 ), -( FPC_Y - FPC_Z ), 0 >, < ( FPC_X * 1.01 ), -( FPC_Y - FPC_Z ), 0 >, SH_Rad
    }
  }
  texture {
    ChromeTex
  }
}

#local CurvedPlateConnector =
union {
  object {
    CurvedPlate
    translate < -CR_Sph, 0, 0 >
  }
  object {
    CurvedPlate
    rotate y*180
    translate < CR_Sph, 0, 0 >
  }
}

// ABOUT THE ALLIGATOR CLIPS:
// Heavy sigh.  No, the alligator clips don't have springs in them.  In real life, their jaws
// would flap uselessly, like those of [insert name of least-favorite politician here].  Should
// you feel the need to insert a spring, feel free to do so.

#local AC_Ht = 0.09375 ;          // Height of each alligator clip jaw 
#local AC_MT = 0.03125 ;          // Thickness of alligator clip metal
#local AC_TL = 0.3125 ;           // Length of toothal area
#local AC_TW = 0.0625 ;           // Width of each tooth at base (AC_TW*5 = AC_TL)
#local AC_TH = 0.03125 ;          // Height of each tooth above jaw
#local AC_JawRot = 5 ;            // Number of degrees to rotate each alligator clip half
#local AC_SideCylRad = 0.1875 ;   // Radius of cylinder at side of each jaw
#local AC_CLen = 0.25 ;           // Length of (clipped) cone at end of upper jaw
#local ACJ_Len = 1.375 ;          // Overall length of alligator clip's lower jaw (ACJ_PLenF+ACJ_PLenR)
#local ACJ_PLenF = 0.8125 ;       // Length from pivot point to front of jaw
#local ACJ_PLenR = 0.5625 ;       // Length from pivot point to rear of jaw (including cylinder)
#local ACJ_CylRad = 0.15625 ;     // Radius of cylinder at end of lower jaw
#local ACJ_FW = 0.09375 ;         // Width of jaw at front end of snout
#local ACJ_RW = 0.1875 ;          // Width of jaw at rear of snout

#local ACJawBase =
prism {
  linear_spline
  0, AC_MT, 5
  < -( ACJ_RW * 0.5 ), -ACJ_PLenR >,
  < -( ACJ_FW * 0.5 ),  ACJ_PLenF >,
  <  ( ACJ_FW * 0.5 ),  ACJ_PLenF >,
  <  ( ACJ_RW * 0.5 ), -ACJ_PLenR >,
  < -( ACJ_RW * 0.5 ), -ACJ_PLenR >
}

#local ACTeeth =
prism {
  linear_spline
  0, AC_MT, 13
  < -0.05,  -AC_TL >,
  < -0.05,   0 >,
  <  0,      0 >,
  <  AC_TH, -( AC_TW * 0.5 ) >,
  <  0,     -( AC_TW * 1.0 ) >,
  <  AC_TH, -( AC_TW * 1.5 ) >,
  <  0,     -( AC_TW * 2.0 ) >,
  <  AC_TH, -( AC_TW * 2.5 ) >,
  <  0,     -( AC_TW * 3.0 ) >,
  <  AC_TH, -( AC_TW * 3.5 ) >,
  <  0,     -( AC_TW * 4.0 ) >,
  <  AC_TH, -( AC_TW * 4.5 ) >,
  < -0.05,  -AC_TL >
}

#local ACFrontLowerTooth =
prism {
  linear_spline
  0, AC_MT, 6
  < -( ACJ_FW * 0.5 ), 0 >,
  < -( ACJ_FW * 0.5 ), AC_Ht >,
  <  0,                ( AC_Ht + ( AC_TH * 0.75 ) ) >,
  <  ( ACJ_FW * 0.5 ), AC_Ht >,
  <  ( ACJ_FW * 0.5 ), 0 >,
  < -( ACJ_FW * 0.5 ), 0 >
}

#local ACFrontUpperTooth =
prism {
  linear_spline
  0, AC_MT, 6
  < -( ACJ_FW * 0.5 ), 0 >,
  < -( ACJ_FW * 0.5 ), ( AC_Ht + ( AC_TH * 0.75 ) ) >,
  <  0,                AC_Ht >,
  <  ( ACJ_FW * 0.5 ), ( AC_Ht + ( AC_TH * 0.75 ) ) >,
  <  ( ACJ_FW * 0.5 ), 0 >,
  < -( ACJ_FW * 0.5 ), 0 >
}

#local ACLowerJawBase =
union {
  difference {
    union {
      difference {
        object {
          ACJawBase
        }
        box {
          < -( ACJ_RW * 0.51 ), -0.01, -( ACJ_PLenR - ACJ_CylRad ) >,
          < ( ACJ_RW * 0.51 ), ( AC_Ht + 1.01 ), -( ACJ_PLenR * 1.01 ) >
        }
      }
      cylinder {
        < 0, 0, -( ACJ_PLenR - ACJ_CylRad ) >, < 0, AC_MT, -( ACJ_PLenR - ACJ_CylRad ) >, ACJ_CylRad
      }
    }
    sphere {
      < 0, -( ACJ_CylRad * 0.9 ), -( ACJ_PLenR - ACJ_CylRad ) >, ACJ_CylRad
    }
  }
  difference {
    cylinder {
      < 0, 0.01, -( ACJ_PLenR - ACJ_CylRad ) >,
      < 0, AC_Ht, -( ACJ_PLenR - ACJ_CylRad ) >, ACJ_CylRad
    }
    cylinder {
      < 0, 0, -( ACJ_PLenR - ACJ_CylRad ) >,
      < 0, ( AC_Ht + 0.01 ), -( ACJ_PLenR - ACJ_CylRad ) >, ( ACJ_CylRad - AC_MT )
    }
  }
  object {
    ACFrontLowerTooth
    rotate x*-90
    translate < 0, 0, ACJ_PLenF >
  }
}

#local ACLowerJawSide =
union {
  box {
    < -( AC_MT * 0.5 ), 0.01, -( ACJ_PLenR - ( ACJ_CylRad * 1.75 ) ) >,
    < ( AC_MT * 0.5 ), AC_Ht, ACJ_PLenF >
  }
  object {
    ACTeeth
    rotate z*90
    rotate x*AC_JawRot
    translate < ( AC_MT * 0.5 ), ( AC_Ht * 0.95 ), ACJ_PLenF >
  }
  difference {
    cylinder {
      < -( AC_MT * 0.5 ), 0, 0 >,
      < ( AC_MT * 0.5 ), 0, 0 >, AC_SideCylRad
      scale < 1, 0.75, 1 >
      translate < 0, AC_Ht, 0 >
    }
    box {
      < -( AC_MT * 0.51 ), ( ( AC_Ht * 0.99 ) - ( AC_SideCylRad * 0.751 ) ), -( AC_SideCylRad * 1.01 ) >,
      < ( AC_MT * 0.51 ), ( AC_Ht * 0.99 ), ( AC_SideCylRad * 1.01 ) >
    }
  }
}

#local ACLowerJaw =
union {
  object {
    ACLowerJawBase
  }
  object {
    ACLowerJawSide
    rotate y*2.0
    translate < -( ACJ_RW - ACJ_FW - AC_MT ), 0, 0 >
  }
  object {
    ACLowerJawSide
    rotate y*-2.0
    translate < ( ACJ_RW - ACJ_FW - AC_MT ), 0, 0 >
  }
  texture {
    ChromeTex
  }
}

#local ACUpperJawBase =
union {
  object {
    ACJawBase
  }
  object {
    ACFrontUpperTooth
    rotate x*-90
    translate < 0, 0, ACJ_PLenF >
  }
}

#local ACUpperJawSide =
union {
  box {
    < -( AC_MT * 0.5 ), 0.01, -ACJ_PLenR >,
    < ( AC_MT * 0.5 ), AC_Ht, ACJ_PLenF >
  }
  object {
    ACTeeth
    rotate z*90
    rotate x*AC_JawRot
    translate < ( AC_MT * 0.5 ), ( AC_Ht * 0.95 ), ( ACJ_PLenF - ( AC_TW * 0.5 ) ) >
  }
  difference {
    cylinder {
      < -( AC_MT * 0.45 ), 0, 0 >,
      < ( AC_MT * 0.45 ), 0, 0 >, AC_SideCylRad
      scale < 1, 0.75, 1 >
      translate < 0, AC_Ht, 0 >
    }
    box {
      < -( AC_MT * 0.51 ), ( ( AC_Ht * 0.99 ) - ( AC_SideCylRad * 0.751 ) ), -( AC_SideCylRad * 1.01 ) >,
      < ( AC_MT * 0.51 ), ( AC_Ht * 0.99 ), ( AC_SideCylRad * 1.01 ) >
    }
  }
}

#local ACUpperJaw =
union {
  object {
    ACUpperJawBase
  }
  object {
    ACUpperJawSide
    rotate y*2.0
    translate < -( ACJ_RW - ACJ_FW - AC_MT ), 0, 0 >
  }
  object {
    ACUpperJawSide
    rotate y*-2.0
    translate < ( ACJ_RW - ACJ_FW - AC_MT ), 0, 0 >
  }
  union {
    difference {
      cone {
        < 0, ( AC_Ht * 0.5 ), -( ACJ_PLenR - 0.01 ) >, ( ACJ_RW * 0.75 ),
        < 0, ( AC_Ht * 0.5 ), -( ACJ_PLenR + AC_CLen ) >, ( CPC_Rad - FPC_T - 0.01 )
      }
      union {
        cone {
          < 0, ( AC_Ht * 0.5 ), -( ACJ_PLenR - 0.02 ) >, ( ( ACJ_RW * 0.5 ) - AC_MT ),
          < 0, ( AC_Ht * 0.5 ), -( ACJ_PLenR + AC_CLen + 0.01 ) >, ( ( CPC_Rad - FPC_T - 0.01 ) - AC_MT )
        }
        box {
          < -ACJ_RW, -( ( AC_Ht * 0.5 ) + ACJ_RW ), -( ACJ_PLenR - 0.02 ) >,
          < ACJ_RW, 0, -( ACJ_PLenR + AC_CLen + 0.01 ) >
        }
        box {
          < -ACJ_RW, AC_Ht, -( ACJ_PLenR - 0.02 ) >,
          < ACJ_RW, ( ( AC_Ht * 0.5 ) + ACJ_RW ), -( ACJ_PLenR + AC_CLen + 0.01 ) >
        }
        box {
          < -( ACJ_RW * 0.5 ), -( ( AC_Ht * 0.5 ) + ( ACJ_RW * 0.5 ) ), -( ACJ_PLenR - 0.02 ) >,
          < -ACJ_RW, ( ( AC_Ht * 0.5 ) + ( ACJ_RW * 0.5 ) ), -( ACJ_PLenR + AC_CLen + 0.01 ) >
        }
        box {
          < ( ACJ_RW * 0.5 ), -( ( AC_Ht * 0.5 ) + ( ACJ_RW * 0.5 ) ), -( ACJ_PLenR - 0.02 ) >,
          < ACJ_RW, ( ( AC_Ht * 0.5 ) + ( ACJ_RW * 0.5 ) ), -( ACJ_PLenR + AC_CLen + 0.01 ) >
        }
      }
    }
    difference {
      cylinder {
        < 0, ( AC_Ht * 0.5 ), -( ACJ_PLenR + AC_CLen ) >,
        < 0, ( AC_Ht * 0.5 ), -( ACJ_PLenR + AC_CLen + FPC_W ) >, ( CPC_Rad - FPC_T - 0.01 )
      }
      cylinder {
        < 0, ( AC_Ht * 0.5 ), -( ACJ_PLenR + AC_CLen - 0.1 ) >,
        < 0, ( AC_Ht * 0.5 ), -( ACJ_PLenR + AC_CLen + FPC_W + 0.1 ) >,
        ( ( CPC_Rad - FPC_T - 0.01 ) * 0.5 )
      }
    }
  }
  texture {
    ChromeTex
  }
}

#local AlligatorClip =
union {
  union {
    object {
      ACUpperJaw
      rotate z*180
      rotate x*AC_JawRot
      translate < 0, ( AC_SideCylRad * 0.95 ), 0 >
    }
    object {
      ACLowerJaw
      rotate x*-AC_JawRot
      translate < 0, -( AC_SideCylRad * 0.95 ), 0 >
    }
    rotate x*-AC_JawRot
  }
  cylinder {
    < -( ACJ_RW * 0.5 ), 0, 0 >, < ( ACJ_RW * 0.5 ), 0, 0 >, AC_MT
    texture {
      ChromeTex
    }
  } // Put < 0, 0, 0 > at center of upper jaw extension
  translate < 0, -( ( AC_SideCylRad * 0.95 ) - ( AC_Ht * 0.5 ) ), ( ACJ_PLenR + AC_CLen+ FPC_Z  ) >
}

// Rotation of lower jaw of left alligator clip.
// X-axis rotation is subtracted from closed rotation.
#ifndef ( SSMC_Left_AC_J_XRot )
  #local SSMC_Left_AC_J_XRot = 0 ;
#end

// Object to be held in jaws of right alligator clip.
#ifndef ( SSMC_Left_AC_UseObj )
  #local SSMC_Left_AC_UseObj = false ;
#end

#local LeftAlligatorClip =
union {
  union {
    object {
      ACUpperJaw
      rotate z*180
      rotate x*AC_JawRot
      translate < 0, ( AC_SideCylRad * 0.95 ), 0 >
    }
    object {
      ACLowerJaw
      rotate x*-( AC_JawRot - SSMC_Left_AC_J_XRot )
      translate < 0, -( AC_SideCylRad * 0.95 ), 0 >
    }
    #if ( SSMC_Left_AC_UseObj = true )
      object {
        SSMC_Left_AC_Obj
      }
    #end
    rotate x*-AC_JawRot
  }
  cylinder {
    < -( ACJ_RW * 0.5 ), 0, 0 >, < ( ACJ_RW * 0.5 ), 0, 0 >, AC_MT
    texture {
      ChromeTex
    }
  } // Put < 0, 0, 0 > at center of upper jaw extension
  translate < 0, -( ( AC_SideCylRad * 0.95 ) - ( AC_Ht * 0.5 ) ), ( ACJ_PLenR + AC_CLen+ FPC_Z  ) >
}

// Rotation of lower jaw of right alligator clip.
// X-axis rotation is subtracted from closed rotation.
#ifndef ( SSMC_Right_AC_J_XRot )
  #local SSMC_Right_AC_J_XRot = 0 ;
#end

// Object to be held in jaws of right alligator clip.
#ifndef ( SSMC_Right_AC_UseObj )
  #local SSMC_Right_AC_UseObj = false ;
#end

#local RightAlligatorClip =
union {
  union {
    object {
      ACUpperJaw
      rotate z*180
      rotate x*AC_JawRot
      translate < 0, ( AC_SideCylRad * 0.95 ), 0 >
    }
    object {
      ACLowerJaw
      rotate x*-( AC_JawRot - SSMC_Right_AC_J_XRot )
      translate < 0, -( AC_SideCylRad * 0.95 ), 0 >
    }
    #if ( SSMC_Right_AC_UseObj = true )
      object {
        SSMC_Right_AC_Obj
      }
    #end
    rotate x*-AC_JawRot
  }
  cylinder {
    < -( ACJ_RW * 0.5 ), 0, 0 >, < ( ACJ_RW * 0.5 ), 0, 0 >, AC_MT
    texture {
      ChromeTex
    }
  } // Put < 0, 0, 0 > at center of upper jaw extension
  translate < 0, -( ( AC_SideCylRad * 0.95 ) - ( AC_Ht * 0.5 ) ), ( ACJ_PLenR + AC_CLen+ FPC_Z  ) >
}

#local BC_Len = 0.5 ;             // Length of "barrel" part of barrel connector
#local BC_Rad = 0.25 ;            // Radius of barrel connector
#local BC_Ext = 0.1875 ;          // Length of extension (radius=Rod_Rad)

#local BarrelConnector =
union {
  difference {
    cylinder {
      < 0, -( BC_Len * 0.5 ), 0 >, < 0, ( BC_Len * 0.5 ), 0 >, BC_Rad
    }
    union {
      cylinder {
        < -( BC_Len * 0.51 ), 0, 0 >, < ( BC_Len * 0.51 ), 0, 0 >, ( Rod_Rad * 1.1 )
      }
      cone {
        < 0, 0, ( BC_Rad * 1.1 ) >, ( Screw_Rad + ( Thread_Ht * 3 ) ), < 0, 0, 0 >, Screw_Rad
      }
    }
  }
  cylinder {
    < 0, ( BC_Len * 0.51 ), 0 >, < 0, ( ( BC_Len * 0.5 ) + BC_Ext ), 0 >, Rod_Rad
  }
  object {
    ConnectingRod
    rotate z*90
    translate < 0, ( ( BC_Len * 0.5 ) + BC_Ext ), 0 >
  }
  texture {
    ChromeTex
  }
}

#local SupportRod =
union {
  cylinder {
    < 0, 0, -0.25 >, < 0, 0, 0.25 >, 0.125
  }
  sphere {
    < 0, 0, 0 >, 0.125
    scale < 1, 1, 0.5 >
    translate < 0, 0, -0.25 >
  }
  object {
    ConnectingRod
    rotate y*90
    translate < 0, 0, -0.1875 >
  }
  translate < 0, 0, ( CR_Len + 0.1875 ) > // Put z=0 at center of connecting rod sphere
  texture {
    ChromeTex
  }
}

#local LongRod =
union {
  cylinder {
    < -Rod_X, 0, 0 >, < Rod_X, 0, 0 >, Rod_Rad
  }
  sphere {
    < 0, 0, 0 >, Rod_Rad
    scale < 0.5, 1, 1 >
    translate < -Rod_X, 0, 0 >
  }
  sphere {
    < 0, 0, 0 >, Rod_Rad
    scale < 0.5, 1, 1 >
    translate < Rod_X, 0, 0 >
  }
  object {
    ConnectingRod
    rotate z*180
    translate < -( Rod_X + ( Rod_Rad * 0.25 ) ), 0, 0 >
  }
  object {
    ConnectingRod
    translate < ( Rod_X + ( Rod_Rad * 0.25 ) ), 0, 0 >
  }
  texture {
    ChromeTex
  }
}

//------------------------------------------------------------------------------
// The magnifying glass
//
// (This particular big, thick lens was set up with caustics in mind, rather than
// an accurate re-creation of the lens in the magnifier that's sitting here on my
// desk.  The variables "Lens_Edge" and "Lens_Ctr" control the thickness of the
// lens; feel free to modify them however you like.)
//------------------------------------------------------------------------------

// Use photons with lens?
#ifndef ( SSMC_Mag_UsePhotons )
  #local SSMC_Mag_UsePhotons = false ;
#end

#local Lens_Rad = 1.1875 ;      // Lens radius
#local Lens_Edge = 0.125 ;      // Thickness of lens at edge
#local Lens_Ctr = 0.575 ;       // Thickness of lens at center
#local Ring_T = 0.03125 ;       // Thickness of chrome ring around lens
#local Ring_W = 0.4375 ;        // Width of chrome ring around lens
#local Att_Len = 0.375 ;        // Length of ring attachment rod

#local H = 0.5 * ( Lens_Ctr - Lens_Edge ) ;
#local SRad = 0.5 * ( ( pow( ( Lens_Rad * 2 ), 2 ) / ( 4 * H ) ) + H ) ;
#local D = SRad - H ;
#local CRad = Lens_Edge * 0.5 ;

#local LensCurve =
difference {
  intersection {
    sphere {
      < 0, 0, 0 >, SRad
      translate < 0, 0, -D >
    }
    sphere {
      < 0, 0, 0 >, SRad
      translate < 0, 0, D >
    }
  }
  cylinder {
    < 0, 0, 0 >, < 0, 0, ( D * 1.01) >, ( Lens_Rad * 1.01 )
  }
}

#local Lens =
merge {
  object {
    LensCurve
    translate < 0, 0, -CRad >
  }
  cylinder {
    < 0, 0, -( CRad * 1.001 ) >, < 0, 0, ( CRad * 1.001 ) >, Lens_Rad
  }
  object {
    LensCurve
    rotate y*180
    translate < 0, 0, CRad >
  }
  material {
    texture {
      pigment {
        color rgbf < 0.98, 0.98, 0.98, 0.98 >
      }
      finish {
        ambient 0.1
        diffuse 0.1
        specular 0.8
        roughness 0.003
        reflection {
          0.025
        }
      }
    }
    interior {
      ior 1.7
      #if ( SSMC_Mag_UsePhotons = false )
        caustics 1.5
      #end
    }
  }
  #if ( SSMC_Mag_UsePhotons = true )
    photons {
      target
      reflection off
      refraction on
      collect off
    }
  #end
}

#local Ring =
difference {
  cylinder {
    < 0, 0, -( Ring_W * 0.5 ) >, < 0, 0, ( Ring_W * 0.5 ) >, ( Lens_Rad + ( Ring_T * 0.5 ) )
  }
  cylinder {
    < 0, 0, -( Ring_W * 0.51 ) >, < 0, 0, ( Ring_W * 0.51 ) >, ( Lens_Rad - ( Ring_T * 0.5 ) )
  }
  texture {
    ChromeTex
  }
}

#local RingAttachment =
difference {
  union {
    cylinder {
      < 0, 0, -( ( Ring_W * 0.5 ) + Ring_T ) >, < 0, 0, ( ( Ring_W * 0.5 ) + Ring_T ) >, ( Lens_Rad + Ring_T )
    }
    torus {
      ( Lens_Rad + Ring_T - ( Ring_T * 0.25 ) ), ( Ring_T * 0.25 )
      rotate x*90
      translate < 0, 0, ( ( Ring_W * 0.5 ) + Ring_T ) >
    }
    torus {
      ( Lens_Rad + Ring_T - ( Ring_T * 0.25 ) ), ( Ring_T * 0.25 )
      rotate x*90
      translate < 0, 0, -( ( Ring_W * 0.5 ) + Ring_T ) >
    }
  }
  union {
    cylinder {
      < 0, 0, -( ( Ring_W * 0.51 ) + Ring_T ) >, < 0, 0, ( ( Ring_W * 0.51 ) + Ring_T ) >, ( Lens_Rad - ( Ring_W * 0.125 ) )
    }
    cylinder {
      < 0, 0, -( Ring_W * 0.5 ) >, < 0, 0, ( Ring_W * 0.5 ) >, ( Lens_Rad + Ring_T )
    }
    box {
      < -( Lens_Rad * 1.5 ), 0, -( Ring_W * 1.5 ) >, < ( Lens_Rad * 1.5 ), ( Lens_Rad * 1.5 ), ( Ring_W * 1.5 ) >
      rotate z*75
    }
    box {
      < -( Lens_Rad * 1.5 ), 0, -( Ring_W * 1.5 ) >, < ( Lens_Rad * 1.5 ), ( Lens_Rad * 1.5 ), ( Ring_W * 1.5 ) >
      rotate z*-75
    }
  }
  texture {
    ChromeTex
  }
}

#local RingAttachmentRod =
union {
  cylinder {
    < 0, 0.01, 0 >, < 0, -Att_Len, 0 >, Rod_Rad
  }
  sphere {
    < 0, 0, 0 >, Rod_Rad
    scale < 1, 0.5, 1 >
    translate < 0, -Att_Len, 0 >
  }
  object {
    ConnectingRod
    rotate z*-90
    translate < 0, -Att_Len, 0 >
  }
  translate < 0, -( Lens_Rad + Ring_T ), 0 >
  texture {
    ChromeTex
  }
}

#local Magnifier =
union {
  object {
    Lens
  }
  object {
    Ring
  }
  object {
    RingAttachment
  }
  object {
    RingAttachmentRod
  }
  translate < 0, ( Lens_Rad + Ring_T + Att_Len + CR_Len ), 0 >   // Put y=0 at center of sphere at end of connecting rod
}

// *****  MAGNIFIER ASSEMBLY  *****

// Rotation of magnifier assembly on flate plate connector.
// Y-axis rotation will be applied before X-axis rotation.  (Although a minimal amount of z-axis
// rotation is possible, it is just way too trivial to mess with and is ignored.)
#ifndef ( SSMC_Mag_on_FPC_Rot )
  #local SSMC_Mag_on_FPC_Rot = < 0, 0, 0 > ;
#end

// Rotation of screw and wing nut in flate plate connector.
// X-axis rotation is applied to the screw.  Y-axis value is applied as x-axis rotation to the wing nut.
// The z-axis value is ignored.
#ifndef ( SSMC_Mag_FPC_SW_Rot )
  #local SSMC_Mag_FPC_SW_Rot = < 0, 0, 0 > ;
#end

// Rotation of flate plate connector at bottom of magnifier assembly on barrel connector.
// X-axis rotation will be applied before Y-axis rotation.  (Although a minimal amount of z-axis
// rotation is possible, it is just way too trivial to mess with and is ignored.)
#ifndef ( SSMC_Mag_FPC_on_BC_Rot )
  #local SSMC_Mag_FPC_on_BC_Rot = < 0, 0, 0 > ;
#end

// Rotation of screw in barrel connector.
// X-axis rotation is applied to the screw (actually applied to z-axis after 90 degree y-axis rotation
// need to move screw to back of assembly).
#ifndef ( SSMC_Mag_BC_S_XRot )
  #local SSMC_Mag_BC_S_XRot = 0 ;
#end

#local MagnifierAssembly =
union {
  union {
    object {
      FlatPlateConnector
    }
    object {
      Screw
      rotate x*SSMC_Mag_FPC_SW_Rot.x
      translate < -( CR_Sph - FPC_X + ( Screw_Len * 0.45 ) ), 0, 0 > 
    }
    object {
      WingNut
      rotate x*SSMC_Mag_FPC_SW_Rot.y
      translate < -( CR_Sph + ( FPC_X * 1.1 ) ), 0, 0 >
    }
    object {
      Magnifier
      rotate y*SSMC_Mag_on_FPC_Rot.y
      rotate x*SSMC_Mag_on_FPC_Rot.x
      translate < 0, ( FPC_Y - FPC_Z ), 0 >
    }
    translate < 0, ( FPC_Y - FPC_Z ), 0 >
    rotate x*SSMC_Mag_FPC_on_BC_Rot.x
    rotate y*SSMC_Mag_FPC_on_BC_Rot.y
    translate < 0, ( CR_Len + BC_Ext + ( BC_Len * 0.5 ) ), 0 >
  }
  object {
    BarrelConnector
  }
  object {
    Screw
    rotate y*-90
    rotate z*SSMC_Mag_BC_S_XRot
    translate < 0, 0, ( Rod_Rad * 0.5 ) >
  }
}

// *****  Left Alligator Clip Assembly  *****

// Rotation of left alligator clip.
// Y-axis rotation is applied to alligator clip extension within curved plate connector (actually applied
// as x-axis rotation before complete assembly is rotated into final position).  X-axis and z-axis
// rotation are applied to completed assembly.  (Note that z-axis rotation is applied after the 90
// degree rotation that brings the assembly into the extended position at the end of the long rod.)
#ifndef ( SSMC_Left_AC_Rot )
  #local SSMC_Left_AC_Rot = < 0, 0, 0 > ;
#end

// Rotation of screw and wing nut in curved plate connector.
// X-axis rotation is applied to the screw.  Y-axis value is applied as x-axis rotation to the wing nut.
// The z-axis value is ignored.
#ifndef ( SSMC_Left_AC_CPC_SW_Rot )
  #local SSMC_Left_AC_CPC_SW_Rot = < 0, 0, 0 > ;
#end

#local LeftAlligatorClipAssembly =
union {
  union {
    object {
      CurvedPlateConnector
    }
    object {
      Screw
      rotate x*SSMC_Left_AC_CPC_SW_Rot.x
      translate < -( CR_Sph - FPC_X + ( Screw_Len * 0.45 ) ), 0, 0 > 
    }
    object {
      WingNut
      rotate x*SSMC_Left_AC_CPC_SW_Rot.y
      translate < -( CR_Sph + ( FPC_X * 1.1 ) ), 0, 0 >
    }
    rotate y*90
    translate < 0, ( FPC_Y - FPC_Z ), 0 >
  }
  object {
    LeftAlligatorClip
    rotate y*90
    rotate x*SSMC_Left_AC_Rot.y
    translate < 0, ( FPC_Y + FPC_Z + ( CR_Sph * 0.5 ) ), 0 >
  }
  rotate z*90
  rotate z*SSMC_Left_AC_Rot.z
  rotate x*SSMC_Left_AC_Rot.x
}

// *****  Right Alligator Clip Assembly  *****

// Rotation of right alligator clip.
// Y-axis rotation is applied to alligator clip extension within curved plate connector (actually applied
// as x-axis rotation before complete assembly is rotated into final position).  X-axis and z-axis
// rotation are applied to completed assembly.  (Note that z-axis rotation is applied after the 90
// degree rotation that brings the assembly into the extended position at the end of the long rod.)
#ifndef ( SSMC_Right_AC_Rot )
  #local SSMC_Right_AC_Rot = < 0, 0, 0 > ;
#end

// Rotation of screw and wing nut in curved plate connector.
// X-axis rotation is applied to the screw.  Y-axis value is applied as x-axis rotation to the wing nut.
// The z-axis value is ignored.
#ifndef ( SSMC_Right_AC_CPC_SW_Rot )
  #local SSMC_Right_AC_CPC_SW_Rot = < 0, 0, 0 > ;
#end

#local RightAlligatorClipAssembly =
union {
  union {
    object {
      CurvedPlateConnector
    }
    object {
      Screw
      rotate x*SSMC_Right_AC_CPC_SW_Rot.x
      translate < -( CR_Sph - FPC_X + ( Screw_Len * 0.45 ) ), 0, 0 > 
    }
    object {
      WingNut
      rotate x*SSMC_Right_AC_CPC_SW_Rot.y
      translate < -( CR_Sph + ( FPC_X * 1.1 ) ), 0, 0 >
    }
    rotate y*90
    translate < 0, ( FPC_Y - FPC_Z ), 0 >
  }
  object {
    RightAlligatorClip
    rotate y*-90
    rotate x*SSMC_Right_AC_Rot.y
    translate < 0, ( FPC_Y + FPC_Z + ( CR_Sph * 0.5 ) ), 0 >
  }
  rotate z*-90
  rotate z*SSMC_Right_AC_Rot.z
  rotate x*SSMC_Right_AC_Rot.x
}

// *****  Long Rod Assembly  *****

// X-translation of magnifier assembly.
#ifndef ( SSMC_Mag_X_Trans )
  #local SSMC_Mag_X_Trans = -1.25 ;
#end

// X-rotation of magnifier assembly barrel connector around long rod.
#ifndef ( SSMC_Mag_BC_on_Rod_Rot )
  #local SSMC_Mag_BC_on_Rod_Rot = 0 ;
#end

// X-rotation of long rod assembly in central barrel connector.
#ifndef ( SSMC_Rod_in_BC_Rot )
  #local SSMC_Rod_in_BC_Rot = 0 ;
#end

// Rotation of screw in barrel connector.
// X-axis rotation is applied to the screw.
#ifndef ( SSMC_Rod_BC_S_XRot )
  #local SSMC_Rod_BC_S_XRot = 0 ;
#end

#local LongRodAssembly =
union {
  union {
    object {
      LongRod
    }
    object {
      LeftAlligatorClipAssembly
      translate < -( Rod_X + CR_Len + ( Rod_Rad * 0.25 ) ), 0, 0 >
    }
    object {
      MagnifierAssembly
      translate < SSMC_Mag_X_Trans, 0, 0 >
      rotate x*SSMC_Mag_BC_on_Rod_Rot
    }
    object {
      RightAlligatorClipAssembly
      translate < ( Rod_X + CR_Len + ( Rod_Rad * 0.25 ) ), 0, 0 >
    }
    rotate x*SSMC_Rod_in_BC_Rot
  }
  object {
    BarrelConnector
    rotate z*180
  }
  object {
    Screw
    rotate y*-90
    rotate z*SSMC_Rod_BC_S_XRot
    translate < 0, 0, ( Rod_Rad * 0.5 ) >
  }
  translate < 0, ( CR_Len + BC_Ext + ( BC_Len * 0.5 ) ), 0 >
}

// *****  Mag-and-Clip Assembly  *****

// Rotation of long rod assembly on flate plate connector.
// Y-axis rotation will be applied before X-axis rotation.  (Although a minimal amount of z-axis
// rotation is possible, it is just way too trivial to mess with and is ignored.)
#ifndef ( SSMC_Rod_on_FPC_Rot )
  #local SSMC_Rod_on_FPC_Rot = < 0, 0, 0 > ;
#end

// Rotation of screw and wing nut in flate plate connector.
// X-axis rotation is applied to the screw.  Y-axis value is applied as x-axis rotation to the wing nut.
// The z-axis value is ignored.
#ifndef ( SSMC_Rod_FPC_SW_Rot )
  #local SSMC_Rod_FPC_SW_Rot = < 0, 0, 0 > ;
#end

#local MagAndClipAssembly =
union {
  object {
    LongRodAssembly
    rotate y*SSMC_Rod_on_FPC_Rot.y
    rotate x*SSMC_Rod_on_FPC_Rot.x
    translate < 0, ( FPC_Y - FPC_Z ), 0 >
  }
  union {
    object {
      FlatPlateConnector
    }
    object {
      Screw
      rotate x*SSMC_Rod_FPC_SW_Rot.x
      translate < -( CR_Sph - FPC_X + ( Screw_Len * 0.45 ) ), 0, 0 > 
    }
    object {
      WingNut
      rotate x*SSMC_Rod_FPC_SW_Rot.y
      translate < -( CR_Sph + ( FPC_X * 1.1 ) ), 0, 0 >
    }
  }
}

// *****  Base Assembly  *****

#local BaseAssembly =
union {
  object {
    Base
  }
  object {
    SupportRod
    translate < 0, ( 1.6875 + Base_PH ), -( CR_Len * 1.75 ) >
  }
}

// *****  The Finished Gadget  *****

// Rotation of complete mag-and-clip assembly on flate plate connector.
// X-axis and z-axis rotation are applied to completed assembly, in that order. Y value is ignored.
#ifndef ( SSMC_Ass_Rot )
  #local SSMC_Ass_Rot = < 0, 0, 0 > ;
#end

#declare SSMC_STAND_MAG_OBJ =
union {
  object {
    BaseAssembly
  }
  object {
    MagAndClipAssembly
    translate < 0, ( FPC_Y - FPC_Z ), 0 >
    rotate x*SSMC_Ass_Rot.x
    rotate z*SSMC_Ass_Rot.z 
    translate < 0, ( 1.6875 + Base_PH ), -( CR_Len * 1.75 ) >
  }
}






