//-------------------------------------------------------------------------------------------------
// Author: Paolo Brasolin
// Title: Strawberry Juice
// Date: 5/2/2004
// Info: My entry for the IRTC competitions of May-June 2004 (Topic: Epic Proportions)
// Email: pbrasolin@yahoo.it
// Homepage: http://paolo.brasolin.free.fr
// Render options: +W1600 +H800 +A
//-------------------------------------------------------------------------------------------------

//-----------------------------------------------------------------------------------Initialization
#include "math.inc"
#include "rand.inc"

global_settings{
  max_trace_level 128
  radiosity{}
}

#declare stream=seed(6888);

//-------------------------------------------------------------------------------------------Camera
camera {
  location <3,0,-8>
  look_at x*-3
  right x*2
}

//----------------------------------------------------------------------------------------------Sky
// I used galaxy.inc by Chris Colefax (http://www.geocities.com/SiliconValley/Lakes/1434/)
sky_sphere{pigment{wrinkles color_map{[0 rgb 0][1 blue 0.1]}}scale 0.5}
#declare star_count=20000;
#declare star_color=1;
#declare star_type=1;
#declare star_scale=0.6;
#declare star_distance=20000;
#declare star_spread=8;
#declare star_distribution=1;
#declare star_color_turb=0.5;
#declare star_brightness_turb=1;
#include "GALAXY.SF"

//--------------------------------------------------------------------------------------------Glass
// Just a lathe
#declare glass = lathe { 
  cubic_spline
  15 
  <0.00,0.00>,
  <0.00,0.40>,
  <1.60,0.00>,
  <1.60,0.10>,
  <0.70,0.30>,
  <0.55,0.40>,
  <0.35,0.50>,
  <0.30,0.60>,
  <0.20,0.95>,
  <0.20,2.65>,
  <0.50,3.40>,
  <2.00,5.00>,
  <1.90,5.00>,
  <0.00,3.30>,
  <0.00,4.00>

  texture {
    pigment { color rgbt <1,1,1,0.9> quick_color rgb <0,1,1> }
    finish {
      specular 0.7
      roughness 0.001
      ambient 0
      diffuse 0
      reflection { 0.1, 0.6 fresnel on }
      conserve_energy
    }
  }

  interior {
   ior 1.5
   fade_distance 4
   fade_power 1
  }

  photons {
    target
    reflection on
    refraction on
  }
  
}

//-------------------------------------------------------------------------------------------Liquid
// A perturbed isosurface box intersected with a cone
#declare liquid = intersection {
  cone{<0,3.30,0>,0,<0,5,0>,1.9}
  isosurface {
    function { f_rounded_box(x,y,z,0,0.5,0.5,0.5)-f_noise3d(x*8,y*8,z*8)/32 }
    scale 10
    translate y*-0.4
  }
  texture {
    pigment { color rgbf <1,0,0,0.9> quick_color red 1 }
    finish {
      specular 0.7
      roughness 0.03
      reflection { 0, 0.5 fresnel on }
      conserve_energy
    }
  }
  interior {
   ior 1.333
   fade_distance 2
   fade_power 1
  }
  photons {
    target
    reflection on
    refraction on
  }
}

//--------------------------------------------------------------------------------------Protofraise
// It's the starting shape
#declare protofraise = blob {
  threshold 0.6
  sphere { <0, 2.1,0>, 1.5,  0.35 }
  sphere { <0, 0.0,0>, 3.0,  0.85 }
  sphere { <0,-1.3,0>, 1.0, -0.10 }
}

//----------------------------------------------------------------------------------------Prefraise
// The basic shape
#declare prefraise = blob {
  threshold 0.6
  sphere{ <0, 2.1,0>, 1.5,  0.35 }
  sphere{ <0, 0.0,0>, 3.0,  0.85 }
  sphere{ <0,-1.3,0>, 1.0, -0.10 }

// is deformed here
  #declare n=0;
  #while (n<60)
    #declare m=<sin(n)*2,(3/60*n)-1,cos(n)*2>;
    #declare l=trace(protofraise,m,<0-m.x,m.y,0-m.z>);
    #if (l.x!=0|l.y!=0|l.z!=0)
      sphere {l,0.5,-0.04 }
    #end
    #declare n=n+1;
  #end

// and the seeds holes are added here
  #declare n=0;
  #while (n<600)
    #declare m=<sin(n)*2,(3/600*n)-1,cos(n)*2>;
    #declare l=trace(protofraise,m,<0-m.x,m.y,0-m.z>);
    #if (l.x!=0|l.y!=0|l.z!=0)
      sphere {l,0.15,-0.008 }
    #end
    #declare n=n+1;
  #end
}

//-------------------------------------------------------------------------------------------Fraise
#declare fraise = union{

  object { prefraise  
    texture {
      pigment { bozo color_map { [0 red 1] [1 red 0.5] } scale 0.5 quick_color red 1 }      
      finish {
        specular 0.7 roughness 0.03
        phong 0.3 phong_size 5
        crand 0.1
      }
    }
  }

// The seeds are added here
  union {
    #declare n=0;
    #declare i=<0,0,0>;
    #while (n<600)
      #declare m=<sin(n)*2,(3/600*n)-1,cos(n)*2>;
      #declare l=trace(prefraise,m,<0-m.x,m.y,0-m.z>,i);
      #if (l.x!=0|l.y!=0|l.z!=0)
        sphere {0,0.03 scale <1.5,2.5,1.5>*rand(seed(6888)) rotate i*36 translate l}
      #end
      #declare n=n+1;
    #end
  
    texture {
      pigment { color rgb <0.7,0.6,0> quick_color rgb <1,1,0> }      
      finish { specular 1 roughness 0.05 }
    }
  }

// The leaves are added here
  union{
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.029, 0.188, 0.742> <-0.167, 0.174, 0.660> <-0.350, 0.155, 0.551> <-0.488, 0.141, 0.469> <-0.112, 0.207, 0.868> <-0.250, 0.193, 0.786> <-0.395, 0.172, 0.680> <-0.533, 0.158, 0.598> <-0.355, 0.131, 0.872> <-0.353, 0.141, 0.734> <-0.353, 0.141, 0.734> <-0.493, 0.117, 0.790> <-0.495, 0.106, 0.928> <-0.495, 0.106, 0.928> <-0.495, 0.106, 0.928> <-0.495, 0.106, 0.928> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.414, 0.299, 0.420> < 0.298, 0.223, 0.507> < 0.142, 0.122, 0.622> < 0.025, 0.047, 0.708> < 0.490, 0.284, 0.580> < 0.373, 0.208, 0.666> < 0.236, 0.110, 0.738> < 0.119, 0.034, 0.825> < 0.462, 0.044, 0.743> < 0.325, 0.078, 0.691> < 0.325, 0.078, 0.691> < 0.345,-0.032, 0.829> < 0.482,-0.065, 0.881> < 0.482,-0.065, 0.881> < 0.482,-0.065, 0.881> < 0.482,-0.065, 0.881> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.514,-0.099, 0.469> < 0.576, 0.004, 0.326> < 0.659, 0.141, 0.135> < 0.722, 0.244,-0.009> < 0.643,-0.124, 0.507> < 0.705,-0.022, 0.363> < 0.821, 0.104, 0.220> < 0.883, 0.207, 0.077> < 0.826,-0.141, 0.421> < 0.755,-0.020, 0.298> < 0.755,-0.020, 0.298> < 0.889,-0.038, 0.278> < 0.960,-0.159, 0.400> < 0.960,-0.159, 0.400> < 0.960,-0.159, 0.400> < 0.960,-0.159, 0.400> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-1.089,-0.104, 0.438> <-0.921,-0.052, 0.432> <-0.695, 0.048, 0.473> <-0.530, 0.068, 0.418> <-0.999,-0.062, 0.340> <-0.832,-0.011, 0.334> <-0.774, 0.037, 0.381> <-0.608, 0.058, 0.325> <-0.912, 0.018, 0.178> <-0.833, 0.028, 0.270> <-0.833, 0.028, 0.270> <-0.713, 0.044, 0.202> <-0.792, 0.034, 0.110> <-0.792, 0.034, 0.110> <-0.792, 0.034, 0.110> <-0.792, 0.034, 0.110> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-1.052,-0.069,-0.366> <-0.957, 0.007,-0.272> <-0.875, 0.170,-0.116> <-0.736, 0.184,-0.051> <-0.911,-0.001,-0.392> <-0.816, 0.075,-0.297> <-0.829, 0.163,-0.236> <-0.690, 0.176,-0.171> <-0.735, 0.144,-0.492> <-0.781, 0.152,-0.372> <-0.781, 0.152,-0.372> <-0.628, 0.166,-0.331> <-0.582, 0.158,-0.451> <-0.582, 0.158,-0.451> <-0.582, 0.158,-0.451> <-0.582, 0.158,-0.451> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.455, 0.115,-0.908> <-0.337, 0.124,-0.834> <-0.134, 0.165,-0.780> <-0.063, 0.145,-0.661> <-0.471, 0.085,-0.781> <-0.354, 0.094,-0.707> <-0.268, 0.126,-0.726> <-0.197, 0.107,-0.608> <-0.546, 0.035,-0.609> <-0.412, 0.074,-0.662> <-0.412, 0.074,-0.662> <-0.376, 0.055,-0.537> <-0.510, 0.017,-0.484> <-0.510, 0.017,-0.484> <-0.510, 0.017,-0.484> <-0.510, 0.017,-0.484> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.557, 0.090,-0.438> < 0.689, 0.061,-0.478> < 0.831,-0.091,-0.418> < 0.948,-0.168,-0.409> < 0.583, 0.084,-0.327> < 0.715, 0.056,-0.367> < 0.739,-0.019,-0.315> < 0.857,-0.096,-0.307> < 0.617, 0.078,-0.178> < 0.747, 0.046,-0.258> < 0.747, 0.046,-0.258> < 0.772, 0.041,-0.147> < 0.643, 0.072,-0.067> < 0.643, 0.072,-0.067> < 0.643, 0.072,-0.067> < 0.643, 0.072,-0.067> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.354, 0.042,-0.465> < 0.276, 0.110,-0.520> < 0.172, 0.201,-0.594> < 0.095, 0.269,-0.649> < 0.404, 0.087,-0.611> < 0.327, 0.155,-0.666> < 0.247, 0.238,-0.703> < 0.169, 0.306,-0.758> < 0.409, 0.183,-0.808> < 0.308, 0.191,-0.716> < 0.308, 0.191,-0.716> < 0.331, 0.251,-0.863> < 0.433, 0.243,-0.955> < 0.433, 0.243,-0.955> < 0.433, 0.243,-0.955> < 0.433, 0.243,-0.955> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.032, 0.003,-0.128> <-0.004, 0.003,-0.128> < 0.034, 0.003,-0.112> < 0.053, 0.003,-0.092> <-0.026, 0.122,-0.082> < 0.002, 0.122,-0.082> <-0.006, 0.120,-0.080> < 0.014, 0.120,-0.061> <-0.054, 0.300,-0.087> <-0.037, 0.300,-0.087> <-0.008, 0.299,-0.075> < 0.003, 0.299,-0.063> <-0.032, 0.426,-0.078> <-0.016, 0.426,-0.078> < 0.006, 0.426,-0.068> < 0.018, 0.426,-0.057> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.053, 0.003,-0.092> < 0.073, 0.003,-0.073> < 0.089, 0.003,-0.035> < 0.089, 0.003,-0.007> < 0.066,-0.033,-0.102> < 0.085,-0.033,-0.082> < 0.110,-0.040,-0.032> < 0.110,-0.040,-0.004> < 0.128,-0.002,-0.148> < 0.176,-0.002,-0.106> < 0.206,-0.014,-0.041> < 0.203,-0.014, 0.022> < 0.160, 0.003,-0.170> < 0.208, 0.003,-0.129> < 0.252, 0.003,-0.041> < 0.248, 0.003, 0.022> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.089, 0.003,-0.007> < 0.089, 0.003, 0.021> < 0.073, 0.003, 0.059> < 0.053, 0.003, 0.079> < 0.032, 0.117,-0.015> < 0.032, 0.117, 0.013> < 0.038, 0.121, 0.026> < 0.018, 0.121, 0.045> < 0.027, 0.299,-0.007> < 0.027, 0.299, 0.010> < 0.014, 0.299, 0.039> < 0.003, 0.299, 0.050> < 0.039, 0.426,-0.007> < 0.039, 0.426, 0.009> < 0.029, 0.426, 0.032> < 0.018, 0.426, 0.043> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.053, 0.003, 0.079> < 0.034, 0.003, 0.098> <-0.004, 0.003, 0.114> <-0.032, 0.003, 0.114> < 0.018, 0.121, 0.045> <-0.001, 0.121, 0.065> <-0.003, 0.122, 0.067> <-0.030, 0.122, 0.067> < 0.003, 0.299, 0.050> <-0.009, 0.299, 0.062> <-0.038, 0.300, 0.073> <-0.054, 0.300, 0.073> < 0.018, 0.426, 0.043> < 0.006, 0.426, 0.055> <-0.016, 0.426, 0.064> <-0.032, 0.426, 0.064> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.032, 0.003, 0.114> <-0.060, 0.003, 0.114> <-0.098, 0.003, 0.098> <-0.117, 0.003, 0.079> <-0.033,-0.030, 0.127> <-0.060,-0.030, 0.127> <-0.110,-0.032, 0.107> <-0.129,-0.032, 0.088> <-0.038,-0.007, 0.199> <-0.094,-0.007, 0.174> <-0.155,-0.006, 0.166> <-0.199,-0.006, 0.123> <-0.038, 0.003, 0.233> <-0.094, 0.003, 0.208> <-0.181, 0.003, 0.193> <-0.224, 0.003, 0.150> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.117, 0.003, 0.079> <-0.137, 0.003, 0.059> <-0.153, 0.003, 0.021> <-0.153, 0.003,-0.007> <-0.129,-0.032, 0.088> <-0.149,-0.032, 0.068> <-0.169,-0.034, 0.020> <-0.169,-0.034,-0.007> <-0.199,-0.006, 0.123> <-0.236,-0.006, 0.087> <-0.248,-0.009, 0.043> <-0.250,-0.009,-0.010> <-0.224, 0.003, 0.150> <-0.262, 0.003, 0.114> <-0.287, 0.003, 0.040> <-0.289, 0.003,-0.013> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.153, 0.003,-0.007> <-0.153, 0.003,-0.035> <-0.137, 0.003,-0.073> <-0.117, 0.003,-0.092> <-0.169,-0.034,-0.007> <-0.169,-0.034,-0.035> <-0.150,-0.033,-0.080> <-0.131,-0.033,-0.100> <-0.250,-0.009,-0.010> <-0.252,-0.009,-0.053> <-0.239, 0.002,-0.089> <-0.208, 0.002,-0.119> <-0.289, 0.003,-0.013> <-0.291, 0.003,-0.056> <-0.267, 0.003,-0.116> <-0.236, 0.003,-0.147> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.117, 0.003,-0.092> <-0.098, 0.003,-0.112> <-0.060, 0.003,-0.128> <-0.032, 0.003,-0.128> <-0.131,-0.033,-0.100> <-0.111,-0.033,-0.120> <-0.062,-0.029,-0.140> <-0.034,-0.029,-0.140> <-0.208, 0.002,-0.119> <-0.165, 0.002,-0.162> <-0.124,-0.012,-0.207> <-0.063,-0.012,-0.210> <-0.236, 0.003,-0.147> <-0.192, 0.003,-0.189> <-0.116, 0.003,-0.237> <-0.055, 0.003,-0.241> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.038, 0.003, 0.233> <-0.094, 0.003, 0.208> <-0.181, 0.003, 0.193> <-0.224, 0.003, 0.150> <-0.037, 0.049, 0.389> <-0.093, 0.049, 0.364> <-0.268, 0.036, 0.285> <-0.312, 0.036, 0.242> < 0.060, 0.168, 0.607> <-0.078, 0.154, 0.525> <-0.308, 0.139, 0.428> <-0.446, 0.125, 0.346> <-0.029, 0.188, 0.742> <-0.167, 0.174, 0.660> <-0.350, 0.155, 0.551> <-0.488, 0.141, 0.469> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.224, 0.003, 0.150> <-0.262, 0.003, 0.114> <-0.287, 0.003, 0.040> <-0.289, 0.003,-0.013> <-0.316, 0.023, 0.231> <-0.354, 0.023, 0.194> <-0.438, 0.012, 0.077> <-0.440, 0.012, 0.024> <-0.414, 0.082, 0.379> <-0.492, 0.072, 0.287> <-0.579, 0.062, 0.126> <-0.657, 0.052, 0.034> <-0.530, 0.068, 0.418> <-0.608, 0.058, 0.325> <-0.713, 0.044, 0.202> <-0.792, 0.034, 0.110> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.289, 0.003,-0.013> <-0.291, 0.003,-0.056> <-0.267, 0.003,-0.116> <-0.236, 0.003,-0.147> <-0.427, 0.046,-0.023> <-0.429, 0.046,-0.066> <-0.371, 0.050,-0.208> <-0.340, 0.050,-0.238> <-0.605, 0.196, 0.010> <-0.559, 0.188,-0.110> <-0.488, 0.178,-0.293> <-0.442, 0.170,-0.414> <-0.736, 0.184,-0.051> <-0.690, 0.176,-0.171> <-0.628, 0.166,-0.331> <-0.582, 0.158,-0.451> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.236, 0.003,-0.147> <-0.192, 0.003,-0.189> <-0.116, 0.003,-0.237> <-0.055, 0.003,-0.241> <-0.329, 0.006,-0.239> <-0.285, 0.006,-0.281> <-0.119, 0.046,-0.363> <-0.058, 0.046,-0.367> <-0.475,-0.001,-0.360> <-0.341, 0.037,-0.413> <-0.129, 0.088,-0.495> < 0.005, 0.126,-0.548> <-0.510, 0.017,-0.484> <-0.376, 0.055,-0.537> <-0.197, 0.107,-0.608> <-0.063, 0.145,-0.661> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.055, 0.003,-0.241> < 0.013, 0.003,-0.245> < 0.109, 0.003,-0.215> < 0.160, 0.003,-0.170> <-0.023, 0.071,-0.374> < 0.045, 0.071,-0.378> < 0.167, 0.015,-0.303> < 0.218, 0.015,-0.258> < 0.011, 0.228,-0.527> < 0.089, 0.160,-0.472> < 0.242, 0.081,-0.423> < 0.320, 0.013,-0.368> < 0.095, 0.269,-0.649> < 0.172, 0.201,-0.594> < 0.276, 0.110,-0.520> < 0.354, 0.042,-0.465> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.055, 0.003,-0.241> < 0.013, 0.003,-0.245> < 0.109, 0.003,-0.215> < 0.160, 0.003,-0.170> <-0.063,-0.012,-0.210> < 0.005,-0.012,-0.214> < 0.076,-0.002,-0.192> < 0.128,-0.002,-0.148> <-0.034,-0.029,-0.140> <-0.006,-0.029,-0.140> < 0.046,-0.033,-0.122> < 0.066,-0.033,-0.102> <-0.032, 0.003,-0.128> <-0.004, 0.003,-0.128> < 0.034, 0.003,-0.112> < 0.053, 0.003,-0.092> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.160, 0.003,-0.170> < 0.208, 0.003,-0.129> < 0.252, 0.003,-0.041> < 0.248, 0.003, 0.022> < 0.279, 0.024,-0.252> < 0.327, 0.024,-0.210> < 0.370, 0.024,-0.068> < 0.367, 0.024,-0.004> < 0.420, 0.119,-0.397> < 0.446, 0.114,-0.285> < 0.515, 0.103,-0.115> < 0.540, 0.097,-0.004> < 0.557, 0.090,-0.438> < 0.583, 0.084,-0.327> < 0.617, 0.078,-0.178> < 0.643, 0.072,-0.067> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.248, 0.003, 0.022> < 0.246, 0.003, 0.075> < 0.174, 0.003, 0.120> < 0.143, 0.003, 0.162> < 0.398, 0.060, 0.022> < 0.395, 0.060, 0.074> < 0.210,-0.139, 0.138> < 0.178,-0.139, 0.180> < 0.584, 0.275,-0.082> < 0.521, 0.172, 0.061> < 0.437, 0.032, 0.285> < 0.375,-0.071, 0.429> < 0.722, 0.244,-0.009> < 0.659, 0.141, 0.135> < 0.576, 0.004, 0.326> < 0.514,-0.099, 0.469> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.248, 0.003, 0.022> < 0.246, 0.003, 0.075> < 0.174, 0.003, 0.120> < 0.143, 0.003, 0.162> < 0.203,-0.014, 0.022> < 0.200,-0.014, 0.075> < 0.193, 0.003, 0.089> < 0.161, 0.003, 0.131> < 0.110,-0.040,-0.004> < 0.110,-0.040, 0.024> < 0.083,-0.031, 0.069> < 0.063,-0.031, 0.088> < 0.089, 0.003,-0.007> < 0.089, 0.003, 0.021> < 0.073, 0.003, 0.059> < 0.053, 0.003, 0.079> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.414, 0.299, 0.420> < 0.298, 0.223, 0.507> < 0.142, 0.122, 0.622> < 0.025, 0.047, 0.708> < 0.353, 0.311, 0.291> < 0.237, 0.236, 0.378> < 0.052, 0.134, 0.509> <-0.065, 0.059, 0.595> < 0.108, 0.141, 0.145> < 0.079, 0.141, 0.195> <-0.056, 0.010, 0.317> <-0.068, 0.014, 0.373> < 0.143, 0.003, 0.162> < 0.113, 0.003, 0.212> <-0.026,-0.001, 0.176> <-0.038, 0.003, 0.233> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.153, 0.003,-0.007> <-0.101, 0.120,-0.005> <-0.133, 0.302,-0.007> <-0.103, 0.426,-0.007> <-0.153, 0.003, 0.021> <-0.101, 0.120, 0.022> <-0.133, 0.302, 0.009> <-0.103, 0.426, 0.009> <-0.137, 0.003, 0.059> <-0.097, 0.121, 0.029> <-0.122, 0.301, 0.038> <-0.094, 0.426, 0.032> <-0.117, 0.003, 0.079> <-0.078, 0.121, 0.049> <-0.110, 0.301, 0.049> <-0.082, 0.426, 0.043> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.103, 0.426,-0.007> <-0.076, 0.535,-0.007> <-0.005, 0.670,-0.006> < 0.037, 0.774,-0.006> <-0.103, 0.426, 0.009> <-0.076, 0.535, 0.010> <-0.005, 0.669, 0.009> < 0.037, 0.774, 0.010> <-0.094, 0.426, 0.032> <-0.069, 0.533, 0.026> < 0.004, 0.668, 0.031> < 0.045, 0.770, 0.030> <-0.082, 0.426, 0.043> <-0.058, 0.533, 0.038> < 0.014, 0.663, 0.042> < 0.055, 0.765, 0.041> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.137, 0.723,-0.055> < 0.101, 0.634,-0.055> < 0.029, 0.521,-0.052> < 0.018, 0.426,-0.057> < 0.127, 0.728,-0.065> < 0.092, 0.639,-0.066> < 0.018, 0.521,-0.063> < 0.006, 0.426,-0.068> < 0.109, 0.737,-0.074> < 0.060, 0.653,-0.107> < 0.002, 0.527,-0.070> <-0.016, 0.426,-0.078> < 0.095, 0.744,-0.073> < 0.046, 0.660,-0.107> <-0.015, 0.527,-0.070> <-0.032, 0.426,-0.078> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.018, 0.426,-0.057> < 0.029, 0.426,-0.045> < 0.039, 0.426,-0.023> < 0.039, 0.426,-0.007> < 0.003, 0.299,-0.063> < 0.015, 0.299,-0.052> < 0.027, 0.299,-0.023> < 0.027, 0.299,-0.007> < 0.014, 0.120,-0.061> < 0.033, 0.120,-0.041> < 0.032, 0.117,-0.042> < 0.032, 0.117,-0.015> < 0.053, 0.003,-0.092> < 0.073, 0.003,-0.073> < 0.089, 0.003,-0.035> < 0.089, 0.003,-0.007> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.039, 0.426,-0.007> < 0.039, 0.426, 0.009> < 0.029, 0.426, 0.032> < 0.018, 0.426, 0.043> < 0.047, 0.518,-0.007> < 0.047, 0.518, 0.009> < 0.041, 0.521, 0.026> < 0.029, 0.521, 0.038> < 0.120, 0.627,-0.008> < 0.121, 0.627, 0.008> < 0.127, 0.629, 0.061> < 0.117, 0.634, 0.072> < 0.155, 0.714,-0.008> < 0.156, 0.714, 0.007> < 0.148, 0.717, 0.028> < 0.139, 0.722, 0.039> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.018, 0.426, 0.043> < 0.006, 0.426, 0.055> <-0.016, 0.426, 0.064> <-0.032, 0.426, 0.064> < 0.029, 0.521, 0.038> < 0.018, 0.521, 0.049> < 0.002, 0.527, 0.056> <-0.014, 0.527, 0.056> < 0.117, 0.634, 0.072> < 0.108, 0.639, 0.083> < 0.087, 0.642, 0.092> < 0.073, 0.649, 0.092> < 0.139, 0.722, 0.039> < 0.129, 0.727, 0.050> < 0.111, 0.736, 0.059> < 0.097, 0.743, 0.060> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.032, 0.426, 0.064> <-0.048, 0.426, 0.064> <-0.071, 0.426, 0.055> <-0.082, 0.426, 0.043> <-0.054, 0.300, 0.073> <-0.070, 0.300, 0.073> <-0.099, 0.301, 0.061> <-0.110, 0.301, 0.049> <-0.030, 0.122, 0.067> <-0.058, 0.122, 0.067> <-0.058, 0.121, 0.068> <-0.078, 0.121, 0.049> <-0.032, 0.003, 0.114> <-0.060, 0.003, 0.114> <-0.098, 0.003, 0.098> <-0.117, 0.003, 0.079> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.103, 0.426,-0.007> <-0.103, 0.426,-0.023> <-0.094, 0.426,-0.045> <-0.082, 0.426,-0.057> <-0.133, 0.302,-0.007> <-0.133, 0.302,-0.023> <-0.121, 0.301,-0.052> <-0.110, 0.301,-0.063> <-0.101, 0.120,-0.005> <-0.101, 0.120,-0.033> <-0.094, 0.121,-0.048> <-0.075, 0.121,-0.067> <-0.153, 0.003,-0.007> <-0.153, 0.003,-0.035> <-0.137, 0.003,-0.073> <-0.117, 0.003,-0.092> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.082, 0.426,-0.057> <-0.071, 0.426,-0.068> <-0.048, 0.426,-0.078> <-0.032, 0.426,-0.078> <-0.110, 0.301,-0.063> <-0.098, 0.301,-0.075> <-0.070, 0.300,-0.087> <-0.054, 0.300,-0.087> <-0.075, 0.121,-0.067> <-0.055, 0.121,-0.087> <-0.053, 0.122,-0.082> <-0.026, 0.122,-0.082> <-0.117, 0.003,-0.092> <-0.098, 0.003,-0.112> <-0.060, 0.003,-0.128> <-0.032, 0.003,-0.128> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.137, 0.723,-0.055> < 0.147, 0.718,-0.044> < 0.155, 0.714,-0.023> < 0.155, 0.714,-0.008> < 0.101, 0.634,-0.055> < 0.111, 0.629,-0.045> < 0.120, 0.627,-0.023> < 0.120, 0.627,-0.008> < 0.029, 0.521,-0.052> < 0.041, 0.521,-0.040> < 0.047, 0.518,-0.023> < 0.047, 0.518,-0.007> < 0.018, 0.426,-0.057> < 0.029, 0.426,-0.045> < 0.039, 0.426,-0.023> < 0.039, 0.426,-0.007> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.155, 0.714,-0.008> < 0.156, 0.714, 0.007> < 0.148, 0.717, 0.028> < 0.139, 0.722, 0.039> < 0.155, 0.714,-0.023> < 0.155, 0.714,-0.008> < 0.156, 0.751, 0.016> < 0.147, 0.756, 0.027> < 0.147, 0.718,-0.044> < 0.137, 0.723,-0.055> < 0.126, 0.768,-0.062> < 0.112, 0.775,-0.062> < 0.137, 0.723,-0.055> < 0.127, 0.728,-0.065> < 0.109, 0.737,-0.074> < 0.095, 0.744,-0.073> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.139, 0.722, 0.039> < 0.129, 0.727, 0.050> < 0.111, 0.736, 0.059> < 0.097, 0.743, 0.060> < 0.147, 0.756, 0.027> < 0.137, 0.761, 0.038> < 0.119, 0.770, 0.048> < 0.106, 0.777, 0.048> < 0.112, 0.775,-0.062> < 0.099, 0.782,-0.061> < 0.080, 0.791,-0.053> < 0.071, 0.796,-0.042> < 0.095, 0.744,-0.073> < 0.081, 0.751,-0.073> < 0.063, 0.761,-0.064> < 0.053, 0.766,-0.053> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.097, 0.743, 0.060> < 0.083, 0.750, 0.060> < 0.065, 0.760, 0.052> < 0.055, 0.765, 0.041> < 0.106, 0.777, 0.048> < 0.092, 0.784, 0.048> < 0.055, 0.765, 0.041> < 0.045, 0.770, 0.030> < 0.071, 0.796,-0.042> < 0.061, 0.801,-0.031> < 0.037, 0.774,-0.006> < 0.037, 0.774, 0.010> < 0.053, 0.766,-0.053> < 0.044, 0.770,-0.042> < 0.036, 0.774,-0.021> < 0.037, 0.774,-0.006> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.097, 0.743, 0.060> < 0.083, 0.750, 0.060> < 0.065, 0.760, 0.052> < 0.055, 0.765, 0.041> < 0.073, 0.649, 0.092> < 0.060, 0.656, 0.093> < 0.023, 0.658, 0.052> < 0.014, 0.663, 0.042> <-0.014, 0.527, 0.056> <-0.031, 0.527, 0.056> <-0.046, 0.533, 0.049> <-0.058, 0.533, 0.038> <-0.032, 0.426, 0.064> <-0.048, 0.426, 0.064> <-0.071, 0.426, 0.055> <-0.082, 0.426, 0.043> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.037, 0.774,-0.006> < 0.036, 0.774,-0.021> < 0.044, 0.770,-0.042> < 0.053, 0.766,-0.053> <-0.005, 0.670,-0.006> <-0.005, 0.670,-0.021> <-0.008, 0.679,-0.076> < 0.002, 0.675,-0.087> <-0.076, 0.535,-0.007> <-0.076, 0.535,-0.023> <-0.070, 0.533,-0.040> <-0.058, 0.533,-0.051> <-0.103, 0.426,-0.007> <-0.103, 0.426,-0.023> <-0.094, 0.426,-0.045> <-0.082, 0.426,-0.057> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 < 0.053, 0.766,-0.053> < 0.063, 0.761,-0.064> < 0.081, 0.751,-0.073> < 0.095, 0.744,-0.073> < 0.002, 0.675,-0.087> < 0.011, 0.670,-0.098> < 0.033, 0.667,-0.107> < 0.046, 0.660,-0.107> <-0.058, 0.533,-0.051> <-0.047, 0.533,-0.063> <-0.031, 0.527,-0.070> <-0.015, 0.527,-0.070> <-0.082, 0.426,-0.057> <-0.071, 0.426,-0.068> <-0.048, 0.426,-0.078> <-0.032, 0.426,-0.078> }
    bicubic_patch { type 1 flatness 0 u_steps 3 v_steps 3 <-0.032, 0.003, 0.114> <-0.033,-0.030, 0.127> <-0.038,-0.007, 0.199> <-0.038, 0.003, 0.233> <-0.004, 0.003, 0.114> <-0.005,-0.030, 0.127> <-0.026,-0.011, 0.142> <-0.026,-0.001, 0.176> < 0.034, 0.003, 0.098> < 0.044,-0.031, 0.108> < 0.132, 0.003, 0.181> < 0.113, 0.003, 0.212> < 0.053, 0.003, 0.079> < 0.063,-0.031, 0.088> < 0.161, 0.003, 0.131> < 0.143, 0.003, 0.162> }
    rotate z*180
    translate y*-1
  
    texture {
      pigment {
        ripples
        color_map { [0 rgb <0.4, 0.6, 0.0>] [1 rgb <0.3, 0.5, 0.0>] }
        turbulence 1
        omega 0.5
        quick_color green 1
      }
      finish { phong 0.1 phong_size 5 crand 0.2 }
      normal { wrinkles 0.2 scale 0.2 }
    }
  }

}

//----------------------------------------------------------------------------------------------Ice
// It's just an isosurface
#declare ice = isosurface {
  function { f_rounded_box(x,y,z,0,0.5,0.5,0.5)-f_noise3d(x*1.5,y*1.5,z*1.5)/6 }

  texture {
    pigment {
      agate
      agate_turb 0.8
      color_map{[0.5 color rgbt 1][1 color rgbt <1,1,1,0.6>]}
      quick_color rgb 1
    }
    finish {
      specular 0.7
      roughness 0.001
      reflection { 0,1 fresnel on }
      conserve_energy
    }
  }

  interior {
   ior 1.333
   fade_distance 8
   fade_power 1
  }

  photons {
    target
    reflection on
    refraction on
  }
}

//------------------------------------------------------------------------------------------Martini
// The final Martini
#declare martini = union {
  object {glass}
  object {liquid}
  object {fraise scale 0.5 rotate <0,0,140> translate <2.2,5.4,0>}
  object {ice rotate 6888 translate <-0.3,5,0.1>}
}

object{martini translate y*-3 rotate <-50,-40,10>}

//----------------------------------------------------------------------------------------------Sun
// A distant parallel area light
light_source {<1,0,0>*100 color rgb 1 parallel point_at 0 area_light z,y,2,2 circular orient}

//--------------------------------------------------------------------------------------------Earth
// The Planet with a blue halo and some spheres scattered around it
// Download land_ocean_ice_cloud_2048.jpg from http://earthobservatory.nasa.gov/Newsroom/BlueMarble/
// Download moonmap1k.jpg and moonbump1k.jpg from http://gw.marketingden.com/planets/moon.html
sphere {
  0, 0.5
  pigment{image_map{jpeg "land_ocean_ice_cloud_2048.jpg" map_type 2 interpolate 2} translate y*0.5 rotate y*90 quick_color blue 1}
  scale 15
  translate x*-15
}

sphere {
  0, 0.5
  pigment{color rgbf 1 quick_color blue 1}
  interior{media{emission 0.05 density{blue 0.1}}}
  scale 20
  translate x*-15
  hollow
}

union{
  #declare n=0;
  #while (n<500)
    sphere {VRand_On_Sphere(stream)*(rand(stream)/10+0.5),0.002 pigment{color rgb rand(stream)/2+0.5} finish{specular 1} no_shadow}
    #declare n=n+1;
  #end
  scale 15
  translate x*-15
}

//---------------------------------------------------------------------------------------------Moon
// The satellite with a white halo
sphere {
  0.0, 0.5
  texture {
    pigment{image_map{jpeg"moonmap1k.jpg"map_type 2}quick_color rgb 0.5}
    normal{bump_map{jpeg"moonbump1k.jpg"map_type 2 bump_size 1}}
    finish{ambient 0.02}
    translate y*0.5
  }
  scale 8
  translate <-3,0,9>
}

sphere {
  0, 0.55
  pigment{color rgbf 1 quick_color rgb 0.5}
  interior{media{emission 0.05 density{rgb 0.1}}}
  scale 9
  translate <-3,0,9>
  hollow
}

//----------------------------------------------------------------------------------------------HUD
// This is quite tricky (at least for me...): I placed the hud elements around the origin in a 
// <-1,0,-0.5>,<1,0,0.5> box. After that I oriented them towards the camera, I moved it and saled 
// it a bit to make it fit the field of view. Then I added ambient 1 finish and no_shadow.
union {

  plane {y,-0.0001
    texture{
      spherical
      texture_map{
        [0.0 pigment{gradient z color_map{[0 transmit 1][0.5 rgbt <0.0, 1.0, 0.0, 0.5>][1 transmit 1]}scale 0.005}]
        [0.7 pigment{color rgbt 1}]
      }
      translate <(0.824-0.018)/2+0.018,0,(0.419+0.375)/2-0.375>
    }
  }
  
  union{
    disc {<(0.824-0.018)/2+0.018,0,(0.419+0.375)/2-0.375> y, 0.4*0.02,0}
    difference{
      box {<0.018-0.005,-0.0001,-0.375-0.005>,<0.824+0.005,0.0001,0.419+0.005>}
      union{
        box {<0.018+0.015,-0.0002,-0.375+0.015>,<0.824-0.015,0.0002,0.419-0.015>}
        box {<0.018+0.10,-0.0002,-0.375-0.10>,<0.824-0.10,0.0002,0.419+0.10>}
        box {<0.018-0.10,-0.0002,-0.375+0.10>,<0.824+0.10,0.0002,0.419-0.10>}
      }
    }
    pigment{color rgbt <0,4,0,0.5>}
  }
  
  union{
    disc {<(0.824-0.018)/2+0.018,0,(0.419+0.375)/2-0.375> y, 1.6*0.02,0.4*0.02}
    difference{
      box {<0.018,-0.0001,-0.375>,<0.824,0.0001,0.419>}
      box {<0.018+0.01,-0.0002,-0.375+0.01>,<0.824-0.01,0.0002,0.419-0.01>}
    }
    pigment{color rgbt <0,2,0,0.5>}
  }
  
  union{
    disc {<-0.3, 0, 8> y, 0.4,0}
    box {<-0.4,-0.0001,0>,<-0.2,0.0001,8>}
    text{ttf "courbd" "Earth, planet." 0,0 rotate x*90 translate z*6}
    text{ttf "courbd" "Mass: 5.972e24 kg" 0,0 rotate x*90 translate z*5}
    text{ttf "courbd" "Diameter: 12,756.3 km" 0,0 rotate x*90 translate z*4}
    text{ttf "courbd" "Orbit: 149,600,000 km from Sun" 0,0 rotate x*90 translate z*3}
    text{ttf "courbd" "Composition: 34.6% Fe, 29.5% O, 15.2% Si," 0,0 rotate x*90 translate z*2}
    text{ttf "courbd" "12.7% Mg, 2.4% Ni, 1.9% S, 0.05% Ti" 0,0 rotate x*90 translate z*1}
    text{ttf "courbd" "Atmosphere: 77% N, 21% O, traces of Ar, CO2 and H20" 0,0 rotate x*90 translate z*0}
    scale 0.02
    translate <-0.8,0,0.2>
    pigment{color rgbt <0,2,0,0.5>}
  }
  
  union{
    disc {<-0.3, 0, 5> y, 0.4,0}
    box {<-0.4,-0.0001,0>,<-0.2,0.0001,5>}
    text{ttf "courbd" "Moon, satellite." 0,0 rotate x*90 translate z*3}
    text{ttf "courbd" "Mass: 7.35e22 kg" 0,0 rotate x*90 translate z*2}
    text{ttf "courbd" "Diameter: 3476 km" 0,0 rotate x*90 translate z*1}
    text{ttf "courbd" "Orbit: 384,400 km from Earth" 0,0 rotate x*90 translate z*0}
    scale 0.02
    translate <0.45,0,-0.15>
    pigment{color rgbt <0,2,0,0.5>}
  }
  
  union{
    text{ttf "courbd" "Object under analysis" 0,0 rotate x*90 translate z*6}
    text{ttf "courbd" "Mass: 6.888e13 kg" 0,0 rotate x*90 translate z*5}
    text{ttf "courbd" "Composition: 36.8% Sio2, 23.5% H2O, 12% fructose," 0,0 rotate x*90 translate z*4} 
    text{ttf "courbd" "             8% Na2O, 2.6% CaO, 2.1% chlorophyll," 0,0 rotate x*90 translate z*3} 
    text{ttf "courbd" "             1.8% MgO, 0.5% Al2O3, 0.3% K2O," 0,0 rotate x*90 translate z*2} 
    text{ttf "courbd" "             12.4% under analysis..." 0,0 rotate x*90 translate z*1}
    text{ttf "courbd" "Analysis: no match in database" 0,0 rotate x*90 translate z*0}
    scale 0.02
    translate <0.038,0,-0.355>
    pigment{color rgbt <0,4,0,0.5>}
  }
  
  text{ttf "courbd" "Probe ID: 2431-1248-6997-6888. Nickname: StrawberryJuice. Signature: Paolo Brasolin inc.. Date: 4/2/2004" 0,0
    rotate x*90
    scale 0.03
    translate <-0.975,0,-0.47>
    pigment{color rgbt <0,2,0,0.5>}
  }
  
  text{ttf "courbd" "Analyzing molecular structures..." 0,0
    rotate x*90
    scale 0.03
    translate <-0.83,0,-0.35>
    pigment{color rgbt <0,3,0,0.5>}
  }

// This is the chlorophyll molecula
  #declare Proportion = 0.2 ;
  #declare Unknown_Radius = 1.6 * Proportion ;
  #declare Carbon_Radius = 1.85 * Proportion ;
  #declare Nitrogen_Radius = 1.54 * Proportion ;
  #declare Oxygen_Radius = 1.4 * Proportion ;
  #declare Hydrogen_Radius = 1.2 * Proportion ;
  #declare Unknown_Color = color rgb < 0.6,1,0.6 > ;
  #declare Carbon_Color = color rgb < 0,1,0 > ;
  #declare Nitrogen_Color = color rgb < 0.6,0,0 > ;
  #declare Oxygen_Color = color rgb < 0,0,0.6 > ;
  #declare Hydrogen_Color = color rgb < 0.6,1,0 > ;
  #declare StickRadius = 0.1 ;
  #declare AtomObject = sphere { < 0 , 0 , 0 > , 1 }
  #declare StickObject = cylinder { < 0 , 0 , 0 > , < 0 , 0 , 1 > , 1 }
  
  union { 
    object { AtomObject pigment { Unknown_Color } scale Unknown_Radius translate < 5.814 , 7.901 , -30.79 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 3.149 , 9.127 , -32.553 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 7.561 , 7.561 , -33.767 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 8.734 , 7.364 , -29.079 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 4.284 , 8.853 , -27.804 > }
    object { AtomObject pigment { Nitrogen_Color } scale Nitrogen_Radius translate < 5.414 , 8.329 , -32.81 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 4.176 , 8.724 , -33.336 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 4.234 , 8.885 , -34.812 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 5.508 , 8.11 , -35.193 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 6.231 , 7.974 , -33.857 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 5.069 , 6.778 , -35.737 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 4.414 , 10.364 , -34.998 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 5.765 , 10.849 , -34.445 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 5.916 , 12.319 , -34.97 > }
    object { AtomObject pigment { Oxygen_Color } scale Oxygen_Radius translate < 5.142 , 12.71 , -35.755 > }
    object { AtomObject pigment { Oxygen_Color } scale Oxygen_Radius translate < 6.849 , 13.1 , -34.365 > }
    object { AtomObject pigment { Nitrogen_Color } scale Nitrogen_Radius translate < 7.823 , 7.602 , -31.299 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 8.325 , 7.402 , -32.599 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 9.709 , 6.944 , -32.504 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 10.063 , 6.898 , -31.198 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 8.84 , 7.323 , -30.421 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 10.619 , 6.52 , -33.613 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 11.404 , 6.476 , -30.694 > }
    object { AtomObject pigment { Oxygen_Color } scale Oxygen_Radius translate < 12.23 , 5.86 , -31.424 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 11.98 , 6.812 , -29.317 > }
    object { AtomObject pigment { Nitrogen_Color } scale Nitrogen_Radius translate < 6.414 , 8.128 , -28.791 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 7.622 , 7.715 , -28.293 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 7.607 , 7.55 , -26.801 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 6.354 , 8.377 , -26.431 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 5.58 , 8.411 , -27.7 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 7.295 , 6.054 , -26.6 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 6.117 , 9.115 , -25.348 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 6.78 , 8.846 , -23.951 > }
    object { AtomObject pigment { Nitrogen_Color } scale Nitrogen_Radius translate < 4.151 , 8.893 , -30.252 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 3.559 , 9.101 , -29.003 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 2.191 , 9.503 , -29.146 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 1.987 , 9.548 , -30.54 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 3.201 , 9.181 , -31.159 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 1.206 , 9.693 , -28.061 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 1.006 , 9.719 , -31.567 > }
    object { AtomObject pigment { Oxygen_Color } scale Oxygen_Radius translate < -0.232 , 9.845 , -31.472 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 1.731 , 9.571 , -32.936 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 0.954 , 8.467 , -33.637 > }
    object { AtomObject pigment { Oxygen_Color } scale Oxygen_Radius translate < 0.49 , 8.533 , -34.647 > }
    object { AtomObject pigment { Oxygen_Color } scale Oxygen_Radius translate < 0.87 , 7.311 , -32.986 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 0 , 6.33 , -33.596 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 7.224 , 14.416 , -34.971 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 8.03 , 14.064 , -36.222 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 9.229 , 14.456 , -36.548 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 10 , 15.424 , -35.643 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 9.845 , 13.847 , -37.852 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 10.21 , 14.811 , -39.006 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 10.445 , 13.908 , -40.267 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 11.728 , 14.401 , -40.937 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 11.656 , 15.933 , -41.054 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 11.893 , 13.769 , -42.326 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 13.295 , 14.105 , -42.858 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 13.478 , 13.468 , -44.244 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 14.922 , 13.678 , -44.734 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 15.274 , 15.178 , -44.701 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 15.069 , 13.129 , -46.168 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 16.549 , 13.15 , -46.6 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 16.688 , 12.603 , -48.036 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 18.176 , 12.549 , -48.442 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 18.306 , 12.13 , -49.919 > }
    object { AtomObject pigment { Carbon_Color } scale Carbon_Radius translate < 18.932 , 11.55 , -47.546 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 8.063 , 7.337 , -34.708 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 9.632 , 7.088 , -28.527 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 3.754 , 9.033 , -26.869 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 3.346 , 8.469 , -35.288 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 6.1 , 8.66 , -35.924 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 5.945 , 6.194 , -36.02 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 4.506 , 6.241 , -34.974 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 4.437 , 6.933 , -36.612 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 3.637 , 10.934 , -34.488 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 4.379 , 10.647 , -36.05 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 6.503 , 10.162 , -34.858 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 5.668 , 10.782 , -33.361 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 10.549 , 7.234 , -34.434 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 11.646 , 6.486 , -33.248 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 10.326 , 5.531 , -33.966 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 12.343 , 7.84 , -29.315 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 11.203 , 6.7 , -28.561 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 12.805 , 6.136 , -29.093 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 8.518 , 7.936 , -26.344 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 7.262 , 5.829 , -25.534 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 6.33 , 5.82 , -27.05 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 8.072 , 5.454 , -27.073 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 5.046 , 9.035 , -25.159 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 6.425 , 10.129 , -25.603 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 6.426 , 9.584 , -23.232 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 6.512 , 7.847 , -23.607 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 7.864 , 8.92 , -24.042 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 1.096 , 10.757 , -27.852 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 0.244 , 9.282 , -28.367 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 1.551 , 9.181 , -27.163 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 1.753 , 10.514 , -33.483 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < -0.003 , 5.422 , -32.993 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < -1.013 , 6.729 , -33.656 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 0.359 , 6.099 , -34.599 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 7.811 , 14.925 , -34.206 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 6.277 , 14.909 , -35.19 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 7.425 , 14.425 , -37.054 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 8.131 , 12.979 , -36.191 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 10.832 , 15.857 , -36.199 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 9.333 , 16.219 , -35.309 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 10.384 , 14.885 , -34.777 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 9.116 , 13.146 , -38.259 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 10.77 , 13.344 , -37.57 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 11.113 , 15.333 , -38.688 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 9.357 , 15.48 , -39.121 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 9.577 , 14.032 , -40.914 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 10.542 , 12.882 , -39.911 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 12.575 , 14.104 , -40.318 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 12.563 , 16.305 , -41.53 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 10.791 , 16.212 , -41.655 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 11.563 , 16.37 , -40.06 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 11.131 , 14.178 , -42.99 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 11.774 , 12.689 , -42.236 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 14.045 , 13.711 , -42.172 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 13.401 , 15.187 , -42.935 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 12.793 , 13.925 , -44.958 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 13.274 , 12.398 , -44.193 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 15.615 , 13.136 , -44.09 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 16.297 , 15.319 , -45.049 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 14.591 , 15.726 , -45.35 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 15.183 , 15.551 , -43.681 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 14.494 , 13.737 , -46.867 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 14.706 , 12.102 , -46.219 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 17.143 , 12.531 , -45.927 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 16.932 , 14.17 , -46.572 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 16.156 , 13.247 , -48.736 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 16.271 , 11.598 , -48.097 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 18.635 , 13.529 , -48.312 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 19.36 , 12.095 , -50.196 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 17.861 , 11.145 , -50.058 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 17.789 , 12.854 , -50.549 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 19.981 , 11.519 , -47.84 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 18.856 , 11.865 , -46.505 > }
    object { AtomObject pigment { Hydrogen_Color } scale Hydrogen_Radius translate < 18.494 , 10.558 , -47.657 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.35285882486 > rotate < 17.3307774065 , 127.322433497 , 0 > translate < 3.149 , 9.127 , -32.553 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.39601432657 > rotate < -2.21684270746 , 2.13629824813 , 0 > translate < 3.149 , 9.127 , -32.553 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.53445397455 > rotate < -16.8192694811 , -105.114847474 , 0 > translate < 3.149 , 9.127 , -32.553 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.39555329529 > rotate < -17.2139106261 , -93.871256232 , 0 > translate < 7.561 , 7.561 , -33.767 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.40470673096 > rotate < 6.49928912436 , 33.1890838116 , 0 > translate < 7.561 , 7.561 , -33.767 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08979860525 > rotate < 11.8612620896 , 151.921192863 , 0 > translate < 7.561 , 7.561 , -33.767 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.34680399465 > rotate < 1.7444928586 , 175.483778398 , 0 > translate < 8.734 , 7.364 , -29.079 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.40625068889 > rotate < -14.4538362821 , -54.7460107905 , 0 > translate < 8.734 , 7.364 , -29.079 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08962562378 > rotate < 14.672757511 , 58.4209853981 , 0 > translate < 8.734 , 7.364 , -29.079 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.37324287728 > rotate < 18.7758017959 , 85.4120218759 , 0 > translate < 4.284 , 8.853 , -27.804 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.42293007558 > rotate < -10.0372416236 , -148.839881712 , 0 > translate < 4.284 , 8.853 , -27.804 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08973620661 > rotate < -9.50755116636 , -29.5465123971 , 0 > translate < 4.284 , 8.853 , -27.804 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Nitrogen_Color ] [ 1 color Carbon_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.40190762891 > rotate < -16.3652115114 , -113.019611699 , 0 > translate < 5.414 , 8.329 , -32.81 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Nitrogen_Color ] [ 1 color Carbon_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.37467196087 > rotate < 14.9658588887 , 142.034202879 , 0 > translate < 5.414 , 8.329 , -32.81 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.48588727702 > rotate < -6.220368584 , 177.749697604 , 0 > translate < 4.176 , 8.724 , -33.336 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.53911078224 > rotate < 30.2343165877 , 106.649719694 , 0 > translate < 4.234 , 8.885 , -34.812 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.50147827157 > rotate < -80.0733464597 , 135.939190946 , 0 > translate < 4.234 , 8.885 , -34.812 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09003486183 > rotate < 22.4352536408 , -118.192912264 , 0 > translate < 4.234 , 8.885 , -34.812 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.52516261428 > rotate < 5.11590674309 , 28.4208189742 , 0 > translate < 5.508 , 8.11 , -35.193 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.504287539 > rotate < 62.3092933629 , -141.096980559 , 0 > translate < 5.508 , 8.11 , -35.193 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08964443742 > rotate < -30.3148782206 , 140.997733918 , 0 > translate < 5.508 , 8.11 , -35.193 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09019310216 > rotate < 32.3903726363 , 107.903536357 , 0 > translate < 5.069 , 6.778 , -35.737 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.0897279477 > rotate < 29.5237006193 , -36.4227555231 , 0 > translate < 5.069 , 6.778 , -35.737 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09044669746 > rotate < -8.1719051652 , -144.159956335 , 0 > translate < 5.069 , 6.778 , -35.737 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.53825713065 > rotate < -18.3784358982 , 67.7394177834 , 0 > translate < 4.414 , 10.364 , -34.998 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09028849393 > rotate < -31.5200839378 , -56.7202801903 , 0 > translate < 4.414 , 10.364 , -34.998 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08996238467 > rotate < -15.0488200246 , -178.094474456 , 0 > translate < 4.414 , 10.364 , -34.998 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.56822383606 > rotate < -69.6138445361 , 163.953756716 , 0 > translate < 5.765 , 10.849 , -34.445 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08957881771 > rotate < 39.0884138729 , 119.232275064 , 0 > translate < 5.765 , 10.849 , -34.445 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09039167275 > rotate < 3.52280475779 , -5.11340173842 , 0 > translate < 5.765 , 10.849 , -34.445 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Oxygen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.16969312215 > rotate < -19.5285003282 , -135.404261139 , 0 > translate < 5.916 , 12.319 , -34.97 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Oxygen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.35885061725 > rotate < -35.0821577369 , 57.0387743648 , 0 > translate < 5.916 , 12.319 , -34.97 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Oxygen_Color ] [ 1 color Carbon_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.49656840806 > rotate < -61.563456115 , 148.250285686 , 0 > translate < 6.849 , 13.1 , -34.365 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Nitrogen_Color ] [ 1 color Carbon_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.40783663825 > rotate < 8.16717927702 , 158.885740424 , 0 > translate < 7.823 , 7.602 , -31.299 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Nitrogen_Color ] [ 1 color Carbon_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.37222957263 > rotate < 11.7310978884 , 49.1951852375 , 0 > translate < 7.823 , 7.602 , -31.299 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.46090554109 > rotate < 18.2705363623 , 86.0732843633 , 0 > translate < 8.325 , 7.402 , -32.599 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.3539084164 > rotate < 1.94703951712 , 15.1659887057 , 0 > translate < 9.709 , 6.944 , -32.504 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.49591343332 > rotate < 16.4655543878 , 140.629099367 , 0 > translate < 9.709 , 6.944 , -32.504 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.50999437085 > rotate < -16.3472421614 , -57.5712669917 , 0 > translate < 10.063 , 6.898 , -31.198 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.49344601509 > rotate < 16.4135287947 , 69.4018255384 , 0 > translate < 10.063 , 6.898 , -31.198 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09029216268 > rotate < -40.9098192292 , -175.126640664 , 0 > translate < 10.619 , 6.52 , -33.613 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09046320433 > rotate < 1.78673821322 , 70.4345789137 , 0 > translate < 10.619 , 6.52 , -33.613 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09021970263 > rotate < 65.1153255704 , -140.306365817 , 0 > translate < 10.619 , 6.52 , -33.613 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Oxygen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.26278739303 > rotate < 29.1967240482 , 131.469516911 , 0 > translate < 11.404 , 6.476 , -30.694 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.52996764672 > rotate < -12.6862735217 , 22.6995900072 , 0 > translate < 11.404 , 6.476 , -30.694 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09020961287 > rotate < -70.550946471 , 89.6843239682 , 0 > translate < 11.98 , 6.812 , -29.317 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08986650559 > rotate < 5.89840691137 , -45.784824603 , 0 > translate < 11.98 , 6.812 , -29.317 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08985182479 > rotate < 38.3356937093 , 74.8095425575 , 0 > translate < 11.98 , 6.812 , -29.317 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Nitrogen_Color ] [ 1 color Carbon_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.37034192813 > rotate < 17.5407864207 , 67.5960135717 , 0 > translate < 6.414 , 8.128 , -28.791 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Nitrogen_Color ] [ 1 color Carbon_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.40211483125 > rotate < -11.6444586618 , -37.3956088042 , 0 > translate < 6.414 , 8.128 , -28.791 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.50117087635 > rotate < 6.31036975678 , -0.576010548608 , 0 > translate < 7.622 , 7.715 , -28.293 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.54623348819 > rotate < -32.3336086172 , -73.5485878707 , 0 > translate < 7.607 , 7.55 , -26.801 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.54135038197 > rotate < 76.0668906875 , -57.2091197585 , 0 > translate < 7.607 , 7.55 , -26.801 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08984677822 > rotate < -20.7430882444 , 63.359526603 , 0 > translate < 7.607 , 7.55 , -26.801 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.48680630884 > rotate < -1.31034306005 , -148.619770578 , 0 > translate < 6.354 , 8.377 , -26.431 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.33180403964 > rotate < -33.6511960956 , -12.3438205108 , 0 > translate < 6.354 , 8.377 , -26.431 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08998623845 > rotate < 11.9129079358 , -1.77313047122 , 0 > translate < 7.295 , 6.054 , -26.6 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09017475663 > rotate < 12.394670907 , -115.000639357 , 0 > translate < 7.295 , 6.054 , -26.6 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08970546479 > rotate < 33.408699887 , 121.331032139 , 0 > translate < 7.295 , 6.054 , -26.6 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.56956650066 > rotate < 9.86834991282 , 25.3884736491 , 0 > translate < 6.117 , 9.115 , -25.348 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09048704715 > rotate < 4.20709614372 , -79.9920201986 , 0 > translate < 6.117 , 9.115 , -25.348 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08999311924 > rotate < -68.4786760863 , 129.622107808 , 0 > translate < 6.117 , 9.115 , -25.348 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08945904007 > rotate < -42.6408343504 , -26.2133515907 , 0 > translate < 6.78 , 8.846 , -23.951 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09002798129 > rotate < 66.4182544462 , -37.9210783954 , 0 > translate < 6.78 , 8.846 , -23.951 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09032701517 > rotate < -3.89163013077 , 94.7986341466 , 0 > translate < 6.78 , 8.846 , -23.951 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Nitrogen_Color ] [ 1 color Carbon_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.39775856284 > rotate < -8.55795189535 , -25.3599778776 , 0 > translate < 4.151 , 8.893 , -30.252 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Nitrogen_Color ] [ 1 color Carbon_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.34465348696 > rotate < -12.3675164759 , -133.673517326 , 0 > translate < 4.151 , 8.893 , -30.252 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.43299581297 > rotate < -16.291909475 , -95.9675789227 , 0 > translate < 3.559 , 9.101 , -29.003 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.40956624534 > rotate < -1.82946225559 , -171.67434967 , 0 > translate < 2.191 , 9.503 , -29.146 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.4776839987 > rotate < -7.3875198599 , -42.2342381658 , 0 > translate < 2.191 , 9.503 , -29.146 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.41125688661 > rotate < 15.0731402674 , 117.016337778 , 0 > translate < 1.987 , 9.548 , -30.54 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.43050026215 > rotate < -6.86547468681 , -136.312323184 , 0 > translate < 1.987 , 9.548 , -30.54 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08989770162 > rotate < -77.4847622322 , -27.7585406011 , 0 > translate < 1.206 , 9.693 , -28.061 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08995458621 > rotate < 22.1529237304 , -107.645207393 , 0 > translate < 1.206 , 9.693 , -28.061 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08975823007 > rotate < 28.0231959688 , 21.0161304086 , 0 > translate < 1.206 , 9.693 , -28.061 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Oxygen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.24801642617 > rotate < -5.79446628668 , -85.6119121839 , 0 > translate < 1.006 , 9.719 , -31.567 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.55617801038 > rotate < 5.45735177859 , 152.095037744 , 0 > translate < 1.006 , 9.719 , -31.567 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.52116600015 > rotate < 46.5320290025 , -132.056395519 , 0 > translate < 1.731 , 9.571 , -32.936 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09038617013 > rotate < -59.8634381833 , 177.696840688 , 0 > translate < 1.731 , 9.571 , -32.936 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Oxygen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.11344151171 > rotate < -3.39823814673 , -155.325668792 , 0 > translate < 0.954 , 8.467 , -33.637 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Oxygen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.32935811578 > rotate < 60.4113331831 , -7.35237935989 , 0 > translate < 0.954 , 8.467 , -33.637 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Oxygen_Color ] [ 1 color Carbon_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.44615386457 > rotate < 42.7149226719 , -125.03619581 , 0 > translate < 0.87 , 7.311 , -32.986 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08999174309 > rotate < 56.4116591368 , -0.285051277583 , 0 > translate < 0 , 6.33 , -33.596 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09039900954 > rotate < -21.4642794822 , -93.3896694149 , 0 > translate < 0 , 6.33 , -33.596 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09006926385 > rotate < 12.2344907124 , 160.306379675 , 0 > translate < 0 , 6.33 , -33.596 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.52922889065 > rotate < 13.3077499465 , 147.206941283 , 0 > translate < 7.224 , 14.416 , -34.971 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09035544663 > rotate < -27.8280891196 , 37.4997574233 , 0 > translate < 7.224 , 14.416 , -34.971 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08987109329 > rotate < -26.8943697836 , -103.02112625 , 0 > translate < 7.224 , 14.416 , -34.971 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.30289715634 > rotate < -17.5097418238 , 105.210631271 , 0 > translate < 8.03 , 14.064 , -36.222 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09021557501 > rotate < -19.3372045079 , -143.976721511 , 0 > translate < 8.03 , 14.064 , -36.222 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09013164343 > rotate < 84.4384364346 , 72.9371428265 , 0 > translate < 8.03 , 14.064 , -36.222 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.53313078372 > rotate < -39.1525827799 , 40.4287962405 , 0 > translate < 9.229 , 14.456 , -36.548 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.5654881028 > rotate < 22.8932890826 , 154.714294515 , 0 > translate < 9.229 , 14.456 , -36.548 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09034352385 > rotate < -23.3984179513 , 123.75358832 , 0 > translate < 10 , 15.424 , -35.643 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09016971156 > rotate < -46.8230747163 , -63.4005988393 , 0 > translate < 10 , 15.424 , -35.643 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.0899233918 > rotate < 29.6387739736 , 23.9133607727 , 0 > translate < 10 , 15.424 , -35.643 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.54733222031 > rotate < -38.5361121894 , 162.448297378 , 0 > translate < 9.845 , 13.847 , -37.852 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09017934304 > rotate < 40.0168950035 , -119.174581126 , 0 > translate < 9.845 , 13.847 , -37.852 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09002660518 > rotate < 27.4811409565 , 73.0453780913 , 0 > translate < 9.845 , 13.847 , -37.852 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.56867938088 > rotate < 35.1447053737 , 169.443454843 , 0 > translate < 10.21 , 14.811 , -39.006 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09042056107 > rotate < -28.6014695948 , 70.5998023144 , 0 > translate < 10.21 , 14.811 , -39.006 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.0901353127 > rotate < -37.8564527471 , -97.6782225654 , 0 > translate < 10.21 , 14.811 , -39.006 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.52906441983 > rotate < -18.8093019127 , 117.574174086 , 0 > translate < 10.445 , 13.908 , -40.267 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08968298142 > rotate < -6.53410224971 , -126.700540579 , 0 > translate < 10.445 , 13.908 , -40.267 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09033068378 > rotate < 70.2199949724 , 15.2414995496 , 0 > translate < 10.445 , 13.908 , -40.267 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.53814726213 > rotate < -84.8758211008 , -148.392497754 , 0 > translate < 11.728 , 14.401 , -40.937 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.53491693586 > rotate < 24.3147306644 , 173.225552016 , 0 > translate < 11.728 , 14.401 , -40.937 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.0903114234 > rotate < 15.8070854535 , 53.8401185658 , 0 > translate < 11.728 , 14.401 , -40.937 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08977474737 > rotate < -19.9594462114 , 117.690805887 , 0 > translate < 11.656 , 15.933 , -41.054 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08961782291 > rotate < -14.8359940647 , -124.791483561 , 0 > translate < 11.656 , 15.933 , -41.054 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08979539364 > rotate < -23.6402517801 , -5.34511121255 , 0 > translate < 11.656 , 15.933 , -41.054 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.53672508927 > rotate < -12.6295645922 , 110.779646096 , 0 > translate < 11.893 , 13.769 , -42.326 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09033068378 > rotate < -22.0314550437 , -131.068603369 , 0 > translate < 11.893 , 13.769 , -42.326 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09025730908 > rotate < 82.1344168022 , -52.8996916143 , 0 > translate < 11.893 , 13.769 , -42.326 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.53631181731 > rotate < 24.4959782795 , 172.478479396 , 0 > translate < 13.295 , 14.105 , -42.858 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09010641682 > rotate < 21.1882031834 , 47.5518836137 , 0 > translate < 13.295 , 14.105 , -42.858 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08990320671 > rotate < -83.0958833852 , 125.995220875 , 0 > translate < 13.295 , 14.105 , -42.858 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.53926475955 > rotate < -7.84124693216 , 108.743882011 , 0 > translate < 13.478 , 13.468 , -44.244 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08989449031 > rotate < -24.7908207068 , -136.187519426 , 0 > translate < 13.478 , 13.468 , -44.244 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09046641397 > rotate < 78.8818132556 , -75.9637565321 , 0 > translate < 13.478 , 13.468 , -44.244 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.54110122964 > rotate < -76.7376822284 , 84.6441749571 , 0 > translate < 14.922 , 13.678 , -44.734 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.54251936779 > rotate < 20.8493032661 , 174.147027915 , 0 > translate < 14.922 , 13.678 , -44.734 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09029766578 > rotate < 29.8091114636 , 47.0989059332 , 0 > translate < 14.922 , 13.678 , -44.734 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08973115951 > rotate < -7.43432749427 , 108.787069461 , 0 > translate < 15.274 , 15.178 , -44.701 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08995137506 > rotate < -30.183744574 , -133.537812626 , 0 > translate < 15.274 , 15.178 , -44.701 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08986696436 > rotate < -20.0136257592 , -5.0981846511 , 0 > translate < 15.274 , 15.178 , -44.701 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.54190304494 > rotate < -0.780365908509 , 106.272096734 , 0 > translate < 15.069 , 13.129 , -46.168 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.0903623251 > rotate < -33.8908678843 , -140.559158749 , 0 > translate < 15.069 , 13.129 , -46.168 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09045816059 > rotate < 70.3570904813 , -97.9974734718 , 0 > translate < 15.069 , 13.129 , -46.168 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.54292773648 > rotate < 20.7640791582 , 174.47118471 , 0 > translate < 16.549 , 13.15 , -46.6 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09037883325 > rotate < 34.5894802926 , 41.4321117622 , 0 > translate < 16.549 , 13.15 , -46.6 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08989586659 > rotate < -69.3690310458 , 85.8187131762 , 0 > translate < 16.549 , 13.15 , -46.6 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.54333923685 > rotate < 2.00513513766 , 105.26163477 , 0 > translate < 16.688 , 12.603 , -48.036 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08984402554 > rotate < -36.2216285759 , -142.765166018 , 0 > translate < 16.688 , 12.603 , -48.036 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08978667637 > rotate < 67.2496829738 , -98.322369349 , 0 > translate < 16.688 , 12.603 , -48.036 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.54077577863 > rotate < 15.7797962659 , 174.970002566 , 0 > translate < 18.176 , 12.549 , -48.442 > }
    object { StickObject pigment { Carbon_Color } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.54024446112 > rotate < 40.4360058591 , 40.1559996249 , 0 > translate < 18.176 , 12.549 , -48.442 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08994541148 > rotate < -64.0436298081 , 74.1865837781 , 0 > translate < 18.176 , 12.549 , -48.442 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.0903531538 > rotate < 1.83949284533 , 104.724830856 , 0 > translate < 18.306 , 12.13 , -49.919 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08975731243 > rotate < 64.6716485138 , -107.346693123 , 0 > translate < 18.306 , 12.13 , -49.919 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09012155286 > rotate < -41.6168513754 , -140.626502008 , 0 > translate < 18.306 , 12.13 , -49.919 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.08986145909 > rotate < 1.62993997259 , 105.656427076 , 0 > translate < 18.932 , 11.55 , -47.546 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09026693979 > rotate < -16.793313205 , -4.17556912442 , 0 > translate < 18.932 , 11.55 , -47.546 > }
    object { StickObject pigment { gradient z color_map { [ 0 color Carbon_Color ] [ 1 color Hydrogen_Color ] } } scale z + (x+y) * StickRadius scale < 1 , 1 , 1.09005917271 > rotate < 65.5112032239 , -104.2207703 , 0 > translate < 18.932 , 11.55 , -47.546 > }
    translate <-8.9236497879 , -10.6597642899 , 35.9894638062 >
    scale 0.025 
    rotate y*-30
    translate <-0.7,-0.2,-0.15>
  }

  rotate <-90,asin(vnormalize(<6,0,8>).z)/pi*180-90,0>
  translate <-3,0,0>+<6,0,-8>*0.9
}
//--------------------------------------------------------------------------------------End Of File