import java.io.*;
import java.awt.*;
import javax.swing.*;
import java.util.*;
import java.awt.event.*;

public class FractalTree extends JFrame implements MouseListener
{
    private static class Point3D
    {
        private static final Point3D ORIGIN = new Point3D(0,0,0);

        public double x, y, z;

        public Point3D(double x, double y, double z)
        {
            this.x = x;
            this.y = y;
            this.z = z;
        }
    }

    private static class Matrix
    {
        private double a0, a1, a2, a3;
        private double b0, b1, b2, b3;
        private double c0, c1, c2, c3;
        private double d0, d1, d2, d3;

        public Matrix()
        {
            a0 = b1 = c2 = d3 = 1;
        }

        public Matrix(double pa0, double pa1, double pa2, double pa3,
                      double pb0, double pb1, double pb2, double pb3,
                      double pc0, double pc1, double pc2, double pc3,
                      double pd0, double pd1, double pd2, double pd3)
        {
            a0 = pa0; a1 = pa1; a2 = pa2; a3 = pa3;
            b0 = pb0; b1 = pb1; b2 = pb2; b3 = pb3;
            c0 = pc0; c1 = pc1; c2 = pc2; c3 = pc3;
            d0 = pd0; d1 = pd1; d2 = pd2; d3 = pd3;
        }

        public static Matrix xRotate(double angle)
        {
            double c = Math.cos(angle);
            double s = Math.sin(angle);

            return new Matrix(1, 0,  0, 0,
                              0, c, -s, 0,
                              0, s,  c, 0,
                              0, 0,  0, 1);
        }

        public static Matrix yRotate(double angle)
        {
            double c = Math.cos(angle);
            double s = Math.sin(angle);

            return new Matrix( c, 0, s, 0,
                               0, 1, 0, 0,
                              -s, 0, c, 0,
                               0, 0, 0, 1);
        }

        public static Matrix zRotate(double angle)
        {
            double c = Math.cos(angle);
            double s = Math.sin(angle);

            return new Matrix(c, -s,  0, 0,
                              s,  c,  0, 0,
                              0,  0,  1, 0,
                              0,  0,  0, 1);
        }

        public static Matrix scale(double x, double y, double z)
        {
            return new Matrix(x, 0, 0, 0,
                              0, y, 0, 0,
                              0, 0, z, 0,
                              0, 0, 0, 1);
        }

        public Matrix multiply(Matrix m)
        {
            return new Matrix(a0*m.a0 + a1*m.b0 + a2*m.c0 + a3*m.d0,
                              a0*m.a1 + a1*m.b1 + a2*m.c1 + a3*m.d1,
                              a0*m.a2 + a1*m.b2 + a2*m.c2 + a3*m.d2,
                              a0*m.a3 + a1*m.b3 + a2*m.c3 + a3*m.d3,

                              b0*m.a0 + b1*m.b0 + b2*m.c0 + b3*m.d0,
                              b0*m.a1 + b1*m.b1 + b2*m.c1 + b3*m.d1,
                              b0*m.a2 + b1*m.b2 + b2*m.c2 + b3*m.d2,
                              b0*m.a3 + b1*m.b3 + b2*m.c3 + b3*m.d3,

                              c0*m.a0 + c1*m.b0 + c2*m.c0 + c3*m.d0,
                              c0*m.a1 + c1*m.b1 + c2*m.c1 + c3*m.d1,
                              c0*m.a2 + c1*m.b2 + c2*m.c2 + c3*m.d2,
                              c0*m.a3 + c1*m.b3 + c2*m.c3 + c3*m.d3,

                              d0*m.a0 + d1*m.b0 + d2*m.c0 + d3*m.d0,
                              d0*m.a1 + d1*m.b1 + d2*m.c1 + d3*m.d1,
                              d0*m.a2 + d1*m.b2 + d2*m.c2 + d3*m.d2,
                              d0*m.a3 + d1*m.b3 + d2*m.c3 + d3*m.d3);
        }
        
        public static Matrix translate(double x, double y, double z)
        {
            return new Matrix(1, 0, 0, x,
                              0, 1, 0, y,
                              0, 0, 1, z,
                              0, 0, 0, 1);
        }

        public Point3D transform(Point3D p)
        {
            /* WARNING: ignoring last row of matrix for now. */
            return new Point3D(a0*p.x + a1*p.y + a2*p.z + a3,
                               b0*p.x + b1*p.y + b2*p.z + b3,
                               c0*p.x + c1*p.y + c2*p.z + c3);
        }

        public void writePOV(PrintStream out)
        {
            out.println("  matrix <" + a0 + ", " + b0 + ", " + c0 + ",");
            out.println("          " + a1 + ", " + b1 + ", " + c1 + ",");
            out.println("          " + a2 + ", " + b2 + ", " + c2 + ",");
            out.println("          " + a3 + ", " + b3 + ", " + c3 + ">");
            
        }
    }

    private double minScale;
    private double maxScale;
    private double maxRot;
    private double leafSize;

    private double splitProb;
    private double splitMinScale;
    private double splitMaxScale;
    private double splitMaxRot;

    private double scale;
    private double yScale;

    private TreeNode root;
    private Vector<TreeNode> startNodes = new Vector<TreeNode>();
    private int count;
    private Random rand;
    private long seed;

    private class TreeNode
    {
        private int id;
        private boolean isStartNode;
        private TreeNode parent;
        private Matrix matrix;
        private double size;
        private Vector<TreeNode> children;

        public TreeNode()
        {
            id = count++;

            isStartNode = true;
            startNodes.add(this);
            matrix = Matrix.scale(1, yScale, 1);
            size = 1;

            setChildren();
        }

        public TreeNode(TreeNode parent, Matrix matrix, double size, boolean isStartNode)
        {
            id = count++;

            this.isStartNode = isStartNode;
            this.parent = parent;
            if (isStartNode)
                startNodes.add(this);

            this.matrix = matrix;
            this.size = size;
            
            setChildren();
        }

        private void setChildren()
        {
            children = new Vector<TreeNode>();

            if (size < leafSize)
                return;

            if (rand.nextDouble() < splitProb)
            {
                double yRot1 = rand.nextDouble() * 2 * Math.PI;
                double yRot2 = yRot1 + Math.PI;
                double zRot1 = rand.nextDouble() * splitMaxRot;
                double zRot2 = rand.nextDouble() * splitMaxRot;
                //double zRot1 = (rand.nextDouble()*2-1) * splitMaxRot;
                //double zRot2 = (rand.nextDouble()*2-1) * splitMaxRot;
                double size1 = (rand.nextDouble() * (splitMaxScale - splitMinScale)
                    + splitMinScale) * size;
                double size2 = (rand.nextDouble() * (splitMaxScale - splitMinScale)
                    + splitMinScale) * size;
                
                addChild(yRot1, zRot1, size1, false);
                addChild(yRot2, zRot2, size2, true);
            }
            else
            {
                double yRot = rand.nextDouble() * 2 * Math.PI;
                double zRot = rand.nextDouble() * maxRot;
                //double zRot = (rand.nextDouble()*2-1) * maxRot;
                double newSize = (rand.nextDouble() * (maxScale - minScale)
                    + minScale) * size;
                
                addChild(yRot, zRot, newSize, false);
            }
        }

        private void addChild(double yRot, double zRot, double newSize, boolean start)
        {
            //yRot = 0;
            Matrix trans =
                matrix.multiply(Matrix.yRotate(yRot))
                .multiply(Matrix.zRotate(zRot))
                .multiply(Matrix.translate(0, newSize, 0));
            
            children.add(new TreeNode(this, trans, newSize, start));
        }

        public void paint(Graphics g)
        {
            Point3D p = matrix.transform(Point3D.ORIGIN);
            int x1 = (int)(p.x*scale + 200);
            int y1 = (int)(400 - p.y*scale);
            if (children.isEmpty())
            {
                g.setColor(Color.GREEN);
                g.drawOval(x1-1, y1-1, 3, 3);
            }
            else
            {
                for (TreeNode node : children)
                {
                    g.setColor(Color.RED);
                    Point3D np = node.matrix.transform(Point3D.ORIGIN);
                    int x2 = (int)(np.x*scale + 200);
                    int y2 = (int)(400 - np.y*scale);
                    g.drawLine(x1, y1, x2, y2);

                    node.paint(g);
                }
            }
        }

        public void writePOV(PrintStream out)
        {
            int count = countChildren();
            if (count == 1)
                return;

            if (parent != null)
                count++;
            
            out.println("sphere_sweep");
            out.println("{");
            out.println("  cubic_spline,");
            out.println("  " + (count + 1));
            
            if (parent == null)
                out.println("  <0, -1, 0>, BranchRadius(1.0)");
            else
            {
                if (parent.parent == null)
                    out.println("  <0, -1, 0>, BranchRadius(1.0)");
                else
                    parent.parent.writePoint(out);

                parent.writePoint(out);
            }

            writePoints(out);
            
            out.println("}");
            out.println();
        }

        private int countChildren()
        {
            int max = 0;
            for (TreeNode child : children)
            {
                if (!child.isStartNode)
                {
                    int count = child.countChildren();
                    if (count > max)
                        max = count;
                }
            }

            return max+1;
        }

        private void writePoint(PrintStream out)
        {
            Point3D p = matrix.transform(Point3D.ORIGIN);
            out.println("  <" + p.x + ", " + p.y + ", " + p.z + ">, BranchRadius(" + size + ")");
        }

        private void writePoints(PrintStream out)
        {
            writePoint(out);

            for (TreeNode child : children)
                if (!child.isStartNode)
                    child.writePoints(out);
        }

        public void writeLeaves(PrintStream out)
        {
            if (children.isEmpty())
            {
                out.println("object");
                out.println("{");
                out.println("  BonsaiLeaf(" + size + ")");
                matrix.writePOV(out);
                out.println("}");
                out.println();
            }
            else
            {
                for (TreeNode child : children)
                    child.writeLeaves(out);
            }
        }
    }

    public FractalTree(String[] args)
    {
        minScale      = Double.parseDouble(args[0]);
        maxScale      = Double.parseDouble(args[1]);
        maxRot        = Double.parseDouble(args[2]);
        leafSize      = Double.parseDouble(args[3]);
        splitProb     = Double.parseDouble(args[4]);
        splitMinScale = Double.parseDouble(args[5]);
        splitMaxScale = Double.parseDouble(args[6]);
        splitMaxRot   = Double.parseDouble(args[7]);
        yScale        = Double.parseDouble(args[8]);
        scale         = Double.parseDouble(args[9]);
        
        createTree();

        setSize(450, 450);
        setDefaultCloseOperation(EXIT_ON_CLOSE);
        setContentPane(new JPanel()
            {
                public void paint(Graphics g)
                {
                    g.setColor(Color.BLACK);
                    g.fillRect(0, 0, getWidth(), getHeight());
                    root.paint(g);
                }
            });
        addMouseListener(this);
        setVisible(true);
    }

    private void createTree()
    {
        count = 0;
        startNodes.clear();
        seed = System.currentTimeMillis();
        rand = new Random(seed);
        root = new TreeNode();

        repaint();

        String filename = "bonsai_" + seed + ".pov";
        System.out.println(filename);

        try
        {
            PrintStream out = new PrintStream(new FileOutputStream(filename));

            out.println("#macro BranchRadius(rawSize)");
            out.println("  sqrt(rawSize) * 0.2");
            out.println("#end");
            out.println();
            out.println("#macro BonsaiLeaf(rawSize)");
            out.println("  sphere{<0,20,0>,20 scale<0.2,1,0.5>*rawSize pigment{rgb <0,0.7,0>}}");
            out.println("#end");
            out.println("union {");

            for (TreeNode node : startNodes)
                node.writePOV(out);

            out.println("pigment{rgb <0.7, 0.5, 0.3>} finish{diffuse 0.8 ambient 0.3}");
            out.println("}");
            out.println();

            root.writeLeaves(out);

            out.println("light_source{<100,100,-300> rgb 1}");
            out.println("camera{location <0,0,-30> look_at y*3 angle 18}");
            
            out.close();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
    }

    public void mouseClicked(MouseEvent e)
    {
        createTree();
    }

    public void mouseEntered(MouseEvent e) {}
    public void mouseExited(MouseEvent e) {}
    public void mousePressed(MouseEvent e) {}
    public void mouseReleased(MouseEvent e) {}

    public static void main(String[] args)
    {
        FractalTree ft = new FractalTree(args);
    }
}

