// PERSISTENCE OF VISION RAYTRACER SCENE DESCRIPTION FILE
// file: scrabble.pov
// auth: AJ Ashton
// date: Aug 07 2004 (begun)
//       Oct 31 2004 (last updated)
// desc: A Scrabble game, based on my "Deluxe Turnable" Scrabble set.
//       The setup of the letters is completely customizable.
//       To add a letter on the board add 'object { letterX }' under
//       the '// LETTERS...' comment at the end of the Scrabble Board
//       union. Translating a letter 1 unit in the x or z direction
//       moves the letter one square on the board. For added realism
//       rotate all or some of the letters by a few degrees around the 
//       y-axis (but remember to rotate _before_ you translate!! ;)

/* NOTES ==============================================================

SEPT 13 2004 TO-DO LIST:
[?] -fix maroon plastic texture (too purple.. go darker, less saturated)
[?] -fix boardtop texture (too dark, maybe add more contrast)
[X] -add more letter pieces to the board
[X] -add letter trays
[ ] -add the letter bag(?)
[X] -add the rotating base(?)
[X] -texture the table (plane)
[ ] -add a background

OCT 02 2004:
-changed maroon plastic colour; unsure if it's better (must check on real monitor)
-added new letter arrangement based on an actual game (#declare game = 2;)
-started modeling the letter trays

OCT 17 2004:
-letter trays complete except for texture; added them to the scene
-added a tabletop & gave it a wood texture

TO-DO (in addition to the above list)
[ ] -add letters to trays
[ ] -fix wood texture on table
[ ] -add chairs(?)
[ ] -add scorepad(?)
[ ] -texture label on letter trays
[ ] -add logo to game board(?)

*/


// it probably works with version 3.5 as well, though I haven't tested it
#version 3.6;


// settings ===========================================================
// cameraAngle 1 is a medium-wide shot capturing the entire board from a slight angle.
// cameraAngle 2 is a fisheye top-down shot capturing the entire board.
// cameraAngle 3 is an extreme close-up on the centre square of the board.
// cameraAngle 4 is looking down the centre word (made fore #declare game = 3;)
// cameraAngle 5 is a wide angled top-down shot of the scene
#declare cameraAngle 	= 1;	// [1-5]
#declare lights	 	= on;	// [bool]
#declare fillLight 	= on;	// [bool]
#declare areaLights	= on;	// [bool]
#declare areaFillLight 	= off;	// [bool]
#declare focalBlur 	= on;	// [bool]
#declare Radiosity 	= on;	// [bool]
#declare radSetting 	= 2;	// [1-5]
#declare radBrightness 	= .8;	// [int]
// if yes then the letter pieces will be based on simple untextured boxes (good for quick test renders)
// if no then they will be based on superellipsoids (good for most renders... doesn't slow it down too much)
#declare letterHardEdges = no;	// [bool]
// if no the letter pieces won't render (speeds up test renders)
#declare showLetters 	= yes;	// [bool]
// Game 1 features words relating to POV-Ray and raytracing in general.
// Game 2 is a game that actually happened and in which I got my all-time high score in one turn: 104 points.
// Game 3 is simply 'Linux'
#declare game 		= 2;	// [1-3]
#declare table	 	= on;	// [bool]
// font to use for the letter pieces & the words on the double word spaces, etc.
// cyrvetic comes with POV-Ray, so you should have it
#declare letterFont 	= "cyrvetic.ttf";	// [str]
// Luxi Sans Bold (luxisb.ttf) generally found on Linux computers, so if you're on
// Windows or whatever you might not have it. Either get it or replace it with one you have.
#declare numberFont 	= "luxisb.ttf";	// [str]


// colours ============================================================
#declare White 		= rgb <1, 1, 1>;
#declare Black 		= rgb <0, 0, 0>;
#declare LightGrey 	= rgb <.75, .75, .75>;
#declare Grey 		= rgb <.5, .5, .5>;
#declare Maroon 	= rgb <.35, .05, .2>;
#declare Yellow 	= rgb <1, .95, .5>;


// camera =============================================================
camera { 
	#if (cameraAngle = 1)
		location <5, 15, -30> 
		look_at <-2, 1.75, -5> 
		//angle 45
		angle 50
	#end
	#if (cameraAngle = 2)	// straight down
		//orthographic
		//fisheye
		location <0, 60, 0>
		look_at <0, 1.75, 0>
		//angle 180
	#end
	#if (cameraAngle = 3) // close-up on origin
		location <1, 2.5, -4>
		look_at <0, 1.75, 0>
		angle 35
	#end
	#if (cameraAngle = 4) // looking down first word 
		location <-.5, 3, -2>
		look_at <0, 1.75, 1.5>
	#end
	#if (cameraAngle = 5) // wide top-down
		location <0, 0, 30>
		look_at <0, 0, 0>
	#end
	#if (focalBlur)
		focal_point <0, 2, -5>
		aperture .5
		variance .0001
		blur_samples 60
	#end
}


// lights =============================================================
#if (lights)
	light_source { 
		<-15, 30, -15> 
		colour White
		fade_distance 35
		fade_power 2
		// shadowless
		#if (areaLights)
			area_light <10, 0, 0>, <0, 0, 10>, 5, 5
			adaptive 3
			jitter
		#end
	
	}
	#if (fillLight)
		light_source {
			<15, 20, 5>
			colour Grey
			fade_distance 25
			fade_power 1
			#if (areaFillLight)
				area_light <20, 0, 0>, <0, 0, 20,> 5, 5
				adaptive 1
				jitter
			#else
				shadowless
			#end
		}
	#end
#end


// global settings ====================================================
global_settings
{
	max_trace_level 2
	#if (Radiosity)
		radiosity {
			#if (radSetting = 1)
				pretrace_start 0.08
				pretrace_end   0.02
				count 50
				error_bound 0.5
				recursion_limit 1
			#end
			#if (radSetting = 2)
				pretrace_start 0.08
				pretrace_end   0.01
				count 120
				error_bound 0.25
				recursion_limit 1
			#end
			#if (radSetting = 3)
				pretrace_start 0.08
				pretrace_end   0.005
				count 400
				error_bound 0.1
				recursion_limit 1
			#end
			#if (radSetting = 4)
				pretrace_start 0.08
				pretrace_end   0.005
				count 350
				error_bound 0.15
				recursion_limit 2
			#end
			#if (radSetting = 5)
				pretrace_start 0.08
				pretrace_end 0.04
				count 35
				error_bound 1.8
				recursion_limit 1
			#end
			nearest_count 5
			low_error_factor .5
			gray_threshold 0
			minimum_reuse 0.015
			brightness radBrightness
			adc_bailout 0.01/2
		}
	#end
}


// textures ===========================================================
#declare Wood1 = material {
	texture {  
		pigment { 
			bozo
			turbulence 0.04
			lambda 2.5
			omega 0.1
			octaves 7
			scale <0.5, 0.05, 0.05>
			color_map { 
				[0.000, 0.256 color rgb <0.20, 0.11, 0.07>*1.25 color rgb <0.23, 0.12, 0.09>*1.25]
				[0.256, 0.393 color rgb <0.23, 0.12, 0.09>*1.25 color rgb <0.24, 0.13, 0.09>*1.25]
				[0.393, 0.581 color rgb <0.24, 0.13, 0.09>*1.25 color rgb <0.20, 0.11, 0.07>*1.25]
				[0.581, 0.726 color rgb <0.20, 0.11, 0.07>*1.25 color rgb <0.25, 0.12, 0.10>*1.25]
				[0.726, 0.983 color rgb <0.25, 0.12, 0.10>*1.25 color rgb <0.23, 0.12, 0.08>*1.25]
				[0.983, 1.000 color rgb <0.23, 0.12, 0.08>*1.25 color rgb <0.20, 0.11, 0.07>*1.25]
			}
		}
	} 
	texture {  
		pigment {
			wood
			turbulence <0.1, 0.04, 1>
			scale <0.15, 0.5, 1>
			rotate x*2
			color_map {
				[0.000, 0.139 color rgbt <0.545, 0.349, 0.247, 1.000> color rgbt <0.450, 0.350, 0.320, 0.800>]
				[0.139, 0.148 color rgbt <0.450, 0.350, 0.320, 0.800> color rgbt <0.450, 0.350, 0.320, 0.800>]
				[0.148, 0.287 color rgbt <0.450, 0.350, 0.320, 0.800> color rgbt <0.545, 0.349, 0.247, 1.000>]
				[0.287, 0.443 color rgbt <0.545, 0.349, 0.247, 1.000> color rgbt <0.545, 0.349, 0.247, 1.000>]
				[0.443, 0.626 color rgbt <0.545, 0.349, 0.247, 1.000> color rgbt <0.450, 0.350, 0.320, 0.800>]
				[0.626, 0.635 color rgbt <0.450, 0.350, 0.320, 0.800> color rgbt <0.450, 0.350, 0.320, 0.800>]
				[0.635, 0.843 color rgbt <0.450, 0.350, 0.320, 0.800> color rgbt <0.545, 0.349, 0.247, 1.000>]
				[0.843, 1.000 color rgbt <0.545, 0.349, 0.247, 1.000> color rgbt <0.545, 0.349, 0.247, 1.000>] 
			}
		}
	}
	scale 5
}

#declare Wood2 = material {	// for the letter squares
	texture { 
		pigment { 
			wood
			turbulence 0.04
			octaves 4
			omega 1.25
			scale <0.3, .3, 1>
			color_map { 
				[0.0, 0.3 color rgb <0.35, 0.05, 0.5> color rgb <0.35, 0.10, 0.4>]
				[0.3, 0.6 color rgb <0.35, 0.10, 0.4> color rgb <0.55, 0.15, 0.4>]
				[0.6, 1.0 color rgb <0.55, 0.15, 0.4> color rgb <0.35, 0.05, 0.4>]
			}
		}
	}
	texture { 
		pigment { 
			bozo
			turbulence <0.03, 0.03, 0.1>
			octaves 5
			lambda 3.25
			scale <0.05, .05, 1.0> 
			color_map {
				[0.0, 0.3 color rgbt <0.4, 0.15, 0.22, 0.00> color rgbt <0.4, 0.13, 0.20, 0.20>]
				[0.3, 0.4 color rgbt <0.4, 0.13, 0.18, 0.35> color rgbt <0.6, 0.25, 0.28, 0.40>]
				[0.4, 0.6 color rgbt <0.6, 0.25, 0.28, 0.20> color rgbt <0.4, 0.13, 0.18, 0.35>]
				[0.6, 1.0 color rgbt <0.4, 0.13, 0.20, 0.50> color rgbt <0.4, 0.15, 0.25, 0.00>]
			}
		}			
		rotate 15*y
	}
}


// objects ============================================================
#if (showLetters)	// if not a lot of parse time is saved (400 lines!)
#declare scaleLtrs = .5;
#declare scaleNums = .25;
#declare letterThick = .0325;
#declare numThick = .065;
#declare numXtrans = .2;
#declare numZtrans = -.2;
#if (letterHardEdges)
	#declare = letterBlank = box { <.4, 0, .4>, <-.4, .2, -.4> material { Wood2 } finish { specular .25 } }
#else
	#declare letterBlank = superellipsoid { 
		<.08,.08> 
		scale <.4, .1, .4> 
		translate y*.1 
		material { Wood2 }
		finish { specular .25 }
	}
#end

#declare letterA = difference {	// there are 9 As in Scrabble
	object { letterBlank }
	union {
		text {	
			ttf letterFont "A" .0325, 0 translate <-.3, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text {	
			ttf numberFont " 1" .065, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterB = difference {	// there are 2 Bs in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "B" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 3" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterC = difference {	// there are 2 Cs in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "C" letterThick, 0 translate <-.3, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 3" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterD = difference {	// there are 4 Ds in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "D" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 2" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterE = difference {	// there are 12 Es in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "E" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 1" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterF = difference {	// there are 2 Fs in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "F" letterThick, 0 translate <-.2, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont "4" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterG = difference {	// there are 3 Gs in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "G" letterThick, 0 translate <-.35, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 2" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterH = difference {	// there are 2 Hs in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "H" letterThick, 0 translate <-.3, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 4" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterI = difference {	// there are 9 Is in Scrabble
	object { letterBlank }
	union {
		
		text { 
			ttf letterFont "I" letterThick, 0 translate <-.05, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont "1" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterJ = difference {	// there is 1 J in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "J" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont "8" numThick, 0 translate <-.2, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterK = difference {	// there is 1 K in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "K" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 5" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterL = difference {	// there are 4 Ls in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "L" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont "1" numThick, 0 translate <-.15, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterM = difference {	// there are 2 Ms in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "M" letterThick, 0 translate <-.35, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 3" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterN = difference {	// there are 6 Ns in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "N" letterThick, 0 translate <-.3, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 1" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterO = difference {	// there are 8 Os in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "O" letterThick, 0 translate <-.35, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 1" numThick, 0 translate <-.3, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterP = difference {	// there are 2 Ps in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "P" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont "2" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterQ = difference {	// there is 1 Q in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "Q" letterThick, 0 translate <-.35, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text {
			ttf numberFont "10" .3, <-.1, 0, 0>  translate <-.2, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterR = difference {	// there are 6 Rs in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "R" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 1" numThick, 0 translate <-.35, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterS = difference {	// there are 4 Ss in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "S" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 1" numThick, 0 translate <-.3, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterT = difference {	// there are 6 Ts in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "T" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont "1" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterU = difference {	// there are 4 Us in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "U" letterThick, 0 translate <-.3, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 1" numThick, 0 translate <-.35, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterV = difference {	// there are 2 Vs in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "V" letterThick, 0 translate <-.3, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 4" numThick, 0 translate <-.6, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterW = difference {	// there are 2 Ws in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "W" letterThick, 0 translate <-.45, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 4" numThick, 0 translate <-.3, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterX = difference {	// there is 1 X in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "X" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 8" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterY = difference {	// there are 2 Ys in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "Y" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont " 4" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#declare letterZ = difference {	// there is 1 Z in Scrabble
	object { letterBlank }
	union {
		text { 
			ttf letterFont "Z" letterThick, 0 translate <-.25, -.375, 0> 
			scale scaleLtrs rotate x*90 translate y*.21 
		}
		text { 
			ttf numberFont "10" numThick, 0 translate <-.25, -.275, 0> 
			scale scaleNums rotate x*90 translate <numXtrans, .21, numZtrans> 
		}
		pigment { colour Yellow }
	}
}
#end
// the commented objects below are for use when image maps are not available
/*#declare tripleLetter = union {
	box { <.5, 0.001, .5>, <-.5, -0.1, -.5> pigment { colour rgb <.1, .6, 1> } }
	union {
		text { ttf letterFont "TRIPLE" .1, 0 translate <-1.65, -.375, 0> scale .15 rotate x*90 translate <0, .003, .2> }
		text { ttf letterFont "LETTER" .1, 0 translate <-1.7, -.375, 0> scale .15 rotate x*90 translate <0, .003, 0> }
		text { ttf letterFont "SCORE" .1, 0 translate <-1.6, -.375, 0> scale .15 rotate x*90 translate <0, .003, -.2> }
		pigment { colour Black }
	}
}
#declare doubleLetter = union {
	box { <.5, 0.001, .5>, <-.5, -0.1, -.5> pigment { colour rgb <.7, .8, 1> } }
	union {
		text { ttf letterFont "DOUBLE" .1, 0 translate <-1.9, -.375, 0> scale .15 rotate x*90 translate <0, .003, .2> }
		text { ttf letterFont "LETTER" .1, 0 translate <-1.7, -.375, 0> scale .15 rotate x*90 translate <0, .003, 0> }
		text { ttf letterFont "SCORE" .1, 0 translate <-1.6, -.375, 0> scale .15 rotate x*90 translate <0, .003, -.2> }
		pigment { colour Black }
	}
}
#declare tripleWord = union {
	box { <.5, 0.001, .5>, <-.5, -0.1, -.5> pigment { colour rgb <.9, .3, .2> } }
	union {
		text { ttf letterFont "TRIPLE" .1, 0 translate <-1.65, -.375, 0> scale .15 rotate x*90 translate <0, .003, .2> }
		text { ttf letterFont "WORD" .1, 0 translate <-1.45, -.375, 0> scale .15 rotate x*90 translate <0, .003, 0> }
		text { ttf letterFont "SCORE" .1, 0 translate <-1.6, -.375, 0> scale .15 rotate x*90 translate <0, .003, -.2> }
		pigment { colour Black }
	}
}
#declare doubleWord = union {
	box { <.5, 0.001, .5>, <-.5, -0.1, -.5> pigment { colour rgb <1, .75, .75> } }
	union {
		text { ttf letterFont "DOUBLE" .1, 0 translate <-1.9, -.375, 0> scale .15 rotate x*90 translate <0, .003, .2> }
		text { ttf letterFont "WORD" .1, 0 translate <-1.45, -.375, 0> scale .15 rotate x*90 translate <0, .003, 0> }
		text { ttf letterFont "SCORE" .1, 0 translate <-1.6, -.375, 0> scale .15 rotate x*90 translate <0, .003, -.2> }
		pigment { colour Black }
	}
}*/
#declare tripleLetter = box {
	<.5, .001, .5>, <-.5, -.1, -.5>
	pigment { image_map { png "tripleletter.png" } translate <-.5, -.5, 0> rotate x*90 }
}
#declare doubleLetter = box {
	<.5, .001, .5>, <-.5, -.1, -.5>
	pigment { image_map { png "doubleletter.png" } translate <-.5, -.5, 0> rotate x*90 }
}
#declare tripleWord = box {
	<.5, .001, .5>, <-.5, -.1, -.5>
	pigment { image_map { png "tripleword.png" } translate <-.5, -.5, 0> rotate x*90 }
}
#declare doubleWord = box {
	<.5, .001, .5>, <-.5, -.1, -.5>
	pigment { image_map { png "doubleword.png" } translate <-.5, -.5, 0> rotate x*90 }
}

#declare tray = union {	// plastic tray to hold player's letter pieces
	mesh {
		// back
		triangle { <-4, 1, -1>, <-4, 0, -1>, < 4, 0, -1> }
		triangle { < 4, 0, -1>, < 4, 1, -1>, <-4, 1, -1> }
		// top
		triangle { <-4, 1.1, -.9>, < 4,  1.1, -.9>, < 4, .625, .1> }
		triangle { <-4, 1.1, -.9>, <-4, .625,  .1>, < 4, .625, .1> }
		// inset back
		// 60.255 degrees
		triangle { <-4, .625, .1>, <-4, .1, .4>, < 4, .625, .1> }
		triangle { < 4, .625, .1>, < 4, .1, .4>, <-4, .1, .4> }
		// inset bottom
		triangle { <-4, .1, .4>, <-4, .3, .8>, < 4, .1, .4> }
		triangle { < 4, .1, .4>, < 4, .3, .8>, <-4, .3, .8> }
		// front
		triangle { <-4, .3, .8>, <-4, 0, 1>, <4, .3, .8> }
		triangle { < 4, .3, .8>, <4, 0, 1>, <-4, 0, 1> }
		// inset sides
		triangle { <-3.9, .625, .1>, <-3.9, .1, .4>, <-3.9, .3, .8> }
		triangle { <3.9, .625, .1>, <3.9, .1, .4>, <3.9, .3, .8> }
		// sides
		triangle { <-4.075, .278, .7>, <-4.075, 1, -.9>, <-4.075, 0, -.9> }
		triangle { <4.075, .278, .7>, <4.075, 1, -.9>, <4.075, 0, -.9> }
		triangle { <-4.075, 0, -.9>, <-4.075, .278, .7>, <-4.075, 0, .925> }
		triangle { <4.075, 0, -.9>, <4.075, .278, .7>, <4.075, 0, .925> }
	}
	cylinder { <-4, 1, -.9>, <4, 1, -.9>, .1 }
	cylinder { <-4, .278, .7>, <4, .278, .7>, .1 }
	torus { .1, .05 rotate z*90 translate <-3.95, .278, .7> }
	torus { .1, .05 rotate z*90 translate <3.95, .278, .7> }
	torus { .1, .05 rotate z*90 translate <-3.95, 1, -.9> }
	torus { .1, .05 rotate z*90 translate <3.95, 1, -.9> }
	cylinder { <-3.95, 1.085, -.85>, <-3.95, .35, .775>, .05 } 
	cylinder { <3.95, 1.085, -.85>, <3.95, .35, .775>, .05 } 
	cylinder { <-3.95, 1, -1>, <-3.95, 0, -1>, .05 }
	cylinder { <3.95, 1, -1>, <3.95, 0, -1>, .05 }
	sphere { <-4, 1, -.9>, .075 }
	sphere { <4, 1, -.9>, .075 }
	cylinder { <4, 1, -.9>, <4, .278, .7>, .075 }
	cylinder { <-4, 1, -.9>, <-4, .278, .7>, .075 }
	sphere { <-4, .278, .7>, .075 }
	sphere { <4, .278, .7>, .075 }
	cylinder { <4, 1, -.9>, <4, 0, -.9>, .075 }
	cylinder { <-4, 1, -.9>, <-4, 0, -.9>, .075 }
	cylinder { <-3.95, .3, .8>, <-3.95, 0, 1>, .05 }
	cylinder { <3.95, .3, .8>, <3.95, 0, 1>, .05 }
	sphere { <3.95, 0, 1>, .05 }
	sphere { <-3.95, 0, 1>, .05 }
	cylinder { <-4, .278, .7>, <-4, 0, .925>, .075 }
	cylinder { <4, .278, .7>, <4, 0, .925>, .075 }
	sphere { <4, 0, .925>, .075 }
	sphere { <-4, 0, .925>, .075 }
	// top label border
	union {
		difference {
			union {
				torus { .1, .025 translate <3.5, 0, 0> }
				torus { .1, .025 translate <-3.5, 0, 0> }
				torus { .1, .025 translate <3.5, 0, -.55> }
				torus { .1, .025 translate <-3.5, 0, -.55> }
			}
			union {
				box { <-4, 1, 0>, <4, -1, -.55> }
				box { <-3.5, 1, 1>, <3.5, -1, -1> }
			}
		}
		cylinder { <-3.6, 0, 0>, <-3.6, 0, -.56>, .025 }
		cylinder { <3.6, 0, 0>, <3.6, 0, -.56>, .025 }
		cylinder { <3.5, 0, .1>, <-3.5, 0, .1>, .025 }
		cylinder { <3.5, 0, -.65>, <-3.5, 0, -.65>, .025 }
		rotate x*25
		translate <0, .75, -.12>
	}
	// top label
	/*merge {
		cylinder { <3.5, 0, 0>, <3.5, .01004, 0>, .1 }
		cylinder { <-3.5, 0, 0>, <-3.5, .01003, 0>, .1 }
		cylinder { <3.5, 0, -.55>, <3.5, .01002, -.55>, .1 }
		cylinder { <-3.5, 0, -.55>, <-3.5, .01001, -.55>, .1 }
		box { <-3.6, 0, 0>, <3.6, .01005, -.55> }
		box { <3.5, 0, -.65>, <-3.5, .01006, .1> }
		pigment { colour rgbt <1, 1, 1, .9> }
		finish { reflection 0 specular 0 }
		rotate x*25
		translate <0, .75, -.12>
	}*/
	pigment { colour Maroon }
	finish { specular .25 reflection .01 }
}


// scene ==============================================================
sphere {	// sky/background
	<0, 0, 0>, 1000 hollow on no_shadow 
	pigment { colour White } 
	finish { diffuse 0 ambient 1 }
}

plane { // ground plane 
	<0, 1, 0>, #if (table) -50 #else 0 #end
	pigment { color White }
}

#if (table)
	#declare T_pine = material {	// stolen from "woods.inc" and "woodmaps.inc"
		texture { 
			pigment {
				wood  
				turbulence 0.02
				octaves 4
				lambda 3
				scale 0.175
				rotate <2, 2, 0>
				color_map { 
					[0.0, 0.5 color rgb <1.00, 0.85, 0.50>*.8 color rgb <0.90, 0.70, 0.46>*.8]
					[0.5, 0.7 color rgb <0.90, 0.70, 0.46>*.8 color rgb <0.90, 0.70, 0.46>*.8]
					[0.7, 1.0 color rgb <0.90, 0.70, 0.46>*.8 color rgb <1.00, 0.85, 0.50>*.8]
				}
			}
		} 
    		/*texture { 
			pigment { 
				wood
				turbulence 0.02
				octaves 4
				lambda 2.8
				scale 0.2
				rotate <2, 2, 0>
				translate <0.0175, 0.0175, 0.0175>
				color_map { 
					[0.0, 0.5 color rgbt <1.00, 0.45, 0.10, 0.80>*.8 color rgbt <0.85, 0.65, 0.40, 0.40>*.8]
					[0.5, 0.7 color rgbt <0.85, 0.65, 0.40, 0.40>*.8 color rgbt <0.85, 0.65, 0.40, 0.40>*.8]
					[0.7, 1.0 color rgbt <0.85, 0.65, 0.40, 0.40>*.8 color rgbt <1.00, 0.45, 0.10, 0.80>*.8]
      				}
			}
		}*/
		scale 1
	}              
	#declare Twoodmap = material { texture {
	        pigment {
	                image_map { jpeg "lightwood2.jpg" }
	                scale 50
	        }             
	} }
	union {	// tabletop
		union { 
			box { <-35, 0, 20>, <35, -3, 12> }
			cylinder { <-35, -1.5, 20>, <35, -1.5, 20>, 1.5 }
			cylinder { <-35, -1.5, 20>, <-35, -1.5, 12>, 1.5 }
			cylinder { <35, -1.5, 20>, <35, -1.5, 12>, 1.5 }
			sphere { <35, -1.5, 20>, 1.5 }
			sphere { <-35, -1.5, 20>, 1.5 }
			material { Twoodmap rotate <35, 90, 30> }
		}
		union { 
			box { <-35, 0, 12>, <35, -3, 4> }
			cylinder { <-35, -1.5, 12>, <-35, -1.5, 4>, 1.5 }
			cylinder { <35, -1.5, 12>, <35, -1.5, 4>, 1.5 }
			material { Twoodmap rotate <25, 95, 25> }
		}
		union {
			box { <-35, 0, 4>, <35, -3, -4> }
			cylinder { <-35, -1.5, 4>, <-35, -1.5, -4>, 1.5 }
			cylinder { <35, -1.5, 4>, <35, -1.5, -4>, 1.5 }
			material { Twoodmap rotate <35, 85, 40> }
		}
		union {
			box { <-35, 0, -4>, <35, -3, -12> }
			cylinder { <-35, -1.5, -4>, <-35, -1.5, -12>, 1.5 }
			cylinder { <35, -1.5, -4>, <35, -1.5, -12>, 1.5 }
			material { Twoodmap rotate <28, 92, 43> }
		}
		union {
			box { <-35, 0, -12>, <35, -3, -20> }
			cylinder { <-35, -1.5, -20>, <35, -1.5, -20>, 1.5 }
			cylinder { <-35, -1.5, -12>, <-35, -1.5, -20>, 1.5 }
			cylinder { <35, -1.5, -12>, <35, -1.5, -20>, 1.5 }
			sphere { <35, -1.5, -20>, 1.5 }
			sphere { <-35, -1.5, -20>, 1.5 }
			material { Twoodmap rotate <33, 83, 35> }
		}
		finish { specular 1 reflection .2 }
		translate x*10
	}
#end

union { // the Scrabble board
	union { // the top face of the board
		union {	// the raised bars on the board
			#declare bumpsRad = 0.05;	// radius of the bars
			#declare i = -7.5;
			#while (i < 7.6)
				//cylinder { <i, 0, 7.5>, <i, 0, -7.5>, bumpsRad }
				superellipsoid { <.625, .005> scale <bumpsRad, bumpsRad, 7.5> translate <i, 0, 0> }
				#declare i = i + 1;
			#end	
			#declare i = -7.5;
			#while (i < 7.6)
				//cylinder { <7.5, 0, i>, <-7.5, 0, i>, bumpsRad }
				superellipsoid { <.625, .005> rotate y*90 scale <7.5, bumpsRad, bumpsRad> translate <0, 0, i> }
				#declare i = i + 1;
			#end
			sphere { <7.5, 0, 7.5>, bumpsRad }
			sphere { <7.5, 0, -7.5>, bumpsRad }
			sphere { <-7.5, 0, 7.5>, bumpsRad }
			sphere { <-7.5, 0, -7.5>, bumpsRad }
			pigment { colour rgb <.9, .9, .9> }
			finish { specular .25 reflection .05 }
		}
		box {	// the flat top
			<7.85, 0, 7.85>,
			<-7.85, -.1, -7.85>
			pigment { 
				//color rgb <.9, .85, .7> 
				image_map { png "facetile.png" }
				rotate x*90 scale 2
			}
			finish { specular .25 reflection .1 }
		}
		box {	// the flat top
			<7.85, 0.0005, 7.85>,
			<-7.85, -.1, -7.85>
			pigment { 
				image_map { png "gradoverlay.png" }
				translate <-.5, -.5, 0> rotate x*90 scale 15
			}
			finish { specular .25 reflection .05 }
		}

		// all of the coloured squares:
		object { tripleWord translate <7, 0, 7> }
		object { tripleWord translate <-7, 0, 7> }
		object { tripleWord translate <7, 0, -7> }
		object { tripleWord translate <-7, 0, -7> }
		object { tripleWord translate <7, 0, 0> }
		object { tripleWord translate <-7, 0, 0> }
		object { tripleWord translate <0, 0, 7> }
		object { tripleWord translate <0, 0, -7> }
		object { doubleWord translate <3, 0, 3> }
		object { doubleWord translate <4, 0, 4> }
		object { doubleWord translate <5, 0, 5> }
		object { doubleWord translate <6, 0, 6> }
		object { doubleWord translate <-3, 0, -3> }
		object { doubleWord translate <-4, 0, -4> }
		object { doubleWord translate <-5, 0, -5> }
		object { doubleWord translate <-6, 0, -6> }
		object { doubleWord translate <-3, 0, 3> }
		object { doubleWord translate <-4, 0, 4> }
		object { doubleWord translate <-5, 0, 5> }
		object { doubleWord translate <-6, 0, 6> }
		object { doubleWord translate <3, 0, -3> }
		object { doubleWord translate <4, 0, -4> }
		object { doubleWord translate <5, 0, -5> }
		object { doubleWord translate <6, 0, -6> }
		object { tripleLetter translate <2, 0, 2> }
		object { tripleLetter translate <2, 0, -2> }
		object { tripleLetter translate <-2, 0, 2> }
		object { tripleLetter translate <-2, 0, -2> }
		object { tripleLetter translate <2, 0, 6> }
		object { tripleLetter translate <2, 0, -6> }
		object { tripleLetter translate <-2, 0, 6> }
		object { tripleLetter translate <-2, 0, -6> }
		object { tripleLetter translate <6, 0, 2> }
		object { tripleLetter translate <6, 0, -2> }
		object { tripleLetter translate <-6, 0, 2> }
		object { tripleLetter translate <-6, 0, -2> }
		object { doubleLetter translate <4, 0, 7> }
		object { doubleLetter translate <-4, 0, 7> }
		object { doubleLetter translate <4, 0, -7> }
		object { doubleLetter translate <-4, 0, -7> }
		object { doubleLetter translate <7, 0, 4> }
		object { doubleLetter translate <7, 0, -4> }
		object { doubleLetter translate <-7, 0, 4> }
		object { doubleLetter translate <-7, 0, -4> }
		object { doubleLetter translate <1, 0, 5> }
		object { doubleLetter translate <1, 0, -5> }
		object { doubleLetter translate <-1, 0, 5> }
		object { doubleLetter translate <-1, 0, -5> }
		object { doubleLetter translate <0, 0, 4> }
		object { doubleLetter translate <0, 0, -4> }
		object { doubleLetter translate <4, 0, 0> }
		object { doubleLetter translate <-4, 0, 0> }
		object { doubleLetter translate <1, 0, 1> }
		object { doubleLetter translate <1, 0, -1> }
		object { doubleLetter translate <-1, 0, 1> }
		object { doubleLetter translate <-1, 0, -1> }
		object { doubleLetter translate <5, 0, 1> }
		object { doubleLetter translate <5, 0, -1> }
		object { doubleLetter translate <-5, 0, 1> }
		object { doubleLetter translate <-5, 0, -1> }
		box { <.5, 0.001, .5>, <-.5, -0.1, -.5> pigment { colour rgb <1, .75, .75> } }	// centre square
		text {	// the board doesn't need to be *that* accuate ;D
			ttf "povlogo.ttf" "P" .1, 0 
			translate <-.25, 0, 0> rotate x*90 translate <0, .003, -.5> scale .75 
			pigment { colour Black } 
		}
		finish { specular .75 reflection .1 }
	}

	// BASE - the maroon plastic base to the board
	#declare oneSide = union {
		difference { // the little top corners
			union {
				torus { .1, .05 translate <7.8, .05, 7.8> }
				cone { <7.8, -.002, 7.8>, .2 <7.8, .05, 7.8>, .15 }
				cylinder { <7.8, 0, 7.8>, <7.8, .1, 7.8>, .1 }
			}
			box { <7.8, 1, 7.8>, <7.5, -1, 7.5> }
		}
		cylinder { <7.9, .05, 7.8>, <7.9, .05, -7.8>, .05 } // horizontal soft edge
		box { <7.9, .1, 7.8>, <7.7, -.1, -7.8> } // top edge
		union {
		mesh {	// the little top flat sides
			triangle { <-7.8, 0, 8>, <-7.8, .05, 7.95>, <7.8, 0, 8> }
			triangle { <-7.8, .05, 7.95>, <7.8, 0, 8>, <7.8, .05, 7.95> }
		}
			// the big cone corners
			cone { <7.8, -.001, 7.8>, .2 <7.8, -.9, 7.8> 1.5 }
			mesh {	// the big, flat sides
				triangle { <7.8, 0, 8>, <7.8, -.9, 9.3>, <-7.8, 0, 8> }
				triangle { <-7.8, -.9, 9.3>, <-7.8, 0, 8>, <7.8, -.9, 9.3> }
			}
			pigment { colour Maroon }
			normal { bumps .25 scale .01 }
			finish { specular .5 reflection .05 }
		}
		cylinder { <7.8, -.9, 7.8>, <7.8, -1, 7.8>, 1.5 }
		box { <7.8, -.9, 9.25>, <-7.8, -1, 9.3> }
		cylinder { <7.8, -.975, 7.8>, <7.8, -1.3, 7.8>, 1.55 }
		box { <7.8, -.975, 9.3>, <-7.8, -1.3, 9.35> }
		pigment { colour Maroon }
		finish { specular .65 reflection .1 }
	}
	object { oneSide }
	object { oneSide rotate y*90 }
	object { oneSide rotate y*180 }
	object { oneSide rotate y*-90 }
	union {	// turn base
		cylinder { <0, -2.25, 0>, <0, -.25, 0>, 6 }
		torus { 5.75, .25 translate y*-2.25 }
		pigment { colour Maroon }
		finish { specular .25 reflection .05 }
	}
	
	// LETTERS - letter pieces on the board (that will rotate & translate with it)
	#if (showLetters)
		#if (game = 1)
			// 'POVRAY'
			object { letterP rotate y*2 translate x*-1 }
			object { letterO rotate y*-4 }
			object { letterV rotate y*1 translate x*1 }
			object { letterR rotate y*7 translate x*2 }
			object { letterA translate x*3 }
			object { letterY rotate y* -2 translate x*4 }
			// 'GRAPHICS' ('P' alreadt in 'povray')
			object { letterG translate <-1, 0, 3> }
			object { letterR translate <-1, 0, 2> }
			object { letterA translate <-1, 0, 1> }
			object { letterH translate <-1, 0, -1> }
			object { letterI translate <-1, 0, -2> }
			object { letterC translate <-1, 0, -3> }
			object { letterS translate <-1, 0, -4> }
			// 'RENDER' (last 'R' already in 'graphics')
			object { letterR translate <-6, 0, 2> }
			object { letterE translate <-5, 0, 2> }
			object { letterN translate <-4, 0, 2> }
			object { letterD translate <-3, 0, 2> }
			object { letterE translate <-2, 0, 2> }
			// 'PLANE' ('N' already in 'render')
			object { letterP translate <-4, 0, 5> }
			object { letterL translate <-4, 0, 4> }
			object { letterA translate <-4, 0, 3> }
			object { letterE translate <-4, 0, 1> }
			// 'SPHERE' ('P' already in 'plane')
			object { letterS translate <-5, 0, 5> }
			object { letterH translate <-3, 0, 5> }
			object { letterE translate <-2, 0, 5> }
			object { letterR translate <-1, 0, 5> }
			object { letterE translate <0, 0, 5> }
			// 'TORUS' ('S' already in 'graphics')
			object { letterT translate <-5, 0, -4> }
			object { letterO translate <-4, 0, -4> }
			object { letterR translate <-3, 0, -4> }
			object { letterU translate <-2, 0, -4> }
		#end
		#if (game = 2)
			// a game that actually happened
			object { letterL translate <-1, 0,  7> }
			object { letterR translate < 2, 0,  7> }
			object { letterP translate <-2, 0,  6> }
			object { letterA translate <-1, 0,  6> }
			object { letterT translate < 1, 0,  6> }
			object { letterE translate < 2, 0,  6> }
			object { letterN translate < 3, 0,  6> }
			object { letterN translate < 4, 0,  6> }
			object { letterI translate < 5, 0,  6> }
			object { letterS translate < 6, 0,  6> }
			object { letterA translate <-7, 0,  5> }
			object { letterH translate <-6, 0,  5> }
			object { letterP translate <-1, 0,  5> }
			object { letterI translate < 0, 0,  5> }
			object { letterE translate < 1, 0,  5> }
			object { letterA translate < 6, 0,  5> }
			object { letterL translate <-7, 0,  4> }
			object { letterO translate <-6, 0,  4> }
			object { letterO translate <-5, 0,  4> }
			object { letterM translate <-4, 0,  4> }
			object { letterJ translate <-2, 0,  4> }
			object { letterA translate < 1, 0,  4> }
			object { letterN translate < 2, 0,  4> }
			object { letterF translate < 4, 0,  4> }
			object { letterR translate < 5, 0,  4> }
			object { letterI translate < 6, 0,  4> }
			object { letterT translate < 7, 0,  4> }
			object { letterG translate <-6, 0,  3> }
			object { letterO translate <-4, 0,  3> }
			object { letterE translate <-2, 0,  3> }
			object { letterS translate <-1, 0,  3> }
			object { letterC translate < 0, 0,  3> }
			object { letterR translate < 1, 0,  3> }
			object { letterO translate < 2, 0,  3> }
			object { letterW translate < 3, 0,  3> }
			object { letterL translate < 6, 0,  3> }
			object { letterZ translate <-4, 0,  2> }
			object { letterI translate <-3, 0,  2> }
			object { letterT translate <-2, 0,  2> }
			object { letterS translate < 2, 0,  2> }
			object { letterH translate < 7, 0,  2> }
			object { letterA translate <-6, 0,  1> }
			object { letterN translate <-5, 0,  1> }
			object { letterBlank  translate <-4, 0,  1> }
			object { letterU translate < 0, 0,  1> }
			object { letterE translate < 2, 0,  1> }
			object { letterV translate < 3, 0,  1> }
			object { letterE translate < 4, 0,  1> }
			object { letterR translate < 5, 0,  1> }
			object { letterO translate < 7, 0,  1> }
			object { letterR translate <-3, 0,  0> }
			object { letterG translate < 0, 0,  0> }
			object { letterU translate < 1, 0,  0> }
			object { letterY translate < 2, 0,  0> }
			object { letterT translate < 4, 0,  0> }
			object { letterE translate < 5, 0,  0> }
			object { letterA translate < 6, 0,  0> }
			object { letterM translate < 7, 0,  0> }
			object { letterB translate <-4, 0, -1> }
			object { letterE translate <-3, 0, -1> }
			object { letterT translate <-2, 0, -1> }
			object { letterL translate < 0, 0, -1> }
			object { letterQ translate < 5, 0, -1> }
			object { letterE translate < 7, 0, -1> }
			object { letterD translate <-3, 0, -2> }
			object { letterO translate <-2, 0, -2> }
			object { letterR translate <-1, 0, -2> }
			object { letterY translate < 0, 0, -2> }
			object { letterU translate < 5, 0, -2> }
			object { letterO translate <-3, 0, -3> }
			object { letterF translate < 4, 0, -3> }
			object { letterI translate < 5, 0, -3> }
			object { letterV translate < 6, 0, -3> }
			object { letterE translate < 7, 0, -3> }
			object { letterD translate <-7, 0, -4> }
			object { letterS translate <-5, 0, -4> }
			object { letterI translate <-4, 0, -4> }
			object { letterN translate <-3, 0, -4> }
			object { letterG translate <-2, 0, -4> }
			object { letterK translate < 2, 0, -4> }
			object { letterBlank  translate < 5, 0, -4> }
			object { letterA translate <-7, 0, -5> }
			object { letterO translate <-5, 0, -5> }
			object { letterE translate <-3, 0, -5> }
			object { letterI translate < 2, 0, -5> }
			object { letterW translate < 4, 0, -5> }
			object { letterE translate < 5, 0, -5> }
			object { letterB translate < 6, 0, -5> }
			object { letterT translate <-7, 0, -6> }
			object { letterA translate <-6, 0, -6> }
			object { letterX translate <-5, 0, -6> }
			object { letterD translate < 2, 0, -6> }
			object { letterI translate < 3, 0, -6> }
			object { letterE translate < 4, 0, -6> }
			object { letterD translate < 5, 0, -6> }
			object { letterA translate <-7, 0, -7> }
			object { letterN translate < 3, 0, -7> }
		#end
		#if (game = 3)
			// 'Linux'
			object { letterL translate <-2, 0, 0> }
			object { letterI translate <-1, 0, 0> }
			object { letterN translate < 0, 0, 0> }
			object { letterU translate < 1, 0, 0> }
			object { letterX translate < 2, 0, 0> }
		#end
	#end
	rotate y*8
	translate y*2.5
}

object { tray rotate y*-100 translate <-18, 0, 3> }
object { tray rotate y*5 translate <4, 0, 18> }
object { tray rotate y*-187 translate <-3, 0, -14> }
               
#if (showLetters)               
union { // letters in tray
        object { letterC rotate <280, 180, 0> translate <-2.2, .6, .45> } 
        object { letterE rotate <280, 180, 0> translate <-1.5, .6, .45> }
        object { letterG rotate <280, 180, 0> translate <1, .6, .45> }
        object { letterI rotate <280, 180, 0> translate <0, .6, .45> }
        rotate y*-187 translate <-3, 0, -14>
}  
#end

// 1C 2E 1G 1H 1I  

box {
        <-50, 10, 50>, <50, -20, 51>
        pigment { 
                image_map { jpeg "room.jpg" }  
                scale 100 translate y*-60
        }                    
        finish { diffuse 0 ambient 1 }
        rotate <0, -15, 0>
}                                       


