// Persistence of Vision Ray Tracer Include File
// File: SS_RouterBits.inc (file version 1.0)
// Vers: 3.5
// Desc: Some edge-cutting objects and macros
// Date: 05/22/2004
// Auth: Copyright  2004 by Sherry K. Shaw


//------------------------------------------------------------------------------
// Includes
//------------------------------------------------------------------------------

// Try to avoid including anything more than once...
#ifndef(_transforms_INC)
  #include "transforms.inc"
  #declare _transforms_INC = 1 ;
#end


//******************************************************************************
//
//  ***  THE ROUTER BITS  ***
//  These objects are used by the macros at the end of this file, and may also
//  be used separately.
//
//  HOW TO USE:  Each router bit is based on a one-unit cube, starting at zero
//  x & y, and centered on z.  The cutting surface is on the lower right.
//  Scale in the x & y dimensions to set the size and proportions of the cut.
//  Scale the z dimension slightly longer than the length of the "board."
//
//******************************************************************************

// Rabbet bit (and the starting point for     //      0  1
// bits 02 through 13)                        //      |
#declare RB_00_OBJ =                          //  0 --####--
box {                                         //      ####
  < 0.0, -1.0, -0.5 >, < 1.0,  0.0,  0.5 >    // -1   ####
}                                             //      |

// Chamfer bit (just a slanted edge)
#declare RB_01_OBJ =                                //      0  1
prism {                                             //      |
  linear_spline                                     //  0 --####--
  -0.5, 0.5, 4,                                     //      ###
  < 0, 0 >, < 0, -1 >, < 1, 0 >, < 0, 0 >           //      ##
  rotate x*-90                                      // -1   #
}                                                   //      |

// Roundover bit (basic rounded corner)
#declare RB_02_OBJ =                                      //     0       1
difference {                                              //     |
  object {                                                //  0 -#########-
    RB_00_OBJ                                             //     #######
  }                                                       //     ####
  cylinder {                                              //     ##
    < 1.0, -1.0, -0.51 >, < 1.0, -1.0, 0.51 >, 1.0        //     #
  }                                                       // -1  #
}                                                         //     |

// Stepped roundover bit with peak in center
#declare RB_03_OBJ =
difference {
  object {
    RB_00_OBJ
  }
  union {
    cylinder {
      < 1.0, -0.5, -0.51 >, < 1.0, -0.5, 0.51 >, 0.5
    }
    cylinder {
      < 0.5, -1.0, -0.51 >, < 0.5, -1.0, 0.51 >, 0.5
    }
  }
}

// Beaded roundover bit (big bump in center, bead above and below)
#declare RB_04_OBJ =
difference {
  object {
    RB_00_OBJ
  }
  union {
    cylinder {
      < 1.0, -1.0, -0.51 >, < 1.0, -1.0, 0.51 >, 0.875
    }
    cylinder {
      < 1.0, -0.125, -0.51 >, < 1.0, -0.125, 0.51 >, 0.125
    }
    cylinder {
      < 0.125, -1.0, -0.51 >, < 0.125, -1.0, 0.51 >, 0.125
    }
  }
}

// Beaded roundover bit (big bump in center, bead above and below)
#declare RB_05_OBJ =
difference {
  object {
    RB_00_OBJ
  }
  union {
    cylinder {
      < 1.0, -1.0, -0.51 >, < 1.0, -1.0, 0.51 >, 0.875
    }
    cylinder {
      < 1.0, -0.25, -0.51 >, < 1.0, -0.25, 0.51 >, 0.25
    }
    cylinder {
      < 0.25, -1.0, -0.51 >, < 0.25, -1.0, 0.51 >, 0.25
    }
  }
}

// Beaded roundover bit (big bump in center, bead above)
#declare RB_06_OBJ =
difference {
  object {
    RB_00_OBJ
  }
  union {
    cylinder {
      < 0.0, 0.0, -0.51 >, < 0.0, 0.0, 0.51 >, 1.0
      scale < 1, 0.875, 1 >
      translate < 1, -1, 0 >
    }
    cylinder {
      < 1.0, -0.125, -0.51 >, < 1.0, -0.125, 0.51 >, 0.125
    }
  }
}

// Beaded roundover bit (big bump in center, bead above)
#declare RB_07_OBJ =
difference {
  object {
    RB_00_OBJ
  }
  union {
    cylinder {
      < 0.0, 0.0, -0.51 >, < 0.0, 0.0, 0.51 >, 1.0
      scale < 1, 0.875, 1 >
      translate < 1, -1, 0 >
    }
    cylinder {
      < 1.0, -0.25, -0.51 >, < 1.0, -0.25, 0.51 >, 0.25
    }
  }
}

// Beaded roundover bit (big bump in center, bead below)
#declare RB_08_OBJ =
difference {
  object {
    RB_00_OBJ
  }
  union {
    cylinder {
      < 0.0, 0.0, -0.51 >, < 0.0, 0.0, 0.51 >, 1.0
      scale < 0.875, 1, 1 >
      translate < 1, -1, 0 >
    }
    cylinder {
      < 0.125, -1.0, -0.51 >, < 0.125, -1.0, 0.51 >, 0.125
    }
  }
}

// Beaded roundover bit (big bump in center, bead below)
#declare RB_09_OBJ =
difference {
  object {
    RB_00_OBJ
  }
  union {
    cylinder {
      < 0.0, 0.0, -0.51 >, < 0.0, 0.0, 0.51 >, 1.0
      scale < 0.875, 1, 1 >
      translate < 1, -1, 0 >
    }
    cylinder {
      < 0.25, -1.0, -0.51 >, < 0.25, -1.0, 0.51 >, 0.25
    }
  }
}

// Cove bit (rounded-out)
#declare RB_10_OBJ =
intersection {
  object {
    RB_00_OBJ
  }
  cylinder {
    < 0.0, 0.0, -0.51 >, < 0.0, 0.0, 0.51 >, 1.0
  }
}

// Ogee bit (rounded-out at the top, in at the bottom)
#declare RB_11_OBJ =
union {
  difference {
    object {
      RB_00_OBJ
      scale < 0.5, 1, 1 >
    }
    cylinder {
      < 0.5, -1.0, -0.51 >, < 0.5, -1.0, 0.51 >, 0.5
    }
  }
  intersection {
    object {
      RB_00_OBJ
    }
    cylinder {
      < 0.5, 0.0, -0.51 >, < 0.5, 0.0, 0.51 >, 0.5
    }
  }
}

// Roman ogee bit (rounded-in at the top, out at the bottom; not to be confused with the Roman orgy
// bit, which is something entirely different)
#declare RB_12_OBJ =
union {
  difference {
    object {
      RB_00_OBJ
      scale < 1, 0.5, 1 >
    }
    cylinder {
      < 1.0, -0.5, -0.51 >, < 1.0, -0.5, 0.51 >, 0.5
    }
  }
  intersection {
    object {
      RB_00_OBJ
    }
    cylinder {
      < 0.0, -0.5, -0.51 >, < 0.0, -0.5, 0.51 >, 0.5
    }
  }
}

// Round groove bit
#declare RB_13_OBJ =
intersection {
  object {
    RB_00_OBJ
    scale < 1.01, 1, 1 >
  }
  cylinder {
    < 0, -0.5, -0.51 >, < 0, -0.5, 0.51 >, 0.5
    scale < 2, 1, 1 >
  }
}

// Pointy groove bit
#declare RB_14_OBJ =
prism {
  linear_spline
  -0.5, 0.5, 4,
  < 0, 0 >, < 0, -1 >, < 1, -0.5 >, < 0, 0 >
  rotate x*-90
}

// Finally, a bit of fudge factor
#ifndef( RB_SMIDGEN )
  #declare RB_SMIDGEN = 0.01 ;
#end


//******************************************************************************
//
//  macEdgeCutter--Builds an edge-cutting object from a stack of router bit objects.
//
//  Parameters:
//
//    Thickness = Float; thickness (y size) of target object to be cut by edge-cutter.
//                (Note that the height of the finished edge-cutting object will equal
//                Smidgen+Thickness+Smidgen, with the top of the object at y=Smidgen.)
//    Width     = Float; width (z size) of the edge-cutting object.  (Set to 1 to use
//                with macEdgedPolygon or macEdgedCircle.)
//    CutTop    = Boolean; true=object will be placed at top of stack to cut top
//                edge of target; false=no object at top.
//    CutMiddle = Boolean; true=additional objects will be added to the middle of
//                the stack to cut into the edge of the target; false=no middle
//                objects.
//    CutBottom = Boolean; true=object will be placed at bottom of stack to cut
//                bottom edge of target; false=no object at bottom.
//    AddSpacer = Boolean; true=add block at back (-x side) of object to cut flat
//                surface of target; false=no spacer block.  (Set to false to use with
//                macEdgedPolygon.)
//    Smidgen   = Float; offset value used to avoid coincident surface problems.
//                Smidgen<0 = use absolute value as local variable.
//                Smidgen>0 = set RB_SMIDGEN to value & use.  (Use a value in this range
//                if you intend using the created object in macEdgedPolygon or macEdgedCircle.)
//                Smidgen=0 = use current value of RB_SMIDGEN (default = 0.01).
//
//  Any of the following variables to be used in this macro MUST be #declared as noted
//  below BEFORE calling this macro:
//
//  If CutTop=true:
//    RB_EC_TOP_OBJ   = Object; router bit at top of stack.
//    RB_EC_TOP_SCALE = 2d vector used to scale RB_EC_TOP_OBJ; u=depth; v=height.
//    RB_EC_TOP_INV   = Boolean; true=rotate RB_EC_TOP_OBJ x*180; false (or undefined)=
//                      no rotation.
//
//  If CutMiddle=true:
//    RB_EC_NUM_MIDDLE_BITS = Integer; number of router bit objects to be added to the
//                            middle of the stack.
//    RB_EC_MIDDLE_OBJ[n]   = Array of router bit objects.
//    RB_EC_MIDDLE_SCALE[n] = Array of 2d vectors used for xy scaling each object.
//    RB_EC_MIDDLE_INV[n]   = Array of boolean values; true=rotate RB_EC_MIDDLE_OBJ[n] x*180;
//                            false (or undefined)=no rotation.
//    RB_EC_MIDDLE_Y_OFF[n] = Array of floats; y-offset of center of each router bit from
//                            center of stack (for example, to center a groove-cutting bit
//                            on the edge of the target, use RB_EC_MIDDLE_Y_OFF[n]=0).
//
//  If CutBottom=true:
//    RB_EC_BOTTOM_OBJ   = Object; router bit to be used to cut the bottom edge of the target.
//    RB_EC_BOTTOM_SCALE = 2d vector used to scale RB_EC_BOTTOM_OBJ; u=depth; v=height.
//    RB_EC_BOTTOM_INV   = Boolean; true=rotate RB_EC_BOTTOM_OBJ x*180; false (or undefined)=
//                         no rotation.
//
//******************************************************************************

#macro macEdgeCutter( Thickness, Width, CutTop, CutMiddle, CutBottom, AddSpacer, Smidgen )

// Set up Smidgen value
#if ( Smidgen < 0 )
  #local Offset = abs( Smidgen ) ;
#else
  #if ( Smidgen > 0 )
    #local Offset = Smidgen ;
    #declare RB_SMIDGEN = Smidgen ;
  #else
    #local Offset = RB_SMIDGEN ;
  #end
#end

// Build edge-cutter
union {
  #if ( CutTop = true )
    object {
      RB_EC_TOP_OBJ
      #ifndef( RB_EC_TOP_INV )
        #declare RB_EC_TOP_INV = false ;
      #end
      #if ( RB_EC_TOP_INV = true )
        rotate x*180
        translate < 0, -1, 0 >
      #end
      scale < RB_EC_TOP_SCALE.u, RB_EC_TOP_SCALE.v, Width >
      translate < 0, Offset, 0 >
    }
  #end
  #if ( CutMiddle = true )
    #local Ix = 0 ;
    #while ( Ix < RB_EC_NUM_MIDDLE_BITS )
      object {
        RB_EC_MIDDLE_OBJ[ Ix ]
        #ifndef( RB_EC_MIDDLE_INV[ Ix ] )
          #declare RB_EC_MIDDLE_INV[ Ix ] = false ;
        #end
        #if ( RB_EC_MIDDLE_INV[ Ix ] = true )
          rotate x*180
          translate < 0, -1, 0 >
        #end
        translate < 0, 0.5, 0 >
        scale < RB_EC_MIDDLE_SCALE[ Ix ].u, RB_EC_MIDDLE_SCALE[ Ix ].v, Width >
        translate < 0, ( RB_EC_MIDDLE_Y_OFF[ Ix ] - ( Thickness * 0.5 ) ), 0 >
      }
      #local Ix = Ix + 1 ;
    #end
  #end
  #if ( CutBottom = true )
    object {
      RB_EC_BOTTOM_OBJ
      #ifndef( RB_EC_BOTTOM_INV )
        #declare RB_EC_BOTTOM_INV = false ;
      #end
      #if ( RB_EC_BOTTOM_INV = true )
        rotate x*180
        translate < 0, -1, 0 >
      #end
      translate < 0, 1, 0 >
      scale < RB_EC_BOTTOM_SCALE.u, RB_EC_BOTTOM_SCALE.v, Width >
      translate < 0, -( Thickness + Offset ), 0 >
    }
  #end
  #if ( AddSpacer = true )
    object {
      RB_00_OBJ
      translate < -1, 0, 0 >
      scale < 1, ( Thickness + ( 2 * Offset ) ), Width >
      translate < Offset, 0, 0 >
    }
  #end
}

#end    // macEdgeCutter


//******************************************************************************
//
//  macEdgedCircle--Creates a circular tabletop (or whatever) with the edge cut
//    by the specified router bit object(s).  The final object extends from y=0
//    to y=Thickness.
//
//  Parameters:
//
//    Rad       = Float; radius of cylinder from which tabletop will be cut.
//    Thickness = Float; thickness (y) of the cylinder.
//    NumCuts   = Integer; number of cuts to make around the edge of the cylinder.
//                Higher values make a smoother circle but take longer to render.
//
//  The following variables should be #declared before calling this macro:
//
//    RB_EDGE_CUTTER_OBJ = The object to be used to shape the edge of the polygon.
//                         Note that this may be one of the router bit objects from
//                         this file, but may be any object.  (Use the macro
//                         macEdgeCutter to build an edge-cutting object from
//                         a "stack" of router bits.)  The object is assumed to
//                         have an upper-left corner of x=0, y=Smidgen, and to
//                         extend right and down.  If RB_EDGE_CUTTER_OBJ has not
//                         been #declared, RB_01_OBJ will be used.
//    RB_SMIDGEN = A teensy bit of offset applied to the edge cutter to avoid
//                 problems with coincident surfaces.  If this variable is not
//                 #declared, a default value of 0.01 will be used.  Note that,
//                 when using this macro to cut both the top and bottom edges of
//                 an object, the height of RB_EDGE_CUTTER_OBJ should be equal to
//                 Thickness+(2*RB_SMIDGEN).  (This variable may be set in
//                 macEdgeCutter.)
//
//******************************************************************************

#macro macEdgedCircle ( Rad, Thickness, NumCuts )

#ifndef( RB_SMIDGEN )
  #local RB_SMIDGEN = 0.01 ;
#end

#ifndef( RB_EDGE_CUTTER_OBJ )
  #local RB_EDGE_CUTTER_OBJ =
  object {
    RB_01_OBJ
    translate < 0, RB_SMIDGEN, 0 >
  }
#end

#local Rot = 360 / NumCuts ;

difference {
  cylinder {
    < 0, -Thickness, 0 >, < 0, 0, 0 >, Rad
  }
  union {
    #local Ix = 0 ;
    #while ( Ix < NumCuts )
      object {
        RB_EDGE_CUTTER_OBJ
        translate < -( Rad + RB_SMIDGEN ), 0, 0 >
        rotate y*Rot*Ix
      }
      #local Ix = Ix + 1 ;
    #end
  }
  translate < 0, Thickness, 0 >
}

#end    // macEdgedCircle


//******************************************************************************
//
//  macEdgedPolygon--Creates a regular polygon-shaped tabletop (or whatever) with
//    the edge cut by the specified router bit object(s).  The final object extends
//    from y=0 to y=Thickness.
//
//  Parameters:
//
//    OuterRad  = Float; radius of cylinder from which polygon will be cut.
//    Thickness = Float; thickness (y) of the cylinder.
//    NumSides  = Integer; number of sides to give the polygon.
//
//  The following variables should be #declared before calling this macro:
//
//    RB_EDGE_CUTTER_OBJ = The object to be used to shape the edge of the polygon.
//                         Note that this may be one of the router bit objects from
//                         this file, but may be any object.  (Use the macro
//                         macEdgeCutter to build an edge-cutting object from
//                         a "stack" of router bits.)  Note that only the shaped
//                         portions of the edge-cutting object need to be defined;
//                         the flat bits are added in this macro.  The object is
//                         assumed to have an upper-left corner of x=0, y=Smidgen,
//                         and to extend right and down.  If RB_EDGE_CUTTER_OBJ has
//                         not been #declared, RB_01_OBJ will be used.
//    RB_SMIDGEN = A teensy bit of offset applied to the edge cutter to avoid
//                 problems with coincident surfaces.  If this variable is not
//                 #declared, a default value of 0.01 will be used.  Note that,
//                 when using this macro to cut both the top and bottom edges of
//                 an object, the height of RB_EDGE_CUTTER_OBJ should be equal to
//                 Thickness+(2*RB_SMIDGEN).  (This variable may be set in
//                 macEdgeCutter.)
//
//******************************************************************************

#macro macEdgedPolygon ( OuterRad, Thickness, NumSides )

#ifndef( RB_SMIDGEN )
  #local RB_SMIDGEN = 0.01 ;
#end

#ifndef( RB_EDGE_CUTTER_OBJ )
  #local RB_EDGE_CUTTER_OBJ =
  object {
    RB_01_OBJ
    translate < 0, RB_SMIDGEN, 0 >
  }
#end

#local UncutObj =
cylinder {
  < 0, -Thickness, 0 >, < 0, 0, 0 >, OuterRad
}

#local Rot = 360 / NumSides ;
#local PtA = < OuterRad, 0, 0 > ;
#local PtB = vrotate( PtA, < 0, Rot, 0 > ) ;
#local ZScale = vlength( PtA - PtB ) ;
#local XTrans = sqrt( pow( OuterRad, 2 ) - pow( ( ZScale * 0.5 ), 2 ) ) ;
#local XScale = OuterRad - XTrans ;

#local EdgeCutter =
union {
  object {
    RB_EDGE_CUTTER_OBJ
    scale < 1, 1, ( ZScale + RB_SMIDGEN ) >
  }
  object {
    RB_00_OBJ
    translate < -1, 0, 0 >
    scale < ( XScale + ( 2 * RB_SMIDGEN ) ), ( Thickness + ( 2 * RB_SMIDGEN ) ), ( ZScale + RB_SMIDGEN ) >
    translate < RB_SMIDGEN, RB_SMIDGEN, 0 >
  }
  translate < -RB_SMIDGEN, 0, 0 >
}

difference {
  object {
    UncutObj
  }
  union {
    #local Ix = 0 ;
    #while ( Ix < NumSides )
      object {
        EdgeCutter
        translate < -XTrans, 0, 0 >
        rotate y*Rot*Ix
      }
      #local Ix = Ix + 1 ;
    #end
  }
  translate < 0, Thickness, 0 >
}

#end    // macEdgedPolygon


//******************************************************************************
//
//  macCutTracedEdge--Uses a router bit object to shape the edge of the specified
//    target, using trace to follow the edge.  The target should be centered at
//    the origin, with the top at y=0.
//
//  Parameters:
//
//    TargetObj = Object; the object to be shaped.
//    MaxRad    = Float; maximum radius of target object.
//    NumCuts   = Integer; number of cuts to make around the edge of the object.
//                Higher values make a smoother edge but take longer to render.
//    DropAmt   = Float; distance below y=0 from which to start trace (for example,
//                to start tracing 1.5 units below the top, set DropAmt=-1.5).
//
//  The following variables should be #declared before calling this macro:
//
//    RB_EDGE_CUTTER_OBJ = The object to be used to shape the edge of the target.
//                         Note that this may be one of the router bit objects from
//                         this file, but may be any object.  (Use the macro
//                         macEdgeCutter to build an edge-cutting object from
//                         a "stack" of router bits.)  The object is assumed to
//                         have an upper-left corner of x=0, y=Smidgen, and to
//                         extend right and down.  If RB_EDGE_CUTTER_OBJ has not
//                         been #declared, RB_01_OBJ will be used.
//    RB_SMIDGEN = A teensy bit of offset applied to the edge cutter to avoid
//                 problems with coincident surfaces.  If this variable is not
//                 #declared, a default value of 0.01 will be used.  Note that,
//                 when using this macro to cut both the top and bottom edges of
//                 an object, the height of RB_EDGE_CUTTER_OBJ should be equal to
//                 Thickness+(2*RB_SMIDGEN).  (This variable may be set in
//                 macEdgeCutter.)
//
//******************************************************************************

#macro macCutTracedEdge ( TargetObj, MaxRad, NumCuts, DropAmt )

#ifndef( RB_SMIDGEN )
  #local RB_SMIDGEN = 0.01 ;
#end

#ifndef( RB_EDGE_CUTTER_OBJ )
  #local RB_EDGE_CUTTER_OBJ =
  object {
    RB_01_OBJ
    translate < 0, RB_SMIDGEN, 0 >
  }
#end

#local Rot = 360 / NumCuts ;
#local StartPt = < -( MaxRad + 1 ), DropAmt, 0 > ;

difference {
  object {
    TargetObj
  }
  union {
    #local Ix = 0 ;
    #while ( Ix < NumCuts )
      #local TracePt = vrotate( StartPt, < 0, ( Rot * Ix ), 0 > ) ;
      #local AimVec = vnormalize( vrotate( TracePt, < 0, 180, 0 > ) ) ;
      #local Norm = < 0, 0, 0 > ; 
      #local HitPt = trace( TargetObj, TracePt, AimVec, Norm ) ;
      #if ( vlength( Norm ) != 0 )
        object {
          RB_EDGE_CUTTER_OBJ
          #local Trans = Reorient_Trans( < -1, 0, 0 >, Norm )
          transform Trans
          #if ( HitPt.x = 0 )
            #local DirX = 1 ;
          #else
            #local DirX = HitPt.x / abs( HitPt.x ) ;
          #end
          #if ( HitPt.z = 0 )
            #local DirZ = 1 ;
          #else
            #local DirZ = HitPt.z / abs( HitPt.z ) ;
          #end
          translate < ( HitPt.x + ( RB_SMIDGEN * DirX ) ), 0, ( HitPt.z + ( RB_SMIDGEN * DirZ ) ) >
        }
      #end
      #local Ix = Ix + 1 ;
    #end
  }
}

#end    // macCutTracedEdge


//******************************************************************************
//
//  macCutCSplineEdge--Uses a router bit object to shape the edge of the specified
//    target prism, using the cubic spline that defines the prism to follow the edge.
//    The target should have its top at y=0.
//
//    This is the "if all else fails" macro.  It works pretty well on prisms with
//    concave areas along the edge IF they aren't too terribly concave, and IF the
//    edge cutter is narrow enough not to overlap the adjacent cuts very much, and
//    IF you specify many, many cuts.  Yes, it's very slow.
//
//  Parameters:
//
//    TargetObj = Object; the object to be shaped.
//    NumCuts   = Integer; number of cuts to make around the edge of the object.
//                Higher values make a smoother edge but take longer to render.
//
//  The following variables MUST be #declared BEFORE calling this macro:
//
//    RB_ECS_NUM_PTS = Integer; the number of points in the spline.
//    RB_ECS_PT[ n ] = Array of 2d vectors used to define the prism.
//
//  The following variables should be #declared before calling this macro:
//
//    RB_EDGE_CUTTER_OBJ = The object to be used to shape the edge of the target.
//                         Note that this may be one of the router bit objects from
//                         this file, but may be any object.  (Use the macro
//                         macEdgeCutter to build an edge-cutting object from
//                         a "stack" of router bits.)  The object is assumed to
//                         have an upper-left corner of x=0, y=Smidgen, and to
//                         extend right and down.  If RB_EDGE_CUTTER_OBJ has not
//                         been #declared, RB_01_OBJ will be used.
//    RB_SMIDGEN = A teensy bit of offset applied to the edge cutter to avoid
//                 problems with coincident surfaces.  If this variable is not
//                 #declared, a default value of 0.01 will be used.  Note that,
//                 when using this macro to cut both the top and bottom edges of
//                 an object, the height of RB_EDGE_CUTTER_OBJ should be equal to
//                 Thickness+(2*RB_SMIDGEN).  (This variable may be set in
//                 macEdgeCutter.)
//
//******************************************************************************

#macro macCutCSplineEdge ( TargetObj, NumCuts )

#ifndef( RB_SMIDGEN )
  #local RB_SMIDGEN = 0.01 ;
#end

#ifndef( RB_EDGE_CUTTER_OBJ )
  #declare RB_EDGE_CUTTER_OBJ =
  object {
    RB_01_OBJ
    translate < 0, RB_SMIDGEN, 0 >
  }
#end

#local EdgeCutterObj =
object {
  RB_EDGE_CUTTER_OBJ
  translate < -RB_SMIDGEN, 0, 0 >
  rotate y*-90
}

#local IDInc = 1 / ( RB_ECS_NUM_PTS - 3 ) ;
#local PtInc = 1 / NumCuts ;

#local PathSpline =
spline {
  cubic_spline
  #local Ix = 0 ;
  #while ( Ix < RB_ECS_NUM_PTS )
    #local ID = ( Ix - 1 ) * IDInc ;
    ID, < RB_ECS_PT[ Ix ].u, 0, RB_ECS_PT[ Ix ].v >
    #local Ix = Ix + 1 ;
  #end
}

difference {
  object {
    TargetObj
  }
  union {
    #local Ix = 0 ;
    #while ( Ix < NumCuts )
      #local Pt0 = PathSpline( ( Ix * PtInc ) - 0.0001 ) ;
      #local Pt1 = PathSpline( Ix * PtInc ) ;
      #local Pt2 = PathSpline( ( Ix * PtInc ) + 0.0001 ) ;
      #local Norm = ( Pt0 - Pt2 ) ;
      #local Trans = Reorient_Trans( < -1, 0, 0 >, Norm )
      object {
        EdgeCutterObj
        transform Trans
        translate < Pt1.x, 0, Pt1.z >
      }
      #local Ix = Ix + 1 ;
    #end
  }
}

#end    // macCutCSplineEdge
