///////////////////////////////////////////////////////////////////////////////
// MJR Scene Constants
//
// History
// 1/12/04 - Created
//
// Various Constant definitions used in my scenes and objects
//
// ----------------------------------------------------------------------------
// Units of measure
//
// Metric: mm, cm, m, km
// Customary: in, ft, yd, mi
//
// unit_scale_factor: This value determines how many units will be used as the 
// baseline. Scenes with very large units or very small units should change the
// unit_scale_factor value to prevent rendering artifacts due to lack of 
// floating-point precision. 
//
// Example of usage for units: sphere { <0, 0, 0>*cm, 3*cm ... } 
// which will create a sphere at origin with a radius of 3 centemeters. 
///////////////////////////////////////////////////////////////////////////////

#ifndef(I_MJR_SCENE_CONSTS)
#declare I_MJR_SCENE_CONSTS = 1;

///////////////////////////////////////////////////////////////////////////////
// Units of measure
///////////////////////////////////////////////////////////////////////////////

// ---- Scaling Factor
// If there are problems with accuracy, a scaling factor can be defined to 
// compensate for the inaccuracies within the ray-object intersection algorithms
// If objects begin showing holes or unexpected noise, change the unit scaling
// factor.

#ifndef(unit_scale_factor)
    #declare unit_scale_factor = 1;
#end
    

// ---- Metric units of measure (1cm = 1POV)
#declare cm = 1.0 * unit_scale_factor;      // centimeters
#declare mm = cm  * 0.1;                    // millimeters
#declare m  = cm  * 100;                    // meters
#declare km = m   * 1000;                   // kilometers

// ---- Customary units of measure
#declare in = cm*2.54;                      // inches
#declare ft = in*12;                        // feet
#declare yd = ft*3;                         // yards
#declare mi = ft*5280;                      // miles

#ifndef(epsilon)
    #declare epsilon = 1.0e-10;
#end


///////////////////////////////////////////////////////////////////////////////
// Constants Debugging
///////////////////////////////////////////////////////////////////////////////
#ifdef(MJR_SCENE_CONSTS_DEBUG)

    #debug "\n"
    
    #debug concat("mm = ", str(mm, 0, -1), " units\n")
    #debug concat("cm = ", str(cm, 0, -1), " units\n")
    #debug concat("m = ", str(m, 0, -1), " units\n")
    #debug concat("km = ", str(km, 0, -1), " units\n")

    #debug "\n"

    #debug concat("in = ", str(in, 0, -1), " units\n")
    #debug concat("ft = ", str(ft, 0, -1), " units\n")
    #debug concat("yd = ", str(yd, 0, -1), " units\n")
    #debug concat("mi = ", str(mi, 0, -1), " units\n")

    #debug "\n"

#end

#end
