// Persistence of Vision Ray Tracer Scene Description File
// File: aliums.pov
// Vers: 3.5
// Desc: IRTC Out of place entry
// Date: 01-Feb-05
// Auth: Sean Day
// 
// Thanks to Gilles Tran for writing/releasing the mgrass macros..

#version 3.5;

#include "colors.inc"   
#include "functions.inc"
#include "mgrass.pov"         
#include "shapes.inc"

#declare showPetals     = 1;
#declare showLeaves     = 1;
#declare showStem       = 1;
#declare showStigmata   = 1;      
#declare numFlowers     = 40; //40

#declare useTextures    = 1; // 0 = test tex 1 = full tex
#declare useFocalBlur   = 2; // 0 = none, 1 = low quality, 2 = high quality   
#declare useArea        = 2; // 0 = none, 1 = low quality, 2 = high quality        
#declare useGeo         = 1; 
#declare useRadiosity   = 3; // 0 = none, 1 = load, 2 = save, 3 = low quality 


#declare segBlade= 20;          // number of blade segments - 8 for test 20 for high

global_settings {
 assumed_gamma 1.0
 ambient_light 0.0 
 max_trace_level 10        
#if(useRadiosity) 
 radiosity{ 
  // save settings: modified Outdoor settings from rad_def.inc
  #if (useRadiosity=2) 
    pretrace_start 0.08
    pretrace_end   0.01
    count 250
    nearest_count 20
    error_bound 0.1
    recursion_limit 2
    low_error_factor .9
    gray_threshold 0
    minimum_reuse 0.015
    brightness 1
    adc_bailout 0.01/2
    save_file "flower.rad"
  #end
  #if(useRadiosity=3)
    pretrace_start 0.08           // start pretrace at this size
    pretrace_end   0.04 //0.04    // end pretrace at this size
    count 20    //25             // higher -> higher quality (1..1600) [35]
    nearest_count 8 // 8         // higher -> higher quality (1..10) [5]
    error_bound 1.8 //1.8        // higher -> smoother, less accurate [1.8]
    recursion_limit 3 //3         // how much interreflections are calculated (1..5+) [3]
    low_error_factor .3 //0.3     // reduce error_bound during last pretrace step
    gray_threshold 0.0            // increase for weakening colors (0..1) [0]
    minimum_reuse 0.015           // reuse of old radiosity samples [0.015]
    brightness 1               // brightness of radiosity effects (0..1) [1]

    adc_bailout 0.01/2
  #end
  #if(useRadiosity=1)
    // load settings  
    pretrace_start 1 pretrace_end 1 
    load_file "flower.rad" 
    error_bound 0.1    // it's important to specify the same value used to save data
    recursion_limit 2  // also the same if specified
    always_sample off  // no more samples: specially useful with focal blur
  #end
 }
 #end
}


// perspective (default) camera
camera {
  location  <0.0, 74.0, -130.0>
  look_at   <0.0, 60.0,  0.0>
  right     x*image_width/image_height    
#if(useFocalBlur)
  aperture 6        // [0...N] larger is narrower depth of field (blurrier)
  focal_point <2, 100, -3>     // point that is in focus <X,Y,Z>
  #if(useFocalBlur = 2)
  blur_samples 120        // number of rays per pixel for sampling
  confidence 0.95           // [0...<1] when to move on while sampling (smaller is less accurate)
  variance 1/400000            // [0...1] how precise to calculate (smaller is more accurate)
  #else
  blur_samples 10       // number of rays per pixel for sampling
  confidence 0.90          // [0...<1] when to move on while sampling (smaller is less accurate)
  variance 1/200            // [0...1] how precise to calculate (smaller is more accurate)  
  #end
#end
}

#declare generalRand = seed(582);   

// Macro to return a random number

#macro gr(num)
        rand(generalRand)*num
#end
 // create a regular point light source
light_source {
  0*x                  // light's position (translated below)
  color White+Gold    // light's color
  // light_source { ...
  // put this inside a light_source to add light fading   
  #if(useArea)
        area_light
        <4, 0, 0> <0, 0, 4> // lights spread out across this distance (x * z)
        #if(useArea = 1)
                3, 3                // total number of lights in grid (4x*4z = 16 lights)
        #else
                15, 15
        #end
        adaptive 1          // 0,1,2,3...  
        jitter              // adds random softening of light
        circular            // make the shape of the light circular
        orient              // orient light
  #end

  translate <-50, 100, -100>
  fade_distance 75
  fade_power 2

}     

// Create an infinite sphere around scene and allow any pigment on it
sky_sphere {
  pigment {
   White
  }
} 




//#declare flowerRed = rgbt <0.55, 0.02, 0.2, 0.025>; 
//#declare flowerPink =  rgbt <0.65, 0.2, 0.55, 0.15>; 

#declare StemRed = rgb <0.45, 0.08, 0.125>; 
#declare StemPink =  rgb <0.75, 0.4, 0.375>; 


/*********************************************************************************************************************************
**************** 
****************        MKGRASS Macro written by Gilles Tran (not my own work..)
****************
*********************************************************************************************************************************/

#declare Green1=rgb<0.285,0.65, 0.034>;
#declare Green2=rgb<0.385,0.675,0.085>;
#if(useTextures)
#declare leafTex=texture {                  

                pigment {
                        spotted        
                        lambda 4
                        omega 0.55
                        turbulence 0.275
                                pigment_map {                                
                                    [0.0 Green1]
                                    [0.5 Green2 ]
                                    [1.00 Green1]
                                }    
                                scale 2.5
                        }  

        finish{ambient 0 diffuse 0.7 brilliance 0.9 phong 0.15 phong_size 30 }
        scale <0.1,1,0.1> 
}                            
texture
{ 
        pigment 
        { 
                marble  
                lambda 4    
                omega 0.55
                turbulence 0.375
                color_map
                {
                        [ 0.0 rgbt 1 ]  
                        [ 0.8 rgbt 1 ]  
                        [ 0.85 rgbt <Green1.red*0.8, Green1.green*0.8, Green1.blue*0.8, 0.1> ]
                        [ 1.0 rgbt <Green1.red*0.5, Green1.green*0.6, Green1.blue*0.6, 0.1> ]
                }                                   
                scale 0.05
        }   
        normal { dents 0.15 scale 0.75 }
        finish { ambient 0 diffuse 0.7 brilliance 0.7 phong 0.15 phong_size 30 } 
}
#else
#declare leafTex=texture {                  
               pigment { Green1 }
               finish{ambient 0 diffuse 0.7 brilliance 0.9 }
}
#end
                        
#declare driedLeaf1 = rgb <0.627059, 0.507059, 0.202>;
#declare driedLeaf2 = rgb <0.747059, 0.607059, 0.292>;


#if(useTextures)
#declare driedLeafTex=texture {                  

                pigment {
                        spotted        
                        lambda 4
                        omega 0.55
                        turbulence 0.275
                                pigment_map {                                
                                    [0.0 driedLeaf1]
                                    [0.5 driedLeaf2 ]
                                    [1.00 driedLeaf1]
                                }    
                                scale 2.5
                        }  

        finish{ambient 0 diffuse 0.7 brilliance 0.9 }
        scale <0.1,1,0.1> 
}                            
texture
{ 
        pigment 
        { 
                marble  
                lambda 4    
                omega 0.55
                turbulence 0.375
                color_map
                {
                        [ 0.0 rgbt 1 ]  
                        [ 0.8 rgbt 1 ]  
                        [ 0.85 rgbt <driedLeaf1.red*0.8, driedLeaf1.green*0.8, driedLeaf1.blue*0.8, 0.1> ]
                        [ 1.0 rgbt <driedLeaf1.red*0.5, driedLeaf1.green*0.6, driedLeaf1.blue*0.6, 0.1> ]
                }                                   
                scale 0.05
        }  
        normal { dents 0.15 scale 0.75 }
        finish { ambient 0 diffuse 0.7 brilliance 0.7 } 
}
#else
#declare driedLeafTex=texture {                  
                pigment {driedLeaf1 }
                finish{ambient 0 diffuse 0.7 brilliance 0.9 }
}                            
#end

#declare petalColour=rgb <0.6, 0.15, 1>;

#if(useTextures)
#declare petalTex = texture 
{ 
        pigment 
        { 
                gradient y
                turbulence 0.25
                color_map
                {
                        [ 0.0 rgb <petalColour.red, petalColour.green, petalColour.blue>*0.25 ]
                        [ 0.25 rgb <petalColour.red, petalColour.green, petalColour.blue>*0.45 ]
                        [ 0.5 rgb <petalColour.red, petalColour.green, petalColour.blue>*0.8 ]
                        [ 1.0 rgbt <petalColour.red, petalColour.green, petalColour.blue, 0.05> ]
                          
                }                                   
                scale 1
        }  
        finish { ambient 0 diffuse 0.7 brilliance 0.6 phong 0.1 phong_size 25 } 
}
texture
{ 
        pigment 
        { 
                marble  
                lambda 4    
                omega 0.55
                turbulence 0.275
                color_map
                {
                        [ 0.0 rgbt 1 ]  
                        [ 0.8 rgbt 1 ]  
                        [ 0.9 rgbt <petalColour.red*0.7, petalColour.green*0.7, petalColour.blue*0.7, 0.1> ]
                        [ 1.0 rgbt <petalColour.red*0.4, petalColour.green*0.4, petalColour.blue*0.4, 0.1> ]
                }                                   
                scale 0.05
        }  
        finish { ambient 0 diffuse 0.7 brilliance 0.7 phong 0.1 phong_size 25 } 
}   
#else
#declare petalTex = texture 
{ 
        pigment 
        { 
                        rgb <petalColour.red, petalColour.green, petalColour.blue>*0.8 
        }  
        finish { ambient 0 diffuse 0.7 brilliance 0.6 } 
}
#end



                        
#declare driedPetalColour=rgb <0.785, 0.6095, 0.185>;

#if(useTextures)
#declare driedPetalTex = texture 
{ 
        pigment 
        { 
                gradient y
                turbulence 0.25
                color_map
                {
                        [ 0.0 rgb <driedPetalColour.red, driedPetalColour.green, driedPetalColour.blue>*0.65 ]
                        [ 0.25 rgb <driedPetalColour.red, driedPetalColour.green, driedPetalColour.blue>*0.75 ]
                        [ 0.5 rgb <driedPetalColour.red, driedPetalColour.green, driedPetalColour.blue>*0.9 ]
                        [ 1.0 rgbt <driedPetalColour.red, driedPetalColour.green, driedPetalColour.blue, 0.075> ]
                          
                }                                   
                scale 1
        }  
        finish { ambient 0 diffuse 0.7 brilliance 0.75 } 
}
texture
{ 
        pigment 
        { 
                marble  
                lambda 4    
                omega 0.55
                turbulence 0.275
                color_map
                {
                        [ 0.0 rgbt 1 ]  
                        [ 0.8 rgbt 1 ]  
                        [ 0.9 rgbt <driedPetalColour.red*0.8, driedPetalColour.green*0.8, driedPetalColour.blue*0.8, 0.1> ]
                        [ 1.0 rgbt <driedPetalColour.red*0.5, driedPetalColour.green*0.5, driedPetalColour.blue*0.5, 0.1> ]
                }                                   
                scale 0.05
        }  
        finish { ambient 0 diffuse 0.7 brilliance 0.75 phong 0.1 phong_size 5 } 
}   
#else
#declare driedPetalTex = texture 
{ 
        pigment 
        { 
                        rgb <driedPetalColour.red, driedPetalColour.green, driedPetalColour.blue>*0.9                         
        }  
        finish { ambient 0 diffuse 0.7 brilliance 0.75 } 
}
#end



#if(useTextures)
#declare flowerStemTex = texture 
            { 
                        pigment 
                        { 
                                gradient y 
                                color_map 
                                { 
                                        [ 0 StemRed  ] 
                                        [ 1 StemPink ]                                
                                
                                }
                                scale 2.5   
                                translate <0, -0.2, 0>
                        }   
                        finish 
                        {
                                ambient 0 phong 0.5 phong_size 10
                        }
             }    
#else
#declare flowerStemTex = texture 
            { 
                        pigment 
                        { 
                                        StemRed  
                        }   
                        finish 
                        {
                                ambient 0 
                        }
             }    
#end


#declare driedStem1 = rgb <0.45, 0.32, 0.095>; 
#declare driedStem2 =  rgb <0.55, 0.4, 0.185>; 

#if(useTextures)
#declare driedFlowerStemTex = texture 
            { 
                        pigment 
                        { 
                                gradient y 
                                color_map 
                                { 
                                        [ 0 driedStem1  ] 
                                        [ 1 driedStem2 ]                                
                                
                                }
                                scale 2.5   
                                translate <0, -0.2, 0>
                        }   
                        finish 
                        {
                                phong 0.1 phong_size 10 ambient 0
                        }
             }    
#else
#declare driedFlowerStemTex = texture 
            { 
                        pigment 
                        { 
                                        driedStem1
                        }   
                        finish 
                        {
                                ambient 0
                        }
             }    
#end
             


#if(useTextures)
#declare stigStemTex = texture               
            { 
                        pigment 
                        { 
                                gradient y 
                                color_map 
                                { 
                                        [ 0 (petalColour+White)/2 ] 
                                        [ 1 (petalColour+White) ]                                
                                }
                                scale 1.5    
                                translate <0, -0.2, 0>
                        }   
                        finish 
                        {
                                phong 0.5 phong_size 30 ambient 0 diffuse 0.7 brilliance 0.8 
                        }
             }  
#else
#declare stigStemTex = texture               
            { 
                        pigment 
                        { 
                                        rgb (petalColour+White)/2 
                        }   
                        finish 
                        {
                                ambient 0 diffuse 0.7 brilliance 0.8 
                        }
             }  
#end


#if(useTextures)
#declare driedStigStemTex = texture               
            { 
                        pigment 
                        { 
                                gradient y 
                                color_map 
                                { 
                                        [ 0 (driedPetalColour+White)/4 ] 
                                        [ 1 (driedPetalColour+White)/2 ]                                
                                }
                                scale 1.5    
                                translate <0, -0.2, 0>
                        }   
                        finish 
                        {
                                phong 0.5 phong_size 30 ambient 0 diffuse 0.7 brilliance 0.8 
                        }
             }  
#else
#declare driedStigStemTex = texture               
            { 
                        pigment 
                        { 
                                        rgb (driedPetalColour+White)/4 
                        }   
                        finish 
                        {
                                ambient 0 diffuse 0.7 brilliance 0.8 
                        }
             }  
#end

               
#if(useTextures)
#declare stigTex = texture 
                {
                        pigment
                        {
                                petalColour*0.5
                        }                       
                        finish { ambient 0 phong 0.25 }
                        normal
                        {
                                dents 0.6
                                scale 0.01    
                                turbulence 0.5
                        }  
                }                        
#else
#declare stigTex = texture 
                {
                        pigment
                        {
                                petalColour*0.5
                        }                       
                        finish { ambient 0 }
                }                        
#end


#if(useTextures)
#declare driedStigTex = texture 
                {
                        pigment
                        {
                                driedPetalColour*0.3
                        }                       
                        finish { ambient 0 }
                        normal
                        {
                                dents 0.6
                                scale 0.01    
                                turbulence 0.5
                        }  
                }                                              
#else
#declare driedStigTex = texture 
                {
                        pigment
                        {
                                driedPetalColour*0.3
                        }                       
                        finish { ambient 0 }
                }                                              
#end


#declare stalkGreen1=rgb<0.495,0.75, 0.264>;
#declare stalkGreen2=rgb<0.655,0.775,0.355>;
                     
#if(useTextures)
#declare stalkTex = texture
{
        pigment { radial triangle_wave lambda 4 frequency 30 color_map { [ 0 stalkGreen1 ] [ 1 stalkGreen2 ] } }
        finish { ambient 0 phong 0.15 phong_size 20 }
        normal { average normal_map { [ 1,  wrinkles 0.25  ] [ 0.15, radial frequency 30 ] } }
}
#else
#declare stalkTex = texture
{
        pigment { stalkGreen1 }
        finish { ambient 0 }
}
#end



#if(useTextures)
#declare driedStalkTex = texture
{
        pigment { radial triangle_wave lambda 4 frequency 30 color_map { [ 0 driedLeaf1 ] [ 1 driedLeaf2 ] } }
        finish { ambient 0 phong 0.05 phong_size 5 }
        normal { average normal_map { [ 1,  wrinkles 0.25  ] [ 0.15, radial frequency 30 ] } }
}
#else
#declare driedStalkTex = texture
{
        pigment {driedLeaf1  }
        finish { ambient 0 }
}
#end



#declare stigRand=seed(12415);

#macro stigma(openSize, stigStemTexture, stigTexture)

#local yStep=0.075+(0.05*openSize);
#local yPos = 0;
#local yRand = 0.025;

#local xStep=0.075;
#local xPos = 0;

#local zStep=0.05;
#local zPos = 0;

#local rad=0.0095;
#local radStep = 0.0015;

union
{
// create a curved tube object translating a sphere along a certain path
        sphere_sweep {
            b_spline
            6,
            <-xStep+(rand(stigRand)*xStep*2), -0.5, -zStep+(rand(stigRand)*zStep*2)>,  rad
            < xPos, yPos, zPos>,        rad
            #local xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #local yPos =yPos + yStep+(rand(stigRand)*yStep);
            #local zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #local rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #local xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #local yPos =yPos + yStep+(rand(stigRand)*yStep);
            #local zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #local rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #local xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #local yPos =yPos + yStep+(rand(stigRand)*yStep);
            #local zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #local rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #local exPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #local eyPos =yPos + yStep+(rand(stigRand)*yStep);
            #local ezPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #local rad = rad-(radStep*rand(stigRand));
            < xPos, eyPos, zPos>,        rad
                    
            tolerance 0.1

            texture { stigStemTexture }

        }    
        union
        {         
                #local stigRot=rand(stigRand)*25;
                sphere 
                { 
                        <0, 0, 0> rad*2 scale <1, 4, 1> 
                        rotate <55+stigRot, 0, 0>
                        rotate <zPos*10, 0, -xPos*10>
                        translate <xPos-rad, yPos+rad, zPos>
                }
                sphere 
                { 
                        <0, 0, 0> rad*2 scale <1, 4, 1> 
                        rotate <55+stigRot+rand(stigRand)*2, 0, 0>
                        rotate <zPos*10, 0, -xPos*10>
                        translate <xPos+rad, yPos+rad, zPos>
                }

                texture { stigTexture }
        }  
}
#end    




//==============================================
// MakeBlade macro
//==============================================
// The MakeBlade macro creates a grass blade with a central fold
// The blade is positionned at posBlade and rotated around the y axis according to segBlade
// Its length is lBlade.
// The blade bends from vector startBend to vector endBend
// The  describes how the curbe bends; low power bend faster
// --------------------------------------
// It first creates an array of points (vertices)
// It then calculates the normals (optional)

// petal 1 open petal

#declare doSmooth = true;       // smooth triangles                           
#declare posBlade = <0,0,0>;    // position of blade
#declare rotBlade = 0;        // rotation of blade around y
#declare lBlade = 1;           // length of blade
#declare xBladeStart = 0.095; //.075// width of blade at start
#declare xBladeEnd = 0.025;       // width of blade at the end
#declare zBladeStart = 0.135;     // depth of blade fold at start
#declare startBend = <0,1,0>;   // bending of blade at start (<0,1,0>=no bending)
#declare vBend = <0,1,3>;      // force bending the blade (<0,1,1> = 45)
#declare pwBend = 2;           // bending power (how slowly the curve bends)
#declare dofold = true;        // true creates a fold in the blade (twice more triangles)
#declare dofile = false;        // true creates a mesh file
// --------------------------------------

#declare petal1 = mesh { MakeBlade(doSmooth,posBlade,rotBlade,segBlade,lBlade,xBladeStart,xBladeEnd,zBladeStart,startBend,vBend,pwBend,dofold,dofile)
} 

// petal 2 closed petal
#declare doSmooth = true;       // smooth triangles                           
#declare posBlade = <0,0,0>;    // position of blade
#declare rotBlade = 0;        // rotation of blade around y
#declare lBlade = 0.85;           // length of blade
#declare xBladeStart = 0.065;       // width of blade at start
#declare xBladeEnd = 0.025;       // width of blade at the end
#declare zBladeStart = 0.085;     // depth of blade fold at start
#declare startBend = <0,1,0>;   // bending of blade at start (<0,1,0>=no bending)
#declare vBend = <0,1,1>;      // force bending the blade (<0,1,1> = 45)
#declare pwBend = 3;           // bending power (how slowly the curve bends)
#declare dofold = true;        // true creates a fold in the blade (twice more triangles)
#declare dofile = false;        // true creates a mesh file

#declare petal2 = mesh { MakeBlade(doSmooth,posBlade,rotBlade,segBlade,lBlade,xBladeStart,xBladeEnd,zBladeStart,startBend,vBend,pwBend,dofold,dofile)
}    

// petal 3 wide open petal

#declare doSmooth = true;       // smooth triangles                           
#declare posBlade = <0,0,0>;    // position of blade
#declare rotBlade = 0;        // rotation of blade around y
#declare lBlade = 0.9;           // length of blade
#declare xBladeStart = 0.12; //.09 // width of blade at start
#declare xBladeEnd = 0.025;       // width of blade at the end
#declare zBladeStart = 0.085;     // depth of blade fold at start
#declare startBend = <0,4,2>;   // bending of blade at start (<0,1,0>=no bending)
#declare vBend = <0,3,6>;      // force bending the blade (<0,1,1> = 45)
#declare pwBend = 2;           // bending power (how slowly the curve bends)
#declare dofold = true;        // true creates a fold in the blade (twice more triangles)
#declare dofile = false;        // true creates a mesh file
// --------------------------------------

#declare petal3 = mesh { MakeBlade(doSmooth,posBlade,rotBlade,segBlade,lBlade,xBladeStart,xBladeEnd,zBladeStart,startBend,vBend,pwBend,dofold,dofile)
} 


// leaf1 open leaf
#declare doSmooth = true;       // smooth triangles                           
#declare posBlade = <0,0,0>;    // position of blade
#declare rotBlade = 0;        // rotation of blade around y
#declare lBlade = 0.7;           // length of blade
#declare xBladeStart = 0.150;       // width of blade at start
#declare xBladeEnd = 0.015;       // width of blade at the end
#declare zBladeStart = 0.035;     // depth of blade fold at start
#declare startBend = <0,2,0>;   // bending of blade at start (<0,1,0>=no bending)
#declare vBend = <0,2,5>;      // force bending the blade (<0,1,1> = 45)
#declare pwBend = 2;           // bending power (how slowly the curve bends)
#declare dofold = true;        // true creates a fold in the blade (twice more triangles)
#declare dofile = false;        // true creates a mesh file
// --------------------------------------

#declare leaf1 = mesh { MakeBlade(doSmooth,posBlade,rotBlade,segBlade,lBlade,xBladeStart,xBladeEnd,zBladeStart,startBend,vBend,pwBend,dofold,dofile)
}

// leaf 2 closed leaf 

#declare lBlade = 0.7;           // length of blade
#declare xBladeStart = 0.150;       // width of blade at start
#declare xBladeEnd = 0.015;       // width of blade at the end
#declare zBladeStart = 0.035;     // depth of blade fold at start
#declare startBend = <0,1,0>;   // bending of blade at start (<0,1,0>=no bending)
#declare vBend = <0,2,2>;      // force bending the blade (<0,1,1> = 45)
#declare pwBend = 3;           // bending power (how slowly the curve bends)
#declare dofold = true;        // true creates a fold in the blade (twice more triangles)
#declare dofile = false;        // true creates a mesh file

#declare leaf2 = mesh { MakeBlade(doSmooth,posBlade,rotBlade,segBlade,lBlade,xBladeStart,xBladeEnd,zBladeStart,startBend,vBend,pwBend,dofold,dofile)
} 
    
  
// leaf3 wide  open leaf
#declare doSmooth = true;       // smooth triangles                           
#declare posBlade = <0,0,0>;    // position of blade
#declare rotBlade = 0;        // rotation of blade around y
#declare lBlade = 0.7;           // length of blade
#declare xBladeStart = 0.160;       // width of blade at start
#declare xBladeEnd = 0.015;       // width of blade at the end
#declare zBladeStart = 0.025;     // depth of blade fold at start
#declare startBend = <0,5,3>;   // bending of blade at start (<0,1,0>=no bending)
#declare vBend = <0,4,7>;      // force bending the blade (<0,1,1> = 45)
#declare pwBend = 2;           // bending power (how slowly the curve bends)
#declare dofold = true;        // true creates a fold in the blade (twice more triangles)
#declare dofile = false;        // true creates a mesh file
// --------------------------------------

#declare leaf3 = mesh { MakeBlade(doSmooth,posBlade,rotBlade,segBlade,lBlade,xBladeStart,xBladeEnd,zBladeStart,startBend,vBend,pwBend,dofold,dofile)
}


// flower 1 part open
#macro flower1(stemTexture, leafTexture, petalTexture, stigStemTexture, stigTexture)

union
{   
#declare yStep=2.25;
#declare yPos = 0;
#declare yRand = 0.035;

#declare xStep=0.055;
#declare xPos = 0;

#declare zStep=0.055;
#declare zPos = 0;

#declare rad=0.04;
#declare radStep = 0.008;



        // stem
#if(showStem)
        sphere_sweep {
            b_spline
            6,
            <-xStep+(rand(stigRand)*xStep*2), -0.5, -zStep+(rand(stigRand)*zStep*2)>,  rad
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare yPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare yPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare yPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #declare exPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare eyPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare ezPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, eyPos, zPos>,        rad
                    
            tolerance 0.1
            texture { stemTexture }
        } 
#end
        union
        {
        #if(showPetals)    
                union
                {
                        object { petal1 translate <0, 0, 0.10>  
                        rotate <0, 60, 0>
                        }
                        object { petal1 translate <0, 0, 0.10>  
                        rotate <0, 120, 0>
                        }
                        object { petal1 translate <0, 0, 0.10>  
                        rotate <0, 180, 0>
                        }
                        object { petal1 translate <0, 0, 0.10>  
                        rotate <0, 240, 0>
                        }
                        object { petal1 translate <0, 0, 0.10>  
                        rotate <0, 300, 0>
                        }
                        object { petal1 translate <0, 0, 0.10>  
                        }
                        texture { petalTexture }
                }
        #end
        #if(showStigmata)    
                union
                {
                        object { stigma(0.6, stigStemTexture, stigTexture) scale 1.5 translate <0, 0.23, 0> rotate <10, 60, 0> }
                        object { stigma(0.4, stigStemTexture, stigTexture) scale 1.45 translate <0, 0.23, 0> rotate <10, 120, 0> }
                        object { stigma(0.9, stigStemTexture, stigTexture) scale 1.52 translate <0, 0.23, 0> rotate <10, 180, 0> }
                        object { stigma(0.65, stigStemTexture, stigTexture) scale 1.4 translate <0, 0.23, 0> rotate <10, 240, 0> }
                        object { stigma(0.52, stigStemTexture, stigTexture) scale 1.51 translate <0, 0.23, 0> rotate <10, 300, 0> }
                        object { stigma(0.55, stigStemTexture, stigTexture) scale 1.48 translate <0, 0.23, 0> rotate <10, 0, 0> }
                        texture { stigTexture }
                }
        #end

        #if(showLeaves) 
        union
        {   
                object { leaf1 translate <0, 0, 0.175> rotate <0, 30, 0> }
                object { leaf1 translate <0, 0, 0.175> rotate <0, 90, 0> }
                object { leaf1 translate <0, 0, 0.175> rotate <0, 150, 0> }
                object { leaf1 translate <0, 0, 0.175> rotate <0, 210, 0> }
                object { leaf1 translate <0, 0, 0.175> rotate <0, 270, 0> }
                object { leaf1 translate <0, 0, 0.175> rotate <0, 330, 0> }
                sphere 
                { 
                        <0, 0, 0> 0.235 
                        texture
                        {       
                                average
                                texture_map
                                {      
                                        [ 1 leafTexture ]
                                        [ 0.5 petalTexture ]                        
                                }
                }       }
                texture { leafTexture }
        }
        #end
                translate <xPos, yPos, zPos>
        }   

}   
#end    

// flower 2 closed
#macro flower2(stemTexture, leafTexture, petalTexture, stigStemTexture, stigTexture)

union
{   
#declare yStep=2.25;
#declare yPos = 0;
#declare yRand = 0.035;

#declare xStep=0.055;
#declare xPos = 0;

#declare zStep=0.055;
#declare zPos = 0;

#declare rad=0.04;
#declare radStep = 0.008;



        // stem
#if(showStem)
        sphere_sweep {
            b_spline
            6,
            <-xStep+(rand(stigRand)*xStep*2), -0.5, -zStep+(rand(stigRand)*zStep*2)>,  rad
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare yPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare yPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare yPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #declare exPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare eyPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare ezPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, eyPos, zPos>,        rad
                    
            tolerance 0.1
            texture { stemTexture }
        } 
#end
        union
        {
        #if(showPetals)    
                union
                {
                        object { petal2 translate <0, 0, 0.15>  
                        rotate <0, 60, 0>
                        }
                        object { petal2 translate <0, 0, 0.15>  
                        rotate <0, 120, 0>
                        }
                        object { petal2 translate <0, 0, 0.15>  
                        rotate <0, 180, 0>
                        }
                        object { petal2 translate <0, 0, 0.15>  
                        rotate <0, 240, 0>
                        }
                        object { petal2 translate <0, 0, 0.15>  
                        rotate <0, 300, 0>
                        }
                        object { petal2 translate <0, 0, 0.15>  
                        }
                        texture { petalTexture }
                }
        #end
        #if(showStigmata)    
                union
                {
                        object { stigma(0.9, stigStemTexture, stigTexture) scale 1.5 translate <0, 0.23, 0> rotate <5, 60, 0> }
                        object { stigma(0.9, stigStemTexture, stigTexture) scale 1.45 translate <0, 0.23, 0> rotate <5, 120, 0> }
                        object { stigma(0.8, stigStemTexture, stigTexture) scale 1.52 translate <0, 0.23, 0> rotate <5, 180, 0> }
                        object { stigma(0.85, stigStemTexture, stigTexture) scale 1.4 translate <0, 0.23, 0> rotate <5, 240, 0> }
                        object { stigma(0.72, stigStemTexture, stigTexture) scale 1.51 translate <0, 0.23, 0> rotate <5, 300, 0> }
                        object { stigma(0.85, stigStemTexture, stigTexture) scale 1.48 translate <0, 0.23, 0> rotate <5, 0, 0> }
                        texture { stigTexture }
                }
        #end

        #if(showLeaves) 
        union
        {   
                object { leaf2 translate <0, 0, 0.175> rotate <0, 30, 0> }
                object { leaf2 translate <0, 0, 0.175> rotate <0, 90, 0> }
                object { leaf2 translate <0, 0, 0.175> rotate <0, 150, 0> }
                object { leaf2 translate <0, 0, 0.175> rotate <0, 210, 0> }
                object { leaf2 translate <0, 0, 0.175> rotate <0, 270, 0> }
                object { leaf2 translate <0, 0, 0.175> rotate <0, 330, 0> }
                sphere { <0, 0, 0> 0.235 }
                texture { leafTexture }
        }
        #end
                translate <xPos, yPos, zPos>
        }   

}   
#end

// flower 3 open
#macro flower3(stemTexture, leafTexture, petalTexture, stigStemTexture, stigTexture)

union
{   
#declare yStep=2.25;
#declare yPos = 0;
#declare yRand = 0.035;

#declare xStep=0.055;
#declare xPos = 0;

#declare zStep=0.055;
#declare zPos = 0;

#declare rad=0.04;
#declare radStep = 0.008;



        // stem
#if(showStem)
        sphere_sweep {
            b_spline
            6,
            <-xStep+(rand(stigRand)*xStep*2), -0.5, -zStep+(rand(stigRand)*zStep*2)>,  rad
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare yPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare yPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare yPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare zPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, yPos, zPos>,        rad
            #declare exPos =xPos + (-xStep+(rand(stigRand)*xStep*2));
            #declare eyPos =yPos + yStep+(rand(stigRand)*yStep/10);
            #declare ezPos =zPos + (-zStep+(rand(stigRand)*zStep*2));
            #declare rad = rad-(radStep*rand(stigRand));
            < xPos, eyPos, zPos>,        rad
                    
            tolerance 0.1
            texture { stemTexture }
        } 
#end
        union
        {
        #if(showPetals)    
                union
                {
                        object { petal3 translate <0, 0, 0.11>  
                        rotate <0, 60, 0>
                        }
                        object { petal3 translate <0, 0, 0.11>  
                        rotate <0, 120, 0>
                        }
                        object { petal3 translate <0, 0, 0.11>  
                        rotate <0, 180, 0>
                        }
                        object { petal3 translate <0, 0, 0.11>  
                        rotate <0, 240, 0>
                        }
                        object { petal3 translate <0, 0, 0.11>  
                        rotate <0, 300, 0>
                        }
                        object { petal3 translate <0, 0, 0.11>  
                        }
                        texture { petalTexture }
                }
        #end
        #if(showStigmata)    
                union
                {
                        object { stigma(0.2, stigStemTexture, stigTexture) scale 1.5 translate <0, 0.23, 0> rotate <25, 60, 0> }
                        object { stigma(0.4, stigStemTexture, stigTexture) scale 1.45 translate <0, 0.23, 0> rotate <25, 120, 0> }
                        object { stigma(0.1, stigStemTexture, stigTexture) scale 1.52 translate <0, 0.23, 0> rotate <25, 180, 0> }
                        object { stigma(0.25, stigStemTexture, stigTexture) scale 1.4 translate <0, 0.23, 0> rotate <25, 240, 0> }
                        object { stigma(0.42, stigStemTexture, stigTexture) scale 1.51 translate <0, 0.23, 0> rotate <25, 300, 0> }
                        object { stigma(0.35, stigStemTexture, stigTexture) scale 1.48 translate <0, 0.23, 0> rotate <25, 0, 0> }
                        texture { stigTexture }
                }
        #end

        #if(showLeaves) 
        union
        {   
                object { leaf3 translate <0, 0, 0.175> rotate <0, 30, 0> }
                object { leaf3 translate <0, 0, 0.175> rotate <0, 90, 0> }
                object { leaf3 translate <0, 0, 0.175> rotate <0, 150, 0> }
                object { leaf3 translate <0, 0, 0.175> rotate <0, 210, 0> }
                object { leaf3 translate <0, 0, 0.175> rotate <0, 270, 0> }
                object { leaf3 translate <0, 0, 0.175> rotate <0, 330, 0> }
                sphere { <0, 0, 0> 0.235 }
                texture { leafTexture }
        }
        #end
                translate <xPos, yPos, zPos>
        }   

}   
#end    





#macro Alium(flStemTex, flleafTex, flpetalTex, flstigStemTex, flstigTex, flstalkTex maxCount, missingThreshold)
#declare flowerArray = array[5];
                                           
#declare flowerArray[0] = object { flower1(flStemTex, flleafTex, flpetalTex, flstigStemTex, flstigTex) };
#declare flowerArray[1] = object { flower2(flStemTex, flleafTex, flpetalTex, flstigStemTex, flstigTex) };
#declare flowerArray[2] = object { flower1(flStemTex, flleafTex, flpetalTex, flstigStemTex, flstigTex) };
#declare flowerArray[3] = object { flower3(flStemTex, flleafTex, flpetalTex, flstigStemTex, flstigTex) };
#declare flowerArray[4] = object { flower1(flStemTex, flleafTex, flpetalTex, flstigStemTex, flstigTex) };


#declare fcount = 0;

#declare frand = seed(2435);

    
union
{
        // stalk     
#declare yStep=25;
#declare yPos = 0;

#declare xStep=0.6;
#declare xPos = 0;

#declare zStep=0.4;
#declare zPos = 0;

#declare rad=0.6;
#declare radStep = 0.09;

// create a curved tube object translating a sphere along a certain path
union
{
        sphere_sweep {
            linear_spline
            4,
            //<-xStep+(xStep*2), -0.5, -zStep+(zStep*2)>,  rad
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + -xStep+(xStep*2);
            #declare yPos =yPos + yStep+(yStep/10);
            #declare zPos =zPos + -zStep+(zStep*2);
            #declare rad = rad-radStep;
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + -xStep+(xStep*2);
            #declare yPos =yPos + yStep+(yStep/10);
            #declare zPos =zPos + -zStep+(zStep*2);
            #declare rad = rad-radStep;
            < xPos, yPos, zPos>,        rad
            #declare xPos =xPos + -xStep+(xStep*2);
            #declare yPos =yPos + yStep+(yStep/10);
            #declare zPos =zPos + -zStep+(zStep*2);
            #declare rad = rad-(radStep/2);
            < xPos, yPos, zPos>,        rad
            #declare exPos =xPos + -xStep+(xStep*2);
            #declare eyPos =yPos + yStep+(yStep/10);
            #declare ezPos =zPos + -zStep+(zStep*2);
            #declare rad = rad+radStep;
            //< xPos, eyPos, zPos>,        rad
                    
            tolerance 0.01
            
            }
            sphere { <0, 0, 0> 0.675 scale <1, 1.65, 1> translate <xPos, yPos, zPos> }
            texture 
            {
                flstalkTex
            } 
        }     



union
{
#declare minCount=2;  
#declare zr = 0;
#declare zstep = 90/(maxCount*0.5);   
#declare numfStep = (maxCount-minCount)/(maxCount*0.5);    
#declare flCount = minCount;

#while(zr < 160) 

        #declare yr=0;
        #declare ystep = 360/flCount;  
        #while(yr < 360)
                #declare flowerNum = rand(frand)*5;
                #if(gr(1) > missingThreshold)
                        object { flowerArray[flowerNum] rotate <0, gr(360), 0> translate <0, 0.5, 0> rotate <0, 0, zr-gr(zStep/2)+gr(zStep/2)> rotate <0, yr-gr(yStep/2)+gr(yStep/2), 0> }
                #end
                #declare yr=yr+ystep;
        #end  
        #if(zr > 90)
                #declare flCount=flCount-numfStep;          
        #else                       
                #declare flCount=flCount+numfStep;  
        #end
        #declare zr=zr+zstep;
#end    
        translate <xPos, yPos, zPos>
}

}
#end

//object { Alium(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex, 45, 0.15) translate <-10, 0, 5> }
//object { Alium(driedFlowerStemTex, driedLeafTex, driedPetalTex, driedStigStemTex, driedStigTex, 40, 0.35) translate <10, 0, 5> }





/*
object { flower1(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex) rotate <0, 0, 0> translate <0, -4, -25> } 
object { flower2(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex) rotate <0, 0, 0> translate <-2.5, -4, -25> }
object { flower3(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex) rotate <0, 0, 0> translate <2.5, -4, -25> }

object { flower1(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex) rotate <-45, 0, 0> translate <0, -4, -21> } 
object { flower2(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex) rotate <-45, 0, 0> translate <-2.5, -4, -21> }
object { flower3(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex) rotate <-45, 0, 0> translate <2.5, -4, -21> }

object { flower1(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex) rotate <-90, 0, 0> translate <0, -1, -17> } 
object { flower2(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex) rotate <-90, 0, 0> translate <-2.5, -1, -17> }
object { flower3(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex) rotate <-90, 0, 0> translate <2.5, -1, -17> }

  */

// vase

union
{       
        intersection
        {
                sphere { <0, 0, 0> 6 scale <1, 10, 1> }  
                box { <-9, -40, -9> <9, 0, 9> } 
        }                               
        difference
        {
                torus { 4, 2 scale <1, 8, 1> }
                torus { 3.5, 2 scale <1, 8, 1> }
        }    
        torus { 4.3, 0.175 translate <0, -40, 0>  }
        #if(useTextures)
        texture
        {
                gradient y 
                lambda 4
                omega 0.65
                turbulence 0.15
                
                texture_map
                {
                        [ 0.0
                        pigment 
                        { 
                                granite 
                                colour_map
                                {
                                        [ 0.0 rgb <0.7, 0.6, 0.4> ]
                                        [ 1.0 rgb <0.8, 0.7, 0.5> ]  
                                }                                     
                                scale 1
                        }
                        finish { ambient 0 specular 0.5 roughness 0.075 }
                        normal { granite 0.45 scale 0.05 }
                        ]

                        [ 1.0  
                        pigment 
                        { 
                                granite 
                                colour_map
                                {
                                        [ 0.0 rgb <0.9, 0.9, 0.8> ]
                                        [ 1.0 rgb <1, 0.9, 0.7> ]  
                                }            
                                scale 1
                        }
                        finish { ambient 0 specular 0.3 roughness 0.1 }
                        normal { granite 0.5 scale 0.05 }
                        ]     
                }
                scale 80
                translate <0, -40, 0>
        }
        #else
        texture
        {
                pigment 
                { 
                   rgb <0.7, 0.6, 0.4> 
                }
                finish { ambient 0 }
        }
        #end

         
        translate <0, 40.2, 0> 
}   





#declare SkirtingBoard = union
{       
        
        object { Round_Box_Union(<-1, -1, -0.2>, <1, 1, 0.2>, 0.15) }

        difference 
        {
                box { <-1, 1, -0.2> <1, 1.5, 0.2> }
                cylinder { <-1.1, 1.35, -0.2> <1.1, 1.35, -0.2> 0.25 scale <1, 1, 0.75> }
        }  
        cylinder { <-1, 1.05, -0.2> <1, 1.05, -0.2> 0.05 }
        
        box { <-1, 1.5, 0> <1, 1.7, 0.2> }   
        cylinder { <-1, 1.55, 0> <1, 1.55, 0> 0.05 }
        cylinder { <-1, 1.65, 0> <1, 1.65, 0> 0.05 }
}  

#declare MyPineYellow = colour rgb <1, 0.925, 0.45>;
#declare MyPineOrange = colour rgb <0.74, 0.37, 0>;
#declare MyPineBrown = colour rgb <0.22, 0.0995, 0>;  
   

#declare MyPine = texture 
{ 
        pigment 
        {       
                ripples  
                turbulence 0.75
                //frequency 2  
                octaves 6
                lambda 2
                omega 2   
                colour_map
                { 
                        [ 0.0 MyPineYellow ]
                        [ 1.001 MyPineOrange ]
                }                  
                scale 50
                rotate <30, 40, 20>
        } 
        finish { ambient 0 }         
} 
texture
{
        pigment
        {
                bozo       
                turbulence 0.345   
                colour_map
                {
                        [ 0.0 MyPineYellow+<0, 0, 0, 0.2, 0.3> ] 
                        [ 0.67 MyPineOrange+<0, 0, 0, 0.2, 0.6> ]
                        [ 1.001 MyPineBrown+<0, 0, 0,0.2, 0.8> ]
                }
        }             
        finish { ambient 0 }
}        
texture
{
        pigment
        {
                wood       
                turbulence <0.004, 0.0035, 0.003>
                octaves 10
                lambda 4.5  
                omega 0.6
                
                colour_map
                {
                        [ 0.0   MyPineYellow+<0, 0, 0, 0.0, 0.5> ] 
                        [ 0.85  MyPineOrange+<0, 0, 0, 0.0, 0.4> ]
                        [ 1.001 MyPineBrown+ <0, 0, 0, 0.0, 0.4> ]
                }     
                scale <0.5, 0.45, 0.3>                 
                rotate <1, 4, 0>
        }     
        finish { ambient 0 }
}        
texture
{
        pigment
        {
                wood                               

                turbulence <0.0114, 0.0185, 0.0148>
                octaves 10
                lambda 8  
                omega 0.656
                
                colour_map
                {
                        [ 0.0   MyPineYellow+<0, 0, 0, 0.0, 0.2> ] 
                        [ 0.3   MyPineYellow+<0, 0, 0, 0.0, 0.7> ] 
                        [ 0.35  MyPineOrange+<0, 0, 0, 0.0, 0.8> ]
                        [ 0.41   MyPineYellow+<0, 0, 0, 0.0, 0.5> ] 
                        [ 0.9   MyPineYellow+<0, 0, 0, 0.0, 0.9> ] 
                        [ 0.975  MyPineOrange+<0, 0, 0, 0.0, 0.7> ]
                        [ 1.0 MyPineBrown+<0, 0, 0, 0.0, 0.5> ]
                }     
                scale <0.25, 0.35, 0.3>      
                rotate <1, 0, 0>   
                translate <0.1, 0.1, 0.1>        
        }            
        finish { ambient 0 }
}
texture
{
        pigment
        {
                wood                               

                turbulence <0.0114, 0.0185, 0.0148>
                octaves 10
                lambda 8  
                omega 0.656
                
                colour_map
                {
                        [ 0.0   MyPineYellow+<0, 0, 0, 0.0, 0.2> ] 
                        [ 1.0   MyPineYellow+<0, 0, 0, 0.0, 0.95> ] 
                }     
                scale <0.075, 0.35, 0.3>      
                rotate <1, 0, 0>   
                translate <0.1, 0.1, 0.1>        
        }    
        finish { ambient 0 phong 0.2 phong_size 10}     

        normal
        {              
                average
                normal_map
                {
                        [ 0.6, wrinkles 0.12 turbulence 0.115 scale <1, 0.5, 5> ]
                        [ 0.4, ripples 0.15 turbulence 0.75 octaves 6 lambda 2 omega 2 scale 80 rotate <30, 40, 20> ]
                }
                scale <0.8, 1, 2.5>
        }          

} 



#declare skDirt =  texture { pigment { gradient y triangle_wave omega 0.55 turbulence 0.0875 color_map 
                        { 
                                [ 0.0 rgbt <0.085, 0.025, 0.007, 0.25> ]   
                                [ 0.285 rgbt <0.125, 0.025, 0.007, 0.8> ]   
                                [ 0.395 rgbt <0.225, 0.1, 0.01, 0.9> ]     
                                [ 0.395 rgbt <1, 1, 1, 1> ]     
                                [ 1 rgbt <1, 1, 1, 1> ]  
                        }
                        scale <8, 8, 1> 
                        translate <0, -2.5, 0>
                        } 
                        normal
                        {              
                                average
                                normal_map
                                {
                                        [ 0.5, wrinkles 1 turbulence 0.115 scale <1, 0.5, 5> ]
                                        [ 0.5, ripples 1 turbulence 0.75 octaves 6 lambda 2 omega 2 scale 80 rotate <30, 40, 20> ]
                                        [ 1.0 wood 1 rotate <0, 87, 0> scale <2, 0.35, 1> turbulence <0.0114, 0.0185, 0.0148> octaves 10 lambda 4 ]
                                }                     
                                scale <10,5, 5>
                        }  
                        finish { ambient 0 diffuse 0.4 }
                }          
 
// add Skirting board
object 
{ 
        SkirtingBoard scale <300, 5, 5> 
        #if(useTextures)
                texture { MyPine rotate <0, 50, 80> scale <80, 1, 1> scale 5  } 
                texture { skDirt translate <0, 2.5, 0> scale 5 translate <0, -6.5, 0> }                
        #else
                texture { pigment { MyPineBrown } finish { ambient 0 } }
        #end


        translate <0, 5.6, 18.0> 
}      





//object { Alium(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex, stalkTex, numFlowers, 0.15) rotate <10, 0, 0> translate <0, 5, 0> }
object { Alium(driedFlowerStemTex, driedLeafTex, driedPetalTex, driedStigStemTex, driedStigTex, driedStalkTex, numFlowers*0.825, 0.375) rotate <9, 70, 0> translate <-4.0, 16, 0> }
object { Alium(driedFlowerStemTex, driedLeafTex, driedPetalTex, driedStigStemTex, driedStigTex, driedStalkTex, numFlowers*0.8, 0.3) rotate <8, -90, 0> translate <4, 10, 0> }
object { Alium(driedFlowerStemTex, driedLeafTex, driedPetalTex, driedStigStemTex, driedStigTex, driedStalkTex, numFlowers*0.75, 0.325) rotate <-6.5, -20, 0> translate <-1, 3, 5> }
object { Alium(driedFlowerStemTex, driedLeafTex, driedPetalTex, driedStigStemTex, driedStigTex, driedStalkTex, numFlowers*0.85, 0.4) rotate <7.5, -25, 0> translate <0.5, 9, -4> }
object { Alium(driedFlowerStemTex, driedLeafTex, driedPetalTex, driedStigStemTex, driedStigTex, driedStalkTex, numFlowers*0.825, 0.35) rotate <-7, 40, 0> translate <1, 0, 6> }
object { Alium(driedFlowerStemTex, driedLeafTex, driedPetalTex, driedStigStemTex, driedStigTex, driedStalkTex, numFlowers*0.8, 0.35) rotate <6, -40, 0> translate <-0.5, 22, 3> }
object { Alium(flowerStemTex, leafTex, petalTex, stigStemTex, stigTex, stalkTex, numFlowers, 0.2) rotate <-2, -20, 0> translate <0, 18, 0> }



// floor
//box { <-200, -2, -200> <200, -0.1, 50> pigment { White } finish { ambient 0 } }   

#declare woodF=function{pigment{
  wood 
  turbulence <0.0125, 0.04375, 0.1> 
  lambda 5
  octaves 10
  omega 0.525
  color_map { [0 rgb 1] [0.2 rgb 0 ] [0.3 rgb 1] [0.35 rgb 0] [ 0.6 rgb 1 ] [1 rgb 0] }
  rotate <-1, 1, 0>     
  scale <1, 1, 2>   

warp {
    black_hole <-15, 0, 0>,3
    falloff 1.75
    strength 3
    repeat <15, 0.75, 35>
    turbulence <9, 0.5, 49>
    inverse
  }
  warp {
    black_hole <0, 0, 0>,2
    falloff 1.75
    strength 3
    repeat <1, 0.75, 35>
    turbulence <1, 0.5, 22>
    inverse
  }

/*
  warp {
//    black_hole <-12, 2, -146>, 3
    black_hole <0,0, 0>, 4
    falloff 1.5
    strength 2
    repeat <1,1,555> 
   // turbulence <0.7, 0.35, 0.95>
    inverse
  }
*/
  }
}    



#declare whitePaintTex = texture
{
        pigment { color rgb <1, 0.98, 0.98> }
        finish { phong 0.7 phong_size 100 brilliance 0.9 ambient 0 diffuse 0.7 reflection { 0.025, 0.05 fresnel on } conserve_energy }
        
}   
texture         // Add dirt
{                          
        pigment
        {
                bozo
                
                octaves 10           
                lambda 4
                omega 0.5975
                turbulence 0.275+gr(0.1)
                color_map
                {
                        [ 0.0  rgbt <0.07, 0.04, 0.02, 0.30> ]
                        [ 0.05 rgbt <0.07, 0.04, 0.02, 0.65> ]
                        [ 0.10 rgbt <0.07, 0.04, 0.02, 0.8> ]
                        [ 0.15 rgbt <0.07, 0.04, 0.02, 1.0> ]
                        [ 0.20 rgbt <1, 1, 1, 1> ]
                        [ 1.0  rgbt <1, 1, 1, 1> ]
                }
                    // was 0.02 and 0.002
        }  
        finish { ambient 0 diffuse 0.3 }
        scale <3, 2, 5>
}
texture         // Add dirt
{                          
        pigment
        {
                wrinkles
                scallop_wave
                octaves 10
                lambda 4
                omega 0.6275
                turbulence 0.275+gr(0.1)
                color_map
                {
                        [ 0.0  rgbt <0.07, 0.04, 0.02, 0.30> ]
                        [ 0.05 rgbt <0.07, 0.04, 0.02, 0.55> ]
                        [ 0.11 rgbt <0.07, 0.04, 0.02, 0.7> ]
                        [ 1.01 rgbt <0.07, 0.04, 0.02, 1> ]
                }
                        // was 0.05 and 0.02
        }     
        finish { ambient 0 diffuse 0.3 }  

        scale <1, 2, 4>
}









#declare oldWoodTex = texture
{                
        pigment
        {
                wrinkles
                turbulence 0.275
                octaves 10
                lambda 3
                omega 0.6
                color_map        
                {
                        [ 0.0 rgb <0.12, 0.05, 0.05>*2 ]
                        [ 1.0 rgb <0.325, 0.125, 0.05>*2 ]
                }                                         
                scale 2
        }
        finish { phong 0.1 phong_size 10 ambient 0.0 }
}                    
texture
{
        pigment
        {
                wood  
                turbulence <0.00575, 0.1, 0.2> 

                octaves 10
                lambda 2.5
                omega 0.65

                color_map        
                {
                        [ 0.0 rgbt <0.275, 0.125, 0.05, 1.5>/2 ]
                        [ 0.45 rgbt <0.275, 0.125, 0.05, 0.7> ]
                        [ 1.01 rgbt <0.275, 0.125, 0.05, 0.4>/2 ]
                }            
                scale <0.133, 0.45, 0.5>
                rotate <4, 0, 0>
                warp{black_hole <0.1, 0.1, 0.1>, 0.6 strength 4 inverse repeat <2, 1, 6> falloff 2  turbulence <0.24, 0.8, 0.5> }
                warp{black_hole <1.2, 1.0, 1.8>, 0.4 strength 2.5 inverse repeat <3, 1, 8> falloff 1  turbulence <0.4, 0.5, 0.35> }
        }
        finish { phong 0.1 phong_size 10 ambient 0.0 }
}   
   


#declare xp = -215;

union
{

#while(xp < 200)

#undef woodF
#declare woodF=function{pigment{
  wood 
  turbulence <0.0125, 0.04375, 0.1> 
  lambda 5
  octaves 10
  omega 0.525
  color_map { [0 rgb 1] [0.2 rgb 0 ] [0.3 rgb 1] [0.35 rgb 0] [ 0.6 rgb 1 ] [1 rgb 0] }
  rotate <1, 1, 0>     
  scale <1, 1, 20>   

warp {
    black_hole <-15, 0, 0>,3
    falloff 1.75
    strength 3
    repeat <15, 0.75, 65>
    turbulence <9, 0.5, 42+gr(15)>
    inverse
  }
  warp {
    black_hole <0, 0, 0>,2
    falloff 1.75
    strength 3
    repeat <4, 0.75, 55>
    turbulence <1+gr(1), 0.5, 22+gr(8)>
    inverse
  }
}
}  
#if(useGeo)
union
{
        height_field
        {
                function 500,500 
                {
                        woodF(x*15, y, z*15).gray
        
                } 
                smooth  
                scale <30.0, 0.125, 100.0>
                translate <-15.0, 2, -50.0>
        }
        box { <-15.0, -2.1, -50.0> <15.0, 2, 50.0> }
        translate <xp, -2.1, -30>
}
/*isosurface {
  
  function { f_rounded_box(x,y,z, 0.25, 15, 2, 50)-woodF(x,y,z).gray*0.1}          // function (can also contain declared functions
  contained_by { box { <-15.11, -2.11, -50.11> <15.11, 2.11, 50.11> } }  // container shape
  //threshold 0.0                     // optional threshold value for isosurface [0.0]
  accuracy 0.001                      // accuracy of calculation [0.001]
  max_gradient 4                      // maximum gradient the function can have [1.1]

  translate <xp, -2.2, -30>  
} 
*/ 
#else
box { <-15.0, -2.1, -50.0> <15.0, 2.1, 50.0> translate <xp, -2.2, -30>} 
#end
                           
#declare xp=xp+30.4; 
#end

#if(useTextures)
        texture
        {       
                gradient y    
                lambda 3.5
                omega 0.55
                turbulence <0.0375, 0.325, 0.00725>
                texture_map
                {                               

                        [ 0.0 whitePaintTex ]
                        [ 0.95 whitePaintTex ]
                        [ 0.96  oldWoodTex ]
                        [ 1.0  oldWoodTex ]
                }      
                scale <4, 1, 15>     
                translate <0, -0.95, 0>
        }
#else
        texture { pigment { rgb <1, 0.95, 0.95> } finish { ambient 0 } }
#end

}        

#if(useGeo)
height_field{
    function 1000,1000 {
        f_granite(x, y, z)
//        (sin(y*pi*5)*0.5+sin(x*pi*4)*0.5+1)*0.4+f_ridged_mf(x*7, y*15,z, 1.2,3,6,1.18,5,1)*0.54*0.2
        
    }      
    smooth
    scale <400, 0.5, 400>
    rotate <-90, 0, 0>
    translate <-200, -100, 20>
#else
box { <-200, -100, 20> <200, 300, 25> 

#end 
#if(useTextures)
    texture
    {
        pigment { rgb <1, 0.925, 0.85> }
        finish { ambient 0 diffuse 0.7 brilliance 0.9 }   
    }  
    texture // add dirt 
    {                  
        pigment 
        {
                gradient y 
                triangle_wave     
                lambda 3
                omega 0.625
                turbulence 0.0275
                color_map
                {
                        [ 0.0 rgbt <0.35, 0.2, 0.075, 0.5> ]
                        [ 0.065 rgbt <0.35, 0.2, 0.075, 0.75> ]
                        [ 0.2 rgbt <1, 1, 1, 1> ]
                        [ 1 rgbt <1, 1, 1, 1> ]                      
                        
                }     
                scale <1500, 500, 300>
                translate <0, 8, 0>
        }
        finish { ambient 0 diffuse 0.5 }
    }
#else
    texture
    {
        pigment { rgb <1, 0.925, 0.85> }
        finish { ambient 0 diffuse 0.7 brilliance 0.9 }   
    }  
#end

}


