// This is aquarium.pov [Revision L]
//
// Copyright 2005 Windell Oskay
// ( windell (at) oskay.net )
//
// I claim copyright solely for the purpose of being able to claim 
// that *I too* have the right to use the contents of this scene.  
//
// So... please let me know if you reuse any part of this file.
//
// To render this file: First render label.pov and labelFF.pov,
// if necessary, to generate the three image map files label.png,
// labelFF1.png, and labelFF2.png (the latter two are generated by
// rendering labelFF.pov with two frames of animation.  
//
// In this file I have made extensive use of included POV-Ray templates,
// and I have used code from a number of the POV-Ray sample files as 
// starting points for various parts of the scene.  In particular, 
// the distribution of rocks on the ground from isocacti.pov has been
// used to make the distribution of spilled fish food on the counter top.
//



/**********BEGIN aquarium.pov***************/


#version 3.6;
#include "functions.inc"
#include "arrays.inc"
#include "stdinc.inc"

//Control options that determine rendering time:

#declare Radiosity=on;		//Diffuse lighting (radiosity)
#declare Photons=on;		//Photon mapping
#declare PhocalBlur=on;		//Focal blur

/*These major options really affect the rendering time.  Here are some
examples, rendering a small image, with all other factors constant:
 All (Focal, Photon, Rad) off: 176 s
 Focal On: 965
 Focal, Photon on: 1032 s 
 Focal, Rad on: 2118
 */
 
// Control options that determine which parts of the scene to render:
// [If you want to examine individual objects, it's helpful to turn
// off many of the other objects.]
 
 
#declare foodCan = on;		//Fish food can
#declare showTrees = on;	//Trees outside window
#declare cordOn = on;		//blender power cord
#declare SocketOn = on; 	//Power socket on wall

#declare ShowFish=on; 		//Goldfish
#declare showWater = on;	//Water inside pitcher

#declare blenderOn=on; //Show blender
/*The following seven commands only are applicable if blenderOn=on
they refer to parts of the blender or to rendering options for the 
blender. Note that the cord, goldfish, and water are NOT part of the
blender, so that they can still be drawn without the blender present.*/
#declare showLid = on;		
#declare showDrive = on;	//Blades, etc.
#declare showBody = on; 	//glass body of pitcher
#declare showBase = on;		//ring around base of pitcher
#declare showMotor = on;	//Beehive base for blender
#declare cutOn = off;		//cut-away pitcher, (etc) to see insides
#declare glassOn = on; 		//Turn off to render glass pitcher with






/************ Global Settings **************/

global_settings {
  assumed_gamma 1.0
  max_trace_level 20
  #if (Photons)          // global photon block
    photons {
      spacing 0.1                 // specify the density of photons
    }
  #end
  
   #if (Radiosity)			
   
    radiosity {				 
     pretrace_start 0.08
    pretrace_end   0.02
    count 150
    error_bound 0.5		//0.01 looks great-- damned slow.
    recursion_limit 2  // how much interreflections are calculated (1..5+) [3]
     brightness 5/2 // brightness of radiosity effects (0..1) [1]
    }
    
   #else 
     ambient_light rgb 0.2
  #end
  
}

/************ BEGIN Textures and Materials **************/

// Textures & Materials
  
#declare PolyCarbonate = material {
  texture {
    pigment {rgbt 0.8}
    normal {bumps 0.005 scale < 1, 1/50, 1>}
    finish {
      ambient 0.001
      diffuse 0.05
      specular 0.1 
      reflection {
        0.0001, 0.10
        fresnel on
      }
      conserve_energy
    } 
  }
  interior {
    ior 1.3 
  }   
}

//Two types of glass-- the only difference is the normal

#declare GlassTex = material {	//Glass for pitcher of blender
  texture {
    pigment {rgbf <0.98,1,0.98,1>}
    normal {dents 0.01 poly_wave 2 scale < 1, 0.2, 1>}
    finish {
      ambient 0.001
      diffuse 0.01
      specular 0.04 
      reflection {
        0.0001, 0.15
        fresnel on
      }
      conserve_energy
    } 
  }
  interior {
    ior 1.4
    fade_power 1
    fade_distance 40
    #if (Radiosity=off)
      caustics 1
    #end //  end "#if (Radiosity=off)"
  }   
}

#declare GlassTex2 = material {		//Window pane glass
  texture {
    pigment {rgbf <0.98,1,0.98,1>}

    normal {dents 0.01 poly_wave 2 scale < 1, 1, 1>}
    finish {
      ambient 0.001
      diffuse 0.01
      specular 0.04 
      reflection {
        0.0001, 0.15
        fresnel on
      }
      conserve_energy
    } 
  }
  interior {
    ior 1.4
    fade_power 1
    fade_distance 40
    #if (Radiosity=off)
      caustics 1
    #end //  end "#if (Radiosity=off)"
  }   
}


#declare M_WhitePlastic = 
material{texture { 
	pigment { color rgb 1 } 
	finish {   
	#if (Radiosity= 0)
	  ambient 1 
	#end
    		diffuse 0.55 
    		specular 1
    		roughness 0.01
    		reflection 0.05  
    		  }      	}
}

//One of the quick-render textures
#declare GreenTex = texture { pigment  {color rgb < 0.3, 0.85, 0.4> 	}
		normal {dents 0.1 scale 1/10}}

//Plasticy finish
#declare plasFin  =  finish {
      ambient 0.01
      diffuse 0.50
      specular 0.5 
      reflection { 0.01, 1.0 fresnel on}
       }

#declare M_RedPlastic=   //Lid of fish food can...
material {
  texture {
   pigment { color rgb <1.0, 0.0, 0.0> }
       normal {bumps 0.1  scale 1}
 finish {
      ambient 0
      diffuse 0.50
      specular 0.8 
      reflection { 0.01, 0.1 }
       }
  }
}

#declare M_YellowPlastic=    // Body of fish food can...
material {
  texture {
   pigment { color rgb <1.0, 1, 0.0> }
       normal {bumps 0.1  scale 1/10}
    finish { plasFin}
  }
}

#declare M_BlackPlastic=    // More basic material
material {
  texture {
   pigment { color rgb 0.1 }
       normal {bumps 0.1  scale 1/50}
    finish { plasFin}
  }
}

//For electric power cord, rubber feet of blender
//More of a rubber than a plasttic.
#declare CordPlastic = 
texture { 
	pigment { color rgb 0 } 
	finish {   
		#if (Radiosity= 0)
		  ambient 0.1 
		#end
    		diffuse 0.55 
    		specular 1
    		roughness 0.01
    		reflection 0.05  
    		  }  }

#declare M_Chrome=    // Chrome material
material {
  texture {
    pigment {rgb <0.20, 0.20, 0.20>}
        	normal {
    	    bumps 0.01
	        scale <1, 1, 1>/2   }
    finish {
      ambient 0.0
      brilliance 4
      diffuse 0.2
      specular 0.7
      roughness 0.08
      metallic
      reflection {0.4 0.6}//0.3
    }
  }
}

#declare  formicaFin = 		//Countertop finish
finish {
      	#if (Radiosity= 0)
			ambient 0.2
        #end
      diffuse 0.50
      specular 0.5 
      reflection 0
       }

#declare nuTan = <0.92, 0.78, 0.62>;

#declare M_Tan =    // Face plate texture
material {
  texture {
   pigment { color rgb nuTan }
       normal {bumps 0.005  scale <10,10,0.1>}
    finish { plasFin}
  }
}

#declare M_Tan2 =    // connector texture
material {
  texture {
   pigment { color rgb  <1, 0.95, 0.8> }
 finish {
#if (Radiosity= 0)
      ambient 1
#else
	  ambient 0.1
#end	        
      diffuse 0.50
      specular 0.5 
      reflection {
        0.01, 1.0   } 
    }
      }
}

#declare TrunkTex = texture {	//Tree Trunk
pigment { color red 0.36 green 0.25 blue 0.20}
finish{diffuse 0.4 specular 0.1 ambient 0.01}}

#declare LeafTex = texture {	//Leaves on trees
pigment {color red 0.196078 green 0.8 blue 0.196078}
finish{diffuse 0.3 specular 0.3 ambient 0.01}}

/************ END Textures and Materials **************/

/************ BEGIN Description of objects **************/

//*** The Background ***

  background{ color rgb 1}

//Room furnishings
/* There is actually a full room drawn here, with six walls.  The back
wall has something looking like a doorway, and there are a number of 
other objects outside of the camera view to give realistic and interesting
reflections from the shiny surfaces, not to mention interesting diffuse
reflections when radiosity is turned on. */
  
  //Countertop
#declare counterTop =     superellipsoid {
  <0.2,  0.2>
  scale <28,2,100> //was <29,2,100>
    texture {
     pigment { color rgb <1.0, 0.95, 0.9> }
    finish {formicaFin}
    normal {bumps 0.01  scale 1/20}
    }
    translate <-20,34.5,0>
	} //end superellipsoid
 
 object{counterTop}
 
 //Fish Food Flakes spilled on the counter.  
 #declare FishFoodColors = array[4]
{
	color rgb < 1, 0.8, 0.8>,
	color rgb < 1, 1, 0>,
	color rgb < 0.3, 1, 0.3>,
	color rgb <0.5,0.5,1>
}
#declare RS = seed(11111);

// This part... um... inspired by the beautiful "isocacti.pov" sample 
// file included with POV.  
#declare CtrlPtrn = function {pattern {bozo scale < 7, 1, 2>}}
#declare flake = 0;
#while(flake < 200)
	#declare Pt = trace(counterTop, < rand(RS)*15 - 7.5 - 5 , 40, rand(RS)*30-15 +10>, -y);
	#if(rand(RS) > CtrlPtrn(Pt.x, Pt.y, Pt.z)) 
		isosurface {
			function {f_r(x, y, z) - 1 + f_noise3d(x, y, z)*0.5}
			threshold 0
			max_gradient 1.5
			contained_by {sphere {0, 1}}
			rotate y*rand(RS)*360
			scale (0.02 + pow(rand(RS),2)*0.3)*<1,0.01,1>
			texture {
				pigment {Rand_Array_Item(FishFoodColors, RS)*0.7}
				normal {granite bump_size 0.5 scale 0.01}
			}
			translate Pt + 0.02*y  
		}
		#declare flake = flake + 1;
	#end
#end
 
 
 box {  <-20,-20,-100>, <5, 33, 100> 
 texture{pigment{rgb nuTan/2}
  finish {formicaFin}}
 } // stuff under counter
    
#declare WallTex = texture{
pigment { color rgb <0.99,0.98,1> } 
finish {
      	#if (Radiosity= 0)
			ambient 0.1
        #end
      diffuse 0.7
      specular 0.25 
      reflection 0
       }
normal {bumps 0.01  scale 1}
}
 
 //Back wall with window
box {<-30,-20,-200>, <-50,200,-100> texture {WallTex}}
box {<-30,-20,-100>, <-50,50,-30> texture {WallTex}}
box {<-30,150,-100>, <-50,200,-30> texture {WallTex}}
box {<-30,-20,-30>, <-50,200,200> texture {WallTex}}


//Window Frame!
union{
box {-1,1  rotate 45*z scale <1,1,40> translate <-40,50+66,-65>}
box {-1,1  rotate 45*z scale <1,1,40> translate <-40,50+33,-65>}
box {-1,1  rotate 45*z scale <1,1,40> translate <-40,150,-65>}
box {-1,1  rotate 45*z scale <1,1,40> translate <-40,54.2,-65>}
box {-1,1   scale <1.5,1,40> translate <-40,53.2,-65>}

box {-1,1  rotate 45*y scale <1,55,1> translate <-40,100,-100+70*2/3>}
box {-1,1  rotate 45*y scale <1,55,1> translate <-40,100,-100+70/3>}
box {-1,1  rotate 45*y scale <1,55,1> translate <-40,100,-100>}
box {-1,1  rotate 45*y scale <1,55,1> translate <-40,100,-30>}

pigment { color rgb <0.99,0.98,1> } 
finish {formicaFin}
}

//Window itself
box {<-40.1,54,-100>,<-39.1,150,-30> material{GlassTex2}
photons {  // photon block for an object
    target 1.0 
    reflection off
    refraction on  }
}



superellipsoid { <0.2,  0.01>		//Windowsill
   scale <11.5, 1, 35> translate <-37.5,51,-65> 
   texture {WallTex}}

//Left side wall
box {<-50,-20,-200>, <300,200,-220> texture {WallTex}}

//Right side wall
box {<-50,-20,200>, <300,200,220> texture {WallTex}}

//Wall behind viewer
box {<300,-20,-200>, <280,200,-150> texture {WallTex}}
box {<300,170,-150>, <280,200,-30> texture {WallTex}}
box {<300,-20,-30>, <280,200,200> texture {WallTex}}

box {<300,-20,-30>, <280,170,-150> pigment {color rgb 0.1}}	//dark doorway

box {<-50,-20,-200>, <300,-21,200> texture {WallTex}}	//floor

box {<-50,200,-200>, <300,201,200> texture {WallTex}}	//Ceiling


 // "invisible" objects that are for reflections only   
  cylinder {<-20,-20,-180>, <-20,200,-180>, 10  
    pigment { color rgb 0.25 }} 

 box {<50,-20,-200>, <100,150,-150>  pigment { color rgb 0.9 }} 
 box {<150,-20,-200>, <200,150,-150>  pigment { color rgb 0.9 }} 
 box {<250,-20,-200>, <300,150,-150>  pigment { color rgb 0.9 }} 
   
   
   
// End description of room   
   

//The outside world   
  
#if (showTrees)
/* two isosurface objects: More like a bush (close up) and a tree 
(distant), the latter being visible only as a shadow.
Focal blur is used to suggest distance and to draw the attention of the
viewer away from this part of the scene. */

plane {y,  -21 
    normal {bumps 1  scale 1/50}
pigment {
    bozo scale 10
        color_map {
        [0.00  rgb <0.36,0.25,0.2>]
        [0.33  rgb <0.2,0.5,0.2>]
        [0.66  rgb <0.4,0.5,0.2>]
        [1.00  rgb <0.36,0.25,0.2>]
    }
}}

union {		//Distant Bush
isosurface { //Brown branches
function{(f_granite(x/9,y/9,z/9)*0.4)} // f_granite
threshold 0.05
contained_by {sphere {0,1.9}}
max_gradient 1.274
texture{TrunkTex}
        }  

isosurface {// Green leaves
function{(f_bozo(3*x,4*y,4*z)*0.12)}
threshold 0.042
contained_by {sphere {0,2}}
max_gradient 0.381  
texture{LeafTex}

         }    
scale 28
translate <-475,40,-450>
}

union {	//Distant Tree
isosurface { //Brown branches
function{(f_granite(x/9,y/9,z/9)*0.4)} 
threshold 0.08
contained_by {sphere {0,1.9}}
max_gradient 1.274
texture{TrunkTex}
scale <0.5,2,0.5>
translate -y
        }  

isosurface {// Green leaves
function{(f_bozo(x*2,y*2,z*2)*0.12)}
threshold 0.04
contained_by {sphere {0,3}}
max_gradient 0.381  
texture{LeafTex}
scale 1.5*<1,0.5,1>
         }    
scale 40
rotate 135*y
translate <-750,150,-450>  
}

#else
  plane {y,  -21 pigment { color rgb 1}}
#end //if showTrees


#if (foodCan)

//A can of fish food
/* This is made out of several objects.  The main body of the can is
a lathe object, showing a little lip that hold the lid in place. 
The lid is a separate lathe object, with a nice bevel on the top.
A third object, an open cylinder, is wrapped around the body with the
image maps of the label.
*/

lathe {		//fish food can outside
  linear_spline 
   7// number of points
    <0, 0>
    <1.95, 0.0>
    <2, 0.05>
    <2, 4.0>
    <2.05, 4.05>
    <2.0, 4.1>  
    <2.0, 4.9>
   material {M_YellowPlastic}
     translate <-2,36.5,10>     
}

lathe { //fish food can lid
  linear_spline 
   4// number of points
    <2.15, 4.20>
    <2.15, 5.60>  
    <2.0, 5.75>  
    <0, 5.75>
   material {M_RedPlastic}
     translate <-2,36.5,10>     
}


//Fish Food can label
// This is the "coin" macro, called that because you can use it to put
// two different images on the opposite sides of a coin.
//  If anyone can tell me where this orignally
// came from, I'll happily give credit.

#macro DoubleLabel ( f1,f2 )
  radial
  pigment_map {
    [ .5 f1 ]  [ .5 f2 rotate 180*y ]
  }
#end

/* An image map is used to make the label on the fish food canister.  
(No, it's not cheating-- I made the image maps in POV-Ray.)

The image maps are the files labelFF1.png and labelFF2.png.   
If you don't have the files, you can generate them using the included file
labelFF.pov.  If *THAT* isn't included, you're flat out of luck. 
HA HA Sucks to be you!  Just kidding-- I've attached it as an appendix 
at the end of this file.  =)
*/

#local m1 = pigment {image_map {png "labelFF1.png" once }
		 	scale <4.2, 3.9,1 > translate -2.05*u}
#local m2 = pigment {image_map {png "labelFF2.png" once }
			scale <4.2, 3.9, 1> translate -2.05*u}

cylinder{ 0,3.9*y  2.01 open
  texture { pigment { DoubleLabel(m1,m2) } 
finish {
      	#if (Radiosity= 0)
		  ambient 0.5
		 #end
      diffuse 0.7
      specular 0.8 
      reflection 0       }  
  }
 rotate -60*y
 translate <-2,36.6,10>  
  }
  
#end // #if (foodCan)  



  
#if (SocketOn)   //Wall socket
   //This looks really good-- It would make a good alternate choice for
   //the focus of the whole scene.
   
   
//socket face plate
difference{	

superellipsoid {
  <0.2,  0.1>
  scale 0.9*<0.3,4.5,3>
   material {M_Tan}
	} //end superellipsoid

difference{
cylinder {-x,x,1.3}
	union{
		box {-2,2 translate 3.1*y} 
		box {-2,2 translate -3.1*y} 
	}	//end two-box union
	pigment {color rgb 0}
translate 1.6*y
} // end difference

difference{
cylinder {-x,x,1.3}
	union{
		box {-2,2 translate 3.1*y} 
		box {-2,2 translate -3.1*y} 
	}	//end two-box union
	pigment {color rgb 0}
translate -1.6*y
} // end difference

cone{0.3*x,0.4,0.001*x, 0		//Screw Hole
    pigment {color nuTan} }

translate <-30,45,-45>
}	//End difference for socket face plate
  
 difference {		//Screw itself
  sphere {0 , 0.35 }
 box {<1, -1, -0.1>,<0.1,1,0.1>}
// scale 10
 rotate -40*x
translate <-30,45,-45>
material {M_Chrome} 
}
 
 
 //Sockets (jacks) within face plate
 #local pSace = 0.4; //horizontal offset of prongs
 
difference{ 
cylinder {-0.2*x,0.2*x,1.2 material {M_Tan2}}
   union{
		box {-2,2 translate 3.0*y  } //flat part on top of cylinder
		box {-2,2 translate -3.0*y } //flat part on bottom of cylinder
		//indentations to help guide prongs into place
		box {-0.5,0.5 rotate 45*y scale <0.5,1.1,1> translate <0.4,0,pSace>} 
		box {-0.5,0.5 rotate 45*y scale <0.5,1.1,1> translate <0.4,0,-pSace>} 
		material {M_Tan2} 
		 }
     union{
	 box {<1, -0.4, -0.07>,<0.001,0.4,0.07>  translate pSace*z}	 //slot
	 box {<1, -0.4, -0.07>,<0.001,0.4,0.07>  translate -pSace*z} //slot
	pigment {color rgb 0}
	}
translate 1.6*y
translate <-30,45,-45> 

} // end difference 
 
 difference{ 
cylinder {-0.2*x,0.2*x,1.2 material {M_Tan2}}
   union{
		box {-2,2 translate 3.0*y  } 
		box {-2,2 translate -3.0*y } 
		box {-0.5,0.5 rotate 45*y scale <0.5,1.1,1> translate <0.4,0,pSace>} 
		box {-0.5,0.5 rotate 45*y scale <0.5,1.1,1> translate <0.4,0,-pSace>} 
		material {M_Tan2} 
		 }
	union{	 
	 box {<1, -0.4, -0.07>,<0.001,0.4,0.07>  translate pSace*z}
	 box {<1, -0.4, -0.07>,<0.001,0.4,0.07>  translate -pSace*z}
		pigment {color rgb 0}
	}
translate -1.6*y
translate <-30,45,-45>
} // end difference 
 

#end // if SocketOn  

////Build Goldfish 
/* The goldfish is designed to be animated, actually, swimming around in a
circle.  It's controlled by a clock variable, which is set to a constant 
here.  I've toyed with the idea of doing some POV-Ray animations for
IRTC, but I've never really gotten into it.  

This is at least the third way that I've drawn the fish-- It's a set
of mostly sphere sweeps in its present incarnation. */

#if (ShowFish)
union{

#declare T_GoldFish = 	//Body material, sortof has scales.
texture{
pigment { color <1,0.2,0>  }

 normal {uv_mapping
  pigment_pattern{
     leopard colour_map 
     {[0, rgb 0][1, rgb 1]}
     scale 0.02*<0.1,1,1> 
     }0.05}

 finish {
      ambient 0.2
      diffuse 0.50
      specular 0.5 
      metallic
      reflection {
        0.01, 0.1
      } }};
      

#declare M_GoldFish3 = 	//fin material
material {
  texture {
    pigment { color rgbt <1,0.2,0,0.5>  }
     normal {uv_mapping
  pigment_pattern{
     leopard colour_map 
     {[0, rgb 0][1, rgb 1]}
     scale 0.02*<0.1,1,1> 
     }0.05}
    finish {
      ambient 0.1
      diffuse 0.5
      specular 0.4
      metallic
	reflection {        0.01, 0.1   }
    } 
  }
  interior {
    ior 1.33
    fade_power 2
    fade_distance 0.2
	fade_color <1,0.2,0,0>
  }   
}

#local temptime = 0.5;	

sphere { 0 1 scale <.7, .7, 0.78> translate <5.3,0.35,0> pigment { rgb <1,1,0> }} // Eyes
sphere { 0 1 scale <.5, .5, 0.8> translate <5.41,0.35,0> pigment {rgb 0 }} // Eyes


#local tailWag = 0.2*sin(6*pi*temptime);
#declare tailz = function(x) {tailWag*(x-4)*(x-4)}
#declare tailx2 = function(x) {-1*x}
#declare tailx = function(x) {tailx2(x) + abs(tailWag)*2}


sphere_sweep {              //Main Body
  cubic_spline      
  6,                            // number of specified sphere positions
  <6.71, 0, 0>, 0.0                // position, radius
  <6.7, 0.0, 0>, 0.3           // position, radius
  <4.2, 0.0, 0>, 1.60              // position, radius
  <0, 0, 0>, 2.25               // position, radius
  <-6.3, 0, tailz(6.3)>, 0.8                // position, radius
    <-10, 0, tailz(10)>, 0.1 
scale <1,1,0.5>
        texture { T_GoldFish }
}


merge{
sphere_sweep {              // Top tail
  cubic_spline      
  6,                            // number of specified sphere positions
  <0, 0, 0>, 0.7             // position, radius
  <tailx(4), 0, tailz(4)>, 0.7             // position, radius
  <tailx(6.3), 0, tailz(6.3)>, 0.7                // position, radius
    <tailx(8.4), 1, tailz(8.4)>, 1.2                // position, radius
  <tailx(12.5), 5.5, tailz(12.5)>, 0.1                // position, radius
  <tailx(13), 6, tailz(13)>, 0.0                // position, radius
scale <1,1,0.5/2>
}

sphere_sweep {              // Lower part of tail
  cubic_spline      
  6,                            // number of specified sphere positions
  <0, 0, 0>, 0.7             // position, radius
    <tailx(4), 0, tailz(4)>, 0.7             // position, radius
  <tailx(6.3), 0, tailz(6.3)>, 0.7                // position, radius
    <tailx(8.4), -1, tailz(8.4)>, 1.2                // position, radius
  <tailx(12), -3.2, tailz(12)>, 0.1                // position, radius
  <tailx(13), -3.5, tailz(13)>, 0.0                // position, radius

scale <1,1,0.5/2>
      
}

  material { M_GoldFish3 }
}


sphere_sweep {              // dorsal fin
  linear_spline      
  5,                            // number of specified sphere positions
    <-5.7,1.6,0>, 0.1
    <-1.5,2.4,0>, 1
    <-1,4,0>, 0.25
    <0.5,1,0>, 1
    <-4.0,1,0>, 1
 
scale <1,1,0.5/2>
       material { M_GoldFish3 }
}

sphere_sweep {              // bottom rear Fin
  linear_spline      
  3,                            // number of specified sphere positions
    <-3,0,0>, 1 
   <-5,-2.0,0>, 0.75
   <-8,-3.25,0>, 0.1
 
scale <1,1,0.5/2>
       material { M_GoldFish3 }
}

#declare frontfins = 
sphere_sweep {              // front, lower fins Fin
  linear_spline      
  4,                            // number of specified sphere positions
   <-2.5,-4,0>, 0.1
   <0,-2,0>, 0.6
   <0,0,0>, 0.75
   <-2.5,-4,0>, 0.1 
scale <1,1,0.5/2>
       material { M_GoldFish3 }
}

    object {frontfins rotate  <-25-10*sin(4*pi*temptime),0,0> }// Was -20
    object {frontfins rotate  <25+10*sin(4*pi*temptime),0,0>} // Was +20
    			//Angle of 10-45 degrees OK here.

   #declare sidefins=       
sphere_sweep {              // side Fin
  linear_spline      
  5,                            // number of specified sphere positions
   <2.1,-0.4,0>, 0.25
   <0,-2,0>, 0.25
   <-1.5,-1.5,0>, 0.25
   <2.1,-0.4,0>, 0.25
   <-0.75,-1.75,0>, 0.25
scale <1,1,0.5/2>
       material { M_GoldFish3 }
}
          
   object {sidefins rotate  <-30-20*sin(8*pi*temptime),0,0> translate <0,0,1>} // Was -20
    object {sidefins rotate  <30+20*sin(8*pi*temptime),0,0> translate <0,0,-1>}// Was +20
    
sphere_sweep {              // Gills
  cubic_spline      
  5,                            // number of specified sphere positions
  <4.5, 3, 0>, 0              // position, radius
  <4.2, 1, 0>, .2              // position, radius
  <3.9, 0.0, 0>, 0.95              // position, radius
  <4.5, -0.9, 0>, .6              // position, radius
  <4.5, -3, 0>, 0              // position, radius
scale <1,1,1>
        texture { T_GoldFish }
}
   
   scale 0.25   //Fish Scale (Ha ha.)
   rotate <10,0,-15> 
   rotate <0,90,0>//   rotate <0,100,0>
   translate <1,57,1> // was <1,56.5,0>
	 
}//End Fish Union


#end // if ShowFish

 
///////////////*BLENDER*///////////////

#if (blenderOn) 

#if (showLid)       //Lid
union{
difference{
union{

difference{		//Lower part, sticks into pitcher
	superellipsoid {  <0.6, 0.6>   }
	union{
		cylinder {-0.3*y,-10*y, 10}
		cylinder {0.3*y,10*y, 10}
	}
	scale <4.3,2,4.3>   
}

difference{		//Upper part, sticks out above lip
	superellipsoid {  <0.6, 0.6>   }
	union{
		cylinder {-0.2*y,-10*y, 10}
		superellipsoid {  <0.6, 0.6> scale <1,0.25,1>  translate y*0.15}
		cylinder {0.20*y,10*y, 10}
	}
	scale <4.7,2,4.7>
	translate 1.0*y   
}

rotate 45*y
}

cylinder {-5*y, 5*y, 2} // Cut out for cap
     material {M_BlackPlastic}
}

 // Top cap 
difference{    	//Hollow cylinder
cylinder {0, 1.8*y, 2} 	
cylinder {-0.1*y, 2*y, 1.8} 
   material {PolyCarbonate}
     }
cylinder {1.8*y, 2*y, 2.5	//topmost lip of cap
   material {PolyCarbonate}
     } 
     translate <0,64,0>  
}
#end // if showLid


#if (showBody)  // Show body of blender (pitcher)

 //Define lettering macro for text on sides of pitcher
#macro lettering(textIn, vPos, hPos)
text { ttf "cyrvetic.ttf", textIn, 0.15,  0   
translate <0,0,-0.07>     //center extrusion      
scale 0.5
translate <-0.65,0,0>	  //center text L-R
//texture{pigment {color rgb 0.8*z}} // make visible,  if necessary.
rotate -2.5*x
rotate 238*y
translate vPos*y
translate hPos
}
#end

//Declare isosurface for pitcher body
#declare pitcherBody =
isosurface { 
 
	function { 
	sqrt(x*x + pow(y/2 - sqrt((x*x/4) + (z*z/4))*1.5, 2)/2 + z*z) - 1 
	 -(sin(atan2(x, z)*4)*0.05*(1+y) )
	 +(sin(atan2(x, z)*4)*0.05*(y*y-1))  
	-0.02*ceil(-0.48 +  ceil(0.4-y/2)*0.5*sin(10*(y+1.5)*(y+1.5))) // horiz. stripes
	-0.03*ceil(0.81 + y/5) // Top lip
		}
	threshold 0
        max_gradient 5 
	contained_by {box {<-2,-1.35,-2>, <2,1,2>}} 
}

/* Ah, the magic of isosurfaces! Yes, that's one isosurface to draw the 
outside of the glass body of the pitcher, less the handle 
(an added sphere sweep) and lettering.

I was (as in other places) inspired by the isocacti.pov example file, 
which shows complex, organic forms rendered out of isosurfaces.  
The horizontal stripes and the top lip are made by adding a term with
the ceiling function [ceil(_float_)] to make discrete steps.  
*/

difference{		
merge{ //"Positive" objects
object {pitcherBody rotate <0,-22,0>}

//Letters on sides of pitcher: hard to see (at all) when using
//the glass texture.  But somehow, it's those things that you can't
//see that make things look realistic.  Weird.
union {
object { lettering("64 Oz.", 5.2, <5.05,0,1.3>) }
object { lettering("48 Oz.", 3.2, 0.975*<5.05,0,1.3>) }
object { lettering("32 Oz.", 1.0, 0.935*<5.05,0,1.3>) }
object { lettering("16 Oz.", -1.8, 0.867*<5.05,0,1.3>) }
scale <1,1/1.3,1>/5
}

union {
object { lettering("64 Oz.", 5.2, <5.05,0,1.3>) }
object { lettering("48 Oz.", 3.2, 0.975*<5.05,0,1.3>) }
object { lettering("32 Oz.", 1.0, 0.935*<5.05,0,1.3>) }
object { lettering("16 Oz.", -1.8, 0.867*<5.05,0,1.3>) }
scale <1,1/1.3,1>/5
rotate 90*y
}

#local c1 = <0, 0.55, 1.6>;
#local c2 = <0, -0.4, 1.4>;

sphere_sweep {              //Handle of pitcher
  linear_spline      
  8,                         // number of specified sphere positions
  <0, 0.7, 0.9>, 0.2   		          
  <0, 0.65, 1.15>, 0.10   
  c1+<0,-0.03,0.03>, 0.12          //corner  
  (c1+c2)/2 + <0,0,0.03>, 0.11          //midpoint of handle  
  c2 + <0,0.03,0.03>, 0.12          //corner  
  c2 + <0,-0.03,-0.03>, 0.12          //corner  
  <0, -0.7, 0.8>, 0.10          
  <0, -0.8, 0.6>, 0.14           
} // end sphere sweep

scale 5*<1,1.3,1>		//scale all positive objects
}	//end merge of "postive" objects

sphere {0, 1 scale < 5.25, 14, 3.25>  translate <0,6,0>}
sphere {0, 1 scale < 5.25, 14, 3.25>  rotate 90*y translate <0,6,0>}

sphere {0, 1 scale < 4.10, 14, 4.10> translate <0,6,0>}
sphere {0, 1 scale < 3.5, 10, 3.5> translate <0,0,0>}

#if (cutOn)
	box {<0,-10,-10>, <10,10,10>   }
 #end // if cutOn

 photons {  // photon block for glass pitcher
    target 1.0 
    reflection on
    refraction on
  }

#if (glassOn)
	material {GlassTex}
#else  	 
  	material {texture{GreenTex}}
#end // if glassOn  	  
  	 translate <0,58,0>			  	 
  	 }

#end // if showBody




//Driving mechanism of blender: Blades and support
#if (showDrive)  
difference{
lathe {
  linear_spline 
  8 // number of points
  	<0, 0.5>
    <1.5, 0.5> 
    <1.5, 0.76>   
    <3.1, 0.76>   
    <3.1, 1.2> 
    <1.5, 1.35>   
    <0.75, 2.25>   
    <0, 2.5> 
   material {M_WhitePlastic}
        scale 0.95
     translate <0,48,0>   
}

#if (cutOn)
box {<0,-5,-5>, <5,5,5>   material {M_WhitePlastic}
     translate <0,50,0>  }
#end // #if (cutOn)    
}

//Blades 
union{
cylinder {2.5*y,2.7*y, 1   }

cone {0, 1, 2*z, 0.1 scale <0.5,0.1,1> rotate -45*x translate <0,2.6,0.8>}
cone {0, 1, -2*z, 0.1 scale <0.5,0.1,1> rotate 45*x translate <0,2.6,-0.8>}
cone {0, 1, 1.5*x, 0.1 scale <1,0.1,0.5> rotate -45*z translate <0.8,2.6,0>}
cone {0, 1, -1.5*x, 0.1 scale <1,0.1,0.5> rotate 45*z translate <-0.8,2.6,0>}

   material {M_Chrome}
           scale 0.95
   translate <0,48,0>   
rotate <0,30,0>
}
#end // if showDrive


#if (showBase) //Base of pitcher 
difference{
lathe {
  linear_spline // linear_spline | quadratic_spline | cubic_spline
  9 // number of points
    <4.3, -0.5>
    <3.9, 2.0> //<4,3>
    <3.4, 2.2>//<3, 2.25>
    <3.2, 0.75>//<3, 2.25>
    <1.5, 0.75>//<1.5, 2.25>  
    <1.5, 0.5>   
    <3.5, 0.5>   
    <3.75, -0.5>   
    <4.3, -0.5> 
   material {M_Chrome}
        scale 0.95
     translate <0,48,0>   
}
#if (cutOn)
box {<0,-5,-5>, <5,5,5>   material {M_RedPlastic}
     translate <0,50,0>  }
#end // #if (cutOn)   

photons {  // photon block for an object
    target 1.0 
    reflection on
    refraction off
  }
  
     }
     
#end // if showBase


#if (showMotor)
/* This is the whole structure below the pitcher (except the cord), 
including the little separating cones (each called a baseCone here), 
the label, and the rubber feet. */

#declare baseCone = 
cone {-0.5*y, 0.75, y, 0.4 material {M_Chrome} translate 3.5*x 
translate <0,46.5,0>
photons {  // photon block for an object
    target 1.0 
    reflection on
    refraction off
  }
}
   
object { baseCone rotate 0*y }		
object { baseCone rotate 90*y  }
object { baseCone rotate 180*y  }
object { baseCone rotate 270*y }


/* Rubber feet beneath the motor */

#declare rubberfoot = 
cone {-0.5*y, 0.5, 0.5*y, 0.74 texture {CordPlastic} translate <6,37,0>
rotate 45*y}

object { rubberfoot rotate 0*y }		
object { rubberfoot rotate 90*y }		
object { rubberfoot rotate 180*y }		
object { rubberfoot rotate 270*y }		

/*Beehive-like motor shell.  Again, if you want something interesting,
you've got to make it out of an isosurface! */

difference{			//Beehive-like motor shell
isosurface { 
	function { 
	sqrt(x*x + z*z) - 4.8 + (4.8 - 7)*(1-(y/9))
	+ sin(5*2*pi*y/9 + pi/2)/20 } 
	threshold 0
        max_gradient 5 
	contained_by {box {<-10,9,-10>, <10,0,10>}
}  
		material {M_Chrome} 
}
cylinder {9.5*y, 5*y, 3 pigment {color rgb 0}}	//cut-away for motor shaft
translate <0,46.5-9,0>

photons {  // photon block for an object
    target 1.0 
    reflection on
    refraction off
  }
}



/* An image map is used to make the label on the blender.  
(No, it's not cheating-- I made the image map in POV-Ray, too.)

The image map is the file label.png.   If you don't have the file
label.png, you can generate one using the included file label.pov.  
(I've attached label.pov as an appendix at the end of this file.)

The shape of the label is a section of a cone... Not the usual 
"conic section" (e.g., a parabola) but a CSG object cut out of a 3D cone.  
The cone used is one that pretty closely approximates the shape of the
motor shell, but a little bit larger.  I toyed with cutting the shape
of the label in a couple of different ways; enable the other two cuts
in the "difference" object below to see how that looks.
*/

// Label on blender.  

difference {
cone {0, 4.9, -9*y, 7.1}
union{
cylinder {y, -3*y, 10}
cylinder {-7*y, -10*y, 10}
box {<-10,-10,0>, 10 translate 2*z}		//standard cut
box {<10,10,0>, -10 translate -2*z}		//standard cut
//box {<-10,-10,0>, 10 rotate -20*y}	//Alternate cut
//box {<10,10,0>, -10 rotate 20*y}		//Alternate cut
} 

material {
  texture {
pigment {image_map {png "label.png"} scale 4 translate <-2,-2.65,-2> 
scale 1.07
rotate -90*y}
        	normal {
    	    bumps 0.005
	        scale <1, 1, 1>/2   }
    finish {
      	#if (Radiosity= 0)
			ambient 0.5
        #end
      brilliance 4
      diffuse 0.2
      specular 0.5 
      metallic
      reflection 0.1
    }
  }
}

translate <0,46.5,0>
}

//Switch on faceplate of blender
torus {		//Ring around the switch
  0.25,
  0.1
rotate 90*z
translate <6.25,41.52,0>
material {M_Chrome}
}
//Rod of the switch (looks like a cylinder)
cone {<6.25,41.52,0>, 0.1, <6.25,41.52,0>+ 0.75*x + 0.5*y, 0.15
material {M_Chrome}}
//Rounded end of switch
sphere{<6.25,41.52,0>+ 0.75*x + 0.5*y, 0.15
material {M_Chrome}}		

#end // #if (showMotor)






 


#end // if blenderOn

 
#if (cordOn)

 #if (blenderOn = off)
   //placeholder for blender, if blender is off
    cylinder {-0.5*y, 0.5*y, 6 texture {pigment {color rgb 0.5}} translate <0,38,0>}
 #end    
 
  // Cord End Plug
  
   #local pSace = 0.4; //horizontal offset of prongs
union{
difference{
union{
  	 box {<1, -0.4, -0.07>,<-0.5,0.4,0.07>  translate pSace*z}
  	 box {<1, -0.4, -0.07>,<-0.5,0.4,0.07>  translate -pSace*z}
     rotate 90*y
  }	//end union of positive things
  	 cylinder {-x, x, 0.15  translate -0.6*z }
  	 box {-2,2  translate <0,2.35,1.6>}
  	 box {-2,2  translate <0,-2.35,1.6>}

material {			//brass-like
  texture {
    pigment {rgb <0.65, 0.50, 0.25>}
        	normal {
    	    bumps 0.01
	        scale <1, 1, 1>/2   }
  finish {
    ambient 0.25
    brilliance 4
    diffuse 0.5
    metallic
    specular 0.80
    roughness 1/80
    reflection 0.5
}

}}
  }
  
// Ridges on plug at end of power cord
  
union{
superellipsoid {
  <0.2,  0.1>
  scale 1.2*<1,0.75,0.1>
   translate z
	} //end superellipsoid
	
superellipsoid {
  <0.2,  0.1>
  scale 0.7*<1,0.75,1.6>
   translate z*2
	} //end superellipsoid

superellipsoid {
  <0.2,  0.1>
  scale 1*<1,0.75,0.1>
   translate z*1.5
	} //end superellipsoid
	
superellipsoid {
  <0.2,  0.1>
  scale 0.9*<1,0.75,0.1>
   translate z*2
	} //end superellipsoid

superellipsoid {
  <0.2,  0.1>
  scale 0.8*<1,0.75,0.1>
   translate z*2.5
	} //end superellipsoid
	
superellipsoid {
  <0.2,  0.1>
  scale 0.75*<1,0.75,0.1>
   translate z*3
	} //end superellipsoid
	
	 texture {CordPlastic} 
	translate -0.7*z
	}
	
	rotate -8*y
	rotate 5*z
	rotate 4*x
translate <-20,37.4,-40>	//Match cord end
}
//END Cord-end plug	
	
	
	  
/* Begin the cord itself.  
This was the single hardest thing about the scene-- to get a natural
looking coiled-up cord that looked like the turns were resting on each
other, not intersecting, not kinked, and not too flacid. Trial and error,
man, trial and error.   This sort of thing is enough to make me want to
instead do it in a mechanics simulation in MegaPOV-- but I was aiming to
be a POV-Ray purist here.   Anyway, I don't actually now how to do this 
in MegaPOV, I'm just assuming we can.  But it's probably a lot of work. */

#local CordRad = 0.3; 

sphere_sweep {              //Cord
  b_spline  	// cubic_spline    linear_spline
  16,                            // number of specified sphere positions
  <2, 1, 0>, 0.5                // control point
  <-4, 1, 0>, 2*CordRad                // position, radius
  <-6, 1, 0>, 2*CordRad             // Begin strain relief   
  <-9, 1, 0>, CordRad             // End of strain relief   
  <-14, 0.5, 3>, CordRad              //  Shortly after strain relief
  <-22, 0, 9>, CordRad              //  First bend, away from socket
  <-30+CordRad, 0, 0>, CordRad                	// Back point
  <-20, CordRad, -6>, CordRad              
  <-6.5, 0, 2>, CordRad                // Close to strain relief
  <-17,  4*CordRad, 6>, CordRad                // Between strain relief & second bend
  <-25, 3*CordRad, 5>, CordRad 			// second bend
  <-30+CordRad,1*CordRad,-4>, CordRad					//back point, second loop
  <-17,0*CordRad,-15>, CordRad
  <-22,0,-28>, CordRad					// 
  <-20,CordRad*1.25,-38>, CordRad					// end point, roughly below socket
  <-20,CordRad*2,-50>, CordRad					// control point
          tolerance 3   
   texture {CordPlastic}

        translate <0,36.5+CordRad,0>
}
#end // if CordOn


#if (showWater)

intersection{  cylinder{ -8*y, 3.5*y, 6}

merge{
sphere {0, 1 scale < 5.25, 14, 3.25>  translate <0,6,0>}
sphere {0, 1 scale < 5.25, 14, 3.25>  rotate 90*y translate <0,6,0>}

sphere {0, 1 scale < 4.10, 14, 4.10> translate <0,6,0>}
sphere {0, 1 scale < 3.5, 10, 3.5> translate <0,0,0>}
}

translate <0,58,0>

//	material {texture{GreenTex}}  //Quick texture sub. for water

material{			// water
 interior { ior 1.33
    #if (Radiosity=off)
      caustics 1
    #end //  end "#if (Radiosity=off)"
      }
 texture { 
 pigment { color rgbf <0.9,0.95,1,0.999>} 
    normal { pigment_pattern{function 
   {(0.5+0.25*(sin(pi*sqrt(x*x  + z*z))))} 
   colour_map {[0, rgb 1][1, rgb 0]}}1}
 finish {
      reflection {   0, 0.1 fresnel on }
      conserve_energy
 } } }	//End material statement

scale <0.9999,1,0.9999>
/*Need to have the sides NOT QUITE touching the glass, or else the
water has no (apparent) effect at all-- it's very strange. */

photons {  // photon block for an object
    target 1.0 
    reflection on
    refraction on  }
}

#end //#if (showWater)


/************ END Description of objects **************/







/************ BEGIN Camera **************/

camera {
perspective
	angle 10 
	
/* A number of additional viewpoints and viewing target positions
are provided here for debugg--um... Viewing! the parts of the scene. */

//    look_at <0,51,0> //typical blender 
//    location <1.2*180,90,130> //typical blender 

 //  look_at <-20,37.4,-38>			//cord end
 //  location <-20,37.4,-38> + 50*y	//cord end -- from top
//    location <-20,37.4,-38> + 50*x	//cord end -- from front

//   location <-15,65,-300> //Temporary  //looking along wall (back side)
 // location <-15,65,300> //Temporary  //looking along wall
//    look_at <-15,40,-10> //Temporary  //looking along wall

//   location <400,300,-10> //Distant view, down at corner
// location 7*<400,300,-10> //Distant view
 //  location 10*<0,500,-10> //Distant view	--Above
//   location 2*<-800,50,-10> //Distant view Behind wall
     
look_at <0,51,-8> // Standard View 
location <1.2*180,90,140> //Standard View 

#if (PhocalBlur)
focal_point <0,51,-8>
aperture 2
blur_samples 100 //Slow down rendering HERE!
confidence 0.999           // [0...<1] when to move on while sampling (smaller is less accurate)
variance  1/200

#end //if
}// End camera statement
   
//It's tricky to tune focal blur to give good antialiasing.
//Some test render times:

 //3, 100, 0.999, 1/200	 17 s Good
 // Variance 0: 81 s, but nice looking
 
// 10: 2 s		poor
// 100: 5 s		passable
// 500: 8 s		passable
// 1000: 10 s	good
// 0.9: 11 s	No quality change
// 0.9999: 11s		
// 1/50000: 11 s	No quality change, really.  
// 1/200: 5 s        
// Variance = 0: Damned slow, but pretty.
          
          
/************ End Camera **************/

          
          
          
          
/************ BEGIN LIGHTING AND SKY **************/

sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 rgb <0.6,0.7,1.0>]
      [0.7 rgb <0.0,0.1,0.8>]
    }
  }
}


light_source {		//Sunlight
 4*<-400, 485, -350>   
     color rgb 3.2*<1,1,0.7> 
   area_light <50, 0, 0>, <0, 0, 50>, 2, 2
    adaptive 1
    circular
    jitter
 
       photons {           // photon block for a light source
    refraction on
    reflection on
  }
 } 


 light_source {		//Indoor area light source (New hotness)
<120,199,0>
    color rgb 0.75*<1,0.95,0.90> 
    area_light <20, 0, 0>, <0, 0, 20>, 2, 2		//Make it purdy.
    adaptive 1
    circular
    jitter
   
 spotlight
  point_at <0,50,0>      // direction of spotlight
   falloff 2               // intensity falloff radius (outer, in degrees)
  tightness 100            // tightness of falloff (1...100) lower is softer, higher is tighter
       photons {           // photon block for a light source
    refraction on
    reflection on
  }
 } 


/* Alternate means of lighting the indoors: point a spotlight at the
ceiling; rely only on diffusely reflected light (radiosity) to 
light the scene.  This works okay, but the contrast is a bit poor.
It's useful to put this inside an #if case with respect to radiosity,
because the light will turn off completely if you turn radiosity off.*/


/*
#if (Radiosity)		//Indoor light sources (Old and busted)

 light_source {		//Indoor, indirect light source
<150,150,0>
    color rgb 3*<1,0.97,0.95> 
spotlight
  point_at <150,200,0>      // direction of spotlight
 
       photons {           // photon block for a light source
    refraction off
    reflection off
  }
 } 
  
#else
light_source {		//Indoor light source
<150,180,50>
    color rgb 1.5*<1,0.98,0.97>  
 //   area_light <50, 0, 0>, <0, 0, 50>, 3, 3		//Make it purdy.
 //   adaptive 1
 //   circular
 //   jitter
 
       photons {           // photon block for a light source
    refraction off
    reflection off
  }
 } 
#end // #if (Radiosity)
*/

/************ END LIGHTING AND SKY **************/



/**********END aquarium.pov***************/














/**********BEGIN appendices***************

The appendices contain the support files; label.pov and labelFF.pov.
 Do NOT remove the comments--  If you want to run these, make them 
each into a new file!

// BEGIN File label.pov
//
// Copyright 2005 Windell Oskay
// ( windell (at) oskay.net )
//
// I claim copyright solely for the purpose of being able to claim 
// that *I too* have the right to use the contents of this scene.  
//
// So... please let me know if you reuse any part of this file.
//
// This file is a support file to aquarium.pov, 
// used to generate label.png, 
// the image map for the label on the blender.

camera {   orthographic
   location -2.5*z
   look_at 0  }
   
background {color rgb 1}
#default {finish {ambient 1 diffuse 0}}

union{
text { ttf "povlogo.ttf", "P", 0.1,  0              
scale 0.5 translate <-0.9,0.5,0>
texture{pigment {color rgb 0.8*z}}}

text { ttf "timrom.ttf", "overizer", 0.1,  0              
scale 0.4 translate <-0.6,0.5,0>
texture{pigment {color rgb 0.8*z}}}

superellipsoid {  <0.2, 0.2>  scale 1.1  translate 4*z}

superellipsoid {  <0.2, 0.2>  scale 1.05  translate 2*z
texture{pigment {color rgb 1}}}

difference{
superellipsoid {  <0.2, 0.2>  scale 0.95}
box {<-5,-0.6,-5>, 5}
}

text { ttf "cyrvetic.ttf", "ON", 0.1,  0              
scale 0.2 translate <0.2,0.2,0>
texture{pigment {color rgb 0.8*z}}}

text { ttf "cyrvetic.ttf", "OFF", 0.1,  0              
scale 0.2 translate <0.2,-0.1,0>
texture{pigment {color rgb 0.8*z}}}

text { ttf "cyrvetic.ttf", "PULSE", 0.1,  0              
scale 0.2 translate <0.2,-0.4,0>
texture{pigment {color rgb 0.8*z}}}

scale <4/3,1,1>
}
// END File label.pov


// BEGIN File labelFF.pov
//
// Copyright 2005 Windell Oskay
// ( windell (at) oskay.net )
//
// I claim copyright solely for the purpose of being able to claim 
// that *I too* have the right to use the contents of this scene.  
//
// So... please let me know if you reuse any part of this file.
//
// This file is a support file to aquarium.pov, 
// used to generate labelFF1.png, labelFF2.png, 
// the image maps for the label on the can of fish food.
//
// Run as an animation for two frames to generate the two files.
//


camera {   orthographic
angle 30
   location -10*z 
   look_at 0  }
   
#default {finish {ambient 1 diffuse 0}}
   

#declare titleColor = x;
#declare bgColor = x+y;

background {color rgb bgColor}


#declare ds = 
triangle {
  <-0.2,  0.5, 0>,  
  < 0.2,  0.5, 0>,  
  < 0,  -0.5,  0>  
}

#declare hs = 
object {ds rotate 90*z scale 0.75}

#declare Ht =
union {
object {ds translate -0.25*x}
object {ds translate 0.25*x}
object {hs}
    }
    
#declare At =
union {
object {ds rotate -20*z translate -0.15*x}
object {ds rotate 20*z translate 0.15*x}
object {hs}
    }
    
#declare Pt =
union {
object {ds translate -0.1*x}
object {ds scale 0.75 rotate 50*z translate <0.12,0.25,0>}
object {ds scale 0.75 rotate -50*z translate 0.05*x}
    }    

#declare It =
union {
object {ds translate -0.05*x}
object {hs translate 0.4*y}
object {hs translate -0.3*y}
    }    

#declare Ft =
union {
object {ds translate -0.2*x}
object {hs translate 0.4*y}
object {hs translate 0.1*y}
    }    

#declare St =
union {
object {ds scale 0.75 rotate -50*z translate <-0.15,0.25,0>}
object {ds scale 0.75 rotate 50*z }
object {ds scale 0.75 rotate -50*z translate -0.25*y}
    }    

#declare title = union {
object{Ht  translate -1.9*x}
object{At  translate -1*x}
object{Pt  translate -0.25*x}
object{Pt  translate 0.4*x}
object{It  translate 1.3*x}
object{Ft  translate 2.5*x}
object{It  translate 3.3*x}
object{St  translate 4.0*x}
object{Ht  translate 4.8*x}

pigment{rgb titleColor}
scale <0.2,0.4,1>}



#if (frame_number=1)
union {
object {title translate <-0.25,0.4,0>} 

sphere { <0,-0.25,0>, 0.45		//red spot
pigment{rgb titleColor}
scale <1,1,0.001>
}

union {				//Fish on red spot
triangle {
  <-0.25,  0.15, 0>, < 0.05,  0, 0>,  < -0.15,  0,  0>  }
triangle {
  <-0.25,  -0.15, 0>, < 0.05,  0, 0>,  < -0.15,  0,  0>  }
triangle {
  <0.4,  0, 0>, < 0.0,  0, 0>,  < 0.2,  0.1,  0>  }
triangle {
  <0.4,  0, 0>, < 0.0,  0, 0>,  < 0.2,  -0.1,  0>  }

pigment{rgb bgColor}
translate <-0.05,-0.25,-0.2>
}

sphere { <0.2,-0.12,-1>, 0.03
pigment{rgb bgColor}
scale <1,1,0.001>}

sphere { <0.16,-0.04,-1>, 0.04
pigment{rgb bgColor}
scale <1,1,0.001>}

sphere { <0.2,0.07,-1>, 0.05
pigment{rgb bgColor}
scale <1,1,0.001>}

scale 2.75
}
#end // if frame number = 1


box{ <-0.05,-10,-0.1> <0.05, 10, 0.1>		//Divider line
pigment{rgb titleColor}
translate -2.65*x}

box{ <-0.05,-10,-0.1> <0.05, 10, 0.1>		//Divider line
pigment{rgb titleColor}
translate 2.65*x}

box{ <-10,-0.02,-0.1> <10, 0.02, 0.1>		//Top Stripe
pigment{rgb titleColor}
translate 2*y}

box{ <-10,-0.02,-0.1> <10, 0.02, 0.1>		//Bottom Stripe
pigment{rgb titleColor}
translate -2*y}


#if (frame_number=2)
union {
object {title translate <-0.3,0.4,0>}

union{
text { ttf "cyrvetic.ttf", "HAPPI FISH is the favorite of fish", 0.1, 
 0 }

text { ttf "cyrvetic.ttf", "lovers (and by that we mean pet", 0.1, 
 0 translate -0.9*y}
 
text { ttf "cyrvetic.ttf", "goldfish owners, NOT seafood fans)", 0.1, 
 0 translate -1.8*y}
 
text { ttf "cyrvetic.ttf", "everywhere. Our fish food is made", 0.1, 
 0 translate -0.9*3*y}
 
text { ttf "cyrvetic.ttf", "from the finest ground up, dried...", 0.1, 
 0 translate -0.9*4*y}
 
text { ttf "cyrvetic.ttf", "um... fish.  By the way, if you're", 0.1, 
 0 translate -0.9*5*y}
 
text { ttf "cyrvetic.ttf", "reading this, you need to get out", 0.1, 
 0 translate -0.9*6*y}
 
text { ttf "cyrvetic.ttf", "more often and stop reading other", 0.1, 
 0 translate -0.9*7*y}
 
text { ttf "cyrvetic.ttf", "people's code.  Really, now!", 0.1, 
 0 translate -0.9*8*y}
 
pigment{rgb titleColor}
scale 0.1
translate <-0.75,0.1,0>
}
scale 2.75
}

#end //if frame_number = 2

// END File labelFF.pov

*********END appendices***************/
