// Persistence of Vision Ray Tracer Scene Description File
// File: chembond.pov
// Vers: 3.6
// Desc: IRTC Entry 6/2005 - Forces of Nature
//       Octane molecule on a desk
// Date: 06/24/05
// Auth: David Nash
// Todo:    - add a corkboard on the back wall  - done
//          - scale the can smaller in x and z  - done
//          - add pegs and thumbtacks           - done
//          - complete lighting                 - done
//          - move can right and add sig        - done
//          - recode to cleanup                 - done
//          - final run
// 

// ==========================
// License
// ==========================
// This work is licensed under the Creative Commons
// Attribution-ShareAlike License. To view a copy of
// this license, visit
// http://creativecommons.org/licenses/by-sa/2.5/ 
// or send a letter to
//      Creative Commons
//      559 Nathan Abbott Way
//      Stanford, California 94305, USA.

// ==========================
// Include files and variables
// ==========================
#version 3.6;

// Standard includes                    
#include "colors.inc"
#include "woods.inc"

// Scene includes
#include "pencil.inc"
#include "molecules.inc"

//===============
// Settings
//===============
#declare Area_Lights=on;
#declare Focal_Blur=yes;
#declare Radiosity=on;
#declare R1 = seed(1234);       // initialize random number streams

global_settings {
    assumed_gamma 1.0
    max_trace_level 32
    #if (Radiosity=on)
        ambient_light 0         // Remove all texture ambient and
                                // replace with radiosity
        radiosity {
            brightness 1.0      // Makes the scene brighter or
                                // darker like ambient_light
            
            pretrace_start 0.08 // Size of radiosity block as a %
                                // of the image to start with in
                                // the pretrace.
            pretrace_end   0.01 // Size of radiosity block as a %
                                // of the image to end with in
                                // the pretrace.
            
            count 750           // 35..1600 Rays sent out to
                                // calculate a new radiosity value.
            nearest_count 10    // 5..20 Rays collected from nearby
                                //objects to calculate value.
            error_bound 0.1     //default 1.8 How much error can you
                                // stand?  Very low values (less than
                                // 0.1) need high count values.
            recursion_limit 3   // default 3
            
            low_error_factor 0.5    // reduce error_bound during last pretrace step
            gray_threshold 0.0
            minimum_reuse 0.015
            
            adc_bailout 0.01 / 2	//Leave as this.
        }
    #else
        ambient_light <1, 1, 1> //default
    #end
} // global_settings

/*
// radiosity (global illumination) settings
global_settings {
  radiosity {
    pretrace_start 0.08           // start pretrace at this size
    pretrace_end   0.04           // end pretrace at this size
    count 35                      // higher -> higher quality (1..1600) [35]
    nearest_count 5               // higher -> higher quality (1..10) [5]
    error_bound 1.8               // higher -> smoother, less accurate [1.8]
    recursion_limit 3             // how much interreflections are calculated (1..5+) [3]
    low_error_factor .5           // reduce error_bound during last pretrace step
    gray_threshold 0.0            // increase for weakening colors (0..1) [0]
    minimum_reuse 0.015           // reuse of old radiosity samples [0.015]
    brightness 1                  // brightness of radiosity effects (0..1) [1]

    adc_bailout 0.01/2
    //normal on                   // take surface normals into account [off]
    //media on                    // take media into account [off]
    //save_file "file_name"       // save radiosity data
    //load_file "file_name"       // load saved radiosity data
    //always_sample off           // turn sampling in final trace off [on]
    //max_sample 1.0              // maximum brightness of samples
  }
}
*/


//===============
// Lights and Camera
//===============
camera {
    #if (Focal_Blur=on)
        focal_point <0,0,0>
        aperture 0.25
        blur_samples 200
        confidence 0.95
        variance 1/32768
    #end
    location <-25, 75, -150>
    angle 12.5
    look_at <0,5,0>
}

light_source {<-20, 50, 15> color White
    fade_distance 75.0 fade_power 2
    #if (Area_Lights=on)
        area_light <-20, 0, -4>, <20, 0, 4>, 6, 2
        adaptive 1
        jitter
    #end
}

light_source { <20, 50, 0> color White
    spotlight                 // this kind of light source
    point_at <12.5, 0, 12.5>  // direction of spotlight
    radius 8                  // hotspot (inner, in degrees)
    tightness 25              // tightness of falloff (1...100) lower is softer, higher is tighter
    falloff 16                // intensity falloff radius (outer, in degrees)
    #if (Area_Lights=on)
        area_light <-1, 0, -1>, <1, 0, 1>, 2, 2
        adaptive 1
        jitter
    #end
}

//===============
// Textures
//      T_Page01    - an image_map of a printed page
//      T_Page02    - an image_map of a printed page
//      T_Page03    - an image_map of a printed page
//      T_Desktop   - polished wood
//      T_Wall      - textured white
//      T_Frame     - wooden frame of the corkboard
//      T_Board     - filler of the corkboard
//===============
#declare T_Page01 =
texture {
    pigment{
        image_map { png "2-2-4.png" 
            map_type 0 //planar
            once
            interpolate 2
        }
    }
} //T_Page01
            
#declare T_Page02 =
texture {
    pigment{
        image_map { png "chemical-bond.png" 
            map_type 0 //planar
            once
            interpolate 2
        }
    }
} //T_Page02

#declare T_Page03 =
texture {
    pigment{
        image_map { png "covalent-bond.png" 
            map_type 0 //planar
            once
            interpolate 2
        }
    }
} //T_Page03

#declare T_Desktop = 
texture { T_Wood15
    finish {reflection 0.1 phong 0.6 phong_size 90}
    rotate y*90
    scale 1.5
} //T_Desktop

#declare T_Wall = 
texture {
    pigment { White }
    normal { bozo 0.5 scale 0.2 }
    finish{ phong 0.4 phong_size 20 }
} // T_Wall

#declare T_Frame = 
texture { T_Wood1
    finish {phong 0.6 phong_size 90}
    scale 0.75
} //T_Frame

#declare T_Board =
texture {
    pigment {Gold}
    normal { bozo 0.25 scale 0.5}
} //T_Board    

//===============
// Models
//      Page01      - when I tried one page model texured
//      Page02        three different times, it didn't work.
//      Page03        So, three models it is.
//      Desktop     - a clipped plane for a stage
//      Walls       - planes on 2 sides
//      Board       - a corkboard piece with frame, pushpins, and holes
//===============
#declare tscale=3;
#declare xscale=8.5*tscale;
#declare yscale=11*tscale;

#declare Page01 =
plane {z,0
    clipped_by {box { <0,0,-1> <1,1,1> }}
    texture { T_Page01 }
    scale <xscale,yscale,1>
    translate <-xscale/2,-yscale/2,0>
} // Page01

#declare Page02 =
plane {z,0
    clipped_by {box { <0,0,-1> <1,1,1> }}
    texture { T_Page02 }
    scale <xscale,yscale,1>
    translate <-xscale/2,-yscale/2,0>
} // Page02

#declare Page03 =
plane {z,0
    clipped_by {box { <0,0,-1> <1,1,1> }}
    texture { T_Page03 }
    scale <xscale,yscale,1>
    translate <-xscale/2,-yscale/2,0>
} // Page03

#declare Desktop = 
plane {y,0
    texture { T_Desktop }
} //Desktop

#declare Walls = 
union {
    plane{x,25}
    plane{z,25}
    texture { T_Wall }
} // Walls

#declare Board =
union {
    box { <-25, 5, 25>, <15, 7, 24.5>
        clipped_by { plane {x, 0 rotate z*45 translate <15, 5, 0> }}
        texture { T_Frame rotate y*90 scale 1}
    }

    difference { // cut out the attribution
        box { <13, 5, 25>, <15, 40, 24.5>
            clipped_by { plane {x, 0 rotate z*-135 translate <15, 5, 0> }}
            texture { T_Frame rotate x*90 scale 3}
        }           
        text { ttf "cyrvetic.ttf", "(cc) dbn3 2005", .3, 0 //crystal.ttf
            rotate <0,0,90>
            scale 0.5
            translate <14.5, 6.5, 24.4>
            texture {
                pigment { Black }
                //finish { ambient 0.1 diffuse 0.2 phong 0.1 phong_size 2.5 }
            } 
        } // text
    } //difference    

    difference { // punch out old pushpin holes 
        box { <-25, 7, 25>, <15, 40, 24.75>
            texture {T_Board}
        }
        #declare I=0;
        #while (I<60)
            cylinder { z*0, z*1, 0.1 translate <-16+(30*rand(R1)), 8+(4*rand(R1)), 24> texture {T_Board}}
            #declare I=I+1;
        #end    
    } // difference
    
    // add a few random pins
    #declare I=0;
        #while (I<6)
            cylinder { z*0, z*-0.75, 0.25 translate <-16+(30*rand(R1)), 8+(4*rand(R1)), 24.75> pigment {Green}}
            #declare I=I+1;
    #end
    #declare I=0;
        #while (I<4)
            cylinder { z*0, z*-0.75, 0.25 translate <-16+(30*rand(R1)), 8+(4*rand(R1)), 24.75> pigment {Yellow}}
            #declare I=I+1;
    #end
} // Board
 

//===============
// Object Placement
//===============

object {Page01 rotate <90,25,0> translate <0,0.02,-13>}
object {Page02 rotate <90,0,0> translate <-10,0.01,-15>}
object {Page03 rotate <90,20,0> translate <0,0.03,-22>}
object {Desktop}
object {Walls}
object {Board}

// Something's desparately wrong with this, but I'm not sure what.
union {
    object {Cup} 
    object {Pencil rotate x*90 rotate y*-110 rotate z*10 rotate x*3 translate <-10, 31, 0>}
    object {Pencil rotate x*90 rotate y*-140 rotate x*-10 rotate z*-3 translate <0, 31, -10>}
    object {Pencil rotate x*90 rotate y*-130 rotate x*-10 rotate z*10 translate <-7.5, 31, -7.5>}
    scale <0.35, 0.45, 0.35>
    rotate y*20
    translate <19, 0, 20>
}

object {Octane rotate x*-55  rotate y*50 translate <13,2,13>}
      
// Loose model parts
union {
	object { E_Carbon
		rotate <360*rand(R1),360*rand(R1),360*rand(R1)>
		translate <-13,1,22>}
	object { E_Carbon 
		rotate <360*rand(R1),360*rand(R1),360*rand(R1)>
		translate <-7,1,15>}
	object { E_Carbon 
		rotate <360*rand(R1),360*rand(R1),360*rand(R1)>
		translate <-3,1,20>}
	object { E_Carbon 
		rotate <360*rand(R1),360*rand(R1),360*rand(R1)>
		translate <0,1,18>}
	object { E_Oxygen
		rotate <360*rand(R1),360*rand(R1),360*rand(R1)>
		translate <-15,1,20>}
	object { E_Oxygen
		rotate <360*rand(R1),360*rand(R1),360*rand(R1)>
		translate <-12,1,13>}
	object { E_Oxygen
		rotate <360*rand(R1),360*rand(R1),360*rand(R1)>
		translate <-10,1,21>}
	object { E_Oxygen
		rotate <360*rand(R1),360*rand(R1),360*rand(R1)>
		translate <-3,1,15>}
    object { Bond
		rotate y*360*rand(R1)
		translate <-15,.2,9>}
	object { Bond
		rotate y*360*rand(R1)
		translate <-13,.2,8>}
	object { Bond
		rotate y*360*rand(R1)
		translate <-10,.2,10>}
	object { Bond
		rotate y*360*rand(R1)
		translate <-9,.2,8>}
	object { Bond
		rotate y*360*rand(R1)
		translate <-7,.2,10>}
	object { Bond
		rotate y*360*rand(R1)
		translate <-5,.2,6>}
	object { Bond
		rotate y*360*rand(R1)
		translate <-2,.2,8>}			
}