// Persistence of Vision Ray Tracer Scene Description File
// File: sphere.pov
// Vers: 3.6
// Desc: scene file for "Marble"
// Date: August 26th, 2005
// Auth: Philip Chan

#include "consts.inc"
#include "isotile.inc"

#declare A_Media = true;    // atmospheric media
#declare Q_Photon = true;   // photon quality
#declare Q_Blur = true;     // focal blur quality
#declare Bubbles = true;    // place bubbles
#declare Scratches = true;  // place scratches
#declare Chips = true;      // place chips
#declare A_Light = true;    // use area light
#declare I_Detail = true;  // interior detail
// mode of bubble generation
#declare BM_Union = 0;
#declare BM_Diff = 1;
#declare Bubble_Mode = BM_Union;

#declare Sphere_Loc = <0, 1, 8>;  // location of sphere
#declare N_Bubbles = 50*Bubbles;  // number of bubbles
#declare N_Scratches = 30*Scratches;  // number of scratches
#declare N_Chips = 15*Chips;  // number of chips
#declare N_Components = 5;  // number of blob components in a chip
#declare N_Tile_init = 4;   // number of tiles in the initial row
#declare Tile_init_z = 2.5; // z-position of centre of tile in initial row
#declare N_Tile_rows = 15;  // number of rows of tiles

// global settings
global_settings {
  assumed_gamma 1
  photons {
    #if ( Q_Photon )
      count 120000
    #else
      count 1000
    #end
  }
}

// atmospheric media
#if ( A_Media )
  media {
    scattering { 4, <0.7, 0.7, 0.6> }
    density {
      spotted
      turbulence 0.2
      density_map {
        [0.00 colour rgb <0.030, 0.030, 0.015>]
        [0.50 colour rgb <0.015, 0.015, 0.005>]
        [1.00 colour rgb <0.030, 0.030, 0.015>]
      }
      scale <2, 3, 1>/4
      rotate z*degrees(atan(2/3))
    }
  }
#end

// area light vector calculations (so the plane of the light is perpendicular to the vector to the marble
#declare L_vect = <-10, -10, -4>;
#declare L_axis1 = vnormalize( vcross( L_vect, y ) );
#declare L_axis2 = vnormalize( vcross( L_vect, L_axis1 ) );
// area light spotlight on marble
light_source {
  Sphere_Loc - L_vect
  color rgb <1, 1, 1>
  #if ( A_Light )
    area_light
    3*L_axis1, 3*L_axis2, 5, 5
    jitter
    adaptive 3
  #end
  spotlight
  radius 5
  falloff 10
  tightness 10
  point_at Sphere_Loc
}

// front fill light
light_source {
  Sphere_Loc + <-15, 5, -10>
  colour rgb <0.1, 0.1, 0.1>
  shadowless
  media_interaction off
  fade_distance 13
  fade_power 2 
}

// camera
camera {
  location <0, 3, 0>
  look_at <0, 0, 12>
  focal_point Sphere_Loc
  aperture 0.18
  #if ( Q_Blur )
    blur_samples 100
    confidence 0.98
    variance 1/512
  #else
    blur_samples 25
    confidence 0.75
    variance 1/64
  #end  
}

// floor light group
#declare F_Tile = finish {
  ambient 0 // ambient zero for fade to total blackness at background
  specular 0.3
  roughness 0.075
}

light_group {
  // fill light for floor
  light_source {
    Sphere_Loc + <10, 10, 4>
    colour rgb <0.25, 0.25, 0.25>
    fade_distance 15
    fade_power 1.5
    media_interaction off
    photons {
      reflection off
      refraction off
    }
  }
  
  // floor plane (far), checkerboard with finish used on tiles
  difference {
    plane { y, -0.05
      pigment {
        checker
        colour rgb <1, 1, 1>
        colour rgb <0, 0, 0>
      }
      // slight
      finish { F_Tile }
    }
    plane { z, N_Tile_rows + floor(Tile_init_z) }
  }
  // floor plane (grout)
  difference {
    plane { y, -0.05
      pigment { colour rgb <1, 1, 1> }
      finish { ambient 0 } // ambient zero for fade to total blackness at background
    }
    plane { -z, -( N_Tile_rows + floor(Tile_init_z) ) }
  }
  
  // tiles
  #declare j = 0; // row counter
  #while( j < N_Tile_rows )
    // number of tiles per row increases by one on each side per step along -z (some tiles are still out of sight, but not too many)
    #declare N_in_row = N_Tile_init + j;
    #declare i = -N_in_row; // tile counter
    #while ( i < N_in_row )
      object {
        Isotile()
        scale <0.975, 0.5, 0.975> // scale to make flatter, and allow a little room for grout to show at edges
        pigment { colour rgb mod(abs(i+j), 2) } // colour is either black or white
        finish { F_Tile }
        translate <i+0.5, -0.05, j + Tile_init_z>
      }
      #declare i = i + 1;
    #end
    #declare j = j + 1;
  #end
  
  global_lights on
}

// marble
// union or difference operation based on mode of bubble generation
#switch ( Bubble_Mode )
  #case ( BM_Union )
    union {
  #break
  #case ( BM_Diff )
    difference {
  #break
#end
  difference {
    sphere {
      <0, 0, 0>, 1
    }
    // scratches
    #declare i = 0;
    #declare R_Scratches = seed( 981794 );
    #while ( i < N_Scratches )
      #declare height = 0.9*sqrt(rand(R_Scratches));
      #declare maj_rad = sqrt( 1 - height*height );
      intersection {
        torus {
          maj_rad + 0.01*rand(R_Scratches), 0.0075 + 0.005*rand(R_Scratches)
          scale <1, 0.5 + 0.5*rand(R_Scratches), 1>
        }
        cone {
          <0, 0, -0.75*maj_rad>, 0,
          <0, 0, -1.25*maj_rad>, 0.2/sqrt(maj_rad) + (0.3/sqrt(maj_rad))*rand(R_Scratches)
        }
        translate y*height
        //rotate <360*rand(R_Scratches), 360*rand(R_Scratches), 360*rand(R_Scratches)>
        rotate -x*degrees(asin(height))
        rotate 180*rand(R_Scratches)*z
        rotate (-80 + 160*rand(R_Scratches))*y
        rotate (-35 + 235*rand(R_Scratches))*x
      }
      #declare i = i + 1;
    #end
    // chips
    #if ( Chips )
      blob {
        #declare i = 0;
        #declare R_Chips = seed( 56487 );
        #while ( i < N_Chips )
          #declare R_vect = <-35 + 235*rand(R_Chips), -90 + 135*rand(R_Chips), 180*rand(R_Chips)>;
          #declare j = 0;
          #while ( j < N_Components )
            sphere {
              vrotate( <0, 0, -1 - 0.02*rand(R_Chips)>, R_vect + <(rand(R_Chips)-0.5), (rand(R_Chips)-0.5), (rand(R_Chips)-0.5)>*3 ),
              0.02 + 0.03*rand(R_Chips), 1.0
            }
            #declare j = j + 1;
          #end
          #declare i = i + 1;
        #end
      }
    #end
  }
  // generate bubbles
  #declare i = 0;
  #declare R_Bubbles = seed( 52179 );
  #while ( i < N_Bubbles )
    // generate position using rotations, sqrt function on distance to generate less bubbles near the centre
    #declare Pos = vrotate( vrotate( 0.93*sqrt(rand(R_Bubbles))*z, 360*rand(R_Bubbles)*y ), 360*rand(R_Bubbles)*x );
    sphere {
      <0, 0, 0>, min( 0.01 + 0.03*pow(rand(R_Bubbles), 3), 0.95 - vlength(Pos) )
      // interior statement needed for Union bubbles
      #switch ( Bubble_Mode )
        #case ( BM_Union )
          interior { ior 1.0 }
        #break;
      #end
      scale <0.9 + 0.1*rand(R_Bubbles), 0.9 + 0.1*rand(R_Bubbles), 0.9 + 0.1*rand(R_Bubbles)>
      translate Pos
    }
    #declare i = i + 1;
  #end
  // texture the marble
  //hollow
  // base texture
  texture {
    pigment {
      colour rgbft <0.8, 1.0, 0.8, 0.5, 0.5>
    }
    finish {
      reflection {
        0.01, 0.15
        fresnel on
      }
      conserve_energy
      specular 0.5
      roughness 0.003
    }
  }
  // fading texture (small spots)
  texture {
    pigment {
      bozo
      turbulence 0.5
      colour_map {
        [0.00 colour rgbt <0.75, 0.85, 0.75, 1.0>]
        [0.50 colour rgbt <0.75, 0.85, 0.75, 1.0>]
        [1.00 colour rgbt <0.75, 0.85, 0.75, 0.9>]
      }
    }
    scale 0.05
  }
  // fading texture (large spots)
  texture {
    pigment {
      bozo
      turbulence 0.3
      colour_map {
        [0.00 colour rgbt <0.65, 0.75, 0.65, 1.0>]
        [0.50 colour rgbt <0.65, 0.75, 0.65, 1.0>]
        [1.00 colour rgbt <0.65, 0.75, 0.65, 0.8>]
      }
    }
    scale 0.3
  }
  // interior properties
  interior {
    // physical properties
    ior 1.5
    #if ( I_Detail )
      dispersion 1.005
      fade_colour <0.3, 0.5, 0.3>
      fade_distance 0.5
      fade_power 2
    #end
  }
  // photon block for caustics
  photons {
    target
    refraction on
    collect off
  }
  scale <1, 0.93, 1>
  translate Sphere_Loc
}
