// Box Collision Detector (2D)
// The y axis is ignored in collisions.
// (Boxes above each other will collide)
// Rotation must be around the y axis.
// Returns either
//   true; (collision)
//   false; (no collision)

#include "math.inc"

#macro bCD2D(
          boxOneLocation,
          boxTwoLocation,
          boxOneScale,
          boxTwoScale,
          boxOneRotation,
          boxTwoRotation
          )

     // Create bounding boxes for each box and test to
     // see if they are overlapping.
     
     #local boxOne = 
          box {
               -0.5,0.5
               scale boxOneScale
               rotate y*boxOneRotation
               translate boxOneLocation
               }
     #local boxTwo = 
          box {
               -0.5,0.5
               scale boxTwoScale
               rotate y*boxTwoRotation
               translate boxTwoLocation
               }
               
     #local boundingOneMin = min_extent(boxOne);
     #local boundingOneMax = max_extent(boxOne);
     #local boundingTwoMin = min_extent(boxTwo);
     #local boundingTwoMax = max_extent(boxTwo);                    
               
     // Determine if the bounding boxes are touching.
     
     #if (boundingOneMin.x >= boundingTwoMax.x |
          boundingOneMax.x <= boundingTwoMin.x |
          boundingOneMin.z >= boundingTwoMax.z |
          boundingOneMax.z <= boundingTwoMin.z )
          
          // The boxes cannot be overlaping.
          
          false;
          
     #else
     
          // Test to see if the boxes are too close
          // to not be touching.
          
          #local minDistToFaceOne = 
               min(boxOneScale.x,boxOneScale.z) / 2; 
          #local minDistToFaceTwo = 
               min(boxTwoScale.x,boxTwoScale.z) / 2;
          
          #local distBetweenBoxes =
               sqrt(pow((abs(boxOneLocation.x - boxTwoLocation.x)),2) +
                    pow((abs(boxOneLocation.z - boxTwoLocation.z)),2));
          
          #if (distBetweenBoxes < (minDistToFaceOne + minDistToFaceTwo))
          
               true;
               
          #else
          
               // At this point, intersection detection is 
               // required to know if the boxes are overlapping.
               
               // Cram all the points on each of the boxes into an array
               //
               //   3-----0   3-----0
               //   |     |   |     |  
               //   |     |   |     |            
               //   | bx1 |   | bx2 |
               //   |     |   |     |   
               //   |     |   |     |
               //   2-----1   2-----1
               //
               
               #local pointArray = array[2][4];
               
               #local pointArray[0][0] = vaxis_rotate(<0.5,0,0.5> * boxOneScale,y,boxOneRotation) + boxOneLocation;
               #local pointArray[0][1] = vaxis_rotate(<0.5,0,-.5> * boxOneScale,y,boxOneRotation) + boxOneLocation;
               #local pointArray[0][2] = vaxis_rotate(<-.5,0,-.5> * boxOneScale,y,boxOneRotation) + boxOneLocation;
               #local pointArray[0][3] = vaxis_rotate(<-.5,0,0.5> * boxOneScale,y,boxOneRotation) + boxOneLocation;
               #local pointArray[1][0] = vaxis_rotate(<0.5,0,0.5> * boxTwoScale,y,boxTwoRotation) + boxTwoLocation;
               #local pointArray[1][1] = vaxis_rotate(<0.5,0,-.5> * boxTwoScale,y,boxTwoRotation) + boxTwoLocation;
               #local pointArray[1][2] = vaxis_rotate(<-.5,0,-.5> * boxTwoScale,y,boxTwoRotation) + boxTwoLocation;
               #local pointArray[1][3] = vaxis_rotate(<-.5,0,0.5> * boxTwoScale,y,boxTwoRotation) + boxTwoLocation; 
               
               // Compare each line in box 1 to each line in box 2
               // Determine where an intersection would occur, and 
               // then determine if that intersection would occur 
               // within the span of the lines.
               
               #local intersectionDetected = false;
               #local boxOneLineIndex = 0;
               #while (boxOneLineIndex < 4 & !intersectionDetected)
               
                    // Declare the points of the line.
                    
                    #local pointOne = pointArray[0][boxOneLineIndex];
                    #local pointTwo = pointArray[0][mod(boxOneLineIndex+1,4)];
                    
                    // Determine the slope of the line.
                    // (An infinite slope will be undefined)
                    
                    #if ((pointOne.x - pointTwo.x) != 0)
                         #local boxOneLineSlope = (pointOne.z - pointTwo.z) / (pointOne.x - pointTwo.x);
                    #end 
                    
                    #local boxTwoLineIndex = 0;
                    #while (boxTwoLineIndex < 4 & !intersectionDetected)
                    
                         // Declare the points of the line.
                         
                         #local pointThree = pointArray[1][boxTwoLineIndex];
                         #local pointFour = pointArray[1][mod(boxTwoLineIndex+1,4)];
                         
                         // Determine the slope of the line.
                         // (An infinite slope will be undefined)
                         
                         #if ((pointThree.x - pointFour.x) != 0)
                         #local boxTwoLineSlope = 
                              (pointThree.z - pointFour.z) / (pointThree.x - pointFour.x);
                         #end
                         
                         // BEGIN INTERSECTION DETECTION:
                         
                         // Find the intersection point.
                         
                         #if (!defined(boxOneLineSlope) & !defined(boxTwoLineSlope))
                              // Two infinite slopes
                              #if (pointOne.x = pointThree.x)
                                   true;
                                   #local intersectionDetected = true;
                              #end      
                         #else #if (!defined(boxOneLineSlope) | !defined(boxTwoLineSlope))
                              // One infinite slope
                              #if (!defined(boxOneLineSlope))
                                   // Box one has the undefined slope
                                   // y - mx = b
                                   // Find b for box two's line
                                   #local thisB = pointThree.z - (boxTwoLineSlope * pointThree.x);
                                   // find the y value of line two when it crosses 
                                   // box one's line
                                   // y = mx + b
                                   #local thisY = (boxTwoLineSlope * pointOne.x) + thisB;
                                   #local intersectionPoint = <pointOne.x,0,thisY>;
                              #else
                                   // Box two has the undefined slope
                                   // y - mx = b
                                   // Find b for box one's line
                                   #local thisB = pointOne.z - (boxOneLineSlope * pointOne.x);
                                   // find the y value of line one when it crosses 
                                   // box two's line
                                   // y = mx + b
                                   #local thisY = (boxOneLineSlope * pointThree.x) + thisB;
                                   #local intersectionPoint = <pointThree.x,0,thisY>;
                              #end            
                         #else
                              // No initinite slopes.
                              // Find the b values for each line
                              // y - mx = b
                              #local lineOneB = pointOne.z - (boxOneLineSlope * pointOne.x);
                              #local lineTwoB = pointThree.z - (boxTwoLineSlope * pointThree.x); 
                              
                              // Find the x coordinate of the intersection point
                              // m1*x + b1 = m2*x + b2
                              // (m1-m2)*x = b2 - b1
                              // x = (b2-b1)/(m1-m2)
                              #local thisX = (lineTwoB - lineOneB) / (boxOneLineSlope - boxTwoLineSlope);     //Divided by Zero! Fix!
                              // Find the y value
                              #local thisY = (boxOneLineSlope * thisX) + lineOneB;
                              // Put them together to get the intersection point.
                              #local intersectionPoint = <thisX,0,thisY>;
                              
                         #end #end          
                         
                         // Unless an intersection has been found, use the 
                         // intersectionPoint to determine if there is an 
                         // intersection.
                         #if (!intersectionDetected)
                              
                              #if (!defined(boxOneLineSlope) & !defined(boxTwoLineSlope))
                              #else       
                              // Does the intersection point fall within the bounds 
                              // of the two lines?
                              #if (intersectionPoint.z >= min(pointOne.z,pointTwo.z) &
                                   intersectionPoint.z >= min(pointThree.z,pointFour.z) &
                                   intersectionPoint.z <= max(pointOne.z,pointTwo.z) &
                                   intersectionPoint.z <= max(pointThree.z,pointFour.z))
                                   true;
                                   #local intersectionDetected = true;

                              #end
                              #end               
                                   
                         #end                           
                         
                         // Update the index value and clear the slope.
                         
                         #local boxTwoLineIndex = boxTwoLineIndex + 1;
                         #ifdef (boxTwoLineSlope) #undef boxTwoLineSlope #end
                         #ifdef (slopesAreEqual) #undef slopesAreEqual #end
                         #ifdef (intersectionPoint) #undef intersectionPoint #end
                         #ifdef (thisB) #undef thisB #end
                         #ifdef (thisY) #undef thisY #end
                         #ifdef (thisX) #undef thisX #end 
                         #ifdef (lineOneB) #undef lineOneB #end
                         #ifdef (lineTwoB) #undef lineTwoB #end
                         
                    #end
                    
                    // Update the index value and clear the slope.
                    
                    #local boxOneLineIndex = boxOneLineIndex + 1;
                    #ifdef (boxOneLineSlope) #undef boxOneLineSlope #end
                    
               #end
                        
               #if (!intersectionDetected)
                    false;
               #end
                          
          #end                                             
          
     #end        
     
#end 