// The City
// Written by Ian Mcnamara

#include "lightdistribution.inc" 
#include "math.inc" 

// lightcolor_memeory.txt file must be present...
// It can be generated using City_Placement.pov

// THE SCENE //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////// The City Blocks ////////////////////////////////////////////////////////
#declare showCityBlocks = 1; //
#if (showCityBlocks)            //
//////////////////////////////////

#declare blockTranslate = <0,1,0>;
#declare splineCables = 1;
#declare numberOfPoints = 25;
#declare power = 1.02;

// The first step of showing the blocks is to read
// their information in from the memory file.
#fopen readMemory "lightcolor_memory.txt" read
#read (readMemory,numberOfBlocks)  
#read (readMemory,cableDistFromSide) 
#read (readMemory,cableDistFromTop)
#read (readMemory,cableRadius)
#declare cableRadius = 0.02;  //HACK HACK HACK HACK HACK
#read (readMemory,tackRadius)

#declare blockInfoArray = array[numberOfBlocks][3];
#declare blockTackArray = array[numberOfBlocks*4][3];
#declare i = 0;
#while (i < numberOfBlocks)
     #read (readMemory,readLocation,readScale,readRotation)
     #declare blockInfoArray[i][0] = readLocation;   
     #declare blockInfoArray[i][1] = readScale;
     #declare blockInfoArray[i][2] = readRotation;
     #declare j = 0;
     #while (j < 4)
          #read (readMemory,readTL,readTS,readTR)
          #declare blockTackArray[(i*4)+j][0] = readTL;
          #declare blockTackArray[(i*4)+j][1] = readTS;
          #declare blockTackArray[(i*4)+j][2] = readTR;
          #declare j = j + 1;
     #end       
     #declare i = i + 1;
#end 
#fclose readMemory
/* Create and fill a cable array (start and end points) */
#declare blockCableArray = array[numberOfBlocks*4][2];
#declare i = 0;
#while (i < numberOfBlocks)
     /* Determine locations of block and tack contact */
          // The following contact points should be relative
          // to the location of the block. (sans translation)
     #declare blockContact = 
          <(blockInfoArray[i][1].x/2)-cableDistFromSide,
          blockInfoArray[i][1].y-cableDistFromTop,
          blockInfoArray[i][1].z/2>;
     #declare tackRotatedLocation = blockTackArray[i*4][0] - blockInfoArray[i][0];
     #declare tackLocation = vaxis_rotate(tackRotatedLocation,y,-blockInfoArray[i][2].y);
     #declare tackRadiusVector = <0,0,-(blockTackArray[i*4][1].x / 2)>;
     #declare tackRadiusVector = vaxis_rotate(tackRadiusVector,y,blockTackArray[i*4][2].y);
     #declare tackContact = tackLocation + tackRadiusVector;
     /* Plug the values into an array */
          // These locations still have not been modified
          // to account for block rotation and location.
     #declare blockCableArray[(i*4)+0][0] = (blockContact * <1,1,1>);
     #declare blockCableArray[(i*4)+0][1] = (tackContact * <1,1,1>);
     #declare blockCableArray[(i*4)+1][0] = (blockContact * <1,1,-1>);
     #declare blockCableArray[(i*4)+1][1] = (tackContact * <1,1,-1>);
     #declare blockCableArray[(i*4)+2][0] = (blockContact * <-1,1,-1>);
     #declare blockCableArray[(i*4)+2][1] = (tackContact * <-1,1,-1>);
     #declare blockCableArray[(i*4)+3][0] = (blockContact * <-1,1,1>);
     #declare blockCableArray[(i*4)+3][1] = (tackContact * <-1,1,1>);
     /* Modify the values */
          // Adjust the values to match the rotation 
          // and location of the block. 
     #declare j = 0;     
     #while (j < 4) 
          #declare k = 0;
          #while (k < 2)
               #declare blockCableArray[(i*4)+j][k] = 
                    vaxis_rotate(blockCableArray[(i*4)+j][k],y,blockInfoArray[i][2].y);
               #declare blockCableArray[(i*4)+j][k] =
                    blockCableArray[(i*4)+j][k] + blockInfoArray[i][0];      
               #declare k = k + 1;
          #end
          #declare j = j + 1;
     #end               
     #declare i = i + 1;
#end

// Next, define the textures to be used by the blocks.

#declare blockWhite = 
     texture { 
          pigment { rgb 1 }
          finish {
               ambient 0 
               diffuse 0.9
               }
          }
#declare blockGreen = 
     texture {
          pigment { rgb <0.4,1,0.4> }
          finish { 
               ambient 0.3
               diffuse 2.9
               }
          }
#declare blockRed = 
     texture {
          pigment { rgb <1,0.4,0.4> }
          finish { 
               ambient 0.6
               diffuse 2.9
               }
          }          
// Define a Tack object
#declare tack = 
     intersection {
          cylinder {
               <0,0,0>, <0,0.02,0>, 0.5
               }
          sphere {
               <0,0.02-10,0>, 10
               }
          }
#declare tackTexture =
     texture { 
          pigment { rgb 0.8 }
          finish { 
               ambient 0 
               }
          }         
// Define a Cable texture
#declare cableTexture =
     texture {
          pigment { rgb 0.2 }
          finish { ambient 0 }
          }
                                                             
// Because the blocks have different colors on different
// sides, they will be made using meshes.  The mesh will
// be assembled separately for each block so that their textures
// will not be stretched.
// blockTranslate can be used to translated all the blocks
// by the same amount.   
// Place the blocks and their components:
#declare i = 0;
#while (i < numberOfBlocks)
     // The mesh blocks
     mesh {
          //-x
          triangle { <-0.5,1,-0.5>,<-0.5,0,-0.5>,<-0.5,0,0.5> } 
          triangle { <-0.5,1,-0.5>,<-0.5,1,0.5>,<-0.5,0,0.5> }
          //+x
          triangle { <0.5,1,-0.5>,<0.5,0,-0.5>,<0.5,0,0.5> } 
          triangle { <0.5,1,-0.5>,<0.5,1,0.5>,<0.5,0,0.5> }
          //+y
          triangle { <0.5,1,0.5>,<0.5,1,-0.5>,<-0.5,1,-0.5> }
          triangle { <0.5,1,0.5>,<-0.5,1,0.5>,<-0.5,1,-0.5> }
          scale blockInfoArray[i][1]
          #if (blockInfoArray[i][1].y > 12)
               texture { blockRed }
          #else
               texture { blockGreen }
          #end
          rotate blockInfoArray[i][2]
          translate blockInfoArray[i][0]
          translate blockTranslate
          }
     mesh {
          //-z
          triangle { <0.5,1,-0.5>,<0.5,0,-0.5>,<-0.5,0,-0.5> }
          triangle { <0.5,1,-0.5>,<-0.5,1,-0.5>,<-0.5,0,-0.5> }
          //+z
          triangle { <0.5,1,0.5>,<0.5,0,0.5>,<-0.5,0,0.5> }
          triangle { <0.5,1,0.5>,<-0.5,1,0.5>,<-0.5,0,0.5> }
          scale blockInfoArray[i][1]
          texture { blockWhite }
          rotate blockInfoArray[i][2]
          translate blockInfoArray[i][0]
          translate blockTranslate
          }           
     #declare j = 0;
     #while (j < 4)
          // The Tacks: 
          object {
               tack
               scale blockTackArray[(i*4)+j][1]
               texture { tackTexture }
               rotate y*blockTackArray[(i*4)+j][2].y
               translate blockTackArray[(i*4)+j][0]
               translate blockTranslate
               }
          // The Cable:
          #if (splineCables)
               #declare cableSpline = 
                    spline {
                         linear_spline
                         0, blockCableArray[(i*4)+j][0] + blockTranslate
                         1, blockCableArray[(i*4)+j][1] + blockTranslate
                         }
               #declare dy = cableSpline(0).y - cableSpline(1).y;
               #declare minY = cableSpline(1).y;
               #declare numberOfPoints = numberOfPoints;
               #declare cableSplineArray = array[numberOfPoints+1];
               #declare k = 0;
               #while (k < numberOfPoints)
                    #declare cableSplineArray[k] = cableSpline((1/numberOfPoints)*k);
                    #declare thisY = cableSplineArray[k].y;
                    #declare thisYPercent = (thisY - minY) / dy;
                    #declare newY = (thisY - minY) * pow(thisYPercent,power) + minY;
                    #declare cableSplineArray[k] = 
                         cableSplineArray[k] * <1,0,1> + <0,newY,0>;
                    #declare k = k + 1;
               #end
               #declare cableSplineArray[numberOfPoints] = cableSpline(1);
               #declare k = 0;
               #while (k < numberOfPoints)
                    sphere { 
                         cableSplineArray[k], cableRadius 
                         texture { cableTexture }
                         }
                    cylinder {
                         cableSplineArray[k], cableSplineArray[k+1], cableRadius
                         texture { cableTexture }
                         }
                    #declare k = k + 1;
               #end
               sphere {
                    cableSplineArray[numberOfPoints], cableRadius
                    texture { cableTexture }
                    }                                                                                                                                        
          #else      
          cylinder {
               blockCableArray[(i*4)+j][0],
               blockCableArray[(i*4)+j][1],
               cableRadius
               texture { cableTexture }
               translate blockTranslate
               }
          #end       
          #declare j = j + 1;
     #end                       
     #declare i = i + 1;
#end 
 
///////
#end //
///////////////////////////////////////////////////////// The Platform /////////////////////////////////////////////////////////////
#declare showPlatform = true; //
#if (showPlatform)            //
////////////////////////////////

// Create the texture(s) for the city block platform
#declare platformWhite = 
     texture {
          pigment { rgb 1 }
          finish {
               ambient 0 
               diffuse 0.7
               }
          }

// Create the platform
#declare pHeight = 1;
#declare pRadius = 10;
#declare pEdge = 0.05;
#declare platform = 
     union {
          cylinder {
               <0,0,0>, <0,pHeight-pEdge,0>, pRadius
               texture { platformWhite }
               }
          cylinder {
               <0,pHeight-pEdge,0>, <0,pHeight,0>, pRadius-pEdge
               texture { platformWhite }
               }
          torus {
               pRadius-pEdge, pEdge
               texture { platformWhite }
               translate <0,pHeight-pEdge,0>
               }
          }
     // This Platform's surface is centered
     // at <0,1,0> and has a useable radius
     // of 4.95 units.
     
// Place the platform
object {
     platform
     translate <0,0,0>
     } 
     
///////
#end //
///////////////////////////////////////////////////////// The Floor ///////////////////////////////////////////////////////////// 
#declare showFloor = 1; //
/////////////////////////////

#declare radiusOfPlatform = 10; // Make sure this matches value above 
#declare widthOfInnerRing = 2; 
#declare widthOfWalkingCircle = 4; 
#declare distanceToFarWall = 100;  // From center of platform (x)
#declare distanceToNearWall = 30; // (z)
#declare widthOfWalkway = 5;

#declare fS = seed(1);

//////////////////
#if (showFloor) //
//////////////////
 
// Declare some textures for the floor  
#declare blurSamples = 30; 
#declare tilePigment_01 = 
     pigment {
          agate
          turbulence 1.1
          color_map {
               [ 0.00 rgb <235,235,235>/255 ]
               [ 0.80 rgb <200,200,200>/255 ]
               [ 1.00 rgb <135,130,130>/255 ]
               }
          scale 0.8
          rotate y*20              
          }
#declare tilePigment_02 = 
     pigment {
          marble
          turbulence 1.1
          color_map {
               [ 0.00 rgb <235,235,235>/255 ]
               [ 0.80 rgb <200,200,200>/255 ]
               [ 1.00 rgb <135,130,130>/255 ]
               }
          scale 0.5
          rotate y*-50              
          }          
#declare innerRingTexture = 
     texture {
          average 
          texture_map {
               #declare i = 0;
               #while ( i < blurSamples )
                    [ 
                    1 
                    pigment { 
                         checker
                         pigment { tilePigment_01 },
                         pigment { tilePigment_02 translate 100 }
                         } 
                    finish { 
                         ambient 0 
                         reflection 0.1
                         specular 0.2
                         }
                    normal { bumps 0.1 scale 0.5 translate rand(fS)*100 }
                    ] 
                    #declare i = i+1;
               #end
               }
          }

#declare tilePigment_03 = 
     pigment {
          granite
          turbulence 1.1
          color_map {
               [ 0.00 rgb <035,035,035>/255 ]
               [ 0.70 rgb <050,060,050>/255 ]
               [ 1.00 rgb <139,080,060>/355 ]
               }
          scale 0.5
          rotate y*-50              
          }  
#declare tilePigment_04 = 
     pigment {
          marble
          turbulence 1.1
          color_map {
               [ 0.00 rgb <035,035,035>/255 ]
               [ 0.70 rgb <050,060,050>/255 ]
               [ 1.00 rgb <139,080,060>/355 ]
               }
          scale 0.5
          rotate y*20              
          }                  
#declare walkwayTexture = 
     texture {
          average 
          texture_map {
               #declare i = 0;
               #while ( i < blurSamples )
                    [ 
                    1 
                    pigment { 
                         checker
                         pigment { tilePigment_03 },
                         pigment { tilePigment_04 translate 100 }
                         } 
                    finish { ambient 0 reflection 0.2}
                    normal { bumps 0.1 scale 0.5 translate rand(fS)*100 }
                    ] 
                    #declare i = i+1;
               #end
               }
          }
          
#declare outerFloorTexture = innerRingTexture;

// Put together the floor
#declare innerRingFloor = 
     cylinder {
          <0,-10,0>, <0,0,0>, radiusOfPlatform+widthOfInnerRing
          texture { innerRingTexture }
          }                      
#declare walkwayFloor =      
     difference {
          union {
               cylinder {
                    <0,-10,0>, <0,0,0>, radiusOfPlatform+widthOfInnerRing+widthOfWalkingCircle
                    texture { walkwayTexture }
                    }
               box {
                    <-distanceToFarWall,-10,-widthOfWalkway/2>,<distanceToFarWall,0,widthOfWalkway/2>
                    texture { walkwayTexture }
                    }
               }
          cylinder {
               <0,-11,0>, <0,1,0>, radiusOfPlatform+widthOfInnerRing
               texture { walkwayTexture }
               } 
          }       
#declare outerFloor = 
     difference {
          box {
               <-distanceToFarWall,-10,-distanceToNearWall>
               <distanceToFarWall,0,distanceToNearWall>
               texture { outerFloorTexture }
               }
          box {
               <-distanceToFarWall,-11,-widthOfWalkway/2>,<distanceToFarWall,1,widthOfWalkway/2>
               texture { outerFloorTexture }
               }
          cylinder {
               <0,-11,0>, <0,1,0>, radiusOfPlatform+widthOfInnerRing+widthOfWalkingCircle
               texture { outerFloorTexture }
               }
          }
          
#declare theFloor = 
     union {
          object { innerRingFloor }
          object { walkwayFloor }
          object { outerFloor }
          }
          
object { theFloor }
                           
///////
#end //
///////////////////////////////////////////////////////// The Bench /////////////////////////////////////////////////////////////////                                                                         
#declare showBench = 1; //
#declare frameOnly = 0; //
/////////////////////////////
          
#declare distFromCenter = 
     radiusOfPlatform + widthOfInnerRing + widthOfWalkingCircle + 2;
#declare pairsOfLegs = 1;     // Other than the pair on each end.
#declare legWidth = 0.3;
#declare legHeight = 1;
#declare legUnderBy = 0.1;
#declare seatHeight = 0.5;
#declare depthOfBench = 3;
#declare widthOfEdge = 0.2;
#declare widthOfCenterPieces = 0.15;
#declare numberOfCenterPieces = 10;
#declare numberOfSupportRays = 7;
#declare startDegree = 15;
#declare endDegree = 75;
#declare dividerOfCenterPieces = 1.5;     // seatHeight / this number

//////////////////
#if (showBench) //
//////////////////

// Set up the textures for the bench  
#include "woods.inc"
#declare woodTexture =  
     texture { T_Wood14 rotate x*6 rotate y*45 scale 0.5}
     /* 
     texture {
          pigment { rgb 1 }
          finish { ambient 0 }
          }*/
          
// Put together the bench

#declare S = seed(20);
#declare T = 100;                 
#declare benchObject =
     union {
          //End Legs
          #declare degreeToNextLeg = (endDegree - startDegree) / (pairsOfLegs + 1);
          #declare legObject = 
               box {
                    <-.5,0,-.5>, <0.5,1,0.5>
                    scale <legWidth,legHeight,legWidth>
                    //texture { woodTexture }
                    }
          object {  // Near Inside
               legObject
               translate <legWidth/2,0,(legWidth/2)+legUnderBy>
               translate <distFromCenter+legUnderBy,0,0>
               rotate y*-startDegree
               texture { woodTexture }
               }
          object {  // Near Inside
               legObject
               translate <-legWidth/2,0,(legWidth/2)+legUnderBy>
               translate <distFromCenter+depthOfBench-legUnderBy,0,0>
               rotate y*-startDegree
               texture { woodTexture }
               }
          object {  // Far Inside
               legObject
               translate <legWidth/2,0,-(legWidth/2)-legUnderBy>
               translate <distFromCenter+legUnderBy,0,0>
               rotate y*-endDegree
               texture { woodTexture }
               }
          object {  // Far Inside
               legObject
               translate <-legWidth/2,0,-(legWidth/2)-legUnderBy>
               translate <distFromCenter+depthOfBench-legUnderBy,0,0>
               rotate y*-endDegree
               texture { woodTexture }
               }
          // Additional Legs
          #declare i = 0;
          #while (i < pairsOfLegs)
               object {  //Extra inside
                    legObject
                    translate <legWidth/2,0,0>
                    translate <distFromCenter+legUnderBy,0,0>
                    rotate y*-(startDegree+degreeToNextLeg)
                    rotate y*-(degreeToNextLeg * i)
                    texture { woodTexture }
                    }
               object {  //Extra outside
                    legObject
                    translate <-legWidth/2,0,0>
                    translate <distFromCenter+depthOfBench-legUnderBy,0,0>
                    rotate y*-(startDegree+degreeToNextLeg)
                    rotate y*-(degreeToNextLeg * i)
                    texture { woodTexture }
                    }
               #declare i = i+1;
          #end
          // The Edges
          difference {   // Inner edge
               cylinder {
                    <0,legHeight,0>, <0,legHeight+seatHeight,0>,distFromCenter+widthOfEdge                    
                    }
               cylinder {
                    <0,legHeight-1,0>, <0,legHeight+seatHeight+1,0>,distFromCenter
                    }
               plane {
                    z,0
                    rotate y*-startDegree
                    } 
               plane {
                    -z,0
                    rotate y*-endDegree
                    }         
               texture { woodTexture translate <rand(S)*T,rand(S)*T,rand(S)*T> } 
               no_reflection
               no_shadow
               }
          difference {   // Outer edge
               cylinder {
                    <0,legHeight,0>, <0,legHeight+seatHeight,0>,distFromCenter+depthOfBench                    
                    }
               cylinder {
                    <0,legHeight-1,0>, <0,legHeight+seatHeight+1,0>,distFromCenter+depthOfBench-widthOfEdge
                    }
               plane {
                    z,0
                    rotate y*-startDegree
                    }
               plane {
                    -z,0
                    rotate y*-endDegree
                    }          
               texture { woodTexture translate <rand(S)*T,rand(S)*T,rand(S)*T> }
               no_reflection
               no_shadow
               }
          difference {   // Near and far edges
               union {        
                    intersection {
                         box {     
                              <0,legHeight,0>, <distFromCenter+depthOfBench,legHeight+seatHeight,widthOfEdge>
                              rotate y*-startDegree
                              }
                         cylinder {
                              <0,legHeight-1,0>, <0,legHeight+seatHeight+1,0>,distFromCenter+depthOfBench-widthOfEdge
                              }
                         }
                    intersection {
                         box {     
                              <0,legHeight,0>, <distFromCenter+depthOfBench,legHeight+seatHeight,-widthOfEdge>
                              rotate y*-endDegree
                              }
                         cylinder {
                              <0,legHeight-1,0>, <0,legHeight+seatHeight+1,0>,distFromCenter+depthOfBench-widthOfEdge
                              }
                         } 
                    }        
               cylinder {
                    <0,legHeight-1,0>, <0,legHeight+seatHeight+1,0>,distFromCenter+widthOfEdge
                    }
               texture { woodTexture translate <rand(S)*T,rand(S)*T,rand(S)*T> }
               no_reflection
               no_shadow
               }
          //Fake the supports for shadow/reflection 
          difference {   // Outer edge
               cylinder {
                    <0,legHeight,0>, <0,legHeight+(seatHeight/2),0>,distFromCenter+depthOfBench                    
                    }
               cylinder {
                    <0,legHeight-1,0>, <0,legHeight+seatHeight,0>,distFromCenter
                    }
               plane {
                    z,0
                    rotate y*-startDegree
                    }
               plane {
                    -z,0
                    rotate y*-endDegree
                    }          
               texture { woodTexture translate <rand(S)*T,rand(S)*T,rand(S)*T> }
               no_image
               }      
          #if (!frameOnly)     
          // The support rays (pieces that go from inside to out)
          #declare degreeToNextRay = (endDegree-startDegree) / (numberOfSupportRays+1);
          #declare i = 0;
          #while (i < numberOfSupportRays)
               difference {
                    intersection {
                         box {
                              <0,legHeight+(seatHeight - (seatHeight/dividerOfCenterPieces)),-widthOfCenterPieces/2>
                              <distFromCenter+depthOfBench,legHeight+seatHeight,widthOfCenterPieces/2>
                              }
                         cylinder {
                              <0,legHeight-1,0>, <0,legHeight+seatHeight+1,0>,distFromCenter+depthOfBench-widthOfEdge
                              }
                         }
                    cylinder {
                         <0,legHeight-1,0>, <0,legHeight+seatHeight+1,0>,distFromCenter+widthOfEdge
                         }
                    rotate y*-startDegree
                    rotate y*-(degreeToNextRay * (i+1))
                    translate <0,-0.01,0>
                    texture { woodTexture translate <rand(S)*T,rand(S)*T,rand(S)*T> }  
                    no_reflection
                    no_shadow
                    }
               #declare i = i+1;
          #end
          // The center pieces 
          #declare distToFill = depthOfBench - (widthOfEdge * 2);
          #declare distBetweenPieces = distToFill / (numberOfCenterPieces + 1);
          #declare i = 0;
          #while (i < numberOfCenterPieces)
               #declare thisRadius = distFromCenter+widthOfEdge+(distBetweenPieces*(i+1));
               difference {
                    cylinder {
                         <0,legHeight+(seatHeight - (seatHeight/dividerOfCenterPieces)),0>,
                         <0,legHeight+seatHeight,0>, thisRadius + (widthOfCenterPieces/2)
                         }
                    cylinder {
                         <0,legHeight-1,0>, 
                         <0,legHeight+seatHeight+1,0>,thisRadius - (widthOfCenterPieces/2)
                         }     
                    plane {
                         z,widthOfEdge
                         rotate y*-startDegree
                         }
                    plane {
                         -z,widthOfEdge
                         rotate y*-endDegree
                         }
                    translate <0,0.001,0>
                    texture { woodTexture translate <rand(S)*T,rand(S)*T,rand(S)*T> } 
                    no_reflection
                    no_shadow  
                    }
               #declare i = i+1;
          #end
          #end                                                                                    
          }

object { benchObject }
object { benchObject rotate y*90 }
object { benchObject translate <-0.2,0,1.5> rotate y*180 }
object { benchObject rotate y*270 }                                   

///////               
#end //
/////////////////////////////////////////////////// The Pamphlet ////////////////////////////////////////////////////////////////////
#declare showPamphlet = true; //
#if (showPamphlet)            //
////////////////////////////////

#declare pWidth = 1.31;
#declare pDepth = 1;

#declare pRaise = 0.3;   //Height is determined by adding pRaise
#declare pPower = 2.5;     //raised to pPower based on the x position

#declare nSamples = 100; //This number * 2 = total triangles in mesh.

// Do not touch
#declare dx = pWidth/nSamples;
#declare thirdWidth = pWidth/3;

#declare heightArray = array[nSamples+1];
#declare i = 0;
#while (i < nSamples)
     #declare thisX = i*dx;
     #if (thisX < thirdWidth)
          // First part
          #declare thisY_0 = mod(thisX,thirdWidth);
          #declare thisY_1 = thisY_0/thirdWidth;
          #declare thisY_2 = thisY_1*pRaise;
          #declare thisY = pow(thisY_2,pPower);
     #else #if (thisX > thirdWidth*2)
          // Last part
          #declare thisY_0 = mod(thisX,thirdWidth);
          #declare thisY_1 = 1-(thisY_0/thirdWidth);
          #declare thisY_2 = thisY_1*pRaise;
          #declare thisY = pow(thisY_2,pPower);
     #else
          // Middle part
          #if (thisX < pWidth/2)
               //Left
               #declare thisY_0 = thisX-thirdWidth;
               #declare thisY_1 = mod(thisY_0,thirdWidth/2);
               #declare thisY_2 = 1-(thisY_1/(thirdWidth/2));
               #declare thisY_3 = thisY_2*pRaise;
               #declare thisY = pow(thisY_3,pPower);
          #else
               //Right
               #declare thisY_0 = thisX-pWidth/2;
               #declare thisY_1 = mod(thisY_0,thirdWidth/2);
               #declare thisY_2 = thisY_1/(thirdWidth/2);
               #declare thisY_3 = thisY_2*pRaise;
               #declare thisY = pow(thisY_3,pPower);
               
          #end          
     #end
     #end
     #declare heightArray[i] = thisY;
     #declare i = i+1;
#end
#declare heightArray[nSamples] = 0;               

#declare pamphlet =
     mesh {
          #declare i = 0;
          #while (i < nSamples)    
               triangle { <i*dx,heightArray[i],0>,<i*dx,heightArray[i],pDepth>,<(i+1)*dx,heightArray[i+1],pDepth> }
               triangle { <i*dx,heightArray[i],0>,<(i+1)*dx,heightArray[i+1],0>,<(i+1)*dx,heightArray[i+1],pDepth> }
               #declare i = i+1;
          #end       
          texture {
               pigment {
                    image_map { sys "pamphlet_picture.bmp" }
                    rotate x*90
                    scale <pWidth,1,pDepth>
                    }
               finish { ambient 0.4 }
               }
          }          
          
object { 
     pamphlet
     scale <1.6,1.7,1.7>
     rotate x*-4 
     rotate y*-25
     translate <-14.3,1.501,-16.2>
     }

///////
#end //
/////////////////////////////////////////////////// Dividing Walls ////////////////////////////////////////////////////////////////// 
#declare showDividingWalls = 1; //
#if (showDividingWalls)            //
/////////////////////////////////////
 
#declare startDist = 21.5;  
#declare widthOfWall = 2; 
#declare heightOfWall = 4.5; 
#declare xDist = 4; 
#declare zDist = 4;

#declare dividingWallTexture = 
     texture {
          pigment { 
               dents
               turbulence 0.2
               color_map {
                    [ 0.0 rgb 1.00 ]
                    [ 1.0 rgb 0.95 ]
                    }
               scale .2         
               }
          finish { ambient 0 }
          }

#declare dividingWall =           
     difference {
          cylinder {
               <0,0,0>, <0,heightOfWall,0>, startDist+widthOfWall
               } 
          cylinder {
               <0,-1,0>, <0,heightOfWall+1,0>, startDist
               }               
          plane {
               z, zDist
               }
          plane {
               x, xDist
               }
          texture { dividingWallTexture }
          }             

object { dividingWall rotate y*-0 } 
object { dividingWall rotate y*-90 }
//object { dividingWall rotate y*-180 }
object { dividingWall rotate y*-270 }   

///////
#end //            
/////////////////////////////////////////////////// Walls and Ceiling /////////////////////////////////////////////////////////////////
#declare showWalls = 1; //
#if (showWalls)            //
/////////////////////////////

#declare hWall = 80;          //Height of the wall

#declare nWind = 14;           //Number of windows
#declare wDisp = -2.5;          //Window displacement (z)
#declare dWind = 2;           //Depth of the windows
#declare wWind = 5;           //Width of the windows
#declare hWind = 31;           //Height of the window  
#declare hToWind = 10;         //Distance from floor to window

#declare sWindBy = 1;       //Sink the window into the wall by...
#declare sWidth = 2;        //
#declare sHeight = 10;         //

#declare dBetWind = 8;        //Distance between the windows
#declare dLedge = 0.3;        //Depth of the window ledge
#declare hLedge = 1.5;        //Height from bottom of window up to window ledge

#declare windowLights = 1;     //Include lights between the windows
#declare windowLightOn = 1;
#declare wLightsAfterWind = 0; //Place lights on the outsides of the windows as well
#declare hToWindLt = 8;       //Distance from floor to window light
#declare wWindLt = 1;         //Width of the window light 
#declare hWindLt = 3;         //Height of the window light
#declare dWindLt = 0.5;      //Depth of window light 
#declare hWindLtTop = 0.2;    //Height of the flat part of the top of the light

#declare nWindDH = 9;         //Number of window dividers horizontal
#declare nWindDV = 2;         //Number of window dividers vertical
#declare wWindD = 0.1;        //Width of window divider
#declare dWindD = 0.05;       //Depth of windowdivider

// Textures
    
#declare wallTexture = 
     texture {
          pigment { 
               rgb <241,227,218>/255     
               }              
          finish { 
               ambient 0
               diffuse 0.6
               }
          normal {
               granite 0.002
               scale 5
               }       
          }   
                
#declare windowGlass= 
     texture {
          pigment { rgbt <1,1,1,1> }
          finish {  
               ambient 0
               reflection 0.3
               ior 1.4
               specular 0
               }
          normal {
               bumps 0.002
               scale 3
               }     
          }          

#declare pre_01 = 
     pigment { 
          bumps
          turbulence 0.2
          color_map {
               [ 0.00 color <255,237,202>/255 ]
               [ 0.85 color <245,216,176>/255 ]
               [ 1.00 color <135,109,078>/255 ]
               }
          }                        
#declare pre_02 = 
     pigment { 
          gradient y
          turbulence 0.1
          color_map {
               [ 0.00 color <255,237,202>/255 ]
               [ 0.40 color <245,216,176>/255 ]
               [ 0.50 color <135,109,078>/255 ]
               [ 0.60 color <245,216,176>/255 ]
               [ 1.00 color <255,237,202>/255 ] 
               }
          scale 0.2     
          }       
#declare pre_03 = 
     pigment { 
          gradient x
          turbulence 0.1
          color_map {
               [ 0.00 color <255,237,202>/255 ]
               [ 0.40 color <245,216,176>/255 ]
               [ 0.50 color <135,109,078>/255 ]
               [ 0.60 color <245,216,176>/255 ]
               [ 1.00 color <255,237,202>/255 ] 
               }
          scale 0.2      
          }
#declare pre_04 = 
     pigment { 
          bozo
          turbulence 0.3
          color_map {
               [ 0.00 color <135,109,078>/1655 ] 
               [ 0.35 color <255,237,202>/255 ]
               [ 1.00 color <255,237,202>/255 ]
               } 
          scale 0.2      
          } 
#declare windowLightTexture =
     texture {
          pigment { 
               average
               pigment_map {
                    [ 1.0 pre_01 ]
                    [ 1.0 pre_02 ]
                    [ 1.0 pre_03 ]
                    [1.0 pre_04 ]
                    }
               }       
          finish { 
               ambient 0
               reflection 0
               }
          scale 0.1       
          }
                    
// Close off the -x and x ends of the hall.
box {
     <distanceToFarWall,0,-distanceToNearWall>, <distanceToFarWall+1,hWall,distanceToNearWall>
     texture { wallTexture }
     }
box {
     <-distanceToFarWall-1,0,-distanceToNearWall>, <-distanceToFarWall,hWall,distanceToNearWall>
     texture { wallTexture }
     }
// Throw a ceiling on this whole thing
/*
box {
     <-distanceToFarWall-1,hWall,-distanceToNearWall>, <distanceToFarWall+1,hWall+1,distanceToNearWall>
     texture { wallTexture }
     }*/

// Start the wall real wall building madness.

#declare tL = nWind*wWind + (nWind-1)*dBetWind;   //This is the total length taken up by windows.
#if (tL >= distanceToFarWall*2) #error "Error" #end  // Can't fit windows in wall.
#declare dTL = (distanceToFarWall*2)-tL;     //The distance left over (empty wall)

#declare window =
     union {
          mesh {
               //This is the "sink" of the window into the wall...
               //Here it projects outwards, but it will be pushed back when the window is placed.
               //Top z face
               triangle { <-sWidth,hWind,0>,<-sWidth,hWind+sHeight,0>,<sWidth+wWind,hWind+sHeight,0> }
               triangle { <-sWidth,hWind,0>,<sWidth+wWind,hWind,0>,<sWidth+wWind,hWind+sHeight,0> } 
               //Top y face
               triangle { <-sWidth,hWind+sHeight,0>,<-sWidth,hWind+sHeight,-sWindBy>,<sWidth+wWind,hWind+sHeight,-sWindBy> }
               triangle { <-sWidth,hWind+sHeight,0>,<sWidth+wWind,hWind+sHeight,0>,<sWidth+wWind,hWind+sHeight,-sWindBy> }
               //Bottom z face
               triangle { <-sWidth,0,0>,<-sWidth,-sHeight,0>,<sWidth+wWind,-sHeight,0> }
               triangle { <-sWidth,0,0>,<sWidth+wWind,0,0>,<sWidth+wWind,-sHeight,0> }
               //Bottom y face
               triangle { <-sWidth,-sHeight,0>,<-sWidth,-sHeight,-sWindBy>,<sWidth+wWind,-sHeight,-sWindBy> }
               triangle { <-sWidth,-sHeight,0>,<sWidth+wWind,-sHeight,0>,<sWidth+wWind,-sHeight,-sWindBy> }
               //Left z face
               triangle { <-sWidth,0,0>,<-sWidth,hWind,0>,<0,hWind,0> }
               triangle { <-sWidth,0,0>,<0,0,0>,<0,hWind,0> }
               //Left x face
               triangle { <-sWidth,-sHeight,-sWindBy>,<-sWidth,hWind+sHeight,-sWindBy>,<-sWidth,hWind+sHeight,0> }
               triangle { <-sWidth,-sHeight,-sWindBy>,<-sWidth,-sHeight,0>,<-sWidth,hWind+sHeight,0> }
               //Right z face
               triangle { <wWind,0,0>,<wWind,hWind,0>,<wWind+sWidth,hWind,0> }
               triangle { <wWind,0,0>,<wWind+sWidth,0,0>,<wWind+sWidth,hWind,0> }
               //Right x face
               triangle { <wWind+sWidth,-sHeight,-sWindBy>,<wWind+sWidth,hWind+sHeight,-sWindBy>,<wWind+sWidth,hWind+sHeight,0> }
               triangle { <wWind+sWidth,-sHeight,-sWindBy>,<wWind+sWidth,-sHeight,0>,<wWind+sWidth,hWind+sHeight,0> }  
               //Sides of window (sunk into wall)
               triangle { <0,0,0>,<0,0,dWind>,<0,hWind,dWind> }
               triangle { <0,0,0>,<0,hWind,0>,<0,hWind,dWind> }
               //   far side
               triangle { <wWind,0,0>,<wWind,0,dWind>,<wWind,hWind,dWind> }
               triangle { <wWind,0,0>,<wWind,hWind,0>,<wWind,hWind,dWind> }
               //The ledge of the window
               triangle { <0,hLedge,dWind-dLedge>,<wWind,hLedge,dWind-dLedge>,<wWind,hLedge,dWind> }
               triangle { <0,hLedge,dWind-dLedge>,<wWind,hLedge,dWind>,<0,hLedge,dWind> }
               //Slant to ledge
               triangle { <0,0,0>,<0,hLedge,dWind-dLedge>,<wWind,hLedge,dWind-dLedge> } 
               triangle { <0,0,0>,<wWind,hLedge,dWind-dLedge>,<wWind,0,0> }
               //Top of window
               triangle { <0,hWind,dWind>,<wWind,hWind,dWind>,<wWind,hWind,0> }
               triangle { <0,hWind,dWind>,<wWind,hWind,0>,<0,hWind,0> }                 
               }                                                                                              
          //Dividers
          #declare i = 0;
          #while (i < nWindDH)
               box {
                    <0,-wWindD/2,dWind-dWindD>,<wWind,wWindD/2,dWind>
                    texture { wallTexture }
                    translate <0,hLedge + ((i+1)*((hWind-hLedge)/(nWindDH+1))),0>
                    no_reflection
                    }
               #declare i = i + 1;
          #end 
          #declare i = 0;
          #while (i < nWindDV)
               box {
                    <-wWindD/2,hLedge,dWind-dWindD>,<wWindD/2,hWind,dWind>
                    texture { wallTexture }
                    translate <(i+1)*(wWind/(nWindDV+1)),0,0> 
                    no_reflection
                    }
               #declare i = i + 1;
          #end
          }                        

#declare windowGlassObject = 
     mesh {
          //Pane of glass
          triangle { <0,hLedge,dWind>,<0,hWind,dWind>,<wWind,hWind,dWind> texture{windowGlass} } 
          triangle { <0,hLedge,dWind>,<wWind,hLedge,dWind>,<wWind,hWind,dWind> texture{windowGlass} }
          no_shadow 
          no_reflection
          } 
          
#declare wallBetweenWindows = 
     mesh {
          triangle {
               <0,0-sHeight,0>,<0,hWind+sHeight,0>,<dBetWind-(2*sWidth),hWind+sHeight,0>
               }
          triangle {
               <0,0-sHeight,0>,<dBetWind-(2*sWidth),0-sHeight,0>,<dBetWind-(2*sWidth),hWind+sHeight,0>
               }
          texture { wallTexture }
          } 

#declare ltDX = 0.1;
#declare ltDZ = 0.2;              
#declare windowLightObject =
     union { 
          mesh {
               // Bottom
               triangle { <ltDX,0,0>,<wWindLt-ltDX,0,0>,<wWindLt-ltDX,0,-dWindLt+ltDZ> texture {windowLightTexture} }     
               triangle { <ltDX,0,0>,<ltDX,0,-dWindLt+ltDZ>,<wWindLt-ltDX,0,-dWindLt+ltDZ> texture {windowLightTexture} }
               // Left x side
               triangle { <ltDX,0,0>,<0,hWindLt,0>,<0,hWindLt,-dWindLt> texture {windowLightTexture} }
               triangle { <ltDX,0,0>,<ltDX,0,-dWindLt+ltDZ>,<0,hWindLt,-dWindLt> texture {windowLightTexture} }
               // Right x side
               triangle { <wWindLt-ltDX,0,0>,<wWindLt,hWindLt,0>,<wWindLt,hWindLt,-dWindLt> texture {windowLightTexture} }
               triangle { <wWindLt-ltDX,0,0>,<wWindLt-ltDX,0,-dWindLt+ltDZ>,<wWindLt,hWindLt,-dWindLt> texture {windowLightTexture} }
               // Front
               triangle { <ltDX,0,-dWindLt+ltDZ>,<0,hWindLt,-dWindLt>,<wWindLt,hWindLt,-dWindLt> texture {windowLightTexture} }
               triangle { <ltDX,0,-dWindLt+ltDZ>,<wWindLt-ltDX,0,-dWindLt+ltDZ>,<wWindLt,hWindLt,-dWindLt> texture {windowLightTexture} }
               }  
          #if (windowLightOn)      
          light_source {
               <wWindLt/2,0.1,(-dWindLt+ltDZ)/2>
               rgb <0.85,0.85,1>*10
               fade_power 2
               fade_distance 20
               area_light
               <(wWindLt-(2*ltDX))/3,0,0>,<0,0,(dWindLt-ltDZ)/3>, 5,1
               media_interaction off
               } 
          #end  
          //no_reflection     
          }          
                     
          
// The +Z Wall
// The section beneath the windows...
#if (sHeight < hToWind)
mesh {
     triangle {
          <-distanceToFarWall,0,distanceToNearWall>,
          <-distanceToFarWall,hToWind-sHeight,distanceToNearWall>,
          <distanceToFarWall,hToWind-sHeight,distanceToNearWall>
          texture { wallTexture }
          }
     triangle {
          <-distanceToFarWall,0,distanceToNearWall>,
          <distanceToFarWall,0,distanceToNearWall>,
          <distanceToFarWall,hToWind-sHeight,distanceToNearWall>
          texture { wallTexture }
          }
     }
#end     
// The section between the top of the windows and the ceiling
#declare tWind = hToWind+hWind;
#if (tWind >= hWall) #error "!" #end //Window extends above the wall!
#if ((sHeight*2)+hWind < hWall)
mesh {
     triangle {
          <-distanceToFarWall,tWind+sHeight,distanceToNearWall>,
          <-distanceToFarWall,hWall,distanceToNearWall>,
          <distanceToFarWall,hWall,distanceToNearWall> 
          }
     triangle {
          <-distanceToFarWall,tWind+sHeight,distanceToNearWall>,
          <distanceToFarWall,tWind+sHeight,distanceToNearWall>,
          <distanceToFarWall,hWall,distanceToNearWall>
          }
     texture { wallTexture }     
     }     
#end                            
// Work out the window placements...
#declare wOfAllWind = nWind*wWind + (nWind-1)*dBetWind;
#if (wOfAllWind >= 2*distanceToFarWall) #error #end //Windows won't fit on wall!
#if (wOfAllWind + dBetWind + wWindLt >= 2*distanceToFarWall & wLightsAfterWind & windowLights) 
     #error //lights after windows won't fit. 
#end
 
//Light before window if it's needed.
#if (wLightsAfterWind & windowLights)
     object {
          windowLightObject
          translate <-(wOfAllWind/2)-(dBetWind/2)-(wWindLt/2),hToWindLt,distanceToNearWall>
          translate wDisp*x
          }
#end
// Wall before window...
mesh {
     triangle {
          <-distanceToFarWall,hToWind,distanceToNearWall>,
          <-distanceToFarWall,tWind,distanceToNearWall>,
          <(-wOfAllWind/2)-sWidth+wDisp,tWind,distanceToNearWall>
          }
     triangle {
          <-distanceToFarWall,hToWind,distanceToNearWall>,
          <(-wOfAllWind/2)-sWidth+wDisp,hToWind,distanceToNearWall>,
          <(-wOfAllWind/2)-sWidth+wDisp,tWind,distanceToNearWall>
          }     
     texture { wallTexture }
     }
//Place windows and walls (and lights) with a loop  
#declare windLocArray = array[nWind];   //Record positions for haze to use           
#declare i = 0;
#while (i < nWind)
     object {
          window
          translate <-wOfAllWind/2,hToWind,distanceToNearWall+sWindBy>
          translate <i*(wWind+dBetWind),0,0>
          translate wDisp*x 
          texture { wallTexture }
          #declare windLocArray[i] = <-wOfAllWind/2,hToWind,distanceToNearWall+sWindBy>+<i*(wWind+dBetWind),0,0>+(wDisp*x);
          }
     object {
          windowGlassObject
          translate <-wOfAllWind/2,hToWind,distanceToNearWall+sWindBy>
          translate <i*(wWind+dBetWind),0,0>
          translate wDisp*x 
          texture { windowGlass }
          }        
     #if (i != nWind-1)     
          object {
               wallBetweenWindows
               translate <-(wOfAllWind/2)+wWind+sWidth,hToWind,distanceToNearWall>
               translate <i*(wWind+dBetWind),0,0>
               translate wDisp*x
               texture { wallTexture }
               }
          #if (windowLights)
               object {
                    windowLightObject
                    translate <-(wOfAllWind/2)+wWind+(dBetWind/2)-(wWindLt/2),hToWindLt,distanceToNearWall>
                    translate <i*(wWind+dBetWind),0,0> 
                    translate wDisp*x
                    texture { wallTexture }
                    }
          #end               
     #end
     #declare i = i+1;
#end   
// Light after window if needed
#if (wLightsAfterWind & windowLights)
     object {
          windowLightObject
          translate <(wOfAllWind/2)+(dBetWind/2)-(wWindLt/2),hToWindLt,distanceToNearWall> 
          translate wDisp*x
          }
#end     
// Wall after window...
mesh {
     triangle {
          <distanceToFarWall,hToWind,distanceToNearWall>,
          <distanceToFarWall,tWind,distanceToNearWall>,
          <(wOfAllWind/2)+sWidth+wDisp,tWind,distanceToNearWall>
          }
     triangle {
          <distanceToFarWall,hToWind,distanceToNearWall>,
          <(wOfAllWind/2)+sWidth+wDisp,hToWind,distanceToNearWall>,
          <(wOfAllWind/2)+sWidth+wDisp,tWind,distanceToNearWall>
          }     
     texture { wallTexture }
     }                                               
                                
///////
#end //
//////////////////////////////////////////////////////////////// The Sky (and sun) ///////////////////////////////////////
#declare showSky = 1; //
#if (showSky)         //
////////////////////////

#declare skyPigment_01 = 
     pigment {
          bozo 
          turbulence 0.8
          color_map {
               [0.0 rgbf 01 ]
               [0.6 rgbf 01 ]
               [0.6 rgb 0.3 ]
               [0.7 rgb 0.9 ]
               [1.0 rgb 1.0 ]
               }
          scale <10,7,10>
          }

#declare skyPigment_02 = 
     pigment { 
          gradient y
          turbulence 0.1
          color_map {
               [0.0 rgb <0.70,0.70,1.00> ]
               [1.0 rgb <0.72,0.72,1.00> ]
               }
          scale 60
          translate <0,10,0>
          }       
                  
#declare skyTexture =
     /*texture {
          pigment { rgb <0.7,0.7,1> }
          finish { ambient 1 }
          }*/
     texture {         
          pigment { skyPigment_02 }
          pigment { skyPigment_01 }           
          finish { ambient 1 }
          }                     

#declare Plate_Texture = 
     texture {
          pigment {
               waves 
               turbulence 1
               color_map { 
                    [ 0.0 rgb <0.3,0.3,0.9> ]
                    [ 0.1 rgb <1.2,1.2,1.2> ]
                    [ 0.8 rgb <0.3,0.3,0.9> ]
                    [ 1 rgb <0.4,0.4,0.9> ]
                    }
               scale 8               
               }
          finish { 
               ambient 1
               reflection 0
               }
          rotate x*90         
          scale <7,2,5>
          rotate y*30        
          }
                
#declare sunLocation = <-260,250,280>;
#declare sunPointAt = <0,0,0>; 
#declare sunLight = 
     light_source {
          sunLocation
          rgb <255/255,218/255,205/255>*2.8 
          parallel
          point_at sunPointAt
          area_light
          <1.5,0,0>,<0,1,0>,5,3
          media_interaction off
          }
          
light_group {                              
     object { sunLight }          
     plane {
          z, distanceToNearWall+50
          texture { Plate_Texture }
          no_shadow  
          hollow
          }        
     global_lights false
     }     

object { sunLight }     
///////     
#end //
//////////////////////////////////////////////////////////////// Haze ///////////////////////////////////////////////////
#declare showHaze = 1; 
#if (showHaze & showSky & showWalls)

#debug "MEDIA"


#declare mPA = array[8]; //media point array
//-x,+y,+z point then clockwise around top, repeat for -y
#declare mPA[0] = <-50,tWind,28>;
#declare mPA[1] = <90,tWind,28>;   
#declare mPA[2] = <59,3,-13>;
#declare mPA[3] = <-8,3,-13>;

#declare mPA[4] = <-50,0.1,28>;
#declare mPA[5] = <90,0.1,28>;   
#declare mPA[6] = <59,0.1,-13>;
#declare mPA[7] = <-8,0.1,-13>;

#declare showMediaBoundingBox = 0;
#if (showMediaBoundingBox)
     #declare mBBT = texture { pigment { rgb <1,0,0> } finish { ambient 1 } };
     #declare mBBCR = 0.1;
     cylinder { mPA[0],mPA[1],mBBCR texture{mBBT} } 
     cylinder { mPA[1],mPA[2],mBBCR texture{mBBT} }
     cylinder { mPA[2],mPA[3],mBBCR texture{mBBT} }
     cylinder { mPA[3],mPA[0],mBBCR texture{mBBT} }
     cylinder { mPA[0],mPA[4],mBBCR texture{mBBT} }
     cylinder { mPA[1],mPA[5],mBBCR texture{mBBT} }
     cylinder { mPA[2],mPA[6],mBBCR texture{mBBT} }
     cylinder { mPA[3],mPA[7],mBBCR texture{mBBT} }
     cylinder { mPA[4],mPA[5],mBBCR texture{mBBT} }
     cylinder { mPA[5],mPA[6],mBBCR texture{mBBT} }
     cylinder { mPA[6],mPA[7],mBBCR texture{mBBT} }
     cylinder { mPA[7],mPA[4],mBBCR texture{mBBT} }
#else 
     light_group {                              
          light_source {
               sunLocation
               rgb <255/255,218/255,205/255>*2.8 
               parallel
               point_at sunPointAt
               media_interaction on
               }
          mesh {
               triangle { mPA[0],mPA[1],mPA[2] }  // +y face 
               triangle { mPA[0],mPA[2],mPA[3] }  // +y face 
               triangle { mPA[3],mPA[2],mPA[6] }  // -z face 
               triangle { mPA[3],mPA[6],mPA[7] }  // -z face
               triangle { mPA[2],mPA[1],mPA[5] }  // +x face 
               triangle { mPA[2],mPA[5],mPA[6] }  // +x face
               triangle { mPA[1],mPA[0],mPA[4] }  // +z face 
               triangle { mPA[1],mPA[4],mPA[5] }  // +z face
               triangle { mPA[0],mPA[3],mPA[7] }  // -x face 
               triangle { mPA[0],mPA[7],mPA[4] }  // -x face
               triangle { mPA[4],mPA[5],mPA[6] }  // -y face 
               triangle { mPA[4],mPA[6],mPA[7] }  // -y face
               inside_vector <0,1,0>
               pigment { rgbt 1 }
               finish { ambient 0 }
               hollow
               interior {
                    media {   
                         scattering { 5, 0.01 extinction 0.2 }
                         samples 100
                         }
                    }
               no_reflection     
               }
          global_lights false
          }                        
#end     
       
#end       
/////////////////////////////////////////////////////////// Lights! /////////////////////////////////////////////////////

#declare showCitySpotlight1 = 0;   //Single spotlight above city
#declare showcitySpotlight2 = 0;   //Single spotlight above city (area_light)
#declare showBoxCeilingLights = 1; //Area lights near the ceiling
#declare showWallLight = 0;  //Simple light source for viewing the wall (for testing only)   


#if (showCitySpotlight1)
     light_source {
          <-1,20,-2>
          rgb 1
          spotlight
          point_at <0,4,0>
          radius 25
          falloff 39
          }
#end
#if (showcitySpotlight2)
     light_source {
          <-10,40,-20>
          rgb <0.9,0.9,1>*0.8
          spotlight
          point_at <0,4,0>
          radius 16
          falloff 28
          area_light 
          <2,0,0>,<0,0,2>,2,2
          orient circular
          }
#end
#if (showBoxCeilingLights)
     #declare thisX = 80;
     #declare thisY = 40;
     #declare thisZ = 8;
     #declare numX = 16;
     #declare numZ = 2;
     #declare thisFadeDistance = 25;
     #declare thisColorVector = <1,1,1>/16;
     #declare areaCount = 3;
     #declare areaDist = 0.2;
     #declare thisRadius = 60;
     #declare thisFalloff = 120;
     #declare i = 0;
     #while (i < numX)
          #declare j = 0;
          #while (j < numZ)      
               light_source {
                    <-thisX,thisY,-thisZ>+<i*(thisX*2/(numX-1)),0,j*(thisZ*2/(numZ-1))>
                    rgb thisColorVector
                    spotlight      
                    point_at <-thisX,0,-thisZ>+<i*(thisX*2/(numX-1)),0,j*(thisZ*2/(numZ-1))>
                    radius thisRadius
                    falloff thisFalloff
                    fade_power 2
                    fade_distance thisFadeDistance
                    area_light
                    <areaDist,0,0>,<0,0,areaDist>,areaCount,areaCount
                    orient circular 
                    media_interaction off
                    }                                        
               #declare j = j+1;
          #end
          #declare i = i+1;
     #end            
#end
#if (showWallLight)
     light_source {
          <0,10,0>
          rgb 1
          }
#end
/////////////////
                  
/*                                            
camera {     //view bench  
     //orthographic
     location <-9,5.8,-25.5>
     look_at <10,3,-15>
     right x*image_width/image_height
     angle 40
     } */
     

camera {  // NEAR FINAL
     location <-9,5.8,-25.5>
     look_at <-3,4.5,0>
     right x*image_width/image_height
     //angle 70
     }

/*
camera {  // wall view
     location <0,15,-20>
     look_at <0,17,20>
     right x*image_width/image_height
     //angle 70
     } */
     
/*               
camera { 
     //orthographic 
     location <0,180,-20>
     look_at <0,0,20> 
     right x*image_width/image_height
     } */    

/*lightDistribute(
     10,           //numberOfLights,
     <0,5,0>      //centerOfScene,
     10,            //minRadius,
     10,            //maxAdditionalRadius,
     4,             //minHeight,
     15,            //maxAdditionalHeight,
     0,             //minRotation, 
     360,           //maxAdditionalRotation,
     3.5,           //brightnessMultiplier,
     1423)*/


    
     
#declare radiosityOn = 1;

// Global settings and radiosity                                              
global_settings {
     max_trace_level 10
     #if (radiosityOn)
     radiosity {
          count 200               // number of "rays" cast
          nearest_count 10         // number of considerations for each lighting, 20 is highest
          
          pretrace_start 0.08
          pretrace_end 0.01       // these control number of passes
          
          always_sample off        // on
          error_bound 0.5         // makes the shadows blend, sort of
          low_error_factor 0.1    // lowers error_bound in last pass 
          minimum_reuse 0.015     // reuse of samples
          
          brightness 1            // normal brightness
          max_sample 1            // brightest something can be 
          
          recursion_limit 10       // controlls bouncing (?)
          
          adc_bailout 0.01        // at which point lighting change is ignored
          gray_threshold 0        // 0 is normal, 1 is no color 
          
          normal off             // normal interaction
          media off               // media interaction 
          
          save_file "City_Pass_1"
          }       
     #end        
     }                                                        