// Persistence of Vision Ray Tracer Scene Description File
// File: IRTC-IceFire.pov
// Vers: 3.6
// Desc: ITRC Entry for Jan/Feb 2006 topic: Ice and Fire.
// Date: Feb 2006
// Auth: Bruno Cabasson
//
#declare _iteration = 14;
#version 3.6;

#include "colors.inc"
#include "stones.inc"
#include "textures.inc"
#include "functions.inc"
#include "logo.inc"
#include "fire.inc"

#include "cristal.inc"

// General constants
#declare NONE = -1;

// Render types
#declare TUNING = 0;
#declare RAD_DATA_BUILD = 1;
#declare FINAL = 2;

#declare RENDER_TYPE = TUNING;

// Quality constants
#declare SIMPLE = 0;
#declare NORMAL = 1;
#declare ENHANCED = 2;

// Radiosity operations
#declare LOAD = 0;
#declare SAVE = 1;
#declare RAD_DATA = concat("IRTC-IceFire_", str(_iteration,0,0),".rad");
#declare PHOTON_MAP = concat("IRTC-IceFire_", str(_iteration,0,0),".map");

// Ground types
#declare ISO = 0;
#declare HF = 1;

#switch (RENDER_TYPE)

    // One may change these
    #case (TUNING)
        #declare MAX_TRACE_LEVEL = 25;
        #declare USE_FOG = yes;
            #declare FOG_QUALITY = SIMPLE;
        #declare USE_GLOW = yes;
        #declare USE_TREE = yes;
        #declare USE_AREA_LIGHT = no;
        #declare USE_RAD = yes;
            #declare PRETRACE_START = 0.04;
            #declare PRETRACE_END = 0.02;
            #declare RAD_FILE = NONE;
        #declare USE_PHOTONS = yes;
            #declare PHOTON_SPACING = 0.005;
            #declare PHOTON_MAX_STEPS = 2;
        #declare GROUND_TYPE = ISO;
            #declare GROUND_TEXTURE_QUALITY = NORMAL;
            #declare HF_DENSITY = 2;
        #declare USE_CRISTAL = yes;
    #break
    
    // One should not change these
    #case (RAD_DATA_BUILD)  // performed at low resolution (typ. 512*384)
        #declare MAX_TRACE_LEVEL = 10;
        #declare USE_FOG = yes;
            #declare FOG_QUALITY = SIMPLE;
        #declare USE_GLOW = yes;
        #declare USE_TREE = yes;
        #declare USE_AREA_LIGHT = yes;
        #declare USE_RAD = yes;
            #declare PRETRACE_START = 0.04;
            #declare PRETRACE_END = 0.01;
            #declare RAD_FILE = SAVE;
        #declare USE_PHOTONS = no;
            #declare PHOTON_SPACING = 0.02;
            #declare PHOTON_MAX_STEPS = 2;
        #declare GROUND_TYPE = ISO;
            #declare GROUND_TEXTURE_QUALITY = SIMPLE;
            #declare HF_DENSITY = 2;
        #declare USE_CRISTAL = yes;
    #break
    
    // One should not change these
    #case (FINAL) // performed at final resolution (typ. 1024*768)
        #declare MAX_TRACE_LEVEL = 25;
        #declare USE_FOG = yes;
            #declare FOG_QUALITY = SIMPLE;
        #declare USE_GLOW = yes;
        #declare USE_TREE = yes;
        #declare USE_AREA_LIGHT = no;
        #declare USE_RAD = yes;
            #declare PRETRACE_START = 0.04;
            #declare PRETRACE_END = 0.02;
            #declare RAD_FILE = LOAD;
        #declare USE_PHOTONS = yes;
            #declare PHOTON_SPACING = 0.005;
            #declare PHOTON_MAX_STEPS = 2;
        #declare GROUND_TYPE = ISO;
            #declare GROUND_TEXTURE_QUALITY = SIMPLE;
            #declare HF_DENSITY = 5;
        #declare USE_CRISTAL = yes;
    #break
#end

#if (USE_RAD) #default {finish{ambient 0}} #end

global_settings 
{
    assumed_gamma 1.0
    max_trace_level MAX_TRACE_LEVEL
    #if (USE_RAD) 
      radiosity
      {
          count 100
          nearest_count 8
          error_bound 0.8
          recursion_limit 1
          low_error_factor 0.9
          gray_threshold 0
          minimum_reuse 0.015
          adc_bailout 0.01/2
          media on
          normal on
          pretrace_start PRETRACE_START
          pretrace_end PRETRACE_END
          #switch (RAD_FILE)
             #case (LOAD) load_file RAD_DATA #break
             #case (SAVE) save_file RAD_DATA #break
         #end
          brightness 10
          always_sample off
      }
    #end
    #if (USE_PHOTONS)
      photons {spacing PHOTON_SPACING media PHOTON_MAX_STEPS max_trace_level MAX_TRACE_LEVEL}
    #end
}

background {Black}

#declare SUN_HEADING = 18;
#declare SUN_PITCH = -3;
#declare SUN_DISTANCE = 1E6;
#declare SUN_APPARENT_ANGLE = 0.5;
#declare SUN_RADIUS = SUN_DISTANCE*tan(radians(SUN_APPARENT_ANGLE/2));
#declare EARTH_RADIUS = 30000;
#declare NB_STARS = 500;
#declare SKY_COLOR = VioletRed;
#declare SKY_THICKNESS = 100;
#declare TIME_SINCE_NOON = 5.5;
#declare FOG_ALT = 2.7;
#declare TERRAIN_HEIGHT = 4;
#declare TERRAIN_SIZE = 300;
#declare TERRAIN_FADE_DIST = 0.8*200; // Must be <TERRAIN_SIZE
#declare HF_SIZE = 2*HF_DENSITY*TERRAIN_SIZE;
#declare CAMERA_HEIGHT = 1.7; // Standard human size.
#declare CAMERA_ANGLE = 40;
#declare CAMERA_PITCH = 0;
#declare TREE_POS = 2.8*y+12.5*z;

// ----------------------------------------------------------------------------
#declare _T = 0.7;
#declare C_SNOW_COLOR = <190, 220, 255>/255;
#declare T_Snow_Base = texture
{
    pigment {rgb 0.5}
    normal {granite 0.3 scale 0.002}
    finish {ambient 0.4}
}
texture
{
    pigment 
    {
        bozo scale 0.0000005
        sine_wave
        color_map {[0 rgb 0.7*C_SNOW_COLOR transmit 0.5] [1 rgb C_SNOW_COLOR transmit 0.5]}
    }
    normal {bozo 5 scale 0.003}
    finish {reflection 0.0 specular 0.2 ambient 0.15}
}
texture 
{
    pigment
    {
        granite scale 0.0001
        //turbulence 1 omega 1
        color_map
        {
            [_T transmit 1]
            [_T White]
            [_T*1.01 White]
            [_T*1.01 transmit 1]
        }
    }
    finish {reflection 0.0 specular 0.2 ambient 2}
}

#declare T_Snow = texture
{
    granite scale 0.0001
    //turbulence 1 omega 1
    texture_map
    {
        [_T T_Snow_Base]
        [_T finish {reflection 1}]
        [_T*1.01 finish {reflection 1}]
        [_T*1.01 T_Snow_Base]
    }
}

#declare T_Snow_Far = texture 
{
    pigment {rgb C_SNOW_COLOR} 
    finish {reflection 0.0 specular 0.2 ambient 0.2}
}

#declare T_Snow_Fadeaway = texture 
{
    gradient z
    scale 1000 translate -4*z
    texture_map
    {
        [0  T_Snow_Base]
        [60/1000 T_Snow_Far]
    }
}

#declare T_SimpleGround = texture {pigment {Gray70} finish {specular 0.2 ambient 0.3}}

// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare fn_Terrain = function {max(1-f_r(x/TERRAIN_FADE_DIST,0,z/TERRAIN_FADE_DIST), 0) * f_noise3d(x/4, 0, z/8)}

#declare fn_IsoGround = function 
{
    f_r(x,y+EARTH_RADIUS,z) - EARTH_RADIUS // Approximately equals 'y' because EARTH_RADIUS is big.
  - TERRAIN_HEIGHT*fn_Terrain(x,y,z)
}

    
#declare O_IsoGround = isosurface 
{
    function {fn_IsoGround(x,y,z)}
    accuracy 0.0001
    max_gradient 1.1
    contained_by {box {-<100, 10, 1000>, <100, TERRAIN_HEIGHT, 1000>}}
    translate 3*x // Find a nice point of view
}

#declare O_HFGround = union
{
    height_field
    {
        function HF_SIZE, HF_SIZE {fn_Terrain((x-0.5)*TERRAIN_SIZE,0,-(y-0.5)*TERRAIN_SIZE)}
        smooth
        translate -x/2 -z/2
        scale <TERRAIN_SIZE,TERRAIN_HEIGHT,TERRAIN_SIZE>
        translate 3*x // Find a nice point of view
    }
    
    sphere {0, EARTH_RADIUS pigment {White} translate -EARTH_RADIUS*y}
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare O_Ground = object
{
    #switch (GROUND_TYPE)
        #case (ISO) O_IsoGround #break
        #case (HF)  O_HFGround  #break
    #end
    #switch (GROUND_TEXTURE_QUALITY)
        #case (SIMPLE) texture {T_SimpleGround} #break
        #case (NORMAL) texture {T_Snow_Fadeaway} #break
    #end
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare O_Fog = sphere
{
    0, EARTH_RADIUS + FOG_ALT
    pigment {Clear}
    hollow
    interior
    {
        media
        {
            #switch (FOG_QUALITY)
                #case (SIMPLE) intervals 1 samples 1 #break
                #case (NORMAL) intervals 1 samples 5 #break
                #case (ENHANCED) intervals 2 samples 10 #break
            #end
            scattering {1, 0.02 extinction 1}
            density 
            {
                planar turbulence 0.3
                scale FOG_ALT translate EARTH_RADIUS*y
            }
            density {spherical scale 100 translate EARTH_RADIUS*y}
        }
    }
    #if (USE_PHOTONS)
        photons {pass_through}
    #end
    translate -EARTH_RADIUS*y
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare CM_Gimp_Sky_BC_spline = spline
{
    linear_spline
	0.000000, <0.048155, 0.018543, 0.181818, 0.000000>
	0.085142, <0.091129, 0.043493, 0.265152, 0.000000>
	0.210351, <0.134103, 0.068444, 0.348485, 0.000000>
	0.210351, <0.134103, 0.068444, 0.348485, 0.000000>
	0.267112, <0.282960, 0.075279, 0.209051, 0.000000>
	0.305509, <0.431818, 0.082115, 0.069616, 0.000000>
	0.305509, <0.431818, 0.082115, 0.069616, 0.000000>
	0.340568, <0.609848, 0.217388, 0.076476, 0.000000>
	0.382304, <0.787879, 0.352661, 0.083336, 0.000000>
	0.382304, <0.787879, 0.352661, 0.083336, 0.000000>
	0.442404, <0.527548, 0.364080, 0.503789, 0.000000>
	0.512521, <0.267218, 0.375499, 0.924242, 0.000000>
	0.512521, <0.267218, 0.375499, 0.924242, 0.000000>
	0.636060, <0.278169, 0.465126, 0.962121, 0.000000>
	1.000000, <0.289121, 0.554753, 1.000000, 0.000000>
}

#declare SKY_COLOR = CM_Gimp_Sky_BC_spline(0.01+abs(1-0.5*TIME_SINCE_NOON)/12*0.9);

#declare O_Sky = sphere
{
    0, EARTH_RADIUS + SKY_THICKNESS
    hollow
    no_shadow
    pigment {Clear}
    interior
    {
        media
        {
            samples 5
            #local sc = vnormalize(<SKY_COLOR.red, SKY_COLOR.green, SKY_COLOR.blue>);
            scattering {5, 0.0065*sc/SKY_THICKNESS eccentricity 0.75}
        }
    }
    translate -EARTH_RADIUS*y
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare O_Stars = blob // renders quicker than a union of spheres.
{
    #local Rnd = seed (98.45);
    #local i = 0;
    #while (i<NB_STARS)
        #declare _rot = <rand(Rnd)-0.5, rand(Rnd)-0.5, 0>*CAMERA_ANGLE;
        sphere
        {
            0, SUN_RADIUS/8*(1 + 3*rand(Rnd)), 1
            pigment {color rgb 0.3 + 0.7*<rand(Rnd), rand(Rnd), rand(Rnd)>}
            finish {diffuse 0 ambient 0.4+5*rand(Rnd)}
            translate 2*SUN_DISTANCE*z
            rotate _rot
        }
        #local i = i+1;
    #end
    threshold 0
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare CLUSTER_RADIUS = 6*SUN_RADIUS;
#declare COLOR = White;
#declare O_Cluster = sphere
{
    0, 1
    pigment {rgbt 1}
    hollow
    interior
    {
        media
        {
            emission 800*COLOR/CLUSTER_RADIUS
            samples 100
            density
            {
                spherical
                scale 5 warp {turbulence 0.8 omega 2.2} scale 1/5
                //warp {black_hole 0, 1 strength 1.8 falloff 1.4}
            }
            density 
            {
                spherical
                color_map {[0 rgb 0] [0.7 rgb 0.15] [0.8 rgb 0.7] [1 rgb 1]}
            }
        }
        media {emission 1.5*COLOR/CLUSTER_RADIUS density{spherical scale 1/3}}
    }
    scale CLUSTER_RADIUS
    translate SUN_DISTANCE*z
    rotate -12*x - 17*y
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare COMET_SCALE = 10*SUN_RADIUS;
#declare FLATNESS = 1/12;
#declare COLOR = White;
#declare O_Comet = sphere
{
    0, 1
    pigment {rgbt 1}
    hollow
    scale 1/FLATNESS*x
    interior
    {
        media
        {
            emission 200*COLOR/COMET_SCALE
            density
            {
                spherical
                translate -0.99*x
                scale 1/FLATNESS
                scale 1 warp {turbulence 1 omega 1.8} scale 1
                //warp {black_hole 0, 1 strength 1.8 falloff 1.4}
            }
            density 
            {
                spherical scale 2*x
                translate -1.4*x
                scale 0.7*1/FLATNESS
                color_map {[0 rgb 0] [0.98 rgb 0.05] [0.985 rgb 0.2] [0.99 rgb 0.7] [1 rgb 1]}
            }
        }
    }
    scale COMET_SCALE*FLATNESS
    rotate 145*z
    translate SUN_DISTANCE*z
    rotate -5*x
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare M_Glass= material 
{
  texture 
  {
    pigment {rgbt 1}
    finish 
    {
        ambient 0.0
        diffuse 0.05
        specular 0.6
        roughness 0.005
        reflection 
        {
            0.1, 1.0
            fresnel on
        }
        conserve_energy
     }
  }
  interior 
  {
    ior 1.05
    fade_power 1001
    fade_distance 0.075
    fade_color <0.9,0.4,0.6>
    #if (!USE_PHOTONS) caustics 0.7 #end
  }
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare O_Logo = object
{
    Povray_Logo
    material {M_Glass}
    scale 0.7
    #if (USE_PHOTONS)
        photons 
        {
            target 0.5
            refraction on
            reflection on
        }
    #end
} 
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare O_GlowingSphere = sphere 
{
    0, 1
    pigment {Clear}
    hollow
    interior
    {
        media
        {
            emission 60
            intervals 4 samples 5
            density
            {
                spherical scale 1.3
                scale 3 warp {turbulence 1 omega 1.3} scale 1/3
                warp 
                {
                    black_hole 0, 1
                    strength 2.4 falloff 1.6
                }
                color_map
                {
                    [0.75 0.001*White]
                    [0.75 White]
                }
            }
            density
            {
                spherical 
                color_map
                {
                    [0.0 rgb 0]
                    [0.1 rgb 0.02]
                    [0.125 BrightGold]
                    [0.13 Firebrick]
                    [0.133 Red]
                    [0.135 rgb 0.05]
                    [0.8 rgb 0.1]
                    [0.95 rgb 1]
                    
                }
            }
        }
        media {emission 1.5*BrightGold density {spherical scale 0.6}}
    }
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare O_GlowingLogo = union
{
    object {O_Logo}
    #if (USE_GLOW)
        object {O_GlowingSphere translate 0.1*x + 0.25*y}
    #end
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare DROP_START = 10*y;
#declare norm = <0,0,0>;
#declare DROP_POINT = trace (O_Ground, DROP_START, -y, norm);

camera 
{
  location  0
  angle     CAMERA_ANGLE
  right     x*image_width/image_height
  look_at   z
   
  rotate CAMERA_PITCH*x
  translate DROP_POINT + CAMERA_HEIGHT*y
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare L_PointLight = light_source 
{
    0 color rgb 1
    #if (USE_PHOTONS)
        photons
        {
            refraction on
            reflection on
        }
    #end
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare L_AreaLight = light_source 
{
    0                 // light's position (translated below)
    color rgb 1.0       // light's color
    area_light
    x, y,                 // lights spread out across this distance (x * y)
    4, 4                // total number of lights in grid (4x*4z = 16 lights)
    adaptive 1          // 0,1,2,3...
    circular            // make the shape of the light circular
    //orient              // orient light
    #if (USE_PHOTONS)
        photons
        {
            refraction on
            reflection on
        }
    #end
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare O_Sun = sphere 
{
    0, 1
    pigment {Clear}
    hollow
    no_shadow
    interior
    {
        media
        {
            emission 0.2/SUN_RADIUS
            density
            {
                spherical scale 1.3
                scale 3 warp {turbulence 1 omega 1.0} scale 1/3
                warp 
                {
                    black_hole 0, 1
                    strength 2.4 falloff 1.6
                }
            }
            density
            {
                spherical
                frequency -1
                scallop_wave
                color_map // Gimp's 'IncaNdescent' color map/
                {
                    [0.000000 color rgbt <0.000000, 0.000000, 0.000000, 0.000000>]
                	[0.459098 color rgbt <0.364706, 0.000000, 0.000000, 0.000000>]
                	[0.594324 color rgbt <0.729412, 0.000000, 0.000000, 0.000000>]
                	[0.594324 color rgbt <0.729412, 0.000000, 0.000000, 0.000000>]
                	[0.677796 color rgbt <0.864706, 0.272549, 0.098039, 0.000000>]
                	[0.809683 color rgbt <1.000000, 0.545098, 0.196078, 0.000000>]
                	[0.809683 color rgbt <1.000000, 0.545098, 0.196078, 0.000000>]
                	[0.853088 color rgbt <0.986274, 0.741176, 0.135294, 0.000000>]
                	[0.899833 color rgbt <0.972549, 0.937255, 0.074510, 0.000000>]
                	[0.899833 color rgbt <0.972549, 0.937255, 0.074510, 0.000000>]
                	[0.948247 color rgbt <0.974510, 0.952941, 0.452942, 0.000000>]
                	[1.000000 color rgbt <0.976471, 0.968627, 0.831373, 0.000000>]
                } /* color_map */

            }
            density {spherical}
        }
    }
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare SUN_TRANSFORM = transform
{
    scale 10*SUN_RADIUS
    translate SUN_DISTANCE*z
    rotate SUN_PITCH*x
    rotate SUN_HEADING*y
}
// ----------------------------------------------------------------------------


// ----------------------------------------------------------------------------
#declare O_Cristal = union
{
    object 
    {
        O_Cristal
        scale 1/5 
        material {M_Glass} 
        #if (!USE_PHOTONS) interior {caustics 4} #end
        
        #if (USE_PHOTONS)
            photons 
            {
                target 0.5
                refraction on
                reflection on
            }
        #end
    }
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare T_CristalTransform = transform {translate 2.2*y+11*z rotate 17*y}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare CM_Borealis = color_map {
	[0.000000 color rgbt <0.000000, 0.000000, 0.000000, 0.000000>]
	[0.333532 color rgbt <0.000000, 0.000000, 0.000000, 0.000000>]
	[0.333532 color rgbt <0.961365, 0.226242, 0.520124, 0.000000>]
	[0.570952 color rgbt <0.922731, 0.452483, 0.984953, 0.000000>]
	[0.570952 color rgbt <0.922731, 0.452483, 0.984953, 0.000000>]
	[0.616469 color rgbt <0.522483, 0.386162, 0.784143, 0.000000>]
	[0.664441 color rgbt <0.122236, 0.319840, 0.583333, 0.000000>]
	[0.664441 color rgbt <0.122236, 0.319840, 0.583333, 0.000000>]
	[0.727880 color rgbt <0.090941, 0.659920, 0.570851, 0.000000>]
	[0.756260 color rgbt <0.059646, 1.000000, 0.558369, 0.000000>]
	[0.756260 color rgbt <0.059646, 1.000000, 0.558369, 0.000000>]
	[0.799666 color rgbt <0.514672, 0.974284, 0.545851, 0.000000>]
	[0.843072 color rgbt <0.969697, 0.948568, 0.533333, 0.000000>]
	[0.843072 color rgbt <0.969697, 0.948568, 0.533333, 0.000000>]
	[0.905766 color rgbt <0.984849, 0.719284, 0.766666, 0.000000>]
	[0.949917 color rgbt <1.000000, 0.490000, 1.000000, 0.000000>]
	[0.949917 color rgbt <1.000000, 0.490000, 1.000000, 0.000000>]
	[0.988314 color rgbt <0.619054, 0.340921, 1.000000, 0.000000>]
	[1.000000 color rgbt <0.238108, 0.191841, 1.000000, 0.000000>]
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare O_Cloud = sphere 
{
    0.0, 1
    scale 1/10*y
    pigment {rgbt 1}
    hollow
    interior
    {
        media
        {
            emission 1/500
            samples 100 intervals 1
            density
            {
                pigment_pattern
                {
                    #local T1 = 0.15;
                    #local T2 = 0.65;
                    granite scale 1/3
                    color_map
                    {
                        [T1      rgb 0]
                        [T2      rgb 1]
                        //[T2      rgb 0]
                    } 
                    scale 5*x   
                }
                        
                color_map {CM_Borealis}
            }
            density {spherical translate x  scale 1/10*y scale 0.48 turbulence 0.6 lambda 2.2 omega 0.55}
        }
    }
    scale 3500
    rotate 13*x
    translate 4000*z
    rotate -8*x - 35*y
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare O_Space = union
{
    object {O_Stars}
    object {O_Comet}
    object {O_Cluster}
    object {O_Cloud}
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare LG_PointLight = light_group
{
    light_source {L_PointLight transform  {SUN_TRANSFORM}}
    object {O_Sun transform  {SUN_TRANSFORM}}
    #if (USE_FOG) object {O_Fog} #end
    object {O_Sky}
    object {O_Space}
    //object {O_Logo translate 0.2*y + 11.0*z rotate 14*y}
    #if (USE_CRISTAL) 
    object {O_Cristal transform {T_CristalTransform}}
    #end
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare LG_AreaLight = light_group 
{
    light_source
    {
        #if (USE_AREA_LIGHT)
             L_AreaLight
        #else
            L_PointLight
        #end
        transform  {SUN_TRANSFORM}
    } 
    object {O_Ground} // smooth shadows
    
    #if (USE_TREE)
        #include "WILLOW.inc"
        #declare LEAVES=0;
        #declare LAST_RAMIFICATION_NUMBER=4;
        #declare TRUNK_RADIUS=36;
        #declare BRANCH_RANDOM=0.4606;
        #declare TWIG_RANDOM=0.6615;
        #include "TOMTREE-1.5.inc"
        object{	TREE scale 1.7 scale 1.1*y rotate -20*y translate TREE_POS rotate -8*y}
        
        #if (USE_GLOW)
            object {O_GlowingSphere translate TREE_POS+0.7*y rotate -7.5*y}
        #end
    #end
}
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
#declare LG_NoLight = light_group
{
    #local NB_FLAMES_ATTEMPTS = 200;
    #local rnd = seed (2006);
    #local i=0;
    #while (i<NB_FLAMES_ATTEMPTS)
        #local DROP_START = <2*(rand(rnd)-0.5), 2.5, 2*(rand(rnd)-0.5)>;
        #local norm = <0,0,0>;
        #local DROP_POINT = trace (O_Cristal, DROP_START, -y, norm);
        #if (vlength(norm) != 0 & DROP_POINT.y > 0.10)
            //#debug "=====================> Flame!!!!\n"
            object 
            {
                MakeFire(1.0, 0.25 + 0.4*rand(rnd), 0.0, CM_Flame1, 1.5, -0.4, 0.4, rand(rnd))
                scale 0.12
                scale 0.5 + rand(rnd)
                translate DROP_POINT
                transform {T_CristalTransform}
            }
            //cylinder {DROP_START, DROP_POINT, 0.03 pigment{Green} transform {T_CristalTransform}}
        #end
        #local i=i+1;
    #end
    
    
    global_lights off
}
// ----------------------------------------------------------------------------
    

light_group {LG_PointLight}
light_group {LG_AreaLight}
light_group {LG_NoLight}

