// Copyright 2006 by Christian Froeschlin, www.chrfr.de

#include "colors.inc"
#include "metals.inc"
#include "stones.inc"
#include "strings.inc"
#include "stars.inc"
#include "rand.inc"
#include "woods.inc"
#include "math.inc"
#include "glass.inc"

global_settings {
  assumed_gamma 1.0
}


camera {
  location  <0, 0, -23.0>
  look_at   <0, 0,  50.0>
  angle 90
}


light_source {< -100, 100, -100> color White shadowless}

//7548
#declare R = seed(7549);

#declare SPACE_MAX       = 30; 

#declare NEURON_NUM  = 150;
#declare NEURON_CELL = array[NEURON_NUM]; // Positions of the neuron cores
#declare NEURON_AXON = array[NEURON_NUM]; // Direction (and length) of axon
#declare NEURON_STAT = array[NEURON_NUM]; // Activation state
#declare NEURON_CONN = array[NEURON_NUM]; // Number of connections (derived)

#declare CELL_RADIUS = 0.8;
#declare AXON_RADIUS = 0.15;
#declare CONN_RADIUS = 0.05; // Dendrites and synapses
#declare CONN_DIST   = 10;

#declare AXON_LENGTH_MIN = 4;
#declare AXON_LENGTH_MAX = 7;


#declare T_CELL = texture 
{
  pigment 
  {
    crackle scale 0.2 color_map
    {
      [0 color 2*Red]
      [0.2 color Red]
      [0.3 color Magenta]
      [1 color 0.5*Magenta]
    }
  }
  normal{bumps 0.5 scale 0.15}
  finish{diffuse 0.4 ambient 0.2}
}

#declare T_CELL_ACTIVE = texture 
{
  pigment {color White}
  normal{bumps 0.8 scale 0.1}
  finish{ambient 0.7}
}


#declare T_AXON = texture
{
  pigment {color Red}
  normal{bumps 1 scale 0.08}
  finish{phong 0.4}
}


#declare T_CONN = texture
{
  pigment {color Red}
  normal{bumps 1 scale 0.08}
  finish{phong 0.4}
}



// Return a random unit vector with uniform distribution
#macro RAND_UNIT(R)
  #local x1 = 2*rand(R)-1;
  #local x2 = 2*rand(R)-1;
  #local s1 = x1*x1;
  #local s2 = x2*x2;;  
  #while (s1 + s2 >= 1)
    #local x1 = 2*rand(R)-1;
    #local x2 = 2*rand(R)-1;
    #local s1 = x1*x1;
    #local s2 = x2*x2;  
  #end
  #local V = <2*x1*sqrt(1-s1-s2), 2*x2*sqrt(1-s1-s2), 1-2(s1+s2)>;
  <V.x,V.y,V.z>
#end

// The above was derived from:
// Weisstein, Eric W. "Sphere Point Picking." 
// From MathWorld -- A Wolfram Web Resource.
// http://mathworld.wolfram.com/SpherePointPicking.html 

// Return a random vector within the unit ball with uniform distribution
#macro RAND_BALL(R) 
  #local a = rand(R);
  // Swapped to a non-uniform distribution yielding more pleasing images
  (1 - (1-a)*(1-a)*(1-a))*RAND_UNIT(R)
  //(1 - (1-a)*(1-a))*RAND_UNIT(R)
#end


#macro NEURON(i,R)
union
{
  #local POS    = NEURON_CELL[i];
  #local DIR    = NEURON_AXON[i];
  #local ACTIVE = NEURON_STAT[i];

  #local NEURON_SHAPE = blob     
  {
    #local UDIR = DIR/vlength(DIR);
    sphere {0, CELL_RADIUS,1}
    sphere {0.3 * CELL_RADIUS * UDIR, CELL_RADIUS/2,1}
    sphere {0.4 * CELL_RADIUS * vrotate(UDIR,70*x), CELL_RADIUS/3,1}
    sphere {0.4 * CELL_RADIUS * vrotate(UDIR,-150*y), CELL_RADIUS/3,1}
    sphere {0.4 * CELL_RADIUS * vrotate(UDIR,80*z), CELL_RADIUS/3,1}
    threshold 0.6
    scale 2*<1,0.7,1>
  }
  
  object
  { 
    NEURON_SHAPE
    #if (ACTIVE)
      texture {T_CELL_ACTIVE}
    #else    
      texture {T_CELL}
    #end
  }
  
  sphere_sweep
  {
    b_spline 8,
    0            , 2*AXON_RADIUS
    0 + 0.001*DIR, 2*AXON_RADIUS
    0 + 0.002*DIR, 2*AXON_RADIUS
    DIR/3+x/2, AXON_RADIUS
    2*DIR/3-x/2, AXON_RADIUS
    0.998*DIR, AXON_RADIUS
    0.999*DIR, AXON_RADIUS
    1.000*DIR, AXON_RADIUS
    texture {T_AXON}
  }
  #if (ACTIVE)
    sphere 
    {
      0, CELL_RADIUS*2.0
      hollow pigment {color rgbt 1}
      interior
      {
        media
        {
          emission 5*Yellow
          density {spherical turbulence 0.2 scale 1.2}
        }      
      }
    }  
    
    sphere_sweep
    {
      b_spline 8,
      0            , 3*AXON_RADIUS
      0 + 0.001*DIR, 3*AXON_RADIUS
      0 + 0.002*DIR, 3*AXON_RADIUS
      DIR/3+x/2, 1.5*AXON_RADIUS
      2*DIR/3-x/2, 1.5*AXON_RADIUS
      0.998*DIR, 1.5*AXON_RADIUS
      0.999*DIR, 1.5*AXON_RADIUS
      1.000*DIR, 1.5*AXON_RADIUS
    
      hollow pigment {color rgbt 1}
    
      interior
      {
        media
        {
          emission 5*Yellow
          density 
          {
            gradient -DIR color_map
            {
              [0.0  rgb 0]
              [0.1  rgb 0]
              [1.0  rgb 1]
            }
            scale vlength(DIR)+3*AXON_RADIUS
            turbulence 0.1
          }
        }
      }
    }
  #end
  
  translate POS
}
#end


#macro CONNECT(i,j)
  #local V = NEURON_CELL[j]-(NEURON_CELL[i]+NEURON_AXON[i]);

  #if (vlength(V) < CONN_DIST)

sphere_sweep
{
  b_spline 8,
  NEURON_CELL[i]+NEURON_AXON[i], AXON_RADIUS
  NEURON_CELL[i]+NEURON_AXON[i], AXON_RADIUS
  NEURON_CELL[i]+NEURON_AXON[i], AXON_RADIUS
  NEURON_CELL[i]+1.5*NEURON_AXON[i], CONN_RADIUS
  NEURON_CELL[j]-0.7*NEURON_AXON[j], CONN_RADIUS  
  NEURON_CELL[j]-0.002*NEURON_AXON[j], CONN_RADIUS  
  NEURON_CELL[j]-0.001*NEURON_AXON[j], CONN_RADIUS  
  NEURON_CELL[j], CONN_RADIUS  
}
        
  #declare NEURON_CONN[i] = NEURON_CONN[i] + 1;
  #declare NEURON_CONN[j] = NEURON_CONN[j] + 1;
            
  #end

#end

union
{


// Generate some random neuron distribution

#declare i = 0;
#while (i < NEURON_NUM)
  #declare NEURON_CELL[i] = SPACE_MAX*RAND_BALL(R);
  #declare NEURON_AXON[i] = (AXON_LENGTH_MIN + rand(R)*(AXON_LENGTH_MAX-AXON_LENGTH_MIN))*RAND_UNIT(R);
  #declare NEURON_STAT[i] = (rand(R) < 0.3);
  #declare NEURON_CONN[i] = 0;
  #declare i = i+1;
#end


// Eye-catcher
#declare NEURON_CELL[0] = <-3,-1,-36>;
#declare NEURON_AXON[0] = < 4,1,2>;
#declare NEURON_STAT[0] = 1;



// For each pair of neurons, connect them if the axon of the first is close to the cell of the second

union
{
#declare i = 0;
#while (i < NEURON_NUM)
  #declare j = 0;
  #while (j < i)  
    CONNECT(i,j)
    CONNECT(j,i)
    #declare j = j+1;
  #end  
  #declare i = i+1;
#end
  texture {T_CONN}
}

union
{
#declare i = 0;
#while (i < NEURON_NUM)
  #if (NEURON_CONN[i] > 0)
    NEURON(i,R)
  #end
  #declare i = i+1;
#end
}

  scale 0.5
}
