


/*_______________________________________________________________________________________________________*\

FILE NAME: Make_Scatter_Tree.inc
CREATED BY: Isaac Kulka, Copyright 12-28-2006
NOTES: Created for the Internet Raytracing Competition's Nov-Dec 2006 Contest: 'Complexity' (www.irtc.org)

DESCRIPTION: The macros included in this file can be used to create very simple (non-memory-intensive) trees intended for use in a distant background.

\*_______________________________________________________________________________________________________*/

       




       

  //                                                            \\
 //----------------------MAKE SCATTER TREE-----------------------\\
//                                                                \\

// This macro is called by the Populate_Scatter_Tree_Array() macro to start creating creating a new maples tree, birch tree, or blue-hued background tree for the various SCATTER_TREE_ARRAY's.


#macro Make_Scatter_Tree(HEIGHT, NORM, COLOR_1, SEED)

/**********   VARIABLES   **************\
HEIGHT -    Float -     height of the trunk of the tree
NORM -      <x,y,z> -   direction in which the trunk of the tree grows (typically <0,1,0>)
COLOR_1 -   <r,g,b> -   color of the leaves for the tree
SEED -      Integer -   random seed
\***************************************/  


 
  #declare BRANCH_CNT = 0;
  #declare SCATTER_LEAF_CNT = 0;
  #declare BRANCH_TRIANGLE_CNT = 0;
  
    
  #declare TOTAL_HEIGHT = HEIGHT;
  #declare TOTAL_RADIUS = THICKNESS*HEIGHT;
  
  #local LEVEL = 0;
  #local START = <0,0,0>;
  
  mesh
  {
    //this macro goes on to call the recusive Make_Scatter_Branch() macro, which will complete the rest of the tree
    Make_Scatter_Branch(LEVEL, START, HEIGHT, THICKNESS, NORM, COLOR_1, SEED)
    
    //TREE_TYPE is a global variable defined in the Populate_Scatter_Tree_Array() macro which calls this macro
    #if(TREE_TYPE = 0)
      texture{Scatter_brown_texture} //Maple Tree
    #end

    #if(TREE_TYPE = 1)
      texture{Scattered_Birch_Texture scale 1 translate<0,rand(R1)*1000,0>} //Birch Tree
    #end
  }

#end //Make_Scatter_Tree()







  //                                                            \\
 //----------------------MAKE SCATTER BRANCH---------------------\\
//                                                                \\       

// This is a recursive macro, called by Make_Scatter_Tree().  It will complete the creation of an individual tree.
  
  
#macro Make_Scatter_Branch(LEVEL, START, HEIGHT, THICKNESS, NORM, COLOR_1, SEED)

/**********   VARIABLES   **************\
LEVEL -        Integer - current level of recursion
START -        <x,y,z> - base point of the branch
HEIGHT -       Float -   The length of the current branch
THICKNESS -    Float -   radius of base of the current branch
NORM -         <x,y,z> - this determines the direction in which the base of the branch points as it juts out from its parent branch
COLOR_1 -      rgb -     primary color of the leaves
SEED -         Integer - random seed
\***************************************/  


  #if(LEVEL < LEVEL_MAX)
  
    #local R1 = seed(SEED);
    
    #local norm1 = NORM;
    
   
    //MAKE SPLINE
    //this spline will define the shape of the branch
    #local branch_spline = spline
    {
      natural_spline
      //linear_spline
      
      #local norm2 = vnormalize(<norm1.x+0.1,norm1.y,norm1.z>);
      
      
      #local norm_perp = vnormalize(vcross(norm1, norm2));
      
      
      #local i_step = 0.2;
      #local i = i_step;
      
  
      //initial points
      -i_step, <0,-i_step,0>
      
      0.00, <0,0,0>
      
     
      #local height_step = HEIGHT/(1/i_step);
      
      #while(i <= 1)
        
        #local height_ratio = (HEIGHT*i + TOTAL_HEIGHT-HEIGHT)/TOTAL_HEIGHT;
        
        #local theta = rand(R1)*360;
        
        #local rad = rand(R1)*HEIGHT*THICKNESS*height_ratio*CURVINESS*5;
        
        #local point = norm_perp*rad;
        
        #local final_point = vaxis_rotate(point, norm1, theta)  + norm1*HEIGHT*i;
        
        i, final_point   
        
      
        #local i = i + i_step;
      #end
   
      1.10, norm1*(HEIGHT+1)
    
    }
    
    

    
    
    
    #if(HEIGHT > 0.2) //try to avoid degenerate meshes
      
      
      #local h_step = min(HEIGHT/2,(1-SMOOTHNESS)/2); //distance between points in the branch mesh
      #local h = -h_step;
      #local base_rad = HEIGHT*THICKNESS; //bottom-most radius of the branch
      #local points = max(3,ceil(base_rad*10/(1-SMOOTHNESS))); //number of points per circular cross-section
      #local rad = base_rad; //radius of the current cross-section
      
      #local lower = array[points];
      #local upper = array[points];
      
      #local l_norm = array[points]; //norms of the points of the lower array (used for smooth triangles in the mesh)
      #local u_norm = array[points]; //norms of the points of the upper array
      
      #local dspline = branch_spline(0.01) - branch_spline(0); //vectoral direction of the branch at its bottom-most cross-section
      #local norm = vnormalize(dspline); //normalized dspline vector
      #local norm2 = vnormalize(<norm.x+0.1,norm.y,norm.z>);
      #local norm_perp = vnormalize(vcross(norm, norm2)); //normal vector perpendicular to branch (used to rotate points around norm in order to give your branch some thickness)
      
      //set initial lower points
      #local i=0;
      #while(i < points)
        #local theta = (i/points)*360; //current rotation around the current cross-section
        
        #local point1 = norm_perp*rad; //a single point sticking out a distance of rad from the normal vector
        #local point1 = vaxis_rotate(point1, norm, theta) + branch_spline(h/HEIGHT) + START;
        
        #local point = norm_perp*rad;
        #local lower[i] = vaxis_rotate(point, norm, theta) + branch_spline(h/HEIGHT) + START; //to dermine all the points in the cross-section, start rotating point around norm
        
        #local l_norm[i] = vnormalize(lower[i] - (branch_spline(h/HEIGHT) + START)); //determine the normal of each point (to be used with smooth triangles)
        
        #local i = i + 1;
      #end
      
      #local h = h + h_step;
    
    
    
      #while(h < HEIGHT)
        
        #local h_ratio = h/HEIGHT;
        #local rad = max(base_rad*(1 - h_ratio), 0.0025);
        #local bark_thickness = 0.11 * (1-h_ratio + 0.1);
        
        #local dspline = branch_spline(h_ratio+0.01) - branch_spline(h_ratio); //direction of bottom-most cross-section
        
        #if((dspline.x != 0) | (dspline.y != 0) | (dspline.z != 0)) //catch zero-length vectors
        
          #local norm = vnormalize(dspline);
          #local norm2 = vnormalize(<norm.x+0.1,norm.y,norm.z>);
          #local norm_perp = vnormalize(vcross(norm, norm2));
          
          //set upper points
          #local i=0;
          #while(i < points)
            #local theta = (i/points)*360;
            
            #local point1 = norm_perp*rad; //a single point sticking out a distance of rad from the normal vector
            #local point1 = vaxis_rotate(point1, norm, theta) + branch_spline(h/HEIGHT) + START;
        
            #local point = norm_perp*rad;
            #local upper[i] = vaxis_rotate(point, norm, theta) + branch_spline(h/HEIGHT) + START; //to dermine all the points in the cross-section, start rotating point around norm
        
            //smooth normal
            #local u_norm[i] = vnormalize(upper[i] - (branch_spline(h_ratio) + START));
           
            
            #local i = i + 1;
          #end
          
          //create triangles
          #local i = 0;
          #while(i < points)
            #local j = i+1;
            #if(j >= points) #local j=0; #end
            
            #if(LEVEL <= 1)
              triangle{upper[i], lower[i], lower[j]}
              triangle{upper[j], lower[j], upper[i]}
            #else
              smooth_triangle{upper[i], u_norm[i], lower[i], l_norm[i], lower[j], l_norm[j]}
              smooth_triangle{upper[j], u_norm[j], lower[j], l_norm[j], upper[i], u_norm[i]}
            #end
            
    
            
            #local i = i + 1;
          #end
          
          //set lower equal to upper
          #local i = 0;
          #while(i < points)
            #local lower[i] = upper[i];
            #local l_norm[i] = u_norm[i];
            
            #local i = i + 1;
        #end
          
        #end //end if-statement that catches zero-length vectors
          
        
        #local h = h + h_step;
      #end
            
    #end       
      
      
      
      
      
      
    //MAKE BRANCHES
    
    
    #local h = 0;
    #local branch_step = 0.50 * (1 - LEVEL/LEVEL_MAX);
    #local branch_prob = 0.35;
    #local min_branch_height = HEIGHT*0.20;//0.20
    #local max_branch_height = HEIGHT*0.90;//0.90
    #local theta = 0;
    
    #while(h < HEIGHT)
      
      #local h_ratio = h/HEIGHT;
      
      #if((h > min_branch_height) & (h < max_branch_height) & (rand(R1) < branch_prob + h_ratio*(1-branch_prob)*0.50))
        
        #local phi = rand(R1)*20*(1-min(max(base_rad*(1 - h_ratio), 0.01)/TOTAL_RADIUS, 0.20)) + 30; //rotation away from branch axis
        #local theta = theta + rand(R1)*45 + 45; //rotation around branch axis
        
        #local dspline = vnormalize(branch_spline(h_ratio+0.1) - branch_spline(h_ratio));
        #if((dspline.x != 0) | (dspline.y != 0) | (dspline.z != 0))
          #local norm = vnormalize(dspline);
        #else
          #local norm = NORM;
        #end
        
        #local norm2 = vnormalize(<NORM.x+0.1,NORM.y,NORM.z>);
        #local norm_perp = vnormalize(vcross(NORM, norm2));
        #local norm1 = vnormalize(vaxis_rotate(NORM, norm_perp, phi));
        #local norm1 = vnormalize(vaxis_rotate(norm1, NORM, theta));
        
      
        #local Level = LEVEL + 1;
        #local Thickness = THICKNESS*0.75;
        #local Start = branch_spline(h_ratio)+START;
        #local Height = (rand(R1)*0.25 + 0.75)*(HEIGHT-h);
                
        //determine the color trend of the new branch
        #local C_Var = rand(R1)*C_VAR;
        #local Color = rgb<COLOR_1.red*(1-C_Var) + COLOR_2.red*C_Var, COLOR_1.green*(1-C_Var) + COLOR_2.green*C_Var, COLOR_1.blue*(1-C_Var) + COLOR_2.blue*C_Var>; 
        
        //try to avoid branches that point downward too early in the tree's structure
        #local height_ratio = (h + TOTAL_HEIGHT - HEIGHT)/TOTAL_HEIGHT;
        
        
        
        #local make_branch = true;
        #if((norm1.y < 0) & (height_ratio < 0.70)) //don't make branches if they are pointing down in the negative y-direction, unless you are less than 70% of the way up the tree (in that case the branches can do whatever they want)
          #local make_branch = false;
        #end
        
        //spawn the next branch recursively (as long as the maximum level of recursion hasn't been reached)
        #if(LEVEL < LEVEL_MAX-1 & make_branch)
          Make_Scatter_Branch(Level, Start, Height, Thickness, norm1, Color, rand(R1)*1000)
        #end
      #end
      
      #local h = h + branch_step;
          
    #end
    
    
    
    
    
    //ADD LEAVES
    //Each leaf in a scatter tree is only a single triangle in the mesh
    #local leaf_max = floor(LEAFINESS*100); // scatter this many leaves around each branch tip in the tree 
    #local leaf_texture = texture{ pigment{COLOR_1 filter FILTER} finish{ambient AMBIENT diffuse 0.8 specular LUSTER roughness 0.2}}
      
    #local i = 0;
    #while(i < leaf_max)
      
      //MAKE LEAF TRIANGLE
      #local p1 = 
        <branch_spline(0.90).x+START.x + rand(R1)*0.50,  
         branch_spline(0.90).y+START.y + rand(R1)*0.50, 
         branch_spline(0.90).z+START.z + rand(R1)*0.50>;
      
      #local rad = LEAF_LENGTH;
      #local theta = rand(R1)*2*pi;
      #local phi = rand(R1)*pi;
      
      #local p2 = 
        <p1.x + rad*cos(theta)*sin(phi),
         p1.y + rad*sin(theta)*sin(phi),
         p1.z + rad*cos(phi)>;
      
      #local norm = vnormalize(p2-p1);
      
      #local rot = rand(R1)*360;
      #local p_rot = vaxis_rotate(<LEAF_LENGTH,0,0>,norm,rot);
      
      #local p3 = 
        <p_rot.x + p1.x,
         p_rot.y + p1.y,
         p_rot.z + p1.z>;
      
      triangle{p1, p2, p3 texture{leaf_texture}}
        
      
      #local i = i + 1; 
    #end
    
  
  #end //if-statement at beginning of macro
  
#end //Make_Scatter_Branch() 



                               


   
   
   
   
   

  //                                                         \\
 //---------------POPULATE SCATTER TREE ARRAY-----------------\\
//                                                             \\   

// Use this macro to create an array of birch trees or maple trees.


#macro Populate_Scatter_Tree_Array(SCATTER_TREE_CNT, VARIETY, TREE_TYPE, LEVEL_MAX, HEIGHT, THICKNESS, SMOOTHNESS, NORM, CURVINESS, LEAFINESS, LEAF_LENGTH, COLOR_1, COLOR_2, C_VAR, FILTER, LUSTER, AMBIENT, SEED)

/**********   VARIABLES   **************\
SCATTER_TREE_CNT - Integer - total number of individual trees to generate
VARIETY -      0-1 -     determines how much certain values can deviate below the maximum specified values 

TREE_TYPE -    Integer - determines which kind of tree to make.  0: Maple, 1: Birch
LEVEL_MAX -    Integer - number of levels of recursion to perform on each tree.

HEIGHT -       Float -   maximum hieght of any given tree
THICKNESS -    0-1 -     radius of base of the tree trunk, as a percentage of the height
SMOOTHNESS -   0-1 -     determines the number of trianges in the mesh, higher value means more triangles

NORM -         <x,y,z> - determines the direction in which the trunk of each tree grows (typically <0,1,0>)

CURVINESS -    0-1 -     determines how much the branches curve
LEAFINESS -    0-1 -     determines how many leaves each tree will have
LEAF_LENGTH -  Float -   size of the leaves\

COLOR_1 -      <r,g,b> - primary color of the leaves
COLOR_2 -      <r,g,b> - secondary color of the leaves
C_VAR -        0-1 -     amount by which the primary color may deviate to the secondary color in any given leaf

FILTER -       0-1       filtered light value of the leaves
AMBIENT -      0-1       ambient value of the leaves
LUSTER -       0-1       specular highlight of the leaves

SEED -         Integer - random seed
\***************************************/   


//declare these local variables as global variables to be referenced by Make_Scatter_Branch()
#declare TREE_TYPE = TREE_TYPE;
#declare LEVEL_MAX = LEVEL_MAX;
#declare THICKNESS = THICKNESS;
#declare SMOOTHNESS = SMOOTHNESS;
#declare CURVINESS = CURVINESS;
#declare LEAFINESS = LEAFINESS;
#declare LEAF_LENGTH = LEAF_LENGTH;
#declare COLOR_2 = COLOR_2;
#declare C_VAR = C_VAR;
#declare FILTER = FILTER;
#declare LUSTER = LUSTER;
#declare AMBIENT = AMBIENT;

#local R1 = seed(SEED);
#local INV_VAR = 1-VARIETY;



//global values
#if(TREE_TYPE = 0) //Maple Tree
  #declare SCATTER_MAPLE_ARRAY = array[SCATTER_TREE_CNT];
  #declare SCATTER_MAPLE_MAX = SCATTER_TREE_CNT;
#end

#if(TREE_TYPE = 1) //Birch Tree
  #declare SCATTER_BIRCH_ARRAY = array[SCATTER_TREE_CNT];
  #declare SCATTER_BIRCH_MAX = SCATTER_TREE_CNT;
#end


//populate the array with unique sumac plants
#local i = 0;
#while(i < SCATTER_TREE_CNT)
  #local Height = HEIGHT * (rand(R1)*VARIETY + INV_VAR);
  #local C_Var = C_VAR*rand(R1)*4;
  #local Color_1 = rgb<COLOR_1.red*(1-C_Var) + COLOR_2.red*(C_VAR), COLOR_1.green*(1-C_Var) + COLOR_2.green*(C_VAR), COLOR_1.blue*(1-C_Var) + COLOR_2.blue*(C_VAR)>;
  #local Seed = ceil(rand(R1)*10000);
  
  #if(TREE_TYPE = 0)
    #declare SCATTER_MAPLE_ARRAY[i] = Make_Scatter_Tree(Height, NORM, Color_1, Seed)
  #end
  
  #if(TREE_TYPE = 1)
    #declare SCATTER_BIRCH_ARRAY[i] = Make_Scatter_Tree(Height, NORM, Color_1, Seed)
  #end
                                                
  #local i = i + 1;
#end                         
  

#end //Populate_Scatter_Tree_Array()
                        



           
           
           
           
           
          

  //                                                         \\
 //------------------SCATTER TREE BARK TEXTURES---------------\\
//                                                             \\ 



#declare Scatter_brown_texture =
texture
{
  pigment 
  {
    
    granite
    
    color_map
    {
      [-0.1 rgbft<0.45,0.25,0.15, 0.0, 0.0>] //dark brown
      [ 0.6 rgbft<0.55,0.35,0.15, 0.0, 0.0>] //dark brown
      [ 1.1 rgbft<0.90,0.80,0.70, 0.0, 0.0>] //light orangish-gray
    }
    
    scale 1*<1,8,1>

  }
  
  finish
  {
    ambient 0.00
    diffuse 0.8 
    specular 0.10 
    roughness 0.4 
  }
  
  
  normal
  {   
    granite
    bump_size 0.4
    scale 0.7 
  }
  
}




//this texture is used with the far distant blueground trees
#declare Scatter_blue_texture =
texture
{
  pigment 
  {
    rgb<0.20,0.50,0.35>
  }
  
  finish
  {
    ambient 0.00
    diffuse 0.8 
    specular 0.10 
    roughness 0.4 
  }
  
  
  normal
  {   
    granite
    bump_size 0.4
    scale 0.7 
  }
  
}





//Birch Bark
#declare Scattered_Birch_Texture =
texture
{
  pigment
  {
    granite
    
    turbulence 0.60
    lambda 0.50
    octaves 2
    omega 0.1
    
    color_map
    {
      [0.00 rgb 1]
      [0.55 rgb 1]
      [0.55 rgb <0.9,0.7,0.5>]
      [0.75 rgb <0.9,0.7,0.5>]
      [0.75 rgb 0]
      [1.00 rgb 0]
  
    }
      
    scale <4,0.7,4>  
  } 
 
  finish            
  {                 
     ambient 0.00   
     diffuse 0.90   
     specular 0.10  
     roughness 0.025     
  } 
}                

             
