

/*_______________________________________________________________________________________________________*\

FILE NAME: Make_Sloped_Height_Field.inc
CREATED BY: Isaac Kulka, Copyright 12-28-2006
NOTES: Created for the Internet Raytracing Competition's Nov-Dec 2006 Contest: 'Complexity' (www.irtc.org)

DESCRIPTION: The macro in this file can be used to create a hilly height-field which slopes up gradually away from the camera along the positive z-direction.

\*_______________________________________________________________________________________________________*/






  //                                                            \\
 //-------------------MAKE SLOPED HEIGHT FIELD-------------------\\
//                                                                \\

#macro Make_Sloped_Height_Field(START, END, HEIGHT, STEP, SMOOTH, SLOPED_PERCENT, PRIMARY_FIELD, TEXTURE, FUNC) 

/**********   VARIABLES   **************\
HEIGHT -       Float -    length of the blade of grass from base to tip
START -        <x,y,z> -  starting corner of the height_field (note: the y value in STARt is the only one that counts)
END -          <x,y,z> -  ending corner of the height_field (note: the y value in END is ignored)
HEIGHT -       Float -    height of the height_field
STEP -         Float -    size of each triangle in the field
SMOOTH -       Boolean -  determines whether the mesh will use smooth triangles or not
SLOPED_PERCENT 0-1 -      how quickly or slowly the height-field slopes up from zero to its maximum height as it move away from the camera in the positive z-direction
PRIMARY_FIELD  Boolean -  determines whether this height-field's fucntion will be used by other pov files (like Grassland_Foreground.pov) to calculate where to plant grass and trees
PIGMENT -      Pigment -  pigment of the height_field
FINISH -       Finish  -  finish of the height_field
FUNC -         Function - the bumpy hill function used with the height_field
\***************************************/ 






//special note, the y-value in START is the defining y-value.  The y-value in END is not used

#local triangle_cnt = 0;

#local x_start = min(START.x,END.x);
#local z_start = min(START.z,END.z);
#local y_start = START.y;

#local x_max = max(START.x,END.x);
#local z_max = max(START.z,END.z);

#local mx = x_start;
#local mz = z_start;




//this function describes a plane sloping upwards in the positive z-direction
#local GRAD_FUNC =
function
{ 
  pattern
  {
    gradient y
    
    scale <1,z_max-z_start,1>
    translate<0,z_start,0>
  }
}


//Slope the height_field down toward zero height as it moves in the negative z-direction is SLOPED is true

#if(SLOPED_PERCENT != 0)
  #local HF_FUNC = function{(FUNC(x,y,0)) * min(GRAD_FUNC(x,y,0)*(1/SLOPED_PERCENT),1)};
#else
  #local HF_FUNC = function{FUNC(x,y,0)};
#end




//initialize points
#local p1 = <mx, HF_FUNC(mx,mz,0)*HEIGHT + y_start, mz>;
#local p2 = <mx, HF_FUNC(mx,mz+STEP,0)*HEIGHT + y_start, mz+STEP>;
#local p3 = <0,0,0>;
#local p4 = <0,0,0>;


//create the height_field
mesh
{

  #while(mz < z_max)
      
    #local p3 = <mx+STEP, HF_FUNC(mx+STEP,mz     ,0)*HEIGHT + y_start, mz>;
    #local p4 = <mx+STEP, HF_FUNC(mx+STEP,mz+STEP,0)*HEIGHT + y_start, mz+STEP>;
    
    #if(SMOOTH)
      smooth_triangle{p2, smooth_norm(p2), p1, smooth_norm(p1), p3, smooth_norm(p3)}
      smooth_triangle{p2, smooth_norm(p2), p3, smooth_norm(p3), p4, smooth_norm(p4)}
    #else
      triangle{p2, p1, p3}
      triangle{p2, p3, p4}
    #end
    
    #local triangle_cnt = triangle_cnt + 2;
    
    #local p1 = p3;
    #local p2 = p4;
    
    #local mx = mx + STEP;
    #if(mx >= x_max)
      #local mx = x_start;
      #local mz = mz + STEP;
      
      #local p1 = <mx, HF_FUNC(mx,mz     ,0)*HEIGHT + y_start, mz>;
      #local p2 = <mx, HF_FUNC(mx,mz+STEP,0)*HEIGHT + y_start, mz+STEP>;
    #end
  #end 

  #debug concat("\n\n*********** Height_Field Triangles = ", str(triangle_cnt,0,0), " ***************\n\n")
  
  #declare HF_TRI = triangle_cnt;
  
  texture{TEXTURE}
} 



//declare the HF_FUNC as the global P_FUNC if you intend to reference this function in your main scene
#if(PRIMARY_FIELD)
  
  #ifdef(P_FUNC)
    #warning "\n*********** WARNING! PRIMARY FIELD ALREADY DEFINED! REDEFINING PRIMARY FIELD... ******************"
    #undef P_FUNC
  #end
    
  #declare P_FUNC = function{HF_FUNC(x,y,z)}
  
#end


#end //Make_Sloped_Height_Field() 




   
   
   
   



  //                                                            \\
 //-------------------------SMOOTH_NORM--------------------------\\
//                                                                \\


// This macro is used by the above macro to calculate the normals necessary for smooth triangles


#macro smooth_norm(POINT)
  
/**********   VARIABLES   **************\
POINT -   <x,y,z> - the point on the height_field whose normal will be calculated by this macro
\***************************************/ 
  
  
//this macro is called when smooth triangles are called for use 
#local px = POINT.x;
#local py = POINT.y;
#local pz = POINT.z;

#local dx = 0.50*0.00001;
#local dz = 0.866025*0.00001;

#local p1 = <px-dx, HF_FUNC(px-dx, pz-dz, 0)*HEIGHT, pz-dz>;
#local p2 = <px,    HF_FUNC(px,    pz+dz, 0)*HEIGHT, pz+dz>;
#local p3 = <px+dx, HF_FUNC(px+dx, pz-dz, 0)*HEIGHT, pz-dz>;

vnormalize(vcross(p2-p1,p3-p1))
  
#end //smooth_norm()