 
 
/*_______________________________________________________________________________________________________*\

FILE NAME: Grassland_Foreground.pov
CREATED BY: Isaac Kulka, Copyright 12-28-2006
NOTES: Created for the Internet Raytracing Competition's Nov-Dec 2006 Contest: 'Complexity' (www.irtc.org)

DESCRIPTION: This files is used to generate the foreground of my grassland scene.  The scene includes multiple layers of grass, trees, flowers and shrubs.

\*_______________________________________________________________________________________________________*/





#include "Make_Sloped_Height_Field.inc"
#include "Make_Grass.inc"
#include "Make_Seeded_Grass.inc"
#include "Make_Flower_Susan.inc"
#include "Make_Flower_stalk.inc"
#include "Make_Sumac.inc"
#include "Make_Tree.inc"


//Use these values to enable or disable time-consuming effects from the scene
#local use_area_lights = false;

#local use_radiosity   = false;

#local use_filter = false;
     
     
#local render_grass       = true; //main grass
#local render_tuff_grass  = true; //clump grass
#local render_sgrass      = true; //seeded grass
#local render_susans      = true; //black-eyed susans
#local render_flower_stalk= true; //purple flower_stalk
#local render_trees       = true; //scatter trees
#local render_sumac       = true; //sumac





global_settings
{
  max_trace_level 100
  
  #if(use_radiosity)
    radiosity
    {
  
       
       count 75 //75
       nearest_count 3
       error_bound 3.5 //1.8
       recursion_limit 3 //2
       
        
       pretrace_start .08
       pretrace_end   .01
                           
                           
       low_error_factor 0.25
       minimum_reuse 0.015
       
       //max_sample 0.40
       
       adc_bailout 0.1
       brightness 1.00 
       gray_threshold 0.00 
       
       
       
       always_sample off
       media off          
       normal off  
    }
  #end
  

  
}


background{rgb 0}  
 
 
light_source
{
  <100,500,-1000>
  
  //rgb 1.0
  rgb <1.0, 0.8, 0.0>*0.30 + <1.0, 1.0, 1.0>*0.70
  
  
  #if(use_area_lights)
    area_light <-25, 25, 0>, <25 -25, 10>, 2, 2
    jitter
  #end
}   





//Shadowless Light
light_source
{
  <000,100,-1000>
  
  rgb <1,0.85,0>*0.20 //0.20
  shadowless
}


#local SEED = 6765;
#local R1 = seed(SEED);  
  
  
  
  
 
 

  //                                               \\
 //------------ FOREGROUND HEIGHT_FIELD ------------\\
//                                                   \\

//this hieght field is created using a function which will define not only the height-field's shape, but also where the plants are planted on it.


#local hill_func=
function
{
  pattern
  {
    granite 
    turbulence 0.15
    
    scale 900
    translate y*7875 
  }
}



#local HF_X_START = -55;
#local HF_Z_START = 0;
#local HF_Y_START = 0;

#local HF_X_END = 50;
#local HF_Z_END = 100;
#local HF_Y_END = HF_Y_START;

    
    
    
#local HF_START = <HF_X_START, HF_Y_START, HF_Z_START>;
#local HF_END = <HF_X_END, HF_Y_END, HF_Z_END>;

#local HF_HEIGHT = 25;
#local HF_STEP = 3;
#local HF_SMOOTH = false;

#local HF_SLOPED_PERCENT = 0.85;

#local HF_PRIMARY_FIELD = true;

#local HF_TEXTURE = 
  texture
  {
    pigment{ rgb<0.65,0.80,0.43> }
    finish{ambient 0 diffuse 0.7 specular 0.5 roughness 0.2 crand 0.1}
  }



Make_Sloped_Height_Field(HF_START, HF_END, HF_HEIGHT, HF_STEP, HF_SMOOTH, HF_SLOPED_PERCENT, HF_PRIMARY_FIELD, HF_TEXTURE, hill_func) 















  //                                               \\
 //------------------ WIND FUNCTIONS ---------------\\
//                                                   \\
#local x_wind_func = 
function
{
  pattern
  {
    agate 
      octaves 9
      lambda 2
      omega 0.1
      turbulence 5
      
    scale 50
    rotate y*-100
    translate y*56576
  }
}


#local y_wind_func = 
function
{
  pattern
  {
    agate 
      octaves 9
      lambda 2
      omega 0.1
      agate_turb 2
  
    scale 50 
    rotate y*120
    translate y*32
  }
}

#local x_wind_bend_max = 55;
#local x_wind_bend_min = 10;
#local y_wind_bend = 10;
#local min_grass_bend = 20;











  //                                                  \\
 //---------- POPULATE HEIGHT FIELD -------------------\\
//                                                      \\

//This macro is responsible for plants the various trees, grasses and other plants on the height-field in this scene


#macro Populate_Height_Field(DIST_FUNC, ARRAY, ARRAY_MAX, X_MAX, Z_MAX, X_START, Z_START, STEP, PROB, WIND_BEND, Y_WIND_ROT, FUNC_TYPE)
  
  #declare CNT = 0;
  
  #local Z = Z_START;
  #local X = X_START;
  #local INV_PROB = 1-PROB;

  
  #while (Z < Z_MAX)
    
    #local rx = rand(R1)*STEP + X;
    #local rz = rand(R1)*STEP + Z;
    
    #local ry = 2;
    #if(Z >= 0)
      #local ry = P_FUNC(rx,rz,0)*HF_HEIGHT; //set the y-value to the y-coordinate of the height_field at <rx,rz>
    #end
      
    #local prob = DIST_FUNC(rx,0,rz) - INV_PROB;
    
     
     
     
     
     
    //PLANT ANYTHING BUT GRASS
    #if((rand(R1) < prob) & ((FUNC_TYPE = 0) | (FUNC_TYPE = 3)))
      object
      {
        ARRAY[floor(rand(R1)*ARRAY_MAX)] 
        
        scale rand(R1)*0.25 + 0.75 
        #if(Z < 0) scale 1.25 #end
         
        rotate z*((rand(R1)*0.15 + 0.85)*x_wind_func(rx,0,rz)*WIND_BEND + rand(R1)*15) 
        rotate y*(y_wind_func(rx,0,ry)*y_wind_bend)
        rotate y*(rand(R1)*2-1)*Y_WIND_ROT
        
        translate<rx,ry,rz>
      }
  
      
      #declare CNT = CNT + 1;
    #end
                                     
                                     
                                     
                                     
                                     
    //PLANT SUMAC CLOSE TO THE CAMERA
    #if((FUNC_TYPE = 3) & (Z < Z_START + 16) & (rand(R1) < 0.90*(1 - Z/(Z_START+16))))
      object
      {
        ARRAY[floor(rand(R1)*ARRAY_MAX)] 
        
        scale rand(R1)*0.25 + 0.75 
        #if(Z < 0) scale 1.25 #end
         
        rotate z*((rand(R1)*0.15 + 0.85)*x_wind_func(rx,0,rz)*WIND_BEND + rand(R1)*15) 
        rotate y*(y_wind_func(rx,0,ry)*y_wind_bend)
        rotate y*(rand(R1)*2-1)*Y_WIND_ROT
        
        translate<rx,ry,rz>
      }
  
      
      #declare CNT = CNT + 1;
    #end             
                 
                 
                 
    //PLANT WINDSWEPT GRASS
    #if(FUNC_TYPE = 1)    
      
      #local grass_val = DIST_FUNC(rx,0,rz);
      #local green_chance = rand(R1);
      
      object
      {
         
        #if(grass_val < 0.55 | green_chance < 0.10) //green grass
          GRASS_ARRAY_0[floor(rand(R1)*GRASS_MAX_0)]
        #end
        
        #if(grass_val >= 0.55 & grass_val < 0.65 & green_chance >= 0.10) //randomly pick between green and brown
          #if(rand(R1) < 0.50)
            GRASS_ARRAY_0[floor(rand(R1)*GRASS_MAX_0)]
          #else
            GRASS_ARRAY_1[floor(rand(R1)*GRASS_MAX_1)]
          #end
        #end
        
        #if(grass_val >= 0.65 & green_chance >= 0.10) //brown grass
          GRASS_ARRAY_1[floor(rand(R1)*GRASS_MAX_1)]  
        #end
        
        scale rand(R1)*0.25 + 0.75 
        #if(Z < 0) scale 1.25 #end
        
        rotate z*((rand(R1)*0.15 + 0.85)*x_wind_func(rx,0,rz)*WIND_BEND + min_grass_bend + rand(R1)*15) 
        rotate y*(y_wind_func(rx,0,ry)*y_wind_bend)
        rotate y*(rand(R1)*2-1)*Y_WIND_ROT
        
        translate<rx,ry,rz>
      }
   
      #declare CNT = CNT + 1;
    
    #end //make_grass 
    
    
    
    
    //PLANT CLUSTER GRASS
    #if(FUNC_TYPE = 2)    
      
      #local cluster_val = DIST_FUNC(rx,0,rz);
      #local empty_chance = rand(R1);
      
      
      #if(cluster_val < 0.35) //make very small grass cluster
      
        object
        {
          GRASS_CLUSTER_ARRAY[0]
          
          scale rand(R1)*0.35 + 0.15 
          
          rotate x*rand(R1)*5
          rotate y*rand(R1)*360
          
          translate<rx,ry,rz>
        }
      
      #end
    
      
      #if(cluster_val >= 0.35 & empty_chance >= 0.15)
      
        object
        {          
          #if(cluster_val < 0.425) //brown grass cluster
            GRASS_CLUSTER_ARRAY[1]
          #end
          
          #if(cluster_val >= 0.425 & cluster_val < 0.750) //green grass cluster
            GRASS_CLUSTER_ARRAY[0]
          #end
          
          #if(cluster_val >= 0.750) //large grass
            GRASS_CLUSTER_ARRAY[2]  
          #end
          
          scale rand(R1)*0.25 + 0.75 
          #if(Z < 0) scale 1.25 #end
          
          rotate x*rand(R1)*5
          rotate y*rand(R1)*360
          
          translate<rx,ry,rz>
        }
     
        #declare CNT = CNT + 1;
      
      #end
    
    #end //make_grass 
    
    
    
    
    #local X = X + STEP;
    #if(X >= X_MAX)
      #local X_MAX = min(X_MAX + STEP, HF_X_END);
      #local X = -X_MAX;
      #local Z = Z + STEP;
    #end
  #end
  
#end //Populate_Height_Field 










  //                                               \\
 //------------------ MAIN GRASS -------------------\\
//                                                   \\

//The grass is generated using the Make_Grass.inc file.  The values listed below are described in greater detail in that file.


#if(render_grass)

  #debug "\n\n************ PARSING MAIN GRASS ************\n\n"
  #local SEED = 6765;
  #local R1 = seed(SEED);
  
  
  #local GRASS_CNT = 8;
  #local VARIETY = 0.5;
  
  #local HEIGHT = 3.00; //4.75
  #local THICKNESS = 0.024;//0.028; //0.023 
  
  #local SMOOTHNESS = 0.70;
  #local SEED = 753;
  
  #local RAD_MIN = 0.20;
  #local RAD_MAX = 0.30;
  
  #local X_OFFSET = 0;
  #local Z_OFFSET = 0;
  #local X_BEND = -pi/3;
  #local TWIST = 0;
  
  #local STEM = 0.05;
  #local CENTER = 0.1;
  #local MIDDLE = 0.7;
  
  #local COLOR_1 = rgb<0.40,0.85,0.25>; //green
  #local COLOR_2 = rgb<0.65,0.88,0.35>; //green
  #local COLOR_VAR = 0.50;
  
  #local FILTER = 0.45; //0.40
    #if(!use_filter) #local FILTER = 0; #end
  #local LUSTER = 0.435;
  #local AMBIENT = 0.15;
    #if(use_radiosity) #local AMBIENT = 0.00; #end
  #local WRINKLINESS = 0.00;
  
  #local ARRAY_NUM = 0;
  
  //make green grass
  Populate_Grass_Array(GRASS_CNT, VARIETY, HEIGHT, THICKNESS, SMOOTHNESS, SEED, RAD_MIN, RAD_MAX, X_OFFSET, Z_OFFSET, X_BEND, TWIST, STEM,CENTER,MIDDLE, COLOR_1, COLOR_2, COLOR_VAR, LUSTER, AMBIENT, FILTER, WRINKLINESS, ARRAY_NUM)
  
    
    
    
  //make brown grass
  #local ARRAY_NUM = 1;
  #local SEED = 2439;
  #local COLOR_1 = rgb<0.89,0.78,0.45>; //brown
  #local COLOR_2 = rgb<0.94,0.80,0.55>; //brown
                      
  Populate_Grass_Array(GRASS_CNT, VARIETY, HEIGHT, THICKNESS, SMOOTHNESS, SEED, RAD_MIN, RAD_MAX, X_OFFSET, Z_OFFSET, X_BEND, TWIST, STEM,CENTER,MIDDLE, COLOR_1, COLOR_2, COLOR_VAR, LUSTER, AMBIENT, FILTER, WRINKLINESS, ARRAY_NUM)
    
  
  
  
  
  
  
   
  //populate HF with green and brown grass

  #local grass_func = 
  function
  {
    pattern
    {
      granite 
      turbulence 0.2  
    
      scale 100
      translate y*8493
      translate <10, 0 ,20>
    }
  } 
  
    
  #local ARRAY = GRASS_ARRAY_0;
  #local ARRAY_MAX = GRASS_MAX_0;
  
  #local X_MAX = 5;
  #local Z_MAX = HF_Z_END;
  #local X_START = -X_MAX;
  #local Z_START = -3;
  
  #local STEP = 0.35;//0.35
  
  #local PROB = 0.35; 
  
  #local WIND_BEND = 30;//35;
  #local Y_WIND_ROT = 8;
  
  #local FUNC_TYPE = 1;
    
  Populate_Height_Field(grass_func, ARRAY, ARRAY_MAX, X_MAX, Z_MAX, X_START, Z_START, STEP, PROB, WIND_BEND, Y_WIND_ROT, FUNC_TYPE)
  
  #debug concat("\n************ TOTAL BLADES OF GRASS = ", str(CNT,0,0), " ************\n\n")
  
#end 



      
      
      

  
  
  
  
  
  
  
  //                                               \\
 //------------------ CLUSTER GRASS ----------------\\
//                                                   \\

//The clusters of grass are generated using the Make_Grass.inc file.  The values listed below are described in greater detail in that file.


#if(render_tuff_grass)

  #debug "\n\n************ PARSING CLUMP GRASS ************\n\n"
  #local tuff_cnt = 0; 
  #local SEED = 22222;
  #local R1 = seed(SEED);
  
  
  //Thin Green Grass
  #local GRASS_CNT = 6;
  #local VARIETY = 0.3;
  
  #local HEIGHT = 2.25;
  #local THICKNESS = 0.033; //0.064 //the width of the blade will equal the HEIGHT times this number
  
  #local SMOOTHNESS = 0.70; //0.75
  #local SEED = 1234;
  
  #local RAD_MIN = 0.10;
  #local RAD_MAX = 0.20;
  
  #local X_OFFSET = 0.1;
  #local Z_OFFSET = 0.1;
  #local X_BEND = pi/4;
  #local TWIST = 2*pi;
  
  #local STEM = 0.1;
  #local CENTER = 0.2;
  #local MIDDLE = 0.7;
  
  #local COLOR_1 = rgb<0.40,0.80,0.30>; //green
  #local COLOR_2 = rgb<0.65,0.82,0.40>;
  #local COLOR_VAR = 0.75;//1.00
  
  #local FILTER = 0.45;//0.40;
    #if(!use_filter) #local FILTER = 0; #end
  #local LUSTER = 0.3;//0.40
  #local AMBIENT = 0.15;
    #if(use_radiosity) #local AMBIENT = 0; #end
  #local WRINKLINESS = 0.00;
  #local ARRAY_NUM = 2;
    
  Populate_Grass_Array(GRASS_CNT, VARIETY, HEIGHT, THICKNESS, SMOOTHNESS, SEED, RAD_MIN, RAD_MAX, X_OFFSET, Z_OFFSET, X_BEND, TWIST, STEM,CENTER,MIDDLE, COLOR_1, COLOR_2, COLOR_VAR, LUSTER, AMBIENT, FILTER, WRINKLINESS, ARRAY_NUM)
  
  
  
  //Thin Brown Grass
  #local SEED = 1234;
  
  #local COLOR_1 = rgb<0.89,0.78,0.45>; //brown
  #local COLOR_2 = rgb<0.94,0.80,0.55>; //brown
  
  #local ARRAY_NUM = 3;
    
  Populate_Grass_Array(GRASS_CNT, VARIETY, HEIGHT, THICKNESS, SMOOTHNESS, SEED, RAD_MIN, RAD_MAX, X_OFFSET, Z_OFFSET, X_BEND, TWIST, STEM,CENTER,MIDDLE, COLOR_1, COLOR_2, COLOR_VAR, LUSTER, AMBIENT, FILTER, WRINKLINESS, ARRAY_NUM)

  
  
  
  
  //wide grass
  #local ARRAY_NUM = 4;
  
  #local GRASS_CNT = 4;
  #local VARIETY = 0.20;
  
  #local HEIGHT = 4.75;
  #local THICKNESS = 0.040;
  
  #local SEED = 6764;
  
  #local RAD_MIN = 0.20;
  #local RAD_MAX = 0.40;
 
  #local X_BEND = pi*0.40;
  #local TWIST = pi*2;
   
  #local COLOR_1 = rgb<0.20,0.95,0.25>; //green
  #local COLOR_2 = rgb<0.65,1.00,0.35>; //green
  
  #local FILTER = 0.30;//0.35;
  #if(!use_filter) #local FILTER = 0; #end
                        
  Populate_Grass_Array(GRASS_CNT, VARIETY, HEIGHT, THICKNESS, SMOOTHNESS, SEED, RAD_MIN, RAD_MAX, X_OFFSET, Z_OFFSET, X_BEND, TWIST, STEM,CENTER,MIDDLE, COLOR_1, COLOR_2, COLOR_VAR, LUSTER, AMBIENT, FILTER, WRINKLINESS, ARRAY_NUM)

  
  
  

  
  //Make the clusters
  #local THIN_RADIUS = 3;
  #local WIDE_RADIUS = 3;
  #local THIN_GRASS_CNT = 140;
  #local WIDE_GRASS_CNT = 70;
  #local MAX_BEND = 30;
  #local SEED = 1233;                       
  
  Make_Grass_Cluster(THIN_RADIUS, WIDE_RADIUS, THIN_GRASS_CNT, WIDE_GRASS_CNT, MAX_BEND, SEED)
  
 
 
 
 
  
  //populate HF with Cluster Grass 
  #local grass_cluster_func = 
  function
  {
    pattern
    {
      granite 
      turbulence 0.10  
    
      scale 100
      translate y*7875
      
    }
  } 
  
    
  #local ARRAY = GRASS_CLUSTER_ARRAY;
  #local ARRAY_MAX = GRASS_CLUSTER_MAX;
  
  #local X_MAX = 4;
  #local Z_MAX = HF_Z_END;
  #local X_START = -X_MAX;
  #local Z_START = -6;
  
  #local STEP = 2.00;//2.00
  
  #local PROB = 0.70;
  
  #local WIND_BEND = 0;
  #local Y_WIND_ROT = 360;
  
  #local FUNC_TYPE = 2;
    
  Populate_Height_Field(grass_cluster_func, ARRAY, ARRAY_MAX, X_MAX, Z_MAX, X_START, Z_START, STEP, PROB, WIND_BEND, Y_WIND_ROT, FUNC_TYPE)

 
  #debug concat("\n************ TOTAL CLUSTERS OF GRASS = ", str(CNT,0,0), " ************\n")
  
#end

     
                          
 
 
 
 
 
 
 
 
 
 
 
  //                                               \\
 //----------------- SEEDED GRASS ------------------\\
//                                                   \\

//The seeded grass stalks are generated using the Make_Seeded_Grass.inc file.  The values listed below are described in greater detail in that file.


#if(render_sgrass)

  #debug "\n\n************ PARSING SEEDED GRASS ************\n\n"
  #local seeded_cnt = 0;
  #local SEED = 3452;
  #local R1 = seed(SEED);
  
  

  
  //Seeds
  #local SEED_CNT = 10;
  #local SEED_HEIGHT = 0.200;
  #local RADIUS = 0.015;
  #local VARIENCE = 0.20;
  #local SMOOTHNESS = 0.99;
  #local R_SEED = 23423;
  #local COLOR = rgb<1.00,0.85,0.50>;
  #local COLOR_VAR = 0.2;
  
  #local FILTER = 0.00; //0.10
    #if(!use_filter) #local FILTER = 0; #end
  
  #local LUSTER = 0.10;
  
  #local AMBIENT = 0.10;
    #if(use_radiosity) #local AMBIENT = 0.00; #end
  
  Populate_Seed_Array(SEED_CNT, SEED_HEIGHT, RADIUS, VARIENCE, SMOOTHNESS, R_SEED, COLOR, COLOR_VAR, FILTER, LUSTER, AMBIENT)
 
 
 
 
  
  //Stalks
  #local GRASS_CNT = 10;
  #local VARIETY = 0.2;
  
  #local HEIGHT = 6.00;////5.00;
  #local THICKNESS = 0.04;
                                                                                
  #local SMOOTHNESS = 0.95;
  #local R_SEED = 89822;
  
  #local X_BEND = pi/4;
  
  #local COLOR = rgb<1.00,0.85,0.50>;                                                                            
  #local COLOR_VAR = 0.05;
  
  #local FILTER = 0.00; //0.10
    #if(!use_filter) #local FILTER = 0; #end
  #local LUSTER = 0.10;
  #local AMBIENT = 0.10;
    #if(use_radiosity) #local AMBIENT = 0.00; #end
  
  #local SEEDINESS = 0.30;
  #local SEED_HEIGHT = SEED_HEIGHT; //see above...
  
  Populate_Seeded_Grass_Array(GRASS_CNT, VARIETY, HEIGHT, THICKNESS, SMOOTHNESS, R_SEED, X_BEND, COLOR, COLOR_VAR, FILTER, LUSTER, AMBIENT, SEEDINESS, SEED_HEIGHT)
  
 
  
  //Add seeded_grass to the Landscape
  
  #local seed_func = 
  function
  {
    pattern
    {
      agate 
      turbulence 0.3  
    
      scale 100
      translate y*7875
      translate <4,0,8>
    }
  }  
      
  #local ARRAY = SEEDED_GRASS_ARRAY;
  #local ARRAY_MAX = SEEDED_GRASS_MAX;
  
  #local X_MAX = 4;
  #local Z_MAX = HF_Z_END;
  #local X_START = -X_MAX;
  #local Z_START = -6;
  
  #local STEP = 1.25;//1.00
  
  #local PROB = 0.40; //0.45 
  
  #local WIND_BEND = 25;
  #local Y_WIND_ROT = 10;
  
  #local FUNC_TYPE = 0;
    
  Populate_Height_Field(seed_func, ARRAY, ARRAY_MAX, X_MAX, Z_MAX, X_START, Z_START, STEP, PROB, WIND_BEND, Y_WIND_ROT, FUNC_TYPE)

 
  #debug concat("\n************ SEEDED_CNT = ", str(CNT,0,0), " ************\n\n")
   
#end
                   
                   
 
 
 
 
 
 
 



  //                                               \\
 //----------- BLACK EYED SUSAN FLOWERS ------------\\
//                                                   \\

//The Black-eyed susan flowers are generated using the Make_Flower_Susan.inc file.  The values listed below are described in greater detail in that file.


#if(render_susans)

  #debug "\n\n************ PARSING SUSANS ************\n\n"
  #local SEED = 6674;
  #local R1 = seed(SEED);
  
      
  #local RADIUS = 0.12;//0.20
  #local SMOOTHNESS = 0.80;
  #local COLOR = rgb<0.2,0.1,0.0>;
  #local FILTER = 0.30;
    #if(!use_filter) #local FILTER = 0; #end
  #local LUSTER = 0.1;
  #local AMBIENT = 0.1;
    #if(use_radiosity) #local AMBIENT = 0; #end
  
  Make_Susan_Head(RADIUS, SMOOTHNESS, COLOR, FILTER, LUSTER, AMBIENT) 
   
   
   
  #local PETAL_CNT = 15;
  #local HEIGHT = 0.75;//0.85
  #local THICKNESS = 0.40;
  #local VARIENCE = 0.15;
  #local BEND = pi*0.35;
  #local RUFFLE = 0.25;//0.30
  #local SMOOTHNESS = 0.80;//0.90
  #local COLOR_1 = rgb<1.00,0.85,0.25>;//rgb<1.0,0.9,0.20>;
  #local COLOR_2 = rgb<0.95,0.72,0.12>;//rgb<0.9,0.7,0.0>;
  #local C_VAR = 0.9;
  #local FILTER = 0.30;//0.20
    #if(!use_filter) #local FILTER = 0; #end
  #local LUSTER = 0.4;
  #local AMBIENT = 0.15;//0.20
    #if(use_radiosity) #local AMBIENT = 0.00; #end
    
  #local SEED = 76743;
  
  Populate_Susan_Petal_Array(PETAL_CNT, HEIGHT, THICKNESS, VARIENCE, BEND, RUFFLE, SMOOTHNESS, COLOR_1, COLOR_2, C_VAR, FILTER, LUSTER, AMBIENT, SEED)
  
  
  
   
  #local FLOWER_CNT = 15; //12
  #local HEIGHT =4.85;//6
  #local RADIUS = 0.06;
  #local HEAD_RADIUS = 0.15;//0.20
  #local BEND = pi/6;
  #local SMOOTHNESS = 0.90;
  #local PETALS = 9;
  #local SEED = 67633;
  
  Populate_Susan_Array(FLOWER_CNT, HEIGHT, RADIUS, HEAD_RADIUS, BEND, SMOOTHNESS, PETALS, SEED)


  
  //Add Yellow susans to Landscape
  
  #local flower_func = 
  function
  {
    pattern
    {
      agate 
      turbulence 0.1  
    
      scale 75
      translate y*9050//9098
      translate <12,0,-10>  //<10,0,-10>
    }
  }
    
    
      
  #local ARRAY = SUSAN_ARRAY;
  #local ARRAY_MAX = SUSAN_MAX;
  
  #local X_MAX = 4;
  #local Z_MAX = HF_Z_END;
  #local X_START = -X_MAX;
  #local Z_START = 4;
  
  #local STEP = 1.50; //1.25
  
  #local PROB = 0.19; //0.215
  
  #local WIND_BEND = 20; //30
  #local Y_WIND_ROT = 30;
  
  #local FUNC_TYPE = 0;
    
  Populate_Height_Field(flower_func, ARRAY, ARRAY_MAX, X_MAX, Z_MAX, X_START, Z_START, STEP, PROB, WIND_BEND, Y_WIND_ROT, FUNC_TYPE)

 
  #debug concat("\n************ SUSAN_CNT = ", str(CNT,0,0), " ************\n\n") 
    
    
#end     
 
 

    
    
    
    


  //                                               \\
 //------------------ FLOWER_STALK -----------------\\
//                                                   \\

//The flower stalks are generated using the Make_Flower_Stalk.inc file.  The values listed below are described in greater detail in that file.


#if(render_flower_stalk)

  #debug "\n\n************ PARSING FLOWER_STALK ************\n\n"
  
  #local SEED = 45433;
  #local R1 = seed(SEED);
    
  
  
  //PETALS       
  #local PETAL_CNT = 15;
  #local VARIENCE = 0.2;
  #local HEIGHT = 0.38; //0.41;
  #local THICKNESS = 0.38;
  #local BEND = pi/8;
  #local RUFFLE = 1.5;
  #local SMOOTHNESS = 0.85;
  #local COLOR_1 = rgb<1,0.3,0.9>;
  #local COLOR_2 = rgb<0.90,0.0,0.6>;
  #local C_VAR = 0.9;
  #local FILTER = 0.30; //0.05
    #if(!use_filter) #local FILTER = 0.00; #end   
  #local LUSTER = 0.6;
  #local AMBIENT = 0.15;
    #if(use_radiosity) #local AMBIENT = 0.00; #end
  #local SEED = 5547;   
  
  Populate_FS_Petal_Array(PETAL_CNT, HEIGHT, THICKNESS, VARIENCE, BEND, RUFFLE, SMOOTHNESS, COLOR_1, COLOR_2, C_VAR, FILTER, LUSTER, AMBIENT, SEED)
  
  
  
  
  //FLOWERS
  #local FLOWER_CNT = 15;
  #local SEED = 34356;
  Populate_FS_Flower_Array(FLOWER_CNT, SEED)
  
  
  
  
  //BUDS
  #local BUD_CNT = 4;
  #local HEIGHT = 0.27;
  #local RADIUS = HEIGHT*0.12;
  #local SMOOTHNESS = 0.94;
  #local R_SEED = 2342;
  #local COLOR = rgb<0.6,0.3,0.5>;
  #local COLOR_VAR = 0.1;
  #local FILTER = 0.05;
    #if(!use_filter) #local FILTER = 0.00; #end   
  #local LUSTER = 0.1;
  #local AMBIENT = 0.05;
    #if(use_radiosity) #local AMBIENT = 0.0; #end
  Populate_FS_Bud_Array(BUD_CNT, HEIGHT, RADIUS, SMOOTHNESS, R_SEED, COLOR, COLOR_VAR, FILTER, LUSTER, AMBIENT)
  
  
  
  
  //Plant Array
  #local FLOWER_STALK_CNT = 15;
  #local VARIENCE = 0.3;
  #local HEIGHT = 7.4;
  #local RADIUS = 0.04;
  #local SMOOTHNESS = 0.96;
  #local BEND = pi/8;
  #local STEM_COLOR = rgb<0.5,0.6,0.3>;;
  #local FILTER = 0.05;
    #if(!use_filter) #local FILTER = 0.00; #end
  #local LUSTER = 0.1;
  #local AMBIENT = 0.05;
    #if(use_radiosity) #local AMBIENT = 0.0; #end
  #local SEED = 23437;
  #local FLOWER_DIST = 0.20; //0.15
  #local FLOWERINESS = 0.55;
  Populate_Flower_stalk_Array(FLOWER_STALK_CNT, VARIENCE, HEIGHT, RADIUS, SMOOTHNESS, BEND, STEM_COLOR, FILTER, LUSTER, AMBIENT, SEED, FLOWER_DIST, FLOWERINESS)

 
  
  //Add Purple Flower_stalk to Landscape
  
  #local flower_stalk_func = 
  function
  {
    pattern
    {
      agate 
      turbulence 0.2  
    
      scale 75
      translate y*34543
      translate <20,0,50> //<30,0,53>
    }
  }
   
   
  #local ARRAY = FLOWER_STALK_ARRAY;
  #local ARRAY_MAX = FLOWER_STALK_MAX;
  
  #local X_MAX = 4;
  #local Z_MAX = HF_Z_END;
  #local X_START = -X_MAX;
  #local Z_START = -4;
  
  #local STEP = 2.10; //1.90
  
  #local PROB = 0.310; //0.325 
  
  #local WIND_BEND = 30;
  #local Y_WIND_ROT = 40;
  
  #local FUNC_TYPE = 0;
    
  Populate_Height_Field(flower_stalk_func, ARRAY, ARRAY_MAX, X_MAX, Z_MAX, X_START, Z_START, STEP, PROB, WIND_BEND, Y_WIND_ROT, FUNC_TYPE)

 
  #debug concat("\n************ FLOWER_STALK_CNT = ", str(CNT,0,0), " ************\n\n")
  
#end


    
    
















  //                                               \\
 //---------------------- TREES --------------------\\
//                                                   \\

//The Trees are generated using the Make_Tree.inc file.  The values listed below are described in greater detail in that file.


#if(render_trees)

  #debug "\n\n************ PARSING BACKGROUND TREES ************\n\n"
  
  
  #local tree_cnt = 0;
  #local SEED = 23423;
  #local R1 = seed(SEED);
  
  
  #declare BRANCH_CNT = 0;
  #declare LEAF_CNT = 0; 
  #declare LEAF_TRIANGLES = 0;
  #declare TREE_TRIANGLES = 0;
  
  
  
  
  

  //Leaves
  #local LEAF_CNT = 10;
  #local VARIETY = 0.10;
  #local HEIGHT = 0.90;
  #local THICKNESS = 0.30;
  #local BEND = pi*0.125;
  #local RUFFLE = 0.50;
  #local SMOOTHNESS = 0.75;//0.85;
  #local SEED = 6675;
  
  Populate_Tree_Leaf_Array(LEAF_CNT, VARIETY, HEIGHT, THICKNESS, BEND, RUFFLE, SMOOTHNESS, SEED)
  
  
  
  //Trees
  #local HEIGHT = 32.00;
  #local THICKNESS = 0.025;
  #local SMOOTHNESS = 0.55;//0.55;
  #local NORM = vnormalize(<0,1,0>);
  
  #local CURVINESS = 0.90; //0.50
  #local LEAFINESS =0.30; //0.40
  #local BRANCHINESS = 0.05;
  
  #local MIN_BRANCH_HEIGHT = 1;
  
  #local COLOR_1 = rgb<0.05,0.65,0.0>;  //rgb<0.1,0.90,0>;
  #local COLOR_2 = rgb<0.10,0.55,0.0>;  //rgb<0.2,0.70,0>;
  #local C_VAR = 1.0;
  
  #local LUSTER = 0.50;
  #local AMBIENT = 0.10;
    #if(use_radiosity) #local AMBIENT = 0.00; #end
  #local FILTER = 0.30;
    #if(!use_filter) #local FILTER = 0.00; #end
  
  #local LEVEL_MAX = 4;
  
  #local SEED = 3432;
   
  
  #declare TREE = Make_Tree(LEVEL_MAX, HEIGHT, THICKNESS, SMOOTHNESS, NORM, CURVINESS, LEAFINESS, BRANCHINESS, MIN_BRANCH_HEIGHT, COLOR_1, COLOR_2, C_VAR, FILTER, LUSTER, AMBIENT, SEED)
  
 
  
  //Mid-level right tree
  #local tx = 21.5;//24;
  #local tz = 26;//30;
  object{TREE rotate y*32 translate<tx, P_FUNC(tx,tz,0)*HF_HEIGHT ,tz>}
  
  //very close left tree
  #local tx = -21;
  #local tz = 10;
  //object{TREE translate<tx, 2 ,tz>}
   
   
  //background tree 1 (right)
  #local tx = 32;//-10;
  #local tz = 75;
  object{TREE  scale 1.00 rotate y*177 rotate z*-1 translate<tx, P_FUNC(tx,tz,0)*HF_HEIGHT - 4 ,tz>}
  
  //background tree 2 (left)
  #local tx = 29;//-24;
  #local tz = 80;
  object{TREE  scale 0.94 rotate y*215 rotate z*-8 translate<tx, P_FUNC(tx,tz,0)*HF_HEIGHT - 4 ,tz>}
  
  
#end
    
    
    
    
    
    
    
           
           
           

  //                                    \\
 //----------------- SUMAC --------------\\
//                                        \\

//The sumac plants are generated using the Make_Sumac.inc file.  The values listed below are described in greater detail in that file.


#if(render_sumac)

  #debug "\n\n************ PARSING SUMAC ************\n\n"
  
  #local SEED = 343211;
  #local R1 = seed(SEED);
    
                            
    
  //SUMAC BERRY
  #local RADIUS = 0.025;
  #local SMOOTHNESS = 0.96;
  #local AMBIENT = 0.00;
    #if(use_radiosity) #local AMBIENT = 0.00; #end
  #local LUSTER = 0.00;
  
  Make_Sumac_Berry(RADIUS, SMOOTHNESS, AMBIENT, LUSTER)
  
  
  
  //SUMNAC LEAVES
  #local LEAF_CNT = 10;
  #local VARIETY = 0.1;
  #local LEAF_HEIGHT = 1.00;
  #local LEAF_THICKNESS = 0.25;//0.20;
  #local BEND = pi/4;
  #local RUFFLE = 0.50;
  #local SMOOTHNESS = 0.85;
  #local INNER_COLOR = rgb<0.5,0.85,0.3>;
  #local OUTER_COLOR = rgb<0.7,0.6,0.3>;
  #local C_VAR = 0.2;
  
  #local FILTER = 0.25;//0.35;
    #if(!use_filter) #local FILTER = 0.00; #end
  
  #local LUSTER = 0.30;
  
  #local AMBIENT = 0.10;
    #if(use_radiosity) #local AMBIENT = 0.00; #end
  
  #local SEED = 78754;
  
  Populate_Sumac_Leaf_Array(LEAF_CNT, VARIETY, LEAF_HEIGHT, LEAF_THICKNESS, BEND, RUFFLE, SMOOTHNESS, INNER_COLOR, OUTER_COLOR, C_VAR, FILTER, LUSTER, AMBIENT, SEED)
  
  
  
  
  //SUMAC PLANT
  #local SUMAC_CNT = 10;
  #local VARIETY = 0.3;
  #local HEIGHT = 6.00;
  #local THICKNESS = 0.015;
  #local MAX_BEND = -pi/6;
  #local SMOOTHNESS = 0.85;
  #local CURVINESS = 0.60;
  #local BERRY_RADIUS = RADIUS; //see Make_Sumac_Berry above
  #local LEAF_WIDTH = LEAF_HEIGHT*LEAF_THICKNESS;
  #local SEED = 457733;
  
  
  Populate_Sumac_Array(SUMAC_CNT, VARIETY, HEIGHT, THICKNESS, MAX_BEND, SMOOTHNESS, CURVINESS, BERRY_RADIUS, LEAF_WIDTH, SEED)    

  
  
      

  
  
  
 //Add Sumac to Landscape
  
  #local Sumac_Func = 
  function
  {
    pattern
    {
      agate 
      turbulence 0.2  
    
      scale 75
      translate y*34234//667//6764
      translate <-2,0,-20> //translate <-7,0,-20>
    }
  }
  
  #local ARRAY = SUMAC_ARRAY;
  #local ARRAY_MAX = SUMAC_MAX;
  
  #local X_MAX = 8;
  #local Z_MAX = HF_Z_END*0.85;
  #local X_START = -X_MAX;
  #local Z_START = 0;
  
  #local STEP = 3.40;//1.80;
  
  #local PROB = 0.22;//0.30; 
  
  #local WIND_BEND = 22.5;
  #local Y_WIND_ROT = 360;
  
  #local FUNC_TYPE = 3;
    
  Populate_Height_Field(Sumac_Func, ARRAY, ARRAY_MAX, X_MAX, Z_MAX, X_START, Z_START, STEP, PROB, WIND_BEND, Y_WIND_ROT, FUNC_TYPE)

 
  #debug concat("\n************ SUMAC_CNT = ", str(CNT,0,0), " ************\n\n")
  
#end     
       
       
     
     
     
     
   
   
   



   

  //                                               \\
 //----- Background Image and Camera Location ------\\
//                                                   \\  


//This macro was copied from the PovRay Cyclopedia at: http://www.spiritone.com/~english/cyclopedia/background.html
//downloaded 3-24-06
//John Van Sickle was mentioned in the credits, though it wasn't made exactly clear who created this code.

#declare CamLoc =  <0.00,10.00,-10.00>; //where the camera's location is
#declare CamLook = <0.00,11.00, 10.00>; // Camera's Look_at

#declare cam_z = 1; //the amount of camera zoom you want
#declare back_dist = 4000; // how far away the background is
#declare cam_a = 4/3; // camera aspect ratio
#declare cam_s = <0,1,0>; // camera sky vectoy
#declare cam_d = vnormalize(CamLook-CamLoc); // camera direction vector
#declare cam_r = vnormalize(vcross(cam_s,cam_d)); // camera right vector
#declare cam_u = vnormalize(vcross(cam_d,cam_r)); // camera up vector
#declare cam_dir = cam_d * cam_z; // direction vector scaled
#declare cam_right = cam_r * cam_a; // right vector scaled

#declare fz = vlength(cam_dir);
#declare fx = vlength(cam_right)/2;
#declare fy = vlength(cam_u)/2; 

#macro OrientZ(p1,p2,cs)
  #local nz = vnormalize(p2-p1);
  #local nx = vnormalize(vcross(cs,nz)); 
  #local ny = vcross(nz,nx);
  matrix <nx.x,nx.y,nx.z, ny.x,ny.y,ny.z, nz.x,nz.y,nz.z, p1.x,p1.y,p1.z>          
#end


camera 
{
  location CamLoc
  up cam_u
  right cam_r * cam_a
  direction (cam_d * cam_z) 
}


light_group //insures that the Background does not interact with any light sources
{
  
  //blue filter 
  plane
  {
    <0,0,1>, 0
    
    pigment{rgbt<0.5,0.7,1,0.95>}
    finish{ambient 1.0}
  
    translate<0,0,200>
  }
  
  
  
  box 
  { 
    <0,0,0> 
    <1,1,0.1>
    pigment 
    { 
      image_map 
      {   
        jpeg "Mesoground.jpg"        
      } 
    }
    
    finish 
    { 
      ambient 1.0
    }
    
    translate <-0.5,-0.5,0>
    scale 2*<fx,fy,0.5>
    translate fz*z
    scale back_dist
    OrientZ(CamLoc,CamLook,cam_s)
  }
  
  global_lights false
}

