

/*_______________________________________________________________________________________________________*\

FILE NAME: Make_Flower_Susan.inc
CREATED BY: Isaac Kulka, Copyright 12-28-2006
NOTES: Created for the Internet Raytracing Competition's Nov-Dec 2006 Contest: 'Complexity' (www.irtc.org)

DESCRIPTION: The macros included in this file can be used to create flowers similar to Black-Eyed-Susans.

\*_______________________________________________________________________________________________________*/






  //                                                            \\
 //----------------------POPULATE SUSAN ARRAY--------------------\\
//                                                                \\

// Use this macro to create an array of unique susan flowers.  NOTE: you must run the Populate_Susan_Petal_Array() and Make_Susan_Head() macros before running this macro.


#macro Populate_Susan_Array(SUSAN_CNT, HEIGHT, RADIUS, HEAD_RADIUS, BEND, SMOOTHNESS, PETALS, SEED)

/**********   VARIABLES   **************\
SUSAN_CNT -   Integer - Number of unique flowers to be stored in the array
HEIGHT -      Float -   maximum height of the stem of any one susan plant
RADIUS -      Float -   radius of the stem of each plant  
HEAD_RADIUS - Float -   radius of the head of the plant
BEND -        Radian -  maximum bend of the stem around the z-axis
SMOOTHNESS -  0-1 -     determines the number of triangles in the stem mesh
PETALS -      Integer - maximum number of petals to be found on any one flower
SEED -        Integer - random seed to be used with this macro
\***************************************/


#local R1 = seed(SEED);
#local theta_step = 360/PETALS;

//these are global variables, intended to be called from the main scene in which these flowers are used
#declare SUSAN_ARRAY = array[FLOWER_CNT];
#declare SUSAN_MAX = FLOWER_CNT;

//generate the flowers
#local j = 0;
#while(j < SUSAN_CNT)

  #declare SUSAN_ARRAY[j] =
  union
  {
    #local Height = HEIGHT*(rand(R1)*0.30 + 0.70); 
    #local Stem_X_Bend = BEND*rand(R1);
     
    #local Color = rgb<0.45,0.85,0.35>; 
    #local Filter = 0.0;//0.3;
    
    #local Luster = 0.4; 
    #local Ambient = 0.0;
    
    //each flower gets a uniquely generated stem.  The head and the petals however have already been generated
    Make_Susan_Stem(Height, RADIUS, SMOOTHNESS, Stem_X_Bend, Color, Filter, Luster, Ambient)
    
    object
    {
      SUSAN_HEAD 
      translate<0,-HEAD_RADIUS*0.2,0>
      rotate y*rand(R1)*360 
      rotate z*STEM_BEND 
      translate STEM_END
    }
    
    #local bend = rand(R1)*10;// - 3;
    #local Petals = PETALS*ceil(rand(R1)*0.40 + 0.60);
    
    //add petals around the flower's head
    #local i = 0;
    #while(i < Petals)
      
      #local theta = i*theta_step;
      
      
      object
      {
        SUSAN_PETAL_ARRAY[floor(rand(R1)*SUSAN_PETAL_MAX)] 
        
        translate<0,HEAD_RADIUS*-0.15,-HEAD_RADIUS*0.10>
        rotate x*(bend)
        rotate y*(theta + rand(R1)*theta_step/2) 
        
        //align the petal with the stem
        rotate z*STEM_BEND
        translate STEM_END
      }
    
      #local i = i + 1; 
    #end
  }

  #local j = j + 1;
#end



#end //Populate_Susan_Array()




  
  
  


  //                                                            \\
 //----------------------MAKE Susan Head-------------------------\\
//                                                                \\

// Use this macro to generate a head for you susan plants.  Each plant will use the same head.  This macro must be run before the Populate_Susan_Array() macro.


#macro Make_Susan_Head(RADIUS, SMOOTHNESS, COLOR, FILTER, LUSTER, AMBIENT) 


/**********   VARIABLES   **************\
RADIUS -      Float -   radius of the flower head 
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh   
COLOR -       <r,g,b> - color of the head
FILTER -      0-1 -     filter of the head
LUSTER -      0-1 -     specular highlight of the head
AMBIENT -     0-1 -     ambient value of the head
\***************************************/


//The head is simply a squished spherical mesh 
#declare SUSAN_HEAD =   
  mesh
  {
    #local step = (1-max(SMOOTHNESS,0.40))/16; //distance between points in the mesh
    #local rad = 0; //radius of the current cross section of the sphere
    #local theta = 0; //rotation around the current cross section of the sphere
    #local Height = RADIUS*0.65; //0.50 //this value squishes or elongates the shape of the sphere
    #local h = -Height; //current cross section of the sphere
    
    #local points = ceil(Height/step * 8); //number of points pe circular cross section
    
    #local upper = array[points];
    #local lower = array[points];
    
    //initialize lower array at <0,0,0>
    #local i = 0;
    #while(i < points)
      #local lower[i] = <0,h,0>;
      #local i = i + 1;
    #end
    
    #local h = h + step;
    
    
    
    //create the mesh
    #while(h <= Height)
      
      #local h_ratio = h/Height;
      
      #if(h + step <= Height)
        #local rad = sqrt(pow(RADIUS,2)*(1-(pow(h,2)/pow(Height,2))));
      #else
        #local rad = 0;
      #end
      
      //define upper points  
      #local i = 0;
      #while(i < points)
        
        #local theta = (i/points)*(2*pi);
        
        #local pp = <rad*cos(theta), h, rad*sin(theta)>;
        #local h_off = HEAD_PATTERN(pp.x,pp.y,pp.z)*RADIUS*0.05;
        
        #local upper[i] = <rad*cos(theta), h + h_off, rad*sin(theta)>;
      
        #local i = i + 1;
      #end
      
      //make triangles
      #local i = 0;
      #while(i < points)
        #local j = i+1;
        #if(j = points) #local j = 0; #end 
  
        
        triangle{upper[i],lower[i],lower[j]}
        triangle{upper[i],upper[j],lower[j]}
        
        #local i = i + 1;
      #end
      
      //set lower to upper
      #local i = 0;
      #while( i < points)
        
        #local lower[i] = upper[i];
        
        #local i = i + 1;
      #end
      
      #local h = h + step;
    #end
    
    pigment
      {COLOR filter FILTER}
    finish
      {ambient AMBIENT diffuse 0.8 specular LUSTER roughness 0.2}
    normal{granite bump_size 0.2 scale 0.2}                                                 
  }

#end //Make_Susan_Head()

      
      
      
      


                              

  //                                                            \\
 //----------------------MAKE Susan Petal------------------------\\
//                                                                \\

// This macro is called by the Populate_Susan_Petal_Array() macro to create petals for the SUSAN_PETAL_ARRAY.


#macro Make_Susan_Petal(HEIGHT, WIDTH, BEND, RUFFLE, COLOR, SEED)

/**********   VARIABLES   **************\
HEIGHT -      Float -   length of the petal
WIDTH -       Float -   width of petal at it widest point
BEND -        Radian -  amount the petal curls around the x-axis
RUFFLE -      Float -   amount the petal curves out in the z-direction along its width from the crease
COLOR -       <r,g,b> - color of the petal
SEED -        Integer - random seed
\***************************************/


#local R1 = seed(SEED);

  mesh
  {
    #local RUFFLE = max(RUFFLE, 0.2); //the ruffle cannot go below 0.2 (no flat petals)
    
    #local h_step = HEIGHT*(1-SMOOTHNESS)*0.1;
    #local h = 0;
    
    #local points = max(3, ceil(WIDTH/h_step));
    
    #local lower = array[points];
    #local upper = array[points];
    
    #local rpx = rand(R1)*1000;
    #local rpy = rand(R1)*1000;
    #local rpz = rand(R1)*1000;
    
    #local i = 0;
    #while(i < points)
      
      #local lower[i] = <0,0,0>;
      
      #local i = i + 1;
    #end
    
    
    #while(h < HEIGHT)
      
      #local h_ratio = h/HEIGHT;
      
      #local rad = sqrt(pow(WIDTH,2)*(1-(pow(h_ratio-0.5,2)/pow(0.5,2))));
      #if(h+h_step >= HEIGHT) #local rad = 0; #end
            
      #local i = 0;
      #while(i < points)
        #local i_ratio = i/points;
        
        #local z_bend = BEND * h_ratio;
        
        #local pp = <rad*i_ratio - rad/2,      cos(z_bend)*(h + HEIGHT*0.15) + cos(z_bend)*((1-h_ratio) * rad*RUFFLE*(1-h_ratio)*abs(sin(i_ratio*pi*2))),      sin(z_bend)*(h + HEIGHT*0.15) + sin(z_bend)*((1-h_ratio) * rad*RUFFLE*(1-h_ratio)*abs(sin(i_ratio*pi*2)))>;
        
        #local upper[i] = <pp.x,      pp.y + PETAL_PATTERN(pp.x,pp.y,pp.z)*0.05*HEIGHT,      pp.z + PETAL_PATTERN(pp.x+rpx,pp.y+rpy,pp.z+rpz)*0.05*HEIGHT>;
        
        #local i = i + 1;
      #end
      
      
      #local i = 0;
      #while(i < points-1)
             
        triangle{upper[i], lower[i], lower[i+1]}
        triangle{upper[i+1], upper[i], lower[i+1]}
        
        #local i = i + 1;
      #end
      
      
      #local i = 0;
      #while(i < points)
        
        #local lower[i] = upper[i];
        
        #local i = i + 1;
      #end
      
      #local h = h + h_step;
    #end
    
   
  
   
   pigment
    {COLOR filter FILTER} 
   finish
    {ambient AMBIENT diffuse 0.7 specular LUSTER roughness 0.6}
   
   double_illuminate
    
   rotate x*90
  }

#end //Make_Susan_Petal()









  //                                                            \\
 //------------------POPULATE_SUSAN_PETAL_ARRAY------------------\\
//                                                                \\

// Use this macro to create an array of unique petals for your susan plants.  This macro must be run before the Populate_Susan_Array() macro.
 
 
#macro Populate_Susan_Petal_Array(PETAL_CNT, HEIGHT, THICKNESS, VARIETY, BEND, RUFFLE, SMOOTHNESS, COLOR_1, COLOR_2, C_VAR, FILTER, LUSTER, AMBIENT, SEED)

/**********   VARIABLES   **************\
PETAL_CNT -   Integer - number of unique petals to generate
HEIGHT -      Float -   maximum length of any one petal
THICKNESS -   Percent - determines width of petal, a percentage of the height
VARIETY -     0-1 -     determines how much certain values can deviate below the maximum specified values 
BEND -        Radian -  maximum amount a petal curls around the x-axis
RUFFLE -      Float -   maximum amount a petal curves out in the z-direction along its width from the crease
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh   
COLOR_1 -     rgb -     primary color of the petals
COLOR_2 -     rgb -     secondary color of the petals
C_VAR -       0-1 -     amount by which the primary color may deviate to the secondary color in any given petal
FILTER -      0-1 -     filtered light value
LUSTER -      0-1 -     specular highlight
AMBIENT -     0-1 -     ambient value
SEED -        Integer - random seed
\***************************************/


#local R1 = seed(SEED);

//these two values will be used to control just how much each petal differ from one another
#local INV_VAR = 1-VARIETY; 
#local INV_VAR2 = 1-VARIETY*3;

//global values
#declare SUSAN_PETAL_ARRAY = array[PETAL_CNT];    
#declare SUSAN_PETAL_MAX = PETAL_CNT;

#declare SMOOTHNESS = SMOOTHNESS;
#declare FILTER = FILTER;
#declare LUSTER = LUSTER;
#declare AMBIENT = AMBIENT;
             
             
//randomly generate values and call Make_Susan_Petal() to populate the SUSAN_PETAL_ARRAY with 
#local i = 0;
#while(i < PETAL_CNT)
  
  #local Height = HEIGHT*(rand(R1)*VARIETY + INV_VAR);
  #local Width = THICKNESS*(rand(R1)*VARIETY + INV_VAR)*Height;
  
  #local Bend = BEND*(rand(R1)*VARIETY*3 + INV_VAR2);
  #local Ruffle = RUFFLE*(rand(R1)*VARIETY*3 + INV_VAR2);
  
  #local C_Var = rand(R1)*C_VAR;
  #local Color = rgb<COLOR_1.red*(1-C_Var) + COLOR_2.red*(C_Var), COLOR_1.green*(1-C_Var) + COLOR_2.green*(C_Var), COLOR_1.blue*(1-C_Var) + COLOR_2.blue*(C_Var)>;
  
  #local Seed = rand(R1)*1000;   
  
  #declare SUSAN_PETAL_ARRAY[i] = Make_Susan_Petal(Height, Width, Bend, Ruffle, Color, Seed);
  
  
  #local i = i + 1;
#end

#end //Make_Susan_Petal_Array()








  //                                                            \\
 //----------------------Make Susan Stem-------------------------\\
//                                                                \\

// This macro is called by the Populate_Susan_Array() macro to create stems for the plants in the SUSAN_ARRAY. 


#macro Make_Susan_Stem(HEIGHT, RADIUS, SMOOTHNESS, BEND, COLOR, FILTER, LUSTER, AMBIENT)                                                            


/**********   VARIABLES   **************\
HEIGHT -      Float -   height of the stem
RADIUS -      Float -   radius of the stem
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh
BEND -        Radian -  bend of the stem around the z-axis
COLOR -       <r,g,b> - color of the stem
FILTER -      0-1 -     filter value
LUSTER -      0-1 -     specular highlight
AMBIENT -     0-1 -     ambient value
\***************************************/      

#local circumference = 2*pi*RADIUS; //the circular length around the stem
#local step = (1-SMOOTHNESS)/2; //distance between points in the mesh
#local points = ceil(circumference/step); //number of points per cross section



#local lower = array[points];
#local upper = array[points];

#declare x_bend_rad = HEIGHT/BEND;

mesh
{
  //initialize lower array
  #local i = 0;
  #while(i < points)
    #local theta = (i/points)*(2*pi);
    #local lower[i] = <RADIUS*cos(theta), 0, RADIUS*sin(theta)>;
    #local i = i + 1;
  #end
  
  //begin builing the stem
  #local h = step;
  #while(h <= HEIGHT)
    
    #local h_ratio = h/HEIGHT;
    
    #local rad = RADIUS*((1-(h/HEIGHT))*0.75 + 0.25);
    //generate new points for upper array
    #local i = 0;
    #while(i < points)
      #local theta = (i/points)*(2*pi);
      #local bend = (h/HEIGHT)*BEND;
      #local upper[i] = <rad*cos(theta)*cos(bend) + x_bend_rad*cos(bend) - x_bend_rad,    rad*cos(theta)*sin(bend) + x_bend_rad*sin(bend),    rad*sin(theta)>;
      #local i = i + 1;
    #end
    
    //create the triangles for this pass
    #local i = 0;
    #while(i < points)
      #local j = i+1;
      #if(j = points) #local j = 0; #end 
      
     
      
      triangle{upper[i],lower[i],lower[j]}
      triangle{upper[i],upper[j],lower[j]}
            
      #local i = i + 1;
    #end
    
    //set lower to upper
    #if(h + step <= HEIGHT)
    #local i = 0;
      #while( i < points)
        
        #local lower[i] = upper[i];
        
        #local i = i + 1;
      #end
    #end 
    
    
    #local h = h + step;
  #end
  
  pigment
    {COLOR filter FILTER}
  finish
    {ambient AMBIENT diffuse 0.85 specular LUSTER}

  
  //The values are made global so that the Make_Susan_Array() macro which called this macro can know where to place the head of susan and how much it should be rotated  
  #declare STEM_END = upper[0];
  
  #local dx = upper[0].x - lower[0].x;
  #local dy = upper[0].y - lower[0].y;
  
  #local hyp = sqrt(dx*dx + dy*dy);
  
  #declare STEM_BEND = degrees(asin(dy/hyp)) + 90;
  #declare STEM_BEND = 180-STEM_BEND;
  #if(dx > 0) #declare STEM_BEND = -STEM_BEND; #end
    
  }
#end //Make_Susan_Stem()   










//PETAL WRINKLE PATTERN
#declare PETAL_PATTERN = 
function
{ 
  pattern
  { 
    agate
     
    lambda 0.8
    octaves 3
    omega 5  
    turbulence 0.1

     
    scale 20
    translate y*654
  } 
}



//HEAD WRINKLE PATTERN
#declare HEAD_PATTERN = 
function
{ 
  pattern
  { 
    agate
    
    //turbulence 1
     
    scale 100//4
  } 
}  