

/*_______________________________________________________________________________________________________*\

FILE NAME: Make_Tree.inc
CREATED BY: Isaac Kulka, Copyright 12-28-2006
NOTES: Created for the Internet Raytracing Competition's Nov-Dec 2006 Contest: 'Complexity' (www.irtc.org)

DESCRIPTION: The macros included in this file can be used to create natural looking tree-like 
             structures using the Povray raytracing program.
\*_______________________________________________________________________________________________________*/
       





  //                                                       \\
 //---------------------- MAKE TREE ------------------------\\
//                                                           \\

// Use this macro to begin creating a new tree.  NOTE: if you plan to use leaves, then you must create them ahead of time using the Populate_Tree_Leaf_Array() Macro, provided further below


#macro Make_Tree(LEVEL_MAX, HEIGHT, THICKNESS, SMOOTHNESS, NORM, CURVINESS, LEAFINESS, BRANCHINESS, MIN_BRANCH_HEIGHT, COLOR_1, COLOR_2, C_VAR, FILTER, LUSTER, AMBIENT, SEED)

/**********   VARIABLES   **************\
LEVEL_MAX -    Integer - maximum level of recursion
HEIGHT -       Float -   the length of the trunk of the Tree
THICKNESS -    Float -   determine the radius of the Tree at the base of the trunk, based on the Tree's HEIGHT
SMOOTHNESS -   0-1 -     determines the number of trianges in the mesh, a higher value means more triangles
NORM -         <x,y,z> - this determines the direction in which the tree is growing.  <0,1,0> is a good default value
CURVINESS -    0-1 -     determines how much the branches curve
LEAFINESS -    0-1 -     determines how densely the leaves populate the branches of the Tree.  Set this value to zero if you don't want any leaves
BRANCHINESS -  0-1 -     determines how densely the branches grow on the tree
MIN_BRANCH_HEIGHT 0-1 -  determines at what point along the trunk branches are allowed to begin spawning (a value of 0.20 would mean the no branches will spawn off the bottom 20% of the trunk)
COLOR_1        <r,g,b> - primary color of the leaves
COLOR_2        <r,g,b> - secondary color of the leaves.  This is the color which the leaves may shift towards (think of autumn leaves shifting from green to red)
C_VAR          0-1 -     this value determines how much and how readily the leaves will shift from COLOR_1 to COLOR_2
FILTER -       0-1 -     filter value of the leaves (Higher value means the leaves will be more transparent)
LUSTER -       0-1 -     specular Highlight value of the leaves
AMBIENT -      0-1 -     ambient value of the leaves
SEED -         Integer - random seed
\***************************************/     
                 

  //derived global variables
  #declare MAX_RAD = THICKNESS * HEIGHT; //the radius of the base of the trunk
  #declare MAX_HEIGHT = HEIGHT;
  
  //local variables converted to global variables (to be referenced in the Make_Tree_Branch() macro)
  #declare SMOOTHNESS = SMOOTHNESS;
  #declare LEVEL_MAX = LEVEL_MAX;
  #declare CURVINESS = CURVINESS;
  #declare LEAFINESS = LEAFINESS;
  #declare BRANCHINESS = BRANCHINESS;
  #declare COLOR_2 = COLOR_2;
  #declare C_VAR = C_VAR;
  #declare FILTER = FILTER;

  #declare TREE_FINISH = 
  finish
  {
    ambient AMBIENT 
    diffuse 0.8 
    specular LUSTER
    roughness 0.30
  }
  
  #local START = <0,0,0>;
  #local LEVEL = 0;
  
  //begin making the tree.  Once started, the following macro will call itself recursively until the tree is complete  
  union
  {
    Make_Tree_Branch(LEVEL, START, HEIGHT, THICKNESS, NORM, MIN_BRANCH_HEIGHT, COLOR_1, SEED)
  }
  
  
#end //Make_Tree()



                  
        
        
        
                 
       
       
                  
                  
  //                                                            \\
 //---------------------- MAKE TREE BRANCH ----------------------\\
//                                                                \\

// This macro is called by the Make_Tree() Macro above to begin creating a new tree. This macro then calls itself recursively to finish creating that tree


#macro Make_Tree_Branch(LEVEL, START, HEIGHT, THICKNESS, NORM, MIN_BRANCH_HEIGHT, COLOR_1, SEED)

/**********   VARIABLES   **************\
LEVEL -        Integer - current level of recursion
START -        <x,y,z> - base point of the current branch
HEIGHT -       Float -   the length of the current branch
THICKNESS -    Float -   radius of base of the current branch
NORM -         <x,y,z> - determines the direction in which the base of the branch points as it juts out from its parent branch
MIN_BRANCH_HEIGHT 0-1 -  determines how far up along the branch we must be before new branches may begin spawning off this one
COLOR_1 -      <r,g,b> - current color of the leaves, this color may shift as further branches are spawned from this one
SEED -         Integer - random seed
\***************************************/  

  
  #if(LEVEL < LEVEL_MAX)
  
    #local R1 = seed(SEED);
    
    #local norm1 = NORM;
    
    #local bark_thickness = 0.11; //thickness of the crevaces in the bark
    
    #local bulge_thickness = 0.20; //max thickness of buldges in the trees shape
    
    
    
    //MAKE SPLINE
    //this spline will define the shape of the branch
    #local branch_spline = spline
    {
      natural_spline
      
      #local norm2 = vnormalize(<norm1.x+0.1,norm1.y,norm1.z>);
        
      #local norm_perp = vnormalize(vcross(norm1, norm2));
         
      #local i_step = 0.2;
      #local i = i_step;
      
  
      //initial points
      -i_step, <0,-i_step,0>
      
      0.00, <0,0,0>
      
      
      //middle points
      #local height_step = HEIGHT/(1/i_step);
      
      #while(i <= 1)
        
        #local height_ratio = (HEIGHT*i + MAX_HEIGHT-HEIGHT)/MAX_HEIGHT;
        
        #local theta = rand(R1)*360;
        
        #local rad = rand(R1)*HEIGHT*THICKNESS*height_ratio*CURVINESS*5;
        
        #local point = norm_perp*rad;
        
        #local final_point = vaxis_rotate(point, norm1, theta)  + norm1*HEIGHT*i;
        
        i, final_point   
        
      
        #local i = i + i_step;
      #end
      
      //final point
      1.10, norm1*(HEIGHT+1)
    
    }
    
    

    
    
    
    #if(HEIGHT > 0.2) //No branches below a HEIGHT of 0.2 allowed (this helps to avoid degenerate meshes)
      
      #declare BRANCH_CNT = BRANCH_CNT + 1;
      
      //rotate the grain of the bark pattern with the direction of NORM
      #local x_rot = 0; //rotation around x-axis in degrees
      #local z_rot = 0; //rotation around z-axis
      
      #if(NORM.x != 0) //catch zero length vectors
        #local z_rot = degrees(asin(NORM.y/sqrt(NORM.y*NORM.y + NORM.x*NORM.x))) - 90;
      #end
      #if(NORM.x < 0) #declare z_rot = -z_rot; #end
      
      
      #if(NORM.z != 0) //catch zero length vectors
        #local x_rot = degrees(asin(NORM.y/sqrt(NORM.y*NORM.y + NORM.z*NORM.z))) - 90;
      #end
      #if(NORM.z > 0) #declare x_rot = -x_rot; #end
      
      #local bark_noise = 
      function
      { 
        pattern
        { 
          agate 
          scale<0.2, 1.6, 0.2>
          rotate <x_rot, 0, z_rot> 
        } 
      }
      

 
      
      
      mesh                                                   
      { 
 
        #local Smoothness = SMOOTHNESS;
        
        #if(LEVEL <= 1) 
          #local Smoothness = 0.925; 
        #end
       
        #local h_step = min(HEIGHT/2,(1-Smoothness)/2); //distance between points in the branch mesh
        #local h = -h_step;
        #local base_rad = HEIGHT*THICKNESS; //bottom-most radius of the current branch
        #local points = max(3,ceil(base_rad*10/(1-Smoothness))); //number of points per circular cross-section
        #local rad = base_rad; //radius of the current cross-section
        
        #local lower = array[points];
        #local upper = array[points];
        
        #local l_norm = array[points]; //norms of the points of the lower array (used for smooth triangles in the mesh)
        #local u_norm = array[points]; //norms of the points of the upper array
        
        #local dspline = branch_spline(0.01) - branch_spline(0); //vector of the branch at its bottom-most cross-section
        #local norm = vnormalize(dspline);
        #local norm2 = vnormalize(<norm.x+0.1,norm.y,norm.z>);
        #local norm_perp = vnormalize(vcross(norm, norm2)); //normal vector perpendicular to branch (used to rotate points around norm in order to give your branch some thickness)
        
        
        //set initial lower points
        #local i=0;
        #while(i < points)
          #local theta = (i/points)*360; //current rotation around the current cross-section
          
          #local point1 = norm_perp*rad; //a single point sticking out a distance of rad from the normal vector
          #local point1 = vaxis_rotate(point1, norm, theta) + branch_spline(h/HEIGHT) + START;
          
          #local point = norm_perp*(rad + bark_noise(point1.x,point1.y,point1.z)*rad*bark_thickness + TREE_BULGE_NOISE(point1.x,point1.y,point1.z)*rad*bulge_thickness);
          #local lower[i] = vaxis_rotate(point, norm, theta) + branch_spline(h/HEIGHT) + START; //to dermine all the points in the cross-section, start rotating point around norm
          
          #local l_norm[i] = vnormalize(lower[i] - (branch_spline(h/HEIGHT) + START)); //determine the normal of each point (to be used with smooth triangles)
          
          #local i = i + 1;
        #end
        
        #local h = h + h_step;
      
      
        //generate the rest of the point and create the triangles for the mesh
        #while(h < HEIGHT)
          
          #local h_ratio = h/HEIGHT;
          #local rad = max(base_rad*(1 - h_ratio), 0.0025);
          #local bark_thickness = 0.11 * (1-h_ratio + 0.1);
          
          #local dspline = branch_spline(h_ratio+0.01) - branch_spline(h_ratio); //direction of bottom-most cross-section
          
          #if((dspline.x != 0) | (dspline.y != 0) | (dspline.z != 0)) //catch zero-length vectors
          
            #local norm = vnormalize(dspline);
            #local norm2 = vnormalize(<norm.x+0.1,norm.y,norm.z>);
            #local norm_perp = vnormalize(vcross(norm, norm2));
            
            //set upper points
            #local i=0;
            #while(i < points)
              #local theta = (i/points)*360;
              
              #local point1 = norm_perp*rad; //a single point sticking out a distance of rad from the normal vector
              #local point1 = vaxis_rotate(point1, norm, theta) + branch_spline(h/HEIGHT) + START;
          
              #local point = norm_perp*(rad + bark_noise(point1.x,point1.y,point1.z)*rad*bark_thickness + TREE_BULGE_NOISE(point1.x,point1.y,point1.z)*rad*bulge_thickness);
              #local upper[i] = vaxis_rotate(point, norm, theta) + branch_spline(h/HEIGHT) + START; //to dermine all the points in the cross-section, start rotating point around norm
          
              //smooth normal
              #local u_norm[i] = vnormalize(upper[i] - (branch_spline(h_ratio) + START));
             
              
              #local i = i + 1;
            #end
            
            //create triangles
            #local i = 0;
            #while(i < points)
              #local j = i+1;
              #if(j >= points) #local j=0; #end
              
              #if(LEVEL <= 1)
                triangle{upper[i], lower[i], lower[j]}
                triangle{upper[j], lower[j], upper[i]}
              #else
                smooth_triangle{upper[i], u_norm[i], lower[i], l_norm[i], lower[j], l_norm[j]}
                smooth_triangle{upper[j], u_norm[j], lower[j], l_norm[j], upper[i], u_norm[i]}
              #end
              
              
              #local i = i + 1;
            #end
            
            //set lower array equal to upper array
            #local i = 0;
            #while(i < points)
              #local lower[i] = upper[i];
              #local l_norm[i] = u_norm[i];
              
              #local i = i + 1;
          #end
            
          #end //end if-statement that catches zero-length vectors
            
          
          #local h = h + h_step;
        #end
                                        
        texture{TREE_BROWN_TEXTURE}
          
        #undef bark_noise //since patterns cannot be re-declared or overwritten, I must undefine this pattern before reusing its name 
      }
    #end //end if-statement blocking overly short (often degenerate) branches                 
    
    
    
    
    //SPAWN OFF FURTHER BRANCHES
      #local h = 0; 
      #local max_branch_height = HEIGHT*0.95; 
      #local min_branch_height = MIN_BRANCH_HEIGHT*HEIGHT*0.5;
      #local branch_prob = BRANCHINESS;
      #local theta = 0;
      
      
      #if(LEVEL = 0) //increase the branchiness of the trunk
        #local min_branch_height = HEIGHT*0.20;
        #local branch_prob = 0.95;
      #end
      
      
      
      #while(h < HEIGHT)
        
        #local h_ratio = h/HEIGHT;
        
        #local thick_ratio = max(base_rad*(1 - h_ratio), 0.01)/MAX_RAD;
        #local branch_step = max(thick_ratio,0.50);
        
        #local branch_prob = max(branch_prob, h_ratio*2);
                 
        #if((h > min_branch_height) & (h < max_branch_height) & (rand(R1) < branch_prob))
          
          #local phi = rand(R1)*30*(max(1-thick_ratio,0.30)) + 60; //rotation away from branch axis
          #local theta = theta + 60 + rand(R1)*40; //rotation around branch axis
          
          #local dspline = vnormalize(branch_spline(h_ratio+0.1) - branch_spline(h_ratio));
          #if((dspline.x != 0) | (dspline.y != 0) | (dspline.z != 0))
            #local norm = vnormalize(dspline);
          #else
            #local norm = NORM;
          #end
          
          #local norm2 = vnormalize(<NORM.x+0.1,NORM.y,NORM.z>);
          #local norm_perp = vnormalize(vcross(NORM, norm2));
          #local norm1 = vnormalize(vaxis_rotate(NORM, norm_perp, phi));
          #local norm1 = vnormalize(vaxis_rotate(norm1, NORM, theta));
          #local norm1 = vnormalize(<(norm1.x+NORM.x)/2, (norm1.y+NORM.y)/2, (norm1.z+NORM.z)/2>); 
        
          #local Level = LEVEL + 1;
          #local Thickness = THICKNESS*0.80;
          #local Start = branch_spline(h_ratio)+START;
          #local Height = (rand(R1)*0.2 + 0.8)*(HEIGHT-h);
          
          #local C_Var = rand(R1)*C_VAR;          
          #local Color = rgb<COLOR_1.red*(1-C_Var) + COLOR_2.red*C_Var, COLOR_1.green*(1-C_Var) + COLOR_2.green*C_Var, COLOR_1.blue*(1-C_Var) + COLOR_2.blue*C_Var>;
          
          #local Min_Branch_Height = max(0.05,MIN_BRANCH_HEIGHT*0.50);
          
          //try to avoid branches that point downward early in the tree's structure
          #local height_ratio = (h + MAX_HEIGHT - HEIGHT)/MAX_HEIGHT;
          
          
          
          #local make_branch = true;
          #if((norm1.y < 0) & (height_ratio < (1 - thick_ratio))) //don't make branches if they are pointing down and they are coming too early in the tree's development
            #local make_branch = false;
          #end
          
          #if(LEVEL < LEVEL_MAX-1 & make_branch)
            Make_Tree_Branch(Level, Start, Height, Thickness, norm1, Min_Branch_Height, Color, rand(R1)*1000)
          #end
        #end
        
        #local h = h + branch_step;
            
      #end
       
       
       
       
      //Make Leaves
      #local leaf_max = floor(LEAFINESS*100);
      
      #local leaf_norm = norm1 + <0,-1,0>;
        
      #if(LEAFINESS > 0)
        #local spline_start = max(0, 2*HEIGHT/MAX_HEIGHT);
        #local spline_step = 0.025 * (6/HEIGHT);
        
        #local i = spline_start;
        
        #local y_bend = (rand(R1)*2-1)*60;
        
        #while(i <= 1)
          
          //randomly make leaves based on LEAFINESS   
          #local i_ratio = (i-spline_start)/(1-spline_start);
          
          #local norm = vnormalize(branch_spline(i+spline_step) - branch_spline(i));  
          
          
          #local theta_x = degrees(asin(norm.x/sqrt(norm.x*norm.x + norm.y*norm.y)));
          #if(norm.y > 0) #local theta_x = theta_x * -1; #end
          #if(norm.y < 0) #local theta_x = theta_x - 180; #end
          
          #local theta_z = degrees(asin(norm.z/sqrt(norm.z*norm.z + norm.y*norm.y)));
          
          
          #if(rand(R1) < LEAFINESS)
            object
            {
              //randomly select a leaf from the TREE_LEAF_ARRAY
              TREE_LEAF_ARRAY[floor(rand(R1)*TREE_LEAF_MAX)]   
              
              pigment{Color filter FILTER}
              finish {TREE_FINISH}
            
              
              rotate z*-30*(rand(R1)*0.30+0.70)
              rotate -y*i_ratio*y_bend
              rotate <theta_z+(rand(R1)*0.1+0.9)*-15, 0, theta_x+(rand(R1)*0.1+0.9)*-15> 
              translate branch_spline(min(i + rand(R1)*0.1, 0.95))+START
            }
          #end
          
          
          #if(rand(R1) < LEAFINESS)
            object
            { 
              //randomly select another leaf from the TREE_LEAF_ARRAY
              TREE_LEAF_ARRAY[floor(rand(R1)*TREE_LEAF_MAX)]       
              
              pigment{COLOR_1 filter FILTER}
              finish {TREE_FINISH}
              
              rotate z*30*(rand(R1)*0.30+0.70)
              rotate -y*i_ratio*y_bend
              rotate <theta_z+(rand(R1)*0.1+0.9)*-15, 0, theta_x+(rand(R1)*0.1+0.9)*-15>
              translate branch_spline(min(i + rand(R1)*0.1, 0.95))+START                
            }
          #end
            
            
            
          
          
          #local i = i + spline_step;
        #end
      #end
  
  #end
  
#end //Make_Branch() 


           
           
           
           
           
           
           
           
           
           
           
           
                               

  //                                                          \\
 //----------------------MAKE TREE LEAF------------------------\\
//                                                              \\

// This macro is called by the Populate_Tree_Leaf_Array() macro listed below in order to generate original leaves for the Tree.

#macro Make_Tree_Leaf(HEIGHT, WIDTH, BEND, RUFFLE, SEED)


/**********   VARIABLES   **************\
HEIGHT -      Float -   length of the leaf from stem to tip
WIDTH -       Float -   width of the leaf from side to side at its widest point
BEND -        Radian -  curl of the leaf from stem to tip
RUFFLE -      Float -   amount by which the leaves curl from leaf to right from the central stem
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh
INNER_COLOR - rgb -     color value of the center of the leaf
OUTER_COLOR - rgb -     color value of the edges of the leaf
FILTER -      0-1 -     filtered light value
LUSTER -      0-1 -     specular highlight of the leaf
AMBIENT -     0-1 -     ambient value of the leaf
\***************************************/   

#local R1 = seed(SEED);
 
  mesh
  {    
    //dark green texture of the leaf's stemm
    #local TREE_LEAF_STEM_TEXTURE =  texture{pigment{rgbft<0.34,0.29,0.17, 0.2, 0>} finish{ambient 0 diffuse 0.6 specular 0.1}};
        
    #local RUFFLE = max(RUFFLE, 0.05); //ruffle cannot be below 0.05 (no perfectly flat leaves allowed)
    
    #local leaf_wrinkle = 0.07; //ammount by which the leaf wrinkes
    
    #local h_step = HEIGHT*(1-SMOOTHNESS)*0.1; //distance between points on the mesh
    #local h = 0; //current value along the stem-to-tip length of the leaf
    
    #local points = max(3, ceil(WIDTH/h_step)); //number of points per horizontal cross-section of the leaf
    
    //point arrays which will be used to create the triangles in the mesh from
    #local lower = array[points];
    #local upper = array[points];
    
    #local i = 0;
    #while(i < points)
      
      #local lower[i] = <0,0,0>;
      
      #local i = i + 1;
    #end
    
    #local rpx = rand(R1)*1000;
    #local rpy = rand(R1)*1000;
    #local rpz = rand(R1)*1000;
                      
                      
    //LEAF STEM
    #local sp1 = <-WIDTH*0.025,0,0>;
    #local sp2 = < WIDTH*0.025,0,0>;
    #local sp3 = <-WIDTH*0.020,HEIGHT*0.2,0>;
    #local sp4 = < WIDTH*0.020,HEIGHT*0.2,0>;
    
    triangle{sp1, sp3, sp2 texture{TREE_LEAF_STEM_TEXTURE}}
    triangle{sp3, sp4, sp2 texture{TREE_LEAF_STEM_TEXTURE}}
    
    
    
    
    //LEAF BODY
    #while(h < HEIGHT)
      
      #local h_ratio = h/HEIGHT;
      
      #local rad = sin(h_ratio*pi)*WIDTH;
      #if(h+h_step >= HEIGHT) #local rad = 0; #end
            
      #local i = 0;
      #while(i < points)
        #local i_ratio = i/points;
        
        #local z_bend = BEND * h_ratio;
        
        #local pp = <rad*i_ratio - rad/2,      cos(z_bend)*(h + HEIGHT*0.15) + cos(z_bend)*((1-h_ratio) * rad*RUFFLE*(1-h_ratio)*abs(sin(i_ratio*pi*2))),      sin(z_bend)*(h + HEIGHT*0.15)> + sin(z_bend)*((1-h_ratio) * rad*RUFFLE*(1-h_ratio)*abs(sin(i_ratio*pi*2)));
        
        #local upper[i] = <pp.x,        pp.y + TREE_LEAF_PATTERN(pp.x, pp.y, pp.z)*leaf_wrinkle*HEIGHT,       pp.z + TREE_LEAF_PATTERN(pp.x+rpx,pp.y+rpy,pp.z+rpz)*leaf_wrinkle*HEIGHT>;
        
        #local i = i + 1;
      #end
      
      
      #local i = 0;
      #while(i < points-1)  //Create the triangles
        
        #local i_ratio = i/(points-1);
        

        triangle{upper[i], lower[i], lower[i+1] }
        triangle{upper[i+1], upper[i], lower[i+1] }   
        
        #local i = i + 1;
      #end
      
      
      #local i = 0;
      #while(i < points)
        
        #local lower[i] = upper[i];
        
        #local i = i + 1;
      #end
      
      #local h = h + h_step;
    #end
    
   
   double_illuminate 
   
  }

#end //Make_Leaf()













  //                                                              \\
 //--------------------POPULATE TREE LEAF ARRAY--------------------\\
//                                                                  \\   

// Use this macro to create an array of unique leaves for your tree.  PLEASE NOTE: This macro MUST be run before the Make_Tree() macro, if you plan to use leaves with you tree

#macro Populate_Tree_Leaf_Array(LEAF_CNT, VARIETY, HEIGHT, THICKNESS, BEND, RUFFLE, SMOOTHNESS, SEED)


/**********   VARIABLES   **************\
LEAF_CNT -    Integer - Number of unique leaves to generate in the array
VAREITY -     0-1 -     determines how individual each leaf appears from one another in the array
HEIGHT -      Float -   maximum length of a leaf from stem to tip
THICKNESS -   Float -   determines the width of a given leaf based on its HEIGHT
BEND -        Radian -  maximum curl of a leaf from stem to tip
RUFFLE -      Float -   maximum amount by which a leaf curls out around the central stem
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh (a higher value means more triangles)
SEED -        Integer - random seed 
\***************************************/   


#local R1 = seed(SEED);
#local INV_VAR = 1-VARIETY;

//local variable declared as global variable (to be referenced in Make_Tree_Leaf() macro)
#declare SMOOTHNESS = SMOOTHNESS;


//global values
#declare TREE_LEAF_ARRAY = array[LEAF_CNT];
#declare TREE_LEAF_MAX = LEAF_CNT;

//populate the array with unique leaves
#local i = 0;
#while(i < LEAF_CNT)
  #local Height = HEIGHT * (rand(R1)*VARIETY + INV_VAR);
  #local Width = Height * THICKNESS;
  
  #local Bend = BEND * (rand(R1)*VARIETY + INV_VAR);
  #local Ruffle = RUFFLE * (rand(R1)*VARIETY + INV_VAR);
  #local Seed = rand(R1)*10000;
  
  #declare TREE_LEAF_ARRAY[i] = Make_Tree_Leaf(Height, Width, Bend, Ruffle, Seed)
                                                
  #local i = i + 1;
#end
  

#end // Populate_Tree_Leaf_Array()














  //                                                        \\
 //-----------------PATTERNS AND TEXTURES -------------------\\
//                                                            \\ 
 
#local skew = <1,8,1>;
 
 
 
//BARK PATTERN
#declare TREE_BARK_PATTERN = 
function
{ 
  pattern
  { 
    agate
    
    //turbulence 1
     
    scale 0.2*skew
  } 
} 



//BRANCH BULDGE PATTERN
#declare TREE_BULGE_NOISE =
function
{
  pattern
  {
    agate
    
    scale 5
  }
}


              
//COLOR OF BARK
#declare TREE_BROWN_TEXTURE =
texture
{
  pigment 
  {
    granite
    
    color_map
    {
      [-0.1 rgbft<0.45,0.25,0.15, 0.0, 0.0>] //dark brown
      [ 0.6 rgbft<0.55,0.35,0.15, 0.0, 0.0>] //dark brown
      [ 1.1 rgbft<0.90,0.80,0.70, 0.0, 0.0>] //light orangish-gray
    }
    
    scale 1*skew 
  }
  
  finish
  {
    ambient 0.00
    diffuse 0.8 
    specular 0.25 
    roughness 0.4 
  }
  
  
}



//GREEN MOSS TEXTURE
#declare TREE_MOSS_TEXTURE =
texture
{
  pigment 
  {
    granite
    
    color_map
    {
      [-0.1 rgbft<0.1, 0.4, 0.0, 1.0, 1.0>] //invisible green
      [ 0.2 rgbft<0.1, 0.4, 0.0, 1.0, 1.0>] //invisible green
      [ 0.8 rgbft<0.1, 0.5, 0.0, 0.5, 0.5>] //green fading in
      [ 1.1 rgbft<0.3, 0.6, 0.0, 0.4, 0.4>] //solid green
    }
    
    scale 8 
  }
  

}



//LEAF WRINKLE PATTERN
#declare TREE_LEAF_PATTERN = 
function
{ 
  pattern
  { 
    agate
     
    lambda 0.8
    octaves 3
    omega 5  
    turbulence 0.1

     
    scale 11
    translate y*6765
  } 
}


 
 
